/************************************************************************/
/*									*/
/*		builddata.c						*/
/*									*/
/*	Data management for Make interface				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "build_local.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/dir.h>




/************************************************************************/
/*									*/
/*	Local types							*/
/*									*/
/************************************************************************/


typedef struct _DIRDATA *	DIRDATA;


typedef struct _DIRDATA {
   String name;
   Integer numfiles;
   String * files;
   DIRDATA next;
} DIRDATA_INFO;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_projs;
static	DIRDATA 	all_dirs;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	data_compare();
static	void		add_depend();
static	void		get_match_default();
static	void		get_default();
static	void		get_implied_files();
static	Boolean 	test_source();
static	Boolean 	test_binary();
static	Boolean 	test_depend();
static	Boolean 	test_library();
static	Boolean 	test_macro();
static	void		make_special();
static	void		compute_macro();
static	Boolean 	check_match();
static	void		replace_match();
static	Boolean 	valid_default();
static	Boolean 	check_exists();
static	DIRDATA 	find_directory();
static	void		fill_directory();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String	special_files[] = {
   ".SILENT",
   ".IGNORE",
   ".PRECIOUS",
   ".SUFFIXES",
   NULL
};



static	String	binary_suffix[] = {
   ".o", ".oo", ".do",
   NULL
};



struct {
   String suffix;
   Integer flags;
} source_suffix[] = {
   { ".c", BUILD_TYPE_SOURCE|BUILD_TYPE_C },
   { ".p", BUILD_TYPE_SOURCE|BUILD_TYPE_PASCAL },
   { ".y", BUILD_TYPE_SOURCE|BUILD_TYPE_YACC },
   { ".l", BUILD_TYPE_SOURCE|BUILD_TYPE_LEX },
   { ".h", BUILD_TYPE_HEADER },
   { ".c~", BUILD_TYPE_SOURCE|BUILD_TYPE_C|BUILD_TYPE_SCCS },
   { ".p~", BUILD_TYPE_SOURCE|BUILD_TYPE_PASCAL|BUILD_TYPE_SCCS },
   { ".y~", BUILD_TYPE_SOURCE|BUILD_TYPE_YACC|BUILD_TYPE_SCCS },
   { ".l~", BUILD_TYPE_SOURCE|BUILD_TYPE_LEX|BUILD_TYPE_SCCS },
   { ".h~", BUILD_TYPE_HEADER|BUILD_TYPE_SCCS },
   { NULL, BUILD_TYPE_SOURCE }
};



/************************************************************************/
/*									*/
/*	BUILD_data_init -- module initialization			*/
/*									*/
/************************************************************************/


void
BUILD_data_init()
{
   all_projs = NULL;
   all_dirs = NULL;
};





/************************************************************************/
/*									*/
/*	BUILD_find_project -- find project for file			*/
/*									*/
/************************************************************************/


BUILD_PROJ
BUILD_find_project(file)
   String file;
{
   Character buf[1024];
   String s;
   BUILD_PROJ bp;
   Sequence l;
   struct stat sbuf;

   if (file[0] != '/') sprintf(buf,"%s/%s",MSGinq_wd(),file);
   else strcpy(buf,file);
   s = rindex(buf,'/');

   if (stat(buf,&sbuf) >= 0 && (sbuf.st_mode & S_IFDIR) != 0) {
      ++s;
    }
   else {
      *s++ = 0;
      if (stat(buf,&sbuf) < 0 || (sbuf.st_mode & S_IFDIR) == 0) {
	 return NULL;
       };
    };

   PROTECT;

   forin (bp,BUILD_PROJ,l,all_projs) {
      if (MSGfile_compare(buf,bp->system_dir)) break;
    };

   if (bp == NULL) {
      bp = PALLOC(BUILD_PROJ_INFO);
      all_projs = CONS(bp,all_projs);
      bp->system_name = NULL;
      bp->system_dir = SALLOC(buf);
      bp->systems = NULL;
      bp->commands = NULL;
      bp->files = NULL;
      bp->defaults = NULL;
      bp->macros = NULL;
      bp->librarys = NULL;
      bp->make_name = NULL;
      bp->temp_make = NULL;
      bp->pipe_fd = -1;
      bp->cmd_pid = 0;
      bp->trans_fd = NULL;
      bp->transcript = NULL;
      bp->trans_size = 0;
      bp->trans_len = 0;
      bp->trans_data = NULL;
      bp->uptodate = FALSE;
      bp->xtra[0] = 0;
      bp->match_bits = 0;
      bp->last_file = NULL;
      bp->last_proc = NULL;
      bp->last_line = 0;
      bp->last_cmd = NULL;
      bp->respond_id = -1;
      bp->continue_opt = FALSE;
      bp->show_opt = FALSE;
      bp->touch_opt = FALSE;
      bp->ignore_opt = FALSE;
      bp->autodepend = TRUE;
      bp->nosuffix = FALSE;
      make_special(bp);
      BUILD_file_setup(bp);
    };

   UNPROTECT;

   return bp;
};





/************************************************************************/
/*									*/
/*	BUILD_find_proj_by_file -- find project from file bits		*/
/*									*/
/************************************************************************/


BUILD_PROJ
BUILD_find_proj_by_file(rfgs)
   Integer rfgs[];
{
   BUILD_PROJ bp;
   Sequence l;
   Integer i;

   forin (bp,BUILD_PROJ,l,all_projs) {
      if (bp->pipe_fd >= 0) {
	 i = 1 << bp->pipe_fd;
	 if (i & rfgs[0]) break;
       };
    };

   return bp;
};





/************************************************************************/
/*									*/
/*	BUILD_data_macro_define -- (re)define a macro			*/
/*									*/
/************************************************************************/


void
BUILD_data_macro_define(bp,name,text,perm,lib)
   BUILD_PROJ bp;
   String name;
   String text;
   Boolean perm;
   Boolean lib;
{
   Sequence l;
   BUILD_MACRO bm;

   bp->uptodate = FALSE;

   if (STREQL(name,"VPATH")) {
      if (bp->source_dir != NULL) SFREE(bp->source_dir);
      if (text != NULL) bp->source_dir = SALLOC(text);
      else bp->source_dir = NULL;
      return;
    };

   forin (bm,BUILD_MACRO,l,bp->macros) {
      if (STREQL(bm->name,name)) break;
    };

   if (bm == NULL) {
      bm = PALLOC(BUILD_MACRO_INFO);
      bm->name = SALLOC(name);
      bm->body = NULL;
      bm->option_name = NULL;
      bm->bit_value = 0;
      bm->defined = FALSE;
      bm->perm = FALSE;
      bm->dflt_type = BUILD_TYPE_UNKNOWN;
      bm->library = lib;
      bp->macros = APPEND(bm,bp->macros);
    };

   if (!lib && bm->library) bm->library = FALSE;

   if (perm || !bm->perm || !bm->defined) {
      if (bm->body != NULL) SFREE(bm->body);
      if (text != NULL) bm->body = SALLOC(text);
      else bm->body = NULL;
      bm->defined = TRUE;
      bm->perm = perm;
    };
};






/************************************************************************/
/*									*/
/*	BUILD_data_file -- find/create file from name			*/
/*									*/
/************************************************************************/


BUILD_FILE
BUILD_data_file(bp,name,libfg)
   BUILD_PROJ bp;
   String name;
   Boolean libfg;
{
   Sequence l;
   BUILD_FILE bf;

   if (bp == NULL) return NULL;

   forin (bf,BUILD_FILE,l,bp->files) {
      if (STREQL(bf->name,name)) break;
    };

   if (bf == NULL) {
      bp->uptodate = FALSE;
      bf = PALLOC(BUILD_FILE_INFO);
      bf->name = SALLOC(name);
      bf->real_name = NULL;
      bf->from = NULL;
      bf->depends = NULL;
      bf->build = NULL;
      bf->type = BUILD_TYPE_UNKNOWN|BUILD_TYPE_GET_DFLT;
      bf->dflt = NULL;
      bf->expand_list = NULL;
      bf->match_bits = 0;
      bf->generated = FALSE;
      bf->uptodate = FALSE;
      bf->library = libfg;
      bf->autodepend = FALSE;
      bp->files = APPEND(bf,bp->files);
    };

   return bf;
};





/************************************************************************/
/*									*/
/*	BUILD_data_depend -- add a single dependency			*/
/*									*/
/************************************************************************/


void
BUILD_data_depend(bp,f,t,autofg)
   BUILD_PROJ bp;
   BUILD_FILE f;
   BUILD_FILE t;
   Boolean autofg;
{
   bp->uptodate = FALSE;

   add_depend(f,t,autofg,FALSE);
};





/************************************************************************/
/*									*/
/*	BUILD_data_command -- add dependencies created explicitly	*/
/*									*/
/************************************************************************/


void
BUILD_data_command(bp,fr,to,cmds,autofg)
   BUILD_PROJ bp;
   Sequence fr;
   Sequence to;
   Sequence cmds;
   Boolean autofg;
{
   Sequence la,lb;
   BUILD_FILE fa,fb;
   String c;
   String buf;
   Integer ln;

   bp->uptodate = FALSE;

   forin (fa,BUILD_FILE,la,fr) {
      if (fa->type == BUILD_TYPE_SPECIAL && STREQL(fa->name,".SUFFIXES")) {
	 if (to != NULL) bp->nosuffix = FALSE;
	 else bp->nosuffix = TRUE;
       };
      if (bp->system_name == NULL) {
	 if (valid_default(fa)) bp->system_name = SALLOC(fa->name);
       };
      forin (fb,BUILD_FILE,lb,to) {
	 if (bp->system_name == NULL) {
	    if (valid_default(fb)) bp->system_name = SALLOC(fb->name);
	  };
	 add_depend(fa,fb,autofg,FALSE);
       };
    };

   ln = 0;
   forin (c,String,la,cmds) ln += strlen(c)+1;

   if (cmds != NULL) {
      buf = (String) malloc(ln+2);
      buf[0] = 0;
      forin (c,String,la,cmds) {
	 strcat(buf,c);
	 strcat(buf,"\n");
	 SFREE(c);
       };

      ln = 0;
      forin (fa,BUILD_FILE,la,fr) {
	 if (fa->name[0] != '.' && fa->build != NULL) {
	    fprintf(stderr,"Multiple build rules for %s\n",fa->name);
	  }
	 else {
	    if (ln++ == 0) fa->build = buf;
	    else fa->build = SALLOC(buf);
	  };
       };

      LFREE(cmds);
    };

   LFREE(fr);
   LFREE(to);
};





/************************************************************************/
/*									*/
/*	BUILD_data_set_rules -- set build rules for file		*/
/*	BUILD_data_macro_body -- set macro body 			*/
/*	BUILD_data_macro_option -- set option for macro 		*/
/*									*/
/************************************************************************/


void
BUILD_data_set_rules(bp,bf,rules)
   BUILD_PROJ bp;
   BUILD_FILE bf;
   String rules;
{
   bp->uptodate = FALSE;

   if (bf->build != NULL) SFREE(bf->build);
   else bf->type = BUILD_TYPE_UNKNOWN;

   bf->build = SALLOC(rules);
};





void
BUILD_data_macro_body(bp,bm,body)
   BUILD_PROJ bp;
   BUILD_MACRO bm;
   String body;
{
   bp->uptodate = FALSE;

   if (bm->body != NULL) SFREE(bm->body);
   bm->body = SALLOC(body);
   bm->defined = TRUE;
   bm->perm = TRUE;
   bm->library = FALSE;
};





void
BUILD_data_macro_option(bp,bm,name)
   BUILD_PROJ bp;
   BUILD_MACRO bm;
   String name;
{
   Integer i;

   bp->uptodate = FALSE;

   if (name == NULL || name[0] == 0) {
      if (bm->option_name != NULL) {
	 SFREE(bm->option_name);
	 bm->option_name = NULL;
	 bp->match_bits &= ~bm->bit_value;
	 bm->bit_value = 0;
       }
    }
   else {
      if (bm->option_name == NULL ||
	     STRNEQ(bm->option_name,name)) {
	 if (bm->option_name != NULL) SFREE(bm->option_name);
	 bm->option_name = SALLOC(name);
	 if (bm->bit_value == 0) {
	    for (i = 1; i != 0; i = i << 1) {
	       if ((bp->match_bits & i) == 0) break;
	     };
	    bp->match_bits |= i;
	    bm->bit_value = i;
	  };
       };
    };

   bm->library = FALSE;
};




/************************************************************************/
/*									*/
/*	BUILD_data_cleanup -- handle cleanup when reading is done	*/
/*									*/
/************************************************************************/


Boolean
BUILD_data_cleanup(bp)
   BUILD_PROJ bp;
{
   Sequence l;
   BUILD_FILE bf;
   BUILD_MACRO bm;

   if (bp == NULL || bp->uptodate) return FALSE;

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->name[0] == '.' && bf->name[1] != '/' && bf->name[1] != '.' &&
	     (bf->type & BUILD_TYPE_SPECIAL) == 0 &&
	     STRNEQ(bf->name,".DEFAULT")) {
	 bf->type = BUILD_TYPE_DEFAULT;
	 bp->defaults = APPEND(bf,bp->defaults);
       }
      else if (index(bf->name,'%') != NULL) {
	 bf->type = BUILD_TYPE_MATCH;
	 bp->defaults = APPEND(bf,bp->defaults);
       }
      else if (STREQL(bf->name,"Makefile") || STREQL(bf->name,"makefile") ||
		  STREQL(bf->name,"Projfile")) {
	 bf->type = BUILD_TYPE_BASIC;
       };
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      if ((bf->type & BUILD_TYPE_GET_DFLT) != 0) {
	 get_implied_files(bp,bf);
       };
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      get_match_default(bp,bf);
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      if ((bf->type & BUILD_TYPE_GET_DFLT) != 0) {
	 if (bf->build == NULL) get_default(bp,bf);
	 bf->type &= ~BUILD_TYPE_GET_DFLT;
       };
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->type != BUILD_TYPE_UNKNOWN) continue;
      if (bf->depends == NULL && bf->build != NULL) {
	 bf->type = BUILD_TYPE_COMMAND;
	 bp->commands = APPEND(bf,bp->commands);
       }
      else if (test_source(bf)) ;
      else if (test_binary(bf)) {
	 bf->type = BUILD_TYPE_BINARY;
       }
      else if (test_depend(bf)) {
	 bf->type = BUILD_TYPE_DEPEND;
       }
      else if (test_library(bf)) {
	 bf->type = BUILD_TYPE_LIBRARY;
       }
      else {
	 bf->type = BUILD_TYPE_SYSTEM;
	 bp->systems = APPEND(bf,bp->systems);
       };

      if (test_macro(bf)) bf->type |= BUILD_TYPE_MACRO;
    };

   forin (bm,BUILD_MACRO,l,bp->macros) {
      if (bm->bit_value != 0 || bm->dflt_type != 0) {
	 compute_macro(bp,bm);
       };
    };

   bp->uptodate = TRUE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	BUILD_data_inq_projects -- return array of projects		*/
/*	BUILD_data_inq_info -- return array of files of given type	*/
/*	BUILD_data_inq_macros -- return macros				*/
/*									*/
/************************************************************************/


Integer
BUILD_data_inq_projects(mx,bps)
   Integer mx;
   BUILD_PROJ bps[];
{
   Integer ct;
   BUILD_PROJ bp;
   Sequence l;

   ct = 0;
   forin (bp,BUILD_PROJ,l,all_projs) {
      if (ct < mx) {
	 bps[ct++] = bp;
       };
    };

   return ct;
};





Integer
BUILD_data_inq_info(bpq,ty,nty,mx,nms)
   BUILD_PROJ bpq;
   BUILD_TYPE ty,nty;
   Integer mx;
   String nms[];
{
   Integer ct,i;
   Sequence l,la;
   BUILD_FILE bf;
   BUILD_PROJ bp;

   ct = 0;

   forin (bp,BUILD_PROJ,l,all_projs) {
      if (bpq != NULL && bpq != bp) continue;
      forin (bf,BUILD_FILE,la,bp->files) {
	 if ((ty == 0 || (bf->type&ty) != 0) &&
		(nty == 0 || (bf->type&nty) == 0) &&
		(bf->type & BUILD_TYPE_SPECIAL) == 0 &&
		ct < mx) {
	    for (i = 0; i < ct; ++i) {
	       if (STREQL(nms[i],bf->name)) break;
	     };
	    if (i >= ct) nms[ct++] = bf->name;
	  };
       };
    };

   qsort(nms,ct,sizeof(String),data_compare);

   return ct;
};





Integer
BUILD_data_inq_macros(bpq,mx,nms)
   BUILD_PROJ bpq;
   Integer mx;
   String nms[];
{
   Integer ct;
   Sequence l,la;
   BUILD_MACRO bm;
   BUILD_PROJ bp;

   ct = 0;

   forin (bp,BUILD_PROJ,l,all_projs) {
      if (bpq != NULL && bpq != bp) continue;
      forin (bm,BUILD_MACRO,la,bp->macros) {
	 if (ct < mx) {
	    nms[ct++] = bm->name;
	  };
       };
    };

   qsort(nms,ct,sizeof(String),data_compare);

   return ct;
};





static Integer
data_compare(sa,sb)
   String * sa;
   String * sb;
{
   return strcmp(*sa,*sb);
};





/************************************************************************/
/*									*/
/*	BUILD_find_file -- find file and its project			*/
/*									*/
/************************************************************************/


BUILD_FILE
BUILD_find_file(nm,ty,proj)
   String nm;
   BUILD_TYPE ty;
   BUILD_PROJ * proj;
{
   Sequence l,la;
   BUILD_PROJ bp;
   BUILD_FILE bf;

   bf = NULL;

   if (nm == NULL) {
      if (all_projs != NULL && proj != NULL && *proj == NULL)
	 *proj = CAR(BUILD_PROJ,all_projs);
    }
   else if (proj != NULL && *proj != NULL) {
      bp = *proj;
      forin (bf,BUILD_FILE,l,bp->files) {
	 if (STREQL(nm,bf->name)) {
	    if (ty == BUILD_TYPE_UNKNOWN || (ty&bf->type) != 0) break;
	  };
       };
    }
   else {
      forin (bp,BUILD_PROJ,l,all_projs) {
	 forin (bf,BUILD_FILE,la,bp->files) {
	    if (STREQL(nm,bf->name)) {
	       if (ty == BUILD_TYPE_UNKNOWN || (ty&bf->type) != 0) break;
	     };
	  };
	 if (bf != NULL) break;
       };
      if (proj != NULL) *proj = bp;
    };

   return bf;
};





/************************************************************************/
/*									*/
/*	BUILD_find_macro -- find macro for project			*/
/*									*/
/************************************************************************/


BUILD_MACRO
BUILD_find_macro(bp,nm)
   BUILD_PROJ bp;
   String nm;
{
   Sequence l;
   BUILD_MACRO bm;

   forin (bm,BUILD_MACRO,l,bp->macros) {
      if (STREQL(bm->name,nm)) break;
    };

   return bm;
};





/************************************************************************/
/*									*/
/*	BUILD_data_remove -- remove all elements of a project		*/
/*									*/
/************************************************************************/


void
BUILD_data_remove(bp)
   BUILD_PROJ bp;
{
   Sequence l,la;
   BUILD_FILE bf;
   BUILD_DEPEND bd;
   BUILD_MACRO bm;

   if (bp->system_name != NULL) {
      SFREE(bp->system_name);
      bp->system_name = NULL;
    };

   LFREE(bp->systems);
   bp->systems = NULL;
   LFREE(bp->commands);
   bp->commands = NULL;
   LFREE(bp->defaults);
   bp->defaults = NULL;

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->name != NULL) SFREE(bf->name);
      if (bf->real_name != NULL) SFREE(bf->real_name);
      forin (bd,BUILD_DEPEND,la,bf->depends) free(bd);
      LFREE(bf->depends);
      if (bf->build != NULL) SFREE(bf->build);
      LFREE(bf->expand_list);
      free(bf);
    };
   LFREE(bp->files);
   bp->files = NULL;

   forin (bm,BUILD_MACRO,l,bp->macros) {
      if (bm->name != NULL) SFREE(bm->name);
      if (bm->body != NULL) SFREE(bm->body);
      if (bm->option_name != NULL) SFREE(bm->option_name);
      free(bm);
    };
   LFREE(bp->macros);
   bp->macros = NULL;

   LFREE(bp->librarys);
   bp->librarys = NULL;

   if (bp->make_name != NULL) {
      SFREE(bp->make_name);
      bp->make_name = NULL;
    };
};





/************************************************************************/
/*									*/
/*	BUILD_data_new_wd -- handle new working directory		*/
/*									*/
/************************************************************************/


void
BUILD_data_new_wd()
{
   DIRDATA dd,ndd,ldd;
   Integer i;

   ldd = NULL;
   for (dd = all_dirs; dd != NULL; dd = ndd) {
      ndd = dd->next;
      if (dd->name[0] != '/') {
	 for (i = 0; i < dd->numfiles; ++i) SFREE(dd->files[i]);
	 free(dd->files);
	 free(dd);
	 if (ldd == NULL) all_dirs = ndd;
	 else ldd->next = ndd;
       }
      else ldd = dd;
    };
};





/************************************************************************/
/*									*/
/*	add_depend -- add a dependency					*/
/*									*/
/************************************************************************/


static void
add_depend(fa,fb,autofg,implicit)
   BUILD_FILE fa;
   BUILD_FILE fb;
   Boolean autofg;
   Boolean implicit;
{
   BUILD_DEPEND bd;
   Sequence lc;

   if (fb->type & BUILD_TYPE_SPECIAL) {
      if (STREQL(fb->name,".SILENT")) fa->silent = TRUE;
      else if (STREQL(fb->name,".IGNORE")) fa->ignore = TRUE;
      else if (STREQL(fb->name,".PRECIOUS")) fa->precious = TRUE;
      return;
    };

   if (fa->depends == NULL && (fa->type & BUILD_TYPE_SPECIAL) == 0 )
      fa->type = BUILD_TYPE_UNKNOWN|BUILD_TYPE_GET_DFLT;

   forin (bd,BUILD_DEPEND,lc,fa->depends) {
      if (bd->file == fb) {
	 if (!autofg && bd->automatic) bd->automatic = FALSE;
	 if (!implicit && bd->implicit) bd->implicit = FALSE;
	 break;
       };
    };

   if (bd == NULL) {
      bd = PALLOC(BUILD_DEPEND_INFO);
      bd->file = fb;
      bd->automatic = autofg;
      bd->implicit = implicit;
      fa->depends = APPEND(bd,fa->depends);
      if (autofg) fa->autodepend = TRUE;
    };
};




/************************************************************************/
/*									*/
/*	get_implied_files -- insure all implied files are present	*/
/*									*/
/************************************************************************/


static void
get_implied_files(bp,bf)
   BUILD_PROJ bp;
   BUILD_FILE bf;
{
   String ext,extd,s;
   Integer eln;
   Sequence l;
   BUILD_FILE bfd,bfn;
   Character buf[1024],nbuf[1024];

   bf->from = NULL;
   bf->dflt = NULL;

   ext = rindex(bf->name,'.');
   if (ext != NULL && index(ext,'/') != NULL) ext = NULL;
   if (ext == NULL) return;

   nbuf[0] = 0;
   eln = strlen(ext);

   forin (bfd,BUILD_FILE,l,bp->defaults) {
      if (bfd->type & BUILD_TYPE_DEFAULT && bfd->build != NULL) {
	 if (strncmp(ext,bfd->name,eln) == 0 &&
		(bfd->name[eln] == '.' || bfd->name[eln] == 0)) {
	    strcpy(buf,bf->name);
	    s = rindex(buf,'.');
	    *s = 0;
	    s = bfd->name;
	    extd = index(&bfd->name[1],'.');
	    if (extd != NULL) strcat(buf,extd);
	    bfn = BUILD_data_file(bp,buf,FALSE);
	    bfn->generated = TRUE;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	get_match_default -- attempt to find default rule matching file */
/*									*/
/************************************************************************/


static void
get_match_default(bp,bf)
   BUILD_PROJ bp;
   BUILD_FILE bf;
{
   Sequence l,la,lb;
   BUILD_FILE bfd,bfa,bfb,bff;
   Character buf[1024],nbuf[1024],fbuf[1024];
   BUILD_DEPEND bd,bda;

   forin (bd,BUILD_DEPEND,l,bf->depends) {
      bfd = bd->file;
      if ((bfd->type & BUILD_TYPE_MATCH)) {
	 forin (bfa,BUILD_FILE,la,bp->files) {
	    if (bfa->type & (BUILD_TYPE_DEFAULT|BUILD_TYPE_MATCH|BUILD_TYPE_SPECIAL))
	       continue;
	    if (check_match(bfa->name,bfd->name,nbuf)) {
	       if (bf->type & BUILD_TYPE_MATCH) {
		  if (bf->from == NULL) bf->from = bfd;
		  replace_match(bf->name,nbuf,buf);
		  bff = BUILD_data_file(bp,buf,FALSE);
		  if (bf->build == NULL || (bff->type & BUILD_TYPE_GET_DFLT) != 0) {
		     forin (bda,BUILD_DEPEND,lb,bf->depends) {
			if (bda == bd) add_depend(bff,bfa,TRUE,TRUE);
			else if (bda->file->type & BUILD_TYPE_MATCH) {
			   replace_match(bda->file->name,nbuf,fbuf);
			   bfb = BUILD_data_file(bp,fbuf,FALSE);
			   add_depend(bff,bfb,TRUE,TRUE);
			 }
			else add_depend(bff,bda->file,TRUE,TRUE);
		      };
		   };
		  if ((bff->type & BUILD_TYPE_GET_DFLT) != 0 && bf->build != NULL) {
		     bff->type &= ~BUILD_TYPE_GET_DFLT;
		     bff->dflt = bf;
		     bff->from = bfa;
		   };
		}
	       else {
		  add_depend(bf,bfa,TRUE,TRUE);
		}
	     };
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	get_default -- attempt to find default rule for file		*/
/*									*/
/************************************************************************/


static void
get_default(bp,bf)
   BUILD_PROJ bp;
   BUILD_FILE bf;
{
   String ext,extd,s;
   Integer i;
   Sequence l,la;
   BUILD_FILE bfd,bfa,sbfd;
   Character buf[1024],nbuf[1024],fbuf[1024];
   BUILD_DEPEND bd;

   bf->from = NULL;
   bf->dflt = NULL;

   ext = rindex(bf->name,'.');
   if (ext != NULL && index(ext,'/') != NULL) ext = NULL;

   nbuf[0] = 0;
   bfa = NULL;

   forin (bfd,BUILD_FILE,l,bp->defaults) {
      if (bfd->type & BUILD_TYPE_MATCH) continue;
      extd = index(&bfd->name[1],'.');
      if ((extd == NULL && ext == NULL) ||
	     extd != NULL && ext != NULL && STREQL(ext,extd)) {
	 strcpy(buf,bf->name);
	 if (ext != NULL) {
	    s = rindex(buf,'.');
	    *s = 0;
	  };
	 i = strlen(buf);
	 s = bfd->name;
	 while (s != extd && *s != 0) buf[i++] = *s++;
	 buf[i] = 0;
	 forin (bd,BUILD_DEPEND,la,bf->depends) {
	    if (STREQL(bd->file->name,buf)) break;
	  };
	 if (bd != NULL) {
	    bfa = bd->file;
	    sbfd = bfd;
	    break;
	  }
	 else if (nbuf[0] == 0) {
	    sbfd = bfd;
	    if (buf[0] == '/') strcpy(fbuf,buf);
	    else if (bp->source_dir != NULL)
	       sprintf(fbuf,"%s/%s",bp->source_dir,buf);
	    else sprintf(fbuf,"%s/%s",bp->system_dir,buf);
	    if (check_exists(fbuf)) {
	       strcpy(nbuf,buf);
	     };
	  };
       };
    };

   if (bfa == NULL && nbuf[0] != 0) {
      bfa = BUILD_data_file(bp,nbuf,FALSE);
      add_depend(bf,bfa,TRUE,TRUE);
    };

   if (bfa != NULL) {
      bf->from = bfa;
      bf->dflt = sbfd;
    };
};





/************************************************************************/
/*									*/
/*	test_source -- check type of source file			*/
/*									*/
/************************************************************************/


static Boolean
test_source(bf)
   BUILD_FILE bf;
{
   String s;
   Integer i;

   s = rindex(bf->name,'.');

   if (s == NULL && bf->depends != NULL) return FALSE;
   else if (s == NULL) s = "";

   for (i = 0; source_suffix[i].suffix != NULL; ++i) {
      if (STREQL(s,source_suffix[i].suffix)) break;
    };

   if (bf->depends != NULL && source_suffix[i].suffix == NULL) return FALSE;

   bf->type = source_suffix[i].flags;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	test_binary -- check if file is binary or system		*/
/*									*/
/************************************************************************/


static Boolean
test_binary(bf)
   BUILD_FILE bf;
{
   String s;
   Integer i;

   s = rindex(bf->name,'.');

   if (s != NULL) {
      for (i = 0; binary_suffix[i] != NULL; ++i) {
	 if (STREQL(s,binary_suffix[i])) return TRUE;
       };
    };

   return FALSE;
};






/************************************************************************/
/*									*/
/*	test_depend -- check if file is dependency information		*/
/*									*/
/************************************************************************/


static Boolean
test_depend(bf)
   BUILD_FILE bf;
{
   String s;

   s = rindex(bf->name,'.');

   if (s != NULL && STREQL(s,".dep")) return TRUE;

   return FALSE;
};






/************************************************************************/
/*									*/
/*	test_library -- check if file is library			*/
/*									*/
/************************************************************************/


static Boolean
test_library(bf)
   BUILD_FILE bf;
{
   String s;

   s = rindex(bf->name,'.');

   if (s != NULL && STREQL(s,".a")) return TRUE;

   return FALSE;
};






/************************************************************************/
/*									*/
/*	test_macro -- check if file is just a macro name		*/
/*									*/
/************************************************************************/


static Boolean
test_macro(bf)
   BUILD_FILE bf;
{
   String s;

   if (bf->name[0] == '$' && bf->name[1] == '(') {
      s = index(bf->name,')');
      if (s != NULL && s[1] == 0) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	make_special -- make a special file				*/
/*									*/
/************************************************************************/


static void
make_special(bp)
   BUILD_PROJ bp;
{
   BUILD_FILE bf;
   Integer i;

   for (i = 0; special_files[i] != NULL; ++i) {
      bf = BUILD_data_file(bp,special_files[i],TRUE);
      bf->type = BUILD_TYPE_SPECIAL;
    };
};





/************************************************************************/
/*									*/
/*	compute_macro -- compute contents of a macro			*/
/*									*/
/************************************************************************/


static void
compute_macro(bp,bm)
   BUILD_PROJ bp;
   BUILD_MACRO bm;
{
   Sequence l;
   BUILD_FILE bf;
   Character buf[10240];
   Integer i;

   buf[0] = 0;
   i = 0;

   forin (bf,BUILD_FILE,l,bp->files) {
      if ((bm->bit_value & bf->match_bits) != 0 ||
	     (bm->dflt_type & bf->type) != 0) {
	 if (i == 0) ;
	 else if ((i%4) == 0) strcat(buf,"\n        ");
	 else strcat(buf," ");

	 strcat(buf,bf->name);

	 ++i;
       };
    };

   if (bm->body != NULL) SFREE(bm->body);
   bm->body = SALLOC(buf);
};






/********************************************************************************/
/*										*/
/*	check_match -- check and find match with a simple pattern		*/
/*										*/
/********************************************************************************/


static Boolean
check_match(nm,pat,buf)
   String nm;
   String pat;
   String buf;
{
   Integer i,j,k;
   String s;

   while (*pat != 0 && *pat != '%') {
      if (*pat++ != *nm++) return FALSE;
    };

   if (*pat == 0) return FALSE;
   ++pat;

   i = strlen(pat);
   j = strlen(nm);
   s = buf;
   for (k = 0; k < j-i; ++k) {
      *s++ = *nm++;
    };
   *s = 0;

   while (*pat != 0) {
      if (*pat++ != *nm++) return FALSE;
    };

   return TRUE;
};






static void
replace_match(nm,match,buf)
   String nm;
   String match;
   String buf;
{
   String s;

   s = buf;
   while (*nm != '%' && *nm != 0) *s++ = *nm++;
   if (*nm == '%') {
      ++nm;
      while (*match != 0) *s++ = *match++;
    };
   while (*nm != 0) *s++ = *nm++;
   *s = 0;
};






/************************************************************************/
/*									*/
/*	valid_default -- check if file can be a default 		*/
/*									*/
/************************************************************************/


static Boolean
valid_default(fa)
   BUILD_FILE fa;
{
   if (fa->name[0] == '.') return FALSE;
   if (index(fa->name,'%') != NULL) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_exists -- test if file exists				*/
/*									*/
/************************************************************************/


static Boolean
check_exists(f)
   String f;
{
   String s;
   DIRDATA dd;
   Integer i;

   s = rindex(f,'/');
   if (s == NULL) {
      s = f;
      f = ".";
    }
   else {
      *s++ = 0;
    };

   dd = find_directory(f);

   for (i = 0; i < dd->numfiles; ++i) {
      if (STREQL(s,dd->files[i])) return TRUE;
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	find_directory -- find directory information			*/
/*									*/
/************************************************************************/


static DIRDATA
find_directory(d)
   String d;
{
   DIRDATA dd;

   for (dd = all_dirs; dd != NULL; dd = dd->next) {
      if (STREQL(dd->name,d)) break;
    };

   if (dd == NULL) {
      dd = PALLOC(DIRDATA_INFO);
      dd->name = SALLOC(d);
      dd->numfiles = 0;
      dd->files = NULL;
      fill_directory(dd);
      PROTECT;
      dd->next = all_dirs;
      all_dirs = dd;
      UNPROTECT;
    };

   return dd;
};





static void
fill_directory(dd)
   DIRDATA dd;
{
   struct direct * dp;
   DIR * df;
   Integer i;
   String files[20480];
   Integer ct;

   df = opendir(dd->name);
   if (df == NULL) return;

   ct = 0;
   while ((dp = readdir(df)) != NULL) {
      if (STREQL(dp->d_name,".") || STREQL(dp->d_name,"..")) continue;
      files[ct++] = SALLOC(dp->d_name);
    };

   closedir(df);

   dd->numfiles = ct;
   dd->files = (String *) malloc(sizeof(String)*ct);

   for (i = 0; i < ct; ++i) {
      dd->files[i] = files[i];
    };
};




/* end of builddata.c */
