/************************************************************************/
/*									*/
/*		buildcmd.c						*/
/*									*/
/*	handle executing BUILD commands 				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/



#include "build_local.h"
#include <sys/errno.h>
#include <sys/wait.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TRANS_SIZE	10240
#define END_LINE	"--------------------------------------------------\n"




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		check_makefile();
static	void		exec_makefile();
static	void		exec_input();
static	void		output_line();
static	Boolean 	run_command();
static	Integer 	stop_command();
static	void		handle_reply();





/************************************************************************/
/*									*/
/*	BUILD_cmd_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
BUILD_cmd_init()
{
};





/************************************************************************/
/*									*/
/*	BUILD_cmd_make -- make part of a system 			*/
/*									*/
/************************************************************************/


void
BUILD_cmd_make(bp,item,rid)
   BUILD_PROJ bp;
   String item;
   Integer rid;
{
   if (STREQL(item,"*")) item = NULL;

   BUILD_find_file(item,BUILD_TYPE_UNKNOWN,&bp);

   if (bp == NULL) {
      handle_reply(rid,NULL);
      return;
    };

   if (item == NULL) item = bp->system_name;

   check_makefile(bp);

   exec_makefile(bp,item,rid);
};





/************************************************************************/
/*									*/
/*	BUILD_cmd_compile -- compile a file				*/
/*									*/
/************************************************************************/


void
BUILD_cmd_compile(bp,src,rid)
   BUILD_PROJ bp;
   String src;
   Integer rid;
{
   BUILD_FILE sbf,bf;
   BUILD_DEPEND bd;
   Sequence l,la;
   Boolean fg;

   sbf = BUILD_find_file(src,BUILD_TYPE_SOURCE,&bp);

   if (bp == NULL || sbf == NULL) {
      handle_reply(rid,NULL);
      return;
    };

   check_makefile(bp);

   fg = FALSE;
   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->type & BUILD_TYPE_BINARY) {
	 forin (bd,BUILD_DEPEND,la,bf->depends) {
	    if (bd->file == sbf) {
	       fg = TRUE;
	       exec_makefile(bp,bf->name,rid);
	       break;
	     };
	  };
       };
    };

   if (!fg) handle_reply(rid,NULL);
};




/************************************************************************/
/*									*/
/*	BUILD_cmd_compile_go -- compile a file and make corres system	*/
/*									*/
/************************************************************************/


void
BUILD_cmd_compile_go(bp,src,rid)
   BUILD_PROJ bp;
   String src;
   Integer rid;
{
   BUILD_FILE sbf,cbf,bf;
   BUILD_DEPEND bd;
   Sequence l,la;

   sbf = BUILD_find_file(src,BUILD_TYPE_SOURCE,&bp);

   if (bp == NULL || sbf == NULL) {
      handle_reply(rid,NULL);
      return;
    };


   check_makefile(bp);

   forin (cbf,BUILD_FILE,l,bp->files) {
      if (cbf->type & BUILD_TYPE_BINARY) {
	 forin (bd,BUILD_DEPEND,la,cbf->depends) {
	    if (bd->file == sbf) break;
	  };
	 if (bd != NULL) break;
       };
    };

   if (cbf != NULL) {
      forin (bf,BUILD_FILE,l,bp->files) {
	 if (bf->type & BUILD_TYPE_SYSTEM) {
	    forin (bd,BUILD_DEPEND,la,bf->depends) {
	       if (bd->file == cbf) {
		  exec_makefile(bp,bf->name,rid);
		  return;
		};
	     };
	  };
       };
    };

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->type & BUILD_TYPE_SYSTEM) {
	 forin (bd,BUILD_DEPEND,la,bf->depends) {
	    if (bd->file == sbf) {
	       exec_makefile(bp,bf->name,rid);
	       return;
	     };
	  };
       };
    };

   handle_reply(rid,NULL);
};




/************************************************************************/
/*									*/
/*	BUILD_cmd_depends -- generate .dep filed for dependency analysis*/
/*									*/
/************************************************************************/


void
BUILD_cmd_depends(bp)
   BUILD_PROJ bp;
{
   BUILD_FILE bf;
   Sequence l;

   if (bp == NULL) return;

   check_makefile(bp);

   forin (bf,BUILD_FILE,l,bp->files) {
      if (bf->type & BUILD_TYPE_DEPEND) {
	 exec_makefile(bp,bf->name,-1);
       };
    };

   while (bp->cmd_pid > 0) {		/* wait for it to finish	*/
      CMPXselect(0);
    };
};




/************************************************************************/
/*									*/
/*	check_makefile -- check if makefile is up to date for project	*/
/*									*/
/************************************************************************/


static void
check_makefile(bp)
   BUILD_PROJ bp;
{
   Character buf[32];

   BUILD_update_project(bp);

   if (!bp->uptodate || bp->temp_make == NULL) {
      BUILD_data_cleanup(bp);
      if (bp->temp_make == NULL) {
	 sprintf(buf,"/tmp/Build.makeXXXXXX");
	 mktemp(buf);
	 bp->temp_make = SALLOC(buf);
       };
      BUILD_file_gen_makefile(bp,FALSE);
    };

   if (bp->trans_data == NULL) {
      bp->trans_size = TRANS_SIZE;
      bp->trans_data = (String) malloc(bp->trans_size);
    };

   bp->trans_len = 0;
   bp->trans_data[0] = 0;
};




/************************************************************************/
/*									*/
/*	exec_makefile -- exec command using current makefile		*/
/*									*/
/************************************************************************/


static void
exec_makefile(bp,cmd,rid)
   BUILD_PROJ bp;
   String cmd;
   Integer rid;
{
   Character buf[256],optbuf[256];
   Integer rfgs[1];

   while (bp->cmd_pid > 0) {
      CMPXselect(0);
    };

   strcpy(optbuf,"-r");
   if (bp->continue_opt) strcat(optbuf," -k");
   if (bp->show_opt) strcat(optbuf," -n");
   if (bp->touch_opt) strcat(optbuf," -t");
   if (bp->ignore_opt) strcat(optbuf," -i");

   if (bp->last_cmd != NULL) SFREE(bp->last_cmd);
   bp->last_cmd = SALLOC(cmd);

   sprintf(buf,"(cd %s; make %s -f %s %s)\n",bp->system_dir,optbuf,
	      bp->temp_make,cmd);

   if (bp->transcript == NULL) {
      strcpy(optbuf,"/tmp/transcriptXXXXXX");
      mktemp(optbuf);
      bp->transcript = SALLOC(optbuf);
    };
   if (bp->trans_fd != NULL) fclose(bp->trans_fd);
   bp->trans_fd = NULL;

   bp->respond_id = rid;

   output_line(bp,buf);

   bp->trans_fd = fopen(bp->transcript,"w");

   if (!run_command(bp,buf)) return;

   bp->show_opt = FALSE;
   bp->touch_opt = FALSE;

   rfgs[0] = 1<<bp->pipe_fd;
   CMPXregister(bp->pipe_fd+1,rfgs,NULL,NULL,exec_input);
};





/************************************************************************/
/*									*/
/*	exec_input -- handle input on pipe from make			*/
/*									*/
/************************************************************************/


static void
exec_input(rfgs)
   Integer rfgs[];
{
   BUILD_PROJ bp;
   Character buf[2048];
   Character lin[1025];
   Integer ln;
   Integer sts;
   String s,t;
   extern int errno;

   bp = BUILD_find_proj_by_file(rfgs);

   if (bp == NULL) return;

   ln = read(bp->pipe_fd,buf,1024);

   if (ln <= 0) {
      CMPXclose(bp->pipe_fd);
      s = bp->transcript;
      if (bp->trans_fd != NULL) fclose(bp->trans_fd);
      else s = "*";
      sts = stop_command(bp);
      MSGsenda("BUILD FINISH %d %s %S",sts,s,bp->last_cmd);
      if (bp->respond_id >= 0) {
	 handle_reply(bp->respond_id,s);
	 bp->respond_id = -1;
       };
      return;
    };

   buf[ln] = 0;

   s = buf;
   while (*s != 0) {
      if (bp->xtra[0] != 0) {
	 strcpy(lin,bp->xtra);
	 bp->xtra[0] = 0;
       }
      else lin[0] = 0;
      t = &lin[strlen(lin)];

      while (*s != 0 && *s != '\n') {
	 if (*s != '\r') *t++ = *s;
	 ++s;
       };

      if (*s == 0) {
	 *t = 0;
	 strcpy(bp->xtra,lin);
	 break;
       };

      *t++ = '\n';
      *t = 0;
      output_line(bp,lin);
      ++s;
    };
};






/************************************************************************/
/*									*/
/*	output_line -- handle line coming from pipe			*/
/*									*/
/************************************************************************/


static void
output_line(bp,lin)
   BUILD_PROJ bp;
   String lin;
{
   Integer ln;
   BUILD_WIN bw;

   if (bp->trans_fd != NULL) {
      fputs(lin,bp->trans_fd);
    };

   BUILD_check_output(bp,lin);

   ln = strlen(lin);
   while (ln+bp->trans_len >= bp->trans_size) {
      bp->trans_size *= 2;
      bp->trans_data = (String) realloc(bp->trans_data,bp->trans_size);
    };
   strcpy(&bp->trans_data[bp->trans_len],lin);
   bp->trans_len += ln;

   bw = BUILD_find_window_by_project(bp);
   if (bw != NULL) BUILD_disp_transcript(bw,lin);

   printf("%s",lin);
};





/************************************************************************/
/*									*/
/*	run_command -- run a command for project			*/
/*									*/
/************************************************************************/


static Boolean
run_command(bp,cmd)
   BUILD_PROJ bp;
   String cmd;
{
   int p[2];
   int myside, hisside, pid;

   bp->cmd_pid = 0;
   bp->pipe_fd = -1;

   if (pipe(p) < 0) return FALSE;

   myside = p[0];
   hisside = p[1];

   if ((pid = vfork()) == 0) {
      /* myside and hisside reverse roles in child */
      close(myside);
      if (hisside != 1) {
	 dup2(hisside, 1);
	 close(hisside);
       }
      dup2(1, 2);
      execl("/bin/sh", "sh", "-f", "-c", cmd, (char *)NULL);
      _exit(127);
    }

   if (pid == -1) {
      close(myside);
      close(hisside);
      return FALSE;
    }

   close(hisside);
   bp->cmd_pid = pid;
   bp->pipe_fd = myside;

   return TRUE;
}





static Integer
stop_command(bp)
   BUILD_PROJ bp;
{
   int child, pid, status;

   pid = bp->cmd_pid;
   if (bp->cmd_pid != 0) kill(bp->cmd_pid,9);
   bp->cmd_pid = 0;
   if (bp->pipe_fd != 0) close(bp->pipe_fd);
   bp->pipe_fd = 0;

   status = 0;
   if (pid != 0) {
      while ((child = wait3(&status,WNOHANG,0)) > 0) {
	 if (child == pid) break;
	 status = 0;
       };
      if (child <= 0) status = EDTexec_inq_status(pid);
    };

   output_line(bp,END_LINE);

   return status;
};






/********************************************************************************/
/*										*/
/*	handle_reply -- handle reply to CALL messages				*/
/*										*/
/********************************************************************************/


static void
handle_reply(rid,ans)
   Integer rid;
   String ans;
{
   if (rid >= 0) MSGreply(rid,ans);
};





/* end of buildcmd.c */
