/************************************************************************/
/*									*/
/*		fixannot.c						*/
/*									*/
/*	Main routines for maintaining annotation database for files	*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#define NO_THREADS

#include "datatypes.h"
#include <sys/types.h>
#include <sys/stat.h>

#include "fixannot.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define 	INSERT_COST		1
#define 	DELETE_COST		1
#define 	EDIT_COST		1
#define 	MAX_COST		2




/************************************************************************/
/*									*/
/*	Data type definitions						*/
/*									*/
/************************************************************************/


typedef struct _ANNOT * 	ANNOT;
typedef struct _LINE *		LINE;
typedef struct _CLASS * 	CLASS;


typedef struct _ANNOT {
   ANNOT next;
   Integer line;
   Integer chr;
   FIXANNOT_FLAGS flags;
   String text;
   String sourceline;
   Boolean found;
   Integer dline;
   Integer cost;
} ANNOT_INFO;




typedef struct _LINE {
   LINE next;
   Integer num;
   String text;
} LINE_INFO;




typedef struct _CLASS {
   CLASS next;
   Integer class;
   Boolean keep;
   Boolean delete;
} CLASS_INFO;






/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	CLASS			add_class();
static	void			work_on();
static	void			get_annot_name();
static	void			create_annot_file();
static	ANNOT			get_annot_list();
static	Boolean 		get_annot();
static	void			file_annot();
static	void			show_annots();
static	void			bring_up_to_date();
static	ANNOT			sort_annots();
static	ANNOT			assign_annot();
static	ANNOT			handle_delete();
static	ANNOT			handle_change();
static	ANNOT			get_source_line();
static	LINE			get_new_source();
static	Integer 		difference();
static	void			compute_cost();
static	void			handle_search();






/************************************************************************/
/*									*/
/*	main -- main program						*/
/*									*/
/************************************************************************/


int
main(argc,argv)
   Integer argc;
   String  argv[];
{
   String iofile;
   Boolean update,list,set;
   Integer i;
   CLASS cls;

   iofile = NULL;
   update = TRUE;
   list = FALSE;
   set = FALSE;
   cls = NULL;

   for (i = 1; i < argc; ++i) {
      if (argv[i][0] == '-') {
	 switch (argv[i][1]) {
	    case 'f' :
	       iofile = argv[++i];
	       break;
	    case 'd' :
	       set = TRUE;
	       update = FALSE;
	       break;
	    case 'l' :
	       list = TRUE;
	       break;
	    case 'k' :
	       cls = add_class(cls,argv[++i],TRUE,FALSE);
	       break;
	    case 'r' :
	       cls = add_class(cls,argv[++i],FALSE,TRUE);
	       break;
	    default :
	       fprintf(stderr,"fixannot [-d] [-l] [-k #] [-r #] [-f iofile] file ...\n");
	       exit(1);
	  };
       }
      else {
	 work_on(argv[i],iofile,cls,set,update,list);
       };
    };

   return 0;
};






/************************************************************************/
/*									*/
/*	add_class -- add new element to class list			*/
/*									*/
/************************************************************************/


static CLASS
add_class(clst,id,keep,rmv)
   CLASS clst;
   String id;
   Boolean keep,rmv;
{
   CLASS c,c1;
   Integer cnum;

   cnum = atol(id);
   if (cnum == 0) {
      fprintf(stderr,"Bad class %s\n",id);
      return clst;
    };

   c = PALLOC(CLASS_INFO);
   c->class = cnum;
   c->keep = keep;
   c->delete = rmv;
   c->next = NULL;

   for (c1 = clst; c1 != NULL && c1->next != NULL; c1 = c1->next);
   if (c1 == NULL) clst = c;
   else c1->next = c;

   return clst;
};





/************************************************************************/
/*									*/
/*	work_on -- process annotations for a file			*/
/*									*/
/************************************************************************/


static void
work_on(file,iofile,clst,set,update,list)
   String file;
   String iofile;
   CLASS clst;
   Boolean set;
   Boolean update;
   Boolean list;
{
   Character ann[1024],orig[1024];
   FILE * inf;
   ANNOT a;

   get_annot_name(file,ann,orig);

   if (set) {
      if (iofile != NULL) freopen(iofile,"r",stdin);
      create_annot_file(file,ann,orig);
    }
   else {
      if (update) bring_up_to_date(file,ann,orig);

      if (list) {
	 if (iofile != NULL) freopen(iofile,"w",stdout);
	 inf = fopen(ann,"r");
	 if (inf != NULL) {
	    a = get_annot_list(inf);
	    fclose(inf);
	  };
	 show_annots(a,stdout);
       };
    };
};





/************************************************************************/
/*									*/
/*	get_annot_name -- get name of annotation file for given file	*/
/*									*/
/************************************************************************/


static void
get_annot_name(file,buf,buf1)
   String file;
   String buf;
   String buf1;
{
   String s,t;

   strcpy(buf,file);
   strcpy(buf1,file);

   t = rindex(file,'/');
   if (t == NULL) t = file;
   else ++t;

   s = rindex(buf,'/');
   if (s == NULL) s = buf;
   else ++s;
   sprintf(s,".%s.annot",t);

   s = rindex(buf1,'/');
   if (s == NULL) s = buf1;
   else ++s;
   sprintf(s,".%s.orig",t);
};





/************************************************************************/
/*									*/
/*	create_annot_file -- create a new annotation entry		*/
/*									*/
/************************************************************************/


static void
create_annot_file(file,buf,orig)
   String file;
   String buf;
   String orig;
{
   Character lbuf[10240];
   FILE * inf;
   FILE * otf;
   FILE * anf;
   ANNOT a;

   unlink(buf);
   unlink(orig);

   a = get_annot_list(stdin);
   if (a == NULL) return;

   inf = fopen(file,"r");
   if (inf == NULL) return;
   otf = fopen(orig,"w");
   if (otf == NULL) return;
   anf = fopen(buf,"w");
   if (anf == NULL) return;

   while (fgets(lbuf,10240,inf) != NULL) fputs(lbuf,otf);
   fclose(inf);
   fclose(otf);

   for ( ; a != NULL; a = a->next) {
      fprintf(anf,"%d %d %d %s\n",a->line,a->chr,a->flags,a->text);
    };

   fclose(anf);
};





/************************************************************************/
/*									*/
/*	get_annot_list -- get all annotations from a file		*/
/*									*/
/************************************************************************/


static ANNOT
get_annot_list(inf)
   FILE * inf;
{
   Integer line,chr,flag;
   Character info[10240];
   ANNOT first,last,a;

   first = NULL;
   last = NULL;

   while (get_annot(inf,&line,&chr,&flag,info)) {
      a = PALLOC(ANNOT_INFO);
      a->next = NULL;
      a->line = line;
      a->chr = chr;
      a->flags = flag;
      a->text = SALLOC(info);
      a->sourceline = NULL;
      a->found = FALSE;
      if (last == NULL) first = a;
      else last->next = a;
      last = a;
    };

   return first;
};





/************************************************************************/
/*									*/
/*	get_annot -- read annotation information from stdin		*/
/*									*/
/************************************************************************/


static Boolean
get_annot(inf,lp,cp,fp,info)
   FILE * inf;
   Integer * lp;
   Integer * cp;
   Integer * fp;
   String info;
{
   Integer ch;

   if (fscanf(inf,"%d %d %d",lp,cp,fp) != 3) {
      *lp = -1;
      return FALSE;
    };

   while ((ch = getc(inf)) == ' ');

   while (ch != '\n' && ch != 0) {
      *info++ = ch;
      ch = getc(inf);
    };

   *info = 0;

   return TRUE;
};






/************************************************************************/
/*									*/
/*	show_annots -- show current annotations for file		*/
/*									*/
/************************************************************************/


static void
show_annots(a,otf)
   ANNOT a;
   FILE * otf;
{
   for ( ; a != NULL; a = a->next) {
      fprintf(otf,"%d %d %d %s\n",a->line,a->chr,a->flags,a->text);
    };
};





/************************************************************************/
/*									*/
/*	bring_up_to_date -- bring annotation database up to date	*/
/*									*/
/************************************************************************/


static void
bring_up_to_date(fil,afil,orig)
   String fil;
   String afil;
   String orig;
{
   struct stat fstat,astat;
   FILE * dff;
   Character cmd[128],line[10240];
   Integer n1,n2,n3,n4,act;
   String s;
   FILE * anf;
   ANNOT a,all;
   Integer delta;

   anf = fopen(afil,"r");
   if (anf == NULL) a = NULL;
   else {
      a = get_annot_list(anf);
      fclose(anf);
    };

   if (a == NULL || stat(fil,&fstat) < 0 || stat(orig,&astat) < 0) {
      unlink(afil);
      unlink(orig);
      return;
    };

   if (fstat.st_mtime <= astat.st_mtime) return;

   sprintf(cmd,"diff %s %s",fil,orig);
   dff = popen(cmd,"r");
   all = a;

   delta = 0;

   while (fgets(line,10240,dff) != NULL) {
      if (line[0] == '<' || line[0] == '>' || line[0] == '-') continue;
      n1 = n2 = n3 = n4 = 0;
      act = 0;
      s = line;
      while (isdigit(*s)) n1 = n1*10 + *s++ - '0';
      if (*s == ',') {
	 ++s;
	 while (isdigit(*s)) n2 = n2*10 + *s++ - '0';
       }
      else n2 = n1;
      act = *s++;
      while (isdigit(*s)) n3 = n3*10 + *s++ - '0';
      if (*s == ',') {
	 ++s;
	 while (isdigit(*s)) n4 = n4*10 + *s++ - '0';
       }
      else n4 = n3;

      a = assign_annot(a,n1-1,delta);

      printf("difference %c <%d,%d> :: <%d,%d>\n",act,n1,n2,n3,n4);

      if (act == 'a') {                 /* Addition in new file */
	 a = assign_annot(a,n1,delta);
	 delta += n4-n3+1;
       }
      else if (act == 'd') {            /* Deletion in new file */
	 a = handle_delete(a,n1,n2,n3,dff);
	 delta -= n2-n1+1;
       }
      else {				/* Change between files */
	 a = handle_change(a,n1,n2,n3,n4,dff);
	 delta += n4-n3-n2+n1;
       };
    };

   pclose(dff);

   handle_search(all,fil);

   sprintf(cmd,"cp %s %s",fil,orig);
   system(cmd);

   a = sort_annots(all);

   anf = fopen(afil,"w");
   if (anf != NULL) {
      show_annots(a,anf);
      fclose(anf);
   };
};





/************************************************************************/
/*									*/
/*	sort_annots -- sort list of annotations 			*/
/*									*/
/************************************************************************/


static ANNOT
sort_annots(oa)
   ANNOT oa;
{
   ANNOT first;
   ANNOT a,na,la,nxt;

   first = NULL;
   for (a = oa; a != NULL; a = nxt) {
      nxt = a->next;
      if (!a->found) continue;

      la = NULL;
      for (na = first; na != NULL; na = na->next) {
	 if (na->line > a->line) break;
	 if (na->line == a->line && na->chr > a->chr) break;
	 la = na;
       };
      a->next = na;
      if (la == NULL) first = a;
      else la->next = a;
    };

   return first;
};





/************************************************************************/
/*									*/
/*	assign_annot -- assign annotations through given line		*/
/*									*/
/************************************************************************/


static ANNOT
assign_annot(a,line,delta)
   ANNOT a;
   Integer line;
   Integer delta;
{
   while (a != NULL && a->line <= line) {
      a->line += delta;
      a->found = TRUE;
      a = a->next;
    };

   return a;
};





/************************************************************************/
/*									*/
/*	handle_delete -- handle a deletion between old and new		*/
/*									*/
/************************************************************************/


static ANNOT
handle_delete(a,n1,n2,n3,dff)
   ANNOT a;
   Integer n1,n2,n3;
   FILE * dff;
{
   get_source_line(a,n1,n2,dff);

   while (a != NULL && a->line <= n2) {
      if (a->flags & FIXANNOT_KEEP) {
	 a->line = n3;
	 a->found = TRUE;
       }
      else if (a->flags & FIXANNOT_SEARCH) {
	 a->dline = n3;
       }
      else a->dline = -1;

      a = a->next;
    };

   return a;
};





/************************************************************************/
/*									*/
/*	handle_change -- handle a simple change between old and new	*/
/*									*/
/************************************************************************/


static ANNOT
handle_change(a,n1,n2,n3,n4,dff)
   ANNOT a;
   Integer n1,n2,n3,n4;
   FILE * dff;
{
   LINE l,ll;
   ANNOT aa;
   Integer n,v,i;

   get_source_line(a,n1,n2,dff);

   if (n1 == n2 && n3 == n4) {
      while (a != NULL && a->line == n1) {
	 a->line = n3;
	 a->found = TRUE;
	 a = a->next;
       };
    }
   else {
      l = get_new_source(dff,n3,n4);
      v = -1;
      for (aa = a; aa != NULL && aa->line <= n2; aa = aa->next) {
	 for (ll = l; ll != NULL; ll = ll->next) {
	    i = difference(aa->sourceline,ll->text);
	    if (i >= 0 && (v < 0 || i < v)) {
	       v = i;
	       n = ll->num;
	     };
	  };
	 if (v >= 0) {
	    aa->line = n;
	    aa->found = TRUE;
	  };
       };
      for (aa = a; aa != NULL && aa->line <= n2; aa = aa->next) {
	 if (!aa->found) {
	    if (aa->flags & FIXANNOT_KEEP) {
	       aa->line = n3;
	       aa->found = TRUE;
	     }
	    else if (aa->flags & FIXANNOT_SEARCH) {
	       aa->dline = n3;
	     }
	    else aa->dline = -1;
	  };
       };
      a = aa;
    };

   return a;
};





/************************************************************************/
/*									*/
/*	get_source_line -- get source line for annotations in change	*/
/*									*/
/************************************************************************/


static ANNOT
get_source_line(a,n1,n2,dff)
   ANNOT a;
   Integer n1,n2;
   FILE * dff;
{
   Character lbuf[10240];

   while (a != NULL && a->line < n1) a = a->next;

   lbuf[0] = 0;

   for ( ; n1 <= n2; ++n1) {
      if (fgets(lbuf,10240,dff) == NULL) return a;
      while (a != NULL && a->line == n1) {
	 a->sourceline = SALLOC(&lbuf[2]);
	 a = a->next;
       };
    };

   return a;
};





/************************************************************************/
/*									*/
/*	get_new_source -- get replacement source lines			*/
/*									*/
/************************************************************************/


static LINE
get_new_source(dff,n1,n2)
   FILE * dff;
   Integer n1,n2;
{
   Character lbuf[10240];
   LINE l,ll,fl;
   Integer i;

   fl = NULL;
   ll = NULL;

   do {
      if (fgets(lbuf,10240,dff) == NULL) return NULL;
    }
   while (lbuf[0] != '>');

   for (i = n1; i <= n2; ++i) {
      l = PALLOC(LINE_INFO);
      l->num = i;
      l->text = SALLOC(&lbuf[2]);
      l->next = NULL;
      if (ll == NULL) fl = l;
      else ll->next = l;
      ll = l;
      if (i != n2) {
	 if (fgets(lbuf,10240,dff) == NULL) break;
       };
    };

   return fl;
};





/************************************************************************/
/*									*/
/*	difference -- compare two lines 				*/
/*									*/
/************************************************************************/


#define COST(a,b) cost[((a)+1)*ncol+(b)+1]

static Integer
difference(s1,s2)
   String s1;
   String s2;
{
   Integer * cost;
   Integer nrow,ncol;
   Integer i,j,k;

   nrow = strlen(s1)+1;
   ncol = strlen(s2)+1;
   cost = (Integer *) alloca(sizeof(Integer)*nrow*ncol);

   COST(-1,-1) = 0;

   for (i = 0; i < ncol; ++i) {
      k = nrow;
      if (i < nrow) {
	 for (j = -1; j < i; ++j) {
	    compute_cost(cost,i,j,nrow,ncol,s1,s2);
	  };
	 k = i;
       };
      for (j = -1; j < k; ++j) {
	 compute_cost(cost,j,i,nrow,ncol,s1,s2);
       };
    };

   i = COST(nrow-1,ncol-1);

   if (i >= ncol/2) i = -1;

   return i;
};





/************************************************************************/
/*									*/
/*	compute_cost -- get cost element  for comparison		*/
/*									*/
/************************************************************************/


static void
compute_cost(cost,i,j,nrow,ncol,s1,s2)
   Integer * cost;
   Integer i,j;
   Integer nrow,ncol;
   String s1,s2;
{
   Integer c,c1;

   if (j >= 0) {
      c = COST(i,j-1);
      if (!isspace(s2[j])) c+= INSERT_COST;
    }
   else c = (nrow+ncol+1)*MAX_COST;

   if (i >= 0) {
      c1 = COST(i-1,j);
      if (!isspace(s1[i])) c1 += DELETE_COST;
      if (c1 < c) c = c1;
      if (j >= 0) {
	 if (s1[i] == s2[j]) c1 = COST(i-1,j-1);
	 else c1 = EDIT_COST+COST(i-1,j-1);
	 if (c1 < c) c = c1;
       };
    };

   COST(i,j) = c;
};





/************************************************************************/
/*									*/
/*	handle_search  -- find matching lines that might have moved	*/
/*									*/
/************************************************************************/


static void
handle_search(all,fil)
   ANNOT all;
   String fil;
{
   ANNOT a;
   Integer ct;
   FILE * inf;
   Integer lin;
   Character lbuf[10240];
   Integer c;

   ct = 0;

   for (a = all; a != NULL; a = a->next) {
      if (!a->found && a->dline >= 0 && a->sourceline != NULL) {
	 if (a->sourceline[0] == 0 || a->sourceline[0] == '\n') {
	    a->found = TRUE;
	    a->line = a->dline;
	  }
	 else {
	    ++ct;
	    a->cost = strlen(a->sourceline)*MAX_COST;
	  };
       };
    };

   if (ct == 0) return;

   inf = fopen(fil,"r");
   if (inf != NULL) {
      lin = 0;
      while (fgets(lbuf,10240,inf) != NULL) {
	 ++lin;
	 if (lbuf[0] != '\n') {
	    for (a = all; a != NULL; a = a->next) {
	       if (!a->found && a->dline >= 0) {
		  c = difference(lbuf,a->sourceline);
		  if (c == 0) {
		     a->line = lin;
		     a->found = TRUE;
		     --ct;
		   }
		  else if (c > 0 && c < a->cost) {
		     a->dline = lin;
		     a->cost = c;
		   };
		};
	     };
	    if (ct == 0) break;
	  };
       };
      fclose(inf);
    };

   for (a = all; a != NULL; a = a->next) {
      if  (!a->found && a->dline >= 0) {
	 a->line = a->dline;
	 a->found = TRUE;
       };
    };
};





/* end of fixannot.c */
