/************************************************************************/
/*									*/
/*		annotact.c						*/
/*									*/
/*	Action routines associated with annotations			*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include "annot_local.h"





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		annot_refresh();
static	void		refresh_annot();
static	Boolean 	add_annotation();
static	void		setup_state_win();
static	Boolean 	state_update();
static	Boolean 	state_pdm();
static	void		show_annot_info();
static	Boolean 	get_annot_texts();
static	void		resend_annot();





/************************************************************************/
/*									*/
/*	ANNOT_act_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
ANNOT_act_init()
{
};





/************************************************************************/
/*									*/
/*	ANNOTact_search_for -- serach for next annotation (ext call)	*/
/*									*/
/************************************************************************/


int
ANNOTact_search_for(aw,what)
   ANNOT_WIN aw;
   String what;
{
   Character buf[1024];
   Boolean fg;

   if (aw == NULL) return FALSE;
   if (!ASHinq_valid_window(aw->edit_win)) return FALSE;
   if (!ASHinq_valid_window(aw->window)) return FALSE;

   sprintf(buf,"Search for %s",what);

   fg = ANNOT_act_search_for(aw,"USER",buf);

   return fg;
};





/************************************************************************/
/*									*/
/*	ANNOT_annot_setup -- setup annotation portion of display	*/
/*									*/
/************************************************************************/


void
ANNOT_annot_setup(aw)
   ANNOT_WIN aw;
{
   if (aw->annot_win == NULL) return;

   ASHset_region_refresh(aw->annot_win,annot_refresh);

   BIOnew_input_window(aw->annot_win);
   BIOset_cursor_standard(aw->annot_win,ASH_CURSOR_SMALL_X);

   refresh_annot(aw);

   if (aw->state_win != NULL || aw->state_pdm != STATE_NONE) {
      setup_state_win(aw);
    };
};





/************************************************************************/
/*									*/
/*	ANNOT_annot_btn -- handle Hits in annotation window		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_annot_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ANNOT_WIN aw;
   ANNOT_DESC ad;
   Integer line;
   ANNOTATION an;
   static Integer xtrinfo;
   Boolean fg;

   aw = (ANNOT_WIN) RIPinq_data(rgn);

   if (aw->filename == NULL) return FALSE;

   ASHmap(ASHinq_top_window(aw->annot_win),x,y,aw->annot_win,&x,&y);

   if (btn & RIP_BTN_TAP) xtrinfo = 0;
   else if (btn & RIP_BTN_DOWN) {
      xtrinfo = btn & RIP_BTN_EXTRA;
    }
   else if (btn & RIP_BTN_UP) btn |= xtrinfo;

   line = ANNOT_get_line(aw,y);
   if (line < 0) return FALSE;

   if (btn & RIP_BTN_NONE) return FALSE;
   if (btn & RIP_BTN_DOWN) return FALSE;

   fg = TRUE;

   if (btn & RIP_BTN_MID) {
      an = ANNOT_find(aw,line,NULL);
      if (an == NULL) fg = FALSE;
      else show_annot_info(an);
    }
   else {
      if (btn & RIP_BTN_RIGHT) ad = aw->alt_annot;
      else ad = aw->cur_annot;

      if (btn & RIP_BTN_EXTRA) an = NULL;
      else an = ANNOT_find(aw,line,ad);

      if (aw->alt_annot == aw->cur_annot && an != NULL &&
	     (btn & RIP_BTN_RIGHT) != 0 && !an->desc->unique) an = NULL;

      if (an == NULL) {
	 fg = add_annotation(aw,ad,line,0);
       }
      else {
	 ANNOT_user_remove(an,FALSE);
       };
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_search -- handle SEARCH button			*/
/*	ANNOT_act_search_for -- handle SEARCH for buttons		*/
/*	ANNOT_act_search_again -- handle SEARCH AGAIN button		*/
/*									*/
/************************************************************************/


Integer
ANNOT_act_search(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   String menu[64];
   Integer ct,i,tct;
   Integer typ,dir,start;
   ANNOT_DESC annots[32];
   Character annotd[32][64];

   if (aw->filename == NULL) return FALSE;

   tct = ANNOT_inq_types(aw,32,annots);

   if (tct == 0) return FALSE;

   ct = 0;
   menu[ct++] = "%CAnnotation Search\n";

   if (tct > 1) {
      for (i = 0; i < tct; ++i) {
	 sprintf(annotd[i],"%%0.%do %s",i,annots[i]->name);
	 menu[ct++] = annotd[i];
       };
      menu[ct++] = "";
    };

   menu[ct++] = "%1.0o Forward%M%1.1o Backward";
   menu[ct++] = "%2o Start of File";
   menu[ct++] = "   %a%M   %c";
   menu[ct] = 0;

   typ = 0;
   dir = 0;
   start = 0;

   i = STEMdialog(aw->window,menu,&typ,&dir,&start);

   if (i) {
      dir = (dir ? -1 : 1);
      if (start) start = 0;
      else start = EDTcur_line(aw->edit_info);

      aw->search_type = annots[typ];
      aw->search_dir = dir;
      aw->search_line = start;

      i = ANNOT_act_search_again(aw,mnm,btn);
    };

   return i;
};




/*ARGSUSED*/

Integer
ANNOT_act_search_for(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   ANNOTATION an;
   Integer i,tct;
   ANNOT_DESC annots[32];
   String s;

   if (aw->filename == NULL) return FALSE;

   tct = ANNOT_inq_types(aw,32,annots);

   if (tct == 0) return FALSE;

   s = rindex(btn,' ');
   if (s == NULL) return FALSE;
   ++s;

   for (i = 0; i < tct; ++i) {
      if (STREQL(s,annots[i]->name)) break;
    };

   if (i >= tct) return FALSE;

   aw->search_type = annots[i];
   aw->search_dir = 1;

   an = ANNOT_search(aw,aw->search_type,aw->search_line,aw->search_dir);

   if (an == NULL && aw->search_line != 0) {
      aw->search_line = 0;
      an = ANNOT_search(aw,aw->search_type,aw->search_line,aw->search_dir);
    };

   if (an == NULL) {
      aw->search_line = 0;
      aw->search_dir = 1;
      return FALSE;
    }
   else {
      i = ANNOT_inq_line(an);
      aw->search_line = i;
      EDTgoto(aw->edit_info,i);
    };

   return TRUE;
};






/*ARGSUSED*/

Integer
ANNOT_act_search_again(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   ANNOTATION an;
   Integer i;

   an = ANNOT_search(aw,aw->search_type,aw->search_line,aw->search_dir);

   if (an == NULL) {
      aw->search_line = 0;
      aw->search_dir = 1;
      return FALSE;
    }
   else {
      i = ANNOT_inq_line(an);
      aw->search_line = i;
      EDTgoto(aw->edit_info,i);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_remove -- handle REMOVE_ALL button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_act_remove(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   Integer r[10];
   ANNOT_DESC desc[10];
   String menu[24];
   Integer ct,tct;
   Integer i;
   Character data[10][64];

   if (aw->filename == NULL) return FALSE;

   tct = ANNOT_inq_types(aw,10,desc);

   if (tct == 0) return FALSE;

   if (tct == 1) r[0] = TRUE;
   else {
      ct = 0;
      menu[ct++] = "%CRemove Annotations\n";
      for (i = 0; i < tct; ++i) {
	 sprintf(data[i],"%%%do %s",i,desc[i]->name);
	 r[i] = FALSE;
	 menu[ct++] = data[i];
       };
      menu[ct++] = "   %a%M   %c";
      menu[ct] = 0;
      if (!STEMdialog(aw->window,menu,&r[0],&r[1],&r[2],&r[3],&r[4],&r[5],
			 &r[6],&r[7],&r[8],&r[9])) {
	 return FALSE;
       };
    };

   for (i = 0; i < tct; ++i) {
      if (r[i]) ANNOT_remove_all(aw,desc[i]);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_rotate -- handle ROTATE button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_act_rotate(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   ANNOTATION an;

   if (aw->filename == NULL) return FALSE;

   an = ANNOT_find(aw,EDTcur_line(aw->edit_info),NULL);

   if (an != NULL) ANNOT_rotate(an);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_monitor -- handle MONITOR button			*/
/*									*/
/************************************************************************/


#define BITS_PER	30
#define WORDS		4
#define ITEMS		(BITS_PER*WORDS)


/*ARGSUSED*/

Integer
ANNOT_act_monitor(aw,mnm,btn)
   ANNOT_WIN aw;
   String mnm;
   String btn;
{
   Integer ct;
   ANNOT_DESC ads[ITEMS];
   Character menu[10240],line[1024];
   Integer vals[WORDS],svals[WORDS],i,j,k,l;
   Sequence l1,l2;

   if (aw->filename == NULL) return FALSE;

   ct = ANNOT_inq_types(NULL,ITEMS,ads);

   sprintf(menu,"%%CAnnotations for Window\n\n");

   strcat(menu,"Use  Monitor  Annotation\n");

   for (i = 0; i < WORDS; ++i) vals[i] = svals[i] = 0;

   for (i = 0; i < ct; ++i) {
      j = (i % BITS_PER);
      k = (i / BITS_PER);
      l = (1 << j);
      sprintf(line,"%%%d.%df  %%%d.%df     %s\n", k*2,l,k*2+1,l,ads[i]->name);
      strcat(menu,line);
      if (MEMQ(ads[i],aw->allowed)) vals[k] |= l;
      if (MEMQ(ads[i],aw->set_on)) svals[k] |= l;
    };

   strcat(menu,"\n\n   %a%M   %c");

   if (!STEMdialog1(aw->window,menu,
		       &vals[0],&svals[0],&vals[1],&svals[1],
		       &vals[2],&svals[2],&vals[3],&svals[3])) {
      return FALSE;
    };

   l1 = NULL;
   l2 = NULL;
   for (i = 0; i < ct; ++i) {
      j = (i % BITS_PER);
      k = (i / BITS_PER);
      l = (1 << j);
      if (vals[k]&l) l1 = APPEND(ads[i],l1);
      if (svals[k]&l) l2 = APPEND(ads[i],l2);
    };

   LFREE(aw->allowed);
   LFREE(aw->set_on);
   aw->allowed = l1;
   aw->set_on = l2;

   setup_state_win(aw);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_display -- handle chainging display info		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_act_display(aw,mnm,bnm)
   ANNOT_WIN aw;
   String mnm;
   String bnm;
{
   Character menu[10240];
   Integer lvl;

   strcpy(menu,"%CAnnotation Display Settings\n\nDisplay Level: %0d\n\n   %a%M   %c");

   lvl = aw->level;

   if (!STEMdialog1(aw->window,menu,&lvl)) {
      return FALSE;
    };

   if (aw->level != lvl) {
      aw->level = lvl;
      ANNOT_update(aw,TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_act_update -- handle UPDATE requests			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_act_update(aw,mnm,bnm)
   ANNOT_WIN aw;
   String mnm;
   String bnm;
{
   ANNOT_iterate(aw,ANNOT_send_update,NULL);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	ANNOT_edit_mouse -- handle special mouse buttons		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

Integer
ANNOT_edit_mouse(aw,btn,l,c,fl,fc,tl,tc)
   ANNOT_WIN aw;
   Integer btn;
   Integer l,c;
   Integer fl,fc;
   Integer tl,tc;
{
   Boolean fg;
   ANNOT_DESC ad;

   if (aw->filename == NULL) return FALSE;

   if ((btn & (RIP_BTN_SHIFT|RIP_BTN_META)) == 0) return FALSE;

   if (btn & RIP_BTN_RIGHT) ad = aw->alt_annot;
   else ad = aw->cur_annot;

   fg = add_annotation(aw,ad,l,c);

   return fg;
};





/************************************************************************/
/*									*/
/*	ANNOT_do_add -- actually add an annotation			*/
/*									*/
/************************************************************************/


void
ANNOT_do_add(aw,ad,line,col,buf0,buf1,buf2)
   ANNOT_WIN aw;
   ANNOT_DESC ad;
   Integer line;
   Integer col;
   String buf0,buf1,buf2;
{
   ANNOTATION an;
   String txts[3];
   Character buf[1024];
   Integer i;

   if (ad->display) {
      an = ANNOT_add(aw,ad,line,col);
      txts[0] = ((buf0 == NULL || buf0[0] == 0) ? NULL : buf0);
      txts[1] = ((buf1 == NULL || buf1[0] == 0) ? NULL : buf1);
      txts[2] = ((buf2 == NULL || buf2[0] == 0) ? NULL : buf2);
      for (i = 0; i < 3; ++i) {
	 if (an != NULL && txts[i] != NULL) {
	    if (an->text[i] != NULL) {
	       if (ad->accumulate) {
		  if (strlen(an->text[i]) + strlen(txts[i]) < 1024)
		     sprintf(buf,"%s\n%s",an->text[i],txts[i]);
		  else
		     strcpy(buf,an->text[i]);
		  txts[i] = buf;
		};
	       SFREE(an->text[i]);
	     };
	    an->text[i] = SALLOC(txts[i]);
	  };
       };
    };

   ANNOT_send_add(aw,ad,line,col,buf0,buf1,buf2);
};





/************************************************************************/
/*									*/
/*	annot_refresh -- ASH entry for refresh				*/
/*	refresh_annot -- refresh window routine 			*/
/*									*/
/************************************************************************/


static void
annot_refresh(w)
   ASH_WINDOW w;
{
   ANNOT_WIN aw;

   aw = ANNOT_find_window(w);

   if (aw != NULL) refresh_annot(aw);
};




static void
refresh_annot(aw)
   ANNOT_WIN aw;
{
   ANNOT_update(aw,TRUE);
};





/************************************************************************/
/*									*/
/*	add_annotation -- add annotation				*/
/*									*/
/************************************************************************/


static Boolean
add_annotation(aw,ad,line,ch)
   ANNOT_WIN aw;
   ANNOT_DESC ad;
   Integer line;
   Integer ch;
{
   String txt;
   Character buf0[1024],buf1[1024],buf2[1024];
   Boolean fg;

   if (ad == NULL) return FALSE;

   if (ch == 0) {
      if (line == EDTcur_line(aw->edit_info)) ch = EDTcur_char(aw->edit_info);
      else ch = 1;
    };

   txt = NULL;
   switch (ad->position) {
      default :
      case POS_LINE_BEGIN :
	 ch = 1;
	 break;
      case POS_TEXT_BEGIN :
	 txt = EDTinq_line_text(aw->edit_info,line,NULL);
	 ch = 1;
	 while (isspace(txt[ch])) ++ch;
	 if (txt[ch] == 0) ch = 1;
	 break;
      case POS_LINE_END :
	 txt = EDTinq_line_text(aw->edit_info,line,NULL);
	 ch = strlen(txt);
	 break;
      case POS_USER_POS :
	 break;
      case POS_WORD_BEGIN :
	 txt = EDTinq_line_text(aw->edit_info,line,NULL);
	 ch -= 1;
	 while (ch > 0 && (isalnum(txt[ch]) || txt[ch] == '_' || txt[ch] == '$'))
	    --ch;
	 ch += 1;
	 break;
      case POS_SEMICOLON :
	 txt = EDTinq_line_text(aw->edit_info,line,NULL);
	 ch -= 1;
	 while (txt[ch] != 0 && txt[ch] != ';') ++ch;
	 if (txt[ch] == ';') ++ch;
	 break;
    }

   if (txt != NULL) free(txt);

   fg = get_annot_texts(aw,NULL,ad,line,buf0,buf1,buf2);

   if (!fg) return FALSE;

   ANNOT_do_add(aw,ad,line,ch,buf0,buf1,buf2);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	setup_state_win -- setup state window options			*/
/*									*/
/************************************************************************/


static void
setup_state_win(aw)
   ANNOT_WIN aw;
{
   Character buf[64][64];
   String menu[80];
   Integer ct;
   ANNOT_DESC typ[64];
   ANNOT_DESC ad;
   Integer i,mct,fst,scd;

   if (aw->state_pdm == STATE_NONE) return;

   ct = ANNOT_inq_types(aw,64,typ);

   if (aw->state_setup) {
      if (aw->state_pdm == STATE_MENU) STEMstate_remove(aw->state_win);
      else {
	 for (i = 0; i < ct; ++i) {
	    STEMpdm_btn_remove(aw->menu_win,SELECTION_MENU,typ[i]->name);
	  };
       };
      aw->state_setup = FALSE;
    };

   mct = 0;
   menu[mct++] = "%CAnnotations\n";

   fst = -1;
   scd = -1;
   for (i = 0; i < ct; ++i) {
      ad = typ[i];
      if (!ad->private) {
	 if (fst < 0) fst = i;
	 else if (scd < 0) scd = i;
	 if (aw->cur_annot == ad) fst = i;
	 if (aw->alt_annot == ad) scd = i;
	 if (aw->state_pdm == STATE_MENU) {
	    sprintf(buf[i],"%%C%%0.%d'%s'o",i,ad->name);
	    menu[mct++] = buf[i];
	  }
	 else {
	    STEMpdm_btn_add(aw->menu_win,SELECTION_MENU,ad->name,state_pdm);
	    STEMpdm_btn_select(aw->menu_win,SELECTION_MENU,ad->name,(fst == i));
	  };
       };
    };

   menu[mct] = 0;

   if (aw->state_pdm == STATE_MENU) STEMstate(aw->state_win,menu,state_update,NULL,aw);
   aw->state_setup = TRUE;

   if (fst >= 0) {
      aw->cur_annot = typ[fst];
      if (aw->state_pdm == STATE_MENU) STEMstate_set(aw->state_win,0,fst);
      if (scd >= 0) aw->alt_annot = typ[scd];
      else aw->alt_annot = aw->cur_annot;
    }
   else {
      aw->cur_annot = NULL;
      aw->alt_annot = NULL;
    };
};




/************************************************************************/
/*									*/
/*	state_update -- handle changes to state menu			*/
/*	state_pdm -- handle button requests on pdm state menu		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Boolean
state_update(aw,num,id,btn,win)
   ANNOT_WIN aw;
   Integer num;
   Integer id;
   Integer btn;
   ASH_WINDOW win;
{
   Integer ct;
   ANNOT_DESC typ[64];
   Boolean fg;

   ct = ANNOT_inq_types(aw,64,typ);
   fg = FALSE;

   if (btn & RIP_BTN_RIGHT) {
      if (id < 0 || id > ct) aw->alt_annot = aw->cur_annot;
      else aw->alt_annot = typ[id];
    }
   else {
      if (id < 0 || id > ct) aw->cur_annot = NULL;
      else aw->cur_annot = typ[id];
      fg = TRUE;
    };

   return fg;
};





/*ARGSUSED*/

static Boolean
state_pdm(aw,mnm,bnm,w)
   ANNOT_WIN aw;
   String mnm;
   String bnm;
   ASH_WINDOW w;
{
   ANNOT_DESC typ[64],oty;
   Integer ct,i,id,oid;

   ct = ANNOT_inq_types(aw,64,typ);

   id = -1;
   oid = -1;
   for (i = 0; i < ct; ++i) {
      if (STREQL(bnm,typ[i]->name)) id = i;
      if (aw->cur_annot != NULL && STREQL(typ[i]->name,aw->cur_annot->name)) oid = i;
    };

   oty = aw->cur_annot;
   if (id < 0 || id > ct) aw->cur_annot = NULL;
   else aw->cur_annot = typ[id];
   if (aw->cur_annot == aw->alt_annot) aw->alt_annot = oty;

   if (aw->state_pdm == STATE_PDM) {
      if (oid >= 0) STEMpdm_btn_select(aw->menu_win,SELECTION_MENU,typ[oid]->name,FALSE);
      if (id >= 0) STEMpdm_btn_select(aw->menu_win,SELECTION_MENU,typ[id]->name,TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	show_annot_info -- display information about an annotation	*/
/*									*/
/************************************************************************/


static void
show_annot_info(an)
   ANNOTATION an;
{
   Character buf0[1024],buf1[1024],buf2[1024];

   get_annot_texts(an->window,an,an->desc,an->line,buf0,buf1,buf2);
};





/************************************************************************/
/*									*/
/*	get_annot_texts -- get text for annotation from user		*/
/*									*/
/************************************************************************/


static Boolean
get_annot_texts(aw,an,ad,l,buf0,buf1,buf2)
   ANNOT_WIN aw;
   ANNOTATION an;
   ANNOT_DESC ad;
   Integer l;
   String buf0,buf1,buf2;
{
   Integer i,j;
   Character menu[40960],line[10240];
   Integer fg;
   String txts[3];
   String s;

   if (ad == NULL && an != NULL) ad = an->desc;
   if (an != NULL) ++an->frozen;

   txts[0] = buf0;
   txts[1] = buf1;
   txts[2] = buf2;

   if (!ad->query && an == NULL) return TRUE;

   sprintf(menu,"%%C%s at line %d of %s\n\n",ad->name,l,aw->filename);

   buf0[0] = 0;
   buf1[0] = 0;
   buf2[0] = 0;

   j = 0;
   for (i = 0; i < 3; ++i) {
      if (an == NULL && ad->textinfo[i] == NULL) continue;
      if (ad->query && ad->textinfo[i] != NULL) {
	 sprintf(line,"%s: %%%d.64t\n",ad->textinfo[i],i);
	 if (an != NULL && an->text[i] != NULL) {
	    strncpy(txts[i],an->text[i],64);
	    txts[i][63] = 0;
	  };
       }
      else if (ad->textinfo[i] == NULL && an->text[i] != NULL) {
	 sprintf(line,"%s\n",an->text[i]);
       }
      else if (an->text[i] != NULL) {
	 sprintf(line,"%s: %s\n",ad->textinfo[i],an->text[i]);
       }
      else continue;

      if (strlen(line) >= 512) {
	 s = index(&line[512],'/n');
	 if (s != NULL) *++s = 0;
       };
      strcat(menu,line);
      ++j;
    };

   if (j == 0 && an == NULL) return TRUE;

   if (an != NULL) {
      strcat(menu,"\n   %3.1o Remove\n");
      if (!an->desc->private) strcat(menu,"   %3.2o Resend\n");
      fg = FALSE;
    };

   strcat(menu,"\n   %a%M   %c\n");

   if (!STEMdialog1(aw->window,menu,buf0,buf1,buf2,&fg)) return FALSE;

   if (an != NULL) {
      if (an->frozen < 0) return FALSE;
      if (an->frozen > 0) --an->frozen;
    };

   for (i = 0; i < 3; ++i) {
      if (ad->textinfo[i] == NULL) continue;

      if (an != NULL) {
	 if (an->text[i] == NULL || STRNEQ(an->text[i],txts[i])) {
	    if (an->text[i] != NULL) SFREE(an->text[i]);
	    an->text[i] = SALLOC(txts[i]);
	  };
       };
    };

   if (an != NULL && fg == 1) {
      ANNOT_user_remove(an,TRUE);
    }
   else if (an != NULL && fg == 2) {
      resend_annot(an);
    };

   return TRUE;
};




/************************************************************************/
/*									*/
/*	resend_annot -- handle resending an annotation on request	*/
/*									*/
/************************************************************************/


static void
resend_annot(an)
   ANNOTATION an;
{
   ANNOT_WIN aw;
   ANNOT_DESC ad;
   Integer line,col;
   String buf0,buf1,buf2;

   aw = an->window;
   ad = an->desc;
   line = EDTmark_line(aw->edit_info,an->mark);
   col = EDTmark_char(aw->edit_info,an->mark);
   buf0 = an->text[0];
   buf1 = an->text[1];
   buf2 = an->text[2];

   ANNOT_user_remove(an,TRUE);

   ANNOT_do_add(aw,ad,line,col,buf0,buf1,buf2);
};






/* end of annotact.c */





