/************************************************************************/
/*									*/
/*		dynload.c						*/
/*									*/
/*	This is a machine-independent (sort-of) dynamic loader.  It	*/
/*	operates by loading most of the file together, getting the	*/
/*	names of all symbols, creating a symbol table for later use	*/
/*	and creating linkage segments for undefined symbols.		*/
/*									*/
/************************************************************************/
/*	Copyright 1987 Brown University -- Steven P. Reiss		*/



#include "dl_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TEMP_NAME	"dldXXXXXX"
#define SYMBOLS_NAME	"./dlsym.out"
#define SED_NAME	"/tmp/dlsXXXXXX"

#define INIT_SYM_SIZE	1024
#define INIT_NAM_SIZE	10240

#define NAME_MAP_COUNT	1024




/************************************************************************/
/*									*/
/*	Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _SYM_INFO {
   Integer index;
   Boolean undef;
   Boolean text;
} SYM_INFO;



typedef struct _MAP_INFO {
   String from;
   String to;
} MAP_INFO;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	SYM_INFO *	symtbl;
static	Integer 	symtbl_size;
static	Integer 	symcnt;
static	Integer 	udfcnt;

static	Character *	namtbl;
static	Integer 	namtbl_size;
static	Integer 	namcnt;

static	String		libtbl[10];
static	Integer 	libcnt;

static	Integer 	stub_value;
static	Character	stub_code[64];
static	Integer 	stub_padding;

static	Boolean 	brownfg;

static	MAP_INFO *	name_maps;
static	Integer 	name_map_max = 0;
static	Integer 	name_map_ct = 0;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		get_symbols();
static	void		add_symbol();
static	Boolean 	ignore_symbol();
static	Boolean 	skip_symbol();
static	void		make_file();
static	void		set_name_map();
static	void		do_name_map();





/************************************************************************/
/*									*/
/*	Main Program							*/
/*									*/
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   register String outnm;
   Character cmd[2048],cmd0[2048];
   Character tmp[64],sym[64],symo[64],syms[64],symt[64],symx[64],dyld[64];
   register Boolean debug,cdbg;
   register FILE * nmf;
   register Integer i;
   String dynld,tempdir;
   Boolean useddl;
   String s,a;

   debug = FALSE;

   tempdir = (String) getenv("DYNTMP");
   if (tempdir == NULL) tempdir = "/tmp";

   sprintf(tmp,"%s/%s",tempdir,TEMP_NAME);
   mktemp(tmp);

   strcpy(symo,SYMBOLS_NAME);
   strcpy(sym,symo);
   strcpy(syms,symo);
   strcpy(symt,symo);
   strcpy(symx,symo);
   strcat(symo,".o");
   strcat(sym,".c");
   strcat(syms,".s");
   strcat(symt,".map.s");
   strcat(symx,".stub.s");

   outnm = "a.out";
   cmd0[0] = 0;
   libcnt = 0;
   stub_value = 0;
   cdbg = FALSE;
   brownfg = FALSE;
   useddl = FALSE;

   DL_x_lib_name(dyld);
   dynld = dyld;

   for (i = 1; i < argc; ++i) {
      if (DL_x_check_arg(argv[i])) continue;
      if (STREQL(argv[i],"-r")) ;
      else if (STREQL(argv[i],"-g")) cdbg = TRUE;
      else if (STREQL(argv[i],"-ldl") || STREQL(argv[i],dyld)) {
	 strcat(cmd0," ");
	 strcat(cmd0,argv[i]);
	 useddl = TRUE;
       }
      else if (STREQL(argv[i],"-o")) {
	 ++i;
	 outnm = argv[i];
       }
      else if (STREQL(argv[i],"-XDEBUG")) debug = TRUE;
      else if (STREQL(argv[i],"-LOCAL")) dynld = "dynlib.o";
      else if (STREQL(argv[i],"-INLIB")) {
	 libtbl[libcnt++] = argv[++i];
       }
      else if (STREQL(argv[i],"-BROWN")) brownfg = TRUE;
      else {
	 strcat(cmd0," ");
	 strcat(cmd0,argv[i]);
       };
    };

   DL_x_check_arg(NULL);

   if (!useddl) {
      strcat(cmd0," ");
      strcat(cmd0,dynld);
    };

   sprintf(cmd,"cc -o %s -r %s",tmp,cmd0);

   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   system(cmd);

   DL_x_nm_command(cmd,tmp,FALSE);

   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   nmf = (FILE *) popen(cmd,"r");

   get_symbols(nmf);

   pclose(nmf);

   make_file(sym,symx);

   s = "-S";

   DL_x_cc_command(cmd,s,sym);

   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   i = system(cmd);

#ifdef mips
   if (i != 0) {
      DL_x_cc_command(cmd,"-S -g",sym);
      i = system(cmd);
    };
#endif

   sprintf(cmd,"cat %s >> %s",symx,syms);
   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   system(cmd);

   if (name_map_ct > 0) {
      do_name_map(syms,symt);
      DL_x_as_command(cmd,symt,symo);
    }
   else {
      DL_x_as_command(cmd,syms,symo);
    };

   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   system(cmd);

   DL_x_ld_command(cmd,outnm,tmp,symo,cdbg,cmd0);

   if (debug) fprintf(stderr,"DYNLOAD: %s\n",cmd);
   system(cmd);

   if (!debug) {
      unlink(tmp);
      unlink(sym);
      unlink(symo);
      unlink(syms);
      unlink(symt);
      unlink(symx);
    };

   exit(0);
};





/************************************************************************/
/*									*/
/*	get_symbols -- read in symbol table				*/
/*									*/
/************************************************************************/


static void
get_symbols(nmf)
   FILE * nmf;
{
   Character buf[1024];
   String p;
   unsigned v;
   Integer ct;
   Boolean undef,txt;
   Integer dlta;

   symtbl_size = INIT_SYM_SIZE;
   symtbl = (SYM_INFO *) calloc(sizeof(SYM_INFO),symtbl_size);
   symcnt = 0;
   udfcnt = 0;
   namtbl_size = INIT_NAM_SIZE;
   namtbl = (char *) calloc(1,namtbl_size);
   namcnt = 0;

   while (fgets(buf,1024,nmf) != NULL) {
      buf[strlen(buf)-1] = 0;

      undef = DL_x_nm_read(buf,&p,&v,TRUE,&txt);
      if (p == NULL) continue;

      if (ignore_symbol(p)) continue;

      if (skip_symbol(p,undef,v)) undef = FALSE;

      add_symbol(p,undef,txt);
    };

   while ((p = DL_x_other_syms()) != NULL) {
      add_symbol(p,FALSE,FALSE);
    };
};




static void
add_symbol(p,undef,text)
   String p;
   Boolean undef;
   Boolean text;
{
   Integer ct;

   if (symcnt+1 >= symtbl_size) {
      symtbl_size *= 2;
      symtbl = (SYM_INFO *) realloc(symtbl,sizeof(SYM_INFO)*symtbl_size);
    };
   symtbl[symcnt].undef = undef;
   symtbl[symcnt].text = text;
   symtbl[symcnt].index = namcnt;
   if (symtbl[symcnt].undef) ++udfcnt;
   ++symcnt;

   ct = strlen(p);
   if (namcnt + ct + 1 >= namtbl_size) {
      namtbl_size *= 2;
      namtbl = (Character *) realloc(namtbl,namtbl_size);
    };
   strcpy(&namtbl[namcnt],p);
   namcnt += ct+1;
   namtbl[namcnt-1] = 0;
};





/************************************************************************/
/*									*/
/*	ignore_symbol -- check if symbol should be ignored		*/
/*									*/
/************************************************************************/


static Boolean
ignore_symbol(nm)
   String nm;
{
   register String p;

   if (DL_x_ignore_symbol(nm)) return TRUE;

   if (brownfg) {
      if (DL_x_prepend_()) {
	 if (nm[0] != '_') return FALSE;
	 p = &nm[1];
       }
      else p = nm;

      if (isupper(*p)) {
	 for (++p; isupper(*p); ++p);
	 if (*p++ == '_' && islower(*p)) return TRUE;
       };
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	skip_symbol -- check if symbol should be skipped		*/
/*									*/
/************************************************************************/


static Boolean
skip_symbol(nm,udf,val)
   String nm;
   Boolean udf;
   Integer val;
{
   if (DL_x_prepend_()) ++nm;

   if (STREQL(nm,"DL_stub")) {
      stub_value = val;
    };

   if (STREQL(nm,"end")) return TRUE;
   if (STREQL(nm,"etext")) return TRUE;
   if (STREQL(nm,"edata")) return TRUE;
   if (STREQL(nm,"DL__esd")) return TRUE;
   if (STREQL(nm,"DL__library_count")) return TRUE;
   if (STREQL(nm,"DL__librarys")) return TRUE;
   if (STREQL(nm,"DL__stub_branch_table")) return TRUE;
   if (STREQL(nm,"DL_stub")) return TRUE;
   if (STREQL(nm,"DL_ext_stub")) return TRUE;

   return FALSE;
};




/************************************************************************/
/*									*/
/*	make_file -- produce our result file				*/
/*									*/
/************************************************************************/


static void
make_file(flnm,asnm)
   String flnm;
   String asnm;
{
   register FILE * otf;
   FILE * asf;
   Integer lnksize;
   register Integer i,j,k,udf,di;
   register String nm,lnk0,typ,typ0,nm0;
   Integer * snum;
   Character buf[64];

   otf = fopen(flnm,"w");
   asf = fopen(asnm,"w");

   while ((namcnt&3) != 0) namtbl[namcnt++] = 0;

   fprintf(otf,"typedef int (*dynload_function_type)();\n\n");

   fprintf(otf,"struct dnlist {\n");
   fprintf(otf,"   char * dn_name;\n");
   fprintf(otf,"   char * dn_value;\n");
   fprintf(otf,"   char   dn_stub;\n");
   fprintf(otf,"   char   dn_undef;\n");
   fprintf(otf,"   short  dn_dummy;\n");
   fprintf(otf,"};\n\n");

   fprintf(otf,"struct _DL_esd_s {\n");
   fprintf(otf,"   int num_syms;\n");
   fprintf(otf,"   int num_linkage;\n");
   fprintf(otf,"   struct dnlist sym[%d];\n",symcnt);
   fprintf(otf,"};\n\n");

   fprintf(otf,"struct _DL_link_info {\n");
   fprintf(otf,"   dynload_function_type addr;\n");
   fprintf(otf,"   int symid;\n};\n\n");

   fprintf(otf,"extern struct _DL_esd_s * DL__esd;\n");
   fprintf(otf,"extern struct _DL_link_info DL__stub_branch_table[];\n");
   fprintf(otf,"extern int DL_stub();\n");
   fprintf(otf,"extern int DL_ext_stub();\n");

   fprintf(otf,"\tint\tDL__library_count = %d;\n\n",libcnt);
   fprintf(otf,"\tchar * DL__librarys[] = {\n");
   for (i = 0; i < libcnt; ++i) {
      fprintf(otf,"\t\"%s\",\n",libtbl[i]);
    };
   fprintf(otf,"\t0 };\n\n");

   for (i = 0; i < symcnt; ++i) {
      nm = &namtbl[symtbl[i].index];
      if (DL_x_prepend_()) ++nm;
      if ((DL_x_prepend_() && namtbl[symtbl[i].index] != '_') || nm[0] == '.' ) {
	 nm = &namtbl[symtbl[i].index];
	 sprintf(buf,"_DLSYM_sym_%d_",i);
	 set_name_map(buf,nm);
	 nm = &buf[1];
       };

      typ0 = "";
      typ = "int";
      if (STREQL(nm,"DL__esd")) continue;
      else if (STREQL(nm,"DL__library_count")) continue;
      else if (STREQL(nm,"DL__librarys")) continue;
      else if (STREQL(nm,"DL__stub_branch_table")) continue;
      else if (STREQL(nm,"DL_stub")) continue;
      else if (STREQL(nm,"DL_ext_stub")) continue;
      else if (symtbl[i].undef) typ = "struct _DL__linkage";
      else if (symtbl[i].text) typ0 = "()";

      fprintf(otf,"extern\t%s\t%s%s;\n",typ,nm,typ0);
    };

   fprintf(otf,"\nstatic struct _DL_esd_s DL_esd_data = {\n");
   fprintf(otf,"\t%d, %d, \n{\n",symcnt,udfcnt);

   for (i = 0; i < symcnt; ++i) {
      nm = &namtbl[symtbl[i].index];
      if (DL_x_prepend_()) ++nm;
      if ((DL_x_prepend_() && namtbl[symtbl[i].index] != '_') || nm[0] == '.' ) {
	 sprintf(buf,"DLSYM_sym_%d_",i);
	 nm = buf;
       }
      nm0 = &namtbl[symtbl[i].index];
      udf = (symtbl[i].undef ? 1 : 0);
      if (STREQL(nm,"DL__librarys")) typ = "";
      else if (STREQL(nm,"DL_stub")) typ = "";
      else if (STREQL(nm,"DL_ext_stub")) typ = "";
      else if (STREQL(nm,"DL__stub_branch_table")) typ = "";
      else if (symtbl[i].text) typ = "";
      else typ = "&";
      fprintf(otf,"\t{ \"%s\", ((char *) %s%s), %d, %d},\n",nm0,typ,nm,udf,udf);
    };

   fprintf(otf,"} };\n\n");
   fprintf(otf,"\tstruct _DL_esd_s * DL__esd = &DL_esd_data;\n\n");

   if (udfcnt != 0) {
      fprintf(stderr,"Linkage segments created for:\n");
    };

   fprintf(otf,"struct _DL_link_info DL__stub_branch_table[%d] = {\n",udfcnt+1);
   DL_x_gen_stub(asf,NULL,0,0);

   lnk0 = NULL;

#ifdef DATA_STORED_BACKWARD
   i = symcnt-1;
   di = -1;
#else
   i = 0;
   di = 1;
#endif

   udf = 0;
   for (k = 0; k < symcnt; ++k) {
      if (symtbl[i].undef) {
	 nm = &namtbl[symtbl[i].index];
	 if (DL_x_prepend_()) ++nm;
	 fprintf(stderr,"   %s\n",nm);

	 nm = &namtbl[symtbl[i].index];

	 fprintf(otf,"   { DL_stub, %d},\t/* %s */\n",i,nm);

	 DL_x_gen_stub(asf,nm,udf);
	 ++udf;

	 if (di < 0) lnk0 = nm;
	 else if (lnk0 == NULL) lnk0 = nm;
       };
      i += di;
    };


   fprintf(otf,"   { DL_stub, -1},\t/* Dynamic Undefineds */\n");
   fprintf(otf,"};\n\n");
   DL_x_gen_stub(asf,NULL,udf+1);

   fclose(otf);
   fclose(asf);
};





/************************************************************************/
/*									*/
/*	set_name_map -- add a name mapping				*/
/*									*/
/************************************************************************/


static void
set_name_map(from,to)
   String from;
   String to;
{
   if (name_map_max == 0) {
      name_map_max = NAME_MAP_COUNT;
      name_maps = (MAP_INFO *) malloc(sizeof(MAP_INFO)*name_map_max);
      name_map_ct = 0;
    };

   if (name_map_ct+1 >= name_map_max) {
      name_map_max *= 2;
      name_maps = (MAP_INFO *) realloc(sizeof(MAP_INFO)*name_map_max);
    };

   name_maps[name_map_ct].from = SALLOC(from);
   name_maps[name_map_ct].to = SALLOC(to);
   name_map_ct++;
};





/********************************************************************************/
/*										*/
/*	do_name_map -- perform name mapping					*/
/*										*/
/********************************************************************************/


static void
do_name_map(fromf,tof)
   String fromf;
   String tof;
{
   FILE * inf;
   FILE * otf;
   Character lbuf[1024];
   register String p,q;
   String s;
   Integer i;

   inf = fopen(fromf,"r");
   otf = fopen(tof,"w");
   if (inf == NULL || otf == NULL) {
      printf("problem opening file for name mapping\n");
      exit(1);
    };

   while (fgets(lbuf,1024,inf) != NULL) {
      s = lbuf;
      while (*s != 0) {
	 if (s[0] == '_' && s[1] == 'D' && s[2] == 'L' && s[3] == 'S') {
	    for (i = 0; i < name_map_ct; ++i) {
	       p = s;
	       for (q = name_maps[i].from; *q != 0; ++q)
		  if (*p++ != *q) break;
	       if (*q == 0) {
		  s = p;
		  q = name_maps[i].to;
		  while (*q != 0) putc(*q++,otf);
		  break;
		};
	     };
	  };
	 putc(*s++,otf);
       };
    };

   fclose(inf);
   fclose(otf);
};





/************************************************************************/
/*									*/
/*	Misc definitions						*/
/*									*/
/*	DL_pro_dir -- get project directory				*/
/*	DL_arch -- get architecture name				*/
/*									*/
/************************************************************************/


DL_FunctionPtr
DL_stub_routine()
{
   return NULL;
};






char *
DL_pro_dir()
{
   String s;

   s = (String) getenv("DL_DIR");
   if (s == NULL) s = (String) getenv("PRO");

#ifdef DL_DIR
   if (s == NULL) s = DL_DIR;
#endif

   return s;
};






char *
DL_arch()
{
   String s;

   s = (String) getenv("DL_ARCH");
   if (s == NULL) s = (String) getenv("ARCH");

#ifdef DL_ARCH
   if (s == NULL) s = DL_ARCH;
#endif

   return s;
};







/* end of dynload.c */
