
/************************************************************************/
/*									*/
/*		dlvax.c 						*/
/*									*/
/*	Routines and definitions for VAX dynamic loading		*/
/*									*/
/************************************************************************/


#include "dl_local.h"

#include <ar.h>
#include <a.out.h>
#include <sys/file.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define AR_SYMDEF	"__.SYMDEF"

#ifndef N_DATOFF
#define N_DATOFF(x) (N_TXTOFF(x) + (x).a_text)
#endif
#ifndef N_TRELOFF
#define N_TRELOFF(x) (N_DATOFF(x) + (x).a_data)
#endif
#ifndef N_DRELOFF
#define N_DRELOFF(x) (N_TRELOFF(x) + (x).a_trsize)
#endif






/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		setup_segment();
static	void		setup_reloc();
static	void		setup_symbols();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String	search_libs[32] = {
   "/lib", "/usr/lib", "/usr/local/lib", 0
};

static	Integer search_ct = -3;






/************************************************************************/
/*									*/
/*	DL_x_prepend_ -- indicate if underscore prepended to names	*/
/*									*/
/************************************************************************/


Boolean
DL_x_prepend_()
{
   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_test_archive -- test for archive file			*/
/*	DL_x_next_archive -- return offset for next archive element	*/
/*									*/
/************************************************************************/


Boolean
DL_x_test_archive(fi,offp)
   Integer fi;
   Integer * offp;
{
   Character magbuf[SARMAG];
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,0,L_SET);
   read(fi,magbuf,SARMAG);
   if (strncmp(magbuf,ARMAG,SARMAG) != 0) return FALSE;

   *offp = SARMAG + sizeof(hdr);

   read(fi,&hdr,sizeof(hdr));
   if (STREQL(hdr.ar_name,AR_SYMDEF)) {
      *offp += sizeof(hdr);
      len = atoi(hdr.ar_size);
      *offp += ((len+1) & ~1);
    };

   return TRUE;
};





Boolean
DL_x_next_archive(fi,ename,offp)
   Integer fi;
   String ename;
   Integer *offp;
{
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,*offp-sizeof(hdr),L_SET);
   if (read(fi,&hdr,sizeof(hdr)) != sizeof(hdr)) return FALSE;

   *offp += sizeof(hdr);
   len = atoi(hdr.ar_size);
   *offp += ((len+1) & ~1);

   strncpy(ename,hdr.ar_name,sizeof(hdr.ar_name));

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_setup_buffers -- read in data from a file			*/
/*									*/
/************************************************************************/


Boolean
DL_x_setup_buffers(dlf,fi,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer aoff;
{
   struct exec fhdr;
   long int stlen;

   lseek(fi,aoff,L_SET);
   read(fi,&fhdr,sizeof(fhdr));
   if (N_BADMAG(fhdr)) return FALSE;

   dlf->numseg = 3;
   setup_segment(dlf,0,fi,fhdr.a_text,aoff+N_TXTOFF(fhdr),0);
   setup_segment(dlf,1,fi,fhdr.a_data,aoff+N_DATOFF(fhdr),fhdr.a_text);
   setup_segment(dlf,2,-1,fhdr.a_bss,-1,fhdr.a_text+fhdr.a_data);

   dlf->common = NULL;
   dlf->basereg = NULL;

   setup_reloc(dlf,0,fi,fhdr.a_trsize,aoff+N_TRELOFF(fhdr));
   setup_reloc(dlf,1,fi,fhdr.a_drsize,aoff+N_DRELOFF(fhdr));

   setup_symbols(dlf,fi,fhdr.a_syms,aoff+N_SYMOFF(fhdr),aoff+N_STROFF(fhdr));

   return TRUE;
};



static void
setup_segment(dlf,seg,fi,len,addr,off)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
   Integer off;
{
   Address rgn;

   if (len > 0) {
      rgn = (Address) malloc(len);
      if (fi >= 0) {
	 lseek(fi,addr,L_SET);
	 read(fi,rgn,len);
       }
      else {
	 bzero(rgn,len);
       };
    }
   else rgn = NULL;

   dlf->segs[seg].base = rgn;
   dlf->segs[seg].reloc = NULL;
   dlf->segs[seg].offset = off;
   dlf->segs[seg].poffset = 0;
   dlf->segs[seg].reloc_size = 0;
   dlf->segs[seg].length = len;
}





static void
setup_reloc(dlf,seg,fi,len,addr)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
{
   Address xrgn;
   DL_RELOC rel;
   Integer ct,i;
   struct relocation_info * rp;

   if (len <= 0) return;

   ct = len/sizeof(struct relocation_info);

   rel = (DL_RELOC) calloc(ct,sizeof(DL_RELOC_INFO));
   xrgn = (Address) alloca(len);

   lseek(fi,addr,L_SET);
   read(fi,xrgn,len);

   for (i = 0; i < ct; ++i) {
      rp = (struct relocation_info *) (xrgn + sizeof(struct relocation_info)*i);
      rel[i].dr_address = rp->r_address;
      rel[i].dr_symbol = rp->r_extern;
      if (rel[i].dr_symbol) rel[i].dr_index = rp->r_symbolnum;
      else switch (rp->r_symbolnum & N_TYPE) {
	 case N_TEXT :
	    rel[i].dr_index = 0;
	    break;
	 case N_DATA :
	    rel[i].dr_index = 1;
	    break;
	 case N_BSS :
	    rel[i].dr_index = 2;
	    break;
	 case N_COMM :
	    rel[i].dr_index = 3;
	    break;
	 default :
	    rel[i].dr_index = SEGMENT_ABS;
	    break;
       };
      rel[i].dr_value = 0;
      switch (rp->r_length) {
	 case 0 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP8 : RTYPE_8);
	    break;
	 case 1 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP16 : RTYPE_16);
	    break;
	 case 2 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP32 : RTYPE_32);
	    break;
       };
    };

   dlf->segs[seg].reloc = rel;
   dlf->segs[seg].reloc_size = ct;
};





static void
setup_symbols(dlf,fi,len,addr,straddr)
   DL_FILE dlf;
   Integer fi;
   Integer len;
   Integer addr;
   Integer straddr;
{
   struct nlist * sp;
   Address xbuf;
   DL_ESYM sym;
   Integer ct,stlen,i;

   ct = len / sizeof(struct nlist);

   if (ct == 0) {
      dlf->numsym = 0;
      dlf->strings = NULL;
      dlf->syms = NULL;
      return;
    };

   xbuf = (Address) alloca(len);
   lseek(fi,addr,L_SET);
   read(fi,xbuf,len);

   lseek(fi,straddr,L_SET);
   read(fi,&stlen,4);
   dlf->strings = (String) malloc(stlen);
   lseek(fi,straddr,L_SET);
   read(fi,dlf->strings,stlen);

   sym = (DL_ESYM) calloc(sizeof(DL_ESYM_INFO),ct);
   for (i = 0; i < ct; ++i) {
      sp = (struct nlist *) (xbuf + sizeof(struct nlist)*i);
      sym[i].ds_name = &dlf->strings[sp->n_un.n_strx];
      sym[i].ds_value = sp->n_value;
      sym[i].ds_extern = (sp->n_type & N_EXT) != 0;
      if ((sp->n_type & N_STAB) != 0 || (sp->n_type & N_TYPE) == N_FN)
	 sym[i].ds_segment = SEGMENT_IGNORE;
      else switch (sp->n_type & N_TYPE) {
	 case N_UNDF :
	    sym[i].ds_segment = SEGMENT_UNDEF;
	    break;
	 case N_ABS :
	    sym[i].ds_segment = SEGMENT_ABS;
	    break;
	 case N_TEXT :
	    sym[i].ds_segment = 0;
	    break;
	 case N_DATA :
	    sym[i].ds_segment = 1;
	    break;
	 case N_BSS :
	    sym[i].ds_segment = 2;
	    break;
	 case N_COMM :
	    sym[i].ds_segment = SEGMENT_COMMON;
	    break;
       };
    };

   dlf->numsym = ct;
   dlf->syms = sym;
};





/************************************************************************/
/*									*/
/*	DL_x_machine_reloc -- handle machine relocation types		*/
/*									*/
/************************************************************************/


void
DL_x_machine_reloc(id,rty,ptr,tp,vp)
   Integer id;
   RELOC_TYPE rty;
   Character * ptr;
   Integer * tp;
   Integer * vp;
{
   /* NOT USED */
};





/************************************************************************/
/*									*/
/*	DL_x_swap -- ns32000 byte swapping stubs			*/
/*									*/
/************************************************************************/


Integer
DL_x_swap(x)
   Integer x;
{
   return x;
};




/************************************************************************/
/*									*/
/*	DL_x_nm_command -- generate nm command on a file		*/
/*	DL_x_cc_command -- generate cc command on a file		*/
/*	DL_x_ld_command -- generate ld command on 2 files		*/
/*	DL_x_as_command -- generate as command on 2 files		*/
/*									*/
/************************************************************************/


void
DL_x_nm_command(buf,file,tblfg)
   String buf;
   String file;
   Boolean tblfg;		/* TRUE if for building tables		*/
				/* FALSE if for dynload symbols 	*/
{
   sprintf(buf,"nm -pg %s",file);
};





void
DL_x_cc_command(buf,opts,file)
   String buf;
   String opts;
   String file;
{
   sprintf(buf,"cc %s %s",opts,file);
};





void
DL_x_ld_command(buf,outfile,f1,f2,dbg,args)
   String buf;
   String outfile;
   String f1;
   String f2;
   Boolean dbg;
   String args;
{
   sprintf(buf,"ld -o %s %s %s%s",outfile,f1,f2,(dbg ? " -lg" : ""));
};






void
DL_x_as_command(buf,src,tgt)
   String buf;
   String src;
   String tgt;
{
   sprintf(buf,"as -o %s %s",tgt,src);
};




/************************************************************************/
/*									*/
/*	DL_x_nm_read -- read a nm line -- return true if undefined sym	*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_read(buf,np,vp,dfg,txtp)
   String buf;
   String * np;
   Integer * vp;
   Boolean dfg;
   Boolean * txtp;
{
   String p;
   Character t;
   Integer v,dlta;
   Boolean fg,tfg;

   if (buf[0] == '0' && buf[1] == 'x') dlta = 2;
   else dlta = 0;

   p = &buf[dlta];

   v = 0;
   while (*p != ' ') {
      v = v*16;
      if (isdigit(*p)) v += *p - '0';
      else if (islower(*p)) v += *p - 'a' + 10;
      else if (isupper(*p)) v += *p - 'A' + 10;
      else break;
      ++p;
    };
   *vp = v;

   while (*p == ' ') ++p;

   t = *p++;
   while (*p == ' ') ++p;

   *np = p;
   while (*p != 0 && *p != ' ' && *p != '\n') ++p;
   *p = 0;

   fg = (t == 'U' && v == 0);
   tfg = (t == 'T' || t == 't');

   if (txtp != NULL) *txtp = tfg;

   return fg;
};






/********************************************************************************/
/*										*/
/*	DL_x_other_syms -- extra symbols					*/
/*										*/
/********************************************************************************/


String
DL_x_other_syms()
{
   return NULL;
};





/************************************************************************/
/*									*/
/*	DL_x_nm_element -- check for new element in archive		*/
/*	DL_x_nm_extdata -- check for external data definition		*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_element(buf,elt)
   String buf;
   String elt;
{
   Integer i;

   i = strlen(buf);
   if (buf[i-1] == '\n') --i;
   if (buf[i-1] == ':') {
      strcpy(elt,buf);
      elt[i-1] = 0;
      return TRUE;
    };

   return FALSE;
};





Boolean
DL_x_nm_extdata(buf)
   String buf;
{
   if (buf[0] == ' ') return FALSE;
   while (*buf != ' ') ++buf;
   while (*buf == ' ') ++buf;

   return (*buf == 'D' || *buf == 'T');
};





/************************************************************************/
/*									*/
/*	DL_x_ignore_symbol -- indicate if symbol should be ignored	*/
/*									*/
/************************************************************************/


Boolean
DL_x_ignore_symbol(nm)
   String nm;
{
   if (nm[0] != '_') return TRUE;

   return FALSE;
};





/********************************************************************************/
/*										*/
/*	DL_x_lib_name -- get library name					*/
/*										*/
/********************************************************************************/


void
DL_x_lib_name(buf)
   char * buf;
{
   String s;
   String a;
   Integer i;

   s = DL_pro_dir();
   a = DL_arch();

   for (i = search_ct-1; i >= 0; --i) {
      sprintf(buf,"%s/libdl.a",search_libs[i]);
      if (access(buf,4) >= 0) {
	 sprintf(buf,"-ldl");
	 return;
       };
      if (a != NULL) {
	 sprintf(buf,"%s/libdl_%s.a",search_libs[i],a);
	 if (access(buf,4) >= 0) {
	    sprintf(buf,"-ldl");
	    return;
	  };
       }
    };

   if (getenv("DL_LIBRARY") != NULL) {
      strcpy(buf,getenv("DL_LIBRARY"));
      return;
    };

   buf[0] = 0;
   if (s != NULL && a != NULL) {
      sprintf(buf,"%s/lib/dynlib.%s.o",s,a);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && s != NULL) {
      sprintf(buf,"%s/lib/dynlib.o",s);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && a != NULL) {
      sprintf(buf,"/usr/local/lib/dynlib.%s.o",a);
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.%s.o",a);
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.%s.o",a);
	    if (access(buf,4) < 0) buf[0] = 0;
	  };
       };
    }

   if (buf[0] == 0) {
      sprintf(buf,"/usr/local/lib/dynlib.o");
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.o");
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.o");
	    if (access(buf,4) < 0) sprintf(buf,"/usr/lib/dynlib.o");
	  };
       };
    };
};






/********************************************************************************/
/*										*/
/*	DL_x_check_arg -- special processing of arguments			*/
/*										*/
/********************************************************************************/


Boolean
DL_x_check_arg(arg)
   String arg;
{
   return FALSE;
};





/************************************************************************/
/*									*/
/*	DL_x_gen_stub -- generate assembler code for a stub		*/
/*									*/
/************************************************************************/


Boolean
DL_x_gen_stub(otf,nm,idx)
   FILE * otf;
   String nm;
   Integer idx;
{
   if (nm == NULL) {
      if (idx == 0) {
	 fprintf(otf,"   .text\n");
       }
      else {
	 fprintf(otf,"DL_stub_handler:\n");
	 fprintf(otf,"   movl _DL__stub_branch_table(r0),r1\n");
	 fprintf(otf,"   jmp 2(r1)\n\n");
	 fprintf(otf,"   .globl _DL_stub_routine\n");
	 fprintf(otf,"   .globl _DL_stub\n");
	 fprintf(otf,"   .globl _DL_ext_stub\n");
	 fprintf(otf,"_DL_ext_stub:\n");
	 fprintf(otf,"   .word 0xffc\n");
	 fprintf(otf,"   movl $%d,r0\n",(idx-1)*sizeof(DL_LINK_INFO));
	 fprintf(otf,"   jmp _DL_stub+2\n");
	 fprintf(otf,"_DL_stub:\n");
	 fprintf(otf,"   .word 0xffc\n");
	 fprintf(otf,"   pushl   r0\n");
	 fprintf(otf,"   subl3 $4,16(fp),r0\n");
	 fprintf(otf,"   pushl   r0\n");
	 fprintf(otf,"   calls   $2,_DL_stub_routine\n");
	 fprintf(otf,"   jmp     2(r0)\n");
       };
      return;
    };

   fprintf(otf,"  .globl %s\n",nm);
   fprintf(otf,"%s:\n",nm);
   fprintf(otf,"   .word 0xffc\n");
   fprintf(otf,"   movl $%d,r0\n",idx*sizeof(DL_LINK_INFO));
   fprintf(otf,"   jbr DL_stub_handler\n");

   fprintf(otf,"\n");
};







/************************************************************************/
/*									*/
/*	dummy routine to force loading of needed things 		*/
/*									*/
/************************************************************************/


static void
dummy_routine()
{
   int a,b,c;
   float d,e,f;
   double g,h,i;

   a = b*c;
   b = a%c;
   strcmp(0,1); 		/* force strcmp to be available */
   d = d+e*f;
   d = d/f-e;
   a = d;
   e = b;
   g = g*h+i-g/h;
   e = g;
   h = e;
   b = g;
   h = a;
   vfork(a,b,c,d,e,f);
};






/* end of dlvax.c */
