/************************************************************************/
/*									*/
/*		dlsymtab.c						*/
/*									*/
/*	Internal symbol table routines for dynamic loading		*/
/*									*/
/************************************************************************/
/*	Copyright 1987 Brown University -- Steven P. Reiss		*/


#include "dl_local.h"




/************************************************************************/
/*									*/
/*	Local data structures -- symbol table				*/
/*									*/
/************************************************************************/

/************************************************************************/
/*									*/
/* Stolen largely from the BSD4.2 loader.				*/
/* All internal data structures are segmented and dynamically extended. */
/* The basic structures hold 1103 (NSYM) symbols.			*/
/* For large programs and/or modules, these structures			*/
/* expand to be up to 40 (NSEG) times as large as this as necessary.	*/
/*									*/
/************************************************************************/

#define NSEG	40		/* Number of segments, each data structure */
#define NSYM	1103		/* Number of symbols per segment */

/************************************************************************/
/*									*/
/* Structure describing each symbol table segment.			*/
/* Each segment has its own hash table.  We record the first		*/
/* address in and first address beyond both the symbol and hash 	*/
/* tables, for use in the routine symx and the lookup routine		*/
/* respectively 							*/
/*									*/
/************************************************************************/

static struct	symseg {
   DL_ISYM	first_sym;	/* base of this alloc'ed segment */
   DL_ISYM	last_sym;	/* end of this segment */
   int	   num_syms;		/* symbols used in this seg */
   DL_ISYM *	hfirst; 	/* base of hash table, this seg */
   DL_ISYM *	hlast;		/* end of hash table, this seg */
} symseg[NSEG+1];

static int num_symbols = 0;	/* number of symbols in the table */


/************************************************************************/
/*									*/
/* The lookup routine uses quadratic rehash.  Since a quadratic rehash	*/
/* only probes 1/2 of the buckets in the table, and since the hash	*/
/* table is segmented the same way the symbol table is, we make the	*/
/* hash table have twice as many buckets as there are symbol table slots*/
/* in the segment.  This guarantees that the quadratic rehash will never*/
/* fail to find an empty bucket if the segment is not full and the	*/
/* symbol is not there. 						*/
/*									*/
/************************************************************************/


#define HSIZE	(NSYM*2)






/************************************************************************/
/*									*/
/*	DL_lookup -- find symbol in our tables				*/
/*									*/
/************************************************************************/


DL_ISYM
DL_lookup(name, loc)
   char *name;
   DL_HashKey *loc;
{
   Integer hash_val;
   DL_ISYM * hash_point;
   String cp,cp1;
   struct symseg *gp;
   Integer i;
   DL_HashKey dummy;

   if (loc == NULL) loc = &dummy;

   hash_val = 0;
   for (cp = name; *cp;) hash_val = (hash_val<<1) + *cp++;
   hash_val = (hash_val & 0x7fffffff) % HSIZE;

   for (gp = symseg; gp < &symseg[NSEG]; gp++) {
      if (gp->first_sym == 0) {
	 gp->first_sym = (DL_ISYM) calloc(NSYM, sizeof (DL_ISYM_INFO));
	 gp->hfirst = (DL_ISYM *) calloc(HSIZE, sizeof (DL_ISYM));
	 if (gp->first_sym == 0 || gp->hfirst == 0)
	    DL_error("DL_lookup", "ran out of space for symbol table");
	 gp->last_sym = gp->first_sym + NSYM;
	 gp->hlast = gp->hfirst + HSIZE;
       };
      hash_point = gp->hfirst + hash_val;
      i = 1;
      do {
	 if (*hash_point == 0) {
	    if (gp->num_syms == NSYM) break;
	    *loc = hash_point;
	    return NULL;
	  };
	 cp1 = (*hash_point)->dn_name;
	 for (cp = name; *cp == *cp1++;) {
	    if (*cp++ == 0) {
	       *loc = hash_point;
	       return (*hash_point);
	     }
	  };
	 hash_point += i;
	 i += 2;
	 if (hash_point >= gp->hlast) hash_point -= HSIZE;
       }
      while (i < HSIZE);
      if (i > HSIZE)
	 DL_error("DL_lookup", "hash table botch");
    };

   DL_error("DL_lookup", "symbol table overflow");

   return NULL;
}




/************************************************************************/
/*									*/
/*	DL_nameof -- return symbol name given address			*/
/*									*/
/************************************************************************/


String
DL_nameof(function)
   Integer function;
{
   DL_ISYM *h;
   struct symseg *gp;

   for (gp = symseg; gp < &symseg[NSEG]; gp++) {
      if (gp->first_sym == 0) return NULL;
      h = gp->hfirst;
      for (h = gp->hfirst; h < gp->hlast; h++) {
	 if (*h != NULL) {
	    if ((**h).dn_value == function)
	       return (**h).dn_name;
	  };
       };
    };

   return NULL;
}





/************************************************************************/
/*									*/
/*	DL_enter -- add symbol to table (after lookup)			*/
/*									*/
/* DL_enter should be used with the value set in 'loc' in the lookup    */
/* function. A symbol should be looked up before it is entered. 	*/
/* (Otherwise you will get garbage!)					*/
/*									*/
/************************************************************************/


Boolean
DL_enter(sp,hash_point,replace)
   DL_ISYM sp;
   DL_HashKey hash_point;
   int replace;
{
   struct symseg *cur_symseg;

   if (*hash_point==0) {
      cur_symseg = &symseg[num_symbols/NSYM];
      if (hash_point < cur_symseg->hfirst || hash_point >= cur_symseg->hlast)
	 DL_error("DL_enter", "enter botch");
      *hash_point = cur_symseg->first_sym + cur_symseg->num_syms;
      **hash_point = *sp;
      cur_symseg->num_syms++;
      num_symbols++;
      return TRUE;
    }
   else if (replace) {
      **hash_point = *sp;
      return TRUE;
    };

   return FALSE;
}





/************************************************************************/
/*									*/
/*	DL_symbol_at -- convert symbol index into pointer		*/
/*	DL_symbol_index -- convert symbol pointer into index		*/
/*									*/
/* DL_symbol_at converts symbol table indices (ala x) into symbol table */
/* pointers. symbol_index (harder, but never used in loops) inverts	*/
/* pointers into the symbol table into indices using the symseg[]	*/
/* structure.								*/
/*									*/
/************************************************************************/


DL_ISYM
DL_symbol_at(x)
   int x;
{
   if (x <= num_symbols)
      return (symseg[(x)/NSYM].first_sym+((x)%NSYM));

   return NULL;
}




Integer
DL_symbol_index(sp)
   DL_ISYM sp;
{
   register struct symseg *gp;

   if (sp == 0) return 0;

   for (gp = symseg; gp->first_sym != NULL; gp++) {
      if (sp >= gp->first_sym && sp < gp->last_sym)
	 return ((gp - symseg) * NSYM + sp - gp->first_sym);
    };

   DL_error("DL_symbol_index", "symbol_index botch");

   return 0;
}





/* end of dlsymtab.c */
