/************************************************************************/
/*									*/
/*		dlsun.c 						*/
/*									*/
/*	Routines and definitions for SUN dynamic loading		*/
/*									*/
/************************************************************************/


#include "dl_local.h"

#include <ar.h>
#include <a.out.h>
#include <sys/types.h>
#include <sys/file.h>

#ifndef NO_SHARED_LIBS
#include <dirent.h>
#include <link.h>
#endif




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define AR_SYMDEF	"__.SYMDEF"


#ifdef sparc
#undef relocation_info
#define relocation_info reloc_info_sparc
#endif


#ifndef N_DATOFF
#define N_DATOFF(x) (N_TXTOFF(x) + (x).a_text)
#endif
#ifndef N_TRELOFF
#define N_TRELOFF(x) (N_DATOFF(x) + (x).a_data)
#endif
#ifndef N_DRELOFF
#define N_DRELOFF(x) (N_TRELOFF(x) + (x).a_trsize)
#endif




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		setup_segment();
static	void		setup_reloc();
static	void		setup_symbols();
static	int		name_comp();
static	Boolean 	test_dynamic();
static	Boolean 	add_shared_lib();
static	void		get_shared_libs();
static	void		get_shared_syms();
static	void		add_shared_syms();
static	void		share_search();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


#define DYN_SYM_SIZE	4096


static	String	search_libs[64] = {
   "/lib", "/usr/lib", "/usr/local/lib", 0
};

static	Integer search_ct = -3;


static	String	shared_libs[256] = {
   0
};

static	Integer shared_ct = 0;




static	String *	dyn_names = NULL;
static	Integer 	dyn_name_ct = 0;
static	Integer 	dyn_name_max = 0;
static	Boolean 	dyn_name_sort = FALSE;





/************************************************************************/
/*									*/
/*	DL_x_prepend_ -- indicate if underscore prepended to names	*/
/*									*/
/************************************************************************/


Boolean
DL_x_prepend_()
{
   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_test_archive -- test for archive file			*/
/*	DL_x_next_archive -- return offset for next archive element	*/
/*									*/
/************************************************************************/


Boolean
DL_x_test_archive(fi,offp)
   Integer fi;
   Integer * offp;
{
   Character magbuf[SARMAG];
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,0,L_SET);
   read(fi,magbuf,SARMAG);
   if (strncmp(magbuf,ARMAG,SARMAG) != 0) return FALSE;

   *offp = SARMAG + sizeof(hdr);

   read(fi,&hdr,sizeof(hdr));
   if (STREQL(hdr.ar_name,AR_SYMDEF)) {
      *offp += sizeof(hdr);
      len = atoi(hdr.ar_size);
      *offp += ((len+1) & ~1);
    };

   return TRUE;
};





Boolean
DL_x_next_archive(fi,ename,offp)
   Integer fi;
   String ename;
   Integer *offp;
{
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,*offp-sizeof(hdr),L_SET);
   if (read(fi,&hdr,sizeof(hdr)) != sizeof(hdr)) return FALSE;

   *offp += sizeof(hdr);
   len = atoi(hdr.ar_size);
   *offp += ((len+1) & ~1);

   strncpy(ename,hdr.ar_name,sizeof(hdr.ar_name));

   /* mjn: get rid of sun's trailing space */

   while (!isspace(*ename) && *ename != '\0') ename++;
   *ename = '\0';

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_setup_buffers -- read in data from a file			*/
/*									*/
/************************************************************************/


Boolean
DL_x_setup_buffers(dlf,fi,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer aoff;
{
   struct exec fhdr;
   long int stlen;

   lseek(fi,aoff,L_SET);
   read(fi,&fhdr,sizeof(fhdr));
   if (N_BADMAG(fhdr)) return FALSE;

   dlf->numseg = 3;
   setup_segment(dlf,0,fi,fhdr.a_text,aoff+N_TXTOFF(fhdr),0);
   setup_segment(dlf,1,fi,fhdr.a_data,aoff+N_DATOFF(fhdr),fhdr.a_text);
   setup_segment(dlf,2,-1,fhdr.a_bss,-1,fhdr.a_text+fhdr.a_data);

   dlf->common = NULL;
   dlf->basereg = NULL;

   setup_reloc(dlf,0,fi,fhdr.a_trsize,aoff+N_TRELOFF(fhdr));
   setup_reloc(dlf,1,fi,fhdr.a_drsize,aoff+N_DRELOFF(fhdr));

   setup_symbols(dlf,fi,fhdr.a_syms,aoff+N_SYMOFF(fhdr),aoff+N_STROFF(fhdr));

   return TRUE;
};



static void
setup_segment(dlf,seg,fi,len,addr,off)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
   Integer off;
{
   Address rgn;

   if (len > 0) {
      rgn = (Address) malloc(len);
      if (fi >= 0) {
	 lseek(fi,addr,L_SET);
	 read(fi,rgn,len);
       }
      else {
	 bzero(rgn,len);
       };
    }
   else rgn = NULL;

   dlf->segs[seg].base = rgn;
   dlf->segs[seg].reloc = NULL;
   dlf->segs[seg].offset = off;
   dlf->segs[seg].poffset = 0;
   dlf->segs[seg].reloc_size = 0;
   dlf->segs[seg].length = len;
}





static void
setup_reloc(dlf,seg,fi,len,addr)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
{
   Address xrgn;
   DL_RELOC rel;
   Integer ct,i;
   struct relocation_info * rp;

   if (len <= 0) return;

   ct = len/sizeof(struct relocation_info);

   rel = (DL_RELOC) calloc(ct,sizeof(DL_RELOC_INFO));
   xrgn = (Address) alloca(len);

   lseek(fi,addr,L_SET);
   read(fi,xrgn,len);

   for (i = 0; i < ct; ++i) {
      rp = (struct relocation_info *) (xrgn + sizeof(struct relocation_info)*i);
      rel[i].dr_address = rp->r_address;
      rel[i].dr_symbol = rp->r_extern;
#ifdef sparc
      if (rel[i].dr_symbol) rel[i].dr_index = rp->r_index;
      else switch (rp->r_index & N_TYPE) {
#else
      if (rel[i].dr_symbol) rel[i].dr_index = rp->r_symbolnum;
      else switch (rp->r_symbolnum & N_TYPE) {
#endif
	 case N_TEXT :
	    rel[i].dr_index = 0;
	    break;
	 case N_DATA :
	    rel[i].dr_index = 1;
	    break;
	 case N_BSS :
	    rel[i].dr_index = 2;
	    break;
	 case N_COMM :
	    rel[i].dr_index = 3;
	    break;
	 default :
	    rel[i].dr_index = SEGMENT_ABS;
	    break;
       };
#ifdef sparc
      rel[i].dr_value = rp->r_addend;
      switch (rp->r_type) {
	 case RELOC_8 :
	    rel[i].dr_type = RTYPE_8;
	    break;
	 case RELOC_16 :
	    rel[i].dr_type = RTYPE_16;
	    break;
	 case RELOC_32 :
	    rel[i].dr_type = RTYPE_32;
	    break;
	 case RELOC_DISP8 :
	    rel[i].dr_type = RTYPE_DISP8;
	    break;
	 case RELOC_DISP16 :
	    rel[i].dr_type = RTYPE_DISP16;
	    break;
	 case RELOC_DISP32 :
	    rel[i].dr_type = RTYPE_DISP32;
	    break;
	 case RELOC_WDISP30 :
	    rel[i].dr_type = RTYPE_WDISP30;
	    break;
	 case RELOC_WDISP22 :
	    rel[i].dr_type = RTYPE_WDISP22;
	    break;
	 case RELOC_HI22 :
	    rel[i].dr_type = RTYPE_HI22;
	    break;
	 case RELOC_22 :
	    rel[i].dr_type = RTYPE_22;
	    break;
	 case RELOC_13 :
	    rel[i].dr_type = RTYPE_13;
	    break;
	 case RELOC_LO10 :
	    rel[i].dr_type = RTYPE_LO10;
	    break;
	 case RELOC_SFA_BASE :
	    rel[i].dr_type = RTYPE_SFA_BASE;
	    break;
	 case RELOC_SFA_OFF13 :
	    rel[i].dr_type = RTYPE_SFA_OFF13;
	    break;
	 case RELOC_SEGOFF16 :
	    rel[i].dr_type = RTYPE_SEGOFF16;
	    break;
       };
#else
      rel[i].dr_value = 0;
      switch (rp->r_length) {
	 case 0 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP8 : RTYPE_8);
	    break;
	 case 1 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP16 : RTYPE_16);
	    break;
	 case 2 :
	    rel[i].dr_type = (rp->r_pcrel ? RTYPE_DISP32 : RTYPE_32);
	    break;
       };
#endif
    };

   dlf->segs[seg].reloc = rel;
   dlf->segs[seg].reloc_size = ct;
};





static void
setup_symbols(dlf,fi,len,addr,straddr)
   DL_FILE dlf;
   Integer fi;
   Integer len;
   Integer addr;
   Integer straddr;
{
   struct nlist * sp;
   Address xbuf;
   DL_ESYM sym;
   Integer ct,stlen,i;

   ct = len / sizeof(struct nlist);

   if (ct == 0) {
      dlf->numsym = 0;
      dlf->strings = NULL;
      dlf->syms = NULL;
      return;
    };

   xbuf = (Address) alloca(len);
   lseek(fi,addr,L_SET);
   read(fi,xbuf,len);

   lseek(fi,straddr,L_SET);
   read(fi,&stlen,4);
   dlf->strings = (String) malloc(stlen);
   lseek(fi,straddr,L_SET);
   read(fi,dlf->strings,stlen);

   sym = (DL_ESYM) calloc(sizeof(DL_ESYM_INFO),ct);
   for (i = 0; i < ct; ++i) {
      sp = (struct nlist *) (xbuf + sizeof(struct nlist)*i);
      sym[i].ds_name = &dlf->strings[sp->n_un.n_strx];
      sym[i].ds_value = sp->n_value;
      sym[i].ds_extern = (sp->n_type & N_EXT) != 0;
      if ((sp->n_type & N_STAB) != 0 || (sp->n_type & N_TYPE) == N_FN)
	 sym[i].ds_segment = SEGMENT_IGNORE;
      else switch (sp->n_type & N_TYPE) {
	 case N_UNDF :
	    if (sp->n_value == 0) sym[i].ds_segment = SEGMENT_UNDEF;
	    else sym[i].ds_segment = SEGMENT_COMMON;
	    break;
	 case N_ABS :
	    sym[i].ds_segment = SEGMENT_ABS;
	    break;
	 case N_TEXT :
	    sym[i].ds_segment = 0;
	    break;
	 case N_DATA :
	    sym[i].ds_segment = 1;
	    break;
	 case N_BSS :
	    sym[i].ds_segment = 2;
	    break;
	 case N_COMM :
	    sym[i].ds_segment = SEGMENT_COMMON;
	    break;
       };
    };

   dlf->numsym = ct;
   dlf->syms = sym;
};





/************************************************************************/
/*									*/
/*	DL_x_machine_reloc -- handle machine relocation types		*/
/*									*/
/************************************************************************/


void
DL_x_machine_reloc(id,rty,ptr,tp,vp)
   Integer id;
   RELOC_TYPE rty;
   Character * ptr;
   Integer * tp;
   Integer * vp;
{
   /* NOT USED */
};





/************************************************************************/
/*									*/
/*	DL_x_swap -- ns32000 byte swapping stubs			*/
/*									*/
/************************************************************************/


Integer
DL_x_swap(x)
   Integer x;
{
   return x;
};




/************************************************************************/
/*									*/
/*	DL_x_nm_command -- generate nm command on a file		*/
/*	DL_x_cc_command -- generate cc command on a file		*/
/*	DL_x_ld_command -- generate ld command on 2 files		*/
/*	DL_x_as_command -- generate as command on 2 files		*/
/*									*/
/************************************************************************/


void
DL_x_nm_command(buf,file,tblfg)
   String buf;
   String file;
   Boolean tblfg;		/* TRUE if for building tables		*/
				/* FALSE if for dynload symbols 	*/
{
   sprintf(buf,"nm -pg %s",file);
};





void
DL_x_cc_command(buf,opts,file)
   String buf;
   String opts;
   String file;
{
   sprintf(buf,"cc %s %s",opts,file);
};





void
DL_x_ld_command(buf,outfile,f1,f2,dbg,args)
   String buf;
   String outfile;
   String f1;
   String f2;
   Boolean dbg;
   String args;
{
   sprintf(buf,"cc -u _DL_stub_routine -o %s %s %s%s",
	      outfile,args,f2,(dbg ? " -lg" : ""));
};






void
DL_x_as_command(buf,src,tgt)
   String buf;
   String src;
   String tgt;
{
   sprintf(buf,"as -o %s %s",tgt,src);
};




/************************************************************************/
/*									*/
/*	DL_x_nm_read -- read a nm line -- return true if undefined sym	*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_read(buf,np,vp,dfg,txtp)
   String buf;
   String * np;
   Integer * vp;
   Boolean dfg;
   Boolean * txtp;
{
   String p;
   Character t;
   Integer v,dlta;
   Boolean fg,tfg;

   if (dfg && !dyn_name_sort) {
      qsort(dyn_names,dyn_name_ct,sizeof(String),name_comp);
      dyn_name_sort = TRUE;
    };

   if (buf[0] == '0' && buf[1] == 'x') dlta = 2;
   else dlta = 0;

   p = &buf[dlta];

   v = 0;
   while (*p != ' ') {
      v = v*16;
      if (isdigit(*p)) v += *p - '0';
      else if (islower(*p)) v += *p - 'a' + 10;
      else if (isupper(*p)) v += *p - 'A' + 10;
      else break;
      ++p;
    };
   *vp = v;

   while (*p == ' ') ++p;

   t = *p++;
   while (*p == ' ') ++p;

   *np = p;
   while (*p != 0 && *p != ' ' && *p != '\n') ++p;
   *p = 0;

   fg = (t == 'U' && v == 0);
   tfg = (t == 'T' || t == 't');

   if (fg && dfg && test_dynamic(*np)) fg = FALSE;
   if (txtp != NULL) *txtp = tfg;

   return fg;
};





static int
name_comp(s1,s2)
   String * s1;
   String * s2;
{
   return strcmp(*s1,*s2);
};





static Boolean
test_dynamic(s)
   String s;
{
   Integer l,h,m,i;

   l = 0;
   h = dyn_name_ct-1;

   while (l <= h) {
      m = (l+h)/2;
      i = strcmp(s,dyn_names[m]);
      if (i == 0) return TRUE;
      else if (i < 0) h = m-1;
      else l = m+1;
    };

   return FALSE;
};




/********************************************************************************/
/*										*/
/*	DL_x_other_syms -- extra symbols					*/
/*										*/
/********************************************************************************/


String
DL_x_other_syms()
{
   static Integer otherct = 0;

#ifdef ALL_LIB_SYMS
   while (otherct < dyn_name_ct) {
      return dyn_names[otherct++];
    };
#endif

   return NULL;
};





/************************************************************************/
/*									*/
/*	DL_x_nm_element -- check for new element in archive		*/
/*	DL_x_nm_extdata -- check for external data definition		*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_element(buf,elt)
   String buf;
   String elt;
{
   Integer i;

   i = strlen(buf);
   if (buf[i-1] == '\n') --i;
   if (buf[i-1] == ':') {
      strcpy(elt,buf);
      elt[i-1] = 0;
      return TRUE;
    };

   return FALSE;
};





Boolean
DL_x_nm_extdata(buf)
   String buf;
{
   if (buf[0] == ' ') return FALSE;
   while (*buf != ' ') ++buf;
   while (*buf == ' ') ++buf;

   return (*buf == 'D' || *buf == 'T');
};





/************************************************************************/
/*									*/
/*	DL_x_ignore_symbol -- indicate if symbol should be ignored	*/
/*									*/
/************************************************************************/


Boolean
DL_x_ignore_symbol(nm)
   String nm;
{
/* if (nm[0] != '_') return TRUE;       */

   if (STREQL(nm,"__DYNAMIC")) return TRUE;
   if (STREQL(nm,"__GLOBAL_OFFSET_TABLE_")) return TRUE;

   return FALSE;
};





/********************************************************************************/
/*										*/
/*	DL_x_lib_name -- get library name					*/
/*										*/
/********************************************************************************/


void
DL_x_lib_name(buf)
   char * buf;
{
   String s;
   String a;
   Integer i;

   if (getenv("DL_LIBRARY") != NULL) {
      strcpy(buf,getenv("DL_LIBRARY"));
      return;
    };

   share_search();
   if (add_shared_lib("dl")) {
      sprintf(buf,"-ldl");
      return;
    };

   s = DL_pro_dir();
   a = DL_arch();

   for (i = search_ct-1; i >= 0; --i) {
      sprintf(buf,"%s/libdl.a",search_libs[i]);
      if (access(buf,4) >= 0) {
	 sprintf(buf,"-ldl");
	 return;
       };
      if (a != NULL) {
	 sprintf(buf,"%s/libdl_%s.a",search_libs[i],a);
	 if (access(buf,4) >= 0) {
	    sprintf(buf,"-ldl");
	    return;
	  };
       }
    };

   buf[0] = 0;
   if (s != NULL && a != NULL) {
      sprintf(buf,"%s/lib/dynlib.%s.o",s,a);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && s != NULL) {
      sprintf(buf,"%s/lib/dynlib.o",s);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && a != NULL) {
      sprintf(buf,"/usr/local/lib/dynlib.%s.o",a);
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.%s.o",a);
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.%s.o",a);
	    if (access(buf,4) < 0) buf[0] = 0;
	  };
       };
    }

   if (buf[0] == 0) {
      sprintf(buf,"/usr/local/lib/dynlib.o");
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.o");
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.o");
	    if (access(buf,4) < 0) sprintf(buf,"/usr/lib/dynlib.o");
	  };
       };
    };
};





/********************************************************************************/
/*										*/
/*	DL_x_check_arg -- special processing of arguments			*/
/*										*/
/********************************************************************************/


Boolean
DL_x_check_arg(arg)
   String arg;
{
   Boolean doshare = TRUE;

   if (arg == NULL) {
      add_shared_lib("c");
      return FALSE;
    };

   share_search();

   if (STREQL(arg,"-Bstatic")) doshare = FALSE;
   else if (STREQL(arg,"-Bdynamic")) doshare = TRUE;
   else if (arg[0] == '-' && arg[1] == 'L' && arg[2] != 0) {
      search_libs[search_ct++] = SALLOC(&arg[2]);
    }
   else if (arg[0] == '-' && arg[1] == 'l' && arg[2] != 0) {
      add_shared_lib(&arg[2]);
    };

   return FALSE;
};





/************************************************************************/
/*									*/
/*	DL_x_gen_stub -- generate assembler code for a stub		*/
/*									*/
/************************************************************************/


Boolean
DL_x_gen_stub(otf,nm,idx)
   FILE * otf;
   String nm;
   Integer idx;
{
   if (nm == NULL) {
      if (idx == 0) {
	 fprintf(otf,"   .text\n");
       }
      else {
	 fprintf(otf,"DL_stub_handler:\n");
#ifndef sparc
	 fprintf(otf,"   lea _DL__stub_branch_table,a0\n");
	 fprintf(otf,"   addl d0,a0\n");
	 fprintf(otf,"   movl a0@,a0\n");
	 fprintf(otf,"   jra a0@\n\n");
#else
	 fprintf(otf,"   set _DL__stub_branch_table,%%g2\n");
	 fprintf(otf,"   ld [%%g1+%%g2],%%g2\n");
	 fprintf(otf,"   jmp %%g2\n");
	 fprintf(otf,"   nop\n\n");
#endif
	 fprintf(otf,"  .globl _DL_stub_routine\n");
	 fprintf(otf,"  .globl _DL_stub\n");
	 fprintf(otf,"  .globl _DL_ext_stub\n");
	 fprintf(otf,"_DL_ext_stub:\n");
#ifndef sparc
	 fprintf(otf,"  movew #%d,d0\n",(idx-1)*sizeof(DL_LINK_INFO));
#else
	 fprintf(otf,"  mov %d,%%g1\n",(idx-1)*sizeof(DL_LINK_INFO));
#endif
	 fprintf(otf,"_DL_stub:\n");
#ifndef sparc
	 fprintf(otf,"  movl sp@,a0\n");
	 fprintf(otf,"  movl d0,(sp)@-\n");
	 fprintf(otf,"  subql #4,a0\n");
	 fprintf(otf,"  movl a0,(sp)@-\n");
	 fprintf(otf,"  jbsr _DL_stub_routine\n");
	 fprintf(otf,"  addqw #8,sp\n");
	 fprintf(otf,"  movl d0,a0\n");
	 fprintf(otf,"  jmp a0@\n");
	 fprintf(otf,"__DL_stub_addr: .long _DL_stub\n");
#else
	 fprintf(otf,"  save %%sp,-(16*4),%%sp\n");
	 fprintf(otf,"  mov %%i7,%%o0\n");
	 fprintf(otf,"  mov %%g1,%%o1\n");
	 fprintf(otf,"  sethi %%hi(_DL_stub_routine),%%g1\n");
	 fprintf(otf,"  or %%g1,%%lo(_DL_stub_routine),%%g1\n");
	 fprintf(otf,"  call %%g1,2\n");
	 fprintf(otf,"  nop\n");
	 fprintf(otf,"  jmpl %%o0,%%g0\n");
	 fprintf(otf,"  restore\n");
#endif
       };
      return;
    };

   fprintf(otf,"  .globl %s\n",nm);
   fprintf(otf,"%s:\n",nm);
#ifndef sparc
   fprintf(otf,"   movew #%d,d0\n",idx*sizeof(DL_LINK_INFO));
   fprintf(otf,"   jra DL_stub_handler\n");
#else
   fprintf(otf,"   mov %d,%%g1\n",idx*sizeof(DL_LINK_INFO));
   fprintf(otf,"   b DL_stub_handler\n");
   fprintf(otf,"   nop\n");
#endif

   fprintf(otf,"\n");
};







/********************************************************************************/
/*										*/
/*	add_shared_lib -- get symbols from a shared library			*/
/*	get_shared_libs -- find nested shared librays				*/
/*	get_shared_syms -- get shared symbols for executable			*/
/*	add_shared_syms -- add symbols from shared library			*/
/*										*/
/********************************************************************************/


static Boolean
add_shared_lib(nam)
   String nam;
{
#ifdef NO_SHARED_LIBS
   return FALSE;
#else
   Character buf[1024],fbuf[1024],sbuf[1024];
   DIR * dir;
   struct dirent * d;
   Integer i,ct,j;
   Boolean fg;
   Float ver,nver;
   extern double atof();
   String s;
   Boolean versok;

   fg = FALSE;

   s = index(nam,'.');
   if (s != NULL && strncmp(s,".so.",4) == 0) {
      sprintf(buf,"lib%s",nam);
      *s = 0;
      versok = TRUE;
    }
   else versok = FALSE;

   for (i = 0; i < shared_ct; ++i) {
      if (STREQL(nam,shared_libs[i])) return FALSE;
    };
   shared_libs[shared_ct++] = SALLOC(nam);

   ver = -1.0;
   if (!versok) sprintf(buf,"lib%s.so.",nam);
   ct = strlen(buf);
   for (i = search_ct-1; i >= 0; --i) {
      d = NULL;
      dir = opendir(search_libs[i]);
      if (dir != NULL) {
	 while ((d = readdir(dir)) != NULL) {
	    if (strncmp(buf,d->d_name,ct) == 0) {
	       sprintf(fbuf,"%s/%s",search_libs[i],d->d_name);
	       if (access(fbuf,4) < 0) continue;
	       nver = atof(&d->d_name[ct]);
	       if (versok) nver = 1;
	       if (versok || ver < 0 || nver > ver) {
		  strcpy(sbuf,fbuf);
		  ver = nver;
		  if (versok) break;
		};
	     };
	  };
	 closedir(dir);
	 if (ver >= 0) break;
       };
    };

   if (ver >= 0) {
      add_shared_syms(sbuf);
      get_shared_libs(sbuf);
      fg = TRUE;
    };

   return fg;
#endif
};





static void
get_shared_libs(bin)
   String bin;
{
#ifndef NO_SHARED_LIBS
   Integer f;
   struct exec hdr;
   struct link_dynamic ldyn;
   struct link_dynamic_2 ldyn2;
   struct link_object lobj;
   Character nam[1024];
   Integer loc;
   Integer delta;

   f = open(bin,0);
   read(f,&hdr,sizeof(hdr));
   if (!hdr.a_dynamic) {
      close(f);
      return;
    };

   lseek(f,N_DATOFF(hdr),0);
   read(f,&ldyn,sizeof(ldyn));

#ifndef sparc
   delta = N_DATADDR(hdr) - N_DATOFF(hdr);
   if (delta >= SEGSIZ) delta -= SEGSIZ;
#else
   delta = 0;
#endif

   loc = (Integer) ldyn.ld_un.ld_2;
   lseek(f,loc - delta,0);
   read(f,&ldyn2,sizeof(ldyn2));

   for (loc = ldyn2.ld_need; loc != 0; loc = lobj.lo_next) {
      lseek(f,loc,0);
      read(f,&lobj,sizeof(lobj));
      lseek(f,lobj.lo_name,0);
      read(f,nam,1024);
      add_shared_lib(nam);
    };
#endif
};





static void
get_shared_syms(bin)
   String bin;
{
#ifndef NO_SHARED_LIBS
   FILE * inf;
   Character buf[1024];
   String s,t;
   String libs[128];
   Integer libct,i;

   sprintf(buf,"ldd %s",bin);

   inf = popen(buf,"r");

   libct = 0;
   while (fgets(buf,1024,inf) != NULL) {
      s = index(buf,'-');
      if (s == NULL || strncmp(s,"-lc.",4) == 0) continue;
      s = index(s,'>');
      ++s;
      while (isspace(*s)) ++s;
      t = index(s,'\n');
      if (t != NULL) *t = 0;
      libs[libct++] = SALLOC(s);
    };

   pclose(inf);

   for (i = 0; i < libct; ++i) {
      add_shared_syms(libs[i]);
      SFREE(libs[i]);
    };
#endif
};





static void
add_shared_syms(lnam)
   String lnam;
{
#ifndef NO_SHARED_LIBS
   FILE * f;
   Integer i;
   Character buf[1024];
   Integer addr;
   Character cls;
   Character nam[1024];

   if (dyn_name_max == 0) {
      dyn_name_max = DYN_SYM_SIZE;
      dyn_names = (String *) malloc(dyn_name_max*sizeof(String));
      dyn_name_ct = 0;
    };

   sprintf(buf,"nm -pg %s",lnam);
   f = popen(buf,"r");
   while (fgets(buf,1024,f) != NULL) {
      i = sscanf(buf,"%x %c %s",&addr,&cls,nam);
      if (i != 3 || !isupper(cls)) continue;
      if (dyn_name_ct+1 >= dyn_name_max) {
	 dyn_name_max *= 2;
	 dyn_names = (String *) realloc(dyn_names,dyn_name_max*sizeof(String));
       };
      dyn_names[dyn_name_ct++] = SALLOC(nam);
    };
   dyn_name_sort = FALSE;
   pclose(f);
#endif
};





static void
share_search()
{
#ifndef NO_SHARED_LIBS
   String s,t;
   Character buf[10240];

   if (search_ct < 0) {
      search_ct = -search_ct;
      s = (String) getenv("LD_LIBRARY_PATH");
      if (s != NULL) {
	 strcpy(buf,s);
	 for ( ; ; ) {
	    t = rindex(buf,':');
	    if (t != NULL) {
	       *t++ = 0;
	       if (*t != 0) search_libs[search_ct++] = SALLOC(t);
	     }
	    else if (buf[0] != 0) {
	       search_libs[search_ct++] = SALLOC(buf);
	       break;
	     };
	   };
       };
    };
#endif
};





/************************************************************************/
/*									*/
/*	dummy routine to force loading of needed things 		*/
/*									*/
/************************************************************************/


static void
dummy_routine()
{
   int a,b,c;
   float d,e,f;
   double g,h,i;
   unsigned j,k,l;

   a = b*c;
   b = a%c;
   strcmp(0,1); 		/* force strcmp to be available */
   d = d+e*f;
   d = d/f-e;
   a = d;
   e = b;
   g = g*h+i-g/h;
   e = g;
   h = e;
   b = g;
   h = a;
   j = k*l + l %j + l/k;
   j = a;
   a = j;
   vfork(a,b,c,d,e,f);
   ioctl(j,k,l);
};





/* end of dlsun.c */
