/************************************************************************/
/*									*/
/*		dlmips.c						*/
/*									*/
/*	Routines and definitions for MIPS R2000 dynamic loading (DEC 3100) */
/*									*/
/************************************************************************/


#include "dl_local.h"

#include <ar.h>
#include <a.out.h>
#include <sys/file.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define AR_SYMDEF	"__.SYMDEF"




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		setup_segment();
static	void		setup_reloc();
static	void		setup_symbols();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String	search_libs[32] = {
   "/lib", "/usr/lib", "/usr/local/lib", 0
};

static	Integer search_ct = -3;






/************************************************************************/
/*									*/
/*	DL_x_prepend_ -- indicate if underscore prepended to names	*/
/*									*/
/************************************************************************/


Boolean
DL_x_prepend_()
{
   return FALSE;
};





/************************************************************************/
/*									*/
/*	DL_x_test_archive -- test for archive file			*/
/*	DL_x_next_archive -- return offset for next archive element	*/
/*									*/
/************************************************************************/


Boolean
DL_x_test_archive(fi,offp)
   Integer fi;
   Integer * offp;
{
   Character magbuf[SARMAG];
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,0,L_SET);
   read(fi,magbuf,SARMAG);
   if (strncmp(magbuf,ARMAG,SARMAG) != 0) return FALSE;

   *offp = SARMAG + sizeof(hdr);

   read(fi,&hdr,sizeof(hdr));
   if (STREQL(hdr.ar_name,AR_SYMDEF)) {
      *offp += sizeof(hdr);
      len = atoi(hdr.ar_size);
      *offp += ((len+1) & ~1);
    };

   return TRUE;
};





Boolean
DL_x_next_archive(fi,ename,offp)
   Integer fi;
   String ename;
   Integer *offp;
{
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,*offp-sizeof(hdr),L_SET);
   if (read(fi,&hdr,sizeof(hdr)) != sizeof(hdr)) return FALSE;

   *offp += sizeof(hdr);
   len = atoi(hdr.ar_size);
   *offp += ((len+1) & ~1);

   strncpy(ename,hdr.ar_name,sizeof(hdr.ar_name));

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_setup_buffers -- read in data from a file			*/
/*									*/
/************************************************************************/


Boolean
DL_x_setup_buffers(dlf,fi,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer aoff;
{
   struct filehdr fhdr;
   struct aouthdr ahdr;
   struct scnhdr shdr[32];
   Integer i;
   long int stlen;

   lseek(fi,aoff,L_SET);
   if (read(fi,&fhdr,sizeof(fhdr)) <= 4) return FALSE;
   if (!ISCOFF(fhdr.f_magic)) return FALSE;
   if (fhdr.f_opthdr > 0) {
      lseek(fi,aoff+sizeof(fhdr)+fhdr.f_opthdr,L_SET);
    };
   read(fi,shdr,sizeof(struct scnhdr)*fhdr.f_nscns);

   for (i = 0; i < fhdr.f_nscns; ++i) {
      if (shdr[i].s_scnptr != 0)
	 setup_segment(dlf,i,fi,shdr[i].s_size,aoff+shdr[i].s_scnptr,shdr[i].s_vaddr);
      else
	 setup_segment(dlf,i,-1,shdr[i].s_size,-1,shdr[i].s_vaddr);
      dlf->segs[i].poffset = shdr[i].s_paddr;
    };
   dlf->numseg = fhdr.f_nscns;

   dlf->common = NULL;
   dlf->basereg = NULL;

   for (i = 0; i < fhdr.f_nscns; ++i) {
      if (shdr[i].s_relptr != 0)
	 setup_reloc(dlf,i,fi,shdr[i].s_nreloc,aoff+shdr[i].s_relptr,
			shdr[i].s_vaddr,shdr,fhdr.f_nscns);
    };

   setup_symbols(dlf,fi,fhdr.f_nsyms,fhdr.f_symptr,aoff);

   return TRUE;
};



static void
setup_segment(dlf,seg,fi,len,addr,off)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
   Integer off;
{
   Address rgn;

   if (len > 0) {
      rgn = (Address) malloc(len);
      if (fi >= 0) {
	 lseek(fi,addr,L_SET);
	 read(fi,rgn,len);
       }
      else {
	 bzero(rgn,len);
       };
    }
   else rgn = NULL;

   dlf->segs[seg].base = rgn;
   dlf->segs[seg].reloc = NULL;
   dlf->segs[seg].offset = off;
   dlf->segs[seg].poffset = 0;
   dlf->segs[seg].reloc_size = 0;
   dlf->segs[seg].length = len;
}





static void
setup_reloc(dlf,seg,fi,ct,addr,vaddr,shdr,sct)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer ct;
   Integer addr;
   Integer vaddr;
   struct scnhdr shdr[];
   Integer sct;
{
   Address xrgn;
   DL_RELOC rel;
   Integer len,i,j;
   struct reloc * rp;
   String s;

   if (ct <= 0) return;

   len = ct*RELSZ;

   rel = (DL_RELOC) calloc(ct,sizeof(DL_RELOC_INFO));
   xrgn = (Address) alloca(len);

   lseek(fi,addr,L_SET);
   read(fi,xrgn,len);

   for (i = 0; i < ct; ++i) {
      rp = (struct reloc *) (xrgn + RELSZ*i);
      rel[i].dr_address = rp->r_vaddr - vaddr;
      rel[i].dr_value = 0;
      if (rp->r_extern == 0) {
	 rel[i].dr_symbol = FALSE;
	 switch (rp->r_symndx) {
	    case R_SN_TEXT :
	       s = ".text";
	       break;
	    case R_SN_RDATA :
	       s = ".rdata";
	       break;
	    case R_SN_DATA :
	       s = ".data";
	       break;
	    case R_SN_SDATA :
	       s = ".sdata";
	       break;
	    case R_SN_SBSS :
	       s = ".sbss";
	       break;
	    case R_SN_BSS :
	       s = ".bss";
	       break;
	  };
	 for (j = 0; j < sct; ++j) {
	    if (STREQL(s,shdr[j].s_name)) break;
	  };
	 rel[i].dr_index = j;
       }
      else {
	 rel[i].dr_symbol = TRUE;
	 rel[i].dr_index = rp->r_symndx;
       };

      switch (rp->r_type) {
	 case R_REFHALF :
	    rel[i].dr_type = RTYPE_16;
	    break;
	 case R_REFWORD :
	    rel[i].dr_type = RTYPE_32;
	    break;
	 case R_JMPADDR :
	    rel[i].dr_type = RTYPE_MACHINE(0);
	    break;
	 case R_REFHI :
	    rel[i].dr_type = RTYPE_HI16;
	    break;
	 case R_REFLO :
	    rel[i].dr_type = RTYPE_LO16;
	    break;
	 case R_GPREL :
	    rel[i].dr_type = RTYPE_MACHINE(1);
	    break;
	 case R_LITERAL :
	    rel[i].dr_type = RTYPE_MACHINE(2);
	    break;
	 default :
	    rel[i].dr_type = RTYPE_NONE;
	    break;
       };
    };

   dlf->segs[seg].reloc = rel;
   dlf->segs[seg].reloc_size = ct;
};





static void
setup_symbols(dlf,fi,hdrsize,hdraddr,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer hdrsize;
   Integer hdraddr;
   Integer aoff;
{
   HDRR symhdr;
   Integer ct;
   Integer addr;
   Integer straddr;
   pEXTR xbuf;
   DL_ESYM sym;
   Integer len,stlen,i;
   String eptr;
   Character ebuf[10];
   pEXTR sp;

   lseek(fi,hdraddr+aoff,L_SET);
   read(fi,&symhdr,sizeof(HDRR));

   ct = symhdr.iextMax;
   addr = symhdr.cbExtOffset;
   len = ct * cbEXTR;

   if (ct == 0) {
      dlf->numsym = 0;
      dlf->strings = NULL;
      dlf->syms = NULL;
      return;
    };

   xbuf = (pEXTR) alloca(len);
   lseek(fi,addr+aoff,L_SET);
   read(fi,xbuf,len);

   lseek(fi,symhdr.cbSsExtOffset+aoff,L_SET);
   stlen = symhdr.issExtMax;
   if (stlen > 0) {
      dlf->strings = (String) malloc(stlen+ct*9);
      read(fi,dlf->strings,stlen);
    }
   else {
      dlf->strings = (String) malloc(4+ct*9);
      stlen = 4;
    };
   eptr = &dlf->strings[stlen];

   sym = (DL_ESYM) calloc(sizeof(DL_ESYM_INFO),ct);
   for (i = 0; i < ct; ++i) {
      sp = &xbuf[i];
      sym[i].ds_name = &dlf->strings[sp->asym.iss];
      sym[i].ds_value = sp->asym.value;
      sym[i].ds_extern = TRUE;
      switch (sp->asym.sc) {
	 case scNil :
	 case scUndefined :
	    sym[i].ds_segment = SEGMENT_UNDEF;
	    break;
	 case scAbs :
	    sym[i].ds_segment = SEGMENT_ABS;
	    break;
	 case scCommon :
	 case scSCommon :
	    sym[i].ds_segment = SEGMENT_COMMON;
	    break;
	 default :
	    if (sp->asym.sc > 0) sym[i].ds_segment = sp->asym.sc-1;
	    else sym[i].ds_segment = SEGMENT_IGNORE;
	    break;
       };
    };

   dlf->numsym = ct;
   dlf->syms = sym;
};





/************************************************************************/
/*									*/
/*	DL_x_machine_reloc -- handle machine relocation types		*/
/*									*/
/************************************************************************/


void
DL_x_machine_reloc(id,rty,ptr,tp,vp)
   Integer id;
   RELOC_TYPE rty;
   Character * ptr;
   Integer * tp;
   Integer * vp;
{
   Integer v;
   Integer * xp;
   extern int _gp;

   switch (id) {
      case 0 :
	 switch (rty) {
	    case RTYPE_MACHINE(0) :
	       v = *(long *) ptr;
	       *vp = v & 0xfc000000;
	       v &= 0x03ffffff;
	       v <<= 2;
	       *tp = v;
	       break;
	    case RTYPE_MACHINE(1) :
	    case RTYPE_MACHINE(2) :
	       v = *(long *) ptr;
	       *vp = v & 0xffff0000;
	       *tp = v & 0xffff;
	       break;
	  };
	 break;

      case 1 :
	 switch (rty) {
	    case RTYPE_MACHINE(0) :
	       v = ((int) ptr) & 0xf0000000;
	       if (v == ((*vp)&0xf0000000)) {
		  v = *tp + *vp;
		  v &= 0x3fffffff;
		  *tp = v;
		}
	       else {
		  v = *tp + *vp;
		  xp = (Integer *) malloc(16);
		  xp[0] = 0x3c080000 + ((v >> 16) & 0xffff);
		  xp[1] = 0x35080000 + (v & 0xffff);
		  xp[2] = 0x01000008;
		  xp[3] = 0;
		  *tp = (Integer) xp;
		};
	       break;
	    case RTYPE_MACHINE(1) :
	    case RTYPE_MACHINE(2) :
	       v = *vp - (int) &_gp;
	       *vp = v;
	       break;
	  }
	 break;

      case 2 :
	 switch (rty) {
	    case RTYPE_MACHINE(0) :
	       v = *tp;
	       v >>= 2;
	       v &= 0x03ffffff;
	       v |= *vp;
	       *(long *) ptr = v;
	       break;
	    case RTYPE_MACHINE(1) :
	    case RTYPE_MACHINE(2) :
	       v = *tp >> 16;
	       if (v != 0 && v != -1) {
		  printf("MIPS:  Can't use GP on dynamic code; use cc -G 0\n");
		  v = 0;
		}
	       else {
		  v = *tp & 0xffff;
		  v |= *vp;
		  *(long *) ptr = v;
		  break;
		};
	       break;
	  };
	 break;
    };
};





/************************************************************************/
/*									*/
/*	DL_x_swap -- ns32000 byte swapping stubs			*/
/*									*/
/************************************************************************/


Integer
DL_x_swap(x)
   Integer x;
{
   return x;
};




/************************************************************************/
/*									*/
/*	DL_x_nm_command -- generate nm command on a file		*/
/*	DL_x_cc_command -- generate cc command on a file		*/
/*	DL_x_ld_command -- generate ld command on 2 files		*/
/*	DL_x_as_command -- generate as command on a file		*/
/*									*/
/************************************************************************/


void
DL_x_nm_command(buf,file,tblfg)
   String buf;
   String file;
   Boolean tblfg;		/* TRUE if for building tables		*/
				/* FALSE if for dynload symbols 	*/
{
   sprintf(buf,"nm -pg %s",file);
};





void
DL_x_cc_command(buf,opts,file)
   String buf;
   String opts;
   String file;
{
   sprintf(buf,"cc -G 0 %s %s",opts,file);
};





void
DL_x_ld_command(buf,outfile,f1,f2,dbg,args)
   String buf;
   String outfile;
   String f1;
   String f2;
   Boolean dbg;
   String args;
{
   sprintf(buf,"ld -o %s %s %s",outfile,f1,f2);
};






void
DL_x_as_command(buf,src,tgt)
   String buf;
   String src;
   String tgt;
{
   sprintf(buf,"as -G 0 -o %s %s",tgt,src);
};






/************************************************************************/
/*									*/
/*	DL_x_nm_read -- read a nm line -- return true if undefined sym	*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_read(buf,np,vp,dfg,txtp)
   String buf;
   String * np;
   Integer * vp;
   Boolean dfg;
   Boolean * txtp;
{
   String p;
   Character t;
   Integer v,dlta;
   Boolean fg,tfg;

   if (buf[0] == '0' && buf[1] == 'x') dlta = 2;
   else dlta = 0;

   p = &buf[dlta];

   v = 0;
   while (*p != ' ') {
      v = v*16;
      if (isdigit(*p)) v += *p - '0';
      else if (islower(*p)) v += *p - 'a' + 10;
      else if (isupper(*p)) v += *p - 'A' + 10;
      else break;
      ++p;
    };
   *vp = v;

   while (*p == ' ') ++p;

   t = *p++;
   while (*p == ' ') ++p;

   *np = p;
   while (*p != 0 && *p != ' ' && *p != '\n') ++p;
   *p = 0;

   fg = (t == 'U' && v == 0);
   tfg = (t == 'T' || t == 't');

   if (txtp != NULL) *txtp = tfg;

   return fg;
};






/********************************************************************************/
/*										*/
/*	DL_x_other_syms -- extra symbols					*/
/*										*/
/********************************************************************************/


String
DL_x_other_syms()
{
   return NULL;
};





/************************************************************************/
/*									*/
/*	DL_x_nm_element -- check for new element in archive		*/
/*	DL_x_nm_extdata -- check for external data definition		*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_element(buf,elt)
   String buf;
   String elt;
{
   Integer i;

   i = strlen(buf);
   if (buf[i-1] == '\n') --i;
   if (buf[i-1] == ':') {
      strcpy(elt,buf);
      elt[i-1] = 0;
      return TRUE;
    };

   return FALSE;
};





Boolean
DL_x_nm_extdata(buf)
   String buf;
{
   if (buf[0] == ' ') return FALSE;
   while (*buf != ' ') ++buf;
   while (*buf == ' ') ++buf;

   return (*buf == 'D' || *buf == 'T');
};





/************************************************************************/
/*									*/
/*	DL_x_ignore_symbol -- indicate if symbol should be ignored	*/
/*									*/
/************************************************************************/


Boolean
DL_x_ignore_symbol(nm)
   String nm;
{
   if (STREQL(nm,"_gp")) return TRUE;

   return FALSE;
};





/********************************************************************************/
/*										*/
/*	DL_x_lib_name -- get library name					*/
/*										*/
/********************************************************************************/


void
DL_x_lib_name(buf)
   char * buf;
{
   String s;
   String a;
   Integer i;

   s = DL_pro_dir();
   a = DL_arch();

   for (i = search_ct-1; i >= 0; --i) {
      sprintf(buf,"%s/libdl.a",search_libs[i]);
      if (access(buf,4) >= 0) {
	 sprintf(buf,"-ldl");
	 return;
       };
      if (a != NULL) {
	 sprintf(buf,"%s/libdl_%s.a",search_libs[i],a);
	 if (access(buf,4) >= 0) {
	    sprintf(buf,"-ldl");
	    return;
	  };
       }
    };

   if (getenv("DL_LIBRARY") != NULL) {
      strcpy(buf,getenv("DL_LIBRARY"));
      return;
    };

   buf[0] = 0;
   if (s != NULL && a != NULL) {
      sprintf(buf,"%s/lib/dynlib.%s.o",s,a);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && s != NULL) {
      sprintf(buf,"%s/lib/dynlib.o",s);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && a != NULL) {
      sprintf(buf,"/usr/local/lib/dynlib.%s.o",a);
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.%s.o",a);
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.%s.o",a);
	    if (access(buf,4) < 0) buf[0] = 0;
	  };
       };
    }

   if (buf[0] == 0) {
      sprintf(buf,"/usr/local/lib/dynlib.o");
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.o");
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.o");
	    if (access(buf,4) < 0) sprintf(buf,"/usr/lib/dynlib.o");
	  };
       };
    };
};










/********************************************************************************/
/*										*/
/*	DL_x_check_arg -- special processing of arguments			*/
/*										*/
/********************************************************************************/


Boolean
DL_x_check_arg(arg)
   String arg;
{
   return FALSE;
};





/************************************************************************/
/*									*/
/*	DL_x_gen_stub -- generate assembler code for a stub		*/
/*									*/
/************************************************************************/


Boolean
DL_x_gen_stub(otf,nm,idx)
   FILE * otf;
   String nm;
   Integer idx;
{
   if (nm == NULL) {
      if (idx == 0) {
	 fprintf(otf,"   .text\n");
       }
      else {
	 fprintf(otf,"DL_stub_handler:\n");
	 fprintf(otf,"   lw $9,DL__stub_branch_table($8)\n");
	 fprintf(otf,"   j $9\n\n");
	 fprintf(otf,"   .globl DL_stub_routine\n");
	 fprintf(otf,"   .globl DL_stub\n");
	 fprintf(otf,"   .globl DL_ext_stub\n");
	 fprintf(otf,"DL_ext_stub:\n");
	 fprintf(otf,"   li $8,%d\n",(idx-1)*sizeof(DL_LINK_INFO));
	 fprintf(otf,"   .ent DL_stub 2\n");
	 fprintf(otf,"DL_stub:\n");
	 fprintf(otf,"   subu $sp,20\n");
	 fprintf(otf,"   sw $31,20($sp)\n");
	 fprintf(otf,"   sw $4,16($sp)\n");
	 fprintf(otf,"   sw $5,12($sp)\n");
	 fprintf(otf,"   .mask 0x80000030,0\n");
	 fprintf(otf,"   .frame $sp,12,$31\n");
	 fprintf(otf,"   addu $4,$31,-4\n");
	 fprintf(otf,"   move $5,$8\n");
	 fprintf(otf,"   jal DL_stub_routine\n");
	 fprintf(otf,"   lw $5,12($sp)\n");
	 fprintf(otf,"   lw $4,16($sp)\n");
	 fprintf(otf,"   lw $31,20($sp)\n");
	 fprintf(otf,"   addu $sp,20\n");
	 fprintf(otf,"   j $2\n");
	 fprintf(otf,"   .end DL_stub\n");
       };
      return;
    };

   fprintf(otf,"  .globl %s\n",nm);
   fprintf(otf,"%s:\n",nm);
   fprintf(otf,"   li $8,%d\n",idx*sizeof(DL_LINK_INFO));
   fprintf(otf,"   b DL_stub_handler\n");

   fprintf(otf,"\n");
};







/************************************************************************/
/*									*/
/*	dummy routine to force loading of needed things 		*/
/*									*/
/************************************************************************/


static void
dummy_routine()
{
   int a,b,c;
   float d,e,f;
   double g,h,i;

   a = b*c;
   b = a%c;
   strcmp(0,1); 		/* force strcmp to be available */
   d = d+e*f;
   d = d/f-e;
   a = d;
   e = b;
   g = g*h+i-g/h;
   e = g;
   h = e;
   b = g;
   h = a;
   vfork(a,b,c,d,e,f);
};






/* end of dlmips.c */
