/************************************************************************/
/*									*/
/*		dlloader.c						*/
/*									*/
/*	Machine-independent loading routine for dynload 		*/
/*									*/
/************************************************************************/
/*	Copyright 1987 Brown University -- Steven P. Reiss		*/



#include "dl_local.h"




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static DL_FILE		file_list = NULL;





/************************************************************************/
/*									*/
/*	Forward references						*/
/*									*/
/************************************************************************/


static	Boolean 	setup_namelist();
static	Boolean 	load_archive();
static	DL_FILE 	setup_buffers();
static	Address 	setup_buffer();
static	void		relocate_all();
static	void		relocate();
static	Integer 	relocate_seg();
static	Integer 	round();





/************************************************************************/
/*									*/
/*	DL_load_subinit -- initialize this part of loader		*/
/*									*/
/************************************************************************/


void
DL_load_subinit()
{
   file_list = NULL;
}



/************************************************************************/
/*									*/
/*	DL_load_file -- load a file entry				*/
/*									*/
/************************************************************************/


Boolean
DL_load_file(fname, element)
   char *fname;
   char *element;
{
   Boolean success;
   Integer fi;

   fi = open(fname,0);

   if (fi < 0) {
      DL_error("DL_load_file: cannot open", fname);
      success = FALSE;
    }
   else {
      success = setup_namelist(fname,element,fi,0);
      close(fi);
    }

   return(success);
}





/************************************************************************/
/*									*/
/*	load_archive -- load from archive				*/
/*									*/
/************************************************************************/


static Boolean
load_archive(aname,aelt,fi)
   String aname;
   String aelt;
   Integer fi;
{
   Character ename[257];
   Boolean retval = 0;
   Integer aoff,noff;

   if (!DL_x_test_archive(fi,&noff)) return FALSE;

   for ( ; ; ) {
      aoff = noff;
      if (!DL_x_next_archive(fi,ename,&noff)) break;
      if (aelt == NULL || STREQL(aelt,ename)) {
	 setup_namelist(aname,ename,fi,aoff);
       }
    };

   return TRUE;
}




/************************************************************************/
/*									*/
/*	setup_namelist -- get symbols from file and then load		*/
/*									*/
/************************************************************************/


static Boolean
setup_namelist(fname,ename,fi,aoff)
   String fname;
   String ename;
   Integer fi;
   Integer aoff;
{
   Integer num_syms;
   DL_ESYM symp;
   DL_ISYM_INFO save_symp;
   DL_ISYM old_symp;
   Integer name_len;
   String symname = NULL;
   DL_HashKey loc;
   Integer num_resolved = 0;
   Boolean add_to_symtab;
   DL_FILE new_file;
   Integer common_size;
   Integer common_seg;

   new_file = PALLOC(DL_FILE_INFO);

   new_file->fname = SALLOC(fname);
   new_file->ename = (ename == NULL ? NULL : SALLOC(ename));
   new_file->next = NULL;

   if (!DL_x_setup_buffers(new_file,fi,aoff)) {
      if (aoff == 0) {
	 if (!load_archive(fname,ename,fi)) {
	    DL_error(fname, "bad format");
	    return FALSE;
	  }
	 else return TRUE;
       }
      else return FALSE;
    };

   num_syms = new_file->numsym;
   if (num_syms == 0) {
      DL_error(fname, "no name list");
      return FALSE;
    }

   symp = new_file->syms;
   common_size = 0;

   for (symp = new_file->syms; num_syms--; ++symp) {
      /* Only allocate space for a common symbol if we do
	 ** not already have the symbol defined.  Currently there
	 ** is no check to make sure that common symbols have the
	 ** same length... the first module that defines the symbol
	 ** establishes the area for it!
	 */
      if (symp->ds_segment == SEGMENT_IGNORE) continue;

      old_symp = DL_lookup(symp->ds_name,&loc);
      if (symp->ds_extern && symp->ds_segment == SEGMENT_COMMON &&
	     symp->ds_value != 0 &&
	     (old_symp == NULL || old_symp->dn_undef)) {
	 int size = symp->ds_value;
	 int rnd;

	 if (size >= sizeof (double))
	    rnd = sizeof (double);
	 else if (size >= sizeof (long))
	    rnd = sizeof (long);
	 else
	    rnd = sizeof (short);

	 common_size = round(common_size,rnd);

	 symp->ds_value = common_size;
	 symp->ds_segment = new_file->numseg;

	 common_size += round(size, rnd);
       };
    };

   if (common_size > 0) {
      common_seg = new_file->numseg++;
      new_file->segs[common_seg].base = (Address) calloc(1,common_size);
      new_file->segs[common_seg].reloc = NULL;
      new_file->segs[common_seg].offset = 0;
      new_file->segs[common_seg].reloc_size = 0;
      new_file->segs[common_seg].length = common_size;
    };

   num_syms = new_file->numsym;

   for (symp = new_file->syms; num_syms--; ++symp) {
      if (symp->ds_segment == SEGMENT_IGNORE) continue;

      if (symp->ds_segment != SEGMENT_UNDEF && symp->ds_segment != SEGMENT_ABS) {
	 symp->ds_value += ((unsigned int) new_file->segs[symp->ds_segment].base) -
	    new_file->segs[symp->ds_segment].offset;
       };

      add_to_symtab = 1;
      if (symp->ds_extern) {
	 old_symp = DL_lookup(symp->ds_name,&loc);
	 if (old_symp != NULL) {
	    if (symp->ds_segment == SEGMENT_UNDEF) {
	       if (!old_symp->dn_undef) {
		  symp->ds_value = old_symp->dn_value;
		  symp->ds_segment = SEGMENT_ABS;
		  continue;
		}
	       else {
		  add_to_symtab = 0;
		};
	     }
	    else if (old_symp->dn_undef) {
	       if (old_symp->dn_stub) {
		  if (symp->ds_segment != 0)
		     DL_error("setup_namelist","attempting to resolve a function reference with data");
		  DL_fill_in_linkage_seg(old_symp,symp->ds_value);
		}
	       num_resolved++;
	     }
	    else {
	       char message[256];
	       if (ename) {
		  sprintf(message,"duplicate external: %s(%s: %s)",
			     fname,ename,symname);
		}
	       else {
		  sprintf(message,"duplicate external: %s: %s",
			     fname,symname);
		}
	       DL_error("setup_namelist",message);
	     }
	  }

	 if (add_to_symtab) {
	    save_symp.dn_value = symp->ds_value;
	    save_symp.dn_undef = (symp->ds_segment == SEGMENT_UNDEF);
	    save_symp.dn_stub = 0;
	    save_symp.dn_name = SALLOC(symp->ds_name);
	    if (!DL_enter(&save_symp,loc,1))
	       DL_error("setup_namelist unable to add symbol",symname);
	  }
       }
    };

   relocate(new_file,TRUE);

   if (num_resolved) relocate_all();

   new_file->next = file_list;
   file_list = new_file;

   free(new_file->syms);
   free(new_file->strings);

   return TRUE;
}





/************************************************************************/
/*									*/
/*	relocate -- relocate data loaded for a file			*/
/*									*/
/************************************************************************/


static void
relocate_all()
{
   DL_FILE f;

   for (f = file_list; f != NULL; f = f->next) {
      relocate(f,0);
    };
};





static void
relocate(f,fst)
   DL_FILE f;
   Boolean fst;
{
   Integer i;

   for (i = 0; i < f->numseg; ++i) {
      f->segs[i].reloc_size =
	 relocate_seg(f,f->segs[i].base,f->segs[i].reloc,f->segs[i].reloc_size,
			 f->segs[i].offset,fst);
      if (f->segs[i].reloc != NULL && f->segs[i].reloc_size == 0) {
	 free(f->segs[i].reloc);
	 f->segs[i].reloc = NULL;
       };
    };
};




static Integer
relocate_seg(f,addr,reloc,cnt,off,fst)
   DL_FILE f;
   Address addr;
   DL_RELOC reloc;
   Integer cnt;
   Integer off;
   Boolean fst;
{
   Integer relocate_count;
   long tw,xtw,atw,ytw;
   Address cp,ncp;
   DL_ESYM sp;
   DL_ISYM nsp;
   struct section *seg;
   struct section *relocseg;
   String symnm;
   long   symvl;
   Boolean symudf;
   Integer i;
   String stub;
   DL_RELOC rp,nrp;
   RELOC_TYPE rtype;
   int new_count;
   int ndisp,nstore;

   if (cnt == 0 || reloc == NULL) return 0;

   relocate_count = cnt;
   new_count = 0;

   rp = reloc;
   nrp = reloc;

   while (relocate_count--) {
      ndisp = 0;
      nstore = 0;
      rtype = rp->dr_type;
      cp = addr + rp->dr_address;

      /*
       * Pick up previous value at location to be relocated.
       */

      if (rtype >= RTYPE_MACHINE_BASE) DL_x_machine_reloc(0,rtype,cp,&tw,&xtw);
      else switch (rtype) {
	 case RTYPE_8 :  /* byte */
	 case RTYPE_DISP8 :
	    xtw = 0;
	    tw = *cp;
	    break;

	 case RTYPE_16 : /* word */
	 case RTYPE_DISP16 :
	    xtw = 0;
	    tw = *(short *)cp;
	    break;

	 case RTYPE_32 : /* long */
	 case RTYPE_DISP32 :
	    xtw = 0;
	    tw = *(long *)cp;
	    break;

	 case RTYPE_WDISP30 :
	    tw = *(long *) cp;
	    xtw = tw & 0xc0000000;
	    tw &= 0x3fffffff;
	    tw <<= 2;
	    break;

	 case RTYPE_WDISP22 :
	    tw = *(long *) cp;
	    xtw = tw & 0xffc00000;
	    tw &= 0x3fffff;
	    tw <<= 2;
	    break;

	 case RTYPE_HI22 :
	 case RTYPE_22 :
	    tw = *(long *) cp;
	    xtw = tw & 0xffc00000;
	    tw &= 0x3fffff;
	    break;

	 case RTYPE_13 :
	 case RTYPE_LO10 :
	    tw = *(long *) cp;
	    xtw = tw & 0xffffe000;
	    tw &= 0x1fff;
	    break;

	 case RTYPE_HI16 :
	    ncp = addr + rp[1].dr_address;
	    tw = *(long *) cp;
	    xtw = tw & 0xffff0000;
	    tw &= 0xffff;
	    tw <<= 16;
	    ytw = *(long *) ncp;
	    ytw &= 0xffff;
	    if (ytw & 0x8000) tw -= 0x10000;
	    tw += ytw;
	    break;

	 case RTYPE_LO16 :
	    tw = *(long *) cp;
	    xtw = tw & 0xffff0000;
	    tw &= 0xffff;
	    break;

	 case RTYPE_DELTA8 :
	 case RTYPE_DSDL8 :
	    xtw = 0;
	    tw = *cp;
	    break;

	 case RTYPE_DELTA16 :
	 case RTYPE_DSDL16 :
	    xtw = 0;
	    tw = *(short *) cp;
	    tw = DL_x_swap(tw);
	    break;

	 case RTYPE_DELTA32 :
	 case RTYPE_DSDL32 :
	 case RTYPE_IMMED32 :
	 case RTYPE_SBREL32 :
	    xtw = 0;
	    tw = *(long *) cp;
	    tw = DL_x_swap(tw);
	    break;

	 default :
	    DL_error("relocate", "Unknown relocation type");
	    break;
       };

      /*
       * If relative to an external which is defined,
       * resolve it.  If the external is undefined, just
       * convert the symbol number to the number of the
       * symbol in the result file and leave it undefined.
       */

      atw = rp->dr_value;

      if (rp->dr_symbol) {
	 if (fst) {
	    sp = &f->syms[rp->dr_index];
	    symnm = sp->ds_name;
	    symvl = sp->ds_value;
	    symudf = (sp->ds_segment == SEGMENT_UNDEF);
	  }
	 else {
	    nsp = DL_symbol_at(rp->dr_index);
	    symnm = nsp->dn_name;
	    symvl = nsp->dn_value;
	    symudf = nsp->dn_undef;
	  };

	 if (symudf) {
	    nsp = DL_lookup(symnm,NULL);
	    if (nsp == NULL) DL_error("relocate: couldn't find", symnm);

	    new_count++;
	    *nrp = *rp;
	    nrp->dr_index = DL_symbol_index(nsp);
	    nrp->dr_value = 0;
	    ++nrp;
	    if (!fst) nstore = 1;

	    atw += (unsigned int) DL_ext_stub;
	  }
	 else {
	    if (!fst) {
	       atw -= (unsigned int) DL_ext_stub;
	       ndisp = 1;
	     }
	    else if (sp->ds_segment < f->numseg) {
	       atw -= f->segs[sp->ds_segment].poffset;
	     };
	    atw += symvl;
	  }
       }
      else if (rp->dr_index != SEGMENT_UNDEF &&
		  rp->dr_index != SEGMENT_ABS) {
	 atw += ((unsigned int) f->segs[rp->dr_index].base) -
	    f->segs[rp->dr_index].offset;
       }
      else {
	 DL_error("relocate","relocation format botch (symbol type)");
       }

      /*
       * Relocation is pc relative, so decrease the relocation
       * by the amount the current segment is displaced.
       * (E.g if we are a relative reference to a text location
       * from data space, we added the increase in the text address
       * above, and subtract the increase in our (data) address
       * here, leaving the net change the relative change in the
       * positioning of our text and data segments.)
       */

      if (!ndisp) {
	 switch (rtype) {
	    case RTYPE_DISP8 :
	    case RTYPE_DISP16 :
	    case RTYPE_DISP32 :
	    case RTYPE_WDISP30 :
	    case RTYPE_WDISP22 :
	    case RTYPE_DSDL8 :
	    case RTYPE_DSDL16 :
	    case RTYPE_DSDL32 :
	       atw -= (unsigned int) addr;
	       break;
	    default :
	       break;
	  };
       };

      if (rtype >= RTYPE_MACHINE_BASE) DL_x_machine_reloc(1,rtype,cp,&tw,&atw);
      else switch (rtype) {
	 default :
	    tw += atw;
	    break;

	 case RTYPE_HI22 :
	    tw += atw>>10;
	    break;

	 case RTYPE_LO10 :
	    tw += atw & 0x3ff;
	    break;

	 case RTYPE_HI16 :
	    tw += atw;
	    break;

	 case RTYPE_LO16 :
	    tw += atw & 0xffff;
	    break;

	 case RTYPE_DELTA8 :
	 case RTYPE_DSDL8 :
	    tw += atw;
	    tw &= 0x7f;
	    break;

	 case RTYPE_DELTA16 :
	 case RTYPE_DSDL16 :
	    tw += atw;
	    tw = DL_x_swap((tw&0x3fff) | 0x8000) >> 16;
	    break;

	 case RTYPE_DELTA32 :
	 case RTYPE_DSDL32 :
	    tw += atw;
	    tw = DL_x_swap((tw&0x3fffffff) | 0xc0000000);
	    break;

	 case RTYPE_IMMED32 :
	    tw += atw;
	    tw = DL_x_swap(tw);
	    break;

	 case RTYPE_SBREL32 :
	    tw += atw - (Integer) f->basereg;
	    tw = DL_x_swap((tw&0x3fffffff) | 0xc0000000);
	    break;
       };

      /*
       * Put the value back in the segment,
       * while checking for overflow.
       */

      if (!nstore) {
	 if (rtype >= RTYPE_MACHINE_BASE) DL_x_machine_reloc(2,rtype,cp,&tw,&xtw);
	 else switch (rtype) {
	    case RTYPE_8 :  /* byte */
	    case RTYPE_DISP8 :
	    case RTYPE_DELTA8  :
	    case RTYPE_DSDL8 :
	       tw &= 0xff;
	       *cp = tw;
	       break;

	    case RTYPE_16 : /* word */
	    case RTYPE_DISP16 :
	    case RTYPE_DELTA16 :
	    case RTYPE_DSDL16 :
	       tw &= 0xffff;
	       *(short *)cp = tw;
	       break;

	    case RTYPE_32:  /* long */
	    case RTYPE_DISP32 :
	    case RTYPE_DELTA32 :
	    case RTYPE_DSDL32 :
	    case RTYPE_IMMED32 :
	    case RTYPE_SBREL32 :
	       *(long *)cp = tw;
	       break;

	    case RTYPE_WDISP30 :
	       tw >>= 2;
	       tw &= 0x3fffffff;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_WDISP22 :
	       tw >>= 2;
	       tw &= 0x3fffff;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_HI22 :
	    case RTYPE_22 :
	       tw &= 0x3fffff;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_13 :
	       tw &= 0x1fff;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_LO10 :
	       tw &= 0x3ff;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_HI16 :
	       ytw = tw;
	       tw >>= 16;
	       tw &= 0xffff;
	       if (ytw & 0x8000) tw += 1;
	       *(long *)cp = tw|xtw;
	       break;

	    case RTYPE_LO16 :
	       tw &= 0xffff;
	       *(long *)cp = tw|xtw;
	       break;

	    default :
	       break;
	  }
       };

      rp++;
    }

   return new_count;
}




/************************************************************************/
/*									*/
/*	DL_find_reference -- find symbol for value for stub calls	*/
/*									*/
/************************************************************************/


String
DL_find_reference(val,offset)
   Address val;
   int offset;
{
   DL_FILE f;
   DL_RELOC rp;
   DL_ISYM dsp;
   char * nm;
   int relocate_count;

   for (f = file_list; f; f = f->next) {
      if (val < f->segs[0].base || val > f->segs[0].base+f->segs[0].length) continue;
      relocate_count = f->segs[0].reloc_size;
      rp = f->segs[0].reloc;
      while (relocate_count--) {
	 if (rp->dr_symbol) {
	    nm = NULL;
	    dsp = DL_symbol_at(rp->dr_index);
	    if (dsp != NULL) nm = dsp->dn_name;
	    if (nm != NULL) {
	       if ((rp->dr_address + f->segs[0].base) == val) {
		  DL__unresolvable_routine = nm;
		  DL__unresolvable_caller = f->fname;
		  return nm;
		}
	     }
	    else
	       DL_error("find_reference","Bad relocation record");
	  }
	 rp++;
       }
    }

   return DL_find_basis_reference(val,offset);
}




/************************************************************************/
/*									*/
/*	round -- utility rounding function				*/
/*									*/
/************************************************************************/


static Integer
round(v, r)
   int v;
   int r;
{

   r--;
   v += r;
   v &= ~(long)r;
   return(v);
}





/* end of dlloader.c */
