/************************************************************************/
/*									*/
/*		dlencore.c						*/
/*									*/
/*	Routines and definitions for ENCORE dynamic loading		*/
/*									*/
/************************************************************************/


#include "dl_local.h"

#include <ar.h>
#include <a.out.h>
#include <sgs.h>
#include <sys/file.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define AR_SYMDEF	"__.SYMDEF"




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		setup_segment();
static	void		setup_reloc();
static	void		setup_symbols();





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static	String	search_libs[32] = {
   "/lib", "/usr/lib", "/usr/local/lib", 0
};

static	Integer search_ct = -3;






/************************************************************************/
/*									*/
/*	DL_x_prepend_ -- indicate if underscore prepended to names	*/
/*									*/
/************************************************************************/


Boolean
DL_x_prepend_()
{
   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_test_archive -- test for archive file			*/
/*	DL_x_next_archive -- return offset for next archive element	*/
/*									*/
/************************************************************************/


Boolean
DL_x_test_archive(fi,offp)
   Integer fi;
   Integer * offp;
{
   Character magbuf[SARMAG];
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,0,L_SET);
   read(fi,magbuf,SARMAG);
   if (strncmp(magbuf,ARMAG,SARMAG) != 0) return FALSE;

   *offp = SARMAG + sizeof(hdr);

   read(fi,&hdr,sizeof(hdr));
   if (STREQL(hdr.ar_name,AR_SYMDEF)) {
      *offp += sizeof(hdr);
      len = atoi(hdr.ar_size);
      *offp += ((len+1) & ~1);
    };

   return TRUE;
};





Boolean
DL_x_next_archive(fi,ename,offp)
   Integer fi;
   String ename;
   Integer *offp;
{
   struct ar_hdr hdr;
   Integer len;

   lseek(fi,*offp-sizeof(hdr),L_SET);
   if (read(fi,&hdr,sizeof(hdr)) != sizeof(hdr)) return FALSE;

   *offp += sizeof(hdr);
   len = atoi(hdr.ar_size);
   *offp += ((len+1) & ~1);

   strncpy(ename,hdr.ar_name,sizeof(hdr.ar_name));

   return TRUE;
};





/************************************************************************/
/*									*/
/*	DL_x_setup_buffers -- read in data from a file			*/
/*									*/
/************************************************************************/


Boolean
DL_x_setup_buffers(dlf,fi,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer aoff;
{
   struct filehdr fhdr;
   struct aouthdr ahdr;
   struct scnhdr shdr[32];
   Integer i;
   long int stlen;

   lseek(fi,aoff,L_SET);
   if (read(fi,&fhdr,sizeof(fhdr)) <= 4) return FALSE;
   if (!ISMAGIC(fhdr.f_magic)) return FALSE;
   if (fhdr.f_opthdr > 0) {
      lseek(fi,aoff+sizeof(fhdr)+fhdr.f_opthdr,L_SET);
    };
   read(fi,shdr,sizeof(struct scnhdr)*fhdr.f_nscns);

   for (i = 0; i < fhdr.f_nscns; ++i) {
      if (shdr[i].s_scnptr != 0)
	 setup_segment(dlf,i,fi,shdr[i].s_size,aoff+shdr[i].s_scnptr,shdr[i].s_vaddr);
      else
	 setup_segment(dlf,i,-1,shdr[i].s_size,-1,shdr[i].s_vaddr);
      dlf->segs[i].poffset = shdr[i].s_paddr;
    };
   dlf->numseg = fhdr.f_nscns;

   dlf->common = NULL;
   dlf->basereg = NULL;

   for (i = 0; i < fhdr.f_nscns; ++i) {
      if (shdr[i].s_relptr != 0)
	 setup_reloc(dlf,i,fi,shdr[i].s_nreloc,aoff+shdr[i].s_relptr);
    };

   setup_symbols(dlf,fi,fhdr.f_nsyms,fhdr.f_symptr,
		    fhdr.f_symptr+(fhdr.f_nsyms*SYMESZ),aoff);

   return TRUE;
};



static void
setup_segment(dlf,seg,fi,len,addr,off)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer len;
   Integer addr;
   Integer off;
{
   Address rgn;

   if (len > 0) {
      rgn = (Address) malloc(len);
      if (fi >= 0) {
	 lseek(fi,addr,L_SET);
	 read(fi,rgn,len);
       }
      else {
	 bzero(rgn,len);
       };
    }
   else rgn = NULL;

   dlf->segs[seg].base = rgn;
   dlf->segs[seg].reloc = NULL;
   dlf->segs[seg].offset = off;
   dlf->segs[seg].poffset = 0;
   dlf->segs[seg].reloc_size = 0;
   dlf->segs[seg].length = len;
}





static void
setup_reloc(dlf,seg,fi,ct,addr)
   DL_FILE dlf;
   Integer seg;
   Integer fi;
   Integer ct;
   Integer addr;
{
   Address xrgn;
   DL_RELOC rel;
   Integer len,i;
   struct reloc * rp;

   if (ct <= 0) return;

   len = ct*RELSZ;

   rel = (DL_RELOC) calloc(ct,sizeof(DL_RELOC_INFO));
   xrgn = (Address) alloca(len);

   lseek(fi,addr,L_SET);
   read(fi,xrgn,len);

   for (i = 0; i < ct; ++i) {
      rp = (struct reloc *) (xrgn + RELSZ*i);
      rel[i].dr_address = rp->r_vaddr;
      rel[i].dr_value = 0;
      if (rp->r_symndx < 0) {
	 rel[i].dr_symbol = FALSE;
	 rel[i].dr_index = i;
       }
      else {
	 rel[i].dr_symbol = TRUE;
	 rel[i].dr_index = rp->r_symndx;
       };

      switch (rp->r_type) {
	 case R_NOTHING :
	    rel[i].dr_type = RTYPE_NONE;
	    break;
	 case R_RELBYTE :
	    rel[i].dr_type = RTYPE_8;
	    break;
	 case R_RELWORD :
	    rel[i].dr_type = RTYPE_16;
	    break;
	 case R_RELLONG :
	    rel[i].dr_type = RTYPE_32;
	    break;
	 case R_PCRBYTE :
	    rel[i].dr_type = RTYPE_DISP8;
	    break;
	 case R_PCRWORD :
	    rel[i].dr_type = RTYPE_DISP16;
	    break;
	 case R_PCRLONG :
	    rel[i].dr_type = RTYPE_DISP32;
	    break;
	 case R_RRELBYTE :
	    rel[i].dr_type = RTYPE_DELTA8;
	    break;
	 case R_RRELWORD :
	    rel[i].dr_type = RTYPE_DELTA16;
	    break;
	 case R_RRELLONG :
	    rel[i].dr_type = RTYPE_DELTA32;
	    break;
	 case R_RPCRBYTE :
	    rel[i].dr_type = RTYPE_DSDL8;
	    break;
	 case R_RPCRWORD :
	    rel[i].dr_type = RTYPE_DSDL16;
	    break;
	 case R_RPCRLONG :
	    rel[i].dr_type = RTYPE_DSDL32;
	    break;
	 case R_IMMED | R_S_FOUR :
	    rel[i].dr_type = RTYPE_IMMED32;
	    break;
	 case R_SBREL | R_DISPL | R_S_FOUR :
	    rel[i].dr_type = RTYPE_SBREL32;
	    break;
	 default :
	    rel[i].dr_type = RTYPE_NONE;
	    break;
       };
    };

   dlf->segs[seg].reloc = rel;
   dlf->segs[seg].reloc_size = ct;
};





static void
setup_symbols(dlf,fi,ct,addr,straddr,aoff)
   DL_FILE dlf;
   Integer fi;
   Integer ct;
   Integer addr;
   Integer straddr;
   Integer aoff;
{
   struct syment * sp;
   Address xbuf;
   DL_ESYM sym;
   Integer len,stlen,i;
   String eptr;
   Character ebuf[10];

   len = ct * SYMESZ;

   if (ct == 0) {
      dlf->numsym = 0;
      dlf->strings = NULL;
      dlf->syms = NULL;
      return;
    };

   xbuf = (Address) alloca(len);
   lseek(fi,addr+aoff,L_SET);
   read(fi,xbuf,len);

   lseek(fi,straddr+aoff,L_SET);
   if (read(fi,&stlen,4) != 4) {
      dlf->strings = (String) malloc(stlen+ct*9);
      lseek(fi,straddr+aoff,L_SET);
      read(fi,dlf->strings,stlen);
    }
   else {
      dlf->strings = (String) malloc(4+ct*9);
      stlen = 4;
    };
   eptr = &dlf->strings[stlen];

   sym = (DL_ESYM) calloc(sizeof(DL_ESYM_INFO),ct);
   for (i = 0; i < ct; ++i) {
      sp = (struct syment *) (xbuf + SYMESZ*i);
      if (sp->n_zeroes == 0) {
	 sym[i].ds_name = &dlf->strings[sp->n_offset];
       }
      else {
	 strncpy(ebuf,sp->n_name,SYMNMLEN);
	 ebuf[SYMNMLEN] = 0;
	 strcpy(eptr,ebuf);
	 sym[i].ds_name = eptr;
	 eptr += strlen(ebuf)+1;
       };
      sym[i].ds_value = sp->n_value;
      sym[i].ds_extern = (sp->n_sclass == C_EXT || sp->n_sclass == C_EXTDEF);
      switch (sp->n_scnum) {
	 case N_UNDEF :
	    sym[i].ds_segment = SEGMENT_UNDEF;
	    break;
	 case N_ABS :
	    sym[i].ds_segment = SEGMENT_ABS;
	    break;
	 default :
	    if (sp->n_scnum > 0) sym[i].ds_segment = sp->n_scnum-1;
	    else sym[i].ds_segment = SEGMENT_IGNORE;
	    break;
       };
    };

   dlf->numsym = ct;
   dlf->syms = sym;
};





/************************************************************************/
/*									*/
/*	DL_x_machine_reloc -- handle machine relocation types		*/
/*									*/
/************************************************************************/


void
DL_x_machine_reloc(id,rty,ptr,tp,vp)
   Integer id;
   RELOC_TYPE rty;
   Character * ptr;
   Integer * tp;
   Integer * vp;
{
   /* NOT USED */
};





/************************************************************************/
/*									*/
/*	DL_x_swap -- ns32000 byte swapping stubs			*/
/*									*/
/************************************************************************/


Integer
DL_x_swap(uli)
   unsigned long uli;
{
   Integer x;

   x = ((uli<<24) | (uli<<8)&0xFF0000 | (uli>>8)&0xFF00 | (uli>>24));

   return x;
};




/************************************************************************/
/*									*/
/*	DL_x_nm_command -- generate nm command on a file		*/
/*	DL_x_cc_command -- generate cc command on a file		*/
/*	DL_x_ld_command -- generate ld command on 2 files		*/
/*	DL_x_as_command -- generate as command on 2 files		*/
/*									*/
/************************************************************************/


void
DL_x_nm_command(buf,file,tblfg)
   String buf;
   String file;
   Boolean tblfg;		/* TRUE if for building tables		*/
				/* FALSE if for dynload symbols 	*/
{
   if (tblfg) {
      sprintf(buf,"nm -e %s",file);
    }
   else {
      sprintf(buf,"nm -he %s",file);
    };
};





void
DL_x_cc_command(buf,opts,file)
   String buf;
   String opts;
   String file;
{
   sprintf(buf,"cc %s -q nodirect_code -Wc,-Z151 -w %s",opts,file);
};





void
DL_x_ld_command(buf,outfile,f1,f2,dbg,args)
   String buf;
   String outfile;
   String f1;
   String f2;
   Boolean dbg;
   String args;
{
   sprintf(buf,"ld -o %s %s %s%s",outfile,f1,f2,(dbg ? " -lg" : ""));
};





void
DL_x_as_command(buf,src,tgt)
   String buf;
   String src;
   String tgt;
{
   sprintf(buf,"as -o %s %s",tgt,src);
};





/************************************************************************/
/*									*/
/*	DL_x_nm_read -- read a nm line -- return true if undefined sym	*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_read(buf,np,vp,dfg,txtp)
   String buf;
   String * np;
   Integer * vp;
   Boolean dfg;
   Boolean * txtp;
{
   String p;
   Character t;
   Integer v,s,i;
   Boolean fg,tfg;

   *np = NULL;

   p = buf;
   for (p = buf; *p != '|' && *p != 0; ++p) {
      if (*p == ' ') *p = 0;
    };
   if (*p == 0) return TRUE;
   *p++ = 0;

   v = 0;
   s = 1;
   while (*p != '|' && *p != 0) {
      if (isdigit(*p)) v = v*10 + (*p - '0');
      else if (*p == '-' && v == 0) s = -1;
      ++p;
    };
   if (*p++ == 0) return TRUE;
   if (*p == 's') return TRUE;          /* ignore static symbols for now        */
   *vp = v*s;

   for (i = 0; i < 4; ++i) {
      while (*p != '|' && *p != 0) ++p;
      if (*p++ == 0) break;
    };
   if (i < 4) return TRUE;

   *np = buf;

   fg = (*p == ' ' && v == 0);
   tfg = (*p == '.' && p[1] == 't');

   if (txtp != NULL) *txtp = tfg;


   return fg;
};






/********************************************************************************/
/*										*/
/*	DL_x_other_syms -- extra symbols					*/
/*										*/
/********************************************************************************/


String
DL_x_other_syms()
{
   return NULL;
};





/************************************************************************/
/*									*/
/*	DL_x_nm_element -- check for new element in archive		*/
/*	DL_x_nm_extdata -- check for external data definition		*/
/*									*/
/************************************************************************/


Boolean
DL_x_nm_element(buf,elt)
   String buf;
   String elt;
{
   if (strncmp(buf,"Symbols from ",13) == 0) {
      buf = index(buf,'[');
      strcpy(elt,&buf[1]);
      buf = index(elt,']');
      *buf = 0;
      return TRUE;
    };

   return FALSE;
};





Boolean
DL_x_nm_extdata(buf)
   String buf;
{
   Integer i;

   if (buf[0] == ' ') return FALSE;

   i = 0;
   while (i < 6) {
      if (*buf == 0) break;
      if (*buf == '|') ++i;
      ++buf;
    };

   return (strncmp(buf,".text",5) == 0 || strncmp(buf,".data",5) == 0);
};





/************************************************************************/
/*									*/
/*	DL_x_ignore_symbol -- indicate if symbol should be ignored	*/
/*									*/
/************************************************************************/


Boolean
DL_x_ignore_symbol(nm)
   String nm;
{
   if (nm[0] != '_') return TRUE;

   return FALSE;
};





/********************************************************************************/
/*										*/
/*	DL_x_lib_name -- get library name					*/
/*										*/
/********************************************************************************/


void
DL_x_lib_name(buf)
   char * buf;
{
   String s;
   String a;
   Integer i;

   s = DL_pro_dir();
   a = DL_arch();

   for (i = search_ct-1; i >= 0; --i) {
      sprintf(buf,"%s/libdl.a",search_libs[i]);
      if (access(buf,4) >= 0) {
	 sprintf(buf,"-ldl");
	 return;
       };
      if (a != NULL) {
	 sprintf(buf,"%s/libdl_%s.a",search_libs[i],a);
	 if (access(buf,4) >= 0) {
	    sprintf(buf,"-ldl");
	    return;
	  };
       }
    };

   if (getenv("DL_LIBRARY") != NULL) {
      strcpy(buf,getenv("DL_LIBRARY"));
      return;
    };

   buf[0] = 0;
   if (s != NULL && a != NULL) {
      sprintf(buf,"%s/lib/dynlib.%s.o",s,a);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && s != NULL) {
      sprintf(buf,"%s/lib/dynlib.o",s);
      if (access(buf,4) < 0) buf[0] = 0;
    };
   if (buf[0] == 0 && a != NULL) {
      sprintf(buf,"/usr/local/lib/dynlib.%s.o",a);
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.%s.o",a);
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.%s.o",a);
	    if (access(buf,4) < 0) buf[0] = 0;
	  };
       };
    }

   if (buf[0] == 0) {
      sprintf(buf,"/usr/local/lib/dynlib.o");
      if (access(buf,4) < 0) {
	 sprintf(buf,"/cs/lib/dynlib.o");
	 if (access(buf,4) < 0) {
	    sprintf(buf,"/pro/lib/dynlib.o");
	    if (access(buf,4) < 0) sprintf(buf,"/usr/lib/dynlib.o");
	  };
       };
    };
};










/********************************************************************************/
/*										*/
/*	DL_x_check_arg -- special processing of arguments			*/
/*										*/
/********************************************************************************/


Boolean
DL_x_check_arg(arg)
   String arg;
{
   return FALSE;
};





/************************************************************************/
/*									*/
/*	dummy routine to force loading of needed things 		*/
/*									*/
/************************************************************************/


static void
dummy_routine()
{
   int a,b,c;
   float d,e,f;
   double g,h,i;

   a = b*c;
   b = a%c;
   strcmp(0,1); 		/* force strcmp to be available */
   d = d+e*f;
   d = d/f-e;
   a = d;
   e = b;
   g = g*h+i-g/h;
   e = g;
   h = e;
   b = g;
   h = a;
   vfork(a,b,c,d,e,f);
};






/************************************************************************/
/*									*/
/*	DL_x_gen_stub -- generate assembler code for a stub		*/
/*									*/
/************************************************************************/


Boolean
DL_x_gen_stub(otf,nm,idx)
   FILE * otf;
   String nm;
   Integer idx;
{
   if (nm == NULL) {
      if (idx == 0) {
	 fprintf(otf,"   .text\n");
	 fprintf(otf,"   .align 4\n");
       }
      else {
	 fprintf(otf,"DL_stub_handler:\n");
	 fprintf(otf,"   movd _DL__stub_branch_table(r0),r1\n");
	 fprintf(otf,"   jump r1\n\n");
	 fprintf(otf,"   .globl _DL_stub_routine\n");
	 fprintf(otf,"   .globl _DL_stub\n");
	 fprintf(otf,"   .globl _DL_ext_stub\n");
	 fprintf(otf,"_DL_ext_stub:\n");
	 fprintf(otf,"   movd $%d,r0\n",(idx-1)*sizeof(DL_LINK_INFO));
	 fprintf(otf,"_DL_stub:\n");
	 fprintf(otf,"   movd r0,tos\n");
	 fprintf(otf,"   movd 4(sp),r0\n");
	 fprintf(otf,"   addqd $-4,r0\n");
	 fprintf(otf,"   movd r0,tos\n");
	 fprintf(otf,"   bsr ?_DL_stub_routine\n");
	 fprintf(otf,"   adjspb $-8\n");
	 fprintf(otf,"   jump r0\n");
       };
      return;
    };

   fprintf(otf,"  .globl %s\n",nm);
   fprintf(otf,"%s:\n",nm);
   fprintf(otf,"   movd $%d,r0\n",idx*sizeof(DL_LINK_INFO));
   fprintf(otf,"   br DL_stub_handler\n");

   fprintf(otf,"\n");
};







/* end of dlencore.c */
