/************************************************************************/
/*									*/
/*		dlcommon.c						*/
/*									*/
/*	Dynamic loading routines not involved with normal loading	*/
/*									*/
/************************************************************************/
/*	Copyright 1987 Brown University -- Steven P. Reiss		*/


#include "dl_local.h"





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/

typedef DL_FunctionPtr	(*DL_FFunctionPtr)();

static struct {
   DL_FunctionPtr (*name_heuristic)();
   DL_FunctionPtr (*other_heuristic)();
} load_on_demand;

       String DL__unresolvable_routine = NULL;
       String DL__unresolvable_caller = NULL;



/************************************************************************/
/*									*/
/*	Forward references						*/
/*									*/
/************************************************************************/


static DL_FunctionPtr	load_by_name();
static DL_FunctionPtr	DLsystem_names();





/************************************************************************/
/*									*/
/*	DL_load_init -- initialize dynamic loader routines		*/
/*									*/
/************************************************************************/


void
DL_load_init()
{
   Character buf[256], host[256],lbbf[256];
   FILE * p;
   String s,pro,arch,libs,lib;
   Integer i;

   DL_load_subinit();

   load_on_demand.name_heuristic = load_by_name;
   load_on_demand.other_heuristic = NULL;

   lib = (String) getenv("DL_SYSNAMES_LIBRARY");
   if (lib != NULL && DLload(lib)) {
      load_on_demand.other_heuristic = (DL_FFunctionPtr) DLlookup("DLsystem_names");
      return;
    };

   gethostname(host, sizeof(host));
   pro = DL_pro_dir();
   arch = DL_arch();

   if (pro == NULL) libs = "/usr/local";
   else libs = pro;

   for ( ; ; ) {
      i = strlen(libs);
      if (STREQL("/src",&libs[i-4])) {
	 strcpy(lbbf,libs);
	 lbbf[i-4] = 0;
	 strcat(lbbf,"/lib");
	 libs = lbbf;
       };

      sprintf(buf, "%s/lib/dl_sysnames.%s.o", libs, host);
      if (DLload(buf)) break;

      s = DL_arch();
      if (s != NULL) strcpy(host,s);
      else host[0] = 0;
      sprintf(buf,"%s/lib/dl_sysnames.%s.o",libs, host);
      if (DLload(buf)) break;

      sprintf(buf,"%s/lib/dl_sysnames.o",libs);
      if (DLload(buf)) break;

      else if (libs == pro) libs = "/usr/local";
      else if (STREQL(libs,"/usr/local")) libs = "/cs/lib";
      else if (STREQL(libs,"/cs/lib")) libs = "/pro";
      else return;
    };

   load_on_demand.other_heuristic = (DL_FFunctionPtr) DLlookup("DLsystem_names");
};





/************************************************************************/
/*									*/
/*	DLload_on_demand_heuristics -- set heuristics			*/
/*									*/
/************************************************************************/


void
DLload_on_demand_heuristics(first, second)
   DL_FunctionPtr (*first)();
   DL_FunctionPtr (*second)();
{
   DLinit();

   if (first != NULL) load_on_demand.name_heuristic = first;
   if (second != NULL) load_on_demand.other_heuristic = second;
}





/************************************************************************/
/*									*/
/*	DL_fill_in_linkage_seg -- fill linkage segment with new value	*/
/*									*/
/************************************************************************/


void
DL_fill_in_linkage_seg(olds,val)
   DL_ISYM olds;
   Address val;
{
   int i;
   int sym_id_offset;
   int sym_id;
   DL_LINK_INFO * link;

   sym_id = DL_symbol_index(olds);

   for (i = 0; i < DL__esd->num_linkage; i++) {
      link = &DL__stub_branch_table[i];
      if (sym_id == link->symid) {
	 link->addr = val;
	 return;
       };
    }

   DL_error("fill_in_linkage_seg: Couldn't find linkage block for",
	       olds->dn_name);
}




/************************************************************************/
/*									*/
/*	DL_unable_to_resolve -- print error message at call time	*/
/*									*/
/************************************************************************/


#define NULL_NAME(N) ((N) == NULL ? "<name unknown>" : (N))


static void
DL_unable_to_resolve()
{
   printf("DL: Unable to resolve reference to %s from %s\n",
	     NULL_NAME(DL__unresolvable_routine),
	     NULL_NAME(DL__unresolvable_caller));
   DL__unresolvable_routine = DL__unresolvable_caller = NULL;
}





/************************************************************************/
/*									*/
/*	DL_stub_routine -- entered on call to undefined routine 	*/
/*									*/
/************************************************************************/


DL_FunctionPtr
DL_stub_routine(caller,offset)
   int caller;
   int offset;
{
   String name;
   Integer nval,i;
   DL_ISYM n;
   DL_HashKey loc;
   DL_FunctionPtr resolution = NULL;

   if (!DL__inited) DLinit();

   name = DL_find_reference(caller,offset);

   if (name != NULL) {
      n = DL_lookup(name,&loc);
      if (n != NULL && !n->dn_undef && n->dn_value != 0)
	 return (DL_FunctionPtr) n->dn_value;

      if (load_on_demand.name_heuristic != NULL)
	 resolution = (*load_on_demand.name_heuristic)(name);
      if (resolution == NULL && load_on_demand.other_heuristic != NULL)
	 resolution = (*load_on_demand.other_heuristic)(name);
      if (resolution != NULL)
	 return resolution;
    };

   return (DL_FunctionPtr) DL_unable_to_resolve;
}




/************************************************************************/
/*									*/
/*	DL_find_basis_reference -- find reference in original program	*/
/*									*/
/************************************************************************/


char *
DL_find_basis_reference(val,offset)
   int *val;
   int offset;
{
   DL_ISYM n;
   Integer i;

   DL__unresolvable_caller = "the basis file";

   i = offset / sizeof(DL_LINK_INFO);
   if (i >= 0 && i < DL__esd->num_linkage) {
      i = DL__stub_branch_table[i].symid;
      n = &DL__esd->sym[i];
      DL__unresolvable_routine = n->dn_name;
      return n->dn_name;
    };

   return NULL;
}





/************************************************************************/
/*									*/
/*	DL_pro_dir -- get project directory				*/
/*	DL_arch -- get architecture name				*/
/*									*/
/************************************************************************/


char *
DL_pro_dir()
{
   String s;

   s = (String) getenv("DL_DIR");
   if (s == NULL) s = (String) getenv("PRO");

#ifdef DL_DIR
   if (s == NULL) s = DL_DIR;
#endif

   return s;
};






char *
DL_arch()
{
   String s;

   s = (String) getenv("DL_ARCH");
   if (s == NULL) s = (String) getenv("ARCH");

#ifdef DL_ARCH
   if (s == NULL) s = DL_ARCH;
#endif

   return s;
};






/************************************************************************/
/*									*/
/*	load_by_name -- load a routine using search heuristics		*/
/*									*/
/* Takes a string containing the name of a function which needs 	*/
/* to be loaded, and loads it if it is a standard package.		*/
/* (standard packages are those that obey the graphics group's standards*/
/* or the BWE software standards) Specifically, if we are given 	*/
/* _SQUID_1mung, we attempt to load /pro/lib/libsquid_1.a or		*/
/* /pro/squid_1/lib/squid_1.lib or /pro/bags/lib/libsquid_1.a		*/
/* or /usr/local/lib/libsquid_q.a					*/
/*									*/
/************************************************************************/


static DL_FunctionPtr
load_by_name(name)
   String name;
{
   Character libname[1024];
   Character package[1024];
   String p,pro,bags_pro,bags_ao;
   String n;

   pro = DL_pro_dir();
   if (pro == NULL) pro = "/pro";

   bags_pro = (String) getenv("BAGS_ROOT");
   if (bags_pro == NULL) bags_pro = "/pro/bags";
   bags_ao = (String) getenv("BAGS_AO");
   if (bags_ao == NULL) bags_ao = "*";

   if (*name == '_') name++;

   p = package;
   for (n = name; !islower(*n) && *n != '\0'; n++) {
      *p++ = isupper(*n) ? tolower(*n) : *n;
    }
   *p = '\0';

   if (package[0] == 0) return NULL;

   sprintf(libname,"%s/lib/lib%s.a",pro,package);
   if (DLload(libname))
      return DLlookup(name);

   sprintf(libname,"%s/%s/lib/%s.lib",pro,package,package);
   if (DLload(libname))
      return DLlookup(name);

   sprintf(libname,"%s/lib/%s/O/lib%s.a",bags_pro,bags_ao,package);
   if (DLload(libname))
      return DLlookup(name);

   sprintf(libname,"/usr/local/lib/lib%s.a",package);
   if (DLload(libname))
      return DLlookup(name);

   return NULL;
}





/* end of dlcommon.c */
