/************************************************************************/
/*									*/
/*		build_table.c						*/
/*									*/
/*	Build the system library index table for a machine		*/
/*									*/
/************************************************************************/

/********************************************************************************/
/*										*/
/* This generates several static arrays that are used within DL heuristics	*/
/* code to load .o files from system libraries.  It reads on stdin the		*/
/* output of "nm" for each of the system libraries.  This input is created      */
/* within the "make_look_and_load" script that calls this program.              */
/*										*/
/* Only library names (lines beginning with "L"), .o file names (lines with     */
/* colons), and defined routines and variables ("T" and "D" symbols) are        */
/* included as symbols in the table.						*/
/*										*/
/* The output is:  an array of library names, an array of .o file names,	*/
/* an array of symbols, an array of triples of indices into the first three	*/
/* arrays, and a hash table.  The hash table takes the hashed value of a	*/
/* symbol and returns the index into the array of triples.			*/
/*										*/
/********************************************************************************/


#include "datatypes.h"
#include "dl_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define LIBRARY_STRING	"@LIBRARY@"
#define MAXLINE 	256

#define NAMELEN 	64	/* Max length of lib, file, or symbol name */
#define MAXLIBS 	10	/* Max number of libs			   */
#define MAXOFILES	1000	/* Max number of total .o files 	   */
#define MAXSYMBOLS	10000	/* Max number of total symbols		   */

/* Index into quad array */
#define SYMBOL		0
#define OFILE		1
#define LIB		2

#define HASHSIZE	1024





/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct Quad Quad;

struct Quad {
   int	hash, symbol, ofile, lib;
};




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static char	libs[MAXLIBS][NAMELEN],
		ofiles[MAXOFILES][NAMELEN],
		symbols[MAXSYMBOLS][NAMELEN];

static int	cur_lib = -1,
		cur_ofile = -1,
		cur_symbol = -1,
		hashtab[HASHSIZE];

static Quad	quad[MAXSYMBOLS];





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


extern void	exit();
extern char	*strcpy();





/************************************************************************/
/*									*/
/*	Main program							*/
/*									*/
/************************************************************************/


main(argc,argv)
   Integer argc;
   String argv[];
{
   Character tmpfile[16],cmd[128],buf[MAXLINE],vname[128];
   Integer i,len,libln;
   FILE * fp;
   String np;

   sprintf(tmpfile,"/tmp/DLXXXXXX");
   mktemp(tmpfile);

   for (i = 1; i < argc; ++i) {
      sprintf(cmd,"echo %s %s >>%s",LIBRARY_STRING,argv[i],tmpfile);
      system(cmd);
      DL_x_nm_command(cmd,argv[i],TRUE);
      strcat(cmd," >> ");
      strcat(cmd,tmpfile);
      system(cmd);
    };

   if ((fp = fopen(tmpfile, "r")) == NULL) {
      fprintf(stderr,"build_table: can't open tmp file `%s'", tmpfile);
      exit (1);
    }

   libln = strlen(LIBRARY_STRING);

   while (fgets(buf,MAXLINE,fp) != NULL) {
      len = strlen(buf);
      if (buf[len-1] == '\n') buf[--len] = 0;
      if (len <= 0) continue;

      if (strncmp(buf,LIBRARY_STRING,libln) == 0) set_lib(&buf[libln+1]);
      else if (DL_x_nm_element(buf,vname)) {
	 set_ofile(vname);
       }
      else if (DL_x_nm_extdata(buf) && !DL_x_nm_read(buf,&np,&i,FALSE,NULL) && np != NULL) {
	 add_symbol(np);
       };
   }

   fclose(fp);
   unlink(tmpfile);

   make_hash();
   output();
   exit(0);
}



/************************************************************************/
/*									*/
/*	set_lib -- start a new library					*/
/*									*/
/************************************************************************/


set_lib(libname)
char *libname;
{
   if (++cur_lib >= MAXLIBS) {
      fprintf(stderr, "build_table: Maximum number of libs (%d) exceeded\n",
		 MAXLIBS);
      fprintf(stderr, "             Aborting at library %s\n", libname);
      exit(1);
   }
   (void) strcpy(libs[cur_lib], libname);
}




/************************************************************************/
/*									*/
/*	set_ofile -- start a new .o file				*/
/*									*/
/************************************************************************/


set_ofile(ofile)
char *ofile;
{
   if (++cur_ofile >= MAXOFILES) {
      fprintf(stderr, "build_table: Maximum number of ofiles (%d) exceeded\n",
		 MAXOFILES);
      fprintf(stderr, "             Aborting at file %s in library %s\n",
		 ofile, libs[cur_lib]);
      exit(1);
   }
   (void) strcpy(ofiles[cur_ofile], ofile);
}





/************************************************************************/
/*									*/
/*	add_symbol -- add a symbol					*/
/*									*/
/************************************************************************/


add_symbol(symbol)
char *symbol;
{
   if (++cur_symbol >= MAXSYMBOLS) {
      fprintf(stderr, "build_table: Maximum number of symbols (%d) exceeded\n",
		 MAXSYMBOLS);
      fprintf(stderr,
		 "            Aborting at symbol %s in file %s in library %s\n",
		 symbol, ofiles[cur_ofile], libs[cur_lib]);
      exit(1);
   }
   (void) strcpy(symbols[cur_symbol], symbol);
   quad[cur_symbol].hash   = hashit(symbol);
   quad[cur_symbol].symbol = cur_symbol;
   quad[cur_symbol].ofile  = cur_ofile;
   quad[cur_symbol].lib    = cur_lib;
}




/************************************************************************/
/*									*/
/*	hashit -- hash function 					*/
/*									*/
/************************************************************************/
/*
 * NOTE: This function is used also in DL to hash symbols.  DO NOT CHANGE
 *	 THIS FUNCTION, unless you also change it in DL source.
 */

int
hashit(name)
char *name;
{
   register int 	i = 0;
   register char	*c;

   for (c = name; *c; c++) i += *c;

   return(i % HASHSIZE);
}





/************************************************************************/
/*									*/
/*	comp_quad -- quadratic hash computation 			*/
/*									*/
/************************************************************************/


static int
comp_quad(qp1, qp2)
Quad *qp1, *qp2;
{
   register int diff;

   if (diff = qp1->hash - qp2->hash) return(diff);

   /* If names are the same, use the library # to resolve order, assuming
      the libraries are given in the correct order.  This allows libnm to
      be loaded before libm. */

   return(qp1->lib - qp2->lib);
}




/************************************************************************/
/*									*/
/*	make_hash -- make the hash table				*/
/*									*/
/************************************************************************/


make_hash()
{
   register int i, prevhash = -1;

   (void) qsort((char *) quad, cur_symbol + 1, sizeof(Quad), comp_quad);

   for (i = 0; i < HASHSIZE; i++) hashtab[i] = -1;

   for (i = 0; i <= cur_symbol; i++) {
      if (quad[i].hash != prevhash) hashtab[quad[i].hash] = i;
      prevhash = quad[i].hash;
   }
}



/************************************************************************/
/*									*/
/*	output -- output the result					*/
/*									*/
/************************************************************************/


output()
{
   register int i;

   printf("/* This file is generated by the build_table program from\n");
   printf("   within make_look_and_load.  Do NOT attempt to edit this. */\n\n");

   printf("#define NUMLIBS\t%d\n",    cur_lib    + 1);
   printf("#define NUMOFILES\t%d\n",  cur_ofile  + 1);
   printf("#define NUMSYMBOLS\t%d\n", cur_symbol + 1);
   printf("#define HASHSIZE\t%d\n",   HASHSIZE);

   printf("#define SYMBOL\t%d\n", SYMBOL);
   printf("#define OFILE\t%d\n",  OFILE);
   printf("#define LIB\t%d\n",    LIB);

   printf("static char *DL_libs[NUMLIBS] = {\n");
   for (i = 0; i <= cur_lib; i++) printf("   /* %4d */ \"%s\",\n", i, libs[i]);
   printf("};\n\n");

   printf("static char *DL_ofiles[NUMOFILES] = {\n");
   for (i = 0; i <= cur_ofile; i++)
      printf("   /* %4d */ \"%s\",\n", i, ofiles[i]);
   printf("};\n\n");

   printf("static char *DL_symbols[NUMSYMBOLS] = {\n");
   for (i = 0; i <= cur_symbol; i++)
      printf("   /* %4d */ \"%s\",\n", i, symbols[i]);
   printf("};\n\n");

   printf("static int DL_triples[NUMSYMBOLS][3] = {\n");
   for (i = 0; i <= cur_symbol; i++)
      printf("   /* %4d */ { %4d, %4d, %4d },\n",
		i, quad[i].symbol, quad[i].ofile, quad[i].lib);
   printf("};\n\n");

   printf("static int DL_hashtab[HASHSIZE] = {\n");
   for (i = 0; i < HASHSIZE; i++)
      printf("   /* %4d */ %4d,\n", i, hashtab[i]);
   printf("};\n\n");
}





/************************************************************************/
/*									*/
/*	stubs								*/
/*									*/
/************************************************************************/


DL_FunctionPtr
DL_stub_routine()
{ return NULL; }






char *
DL_pro_dir()
{
   String s;

   s = (String) getenv("DL_DIR");
   if (s == NULL) s = (String) getenv("PRO");

#ifdef DL_DIR
   if (s == NULL) s = DL_DIR;
#endif

   return s;
};






char *
DL_arch()
{
   String s;

   s = (String) getenv("DL_ARCH");
   if (s == NULL) s = (String) getenv("ARCH");

#ifdef DL_ARCH
   if (s == NULL) s = DL_ARCH;
#endif

   return s;
};








/* end of build_table.c */
