/************************************************************************/
/*									*/
/*		wormtype.c						*/
/*									*/
/*	Type management routines for WORM object manager		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "worm_local.h"
#include <varargs.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_SUPER	1024



/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	WORM_TYPE	all_types;

	WORM_TYPE	WORM__type_Any;
	WORM_TYPE	WORM__type_Type;



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	WORM_type_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
WORM_type_init()
{
   all_types = NULL;
   WORM__type_Any = NULL;
   WORM__type_Type = NULL;
};





/************************************************************************/
/*									*/
/*	WORMtype_define -- define a new type				*/
/*		(name,data_len,default,numsuper,sup1,...,supn)		*/
/*									*/
/************************************************************************/


WORM_TYPE
WORMtype_define(va_alist)
   va_dcl
{
   va_list ap;
   WORM_TYPE wt,st,ct;
   String name;
   Universal * df;
   WORM_TYPE sups[MAX_SUPER];
   Integer ns,tots,i,j,k;

   va_start(ap);

   WORM_init();

   name = va_arg(ap,String);

   for (wt = all_types; wt != NULL; wt = wt->next) {
      if (STREQL(name,wt->name)) {
	 WORM_error("duplicate type %s",name);
	 return NULL;
       };
    };

   wt = PALLOC(WORM_TYPE_INFO);

   wt->header.type = WORM__type_Type;
   wt->name = name;
   wt->local_len = va_arg(ap,Integer);

   df = va_arg(ap,Universal *);
   if (df == NULL) wt->dflt = NULL;
   else {
      wt->dflt = (Universal *) malloc(wt->local_len);
      bcopy(df,wt->dflt,wt->local_len);
    };

   tots = va_arg(ap,Integer);
   ns = 0;
   for (i = 0; i < tots; ++i) {
      st = va_arg(ap,WORM_TYPE);
      if (st == WORM__type_Any || st == NULL) continue;
      for (j = 0; j < ns; ++j) {
	 if (sups[j] == st) break;
       };
      if (j >= ns) {
	 sups[ns++] = st;
	 for (k = 0; k < st->numsuper; ++k) {
	    ct = st->super[k].type;
	    if (ct == WORM__type_Any) continue;
	    for (j = 0; j < ns; ++j) {
	       if (sups[j] == ct) break;
	     };
	    if (j >= ns) sups[ns++] = ct;
	  };
       };
    };
   if (WORM__type_Any != NULL) {
      sups[ns++] = WORM__type_Any;
    };

   wt->numsuper = ns;
   if (ns != NULL) {
      wt->super = (WORM_SUPER) calloc(sizeof(WORM_SUPER_INFO),ns);
      for (i = 0; i < ns; ++i) {
	 wt->super[i].type = sups[i];
	 wt->super[i].offset = 0;
       };
    }
   else {
      wt->super = NULL;
    };

   wt->numop = 0;
   wt->op = NULL;

   k = 0;
   wt->data_offset = k;
   k += (wt->local_len+3)/4;

   for (i = 0; i < ns; ++i) {
#ifdef DOUBLE_PAD
      if ((k&1) != 0) ++k;
#endif
      wt->super[i].offset = k;
      k += (wt->super[i].type->local_len+3)/4;
    };

   wt->total_len = k;

   PROTECT(
      wt->next = all_types;
      all_types = wt;
    );

   WORM_op_setup(wt);

   return wt;
};





/************************************************************************/
/*									*/
/*	WORMtype_find -- find type given name				*/
/*									*/
/************************************************************************/


WORM_TYPE
WORMtype_find(name)
   String name;
{
   WORM_TYPE wt;

   WORM_init();

   for (wt = all_types; wt != NULL; wt = wt->next) {
      if (STREQL(name,wt->name)) break;
    };

   return wt;
};





/************************************************************************/
/*									*/
/*	WORMinstance -- create a new object of given type		*/
/*									*/
/************************************************************************/


WORM_OBJ
WORMinstance(wt)
   WORM_TYPE wt;
{
   WORM_OBJ wo;
   WORM_TYPE st;
   Integer i;

   if (wt == NULL) {
      WORM_error("Instance of NULL");
      return NULL;
    }
   else if (wt == WORM__type_Type) {
      WORM_error("Can't instance type Type");
      return NULL;
    };

   wo = (WORM_OBJ) calloc(1,sizeof(WORM_OBJ_INFO)+(wt->total_len-1)*4);
   wo->type = wt;

   if (wt->dflt != NULL) {
      bcopy(wt->dflt,&wo->data[wt->data_offset],wt->local_len);
    };

   for (i = 0; i < wt->numsuper; ++i) {
      st = wt->super[i].type;
      if (st->dflt != NULL) {
	 bcopy(st->dflt,&wo->data[wt->super[i].offset],st->local_len);
       };
    };

   for (i = wt->numsuper-1; i >= 0; --i) {
      st = wt->super[i].type;
      if (st->numop >= WORM__op_Inittype &&
	     st->op[WORM__op_Inittype].fct != NULL &&
	     st->op[WORM__op_Inittype].base == st) {
	 WORMapply_type(WORM__op_Inittype,st,wo);
       };
    };
   if (wt->numop >= WORM__op_Inittype &&
	  wt->op[WORM__op_Inittype].fct != NULL &&
	  wt->op[WORM__op_Inittype].base == wt) {
      WORMapply(WORM__op_Inittype,wo);
    };

   return wo;
};





/************************************************************************/
/*									*/
/*	WORMaccess -- access data for given type from object		*/
/*									*/
/************************************************************************/


WORM_UNIV
WORMaccess(wt,wo)
   WORM_TYPE wt;
   WORM_OBJ wo;
{
   Integer i,j;

   if (wo == NULL) return NULL;

   if (wo->type == wt) j = wt->data_offset;
   else {
      j = -1;
      for (i = 0; i < wo->type->numsuper; ++i) {
	 if (wt == wo->type->super[i].type) {
	    j = wo->type->super[i].offset;
	    break;
	  };
       };
    };

   if (j < 0) {
      WORM_error("Illegal access of type %s for object of type %s",wt->name,wo->type->name);
      return NULL;
    };

   return (WORM_UNIV) &wo->data[j];
};





/********************************************************************************/
/*										*/
/*	WORMinq_type_name -- return name of type of object			*/
/*										*/
/********************************************************************************/


char *
WORMinq_type_name(wo)
   WORM_OBJ wo;
{
   if (wo == NULL) return NULL;

   return wo->type->name;
};





/********************************************************************************/
/*										*/
/*	WORMinq_subtypes -- find all subtypes of a given type			*/
/*										*/
/********************************************************************************/


int
WORMinq_subtypes(base,max,typs)
   WORM_TYPE base;
   Integer max;
   String typs[];
{
   WORM_TYPE wt;
   Integer ct,i;

   ct = 0;

   for (wt = all_types; wt != NULL; wt = wt->next) {
      for (i = 0; i < wt->numsuper; ++i) {
	 if (wt->super[i].type == base) {
	    if (ct < max) typs[ct++] = wt->name;
	    break;
	  };
       };
      if (ct >= max) break;
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	WORM_type_private -- define private types			*/
/*									*/
/************************************************************************/


void
WORM_type_private()
{
   Integer i;

   WORM__type_Any = WORMtype_define("Any",0,NULL,0);

   i = sizeof(WORM_TYPE_INFO) - sizeof(WORM_OBJ_HEADER);
   WORM__type_Type = WORMtype_define("Type",i,NULL,0);

   WORM__type_Any->header.type = WORM__type_Type;
   WORM__type_Type->header.type = WORM__type_Type;
};





/************************************************************************/
/*									*/
/*	WORM_new_op -- define operator for all types			*/
/*									*/
/************************************************************************/


void
WORM_new_op(bt,idx,fct,narg,lcl)
   WORM_TYPE bt;
   WORM_OPID idx;
   Function_Ptr fct;
   Integer narg;
   Boolean lcl;
{
   WORM_TYPE wt;

   PROTECT(
      for (wt = all_types; wt != NULL; wt = wt->next) {
	 WORM_op_enter(wt,bt,idx,fct,narg,lcl);
       };
    );
};





/* end of wormtype.c */
