/************************************************************************/
/*									*/
/*		wormop.c						*/
/*									*/
/*	Operation routines for WORM object manager			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "worm_local.h"
#include <varargs.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define OP_LEEWAY	10





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	num_ops;
static	WORM_OPDEF	all_ops;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	String		find_opname();
static	WORM_OPDEF	find_opdef();



/************************************************************************/
/*									*/
/*	WORM_op_init -- module initialization				*/
/*									*/
/************************************************************************/


void
WORM_op_init()
{
   num_ops = 0;
   all_ops = NULL;
};




/************************************************************************/
/*									*/
/*	WORMop_define -- define a new operation 			*/
/*									*/
/************************************************************************/


WORM_OPID
WORMop_define(name,type,fct,numarg,localfg)
   String name;
   WORM_TYPE type;
   Function_Ptr fct;
   Integer numarg;
   Boolean localfg;
{
   WORM_OPDEF wd;
   WORM_OPID idx;

   WORM_init();

   if (type == NULL) type = WORM__type_Any;

   wd = find_opdef(name);
   idx = wd->index;

   WORM_new_op(type,idx,fct,numarg,localfg);

   return idx;
};





/************************************************************************/
/*									*/
/*	WORMop_find -- find op given name				*/
/*									*/
/************************************************************************/


WORM_OPID
WORMop_find(name)
   String name;
{
   WORM_OPDEF wd;

   WORM_init();

   wd = find_opdef(name);

   return wd->index;
};





/************************************************************************/
/*									*/
/*	WORMapply -- apply operator to object				*/
/*		(opid,obj,p1,...pn)					*/
/*									*/
/************************************************************************/


WORM_UNIV
WORMapply(va_alist)
   va_dcl
{
   va_list ap;
   WORM_OPID id;
   WORM_OBJ obj;
   WORM_TYPE wt;
   WORM_UNIV args[32];
   WORM_OP op;
   Integer rslt,i;

   va_start(ap);

   id = va_arg(ap,WORM_OPID);
   obj = va_arg(ap,WORM_OBJ);
   wt = (obj == NULL ? WORM__type_Any : obj->type);

   op = &(wt->op[id]);
   if (op->fct == NULL) {
      WORM_error("Message %s not found for type %s",find_opname(id),wt->name);
      return NULL;
    };

   i = 0;
   if (op->localdata) args[i++] = WORMaccess(op->base,obj);
   args[i++] = (WORM_UNIV) obj;

   while (i < op->numarg) {
      args[i++] = va_arg(ap,WORM_UNIV);
    };

   switch (i) {
      case 1 :
	 rslt = (*op->fct)(args[0]);
	 break;
      case 2 :
	 rslt = (*op->fct)(args[0],args[1]);
	 break;
      case 3 :
	 rslt = (*op->fct)(args[0],args[1],args[2]);
	 break;
      case 4 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3]);
	 break;
      case 5 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4]);
	 break;
      case 6 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5]);
	 break;
      case 7 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6]);
	 break;
      case 8 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6],args[7]);
	 break;
      default :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6],args[7],
			      args[8],args[9],args[10],args[11],args[12],args[13],args[14],
			      args[15]);
	 break;
    };

   return (WORM_UNIV) rslt;
};





/************************************************************************/
/*									*/
/*	WORMapply_type -- apply instance of operator to object		*/
/*		(opid,type,obj,p1,...,pn)				*/
/*									*/
/************************************************************************/


WORM_UNIV
WORMapply_type(va_alist)
   va_dcl
{
   va_list ap;
   WORM_OPID id;
   WORM_OBJ obj;
   WORM_TYPE wt;
   WORM_UNIV args[32];
   WORM_OP op;
   Integer rslt,i;

   va_start(ap);

   id = va_arg(ap,WORM_OPID);
   wt = va_arg(ap,WORM_TYPE);
   if (wt == NULL) wt = WORM__type_Any;
   obj = va_arg(ap,WORM_OBJ);

   op = &(wt->op[id]);
   if (op->fct == NULL) {
      WORM_error("Message %s not found for type %s",find_opname(id),wt->name);
    };

   i = 0;
   if (op->localdata) args[i++] = WORMaccess(op->base,obj);
   args[i++] = (WORM_UNIV) obj;

   while (i < op->numarg) {
      args[i++] = va_arg(ap,WORM_UNIV);
    };

   switch (i) {
      case 1 :
	 rslt = (*op->fct)(args[0]);
	 break;
      case 2 :
	 rslt = (*op->fct)(args[0],args[1]);
	 break;
      case 3 :
	 rslt = (*op->fct)(args[0],args[1],args[2]);
	 break;
      case 4 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3]);
	 break;
      case 5 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4]);
	 break;
      case 6 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5]);
	 break;
      case 7 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6]);
	 break;
      case 8 :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6],args[7]);
	 break;
      default :
	 rslt = (*op->fct)(args[0],args[1],args[2],args[3],args[4],args[5],args[6],args[7],
			      args[8],args[9],args[10],args[11],args[12],args[13],args[14],
			      args[15]);
	 break;
    };

   return (WORM_UNIV) rslt;
};






/************************************************************************/
/*									*/
/*	WORM_op_setup -- set up ops for new type			*/
/*									*/
/************************************************************************/


void
WORM_op_setup(wt)
   WORM_TYPE wt;
{
   WORM_TYPE st;
   Integer i,j;

   PROTECT(
      wt->numop = num_ops+OP_LEEWAY;
      wt->op = (WORM_OP) calloc(sizeof(WORM_OP_INFO),wt->numop);

      for (j = 0; j < wt->numsuper; ++j) {
	 st = wt->super[j].type;
	 for (i = 0; i < st->numop; ++i) {
	    if (wt->op[i].fct == NULL) {
	       if (st->op[i].fct != NULL) {
		  wt->op[i] = st->op[i];
		};
	     };
	  };
       };
    );
};





/************************************************************************/
/*									*/
/*	WORM_op_enter -- define new operator for given type		*/
/*									*/
/************************************************************************/


void
WORM_op_enter(wt,bwt,idx,fct,numarg,lclfg)
   WORM_TYPE wt;
   WORM_TYPE bwt;
   WORM_OPID idx;
   Function_Ptr fct;
   Integer numarg;
   Boolean lclfg;
{
   Integer i,j;
   WORM_TYPE st,ot;

   if (wt != bwt) {
      if (wt->numop < idx) ot = NULL;
      else ot = wt->op[idx].base;
      if (ot == wt) return;
      for (i = 0; i < wt->numsuper; ++i) {
	 st = wt->super[i].type;
	 if (st == bwt) break;
	 if (st == ot) return;
       };
      if (i >= wt->numsuper) return;
    };

   if (wt->numop < idx) {
      j = num_ops+OP_LEEWAY;
      wt->op = (WORM_OP) realloc(wt->op,j*sizeof(WORM_OP_INFO));
      for (i = wt->numop; i < j; ++i) {
	 wt->op[i].fct = NULL;
	 wt->op[i].base = NULL;
	 wt->op[i].numarg = 0;
	 wt->op[i].localdata = FALSE;
       };
      wt->numop = j;
    };

   wt->op[idx].fct = fct;
   wt->op[idx].base = bwt;
   wt->op[idx].numarg = numarg;
   wt->op[idx].localdata = lclfg;
};





/************************************************************************/
/*									*/
/*	find_opname -- find name of operator given idx			*/
/*									*/
/************************************************************************/


static String
find_opname(id)
   WORM_OPID id;
{
   WORM_OPDEF od;

   for (od = all_ops; od != NULL; od = od->next) {
      if (od->index == id) break;
    };

   if (od == NULL) return "*UNKNOWN*";
   else return od->name;
};





/********************************************************************************/
/*										*/
/*	find_opdef -- find operator definition given index			*/
/*										*/
/********************************************************************************/


static WORM_OPDEF
find_opdef(name)
   String name;
{
   WORM_OPDEF wd;

   PROTECT(
      for (wd = all_ops; wd != NULL; wd = wd->next) {
	 if (STREQL(wd->name,name)) break;
       };

      if (wd == NULL) {
	 wd = PALLOC(WORM_OPDEF_INFO);
	 wd->name = SALLOC(name);
	 wd->index = num_ops++;
	 wd->next = all_ops;
	 all_ops = wd;
       };
    );

   return wd;
};





/* end of wormop.c */
