/********************************************************************************/
/*										*/
/*		wormio.c							*/
/*										*/
/*	Routines for doing I/O with worm objects				*/
/*										*/
/********************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "worm_local.h"
#include <sequence.h>
#include <auxd.h>
#include <ash.h>




/********************************************************************************/
/*										*/
/*	Parameters								*/
/*										*/
/********************************************************************************/


#define DEFAULT_PTR_SIZE	128
#define MAX_ARGS		64

#define QUERY_DEFAULT		((String) 0)
#define QUERY_IGNORE		((String) 1)
#define QUERY_NEST		((String) 2)

#define QUERY_SEPARATOR 	'.'





/********************************************************************************/
/*										*/
/*	Local type definitions							*/
/*										*/
/********************************************************************************/


typedef struct _PTR_DATA {
   String id;
   String object;
} PTR_DATA;




typedef struct _LINK_DATA {
   String * address;
   String id;
} *LINK_DATA, LINK_DATA_INFO;



/********************************************************************************/
/*										*/
/*	Local storage								*/
/*										*/
/********************************************************************************/


static	Integer 	ptr_table_ct;
static	Integer 	ptr_table_max;
static	PTR_DATA *	ptr_table;
static	Sequence	links;

static	WORM_READ_DATA *	io_table;

static	WORM_READ_DATA	sequence_field = { "VALUE", 0, NULL, NULL };





/********************************************************************************/
/*										*/
/*	Macros									*/
/*										*/
/********************************************************************************/


#define IGNORE_QUERY(t) (query_ask_info(t) == QUERY_IGNORE)





/********************************************************************************/
/*										*/
/*	Forward definitions							*/
/*										*/
/********************************************************************************/


static	void		read_object();
static	void		assign_field();
static	void		read_internal_fields();
static	Sequence	read_sequence();
static	void		write_field();
static	void		write_internal_fields();
static	void		clear_object_table();
static	void		enter_object();
static	void		assign_object();
static	void		fixup_object_links();
static	void		enqueue_object();
static	WORM_OBJ	dequeue_object();
static	void		enter_write();
static	Integer 	query_field();
static	Integer 	query_internal_fields();
static	void		query_enum_field();
static	Integer 	set_field();
static	Integer 	set_internal_fields();
static	void		handle_list_query();
static	void		add_to_list();
static	WORM_OBJ	create_item();
static	String		query_ask_info();
static	String		query_print_info();





/********************************************************************************/
/*										*/
/*	WORM_io_init -- module initialization					*/
/*										*/
/********************************************************************************/


void
WORM_io_init()
{
   ptr_table_max = DEFAULT_PTR_SIZE;
   ptr_table_ct = 0;
   ptr_table = (PTR_DATA *) calloc(ptr_table_max,sizeof(PTR_DATA));
};





/********************************************************************************/
/*										*/
/*	WORMset_io_table -- set I/O table					*/
/*										*/
/********************************************************************************/


void
WORMset_io_table(rt,dummy)
   WORM_READ_DATA * rt;
   Integer dummy;
{
   Integer i;

   io_table = rt;

   if (dummy != 0) {
      for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
	 if (io_table[i].offset > 0) io_table[i].offset -= dummy;
       };
    };
};





/********************************************************************************/
/*										*/
/*	WORMread -- read objects with given handle				*/
/*										*/
/********************************************************************************/


void
WORMread(name,name_rtn)
   String name;
   Function_Ptr name_rtn;
{
   AUXD hdl,obj;

   clear_object_table();

   for (hdl = AUXDget_handle(NULL,name); hdl != NULL; hdl = AUXDnext_handle(hdl)) {
      for (obj = AUXDget_handle(hdl,"OBJECT"); obj != NULL; obj = AUXDnext_handle(obj)) {
	 read_object(obj,name,name_rtn);
       };
      fixup_object_links();
    };
};





/********************************************************************************/
/*										*/
/*	WORMwrite_begin -- start writing objects				*/
/*	WORMwrite -- write an object						*/
/*	WORMwrite_end -- finish writing objects 				*/
/*										*/
/********************************************************************************/


void
WORMwrite_begin(otf,name)
   FILE * otf;
   String name;
{
   fprintf(otf,"%s:\n",name);

   clear_object_table();
};




void
WORMwrite(otf,obj,name)
   FILE * otf;
   WORM_OBJ obj;
   String name;
{
   Integer i;
   WORM_TYPE wty;
   String data;
   String typ;

   typ = WORMinq_type_name(obj);

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   enter_write(obj);

   fprintf(otf,"    OBJECT +\n");
   fprintf(otf,"\tID = %d\n\tTYPE = %s\n",obj,typ);
   if (name != NULL) fprintf(otf,"\tNAME = \"%s\"\n",name);

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      wty = WORMtype_find(io_table[i].base);
      data = ((String) WORMaccess(wty,obj)) + io_table[i].offset;
      write_field(otf,data,&io_table[i],io_table[i].type);
    };

   fprintf(otf,"\t;\n\n");
};





void
WORMwrite_end(otf)
   FILE * otf;
{
   WORM_OBJ obj;

   while ((obj = dequeue_object()) != NULL) {
      WORMwrite(otf,obj,NULL);
    };

   fflush(otf);
};





/********************************************************************************/
/*										*/
/*	WORMquery -- request information from user about an object		*/
/*										*/
/********************************************************************************/


int
WORMquery(w,obj)
   ASH_WINDOW w;
   WORM_OBJ obj;
{
   String s;
   Integer i,j;
   Character menu[10240];
   WORM_TYPE wty;
   String data;
   Integer argct;
   WORM_DIALOG_ARG args[MAX_ARGS];
   Universal * argp[MAX_ARGS];

   if (obj == NULL) return FALSE;
   s = WORMinq_type_name(obj);

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(s,io_table[i].fieldname)) break;
    };
   if (io_table[i].offset != WORM_TYPE_OFFSET) return FALSE;

   sprintf(menu,"%%CEdit %s Object\n\n",s);

   argct = 0;

   for (j = i+1; io_table[j].offset >= 0; ++j) {
      wty = WORMtype_find(io_table[j].base);
      data = ((String) WORMaccess(wty,obj)) + io_table[j].offset;
      argct = query_field(menu,args,argct,data,&io_table[j],0);
    };

   strcat(menu,"\n   %a%M   %c");

   for (j = 0; j < argct; ++j) {
      argp[j] = (Universal *) &args[j];
    };

   if (!STEMdialog1_array(w,menu,argp)) return FALSE;

   argct = 0;
   for (j = i+1; io_table[j].offset >= 0; ++j) {
      wty = WORMtype_find(io_table[j].base);
      data = ((String) WORMaccess(wty,obj)) + io_table[j].offset;
      argct = set_field(args,argct,data,&io_table[j],w,obj);
    };

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	WORMquery_field -- request information from user about an object field	*/
/*										*/
/********************************************************************************/


int
WORMquery_field(w,obj,fld)
   ASH_WINDOW w;
   WORM_OBJ obj;
   String fld;
{
   String s,t;
   Integer i,j,k;
   Character menu[10240];
   WORM_TYPE wty;
   String data;
   Integer argct;
   WORM_DIALOG_ARG args[MAX_ARGS];
   Universal * argp[MAX_ARGS];
   Boolean ifg;

   if (obj == NULL) return FALSE;
   s = WORMinq_type_name(obj);

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(s,io_table[i].fieldname)) break;
    };
   if (io_table[i].offset != WORM_TYPE_OFFSET) return FALSE;

   for (j = i+1; io_table[j].offset >= 0; ++j) {
      if (STREQL(io_table[j].fieldname,fld)) break;
    };
   if (io_table[j].offset < 0) return FALSE;

   sprintf(menu,"%%CEdit %s of %s Object\n\n",fld,s);

   argct = 0;
   wty = WORMtype_find(io_table[j].base);
   data = ((String) WORMaccess(wty,obj)) + io_table[j].offset;

   t = io_table[j].type;

   if (STREQL(t,"Integer") || STREQL(t,"Boolean") || STREQL(t,"Float") ||
	  STREQL(t,"String") || t[0] == '^' || t[0] == '*' || t[0] == '#') {
      argct = query_field(menu,args,argct,data,&io_table[j],0);
      ifg = FALSE;
    }
   else {
      argct = query_internal_fields(menu,args,argct,data,io_table[j].type,-1);
      ifg = TRUE;
    };

   strcat(menu,"\n   %a%M   %c");

   for (k = 0; k < argct; ++k) {
      argp[k] = (Universal *) &args[k];
    };

   if (!STEMdialog1_array(w,menu,argp)) return FALSE;

   argct = 0;

   if (!ifg) {
      argct = set_field(args,argct,data,&io_table[j],w,obj);
    }
   else {
      argct = set_internal_fields(args,argct,data,io_table[j].type,w,obj);
    };

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	WORMquery_new -- get a new object of given type 			*/
/*										*/
/********************************************************************************/


WORM_OBJ
WORMquery_new(w,type,flag)
   ASH_WINDOW w;
   String type;
   Boolean flag;
{
   WORM_OBJ wo;

   wo = create_item(w,type,flag);

   return wo;
};





/********************************************************************************/
/*										*/
/*	read_object -- read object from AUXD information			*/
/*										*/
/********************************************************************************/


static void
read_object(hdl,interface,name_rtn)
   AUXD hdl;
   String interface;
   Function_Ptr name_rtn;
{
   String typ;
   Integer id;
   WORM_OBJ obj;
   WORM_TYPE wty;
   String data;
   String name;
   Integer i;

   typ = AUXDget_info(hdl,"TYPE");
   if (typ == NULL) return;

   wty = WORMtype_find(typ);

   obj = WORMinstance(wty);
   id = AUXDget_info_int(hdl,"ID");
   enter_object(obj,id);

   name = AUXDget_info(hdl,"NAME");
   if (name != NULL && name_rtn != NULL) (*name_rtn)(obj,name,interface);

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      wty = WORMtype_find(io_table[i].base);
      data = ((String) WORMaccess(wty,obj)) + io_table[i].offset;
      assign_field(hdl,data,&io_table[i],io_table[i].type);
    };
};





/********************************************************************************/
/*										*/
/*	assign_field -- handle assigning value to field 			*/
/*										*/
/********************************************************************************/


static void
assign_field(hdl,data,tbl,typ)
   AUXD hdl;
   String data;
   WORM_READ_DATA * tbl;
   String typ;
{
   Integer nid;
   String s;

   if (STREQL(typ,"Integer")) {
      *((Integer *) data) = AUXDget_info_int(hdl,tbl->fieldname);
    }
   else if (STREQL(typ,"Boolean")) {
      *((Boolean *) data) = AUXDget_defined(hdl,tbl->fieldname);
    }
   else if (STREQL(typ,"Float")) {
      *((Float *) data) = AUXDget_info_real(hdl,tbl->fieldname);
    }
   else if (STREQL(typ,"String")) {
      s = AUXDget_info(hdl,tbl->fieldname);
      if (s != NULL) s = SALLOC(s);
      *((String *) data) = s;
    }
   else if (typ[0] == '^') {
      *((Integer *) data) = AUXDget_info_int(hdl,tbl->fieldname);
    }
   else if (typ[0] == '#') {
      *((Sequence *) data) = read_sequence(AUXDget_handle(hdl,tbl->fieldname),&typ[1]);
    }
   else if (typ[0] == '*') {
      nid = AUXDget_info_int(hdl,tbl->fieldname);
      if (nid == 0) *((WORM_OBJ *) data) = NULL;
      else assign_object(data,nid);
    }
   else {
      read_internal_fields(AUXDget_handle(hdl,tbl->fieldname),data,typ);
    };
};





/********************************************************************************/
/*										*/
/*	read_internal_fields -- handle internal fields for a type		*/
/*										*/
/********************************************************************************/


static void
read_internal_fields(hdl,data0,typ)
   AUXD hdl;
   String data0;
   String typ;
{
   Integer i;
   String data;

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname))
	 break;
    };

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      data = data0 + io_table[i].offset;
      assign_field(hdl,data,&io_table[i],io_table[i].type);
    };
};






/********************************************************************************/
/*										*/
/*	read_sequence -- read in a sequence field				*/
/*										*/
/********************************************************************************/


static Sequence
read_sequence(hdl,typ)
   AUXD hdl;
   String typ;
{
   Sequence l,la;

   l = NULL;
   la = NULL;
   for ( ; hdl != NULL; hdl = AUXDnext_handle(hdl)) {
      la = APPEND(0,la);
      if (l == NULL) l = la;
      else la = CDR(la);
      assign_field(hdl,&la->Lcar,&sequence_field,typ);
    };

   return l;
};





/********************************************************************************/
/*										*/
/*	write_field -- handle writing values from a field			*/
/*										*/
/********************************************************************************/


static void
write_field(otf,data,tbl,typ)
   FILE * otf;
   String data;
   WORM_READ_DATA * tbl;
   String typ;
{
   Universal * u;
   Sequence l;

   if (STREQL(typ,"Integer")) {
      fprintf(otf,"\t%s = %d\n",tbl->fieldname,*((Integer *) data));
    }
   else if (STREQL(typ,"Boolean")) {
      if (*((Boolean *) data)) fprintf(otf,"\t%s = .\n",tbl->fieldname);
    }
   else if (STREQL(typ,"Float")) {
      fprintf(otf,"\t%s = %f\n",tbl->fieldname,*((Float *) data));
    }
   else if (STREQL(typ,"String")) {
      if (*((String *) data) != NULL)
	 fprintf(otf,"\t%s = \"%s\"\n",tbl->fieldname,*((String *) data));
    }
   else if (typ[0] == '^') {
      fprintf(otf,"\t%s = %d\n",tbl->fieldname,*((Integer *) data));
    }
   else if (typ[0] == '#') {
      forin(u,Universal *,l,*((Sequence *) data)) {
	 fprintf(otf,"\t%s +\n\t",tbl->fieldname);
	 write_field(otf,&u,&sequence_field,&typ[1]);
	 fprintf(otf,"\t\t;\n");
       };
    }
   else if (typ[0] == '*') {
      fprintf(otf,"\t%s = %d\n",tbl->fieldname,*((Integer *) data));
      enqueue_object(*((Integer *) data));
    }
   else {
      fprintf(otf,"\t%s :\n",tbl->fieldname);
      write_internal_fields(otf,data,typ);
    };
};





/********************************************************************************/
/*										*/
/*	write_internal_fields -- handle internal fields for a type		*/
/*										*/
/********************************************************************************/


static void
write_internal_fields(otf,data0,typ)
   FILE * otf;
   String data0;
   String typ;
{
   Integer i;
   String data;

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      data = data0 + io_table[i].offset;
      write_field(otf,data,&io_table[i],io_table[i].type);
    };

   fprintf(otf,"\t;\n");
};





/********************************************************************************/
/*										*/
/*	clear_object_table -- clear table of objects				*/
/*										*/
/********************************************************************************/


static void
clear_object_table()
{
   ptr_table_ct = 0;
   links = NULL;
};






/********************************************************************************/
/*										*/
/*	enter_object -- enter object into table on read 			*/
/*	assign_object -- do object assignment based on id			*/
/*	fixup_object_links -- fixup undone assignments				*/
/*										*/
/********************************************************************************/


static void
enter_object(obj,id)
   String obj;
   String id;
{
   Integer i;

   for (i = 0; i < ptr_table_ct; ++i) {
      if (id == ptr_table[i].id) break;
    };

   if (i >= ptr_table_ct) {
      if (i >= ptr_table_max) {
	 ptr_table_max *= 2;
	 ptr_table = (PTR_DATA *) realloc(ptr_table,ptr_table_max*sizeof(PTR_DATA));
       };
      i = ptr_table_ct++;
      ptr_table[i].id = id;
    };

   ptr_table[i].object = obj;
};





static void
assign_object(loc,id)
   String  * loc;
   String id;
{
   Integer i;
   LINK_DATA ld;

   for (i = 0; i < ptr_table_ct; ++i) {
      if (id == ptr_table[i].id) {
	 *loc = ptr_table[i].object;
	 return;
       };
    };

   ld = PALLOC(LINK_DATA_INFO);
   ld->address = loc;
   ld->id = id;
   links = CONS(ld,links);
};





static void
fixup_object_links()
{
   Sequence l;
   LINK_DATA ld;
   Integer i;
   String v;

   forin (ld,LINK_DATA,l,links) {
      v = NULL;
      for (i = 0; i < ptr_table_ct; ++i) {
	 if (ld->id == ptr_table[i].id) {
	    v = ptr_table[i].object;
	    break;
	  };
       };
      *(ld->address) = v;
      free(ld);
    };

   LFREE(links);
   links = NULL;
};





/********************************************************************************/
/*										*/
/*	enqueue_object -- enter object to do for write				*/
/*	dequeue_object -- find object not yet written				*/
/*	enter_write -- mark object as written					*/
/*										*/
/********************************************************************************/


static void
enqueue_object(obj)
   String obj;
{
   Integer i;

   if (obj == NULL) return;

   for (i = 0; i < ptr_table_ct; ++i) {
      if (obj == ptr_table[i].id) break;
    };

   if (i >= ptr_table_ct) {
      if (i >= ptr_table_max) {
	 ptr_table_max *= 2;
	 ptr_table = (PTR_DATA *) realloc(ptr_table,ptr_table_max*sizeof(PTR_DATA));
       };
      i = ptr_table_ct++;
      ptr_table[i].id = obj;
      ptr_table[i].object = "TO DO";
    };
};





static WORM_OBJ
dequeue_object()
{
   Integer i;

   for (i = 0; i < ptr_table_ct; ++i) {
      if (ptr_table[i].object != NULL) {
	 return (WORM_OBJ) ptr_table[i].id;
       };
    };

   return FALSE;
};





static void
enter_write(obj)
   String obj;
{
   Integer i;

   for (i = 0; i < ptr_table_ct; ++i) {
      if (ptr_table[i].id == obj) {
	 ptr_table[i].object = NULL;
	 break;
       };
    };
};





/********************************************************************************/
/*										*/
/*	query_field -- setup dialog menu for a field				*/
/*										*/
/********************************************************************************/


static Integer
query_field(menu,args,argct,data,tbl,indent)
   String menu;
   WORM_DIALOG_ARG args[];
   Integer argct;
   String data;
   WORM_READ_DATA * tbl;
   Integer indent;
{
   Integer j;
   Character buf[1024];
   Function_Ptr fp;
   String typ,q;

   typ = tbl->type;

   q = query_ask_info(tbl);
   if (q == QUERY_DEFAULT || q == QUERY_NEST) ;
   else if (q == QUERY_IGNORE) return argct;
   else {
      fp = (Function_Ptr) DLlookup(q);
      if (fp != NULL) {
	 argct = (*fp)(menu,args,argct,data,tbl->fieldname,typ,indent);
       }
      return argct;
    };

   buf[0] = 0;

   if (STREQL(typ,"Integer")) {
      sprintf(buf,"%s: %%%dd",query_print_info(tbl),argct);
      args[argct].intarg = *((Integer *) data);
      ++argct;
    }
   else if (STREQL(typ,"Boolean")) {
      sprintf(buf,"%%%do %s",argct,query_print_info(tbl));
      args[argct].intarg = *((Boolean *) data);
      ++argct;
    }
   else if (STREQL(typ,"Float")) {
      sprintf(buf,"%s: %%%dh",query_print_info(tbl),argct);
      args[argct].fltarg = *((Float *) data);
      ++argct;
    }
   else if (STREQL(typ,"String")) {
      sprintf(buf,"%s: %%%d.48t",query_print_info(tbl),argct);
      if (*((String *) data) != NULL)
	 strcpy(args[argct].strarg,*((String *) data));
      else args[argct].strarg[0] = 0;
      ++argct;
    }
   else if (typ[0] == '^') {
      for (j = 0; j < indent; ++j) strcat(menu,"   ");
      sprintf(buf,"%s:\n",query_print_info(tbl));
      strcat(menu,buf);
      buf[0] = 0;
      args[argct].intarg = *((Integer *) data);
      query_enum_field(menu,argct,&typ[1],indent);
      ++argct;
    }
   else if (typ[0] == '*') {
      if (*((WORM_OBJ *) data) == NULL) {
	 sprintf(buf,"%%%d.1o Create %s",argct,query_print_info(tbl));
       }
      else {
	 sprintf(buf,"%%%d.1o Replace %s%%M%%%d.2o Modify %s",
		    argct,query_print_info(tbl),argct,query_print_info(tbl));
       };
      args[argct].intarg = 0;
      ++argct;
    }
   else if (typ[0] == '#') {
      if (*((Sequence *) data) == NULL)
	 sprintf(buf,"%%%do Create list for %s",argct,query_print_info(tbl));
      else
	 sprintf(buf,"%%%do Modify list for %s",argct,query_print_info(tbl));
      args[argct].intarg = FALSE;
      ++argct;
    }
   else if (query_ask_info(tbl) == QUERY_NEST) {
      strcat(menu,"\n");
      for (j = 0; j < indent; ++j) strcat(menu,"   ");
      sprintf(buf,"%s:\n",query_print_info(tbl));
      strcat(menu,buf);
      buf[0] = 0;
      argct = query_internal_fields(menu,args,argct,data,typ,indent);
      strcat(menu,"\n");
    }
   else {
      sprintf(buf,"%%%do Modify %s",argct,query_print_info(tbl));
      args[argct].intarg = FALSE;
      ++argct;
    };

   if (buf[0] != 0) {
      for (j = 0; j < indent; ++j) strcat(menu,"   ");
      strcat(menu,buf);
      strcat(menu,"\n");
    };

   return argct;
};





/********************************************************************************/
/*										*/
/*	query_internal_fields -- handle internal fields for a type		*/
/*										*/
/********************************************************************************/


static Integer
query_internal_fields(menu,args,argct,data0,typ,indent)
   String menu;
   WORM_DIALOG_ARG args[];
   Integer argct;
   String data0;
   String typ;
   Integer indent;
{
   Integer i;
   String data;

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   if (io_table[i].offset == WORM_END_OFFSET || IGNORE_QUERY(&io_table[i]))
      return argct;

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      data = data0 + io_table[i].offset;
      argct = query_field(menu,args,argct,data,&io_table[i],indent+1);
    };

   return argct;
};





/********************************************************************************/
/*										*/
/*	query_enum_field -- handle enumerations 				*/
/*										*/
/********************************************************************************/


static void
query_enum_field(menu,argno,typ,indent)
   String menu;
   Integer argno;
   String typ;
   Integer indent;
{
   Integer i,j,k;
   Character buf[1024];

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   if (io_table[i].offset != WORM_TYPE_OFFSET) return;

   j = 0;
   for (i = i+1; io_table[i].offset == WORM_ENUM_OFFSET; ++i) {
      if ((j&1) == 0) {
	 if (j != 0) strcat(menu,"\n");
	 for (k = 0; k < indent+1; ++k) strcat(menu,"   ");
       }
      else strcat(menu,"%M");

      sprintf(buf,"%%%d.%do %s",argno,j,io_table[i].fieldname);
      strcat(menu,buf);
      ++j;
    };

   strcat(menu,"\n");
};





/********************************************************************************/
/*										*/
/*	set_field -- set new field value after dialog box query 		*/
/*										*/
/********************************************************************************/


static Integer
set_field(args,argct,data,tbl,w,obj)
   WORM_DIALOG_ARG args[];
   Integer argct;
   String data;
   WORM_READ_DATA * tbl;
   ASH_WINDOW w;
   WORM_OBJ obj;
{
   String s,t;
   Function_Ptr fp;
   WORM_OBJ wo;
   String typ,q;

   typ = tbl->type;

   q = query_ask_info(tbl);
   if (q == QUERY_DEFAULT || q == QUERY_NEST) ;
   else if (q == QUERY_IGNORE) return argct;
   else {
      fp = (Function_Ptr) DLlookup(q);
      if (fp != NULL) {
	 argct = (*fp)(NULL,args,argct,data,tbl->fieldname,typ,0);
       }
      return argct;
    };

   if (STREQL(typ,"Integer")) {
      *((Integer *) data) = args[argct].intarg;
      ++argct;
    }
   else if (STREQL(typ,"Boolean")) {
      *((Boolean *) data) = args[argct].intarg;
      ++argct;
    }
   else if (STREQL(typ,"Float")) {
      *((Float *) data) = args[argct].fltarg;
      ++argct;
    }
   else if (STREQL(typ,"String")) {
      if (args[argct].strarg[0] == 0) s = NULL;
      else s = args[argct].strarg;
      t = *((String *) data);
      if (s != NULL && t != NULL && STREQL(s,t)) ;
      else if (s == NULL && t == NULL) ;
      else if (s != NULL) *((String *) data) = SALLOC(s);
      else *((String *) data) = NULL;
      ++argct;
    }
   else if (typ[0] == '^') {
      *((Integer *) data) = args[argct].intarg;
      ++argct;
    }
   else if (typ[0] == '*') {
      if (args[argct].intarg == 1) {
	 wo = create_item(w,&typ[1],TRUE);
	 if (wo != NULL) *((WORM_OBJ *) data) = wo;
       }
      else if (args[argct].intarg == 2) {
	 wo = *((WORM_OBJ *) data);
	 WORMquery(w,wo);
       };
      ++argct;
    }
   else if (typ[0] == '#') {
      if (args[argct].intarg) {
	 handle_list_query(w,obj,query_print_info(tbl),data,typ);
       };
      ++argct;
    }
   else if (query_ask_info(tbl) == QUERY_NEST) {
      argct = set_internal_fields(args,argct,data,typ,w,obj);
    }
   else {
      if (args[argct].intarg) {
	 WORMquery_field(w,obj,tbl->fieldname);
       };
      ++argct;
    };

   return argct;
};





/********************************************************************************/
/*										*/
/*	set_internal_fields -- handle internal fields for a type		*/
/*										*/
/********************************************************************************/


static Integer
set_internal_fields(args,argct,data0,typ,w,obj)
   WORM_DIALOG_ARG args[];
   Integer argct;
   String data0;
   String typ;
   ASH_WINDOW w;
   WORM_OBJ obj;
{
   Integer i;
   String data;

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET &&
	     STREQL(typ,io_table[i].fieldname)) break;
    };

   if (io_table[i].offset == WORM_END_OFFSET || IGNORE_QUERY(&io_table[i]))
      return argct;

   for (i = i+1; io_table[i].offset >= 0; ++i) {
      data = data0 + io_table[i].offset;
      argct = set_field(args,argct,data,&io_table[i],w,obj);
    };

   return argct;
};





/********************************************************************************/
/*										*/
/*	handle_list_query -- handle query request for Sequence field		*/
/*	add_to_list -- add element to list					*/
/*										*/
/********************************************************************************/


static void
handle_list_query(w,obj,fld,data,typ)
   ASH_WINDOW w;
   WORM_OBJ obj;
   String fld;
   Sequence * data;
   String typ;
{
   Character menu[10240],buf[1024];
   Integer ln,i;
   Integer op,itm,cont;
   Sequence l;

   cont = TRUE;

   while (cont) {
      sprintf(menu,"%%CEdit List for %s\n\n",fld);

      ln = LENGTH(*data);

      if (ln == 0) {
	 strcat(menu,"%0.1o Add to List\n\n");
       }
      else {
	 strcat(menu,"%0.1o Add Before selected element\n");
	 strcat(menu,"%0.2o Add After selected element\n");
	 strcat(menu,"%0.3o Remove selected element\n");
	 strcat(menu,"%0.4o Edit selected element\n");
	 for (i = 0; i < ln; ++i) {
	    sprintf(buf,"   %%1.%do Element %d\n",i,i);
	    strcat(menu,buf);
	  };
	 strcat(menu,"\n%2o Continue\n\n");
       };

      strcat(menu,"   %a%M   %c");

      op = 2;
      itm = ln-1;
      cont = TRUE;

      if (ln != 0) {
	 if (!STEMdialog1(w,menu,&op,&itm,&cont)) return;
       };

      switch (op) {
	 case 0 :
	    break;
	 case 1 :
	    add_to_list(w,&typ[1],data,itm-1);
	    if (ln == LENGTH(*data)) cont = FALSE;
	    break;
	 case 2 :
	    add_to_list(w,&typ[1],data,itm);
	    if (ln == LENGTH(*data)) cont = FALSE;
	    break;
	 case 3 :
	    if (itm == 0) *data = CDRF(*data);
	    else {
	       l = *data;
	       for (i = 1; i < itm; ++i) l = CDR(l);
	       RPLACD(l,CDDR(l));
	     };
	    break;
	 case 4 :
	    WORMquery(w,NTH(WORM_OBJ,*data,itm+1));
	    break;
       };
    };
};





static void
add_to_list(w,typ,data,itm)
   ASH_WINDOW w;
   String typ;
   Sequence * data;
   Integer itm;
{
   WORM_OBJ wo;
   Sequence l;
   Integer i;

   for ( ; ; ) {
      if (typ[0] == '*') {
	 wo = create_item(w,&typ[1],TRUE);
       }
      else wo = NULL;

      if (wo == NULL) break;

      if (itm < 0) *data = CONS(wo,*data);
      else {
	 l = *data;
	 for (i = 0; i < itm; ++i) l = CDR(l);
	 RPLACD(l,CONS(wo,CDR(l)));
       };

      ++itm;
    };
};





/********************************************************************************/
/*										*/
/*	create_item -- create a new item for list or field			*/
/*										*/
/********************************************************************************/


static WORM_OBJ
create_item(w,typ,info)
   ASH_WINDOW w;
   String typ;
   Boolean info;
{
   Integer ct,i,k,l;
   Character menu[10240],buf[1024];
   String typs[128],s,ntyp[128],typn[128];
   WORM_TYPE wt;
   WORM_OBJ wo;

   for (i = 0; io_table[i].offset != WORM_END_OFFSET; ++i) {
      if (io_table[i].offset == WORM_TYPE_OFFSET && STREQL(typ,io_table[i].fieldname)) break;
    };

   if (io_table[i].offset == WORM_END_OFFSET || IGNORE_QUERY(&io_table[i])) {
      ct = WORMinq_subtypes(WORMtype_find(typ),128,ntyp);
      l = 0;
      for (i = ct-1; i >= 0; --i) {
	 for (k = 0; io_table[k].offset != WORM_END_OFFSET; ++k) {
	    if (io_table[k].offset == WORM_TYPE_OFFSET &&
		   STREQL(ntyp[i],io_table[k].fieldname)) break;
	  };
	 if (io_table[k].offset != WORM_END_OFFSET && !IGNORE_QUERY(&io_table[k])) {
	    s = query_print_info(&io_table[k]);
	    typn[l] = ntyp[i];
	    typs[l++] = SALLOC(s);
	  };
       };
      ct = l;

      if (ct > 0) {
	 if (ct == 1) typ = typn[0];
	 else {
	    sprintf(menu,"Select type of item to add:\n\n");
	    for (i = 0; i < ct; ++i) {
	       sprintf(buf,"%%0.%do %s\n",i,typs[i]);
	       strcat(menu,buf);
	     };
	    strcat(menu,"\n   %a%M   %c");
	    i = 0;
	    if (!STEMdialog1(w,menu,&i)) typ = NULL;
	    else typ = typn[i];
	  };
       }
      else typ = NULL;
      for (i = 0; i < ct; ++i) SFREE(typs[i]);
    };

   if (typ == NULL) return NULL;

   wt = WORMtype_find(typ);
   if (wt == NULL) return NULL;
   wo = WORMinstance(wt);

   if (info && !WORMquery(w,wo)) return NULL;

   return wo;
};






/************************************************************************/
/*									*/
/*	query_ask_info -- get information about asking from query field */
/*	query_print_info -- get print name from item/query		*/
/*									*/
/************************************************************************/


static String
query_ask_info(wrd)
   WORM_READ_DATA * wrd;
{
   static Character buf[128];
   String s;

   if (wrd->query == NULL) return QUERY_DEFAULT;

   if (index(wrd->query,QUERY_SEPARATOR) == 0) s = wrd->query;
   else {
      strcpy(buf,wrd->query);
      s = index(buf,QUERY_SEPARATOR);
      *s = 0;
      s = buf;
    };

   if (STREQL(s,WORM_QUERY_IGNORE)) return QUERY_IGNORE;
   if (STREQL(s,WORM_QUERY_NEST)) return QUERY_NEST;

   return s;
};






static String
query_print_info(wrd)
   WORM_READ_DATA * wrd;
{
   static Character buf[128];
   String s;

   if (wrd->query == NULL) s = NULL;
   else s = index(wrd->query,QUERY_SEPARATOR);

   if (s == NULL) s = wrd->fieldname;
   else {
      ++s;
      if (index(s,QUERY_SEPARATOR) != NULL) {
	 strcpy(buf,s);
	 s = index(s,QUERY_SEPARATOR);
	 *s = 0;
	 s = buf;
       };
    };

   return s;
};





/* end of wormio.c */
