/************************************************************************/
/*									*/
/*		winddefs.c						*/
/*									*/
/*	Routines for managing user windows				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "wind_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define THREAD_STACK_SIZE	256000
#define THREAD_PRIORITY 	16




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_views;

static	Integer 	thread_size = 0;	/* thread stack size	*/
static	THREAD_FCT	thread_fct = NULL;	/* thread create fct	*/
static	Integer 	thread_priority = 0;	/* thread priority	*/

static	Boolean 	show_unique;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		wind_thread_begin();
static	void		wind_thread_finish();
static	WIND_VIEW	create_new_view();
static	WIND_VIEW	duplicate_view();
static	int		wind_view_control();
static	void		exec_command();





/************************************************************************/
/*									*/
/*	WIND_defs_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
WIND_defs_init()
{
   String s;

   all_views = NULL;

   s = ASHinq_resource("wind.show_unique");
   if (s == NULL) s = getenv("SHOW_UNIQUE");
   if (s == NULL || STREQL(s,"off")) show_unique = FALSE;
   else show_unique = TRUE;

   if (thread_size == 0) thread_size = THREAD_STACK_SIZE;
   if (thread_fct == NULL) thread_fct = THREADcreate;
   if (thread_priority == 0) thread_priority = THREAD_PRIORITY;
};





/************************************************************************/
/*									*/
/*	WINDthread_init -- handle initialization for threads		*/
/*									*/
/************************************************************************/


void
WINDthread_init()
{
   THREADtype1_register(0,NULL,wind_thread_finish,NULL,NULL);
};





/************************************************************************/
/*									*/
/*	WINDdefine -- define a window					*/
/*									*/
/************************************************************************/


void
WINDdefine(wd)
   WIND_DEF wd;
{
   Character buf[1024];
   WIND_VIEW wv;
   String s,t;
   Sequence l;

   WINDinit();

   forin (wv,WIND_VIEW,l,all_views) {
      if (wv->topwindow == NULL && STREQL(wd->name,wv->origname)) break;
    };
   if (wv != NULL) return;

   wv = create_new_view();
   strcpy(buf,wd->name);
   s = index(buf,'@');
   if (s != NULL) {
      t = index(s+1,'@');
      if (t != NULL) {
	 *s++ = 0;
	 *t++ = 0;
	 wv->icon = SALLOC(s);
	 wv->iconfont = ASHloadfont(t);
       };
    }
   else {
      s = index(buf,'^');
      if (s != NULL) {
	 *s++ = 0;
	 wv->menuname = SALLOC(s);
       };
    };

   wv->defname = SALLOC(buf);
   wv->origname = SALLOC(wd->name);
   wv->name = SALLOC(buf);
   wv->rtn = wd->rtn;
   wv->usethreads = wd->usethreads;
   if (wd->action != NULL) wv->action = SALLOC(wd->action);
   wv->unique = wd->unique;
   wv->display = !wd->nogeneric;
   wv->nowindow = wd->nowindow;
   if (wv->rtn == NULL && wv->action != NULL) wv->nowindow = TRUE;

   WIND_draw_update();
};





/************************************************************************/
/*									*/
/*	WINDregister_thread_fct -- specify thread creation fct		*/
/*	WINDthread_stack_size -- set stack size for threads		*/
/*	WINDthread_priority -- set priority for threads 		*/
/*									*/
/************************************************************************/


void
WINDregister_thread_fct(fct)
   THREAD_FCT fct;
{
   thread_fct = fct;
};





void
WINDthread_stack_size(sz)
   Integer sz;
{
   thread_size = sz;
};





void
WINDthread_priority(pr)
   Integer pr;
{
   thread_priority = pr;
};





/************************************************************************/
/*									*/
/*	WINDactivate -- activate a button				*/
/*									*/
/************************************************************************/


ASH_WINDOW
WINDactivate(name,arg,opts)
   String name;
   String arg;
   String opts;
{
   Sequence l;
   WIND_VIEW wv;

   forin (wv,WIND_VIEW,l,all_views) {
      if (wv->topwindow == NULL && STREQL(name,wv->defname)) break;
    };
   if (wv == NULL) return NULL;

   wv = WIND_activate(wv,arg,opts);

   return (wv == NULL ? NULL : wv->window);
};





/************************************************************************/
/*									*/
/*	WIND_activate -- activate a WIND view				*/
/*									*/
/************************************************************************/


WIND_VIEW
WIND_activate(wv,arg,opt)
   WIND_VIEW wv;
   String arg;
   String opt;
{
   Boolean fg;
   ASH_WINDOW tw,w;
   WIND_VIEW_INFO wvi;
   WIND_VIEW wva;
   Character buf[1024];
   String nm,act;
   Sequence la;
   ASH_CURSOR cur;

   if (show_unique && wv->display && wv->unique && wv->window == NULL) {
      forin (wva,WIND_VIEW,la,all_views) {
	 if (wv != wva && STREQL(wva->defname,wv->defname) && !wva->display &&
		wva->window != NULL) {
	    ASHvisible(wva->topwindow,TRUE);
	    ASHpop(wva->topwindow);
	    return wva;
	  };
       };
    };

   fg = TRUE;
   if (wv->topwindow != NULL) {
      ASHvisible(wv->topwindow,TRUE);
      ASHpop(wv->topwindow);
    }
   else {
      if (wv->nowindow) {
	 w = NULL;
	 if (arg != NULL) {
	    wvi = *wv;
	    wv = &wvi;
	    wv->action = arg;
	  };
       }
      else {
	 if (opt != NULL) {
	    sprintf(buf,"%s %s",wv->name,opt);
	    nm = buf;
	  }
	 else nm = wv->name;
	 tw = WINDact_new(ASHinq_parent(WIND_inq_control_window()),nm);
	 if (tw == NULL) fg = FALSE;
	 else {
	    ASHvisible(tw,TRUE);
	    w = WINDsetup(tw);
	    BIOnew_input_window(tw);
	    BIOnew_input_window(w);
	    wv = duplicate_view(wv,tw,w);
	    if (arg != NULL) {
	       ASHset_window_name(w,arg);
	       if (wv->name != NULL) SFREE(wv->name);
	       wv->name = SALLOC(arg);
	     };
	  };
       };
      if (fg) {
	 if (wv->rtn != NULL) {
	    if (wv->usethreads && USE_THREADS) {
	       wv->thread = (*thread_fct)(wind_thread_begin,wv,0,TRUE,
					     thread_size,thread_priority);
	       fg = TRUE;
	     }
	    else if (w != NULL) {
	       cur = BIOset_cursor_standard(tw,ASH_CURSOR_WATCH);
	       if (wv->action == NULL) act = wv->name;
	       else act = wv->action;
	       fg = (*wv->rtn)(w,act);
	       if (!fg && ASHinq_valid_window(w)) ASHremove(w);
	       else if (fg) ASHvisible(w,TRUE);
	       if (ASHinq_valid_window(tw)) BIOset_cursor_pattern(tw,cur);
	     }
	    else {
	       fg = (*wv->rtn)(w,wv->action);
	     };
	  }
	 else if (wv->action != NULL) {
	    exec_command(wv->action);
	    fg = TRUE;
	  }
	 else fg = FALSE;
       }
      if (fg && wv != NULL && wv->unique && !wv->nowindow && !show_unique)
	 WIND_draw_update();
      else if (fg && wv != NULL && show_unique && wv->display)
	 WIND_draw_update();
    };

   return (fg ? wv : NULL);
};






/************************************************************************/
/*									*/
/*	WIND_inq_views_display -- return array of views to display	*/
/*	WIND_inq_views_menu -- return array of views in menus		*/
/*									*/
/************************************************************************/


Integer
WIND_inq_views_display(mx,vws)
   WIND_VIEW vws[];
{
   Integer ct;
   Sequence l,la;
   WIND_VIEW wv,wva;

   ct = 0;
   forin (wv,WIND_VIEW,l,all_views) {
      if (wv->display && wv->unique && wv->window == NULL) {
	 forin (wva,WIND_VIEW,la,all_views) {
	    if (wv != wva && STREQL(wva->defname,wv->defname)) {
	       if (!show_unique || wva->display) break;
	     };
	  };
	 if (wva != NULL) continue;
       };
      if (wv->display && wv->menuname == NULL) {
	 if (ct < mx) vws[ct++] = wv;
       };
    };

   return ct;
};





Integer
WIND_inq_views_menu(mx,vws)
   WIND_VIEW vws[];
{
   Integer ct;
   Sequence l,la;
   WIND_VIEW wv,wva;

   ct = 0;
   forin (wv,WIND_VIEW,l,all_views) {
      if (wv->display && wv->unique && wv->window == NULL) {
	 forin (wva,WIND_VIEW,la,all_views) {
	    if (wv != wva && STREQL(wva->defname,wv->defname)) {
	       if (!show_unique || wva->display) break;
	     };
	  };
	 if (wva != NULL) continue;
       };
      if (wv->display && wv->menuname != NULL) {
	 if (ct < mx) vws[ct++] = wv;
       };
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	wind_thread_begin -- start a thread for a new window		*/
/*									*/
/************************************************************************/


static void
wind_thread_begin(wv)
   WIND_VIEW wv;
{
   Boolean fg;
   String act;

   if (wv->window != NULL) {
      BIOnew_input_window(wv->window);
      if (wv->rtn != NULL) {
	 if (wv->action == NULL) act = wv->name;
	 else act = wv->action;
	 fg = (*wv->rtn)(wv->window,act);
       }
      else fg = TRUE;
      if (!fg && ASHinq_valid_window(wv->window)) ASHremove(wv->window);
      else if (ASHinq_valid_window(wv->window)) ASHvisible(wv->window,TRUE);
      else return;

      for ( ; ; ) {
	 RIPuser_pick(NULL);
       };
    }
   else if (wv->rtn != NULL) {
      (*wv->rtn)(wv->action);
    };
};





/************************************************************************/
/*									*/
/*	wind_thread_finish -- handle thread abort			*/
/*									*/
/************************************************************************/


static void
wind_thread_finish(th)
   THREAD th;
{
   WIND_VIEW wv;
   Sequence l;

   if (th == NULL) return;

   for ( ; ; ) {
      forin (wv,WIND_VIEW,l,all_views) {
	 if (wv->window != NULL && wv->thread == th) {
	    wv->thread = NULL;
	    ASHremove(wv->window);
	    break;
	  };
       };
      if (wv == NULL) break;
    };
};





/************************************************************************/
/*									*/
/*	create_new_view -- generate a WIND_VIEW structure		*/
/*									*/
/************************************************************************/


static WIND_VIEW
create_new_view()
{
   WIND_VIEW wv;

   wv = PALLOC(WIND_VIEW_INFO);
   wv->defname = NULL;
   wv->origname = NULL;
   wv->name = NULL;
   wv->icon = NULL;
   wv->iconfont = NULL;
   wv->menuname = NULL;
   wv->rtn = NULL;
   wv->action = NULL;
   wv->unique = FALSE;
   wv->display = FALSE;
   wv->nowindow = FALSE;
   wv->pid = 0;
   wv->topwindow = NULL;
   wv->window = NULL;
   wv->usethreads = FALSE;
   wv->thread = NULL;
   wv->region = NULL;
   wv->btn = NULL;

   PROTECT;
   all_views = APPEND(wv,all_views);
   UNPROTECT;

   return wv;
};





/************************************************************************/
/*									*/
/*	duplicate_view -- create a new view from generic one		*/
/*									*/
/************************************************************************/


static WIND_VIEW
duplicate_view(gwv,tw,w)
   WIND_VIEW gwv;
   ASH_WINDOW tw;
   ASH_WINDOW w;
{
   WIND_VIEW wv;

   wv = create_new_view();

   *wv = *gwv;
   wv->name = SALLOC(wv->name);
   wv->topwindow = tw;
   wv->window = w;
   if (show_unique && w != NULL && !wv->unique) wv->display = TRUE;
   else wv->display = FALSE;

   if (wv->topwindow != NULL) {
      ASHset_control(wv->topwindow,wind_view_control);
    };

   return wv;
};





/************************************************************************/
/*									*/
/*	wind_view_control -- handle control messages on views		*/
/*									*/
/************************************************************************/


static int
wind_view_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   WIND_VIEW wv;
   Sequence l;
   String s;

   if (STREQL(msg,"ASH$REMOVE")) {
      forin (wv,WIND_VIEW,l,all_views) {
	 if (wv->topwindow == w) break;
       };
      if (wv != NULL) {
	 PROTECT;
	 all_views = REMOB(wv,all_views);
	 UNPROTECT;
	 if (wv->display || wv->unique) WIND_draw_update();
       };
    }
   else if (STREQL(msg,"ASH$INVISIBLE")) {
      forin (wv,WIND_VIEW,l,all_views) {
	 if (wv->topwindow == w) break;
       };
      if (wv != NULL) {
	 s = ASHinq_window_name(wv->window);
	 if (s != NULL && (wv->name == NULL || STRNEQ(s,wv->name))) {
	    if (wv->name != NULL) SFREE(wv->name);
	    wv->name = SALLOC(s);
	  };
	 if (!wv->display && (!wv->unique || !show_unique)) {
	    wv->display = TRUE;
	    WIND_draw_update();
	  };
       };
    }
   else if (STREQL(msg,"ASH$VISIBLE")) {
      forin (wv,WIND_VIEW,l,all_views) {
	 if (wv->topwindow == w) break;
       };
      if (wv != NULL && wv->display && !show_unique) {
	 wv->display = FALSE;
	 WIND_draw_update();
       };
    };

   return ASH_CONTROL_REJECT;
};






/********************************************************************************/
/*										*/
/*	exec_command -- handle a shell command					*/
/*										*/
/********************************************************************************/


static void
exec_command(txt)
   String txt;
{
   String sh;
   Integer pid;

   sh = getenv("SHELL");
   if (sh == NULL) sh = "/bin/csh";

   if ((pid = vfork()) == 0) {
      execl(sh,"sh","-c",txt,0);
      _exit(127);
    };

   while (getpgrp(pid) > 0) {
      CMPXselect(0);
    };
};





/* end of winddefs.c */

