/************************************************************************/
/*									*/
/*		windcontrol.c						*/
/*									*/
/*	Control routines for windows					*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "wind_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MOVE_ICON	"wind.move"
#define SIZE_ICON	"wind.size"

#define ICON_HEIGHT	16
#define ICON_WIDTH	16

#define THREAD_STACK_SIZE	16384
#define THREAD_PRIORITY 	16





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Sequence	all_wins;
static	Boolean 	show_active;
static	WIND_SETUP_TYPE setup_type;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	int		wind_user_control();
static	int		handle_title_hit();
static	int		handle_move_btn();
static	int		handle_size_btn();
static	int		handle_remove_btn();
static	int		handle_iconify_btn();
static	int		handle_push_btn();
static	int		handle_pop_btn();





/************************************************************************/
/*									*/
/*	Icon definitions						*/
/*									*/
/************************************************************************/


static	Integer 	size_icon[] = {
   0xfff00000,
   0x84100000,
   0x84100000,
   0x84100000,
   0x84100000,
   0xfc100000,
   0x82500000,
   0x81500000,
   0x80d00000,
   0x83d00000,
   0x80100000,
   0xfff00000,
};



static Integer		move_icon[] = {
   0xff000000,
   0x81000000,
   0x81000000,
   0x81100000,
   0x81080000,
   0x817c0000,
   0x81080000,
   0x81100000,
   0x81000000,
   0x81000000,
   0x81000000,
   0xff000000,
};





/************************************************************************/
/*									*/
/*	LEAF definitions						*/
/*									*/
/************************************************************************/


#define ROOT_WINDOW	0
#define USER_WINDOW	1
#define UL_REGION	2
#define TITLE_REGION	3
#define UR_REGION	4
#define LL_REGION	5
#define LOWER_REGION_1	6
#define LOWER_REGION_2	7
#define LOWER_REGION_3	8
#define LOWER_REGION_4	9
#define LR_REGION	10
#define LOWER_REGION_0	11



static LEAF_DATA wind_leaf[] = {
   LEAF_ROOT(NULL),
   { UL_REGION, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP|LEAF_TYPE_ICONIC,
	{ LEAF_COORD_LX_LINE, LEAF_COORD_SIZE(LEAF_VALUE_PARAM(0)),
	     LEAF_COORD_SIZE(ICON_WIDTH), LEAF_COORD_TY_LINE },
	handle_move_btn, (int) MOVE_ICON },
   { TITLE_REGION, LEAF_TYPE_TITLE,
	{ LEAF_COORD_NEXT_LINE(UL_REGION), LEAF_COORD_SAME(UL_REGION),
	     LEAF_COORD_NEXT_LINE(UR_REGION), LEAF_COORD_TY_LINE },
	handle_title_hit, NULL },
   { UR_REGION, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP|LEAF_TYPE_ICONIC,
	{ LEAF_COORD_SIZE(ICON_WIDTH), LEAF_COORD_SAME(UL_REGION),
	     LEAF_COORD_RX_LINE, LEAF_COORD_TY_LINE },
	handle_size_btn, (int) SIZE_ICON },
   { USER_WINDOW, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX_LINE, LEAF_COORD_NEXT_LINE(LL_REGION),
	     LEAF_COORD_RX_LINE, LEAF_COORD_NEXT_LINE(UL_REGION) },
	NULL, NULL },
   { LL_REGION, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP|LEAF_TYPE_ICONIC,
	{ LEAF_COORD_LX_LINE, LEAF_COORD_BY_LINE,
	     LEAF_COORD_SIZE(ICON_WIDTH), LEAF_COORD_SIZE(LEAF_VALUE_PARAM(0)) },
	handle_size_btn, (int) SIZE_ICON },
   { LOWER_REGION_0, LEAF_TYPE_TEXT,
	{ LEAF_COORD_NEXT_LINE(LL_REGION), LEAF_COORD_BY_LINE,
	     LEAF_COORD_SIZE(16), LEAF_COORD_SAME(LL_REGION) },
	handle_pop_btn, NULL },
   { LOWER_REGION_1, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP,
	{ LEAF_COORD_NEXT(LOWER_REGION_0), LEAF_COORD_BY_LINE,
	     LEAF_COORD_TEXT_LINE, LEAF_COORD_SAME(LL_REGION) },
	handle_remove_btn, (int) " Hide " },
   { LOWER_REGION_2, LEAF_TYPE_TEXT_CENTERED,
	{ LEAF_COORD_NEXT(LOWER_REGION_1), LEAF_COORD_BY_LINE,
	     LEAF_COORD_NEXT(LOWER_REGION_3), LEAF_COORD_SAME(LL_REGION) },
	handle_pop_btn, NULL },
   { LOWER_REGION_3, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP,
	{ LEAF_COORD_TEXT_LINE, LEAF_COORD_BY_LINE,
	     LEAF_COORD_NEXT(LOWER_REGION_4), LEAF_COORD_SAME(LL_REGION) },
	handle_push_btn, (int) " Push " },
   { LOWER_REGION_4, LEAF_TYPE_TEXT,
	{ LEAF_COORD_SIZE(16), LEAF_COORD_BY_LINE,
	     LEAF_COORD_NEXT_LINE(LR_REGION), LEAF_COORD_SAME(LL_REGION) },
	handle_pop_btn, NULL },
   { LR_REGION, LEAF_TYPE_TEXT|LEAF_TYPE_SENSE_FLIP|LEAF_TYPE_ICONIC,
	{ LEAF_COORD_SIZE(ICON_WIDTH), LEAF_COORD_BY_LINE,
	     LEAF_COORD_RX_LINE, LEAF_COORD_SAME(LL_REGION) },
	handle_size_btn, (int) SIZE_ICON },
   LEAF_END
};



static LEAF_DATA wind_leaf_solo[] = {
   LEAF_ROOT(NULL),
   { USER_WINDOW, LEAF_TYPE_WINDOW|LEAF_TYPE_UPPER_LEFT|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY, LEAF_COORD_RX, LEAF_COORD_TY },
	NULL, NULL },
   LEAF_END
};




/************************************************************************/
/*									*/
/*	WIND_control_init -- module initialization			*/
/*									*/
/************************************************************************/


void
WIND_control_init()
{
   String s;

   all_wins = NULL;
   setup_type = WIND_SETUP_NONE;

   ASHicon_define(SIZE_ICON,12,12,size_icon);
   ASHicon_define(MOVE_ICON,14,12,move_icon);

   s = ASHinq_resource("show_active");
   if (s == NULL) show_active = TRUE;
   else if (STREQL(s,"off")) show_active = FALSE;
   else show_active = TRUE;
};





/************************************************************************/
/*									*/
/*	WINDset_setup_type -- set setup type				*/
/*									*/
/************************************************************************/


void
WINDset_setup_type(typ)
   String typ;
{
   WINDinit();

   if (STREQL(typ,"NO SETUP")) setup_type = WIND_SETUP_NONE;
   else if (STREQL(typ,"BUTTONS")) setup_type = WIND_SETUP_BUTTONS;
   else if (STREQL(typ,"SOLO")) setup_type = WIND_SETUP_SOLO;
};





/************************************************************************/
/*									*/
/*	WINDsetup -- handle setup					*/
/*									*/
/************************************************************************/


ASH_WINDOW
WINDsetup(w)
   ASH_WINDOW w;
{
   WIND_SETUP_TYPE wst;
   String s;

   WINDinit();

   s = ASHinq_window_resource(w,"setup");
   if (s == NULL) wst = setup_type;
   else if (STREQL(s,"NO SETUP")) wst = WIND_SETUP_NONE;
   else if (STREQL(s,"BUTTONS")) wst = WIND_SETUP_BUTTONS;
   else if (STREQL(s,"SOLO")) wst = WIND_SETUP_SOLO;
   else if (STREQL(s,"off") || STREQL(s,"OFF")) wst = WIND_SETUP_NONE;
   else wst = setup_type;

   switch (wst) {
      case WIND_SETUP_NONE :
	 break;
      case WIND_SETUP_SOLO :
	 w = WINDsetup_solo(w);
	 break;
      case WIND_SETUP_BUTTONS :
	 w = WINDsetup_buttons(w);
	 break;
    };

   return w;
};





/************************************************************************/
/*									*/
/*	WINDsetup_solo -- handle simple, no button windows		*/
/*									*/
/************************************************************************/


ASH_WINDOW
WINDsetup_solo(w)
   ASH_WINDOW w;
{
   ASH_WINDOW uw;
   WIND_WIN ww;

   WINDinit();

   ww = PALLOC(WIND_WIN_INFO);
   ww->window = w;
   ww->user_window = NULL;
   ww->title = FALSE;

   PROTECT;
   all_wins = APPEND(ww,all_wins);
   UNPROTECT;

   LEAFsetup_window(w,wind_leaf_solo,ww);

   LEAFset_data(ww,w,ROOT_WINDOW,0);

   LEAFredraw(w);

   uw = LEAFinq_window(w,USER_WINDOW,0);
   ww->user_window = uw;
   if (uw == NULL) return NULL;

   ASHset_control(uw,wind_user_control);

   return uw;
};





/************************************************************************/
/*									*/
/*	WINDsetup_buttons -- setup a new window with surrounding btns	*/
/*									*/
/************************************************************************/


ASH_WINDOW
WINDsetup_buttons(w)
   ASH_WINDOW w;
{
   RIP_REGION rr;
   ASH_WINDOW uw;
   WIND_WIN ww;
   Integer x,y;

   WINDinit();

   ww = PALLOC(WIND_WIN_INFO);
   ww->window = w;
   ww->user_window = NULL;
   ww->title = TRUE;

   PROTECT;
   all_wins = APPEND(ww,all_wins);
   UNPROTECT;

   LEAFsetup_window(w,wind_leaf,ww);

   ASHinq_text_next(w,"Xp_",&x,&y);
   ++y;
   if (y < ICON_HEIGHT) y = ICON_HEIGHT;
   LEAFset_parameters(w,1,y);

   LEAFset_data(ww,w,ROOT_WINDOW,0);

   LEAFredraw(w);

   uw = LEAFinq_window(w,USER_WINDOW,0);
   ww->user_window = uw;
   if (uw == NULL) return NULL;

   ASHset_control(uw,wind_user_control);

   rr = LEAFinq_rip_region(w,UL_REGION,0);
   RIPset_region_id(rr,"Move");
   rr = LEAFinq_rip_region(w,UR_REGION,0);
   RIPset_region_id(rr,"Size");
   rr = LEAFinq_rip_region(w,LL_REGION,0);
   RIPset_region_id(rr,"Size");
   rr = LEAFinq_rip_region(w,LR_REGION,0);
   RIPset_region_id(rr,"Size");

   return uw;
};





/********************************************************************************/
/*										*/
/*	WIND_wind_window -- find window for user action 			*/
/*										*/
/********************************************************************************/


ASH_WINDOW
WIND_wind_window(w)
   ASH_WINDOW w;
{
   Sequence l;
   WIND_WIN ww;

   forin (ww,WIND_WIN,l,all_wins) {
      if (ww->user_window == w) {
	 w = ww->window;
	 break;
       };
    };

   return w;
};





/********************************************************************************/
/*										*/
/*	WIND_control_active -- make window active or not			*/
/*										*/
/********************************************************************************/


void
WIND_control_active(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   Sequence l;
   WIND_WIN ww;

   forin (ww,WIND_WIN,l,all_wins) {
      if (ww->user_window == w) {
	 w = ww->window;
	 break;
       }
      else if (ww->window == w) break;
    };

   if (ww == NULL) return;
   if (ww->window == ww->user_window) return;

   LEAFset_active(ww->window,fg);
   LEAFredraw(ww->window);
};





/************************************************************************/
/*									*/
/*	wind_user_control -- control messages for user window		*/
/*									*/
/************************************************************************/


static int
wind_user_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   WIND_WIN ww;
   Sequence l;
   String s;
   Integer fg;

   if (STREQL(msg,"ASH$TITLE")) {
      forin (ww,WIND_WIN,l,all_wins) {
	 if (ww->user_window == w) break;
       };
      if (ww != NULL && ww->title) {
	 s = ASHinq_window_name(w);
	 LEAFset_data(SALLOC(s),ww->window,TITLE_REGION,0);
	 LEAFredraw(ww->window);
       };
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      forin (ww,WIND_WIN,l,all_wins) {
	 if (ww->user_window == w) break;
       };
      if (ww != NULL) {
	 ww->user_window = NULL;
	 ASHremove(ww->window);
	 ww->window = NULL;
       };
    }
   else if (show_active) {
      if (STREQL(msg,"ASH$ENTER")) fg = 1;
      else if (STREQL(msg,"ASH$EXIT")) fg = 0;
      else fg = -1;
      if (fg >= 0) {
	 forin (ww,WIND_WIN,l,all_wins) {
	    if (ww->user_window == w) break;
	  };
	 if (ww != NULL && ww->title) {
	    LEAFset_active(ww->window,fg);
	    LEAFredraw(ww->window);
	  };
       };
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	handle_title_hit -- handle hits on title window 		*/
/*	handle_move_btn -- handle MOVE button				*/
/*	handle_size_btn -- handle SIZE button				*/
/*	handle_remove_btn -- handle REMOVE button			*/
/*	handle_push_btn -- handle PUSH button				*/
/*	handle_pop_btn -- handle POP button				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
handle_title_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   w = RIPinq_window(rgn);

   if (btn & RIP_BTN_TAP) {
      fg = WINDact_pop(w,x,y);
    }
   else if (btn & (RIP_BTN_UP|RIP_BTN_NONE)) fg = FALSE;
   else {
      fg = WINDact_move(w,x,y);
    };

   return fg;
};





/*ARGSUSED*/

static int
handle_move_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   w = RIPinq_window(rgn);

   if (btn & RIP_BTN_TAP) {
      fg = WINDact_pop(w,x,y);
      return fg;
    };

   if (btn & (RIP_BTN_UP|RIP_BTN_NONE)) return FALSE;

   fg = WINDact_move(w,x,y);

   return fg;
};





/*ARGSUSED*/

static int
handle_size_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   w = RIPinq_window(rgn);

   if (btn & RIP_BTN_TAP) {
      fg = WINDact_pop(w,x,y);
      return fg;
    };

   if (btn & (RIP_BTN_UP|RIP_BTN_NONE)) return FALSE;

   fg = WINDact_size(w,x,y);

   return fg;
};





/*ARGSUSED*/

static int
handle_remove_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;
   static Integer lastbtn = 0;

   if (btn & RIP_BTN_DOWN) lastbtn = btn;

   if (btn & (RIP_BTN_DOWN|RIP_BTN_NONE)) return FALSE;

   w = RIPinq_window(rgn);

   if ((lastbtn & RIP_BTN_EXTRA) != 0 && (lastbtn & btn) != 0) {
      fg = WINDact_remove(w,x,y);
    }
   else {
      fg = WINDact_iconify(w,x,y);
    };

   return fg;
};





/*ARGSUSED*/

static int
handle_iconify_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   if (btn & (RIP_BTN_DOWN|RIP_BTN_NONE)) return FALSE;

   w = RIPinq_window(rgn);

   fg = WINDact_iconify(w,x,y);

   return fg;
};





/*ARGSUSED*/

static int
handle_push_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   if (btn & (RIP_BTN_DOWN|RIP_BTN_NONE)) return FALSE;

   w = RIPinq_window(rgn);

   fg = WINDact_push(w,x,y);

   return fg;
};





/*ARGSUSED*/

static int
handle_pop_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   Boolean fg;

   if (btn & (RIP_BTN_DOWN|RIP_BTN_NONE)) return FALSE;

   w = RIPinq_window(rgn);

   fg = WINDact_pop(w,x,y);

   return fg;
};





/* end of winduser.c */
