/************************************************************************/
/*									*/
/*		windact.c						*/
/*									*/
/*	Handle button action routines for wind				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "wind_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MIN_SIZE	100





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	ASH_WINDOW	real_root;

typedef struct _REPORT_INFO {
   Integer lx,by,rx,ty;
   Integer x0,y0;
   Boolean ldo,rdo,tdo,bdo;
   ASH_WINDOW w;
} REPORT_INFO;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	do_reporting();
static	Integer 	report_move();
static	Integer 	report_size();
static	ASH_WINDOW	get_real_parent();





/************************************************************************/
/*									*/
/*	WIND_act_init -- module initialization				*/
/*									*/
/************************************************************************/


void
WIND_act_init()
{
   real_root = NULL;
};





/************************************************************************/
/*									*/
/*	WINDact_move -- handle moving a act for window MOVE		*/
/*									*/
/************************************************************************/


int
WINDact_move(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   REPORT_INFO ri;
   Boolean fg;
   Integer lx,by,rx,ty;

   w = WIND_wind_window(w);

   fg = do_reporting(w,x,y,&ri,report_move);

   if (fg) {
      ASHpop(w);
      ASHinq_size(w,ASH_SIZE_VIEW,&lx,&by,&rx,&ty);
      ASHview(w,ri.lx,ri.by,lx,by,rx,ty);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_size -- handle moving a act for window RESIZE		*/
/*									*/
/************************************************************************/


int
WINDact_size(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   REPORT_INFO ri;
   Boolean fg;
   Integer xsz,ysz;

   w = WIND_wind_window(w);

   fg = do_reporting(w,x,y,&ri,report_size);

   if (fg) {
      xsz = abs(ri.rx-ri.lx);
      ysz = abs(ri.by-ri.ty);
      if (xsz < MIN_SIZE || ysz < MIN_SIZE) return FALSE;
      ASHresize(w,ri.lx,ri.by,0,ysz,xsz,0);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_pop -- handle popping a window to the top of stack	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

int
WINDact_pop(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   w = WIND_wind_window(w);

   ASHpop(w);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_push -- handle pushing a window to bottom of stack	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

int
WINDact_push(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   w = WIND_wind_window(w);

   ASHpush(w);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	WINDact_iconify -- handle making a window iconic		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

int
WINDact_iconify(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   w = WIND_wind_window(w);

   if (w == WIND_inq_control_window()) return FALSE;

   if (ASHinq_parent(w) != NULL) ASHvisible(w,FALSE);
   else if (WIND_inq_control_window() != NULL) ASHvisible(w,FALSE);
   else {
      ASHvisible(w,FALSE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_remove -- handle removing a window			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

int
WINDact_remove(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   if (!ASHinq_valid_window(w)) return FALSE;

   w = WIND_wind_window(w);

   if (w == WIND_inq_control_window()) return FALSE;

   ASHremove(w);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_enable -- make window active/inactive			*/
/*									*/
/************************************************************************/


int
WINDact_enable(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   WIND_control_active(w,fg);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	WINDact_new -- return a new window in given window		*/
/*									*/
/************************************************************************/


ASH_WINDOW
WINDact_new(w,name)
   ASH_WINDOW w;
   String name;
{
   Integer x0,y0,x1,y1;
   Integer lx,by;
   Character buf[1024];
   ASH_WINDOW sw;

   if (w == NULL) {
      sw = ASHroot_window(ASHinq_display_name(w),name);
      ASHset_window_defaults(sw);
      ASHinq_size(sw,ASH_SIZE_WINDOW,&x0,&y0,&x1,&y1);
      if (abs(x1-x0) < MIN_SIZE || abs(y1-y0) < MIN_SIZE) {
	 ASHremove(sw);
	 sw = NULL;
       };
    }
   else {
      w = WIND_wind_window(w);

      sprintf(buf,"Locate new instance of %s window",name);

      if (!RIPinput_box(w,buf,&x0,&y0,&x1,&y1)) return NULL;
      if (abs(x1-x0) < MIN_SIZE || abs(y1-y0) < MIN_SIZE) return NULL;

      lx = MIN(x0,x1);
      by = MAX(y0,y1);
      sw = ASHcreate(w,lx,by,0,0,abs(x1-x0),abs(y1-y0),0,ASH_WINDOW_VISIBLE|
			ASH_WINDOW_NOSAVE);
    };

   return sw;
};





/************************************************************************/
/*									*/
/*	do_reporting -- do rubberbanding with a track function		*/
/*									*/
/************************************************************************/


static Boolean
do_reporting(w,x,y,ri,fct)
   ASH_WINDOW w;
   Integer x,y;
   REPORT_INFO *ri;
   Function_Ptr fct;
{
   ASH_WINDOW pw;
   Integer lx,by,rx,ty;
   Universal ud;
   ASH_WINDOW ow;
   Integer cr,fg;
   ASH_CURSOR cur;
   THREAD th;

   pw = get_real_parent(w);

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHmap(w,lx,by,pw,&ri->lx,&ri->by);
   ASHmap(w,rx,ty,pw,&ri->rx,&ri->ty);
   ASHmap(ASHinq_top_window(w),x,y,pw,&ri->x0,&ri->y0);

   ud = ASHinq_user_data(pw);
   ASHset_user_data(pw,ri);
   ASHwindow_draw_thru(pw,TRUE);
   cr = ASHinq_combination_rule(pw);
   ASHdraw_hilite(pw,TRUE);
   ow = RIPgrab_window(pw);
   fg = BIOset_cursor(pw,TRUE);
   cur = BIOset_cursor_standard(pw,ASH_CURSOR_SMALL_XHAIRS);
   th = (THREAD) BIOset_window_thread(pw,THREADinq_current());
   RIPtrack(fct,-1,pw,FALSE);
   RIPgrab_window(ow);
   BIOset_window_thread(pw,th);
   BIOset_cursor_pattern(pw,cur);
   BIOset_cursor(pw,fg);
   ASHdraw_hilite(pw,FALSE);
   ASHwindow_draw_thru(pw,FALSE);
   ASHcombination_rule(pw,cr);
   ASHset_user_data(pw,ud);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	report_move -- handle move-type report button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
report_move(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer nlx,nby,nrx,nty;
   REPORT_INFO * ri;

   ri = (REPORT_INFO *) ASHinq_user_data(w);

   if (act == RIP_TRANS_NONE) {
      x = ri->x0;
      y = ri->y0;
    };

   if (act != RIP_TRANS_UP) {
      nlx = ri->lx+(x-ri->x0);
      nby = ri->by+(y-ri->y0);
      nrx = ri->rx+(x-ri->x0);
      nty = ri->ty+(y-ri->y0);
    };

   if (act != RIP_TRANS_NONE) ASHbox(w,ri->lx,ri->by,ri->rx,ri->ty);

   if (act != RIP_TRANS_UP) {
      ri->lx = nlx;
      ri->rx = nrx;
      ri->by = nby;
      ri->ty = nty;
      ri->x0 = x;
      ri->y0 = y;
      ASHbox(w,nlx,nby,nrx,nty);
    };

   if (act == RIP_TRANS_UP) return FALSE;

   return TRUE;
};






/************************************************************************/
/*									*/
/*	report_size -- handle size-type report button			*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
report_size(x,y,ct,act,max,w)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
{
   Integer nlx,nby,nrx,nty;
   REPORT_INFO * ri;

   ri = (REPORT_INFO *) ASHinq_user_data(w);

   if (act == RIP_TRANS_NONE) {
      x = ri->x0;
      y = ri->y0;
      ri->ldo = (abs(x-ri->lx)*2 < abs(x-ri->rx));
      ri->rdo = (abs(x-ri->rx)*2 < abs(x-ri->lx));
      ri->tdo = (abs(y-ri->ty)*2 < abs(y-ri->by));
      ri->bdo = (abs(y-ri->by)*2 < abs(y-ri->ty));
      if (!ri->ldo && !ri->rdo && !ri->bdo && !ri->tdo) {
	 ri->rdo = TRUE;
	 ri->bdo = TRUE;
       };
    };

   if (act != RIP_TRANS_UP) {
      nlx = (ri->ldo && (x < ri->rx)) ? x : ri->lx;
      nrx = (ri->rdo && (x > ri->lx)) ? x : ri->rx;
      nby = (ri->bdo && (y > ri->ty)) ? y : ri->by;
      nty = (ri->tdo && (y < ri->by)) ? y : ri->ty;
    };

   if (act != RIP_TRANS_NONE) ASHbox(w,ri->lx,ri->by,ri->rx,ri->ty);

   if (act != RIP_TRANS_UP) {
      ri->lx = nlx;
      ri->rx = nrx;
      ri->by = nby;
      ri->ty = nty;
      ASHbox(w,ri->lx,ri->by,ri->rx,ri->ty);
    };

   if (act == RIP_TRANS_UP) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	get_real_parent -- get parent window if there is none		*/
/*									*/
/************************************************************************/


static ASH_WINDOW
get_real_parent(w)
   ASH_WINDOW w;
{
   ASH_WINDOW pw;

   pw = ASHinq_parent(w);

   if (pw == NULL) {
      if (real_root == NULL) {
	 real_root = ASHroot_X_window(w);
       };
      pw = real_root;
    };

   return pw;
};




/* end of windact.c */
