/************************************************************************/
/*									*/
/*		tulipmain.c						*/
/*									*/
/*	Main module for terminal utility package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#define TULIP_MAIN

#include "tulip_local.h"






/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/


	Boolean 	TULIP__initfg = 0;	/* initialization state */
	Integer 	TULIP__tracelvl = 0;

	PROT_DECL;

static	Integer 	default_font;
static	Integer 	maxline;
static	Integer 	maxcolumn;
static	Integer 	minheight;
static	Integer 	minwidth;

static	TULIP_INFO	first_terminal = NULL;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


#ifdef LIB
static	void		tulip_cleanup();
#endif

static	Integer 	tulip_control();
static	void		tulip_refresh();
static	void		tulip_setup_window();
static	void		clear_window();
static	void		close_tulip();




/************************************************************************/
/*									*/
/*	TULIPinit -- initialize the module				*/
/*									*/
/************************************************************************/


void
TULIPinit()
{
   ITRACE("TULIPinit");

   PROT_INIT;
   PROTECT;
   if (TULIP__initfg == 0) {
#ifdef LIB
      BROWNregister_cleanup(tulip_cleanup);
#endif

      TULIP__initfg = 1;

      maxline = 0;
      maxcolumn = 0;
      minheight = 0;
      minwidth = 0;
      first_terminal = NULL;

      TULIP_user_init();
      TULIP_exec_init();
      TULIP_line_init();

      default_font = TULIP_getfont(ASHinq_base_font());
    };
   UNPROTECT;
};






#ifdef LIB

static void
tulip_cleanup()
{
   TULIP__tracelvl = 0;
   TULIP__initfg = 0;
};

#endif




/************************************************************************/
/*									*/
/*	TULIPtrace -- set trace flag					*/
/*									*/
/************************************************************************/


void
TULIPtrace(lvl)
   Integer lvl;
{
   TULIP__tracelvl = lvl;
};






/************************************************************************/
/*									*/
/*	TULIPset_limits -- set limits for determining # lines/columns	*/
/*									*/
/************************************************************************/


void
TULIPset_limits(line,col,ht,wd)
   Integer line;
   Integer col;
   Integer ht;
   Integer wd;
{
   ENTER("TULIPset_limits %d %d %d %d",line,col,ht,wd);

   PROTECT;

   if (line >= 0) maxline = line;
   if (col >= 0) maxcolumn = col;
   if (ht >= 0) minheight = ht;
   if (wd >= 0) minwidth = wd;

   UNPROTECT;
};






/************************************************************************/
/*									*/
/*	TULIPopen -- open a terminal in current window			*/
/*									*/
/************************************************************************/


TULIP_ID
TULIPopen(w)
   ASH_WINDOW w;
{
   register TULIP_INFO ti;
   register Integer i;
   String s;

   ENTER("TULIPopen");

   ASHset_control(w,tulip_control);

   ti = PALLOC(TULIP_INFO_BLOCK);

   ti->window = NULL;
   tulip_setup_window(ti,w);

   s = ASHinq_window_resource(w,"font");
   if (s == NULL) ti->curfont = default_font;
   else ti->curfont = TULIP_getfont(s);

   ti->curline = 0;
   ti->curchar = 0;
   ti->modes = TULIP_DEFAULT_MODES;
   ti->insert_mode = FALSE;
   ti->read_string = NULL;
   ti->box = NULL;
   ti->input_state = TULIP_INPUT_BEGIN;
   ti->bufcnt = 0;
   ti->cursor_on = FALSE;
   ti->discard_on = FALSE;
   ti->cursor_type = TULIP_CURSOR_BLOCK;
   ti->lastfont = ti->curfont;

   ti->fg = ASHinq_color(w);
   ti->bg = ASHinq_background_color(w);
   ti->caret_bitmap = NULL;

   for (i = 0; i < NUM_ALTS; ++i)
      ti->alts[i] = NULL;

   if (!L_PROTECT(ti)) return NULL;

   if (ASHinq_saved(w)) {
      ti->modes |= TULIP_MODE_SAVED;
    }
   else {
      ASHset_region_refresh(w,tulip_refresh);
    };

   ASHset_user_data(w,ti);

   ti->next = NULL;

   ASHbatch_mode(TRUE);

   TULIP_setup_font(ti);
   TULIP_setup_space(ti,ti->curfont);
   TULIP_setup_lines(ti);
   TULIP_draw_all(ti);
   TULIP_set_cursor(ti,TRUE);

   ASHbatch_mode(FALSE);

   PROTECT;
   ti->next = first_terminal;
   first_terminal = ti;
   UNPROTECT;

   L_UNPROTECT(ti);

   return (TULIP_ID) ti;
};





/************************************************************************/
/*									*/
/*	TULIPclose -- close specified tulip terminal			*/
/*									*/
/************************************************************************/


void
TULIPclose(ti)
   TULIP_INFO ti;
{
   ENTER("TULIPclose 0x%x",ti);

   PROTECT;

   close_tulip(ti);

   free(ti);

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	TULIPdefine_alternate -- define alternative window		*/
/*									*/
/************************************************************************/


void
TULIPdefine_alternative(curtulip,id,ti)
   TULIP_INFO curtulip;
   Integer id;
   TULIP_INFO ti;
{
   TRACE("TULIPdefine_alternative 0x%x %d 0x%x",curtulip,id,ti);

   if (id == 0) id = 1;
   ALTS[id] = ti;
};





/************************************************************************/
/*									*/
/*	TULIP_trace -- output trace information 			*/
/*	TULIP_error -- output error information 			*/
/*									*/
/************************************************************************/


void
TULIP_trace(msg,a1,a2,a3,a4,a5,a6,a7,a8,a9)
   String msg;
   Integer a1,a2,a3,a4,a5,a6,a7,a8,a9;
{
   Character mbf[1024];

   sprintf(mbf,msg,a1,a2,a3,a4,a5,a6,a7,a8,a9);

   printf("TULIP: %s\n",mbf);
};






void
TULIP_error(msg,a1,a2,a3,a4,a5,a6,a7,a8,a9)
   String msg;
   Integer a1,a2,a3,a4,a5,a6,a7,a8,a9;
{
   Character mbf[1024];

   sprintf(mbf,msg,a1,a2,a3,a4,a5,a6,a7,a8,a9);

   fprintf(stderr,"TULIP: %s\n",mbf);
   fflush(stderr);
};






/********************************************************************************/
/*										*/
/*	TULIP_protect -- lock/unlock this terminal emulator			*/
/*										*/
/********************************************************************************/


Boolean
TULIP_protect(curtulip,fg)
   TULIP_INFO curtulip;
   Boolean fg;
{
   Boolean rfg;

   if (curtulip->window == NULL) rfg = FALSE;
   else if (fg) {
      rfg = ASHlock(VIEWPORT);
    }
   else {
      ASHunlock(VIEWPORT);
      rfg = TRUE;
    };

   return rfg;
};





/************************************************************************/
/*									*/
/*	tulip_control -- handle ASH control messages for window 	*/
/*									*/
/************************************************************************/


static Integer
tulip_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register TULIP_INFO ti;

   ITRACE("tulip_control %s 0x%x",msg,w);

   ti = (TULIP_INFO) ASHinq_user_data(w);

   if (ti == NULL) ;
   else if (ti->window == NULL || ti->window->window != w);
   else if (STREQL(msg,"ASH$RESIZE")) {
      if (L_PROTECT(ti)) {
	 clear_window(ti);
	 tulip_setup_window(ti,w);
	 TULIP_setup_lines(ti);
/*	 TULIP_draw_all(ti);		*/
/*	 TULIP_set_cursor(ti,TRUE);	*/
	 L_UNPROTECT(ti);
       };
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      if (L_PROTECT(ti)) {
	 close_tulip(ti);
	 L_UNPROTECT(ti);
       };
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	tulip_refresh -- refresh TULIP window				*/
/*									*/
/************************************************************************/


static void
tulip_refresh(w)
   ASH_WINDOW w;
{
   register TULIP_INFO ti;

   ITRACE("tulip_refresh 0x%x",w);

   ti = (TULIP_INFO) ASHinq_user_data(w);

   if (ti == NULL) return;

   if (L_PROTECT(ti)) {
      TULIP_draw_all(ti);
      TULIP_set_cursor(ti,TRUE);
      ti->lastfont = -1;
      L_UNPROTECT(ti);
    };
};





/************************************************************************/
/*									*/
/*	tulip_setup_window -- setup window for tulip terminal		*/
/*									*/
/************************************************************************/


static void
tulip_setup_window(ti,w)
   TULIP_INFO ti;
   ASH_WINDOW w;
{
   Integer lx,by,rx,ty;

   DTRACE("tulip_setup_window 0x%x 0x%x",ti,w);

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (ti->window == NULL) {
      ti->window = PALLOC(TULIP_WINDOW_BLOCK);
    };

   ti->window->window = w;
   ti->window->lx = lx;
   ti->window->ty = ty;
   ti->window->xsize = abs(rx-lx);
   ti->window->ysize = abs(by-ty);
   ti->window->dirx = (lx < rx ? 1 : -1);
   ti->window->diry = (ty < by ? 1 : -1);

   if (maxline > 0) ti->maxline = maxline;
   else if (minheight > 0) ti->maxline = (abs(by-ty)+minheight)/minheight + 1;
   else ti->maxline = MIN(DEFAULT_MAX_LINE,(abs(by-ty)+DEFAULT_MIN_HT)/DEFAULT_MIN_HT);

   if (maxcolumn > 0) ti->maxcolumn = maxcolumn;
   else if (minwidth > 0) ti->maxcolumn = (abs(rx-lx)+minwidth)/minwidth;
   else ti->maxcolumn = MIN(DEFAULT_MAX_COL,(abs(rx-lx)+DEFAULT_MIN_WD)/DEFAULT_MIN_WD);

   ti->curline = 0;
   ti->curchar = 0;
   ti->insert_mode = FALSE;
   ti->box = NULL;
   ti->input_state = TULIP_INPUT_BEGIN;
   ti->bufcnt = 0;

   ti->lines = (TULIP_LINE) calloc(ti->maxline,sizeof(TULIP_LINE_BLOCK));
};





/************************************************************************/
/*									*/
/*	clear_window -- clear storage for window			*/
/*									*/
/************************************************************************/


static void
clear_window(ti)
   TULIP_INFO ti;
{
   register TULIP_LINE tl;
   register Integer i;

   DTRACE("clear_window 0x%x",ti);

   tl = ti->lines;
   for (i = 0; i < ti->maxline; ++i) {
      if (tl->chars != NULL) free(tl->chars);
      else break;
      tl->chars = NULL;
      ++tl;
    };

   free(ti->lines);
   ti->lines = NULL;
};





/********************************************************************************/
/*										*/
/*	close_tulip -- close a tulip editor					*/
/*										*/
/********************************************************************************/


static void
close_tulip(ti)
   TULIP_INFO ti;
{
   TULIP_INFO tn;

   PROTECT;
   for (tn = first_terminal; tn != NULL && tn != ti; tn = tn->next);

   if (tn != NULL) {
      if (ti->window != NULL && ti->window->window != NULL) {
	 ASHremove_control(ti->window->window,tulip_control);
	 ASHset_user_data(ti->window->window,NULL);
	 ASHset_refresh(ti->window->window,NULL);
       };

      clear_window(ti);

      free(ti->window);
      ti->window = NULL;

      if (first_terminal == ti) first_terminal = ti->next;
      else {
	 for (tn = first_terminal; tn->next != ti; tn = tn->next);
	 tn->next = ti->next;
       };
    };

   UNPROTECT;
};






/* end of tulipmain.c */
