/************************************************************************/
/*									*/
/*		tuliplines.c						*/
/*									*/
/*	Output routines for terminal utility package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "tulip_local.h"






/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/


static	TULIP_FONT	fonttbl[NFONT];
static	Boolean 	use_colors;





/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


typedef struct _COLOR_MAP {
   String name;
   Integer value;
   Integer fvalue;
} COLOR_MAP;

static COLOR_MAP colormap[] = {
   { "white", 0, 1 },
   { "black", 0, 0 },
   { "green", 0, 1 },
   { "red",   0, 1 },
   { "blue",  0, 0 },
   { "yellow", 0, 1 },
   { "cyan",  0, 1 },
   { "brown", 0, 1 },
   { 0, 0, 0}
};




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		blank_line();
static	Boolean 	move_lines();
static	void		clear_line_region();
static	void		adjust_height();
static	void		clear_char_region();
static	Boolean 	move_in_line();
static	void		check_bottom();
static	void		check_right();
static	void		out_line();
static	void		out_string();
static	void		out_char();
static	void		set_font();
static	Boolean 	do_move();
static	void		do_clear();
static	void		create_caret_bitmap();
static	void		clear_cursor();
static	void		set_caret_cursor();
static	ASH_COLOR	get_color();





/************************************************************************/
/*									*/
/*	Macro definitions						*/
/*									*/
/************************************************************************/


#define char_info(ch,ft,upp,downp,widthp)		\
      { register TULIP_FONT _f; 			\
	_f = fonttbl[ft];				\
	*(upp) = _f->up[ch];				\
	*(downp) = _f->down[ch];			\
	*(widthp) = _f->width[ch];			\
      }





/************************************************************************/
/*									*/
/*	TULIP_line_init -- module initialization			*/
/*									*/
/************************************************************************/


void
TULIP_line_init()
{
   Integer i;
   ASH_WINDOW w;

   ITRACE("TULIP_line_init");

   for (i = 0; i < NFONT; ++i)
      fonttbl[i] = NULL;

   w = ASHinq_top();

   colormap[0].value = ASHinq_background(w);
   colormap[0].fvalue = ASHinq_foreground(w);
   colormap[1].value = colormap[0].fvalue;
   colormap[1].fvalue = colormap[0].value;

   for (i = 2; colormap[i].name != NULL; ++i) {
      if (ASHinq_configuration_depth(w) == 1) {
	 use_colors = FALSE;
	 colormap[i].value = ASHinq_foreground(w);
       }
      else {
	 use_colors = TRUE;
	 colormap[i].value = ASHlookup_color(w,colormap[i].name);
       };
    };

   for (i = 2; colormap[i].name != NULL; ++i) {
      if (ASHinq_configuration_depth(w) == 1) {
	 colormap[i].fvalue = ASHinq_background(w);
       }
      else {
	 colormap[i].fvalue = colormap[colormap[i].fvalue].value;
	 if (colormap[i].fvalue == colormap[i].value) {
	    if (colormap[i].fvalue == colormap[0].fvalue)
	       colormap[i].fvalue = colormap[1].fvalue;
	    else colormap[i].fvalue = colormap[0].fvalue;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	TULIP_getfont -- load a font and return its id			*/
/*									*/
/************************************************************************/


Integer
TULIP_getfont(nm)
   String nm;
{
   register Integer i,j;
   Integer hsp,vsp,ssz,vsp1;
   register Integer mxup,mxdwn;
   register Integer minwd;
   register Boolean safe;
   Integer uwd;

   ITRACE("TULIP_getfont %s",nm);

   PROTECT;
   for (i = 0; i < USER_FONTS; ++i) {
      if (fonttbl[i] == NULL) break;
      else if (strcmp(nm,fonttbl[i]->name) == 0) {
	 UNPROTECT;
	 return i;
       };
    };

   if (i >= USER_FONTS) {
      UNPROTECT;
      ERROR(Too many fonts);
      return 0;
    };

   fonttbl[i] = PALLOC(TULIP_FONT_BLOCK);
   fonttbl[i]->name = SALLOC(nm);

   if (ASHinq_font_info(nm,&hsp,&vsp,&ssz,
			   fonttbl[i]->width,
			   fonttbl[i]->down,
			   fonttbl[i]->up) == 0) {
      UNPROTECT;
      ERROR(Cant load font);
      return 0;
    };

   if (vsp <= 0) vsp = 1;
   vsp1 = vsp/2;
   vsp -= vsp1;

   mxup = 0;
   mxdwn = 0;
   minwd = 0;
   safe = (hsp == 0);
   uwd = fonttbl[i]->width[' '];

   for (j = 0; j < FONTCHARS; ++j) {
      DTRACE("\tCHAR 0x%x -> %d %d %d",j,
		fonttbl[i]->width[j],
		fonttbl[i]->up[j],
		fonttbl[i]->down[j]);
      if (fonttbl[i]->width[j] == 0) fonttbl[i]->width[j] = ssz;
      fonttbl[i]->width[j] += hsp;
      fonttbl[i]->down[j] += vsp;
      fonttbl[i]->up[j] += vsp1;
      if (fonttbl[i]->up[j] > mxup) mxup = fonttbl[i]->up[j];
      if (fonttbl[i]->down[j] > mxdwn) mxdwn = fonttbl[i]->down[j];
      if (fonttbl[i]->width[j] != 0 &&
	     (minwd == 0 || fonttbl[i]->width[j] < minwd))
	 minwd = fonttbl[i]->width[j];
      if (fonttbl[i]->width[j] != uwd)
	 if (j >= ' ' && j != 127) safe = FALSE;
    };

   for (j = 0; j < FONTCHARS; ++j) {
      fonttbl[i]->up[j] = mxup;
      fonttbl[i]->down[j] = mxdwn;
    };

   fonttbl[i]->safe = safe;

   fonttbl[i]->fontid = ASHloadfont(nm);

   for (j = i+USER_FONTS; j < NFONT; j += USER_FONTS) fonttbl[j] = fonttbl[i];

   if (i == 0) TULIPset_limits(0,0,mxup+mxdwn,minwd);

   UNPROTECT;

   return i;
};





/************************************************************************/
/*									*/
/*	TULIP_setup_space -- define font for spaces inserted ourselves	*/
/*									*/
/************************************************************************/


void
TULIP_setup_space(curtulip,sf)
   register TULIP_INFO curtulip;
   Integer sf;
{
   Integer up,down,width;

   ITRACE("TULIP_setup_space 0x%x %d",curtulip,sf);

   char_info(' ',sf,&up,&down,&width);

   SPACEFONT = sf;
   SPACEUP = up;
   SPACEDOWN = down;
   SPACEWIDTH = width;
};





/************************************************************************/
/*									*/
/*	TULIP_setup_lines -- setup initial lines for terminal		*/
/*									*/
/************************************************************************/


void
TULIP_setup_lines(curtulip)
   register TULIP_INFO curtulip;
{
   register Integer i,y;
   register TULIP_LINE lp;

   ITRACE("TULIP_setup_lines 0x%x",curtulip);

   y = VERT_INDENT;
   lp = LINE(0);
   for (i = 0; i < MAXLINE; ++i) {
      blank_line(curtulip,lp);
      lp->ypos = y;
      y += HEIGHT(lp);
      ++lp;
      if (y > BOTTOM) break;
    };

   if (i == 0) i = 1;
   NUMLINE = i;
   clear_line_region(curtulip,0,NUMLINE-1);

   clear_cursor(curtulip);
};





/************************************************************************/
/*									*/
/*	TULIP_setup_font -- force current font for window		*/
/*									*/
/************************************************************************/


void
TULIP_setup_font(curtulip)
   register TULIP_INFO curtulip;
{
   ITRACE("TULIP_setup_font 0x%x",curtulip);

   LASTFONT = -1;

   set_font(curtulip,CURFONT);
};





/************************************************************************/
/*									*/
/*	TULIP_draw_all -- redraw entire screen				*/
/*									*/
/************************************************************************/


void
TULIP_draw_all(curtulip)
   register TULIP_INFO curtulip;
{
   register TULIP_LINE lp;
   register Integer i;

   ITRACE("TULIP_draw_all 0x%x",curtulip);

   LASTFONT = -1;

   clear_cursor(curtulip);

   clear_line_region(curtulip,0,NUMLINE-1);

   lp = LINE(0);
   for (i = 0; i < NUMLINE; ++i) out_line(curtulip,lp++);
};





/************************************************************************/
/*									*/
/*	TULIP_scroll_up -- scroll region up # lines			*/
/*									*/
/************************************************************************/


void
TULIP_scroll_up(curtulip,from,to,ct)
   register TULIP_INFO curtulip;
   Integer from;
   Integer to;
   Integer ct;
{
   register Integer i,j,y,dy,y1;
   register TULIP_LINE lpa,lpb;
   TULIP_LINE_BLOCK temp[MAX_SCROLL];

   ITRACE("TULIP_scroll_up 0x%x %d %d %d",curtulip,from,to,ct);

   if (from < 0) from = 0;
   if (to > NUMLINE-1) to = NUMLINE-1;
   if (ct >= to-from+1) {
      TULIP_clear_lines(curtulip,from,to);
      return;
    };

   if (CURSORON && CURLINE >= from && CURLINE <= to)
      TULIP_set_cursor(curtulip,FALSE);

   lpa = LINE(to);
   y1 = LBOT(lpa);

   lpa = LINE(from);

   lpb = lpa;
   for (i = 0; i < ct; ++i) temp[i] = *lpb++;

   dy = LTOP(lpb)-LTOP(lpa);
   for (i = from+ct; i <= to; ++i) *lpa++ = *lpb++;

   i -= ct;
   move_lines(curtulip,from,i-1,-dy);
   lpb = LINE(i-1);
   y = LBOT(lpb);
   for (j = 0; j < ct; ++j) {
      *lpa = temp[j];
      blank_line(curtulip,lpa);
      lpa->ypos = y;
      y += HEIGHT(lpa);
      ++lpa;
    };

   dy = y-y1;
   if (dy != 0) {
      move_lines(curtulip,from+1,NUMLINE-1,dy);
      check_bottom(curtulip);
    };
};





/************************************************************************/
/*									*/
/*	TULIP_scroll_down -- scroll down region # lines 		*/
/*									*/
/************************************************************************/


void
TULIP_scroll_down(curtulip,from,to,ct)
   register TULIP_INFO curtulip;
   Integer from;
   Integer to;
   Integer ct;
{
   register Integer i,j,y,dy;
   register TULIP_LINE lpa,lpb;
   Integer y1;
   TULIP_LINE_BLOCK temp[MAX_SCROLL];

   ITRACE("TULIP_scroll_down 0x%x %d %d %d",curtulip,from,to,ct);

   if (from < 0) from = 0;
   if (to >= NUMLINE) to = NUMLINE-1;
   if (ct >= to-from+1) {
      TULIP_clear_lines(curtulip,from,to);
      return;
    };

   if (CURSORON && CURLINE >= from && CURLINE <= to)
      TULIP_set_cursor(curtulip,FALSE);

   lpa = LINE(to);
   y1 = LBOT(lpa);

   lpb = lpa;
   for (j = 0; j < ct; ++j) temp[j] = *lpb--;

   for (i = to-ct; i >= from; --i) *lpa-- = *lpb--;

   i += ct;
   lpb = LINE(to);
   dy = y1-LBOT(lpb);
   move_lines(curtulip,i+1,to,dy);

   lpa = LINE(from);
   y = LTOP(lpa);
   for (i = 0; i < ct; ++i) {
      *lpa = temp[i];
      blank_line(curtulip,lpa);
      lpa->ypos = y;
      y += HEIGHT(lpa);
      ++lpa;
    };

   dy = y-LTOP(lpa);
   if (dy != 0) {
      move_lines(curtulip,from+ct,NUMLINE-1,dy);
      check_bottom(curtulip);
    };
};




/************************************************************************/
/*									*/
/*	TULIP_clear_lines -- clear region				*/
/*									*/
/************************************************************************/


void
TULIP_clear_lines(curtulip,from,to)
   register TULIP_INFO curtulip;
   Integer from;
   Integer to;
{
   register Integer y,y0,y1,i;
   register TULIP_LINE lp;

   ITRACE("TULIP_clear_lines 0x%x %d %d",curtulip,from,to);

   if (CURLINE >= from && CURLINE <= to) clear_cursor(curtulip);

   lp = LINE(to);
   y1 = LBOT(lp);

   lp = LINE(from);
   y0 = LTOP(lp);
   y = y0;
   for (i = from; i <= to; ++i) {
      blank_line(curtulip,lp);
      lp->ypos = y;
      y += HEIGHT(lp);
      ++lp;
    };

   if (y != y1) {
      move_lines(curtulip,to+1,NUMLINE-1,y-y1);
      check_bottom(curtulip);
    };

   clear_line_region(curtulip,from,to);
};





/************************************************************************/
/*									*/
/*	TULIP_out_char -- output single character			*/
/*									*/
/************************************************************************/


void
TULIP_out_char(curtulip,ch)
   register TULIP_INFO curtulip;
   Character ch;
{
   register TULIP_LINE lp;
   register TULIP_CHAR cp;
   register Boolean fg,fga;
   Integer oup,odown,owidth;
   Integer nup,ndown,nwidth;

   ITRACE("TULIP_out_char 0x%x %c",curtulip,ch);

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);
   clear_cursor(curtulip);
   if (ch == cp->charv && CURFONT == cp->font) return;

   char_info(cp->charv,cp->font,&oup,&odown,&owidth);
   char_info(ch,CURFONT,&nup,&ndown,&nwidth);

   cp->charv = ch;
   cp->font = CURFONT;

   fg = FALSE;
   if (nup != oup) {
      if (nup > lp->up) fg = TRUE;
      else if (nup > oup && nup == lp->up) ++lp->nmaxup;
      else if (oup == lp->up && --(lp->nmaxup) == 0) fg = TRUE;
    };
   if (ndown != odown) {
      if (ndown > lp->down) fg = TRUE;
      else if (ndown > odown && ndown == lp->down) ++lp->nmaxdown;
      else if (odown == lp->down && --(lp->nmaxdown) == 0) fg = TRUE;
    };
   if (fg) adjust_height(curtulip,CURLINE);

   if (owidth != nwidth) fga = move_in_line(curtulip,lp,CURCHAR+1,nwidth-owidth);
   else fga = FALSE;

   if (!fga) {
      out_char(curtulip,lp,cp);
    };

   if (owidth != nwidth) check_right(curtulip,lp);
   if (fg) check_bottom(curtulip);
};





/************************************************************************/
/*									*/
/*	TULIP_out_string -- output string of characters 		*/
/*									*/
/************************************************************************/


void
TULIP_out_string(curtulip,str)
   register TULIP_INFO curtulip;
   String str;
{
   register Character ch;
   register TULIP_LINE lp;
   register TULIP_CHAR cp;
   register Boolean fg,fga;
   register Integer dwidth,x,pos;
   Integer oup,odown,owidth;
   Integer nup,ndown,nwidth;

   ITRACE("TULIP_out_string 0x%x `%s`",curtulip,str);

   if (str == NULL || *str == 0) return;

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);
   clear_cursor(curtulip);

   fg = FALSE;
   dwidth = 0;
   x = cp->xpos;
   pos = CURCHAR;
   while ((ch = *str++) != 0) {
      if (pos >= lp->nchars) {
	 lp->nchars = pos+1;
	 oup = SPACEUP;
	 odown = SPACEDOWN;
	 owidth = SPACEWIDTH;
       }
      else {
	 char_info(cp->charv,cp->font,&oup,&odown,&owidth);
       };
      char_info(ch,CURFONT,&nup,&ndown,&nwidth);
      dwidth += nwidth-owidth;
      cp->charv = ch;
      cp->font = CURFONT;
      cp->xpos = x;
      x += nwidth;

      if (nup != oup) {
	 if (nup > lp->up) fg = TRUE;
	 else if (nup > oup && nup == lp->up) ++lp->nmaxup;
	 else if (oup == lp->up && --(lp->nmaxup) == 0) fg = TRUE;
       };
      if (ndown != odown) {
	 if (ndown > lp->down) fg = TRUE;
	 else if (ndown > odown && ndown == lp->down) ++lp->nmaxdown;
	 else if (odown == lp->down && --(lp->nmaxdown) == 0) fg = TRUE;
       };
      ++cp;
      ++pos;
      if (pos >= MAXCOLUMN) break;
    };

   if (fg) adjust_height(curtulip,CURLINE);

   if (dwidth != 0) fga = move_in_line(curtulip,lp,pos,dwidth);
   else fga = FALSE;

   if (!fga) out_string(curtulip,lp,CURCHAR,pos-1);

   if (dwidth != 0) check_right(curtulip,lp);
   if (fg) check_bottom(curtulip);
};





/************************************************************************/
/*									*/
/*	TULIP_insert_char -- insert (ct) instances of (ch)		*/
/*									*/
/************************************************************************/


void
TULIP_insert_char(curtulip,ch,ft,ct)
   register TULIP_INFO curtulip;
   Integer ch;
   Integer ft;
   Integer ct;
{
   register TULIP_LINE lp;
   register TULIP_CHAR cp,cpa,cpb;
   register Integer i,j,k,x;
   register Boolean fg;
   Integer nup,ndown,nwidth;

   ITRACE("TULIP_insert_char 0x%x %c %d %d",curtulip,ch,ft,ct);

   if (ct == 0) return;

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);

   j = (CURSORTYPE == TULIP_CURSOR_CARET ? 0 : 1);
   cpa = cp+j;
   for (i = CURCHAR+j; i < lp->nchars; ++i) {
      if (cpa->charv != ' ') break;
      ++cpa;
    };
   if (i >= lp->nchars && CURFONT == ft) {
      TULIP_out_char(curtulip,ch);
      return;
    };

   if (CURSORON) TULIP_set_cursor(curtulip,FALSE);

   char_info(ch,ft,&nup,&ndown,&nwidth);

   j = lp->nchars+ct-1;
   if (j >= MAXCOLUMN) j = MAXCOLUMN-1;

   cpa = LCHAR(lp,j);
   cpb = LCHAR(lp,j-ct);
   k = CURCHAR+ct;
   for (i = j; i >= k; --i) {
      *cpa-- = *cpb--;
    };
   k = CURCHAR;
   x = cp->xpos + (ct-1)*(nwidth);
   while (i-- >= k) {
      cpa->charv = ch;
      cpa->font = ft;
      cpa->xpos = x;
      x -= nwidth;
      --cpa;
    };
   lp->nchars += ct;

   if (nup > lp->up || ndown > lp->down) adjust_height(curtulip,CURLINE);
   else {
      if (nup == lp->up) ++(lp->nmaxup);
      if (ndown == lp->down) ++(lp->nmaxdown);
    };

   fg = move_in_line(curtulip,lp,CURCHAR+ct,(nwidth)*ct);

   if (!fg && (ch != ' ' || ft != SPACEFONT)) {
      j = CURCHAR;
      k = lp->nchars;
      for (i = 0; i < ct && j < k; ++i) {
	 out_char(curtulip,lp,cp);
	 ++j;
       };
    };

   check_right(curtulip,lp);
};






/************************************************************************/
/*									*/
/*	TULIP_insert_string -- insert string into line			*/
/*									*/
/************************************************************************/


void
TULIP_insert_string(curtulip,str)
   register TULIP_INFO curtulip;
   String str;
{
   register Integer ch;
   register TULIP_LINE lp;
   register TULIP_CHAR cp,cpa,cpb;
   register Integer i,j,k,x;
   register Integer ct,wd;
   register Boolean fg;
   Integer nup,ndown,nwidth;

   ITRACE("TULIP_insert_string 0x%x `%s`",curtulip,str);

   if (str == NULL || *str == 0) return;

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);

   cpa = cp;
   for (i = CURCHAR; i < lp->nchars; ++i) {
      if (cpa->charv != ' ') break;
      ++cpa;
    };
   if (i >= lp->nchars) {
      TULIP_out_string(curtulip,str);
      return;
    };

   if (CURSORON) TULIP_set_cursor(curtulip,FALSE);

   ct = strlen(str);
   if (CURCHAR+ct+1 > MAXCOLUMN) ct = MAXCOLUMN-CURCHAR-1;

   j = lp->nchars+ct-1;
   if (j >= MAXCOLUMN) j = MAXCOLUMN-1;
   cpa = LCHAR(lp,j);
   cpb = LCHAR(lp,j-ct);
   k = CURCHAR+ct;
   for (i = j; i >= k; --i) {
      *cpa-- = *cpb--;
    };
   lp->nchars += ct;
   if (lp->nchars > MAXCOLUMN) lp->nchars = MAXCOLUMN;

   fg = FALSE;
   k = CURCHAR;
   x = cp->xpos;
   wd = 0;
   while ((ch = *str++) != 0) {
      char_info(ch,CURFONT,&nup,&ndown,&nwidth);
      if (nup > lp->up || ndown > lp->down) fg = TRUE;
      else {
	 if (nup == lp->up) ++(lp->nmaxup);
	 if (ndown == lp->down) ++(lp->nmaxdown);
       };
      cp->charv = ch;
      cp->font = CURFONT;
      cp->xpos = x;
      x += nwidth;
      wd += nwidth;
      ++cp;
      ++k;
      if (k >= MAXCOLUMN) break;
    };

   if (fg) adjust_height(curtulip,CURLINE);

   fg = move_in_line(curtulip,lp,CURCHAR+ct,wd);

   if (!fg) out_string(curtulip,lp,CURCHAR,k-1);

   check_right(curtulip,lp);
};






/************************************************************************/
/*									*/
/*	TULIP_delete_char -- delete characters in line			*/
/*									*/
/************************************************************************/


void
TULIP_delete_char(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   register TULIP_LINE lp;
   register TULIP_CHAR cp,cpa,cpb;
   Integer up,down,width;
   register Integer i,dx;

   ITRACE("TULIP_delete_char 0x%x %d",curtulip,ct);

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);

   if (ct > lp->nchars-CURCHAR) ct = lp->nchars-CURCHAR;
   if (ct == 0) return;

   clear_cursor(curtulip);

   dx = 0;
   cpa = cp;
   for (i = 0; i < ct; ++i) {
      char_info(cpa->charv,cpa->font,&up,&down,&width);
      dx += width;
      if (up == lp->up) --(lp->nmaxup);
      if (down == lp->down) --(lp->nmaxdown);
      ++cpa;
    };

   cpb = cp;
   for (i = ct+CURCHAR; i < lp->nchars; ++i) {
      *cpb++ = *cpa++;
    };
   lp->nchars -= ct;

   if (lp->nmaxup == 0 || lp->nmaxdown == 0)
      adjust_height(curtulip,CURLINE);

   move_in_line(curtulip,lp,CURCHAR,-dx);

   check_right(curtulip,lp);
};





/************************************************************************/
/*									*/
/*	TULIP_clear_chars -- clear region within a line 		*/
/*									*/
/************************************************************************/


void
TULIP_clear_chars(curtulip,line,from,to)
   register TULIP_INFO curtulip;
   Integer line;
   Integer from;
   Integer to;
{
   Integer up,down,width;
   register Integer i,x;
   register Boolean fg;
   register TULIP_LINE lp;
   register TULIP_CHAR cp;

   ITRACE("TULIP_clear_chars 0x%x %d %d %d",curtulip,line,from,to);

   lp = LINE(line);
   cp = LCHAR(lp,from);
   x = cp->xpos;

   if (CURLINE == line && CURCHAR >= from && CURCHAR <= to)
      clear_cursor(curtulip);

   for (i = from; i <= to; ++i) {
      char_info(cp->charv,cp->font,&up,&down,&width);
      if (up != SPACEUP && up == lp->up) --(lp->nmaxup);
      if (down != SPACEDOWN && down == lp->down) --(lp->nmaxdown);
      cp->charv = ' ';
      cp->font = SPACEFONT;
      cp->xpos = x;
      x += SPACEWIDTH;
      ++cp;
    };

   if (lp->nmaxup == 0 || lp->nmaxdown == 0) adjust_height(curtulip,line);

   if (x != cp->xpos) fg = move_in_line(curtulip,lp,i,x-cp->xpos);
   else fg = FALSE;

   if (!fg) clear_char_region(curtulip,lp,from,to);
};





/************************************************************************/
/*									*/
/*	TULIP_set_cursor -- set cursor to proper position		*/
/*									*/
/************************************************************************/


void
TULIP_set_cursor(curtulip,fg)
   register TULIP_INFO curtulip;
   Boolean fg;
{
   register TULIP_LINE lp;
   register TULIP_CHAR cp;
   register Integer ft;

   ITRACE("TULIP_set_cursor 0x%x %d",curtulip,fg);

   if (!fg && !CURSORON) return;
   if (fg && CURSORON) return;
   if (fg && !CURSORMODE) return;

   lp = LINE(CURLINE);
   cp = LCHAR(lp,CURCHAR);

   if (CURSORTYPE == TULIP_CURSOR_CARET) {
      set_caret_cursor(curtulip,lp,cp,fg);
    }
   else {
      ft = cp->font;
      if (ASHinq_configuration_depth(VIEWPORT) > 1) {
	 ft ^= REVERSE_BIT;
       }
      else switch (CURSORTYPE) {
	 default :
	 case TULIP_CURSOR_BLOCK :
	    ft ^= REVERSE_BIT;
	    break;
	 case TULIP_CURSOR_UNDERLINE :
	    ft ^= UNDERLINE_BIT;
	    break;
       };

      cp->font = ft;
      out_char(curtulip,lp,cp);
    };

   CURSORON = fg;

   if (CURSORMOVEMODE) {
      ASHcursor_move(VIEWPORT,ASH_X(cp->xpos),ASH_Y(YPOS(lp)));
    };
};





/************************************************************************/
/*									*/
/*	TULIP_out_bell -- simulate/ring the bell			*/
/*									*/
/************************************************************************/


void
TULIP_out_bell(curtulip)
   register TULIP_INFO curtulip;
{
   ITRACE("TULIP_out_bell 0x%x",curtulip);

   ASHbell();

#if MARGIN > 1
   for (i = 0; i < BELLCT; ++i) {
      do_clear(curtulip,1,-MARGIN,-1,XSIZE-1,-MARGIN);
      do_clear(curtulip,1,-MARGIN,YSIZE-1,-1,-MARGIN);
      do_clear(curtulip,1,RIGHT+1,YSIZE-1,XSIZE-1,-MARGIN);
      do_clear(curtulip,1,-MARGIN,YSIZE-1,XSIZE-1,BOTTOM+1);

      do_clear(curtulip,0,-MARGIN,-1,XSIZE-1,-MARGIN);
      do_clear(curtulip,0,-MARGIN,YSIZE-1,-1,-MARGIN);
      do_clear(curtulip,0,RIGHT+1,YSIZE-1,XSIZE-1,-MARGIN);
      do_clear(curtulip,0,-MARGIN,YSIZE-1,XSIZE-1,BOTTOM+1);
    };
#endif
};






/************************************************************************/
/*									*/
/*	blank_line -- set up a blank line				*/
/*									*/
/************************************************************************/


static void
blank_line(curtulip,lp)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
{
   register Integer x,i;
   register TULIP_CHAR cp;

   DTRACE("blank_line 0x%x 0x%x",curtulip,lp);

   if (lp->chars == NULL) {
      lp->chars = (TULIP_CHAR) calloc(MAXCOLUMN+1,sizeof(TULIP_CHAR_BLOCK));
    };

   x = HOR_INDENT;
   cp = lp->chars;
   for (i = 0; i < MAXCOLUMN; ++i) {
      cp->charv = ' ';
      cp->font = SPACEFONT;
      cp->xpos = x;
      x += SPACEWIDTH;
      if (x > RIGHT-HOR_INDENT) break;
      ++cp;
    };

   lp->nmaxup = i+1;		  /* insure min size is that of space */
   lp->nmaxdown = i+1;
   lp->up = SPACEUP;
   lp->down = SPACEDOWN;
   lp->nchars = i;
};





/************************************************************************/
/*									*/
/*	move_lines -- move line region					*/
/*									*/
/************************************************************************/


static Boolean
move_lines(curtulip,from,to,delta)
   register TULIP_INFO curtulip;
   Integer from;
   Integer to;
   Integer delta;
{
   register Integer i,y0,y1;
   register TULIP_LINE lp;
   register Boolean fg;

   DTRACE("move_lines 0x%x %d %d %d",curtulip,from,to,delta);

   lp = LINE(from);
   y0 = LTOPC(lp);

   for (i = from; i < to; ++i) {
      lp->ypos += delta;
      ++lp;
    };
   y1 = LBOTC(lp);
   lp->ypos += delta;

   if (do_move(curtulip,0,y1,RIGHT,y0,0,y1+delta)) {
      fg = FALSE;
      if (delta > 0) do_clear(curtulip,0,0,y0+delta-1,RIGHT,y0);
      else do_clear(curtulip,0,0,y1,RIGHT,y1+delta+1);
    }
   else {
      fg = TRUE;
      if (delta < 0) y0 -= delta;
      else y1 += delta;
      do_clear(curtulip,0,0,y1,RIGHT,y0);
      lp = LINE(from);
      for (i = from; i <= to; ++i) out_line(curtulip,lp++);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	clear_line_region -- clear block of window for lines		*/
/*									*/
/************************************************************************/


static void
clear_line_region(curtulip,from,to)
   register TULIP_INFO curtulip;
   Integer from,to;
{
   register Integer y0,y1;

   DTRACE("clear_line_region 0x%x %d %d",curtulip,from,to);

   y0 = LTOPC(LINE(from));
   y1 = LBOTC(LINE(to));

   do_clear(curtulip,0,0,y1,RIGHT,y0);
};





/************************************************************************/
/*									*/
/*	adjust_height -- adjust height of a line			*/
/*									*/
/************************************************************************/


static void
adjust_height(curtulip,line)
   register TULIP_INFO curtulip;
   Integer line;
{
   Integer mxup,mxdwn,upcnt,dwncnt;
   Integer up,down,width;
   register Integer i;
   register TULIP_LINE lp;
   register TULIP_CHAR cp;

   DTRACE("adjust_height 0x%x %d",curtulip,line);

   lp = LINE(line);
   cp = lp->chars;

   upcnt = 1;			/* insure min size is that of space	*/
   dwncnt = 1;
   mxup = SPACEUP;
   mxdwn = SPACEDOWN;

   for (i = 0; i < lp->nchars; ++i) {
      char_info(cp->charv,cp->font,&up,&down,&width);
      if (up == mxup) ++upcnt;
      else if (up > mxup) {
	 mxup = up;
	 upcnt = 1;
       };
      if (down == mxdwn) ++dwncnt;
      else if (down > mxdwn) {
	 mxdwn = down;
	 dwncnt = 1;
       };
    };

   if (mxup+mxdwn != HEIGHT(lp))
      move_lines(curtulip,line,NUMLINE-1,mxup+mxdwn-HEIGHT(lp));

   if (mxdwn != lp->down)
      move_lines(curtulip,line,line,lp->down-mxdwn);

   lp->up = mxup;
   lp->down = mxdwn;
   lp->nmaxup = upcnt;
   lp->nmaxdown = dwncnt;
};





/************************************************************************/
/*									*/
/*	clear_char_region -- clear range of characters			*/
/*									*/
/************************************************************************/


static void
clear_char_region(curtulip,lp,from,to)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
   Integer from,to;
{
   register TULIP_CHAR cpa,cpb;
   register Integer x1;

   DTRACE("clear_char_region 0x%x 0x%x %d %d",curtulip,lp,from,to);

   cpa = LCHAR(lp,from);
   cpb = LCHAR(lp,to);
   x1 = cpb->xpos+ fonttbl[cpb->font]->width[cpb->charv]-1;

   do_clear(curtulip,0,cpa->xpos,LBOTC(lp),x1,LTOP(lp));
};




/************************************************************************/
/*									*/
/*	move_in_line -- move characters around in the line		*/
/*									*/
/************************************************************************/


static Boolean
move_in_line(curtulip,lp,from,delta)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
   Integer from;
   Integer delta;
{
   register TULIP_CHAR cp;
   register Integer i,x0,x1,x2,by,ty;
   register Boolean fg;

   DTRACE("move_in_line 0x%x 0x%x %d %d",curtulip,lp,from,delta);

   cp = LCHAR(lp,lp->nchars-1);
   x2 = cp->xpos+fonttbl[cp->font]->width[cp->charv];

   if (from < lp->nchars) {
      cp = LCHAR(lp,from);
      x0 = cp->xpos;
    }
   else {
      x0 = x2;
    };

   x1 = RIGHT;
   by = LBOTC(lp);
   ty = LTOPC(lp);

   for (i = from; i < lp->nchars; ++i) {
      cp->xpos += delta;
      ++cp;
    };

   if (delta > 0) {
      fg = do_move(curtulip,x0,by,x1-delta,ty,x0+delta,by);
      if (fg) do_clear(curtulip,0,x0,by,x0+delta-1,ty);
    }
   else {
      fg = do_move(curtulip,x0,by,x1,ty,x0+delta,by);
      if (fg) do_clear(curtulip,0,x2+delta+1,by,x1,ty);
    };

   if (!fg) out_line(curtulip,lp);

   return (!fg);
};





/************************************************************************/
/*									*/
/*	check_bottom -- add or remove lines from the display for space	*/
/*									*/
/************************************************************************/


static void
check_bottom(curtulip)
   register TULIP_INFO curtulip;
{
   register TULIP_LINE lpa,lpb;
   register Integer ct,i,y,dy,inc;

   DTRACE("check_bottom 0x%x",curtulip);

   lpa = LINE(NUMLINE-1);
   y = LBOT(lpa);

   if (y+1 > BOTTOM) {
      lpa = LINE(0);
      dy = y+1-BOTTOM;
      inc = 0;
      for (ct = 0; dy >= 0 && ct < NUMLINE-2; ++ct) {
	 inc += HEIGHT(lpa);
	 dy -= HEIGHT(lpa);
	 ++lpa;
       };
      lpa = LINE(0);
      lpb = LINE(ct);
      for (i = 0; i < NUMLINE-ct; ++i) *lpa++ = *lpb++;
      NUMLINE -= ct;
      CURLINE -= ct;
      if (CURLINE < 0) {
	 CURLINE = 0;
	 CURCHAR = 0;
       };
      move_lines(curtulip,0,NUMLINE-1,-inc);
      lpa = LINE(NUMLINE-1);
      y = LBOT(lpa);
      do_clear(curtulip,0,0,BOTTOM,RIGHT,y+1);
    };

   while (NUMLINE < MAXLINE && y+VERT_INDENT+SPACEUP+SPACEDOWN+1 < BOTTOM) {
      ++lpa;
      ++NUMLINE;
      lpa->ypos = y;
      blank_line(curtulip,lpa);
      y += HEIGHT(lpa);
    };
};





/************************************************************************/
/*									*/
/*	check_right -- check for truncation/spacing at right of line	*/
/*									*/
/************************************************************************/


static void
check_right(curtulip,lp)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
{
   register Integer x;
   register TULIP_CHAR cp;
   Integer up,down,width;

   DTRACE("check_right 0x%x 0x%x",curtulip,lp);

   cp = LCHAR(lp,lp->nchars-1);
   x = cp->xpos+fonttbl[cp->font]->width[cp->charv];

   if (x > RIGHT-HOR_INDENT) {
      ++cp;
      do {
	 --cp;
	 char_info(cp->charv,cp->font,&up,&down,&width);
	 if (up == lp->up) --(lp->nmaxup);
	 if (down == lp->down) --(lp->nmaxdown);
	 --(lp->nchars);
       }
      while (cp->xpos > RIGHT-HOR_INDENT);
      x = cp->xpos;
      do_clear(curtulip,0,x,LBOTC(lp),RIGHT+MARGIN,LTOP(lp));
    }
   else {
      while (lp->nchars < MAXCOLUMN && x+SPACEWIDTH <= RIGHT-HOR_INDENT) {
	 ++cp;
	 ++lp->nchars;
	 cp->xpos = x;
	 cp->charv = ' ';
	 cp->font = SPACEFONT;
	 if (SPACEUP == lp->up) ++(lp->nmaxup);
	 if (SPACEDOWN == lp->down) ++(lp->nmaxdown);
	 x += SPACEWIDTH;
       };
    };
};





/************************************************************************/
/*									*/
/*	out_line -- output the given line				*/
/*									*/
/************************************************************************/


static void
out_line(curtulip,lp)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
{
   DTRACE("out_line 0x%x 0x%x",curtulip,lp);

   out_string(curtulip,lp,0,lp->nchars-1);
};





/************************************************************************/
/*									*/
/*	out_string -- output portion of a line				*/
/*									*/
/************************************************************************/


static void
out_string(curtulip,lp,from,to)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
   Integer from,to;
{
   register Integer i,j,f;
   register TULIP_CHAR cp;
   Character buf[MAX_WIDTH];
   register String p,q1;
   register Integer x,y,x1,x2;
   Integer col;

   DTRACE("out_string 0x%x 0x%x %d %d",curtulip,lp,from,to);

   cp = LCHAR(lp,to);
   x1 = cp->xpos+fonttbl[cp->font]->width[cp->charv]-1;
   cp = LCHAR(lp,from);
   x = cp->xpos;

   y = ASH_Y(YPOS(lp));
   for (j = from; j <= to; j += i) {
      p = buf;
      f = cp->font;
      x = cp->xpos;
      x1 = x+fonttbl[f]->width[cp->charv]-1;
      if (cp->charv == ' ' && BASE_FONT(f) == SPACEFONT) x2 = -1;
      else {
	 q1 = p;
	 *p++ = cp->charv;
	 x2 = x;
       };

      for (i = 1; i+j <= to; ++i) {
	 ++cp;
	 if (f != cp->font) break;
	 if (cp->charv != ' ' || BASE_FONT(f) != SPACEFONT) {
	    if (x2 < 0) x2 = cp->xpos;
	    q1 = p;
	  };
	 if (x2 >= 0) *p++ = cp->charv;
	 x1 = cp->xpos+fonttbl[f]->width[cp->charv]-1;
       };
      do_clear(curtulip,COLOR(f),x,LBOTC(lp),x1,LTOP(lp));
      if (x2 >= 0) {
	 *++q1 = 0;
	 set_font(curtulip,f);
	 ASHtext(VIEWPORT,ASH_X(x2),y,buf);
       };
      if (UNDERLINE(f)) {
	 y = ASH_Y(LBOTC(lp)-1);
	 col = ASHcolor(VIEWPORT,ASHinq_text_color(VIEWPORT));
	 ASHline(VIEWPORT,ASH_X(x),y,ASH_X(x1),y);
	 ASHcolor(VIEWPORT,col);
       };
    };
};





/************************************************************************/
/*									*/
/*	out_char -- output a character					*/
/*									*/
/************************************************************************/


static void
out_char(curtulip,lp,cp)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
   TULIP_CHAR cp;
{
   register Integer x,y,ch;
   Character buf[2];
   Integer up,dwn,wid,col;

   DTRACE("out_char 0x%x 0x%x 0x%x %d %c",curtulip,lp,cp,cp->font,cp->charv);

   if (LASTFONT != cp->font) set_font(curtulip,cp->font);

   ch = cp->charv;
   char_info(ch,cp->font,&up,&dwn,&wid);

   if ( !fonttbl[cp->font]->safe || lp->up != up || lp->down != dwn) {
      x = cp->xpos+wid-1;
      do_clear(curtulip,COLOR(cp->font),cp->xpos,LBOTC(lp),x,LTOP(lp));
    };

   buf[0] = ch;
   buf[1] = 0;
   ASHtext(VIEWPORT,ASH_X(cp->xpos),ASH_Y(YPOS(lp)),buf);

   if (UNDERLINE(cp->font)) {
      y = ASH_Y(LBOTC(lp)-1);
      x = cp->xpos+wid-1;
      col = ASHcolor(VIEWPORT,ASHinq_text_color(VIEWPORT));
      ASHline(VIEWPORT,ASH_X(cp->xpos),y,ASH_X(x),y);
      ASHcolor(VIEWPORT,col);
    };
};





/************************************************************************/
/*									*/
/*	set_font -- set font for character output			*/
/*									*/
/************************************************************************/


static void
set_font(curtulip,id)
   register TULIP_INFO curtulip;
   Integer id;
{
   Integer col;

   DTRACE("set_font 0x%x %d",curtulip,id);

   if (LASTFONT < 0 || (BASE_FONT(LASTFONT) != BASE_FONT(id)))
      ASHfont(VIEWPORT,fonttbl[id]->fontid);

   col = COLOR(id);
   if (LASTFONT < 0 || col != COLOR(LASTFONT)) {
      ASHtext_color(VIEWPORT,get_color(curtulip,col,TRUE));
      ASHtext_background_color(VIEWPORT,get_color(curtulip,col,FALSE));
    };

   LASTFONT = id;
};




/************************************************************************/
/*									*/
/*	do_move -- do a move blt					 */
/*	do_clear -- clear a screen region				*/
/*									*/
/************************************************************************/


static Boolean
do_move(curtulip,lx,by,rx,ty,dlx,dby)
   register TULIP_INFO curtulip;
   Integer lx,by;
   Integer rx,ty;
   Integer dlx,dby;
{
   DTRACE("do_blt 0x%x (%d,%d)x(%d,%d) -> (%d,%d)",curtulip,lx,by,rx,ty,dlx,dby);

   lx = ASH_X(lx);
   by = ASH_Y(by);
   rx = ASH_X(rx);
   ty = ASH_Y(ty);
   dlx = ASH_X(dlx);
   dby = ASH_Y(dby);

/* if (!SAVED && !ASHinq_blt(VIEWPORT,cr,lx,by,rx,ty,dlx,dby))
 *    return FALSE;
 **************************** how do we get this to work??? *****/

   ASHblt(VIEWPORT,lx,by,rx,ty,dlx,dby);

   return TRUE;
};





static void
do_clear(curtulip,col,lx,by,rx,ty)
   register TULIP_INFO curtulip;
   Integer col;
   Integer lx,by;
   Integer rx,ty;
{
   Integer scol,fil;

   DTRACE("do_clear 0x%x %d (%d,%d)x(%d,%d)",curtulip,col,lx,by,rx,ty);

   lx = ASH_X(lx);
   by = ASH_Y(by);
   rx = ASH_X(rx);
   ty = ASH_Y(ty);

   fil = ASHfill(VIEWPORT,ASH_FILL_SOLID);
   scol = ASHcolor(VIEWPORT,get_color(curtulip,col,FALSE));
   ASHrectangle(VIEWPORT,lx,by,rx,ty);
   ASHcolor(VIEWPORT,scol);
   ASHfill(VIEWPORT,fil);
};





/************************************************************************/
/*									*/
/*	create_caret_bitmap -- create caret bitmap			*/
/*	clear_cursor -- char of cursor is being cleared 		*/
/*									*/
/************************************************************************/


#define CARET_BITMAP_HEIGHT	15
#define CARET_BITMAP_WIDTH	caret_bitmap_width
#define CARET_HEIGHT		7
#define CARET_COLOR_WIDTH	4
#define CARET_MONO_WIDTH	6

static	Integer 	caret_bitmap_width;



static void
create_caret_bitmap(curtulip)
   TULIP_INFO curtulip;
{
   Integer x[4],y[4];
   Integer col;
   Integer caret_color;
   ASH_WINDOW caret_bitmap;

   if (curtulip->caret_bitmap != NULL) return;

   if (ASHinq_configuration_depth(VIEWPORT) == 1) caret_color = 1;
   else caret_color = 3;

   if (ASHinq_configuration_depth(VIEWPORT) > 1)
      caret_bitmap_width = CARET_COLOR_WIDTH;
   else
      caret_bitmap_width = CARET_MONO_WIDTH;

   caret_bitmap = ASHcreate(ASHinq_top_window(VIEWPORT),0,100,
			       0,0,CARET_BITMAP_WIDTH,CARET_BITMAP_HEIGHT,
			       0,
			       ASH_WINDOW_INVISIBLE|ASH_WINDOW_OFF_SCREEN);

   curtulip->caret_bitmap = caret_bitmap;

   ASHbackground_color(caret_bitmap,0);
   ASHclear_box(caret_bitmap,0,0,CARET_BITMAP_WIDTH,CARET_BITMAP_HEIGHT);

   x[0] = 0;
   y[0] = 0;
   x[1] = CARET_BITMAP_WIDTH/2;
   y[1] = CARET_HEIGHT;
   x[2] = CARET_BITMAP_WIDTH;
   y[2] = 0;

   col = colormap[caret_color].value ^ curtulip->bg;
   if (col == 0) col = 1;

   ASHcolor(caret_bitmap,col);

   ASHconvex_polygon(caret_bitmap,3,x,y);
   ASHline(caret_bitmap,x[1],0,x[1],CARET_BITMAP_HEIGHT);
};





static void
clear_cursor(curtulip)
   register TULIP_INFO curtulip;
{
   if (CURSORTYPE == TULIP_CURSOR_CARET)
      TULIP_set_cursor(curtulip,FALSE);
   else
      CURSORON = FALSE;
};





static void
set_caret_cursor(curtulip,lp,cp,fg)
   register TULIP_INFO curtulip;
   TULIP_LINE lp;
   TULIP_CHAR cp;
   Boolean fg;
{
   Integer lx,by;

   if (curtulip->caret_bitmap == NULL) create_caret_bitmap(curtulip);

   lx = cp->xpos - CARET_BITMAP_WIDTH/2 -1;
/* by = YPOS(lp) + 1 + CARET_BITMAP_HEIGHT;	*/
   by = LBOT(lp)+1;

   ASHinq_background_color(VIEWPORT);

   ASHcombination_rule(VIEWPORT,6);
   ASHsource(VIEWPORT,curtulip->caret_bitmap);
   ASHblt(VIEWPORT,0,0,CARET_BITMAP_WIDTH,CARET_BITMAP_HEIGHT,ASH_X(lx),ASH_Y(by));
   ASHsource(VIEWPORT,VIEWPORT);
   ASHcombination_rule(VIEWPORT,3);
};





/************************************************************************/
/*									*/
/*	get_color -- get appropriate color				*/
/*									*/
/************************************************************************/


static ASH_COLOR
get_color(curtulip,i,fg)
   TULIP_INFO curtulip;
   Integer i;
   Boolean fg;
{
   ASH_COLOR c;

   if (i >= 2 && use_colors) {
      if (fg) c = colormap[i].fvalue;
      else c = colormap[i].value;
    }
   else if (i >= 1) {
      if (fg) c = curtulip->bg;
      else c = curtulip->fg;
    }
   else {
      if (fg) c = curtulip->fg;
      else c = curtulip->bg;
    };

   return c;
};





/* end of tuliplines.c */
