/************************************************************************/
/*									*/
/*		tulipexec.c						*/
/*									*/
/*	User entries for terminal utility package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "tulip_local.h"






/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		vtback();
static	void		vteraseeol();
static	void		vteraseeos();
static	void		vtfontmap();
static	void		vtmove();
static	void		vtnewline();
static	void		vttab();
static	Integer 	getxchar();
static	Integer 	getyline();





/************************************************************************/
/*									*/
/*	TULIP_exec_init -- initialize this portion of tulip		*/
/*									*/
/************************************************************************/


void
TULIP_exec_init()
{
   ITRACE("TULIP_exec_init");
};






/************************************************************************/
/*									*/
/*	TULIPinq_size -- return terminal size				*/
/*	TULIPinq_max_size -- return maximum terminal size		*/
/*									*/
/************************************************************************/


void
TULIPinq_size(curtulip,rp,cp)
   register TULIP_INFO curtulip;
   int *rp;
   int *cp;
{
   TRACE("TULIPinq_size -> %d %d",NUMLINE,NUMCHAR(CURLINE));

   if (curtulip->window == NULL) {
      *rp = 0;
      *cp = 0;
    }
   else {
      *rp = NUMLINE;
      *cp = NUMCHAR(CURLINE);
    };
};





void
TULIPinq_max_size(curtulip,rp,cp)
   register TULIP_INFO curtulip;
   int *rp;
   int *cp;
{
   TRACE("TULIPinq_max_size -> %d %d",MAXLINE,MAXCOLUMN);

   if (curtulip->window == NULL) {
      *rp = 0;
      *cp = 0;
    }
   else {
      *rp = NUMLINE;		/**************/
      *cp = MAXCOLUMN;
    };
};





/************************************************************************/
/*									*/
/*	TULIPcorrelate -- correlate x,y with row/column 		*/
/*									*/
/************************************************************************/


void
TULIPcorrelate(curtulip,x,y,rp,cp)
   register TULIP_INFO curtulip;
   int x,y;
   int *rp;
   int *cp;
{
   TRACE("TULIPcorrelate 0x%x %d %d",curtulip,x,y);

   if (curtulip->window == NULL) {
      *rp = 0;
      *cp = 0;
    }
   else {
      *rp = getyline(curtulip,TULIP_Y(y));
      *cp = getxchar(curtulip,*rp,TULIP_X(x));
      if (*cp == 0 && TULIP_X(x) <= MARGIN) *cp = -1;
    };
};





/************************************************************************/
/*									*/
/*	TULIPchar_delete -- delete characters at current position	*/
/*	TULIPchar_insert -- insert characters at current position	*/
/*	TULIPline_delete -- delete lines at current position		*/
/*	TULIPline_insert -- insert lines at current position		*/
/*	TULIPinsertmode -- set/reset insert mode			*/
/*	TULIPmove -- absolute movement					*/
/*	TULIPmove_rel -- relative movement				*/
/*	TULIPfont_map -- font map a region				*/
/*	TULIPerase_line --erase to end of line				*/
/*	TULIPerase_screen -- erase to end of screen			*/
/*	TULIPcursor -- turn cursor on/off				*/
/*	TULIPinq_font -- return current font				*/
/*									*/
/************************************************************************/


void
TULIPchar_delete(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPchar_delete 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      if (curtulip->window == NULL) return;
      TULIP_delete_char(curtulip,ct);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPchar_insert(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPchar_insert 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      TULIP_insert_char(curtulip,' ',SPACEFONT,ct);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPline_delete(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPline_delete 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      TULIP_scroll_up(curtulip,CURLINE,NUMLINE-1,ct);
      vtmove(curtulip,CURLINE,0);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPline_insert(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPline_insert 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      TULIP_scroll_down(curtulip,CURLINE,NUMLINE-1,ct);
      vtmove(curtulip,CURLINE,0);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPscroll_up(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPscroll_up 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      TULIP_scroll_up(curtulip,0,NUMLINE-1,ct);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPscroll_down(curtulip,ct)
   register TULIP_INFO curtulip;
   Integer ct;
{
   TRACE("TULIPscroll_down 0x%x %d",curtulip,ct);

   if (T_PROTECT) {
      TULIP_scroll_down(curtulip,0,NUMLINE-1,ct);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPinsertmode(curtulip,fg)
   register TULIP_INFO curtulip;
   Boolean fg;
{
   TRACE("TULIPinsertmode 0x%x %d",curtulip,fg);

   if (curtulip->window == NULL) return;

   INSERTMODE = fg;
};





void
TULIPmove(curtulip,line,col)
   register TULIP_INFO curtulip;
   Integer line,col;
{
   TRACE("TULIPmove 0x%x %d %d",curtulip,line,col);

   if (T_PROTECT) {
      vtmove(curtulip,line,col);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPmove_rel(curtulip,dl,dc)
   register TULIP_INFO curtulip;
   Integer dl,dc;
{
   TRACE("TULIPmove_rel 0x%x %d %d",curtulip,dl,dc);

   if (T_PROTECT) {
      vtmove(curtulip,CURLINE+dl,CURCHAR+dc);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPfont_map(curtulip,froml,fromc,tol,toc,fromf,tof)
   register TULIP_INFO curtulip;
   Integer froml,fromc,tol,toc;
   Integer fromf,tof;
{
   Integer parms[6];

   TRACE("TULIPfont_map 0x%x %d %d %d %d %d %d",curtulip,froml,fromc,tol,toc,
	    fromf,tof);

   if (T_PROTECT) {
      parms[0] = froml;
      parms[1] = fromc;
      parms[2] = tol;
      parms[3] = toc;
      parms[4] = fromf;
      parms[5] = tof;

      vtfontmap(curtulip,6,parms);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPhighlight(curtulip,froml,fromc,tol,toc,fromf,onoff,mode)
   register TULIP_INFO curtulip;
   Integer froml,fromc,tol,toc;
   Integer fromf;
   Boolean onoff;
   Integer mode;
{
   Integer parms[8];
   Integer bit,abit,bbit;

   TRACE("TULIPhighlight 0x%x %d %d %d %d %d %d %d",curtulip,froml,fromc,tol,toc,
	    fromf,onoff,mode);

   if (!T_PROTECT) return;

   parms[0] = froml;
   parms[1] = fromc;
   parms[2] = tol;
   parms[3] = toc;

   if (ASHinq_configuration_depth(VIEWPORT) == 1) {
      if (mode == 1) {
	 bit = UNDERLINE_BIT;
	 abit = REVERSE_BIT;
	 bbit = bit|abit;
       }
      else {
	 bit = REVERSE_BIT;
	 abit = UNDERLINE_BIT;
	 bbit = bit|abit;
       };
    }
   else {
      if (mode == 1) {
	 bit = COLOR2_BIT;
	 abit = REVERSE_BIT;
	 bbit = COLOR3_BIT;
       }
      else {
	 bit = REVERSE_BIT;
	 abit = COLOR2_BIT;
	 bbit = COLOR3_BIT;
       };
    };

   if (onoff) {
      parms[4] = fromf;
      parms[5] = fromf|bit;
      parms[6] = fromf|abit;
      parms[7] = fromf|bbit;
    }
   else {
      parms[4] = fromf|bit;
      parms[5] = fromf;
      parms[6] = fromf|bbit;
      parms[7] = fromf|abit;
    };

   vtfontmap(curtulip,8,parms);
   TULIP_set_cursor(curtulip,TRUE);
   T_UNPROTECT;
};





void
TULIPerase_line(curtulip)
   register TULIP_INFO curtulip;
{
   TRACE("TULIPerase_line 0x%x",curtulip);

   if (T_PROTECT) {
      vteraseeol(curtulip,0);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPerase_screen(curtulip)
   register TULIP_INFO curtulip;
{
   TRACE("TULIPerase_screen 0x%x",curtulip);

   if (T_PROTECT) {
      vteraseeos(curtulip,0);
      TULIP_set_cursor(curtulip,TRUE);
      T_UNPROTECT;
    };
};





void
TULIPcursor(curtulip,fg)
   register TULIP_INFO curtulip;
   Boolean fg;
{
   if (T_PROTECT) {
      if (fg) {
	 SET_CURSORMODE;
	 TULIP_set_cursor(curtulip,TRUE);
       }
      else {
	 TULIP_set_cursor(curtulip,FALSE);
	 RESET_CURSORMODE;
       };
      T_UNPROTECT;
    };
};





int
TULIPinq_font(curtulip)
   TULIP_INFO curtulip;
{
   return CURFONT;
};





/************************************************************************/
/*									*/
/*	TULIP_outc -- output normal character				*/
/*									*/
/************************************************************************/


void
TULIP_outc(curtulip,c)
   register TULIP_INFO curtulip;
   Integer c;
{
   register Integer mx;

   ITRACE("TULIP_outc 0x%x %c",curtulip,c);

   mx = NUMCHAR(CURLINE);

   if (DISCARDON) return;

   if (INSERTMODE) TULIP_insert_char(curtulip,c,CURFONT,1);
   else TULIP_out_char(curtulip,c);

   if (CURCHAR+1 == mx) {
      if (WRAP) vtnewline(curtulip);
      else if (DISCARD) DISCARDON = TRUE;
    }
   else {
      vtmove(curtulip,CURLINE,CURCHAR+1);
    };
};





/************************************************************************/
/*									*/
/*	TULIP_outs -- output string of normal characters		*/
/*									*/
/************************************************************************/


void
TULIP_outs(curtulip,s)
   register TULIP_INFO curtulip;
   String s;
{
   register Integer ln;
   register Integer i,ws;
   Character buf[1024];

   ITRACE("TULIP_outs 0x%x `%s`",curtulip,s);

   ln = strlen(s);
   ws = (FIXED ? 0 : WRAP_SAFETY);
   i = NUMCHAR(CURLINE)-ws-CURCHAR;

   if (ln < i) {
      if (INSERTMODE) TULIP_insert_string(curtulip,s);
      else TULIP_out_string(curtulip,s);
      vtmove(curtulip,CURLINE,CURCHAR+ln);
    }
   else {
      if (i > ws && i > 2) {
	 strncpy(buf,s,i-1);
	 buf[i-1] = 0;
	 TULIP_outs(curtulip,buf);
	 s = &s[i-1];
       };
      while (*s != 0) {
	 TULIP_outc(curtulip,*s++);
	 if (CURCHAR == 0 && *s != 0) {
	    TULIP_outs(curtulip,s);
	    return;
	  }
	 else if (DISCARDON) return;
       };
    };
};





/************************************************************************/
/*									*/
/*	TULIP_control -- handle control characters			*/
/*									*/
/************************************************************************/


void
TULIP_control(curtulip,c)
   register TULIP_INFO curtulip;
   Integer c;
{
   ITRACE("TULIP_control 0x%x 0x%x",curtulip,c);

   switch (c) {
      case '\n' :
	 vtnewline(curtulip);
	 break;

      case '\b' :
	 vtback(curtulip);
	 break;

      case '\t' :
	 vttab(curtulip);
	 break;

      case '\r' :
	 vtmove(curtulip,CURLINE,0);
	 break;

      case '\7' :
	 TULIP_out_bell(curtulip);
	 break;

      default :
	 break;
    };
};





/************************************************************************/
/*									*/
/*	TULIP_escape -- handle escape sequence actions			*/
/*									*/
/************************************************************************/


void
TULIP_escape(curtulip,ch,mode,numparm,parm)
   register TULIP_INFO curtulip;
   Integer ch;
   Integer mode;
   Integer numparm;
   Integer parm[];
{
   register Integer p,i;

   ITRACE("TULIP_escape 0x%x %c %d %d (%d,%d,...",curtulip,ch,mode,numparm,parm[0],parm[1]);

   if (numparm == 0) p = 1;
   else p = parm[0];

   switch (ch) {
      case '@' :                                /* INSERT MODE START    */
	 if (parm[0] == 0) INSERTMODE = TRUE;	/* INSERT CHARACTER	*/
	 else TULIP_insert_char(curtulip,' ',SPACEFONT,p);
	 break;

      case 'A' :                                /* UP                   */
	 p = CURLINE-p;
	 if (p < 0) p = 0;
	 if (ABSMODE) vtmove(curtulip,p,CURCHAR);
	 else vtmove(curtulip,p,getxchar(curtulip,p,XPOS(CURLINE,CURCHAR)));
	 break;

      case 'B' :                                /* DOWN                 */
	 p += CURLINE;
	 if (p >= NUMLINE) p = NUMLINE-1;
	 if (ABSMODE) vtmove(curtulip,p,CURCHAR);
	 else vtmove(curtulip,p,getxchar(curtulip,p,XPOS(CURLINE,CURCHAR)));
	 break;

      case 'C' :                                /* RIGHT                */
	 p += CURCHAR;
	 vtmove(curtulip,CURLINE,p);
	 break;

      case 'D' :                                /* LEFT                 */
	 p = CURCHAR-p;
	 if (p < 0) p = 0;
	 vtmove(curtulip,CURLINE,p);
	 break;

      case 'E' :                                /* REDRAW               */
	 TULIP_draw_all(curtulip);
	 break;

      case 'F' :                                /* FONT                 */
	 CURFONT = parm[0];
	 break;

      case 'G' :                                /* SPACE FONT           */
	 TULIP_setup_space(curtulip,parm[0]);
	 break;

      case 'H' :                                /* MOVE                 */
      case 'Y' :
	 vtmove(curtulip,parm[0],parm[1]);
	 break;

      case 'I' :                                /* REVERSE INDEX        */
	 TULIP_scroll_down(curtulip,0,NUMLINE-1,p);
	 break;

      case 'J' :                                /* ERASE TO EOS         */
	 vteraseeos(curtulip,parm[0]);
	 break;

      case 'K' :                                /* ERASE TO EOL         */
	 vteraseeol(curtulip,parm[0]);
	 break;

      case 'L' :                                /* INSERT LINE          */
	 TULIP_scroll_down(curtulip,CURLINE,NUMLINE-1,p);
	 vtmove(curtulip,CURLINE,0);
	 break;

      case 'M' :                                /* DELETE LINE          */
	 TULIP_scroll_up(curtulip,CURLINE,NUMLINE-1,p);
	 vtmove(curtulip,CURLINE,0);
	 break;

      case 'N' :                                /* DELETE CHAR          */
      case 'P' :
	 TULIP_delete_char(curtulip,p);
	 break;

      case 'O' :                                /* EXIT INSERT MODE     */
	 INSERTMODE = FALSE;
	 break;

/*    case 'P'                                     DELETE CHAR (N)      */

      case 'U' :                                /* FONT MAP REGION      */
	 vtfontmap(curtulip,numparm,parm);
	 break;

      case 'W' :                                /* MOVE ABS             */
	 p = getyline(curtulip,TULIP_Y(parm[0]));
	 vtmove(curtulip,p,getxchar(curtulip,p,TULIP_X(parm[1])));
	 break;

/*    case 'Y'                                     POSITIONING (H)      */

      case 'a' :                                /* aLTERNATIVE terminal */
	 break;

      case 'b' :                                /* ERASE begining screen*/
	 vteraseeos(curtulip,1);
	 break;

/*    case 'e'                                     disable flush        */
/*    case 'f'                                     load font            */

      case 'h' :                                /* set modes HIGH       */
	 if (mode == '?' && parm[0] == 7) SET_WRAP;
	 else if (mode == '>' && parm[0] == 5) {
	    TULIP_set_cursor(curtulip,FALSE);
	    RESET_CURSORMODE;
	  }
	 else if (parm[0] == 4) INSERTMODE = TRUE;
	 else if (mode == '>' && parm[0] == 10) SET_FIXED;
	 else if (mode == '>' && parm[0] == 11) SET_CURSORMOVEMODE;
	 else if (mode == '>' && parm[0] == 4) {
	    TULIP_set_cursor(curtulip,FALSE);
	    CURSORTYPE = TULIP_CURSOR_UNDERLINE;
	  }
	 else if (mode == '>' && parm[0] == 12) {
	    TULIP_set_cursor(curtulip,FALSE);
	    CURSORTYPE = TULIP_CURSOR_CARET;
	  };
	 break;

      case 'i' :                                /* INDEX                */
	 TULIP_scroll_up(curtulip,0,NUMLINE-1,p);
	 break;

      case 'l' :                                /* set modes LOW        */
	 if (mode == '?' && parm[0] == 7) RESET_WRAP;
	 else if (mode == '>' && parm[0] == 5) {
	    SET_CURSORMODE;
	  }
	 else if (parm[0] == 4) INSERTMODE = FALSE;
	 else if (mode == '>' && parm[0] == 10) RESET_FIXED;
	 else if (mode == '>' && parm[0] == 11) RESET_CURSORMOVEMODE;
	 else if (mode == '>' && parm[0] == 4) {
	    TULIP_set_cursor(curtulip,FALSE);
	    CURSORTYPE = TULIP_CURSOR_BLOCK;
	  };
	 break;

      case 'm' :                                /* MODES                */
	 switch (p) {
	    case 4:				/*    underline 	*/
	       CURFONT |= UNDERLINE_BIT;
	       break;
	    case 7 :				/*    reverse video	*/
	       CURFONT |= REVERSE_BIT;
	       break;
	    case 6 :				/*    reverse+underline */
	       CURFONT |= (REVERSE_BIT|UNDERLINE_BIT);
	    default :				/*    reset		*/
	       CURFONT &= BASE_BITS;
	       break;
	  };
	 break;

      case 'n' :                                /* position report      */
	  { Character buf[32];
	    sprintf(buf,"%d %d\n",CURLINE,CURCHAR);
	    TULIP_saveinput(curtulip,buf);
	  };
	 break;

      case 'o' :                                /* ERASE LINE START     */
	 vteraseeol(curtulip,1);
	 break;

      case 'p' :                                /* REVERSE VIDEO        */
	 CURFONT |= REVERSE_BIT;
	 break;

      case 'q' :                                /* EXIT REVERSE VIDEO   */
	 CURFONT &= BASE_BITS;
	 break;

      case 's' :                                /* SIZE                 */
	  { Character buf[32];
	    sprintf(buf,"%d %d\n",NUMLINE,NUMCHAR(CURLINE));
	    TULIP_saveinput(curtulip,buf);
	  };
	 break;

      case 'v' :                                /* WRAP AROUND          */
	 SET_WRAP;
	 break;

      case 'w' :                                /* NO WRAP AROUND       */
	 RESET_WRAP;
	 break;

      case 'y' :                                /* text request         */
	 { Character buf[256];
	   switch (parm[0]) {
	      case 1 :				/*    line		*/
		 for (i = 0; i < NUMCHAR(CURLINE); ++i)
		    buf[i] = CCHAR(i)->charv;
		 buf[i] = 0;
		 break;
	      case 2:				/*    character 	*/
		 buf[0] = CHAR->charv;
		 buf[1] = 0;
		 break;
	    };
	   TULIP_saveinput(curtulip,buf);
	 };
	 break;

      case 'z' :                                /* font request           */
	 { Character buf[256];
	   switch (parm[0]) {
	      case 1 :				/*    line		*/
		 for (i = 0; i < NUMCHAR(CURLINE); ++i)
		    buf[i] = CCHAR(i)->font+' ';
		 buf[i] = 0;
		 break;
	      case 2:				/*    character 	*/
		 buf[0] = CHAR->font+' ';
		 buf[1] = 0;
		 break;
	    };
	   TULIP_saveinput(curtulip,buf);
	 };
	 break;

      case '!' :                                /* SCROLL MODES         */
	 switch (mode) {
	    case '0' : RESET_SCROLLMODE; break;
	    case '1' : SET_SCROLLMODE; break;
	    case '2' : RESET_WRAP; break;
	    case '3' : SET_WRAP; break;
	    case '4' : RESET_DISCARD; DISCARDON = FALSE; break;
	    case '5' : SET_DISCARD; break;
	  };
	 break;

      case '#' :                                /* CLEAR MODES          */
	 switch (mode) {
	    case '0' : RESET_NLCLEARMODE; break;
	    case '1' : SET_NLCLEARMODE; break;
	    case '2' : RESET_BSCLEARMODE; break;
	    case '3' : SET_BSCLEARMODE; break;
	    case '4' : RESET_TABCLEARMODE; break;
	    case '5' : SET_TABCLEARMODE; break;
	  };
	 break;

      case ')' :                                /* ABSOLUTE MODE        */
	 switch (mode) {
	    case '0' : RESET_ABSMODE; break;
	    case '1' : SET_ABSMODE; break;
	  };
	 break;

      case '(' :                                /* ANSI fonts           */
	 if (isdigit(mode)) CURFONT = mode-'0';
	 else if (isupper(mode)) CURFONT = mode-'A'+10;
	 else if (islower(mode)) CURFONT = mode-'a'+10;
	 else CURFONT = 0;
	 break;

      default :
	 break;
    };
};





/************************************************************************/
/*									*/
/*	vtback -- backspace						*/
/*									*/
/************************************************************************/


static void
vtback(curtulip)
   register TULIP_INFO curtulip;
{
   DTRACE("vtback 0x%x",curtulip);

   if (CURCHAR == 0) return;

   vtmove(curtulip,CURLINE,CURCHAR-1);

   if (INSERTMODE) TULIP_delete_char(curtulip,1);
   else if (BSCLEARMODE) {
      TULIP_out_char(curtulip,' ');
      vtmove(curtulip,CURLINE,CURCHAR-1);
    };
};





/************************************************************************/
/*									*/
/*	vteraseeol -- erase end/front/all of current line		*/
/*									*/
/************************************************************************/


static void
vteraseeol(curtulip,p)
   register TULIP_INFO curtulip;
   Integer p;
{
   DTRACE("vteraseeol 0x%x %d",curtulip,p);

   switch (p) {
      case 0 :
	 if (CURCHAR == 0) TULIP_clear_lines(curtulip,CURLINE,CURLINE);
	 else TULIP_clear_chars(curtulip,CURLINE,CURCHAR,NUMCHAR(CURLINE)-1);
	 break;

      case 1 :
	 TULIP_clear_chars(curtulip,CURLINE,0,CURCHAR);
	 break;

      case 2 :
	 TULIP_clear_lines(curtulip,CURLINE,CURLINE);
	 break;
    };
};





/************************************************************************/
/*									*/
/*	vteraseeos -- erase end/front/all of display			*/
/*									*/
/************************************************************************/


static void
vteraseeos(curtulip,p)
   register TULIP_INFO curtulip;
   Integer p;
{
   DTRACE("vteraseeos 0x%x %d",curtulip,p);

   switch (p) {
      case 0 :
	 if (CURCHAR == 0) TULIP_clear_lines(curtulip,CURLINE,NUMLINE-1);
	 else {
	    vteraseeol(curtulip,p);
	    TULIP_clear_lines(curtulip,CURLINE+1,NUMLINE-1);
	  };
	 break;

      case 1 :
	 vteraseeol(curtulip,p);
	 TULIP_clear_lines(curtulip,0,CURLINE-1);
	 break;

      case 2 :
	 TULIP_clear_lines(curtulip,0,NUMLINE-1);
	 vtmove(curtulip,0,0);
	 break;
    };
};





/************************************************************************/
/*									*/
/*	vtfontmap -- map fonts in given area				*/
/*									*/
/************************************************************************/


static void
vtfontmap(curtulip,np,p)
   register TULIP_INFO curtulip;
   Integer np;
   Integer p[];
{
   register Integer i,j,k,j0,j1;
   Integer ol,oc,of;
   register TULIP_LINE lp;
   register TULIP_CHAR cp;
   Integer f0,f1,f2;
   Character buf[MAX_WIDTH];

   DTRACE("vtfontmap 0x%x %d (%d %d %d %d %d %d)",curtulip,np,p[0],p[1],p[2],p[3],p[4],p[5],p[6]);

   TULIP_set_cursor(curtulip,FALSE);

   if (np < 6) return;
   ol = CURLINE;
   oc = CURCHAR;
   of = CURFONT;
   f2 = 0;

   if (p[2] >= MAXLINE) p[2] = MAXLINE-1;
   lp = LINE(p[0]);

   for (i = p[0]; i <= p[2]; ++i) {
      CURLINE = i;
      j0 = (i == p[0]) ? p[1] : 0;
      j1 = (i == p[2]) ? p[3]+1 : NUMCHAR(CURLINE);
      if (j1 > NUMCHAR(CURLINE)) j1 = NUMCHAR(CURLINE);
      cp = LCHAR(lp,j0);
      CURCHAR = j0;
      f0 = -1;
      for (j = j0; j < j1; ++j) {
	 f1 = -1;
	 for (k = 4; k < np; k += 2) {
	    if (cp->font == p[k]) {
	       f1 = p[k+1];
	       break;
	     };
	  };
	 if (f0 >= 0 && f0 != f1) {
	    buf[f2] = 0;
	    CURCHAR -= f2;
	    CURFONT = f0;
	    TULIP_out_string(curtulip,buf);
	    if (j1 > NUMCHAR(CURLINE)) j1 = NUMCHAR(CURLINE);
	    CURCHAR += f2;
	    f0 = -1;
	  };
	 if (f1 >= 0) {
	    if (f0 < 0) {
	       f2 = 0;
	       f0 = f1;
	     };
	    buf[f2++] = cp->charv;
	  };
	 ++cp;
	 ++CURCHAR;
       };
      if (f0 >= 0) {
	 buf[f2] = 0;
	 CURCHAR -= f2;
	 CURFONT = f0;
	 TULIP_out_string(curtulip,buf);
	 if (j1 > NUMCHAR(CURLINE)) j1 = NUMCHAR(CURLINE);
	 CURCHAR += f2;
	 f0 = -1;
       };
      ++lp;
    };

   vtmove(curtulip,ol,oc);
   CURFONT = of;
};






/************************************************************************/
/*									*/
/*	vtmove -- move current position 				*/
/*									*/
/************************************************************************/


static void
vtmove(curtulip,l,c)
   register TULIP_INFO curtulip;
   Integer l;
   Integer c;
{
   register Integer i;

   DTRACE("vtmove 0x%x %d %d",curtulip,l,c);

   if (l >= NUMLINE) l = NUMLINE-1;
   else if (l < 0)   l = 0;

   i = NUMCHAR(l)-1;
   if (c > i) c = i;
   else if (c < 0) c = 0;

   if (CURSORON && (CURCHAR != c || CURLINE != l))
      TULIP_set_cursor(curtulip,FALSE);

   CURLINE = l;
   CURCHAR = c;

   DISCARDON = FALSE;
};





/************************************************************************/
/*									*/
/*	vtnewline -- handle new line					*/
/*									*/
/************************************************************************/


static void
vtnewline(curtulip)
   register TULIP_INFO curtulip;
{
   DTRACE("vtnewline 0x%x",curtulip);

   if (NLCLEARMODE) vteraseeol(curtulip,0);

   if (SCROLLMODE && CURLINE+1 == NUMLINE) {
      TULIP_scroll_up(curtulip,0,NUMLINE-1,1);
      vtmove(curtulip,CURLINE,0);
    }
   else {
      vtmove(curtulip,CURLINE+1,0);
    };
};





/************************************************************************/
/*									*/
/*	vttab -- handle normal tab character				*/
/*									*/
/************************************************************************/


static void
vttab(curtulip)
   register TULIP_INFO curtulip;
{
   register Integer i;

   DTRACE("vttab 0x%x",curtulip);

   i = ((CURCHAR+8)&(~7)) - CURCHAR;

   if (TABCLEARMODE) while (i-- > 0) TULIP_outc(curtulip,' ');
   else vtmove(curtulip,CURLINE,CURCHAR+i);
};





/************************************************************************/
/*									*/
/*	getxchar -- get character for x position in given line		*/
/*									*/
/************************************************************************/


static Integer
getxchar(curtulip,l,x)
   register TULIP_INFO curtulip;
   Integer l;
   Integer x;
{
   register TULIP_CHAR cp,ncp;
   register Integer i,cnt,v;

   DTRACE("getxchar 0x%x %d %d",curtulip,l,x);

   cnt = LINE(l)->nchars;
   cp = LCCHAR(l,0);
   ncp = LCCHAR(l,1);

   for (i = 0; i < cnt-1; ++i) {
      v = (cp->xpos + ncp->xpos + 1)/2;
      if (x <= v) break;
      ++cp;
      ++ncp;
    };

   return i;
};






/************************************************************************/
/*									*/
/*	getyline -- get line for y coordinate				*/
/*									*/
/************************************************************************/


static Integer
getyline(curtulip,y)
   register TULIP_INFO curtulip;
   Integer y;
{
   register TULIP_LINE lp;
   register Integer i;

   DTRACE("getyline 0x%x %d",curtulip,y);

   lp = LINE(1);
   for (i = 0; i < NUMLINE-1; ++i) {
      if (y < LTOP(lp)) break;
      ++lp;
    };

   return i;
};





/* end of tulipexec.c */
