/************************************************************************/
/*									*/
/*		tulip_local.h						*/
/*									*/
/*	Definitions for virtual terminal handler (local)		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#define TULIP_LOCAL

#define PROT_NAME	TULIP__sema
#include <bwethread.h>

#include <ctype.h>
#include "DATATYPES.h"
#include "tulip.h"
#include "ash.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_MAX_LINE	128	/* maximum number of lines	*/
#define DEFAULT_MAX_COL 	256	/* maximum number of columns	*/
#define DEFAULT_MIN_HT		6	/* minimum font height		*/
#define DEFAULT_MIN_WD		4	/* minimum font width		*/


#define NPARM		16		/* # parameters for escape seq	*/
#define MARGIN		1		/* margin size in pixels	*/
#define FONTCHARS	128		/* characters in font		*/
#define BELLCT		1		/* flash count for bells	*/
#define MAX_PUSH_STACK	64		/* depth of PUSH/POP stack	*/
#define MAX_SCROLL	128		/* maximum scroll by size	*/
#define MAX_WIDTH	1024		/* absolute maximum width	*/
#define WRAP_SAFETY	5		/* safety margin for string output */
#define STRING_BUF_SIZE 165		/* buffer size for gathering strings */
#define NUM_ALTS	2		/* # alternative windows	*/

#define HOR_INDENT	2		/* initial horizontal indent	*/
#define VERT_INDENT	3		/* initial vertical indent	*/

#define NFONT		256		/* # of active fonts		*/
#define USER_FONTS	16		/* different user fonts 	*/
#define UNDERLINE_BIT	0x10		/* underline default font	*/
#define REVERSE_BIT	0x20		/* reverse video this font	*/
#define COLOR_BITS	0xe0		/* color bkg (if not reverse)	*/
#define BASE_BITS	0x0f		/* base font contained here	*/
#define BASE_FONT(f)	((f)&BASE_BITS)
#define UNDERLINE(f)	(((f)&UNDERLINE_BIT) != 0)
#define REVERSE(f)	(((f)&COLOR_BITS) == REVERSE_BIT)
#define COLOR(f)	(((f)&COLOR_BITS) >> 5)
#define IS_COLOR(f)	(((f)&COLOR_BITS) > 0x20)
#define COLOR1_BIT	0x20
#define COLOR2_BIT	(2*COLOR1_BIT)
#define COLOR3_BIT	(3*COLOR1_BIT)
#define COLOR4_BIT	(4*COLOR1_BIT)
#define COLOR5_BIT	(5*COLOR1_BIT)
#define COLOR6_BIT	(6*COLOR1_BIT)
#define COLOR7_BIT	(7*COLOR1_BIT)







/************************************************************************/
/*									*/
/*	Tracing definitions						*/
/*									*/
/************************************************************************/


extern	Integer TULIP__tracelvl;
extern	Boolean TULIP__initfg;

#define TRACE	if (TULIP__tracelvl & TULIP_TRACE_ON) TULIP_trace
#define ITRACE	if (TULIP__tracelvl & TULIP_TRACE_INT) TULIP_trace
#define DTRACE	if (TULIP__tracelvl & TULIP_TRACE_DEBUG) TULIP_trace

#define CHECKINIT if (!TULIP__initfg) TULIPinit()
#define ENTER	CHECKINIT; TRACE

#define ERROR(msg) TULIP_error("Error: msg")
#define ABORT(msg) (TULIP_error("ABORT: msg"), abort())





/************************************************************************/
/*									*/
/*	Miscellaneous types						*/
/*									*/
/************************************************************************/


typedef enum {				/* input escape reader states	*/
   TULIP_INPUT_BEGIN,
   TULIP_INPUT_ESCAPE,
   TULIP_INPUT_MODE,
   TULIP_INPUT_PARM,
   TULIP_INPUT_PARM1
} TULIP_INPUT;




typedef Integer 	TULIP_MODES;

#define TULIP_MODE_SAVED	0x1
#define TULIP_MODE_ABSOLUTE	0x2
#define TULIP_MODE_NLCLEAR	0x4
#define TULIP_MODE_BSCLEAR	0x8
#define TULIP_MODE_TABCLEAR	0x10
#define TULIP_MODE_SCROLL	0x20
#define TULIP_MODE_INSERT	0x40
#define TULIP_MODE_CURSOR	0x80
#define TULIP_MODE_WRAP 	0x100
#define TULIP_MODE_FIXED	0x200
#define TULIP_MODE_MOVECURSOR	0x400
#define TULIP_MODE_DISCARD	0x800

#define TULIP_DEFAULT_MODES	0xa0




typedef enum {
   TULIP_SCROLL_UP,
   TULIP_SCROLL_DOWN
} TULIP_SCROLL;





typedef enum {
   TULIP_CURSOR_BLOCK,
   TULIP_CURSOR_UNDERLINE,
   TULIP_CURSOR_CARET
} TULIP_CURSOR;





/************************************************************************/
/*									*/
/*	Line data structures						*/
/*									*/
/************************************************************************/


typedef struct _TULIP_LINE *TULIP_LINE;
typedef struct _TULIP_CHAR *TULIP_CHAR;



typedef struct _TULIP_LINE {		/* information kept for each line */
   Short	ypos;			/*    y position on line	*/
   Short	nchars; 		/*    # characters		*/
   Short	up;			/*    height of line		*/
   Short	down;			/*    down spacing of line	*/
   Character	nmaxup; 		/*    # chars at max up 	*/
   Character	nmaxdown;		/*    # chars at max down	*/
   TULIP_CHAR	chars;			/*    character information	*/
} __TULIP_LINE, TULIP_LINE_BLOCK;



typedef struct _TULIP_CHAR {		/* information for each char	*/
   Character	charv;			/*    character 		*/
   unsigned char font;			/*    font id			*/
   Short	xpos;			/*    x position		*/
} __TULIP_CHAR, TULIP_CHAR_BLOCK;





/************************************************************************/
/*									*/
/*	Box data structures						*/
/*									*/
/************************************************************************/


typedef struct _TULIP_BOX *TULIP_BOX;



typedef struct _TULIP_BOX {	/* box descriptor			*/
   TULIP_BOX	next_box;	/*    link to next box for window	*/
   Short box_flags;		/*    type of box flags 		*/
   Short char0l;		/*    start position			*/
   Short char0c;		/*					*/
   Short char1l;		/*    end position			*/
   Short char1c;		/*					*/
   Short box_id;		/*    identity of box			*/
} __TULIP_BOX, TULIP_BOX_BLOCK;



typedef struct _BOX_DIMS {
   Integer x0,y0,x1,y1;
} BOX_DIMS;


#define BOX_RECTANGLE	0x1	/* rectangular (vs text) box		*/
#define BOX_OUTLINE	0x2	/* outline (vs solid) box		*/
#define BOX_VISIBLE	0x4	/* box currently visible		*/

#define IS_VISIBLE(b) (((b)->box_flags&BOX_VISIBLE) != 0)
#define IS_RECTANGLE(b) (((b)->box_flags&BOX_RECTANGLE) != 0)
#define IS_OUTLINE(b) (((b)->box_flags&BOX_OUTLINE) != 0)

#define SET_VISIBLE(b) ((b)->box_flags |= BOX_VISIBLE)
#define SET_INVISIBLE(b) ((b)->box_flags &= ~BOX_VISIBLE)





/************************************************************************/
/*									*/
/*	Font data structures						*/
/*									*/
/************************************************************************/



typedef struct _TULIP_FONT *TULIP_FONT;




typedef struct _TULIP_FONT {	/* font information			*/
   String name; 		/*    name given by user		*/
   Integer fontid;		/*    font id from ash			*/
   Character width[FONTCHARS];	/*    width of each character		*/
   Character down[FONTCHARS];	/*    below line for each character	*/
   Character up[FONTCHARS];	/*    above line for each character	*/
   Boolean safe;		/*    font doesn't rq background blt    */
} __TULIP_FONT, TULIP_FONT_BLOCK;






/************************************************************************/
/*									*/
/*	Local information for each terminal				*/
/*									*/
/************************************************************************/


typedef struct _TULIP_WINDOW *TULIP_WINDOW;
typedef struct _TULIP_INFO *TULIP_INFO;




typedef struct _TULIP_WINDOW {	/* information on ASH window for tulip	*/
   ASH_WINDOW window;		/*    ash window being used		*/
   Integer lx,ty;		/*    start of the window		*/
   Integer xsize;		/*    width of the window		*/
   Integer ysize;		/*    height of the window		*/
   Integer dirx;		/*    x direction in the window 	*/
   Integer diry;		/*    y direction in the window 	*/
} __TULIP_WINDOW, TULIP_WINDOW_BLOCK;




typedef struct _TULIP_INFO {	/* virtual terminal descriptor		*/
   TULIP_INFO next;		/*    next virtual terminal		*/
   TULIP_WINDOW window; 	/*    window information		*/
   Integer curline;		/*    current line			*/
   Integer curchar;		/*    current character 		*/
   Integer curfont;		/*    current font			*/
   TULIP_MODES modes;		/*    mode flags			*/
   Integer maxline;		/*    maximum line number		*/
   Integer maxcolumn;		/*    maximum column			*/
   Integer numline;		/*    # lines currently displayed	*/
   String read_string;		/*    string for user to read back	*/
   Boolean cursor_on;		/*    internal cursor currently drawn	*/
   Boolean discard_on;		/*    internal discard flag set 	*/
   TULIP_CURSOR cursor_type;	/*    type of internal cursor		*/
   TULIP_BOX box;		/*    boxes for this window		*/
   TULIP_LINE lines;		/*    array of line blocks		*/
   TULIP_INPUT input_state;	/*    input scanning status		*/
   Integer numparm;		/*    number of parameters on input	*/
   Integer parms[NPARM];	/*    parameters on input		*/
   Integer modechar;		/*    mode character for escape input	*/
   Integer lastfont;		/*    last font used on output		*/
   Integer spacefont;		/*    font for blank space		*/
   Integer spacewidth;		/*    width of a space			*/
   Integer spaceup;		/*    up area used by a space		*/
   Integer spacedown;		/*    down area used by a space 	*/
   Integer bufcnt;		/*    counter for string buffer 	*/
   Character buffer[STRING_BUF_SIZE];
   Boolean insert_mode; 	/*    currently in insert mode		*/
   TULIP_INFO alts[NUM_ALTS];	/*    alternate windows 		*/
   ASH_WINDOW caret_bitmap;
   ASH_COLOR fg;
   ASH_COLOR bg;
} __TULIP_INFO, TULIP_INFO_BLOCK;






/************************************************************************/
/*									*/
/*	Global storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Access macros for current virtual terminal			*/
/*									*/
/************************************************************************/



#define VIEWPORT	curtulip->window->window
#define XSIZE		curtulip->window->xsize
#define YSIZE		curtulip->window->ysize
#define RIGHT		XSIZE-2*MARGIN-1
#define BOTTOM		YSIZE-2*MARGIN-1
#define LEFTX		curtulip->window->lx
#define TOPY		curtulip->window->ty
#define DIRX		curtulip->window->dirx
#define DIRY		curtulip->window->diry
#define MAXLINE 	curtulip->maxline
#define MAXCOLUMN	curtulip->maxcolumn
#define NUMLINE 	curtulip->numline
#define CURLINE 	curtulip->curline
#define CURCHAR 	curtulip->curchar
#define CURFONT 	curtulip->curfont
#define MODES		curtulip->modes
#define CURSORON	curtulip->cursor_on
#define DISCARDON	curtulip->discard_on
#define CURSORTYPE	curtulip->cursor_type
#define BOX		curtulip->box
#define INPUTSTATE	curtulip->input_state
#define PARMS		curtulip->parms
#define NUMPARM 	curtulip->numparm
#define MODECHAR	curtulip->modechar
#define READDATA	curtulip->read_string
#define INSERTMODE	curtulip->insert_mode
#define ALTS		curtulip->alts
#define LINES		curtulip->lines
#define LASTFONT	curtulip->lastfont
#define SPACEFONT	curtulip->spacefont
#define SPACEWIDTH	curtulip->spacewidth
#define SPACEUP 	curtulip->spaceup
#define SPACEDOWN	curtulip->spacedown
#define BUFCNT		curtulip->bufcnt
#define BUFFER		curtulip->buffer




/************************************************************************/
/*									*/
/*	Access macros for line/character				*/
/*									*/
/************************************************************************/


#define LINE(i) 	(&(LINES[i]))
#define LCCHAR(l,c)	LCHAR(LINE(l),c)
#define CCHAR(c)	LCCHAR(CURLINE,c)
#define CHAR		LCCHAR(CURLINE,CURCHAR)
#define LCHAR(lp,c)	(&((lp)->chars[c]))

#define NUMCHAR(l)	(LINE(l)->nchars)
#define XPOS(l,c)	(LCCHAR(l,c)->xpos)
#define LTOP(lp)	((lp)->ypos)
#define LBOT(lp)	((lp)->ypos+(lp)->down+(lp)->up)
#define LTOPC(lp)	(LTOP(lp))
#define LBOTC(lp)	(LBOT(lp)-1)
#define HEIGHT(lp)	((lp)->up+(lp)->down)
#define YPOS(lp)	((lp)->ypos+(lp)->up)





/************************************************************************/
/*									*/
/*	Access macros for setting/resetting specific modes		*/
/*									*/
/************************************************************************/


#define NLCLEARMODE	((MODES & TULIP_MODE_NLCLEAR) != 0)
#define BSCLEARMODE	((MODES & TULIP_MODE_BSCLEAR) != 0)
#define TABCLEARMODE	((MODES & TULIP_MODE_TABCLEAR) != 0)
#define SCROLLMODE	((MODES & TULIP_MODE_SCROLL) != 0)
#define CURSORMODE	((MODES & TULIP_MODE_CURSOR) != 0)
#define CURSORMOVEMODE	((MODES & TULIP_MODE_MOVECURSOR) != 0)
#define ABSMODE 	((MODES & TULIP_MODE_ABSOLUTE) != 0)
#define WRAP		((MODES & TULIP_MODE_WRAP) != 0)
#define SAVED		((MODES & TULIP_MODE_SAVED) != 0)
#define FIXED		((MODES & TULIP_MODE_FIXED) != 0)
#define DISCARD 	((MODES & TULIP_MODE_DISCARD) != 0)

#define SET_NLCLEARMODE  (MODES |= TULIP_MODE_NLCLEAR)
#define SET_BSCLEARMODE  (MODES |= TULIP_MODE_BSCLEAR)
#define SET_TABCLEARMODE (MODES |= TULIP_MODE_TABCLEAR)
#define SET_SCROLLMODE	 (MODES |= TULIP_MODE_SCROLL)
#define SET_CURSORMODE	 (MODES |= TULIP_MODE_CURSOR)
#define SET_CURSORMOVEMODE (MODES |= TULIP_MODE_MOVECURSOR)
#define SET_ABSMODE	 (MODES |= TULIP_MODE_ABSOLUTE)
#define SET_WRAP	 (MODES |= TULIP_MODE_WRAP)
#define SET_SAVED	 (MODES |= TULIP_MODE_SAVED)
#define SET_FIXED	 (MODES |= TULIP_MODE_FIXED)
#define SET_DISCARD	 (MODES |= TULIP_MODE_DISCARD)

#define RESET_NLCLEARMODE  (MODES &= ~TULIP_MODE_NLCLEAR)
#define RESET_BSCLEARMODE  (MODES &= ~TULIP_MODE_BSCLEAR)
#define RESET_TABCLEARMODE (MODES &= ~TULIP_MODE_TABCLEAR)
#define RESET_SCROLLMODE   (MODES &= ~TULIP_MODE_SCROLL)
#define RESET_CURSORMODE   (MODES &= ~TULIP_MODE_CURSOR)
#define RESET_CURSORMOVEMODE (MODES &= ~TULIP_MODE_MOVECURSOR)
#define RESET_ABSMODE	   (MODES &= ~TULIP_MODE_ABSOLUTE)
#define RESET_WRAP	   (MODES &= ~TULIP_MODE_WRAP)
#define RESET_SAVED	   (MODES &= ~TULIP_MODE_SAVED)
#define RESET_FIXED	   (MODES &= ~TULIP_MODE_FIXED)
#define RESET_DISCARD	   (MODES &= ~TULIP_MODE_DISCARD)

#define T_PROTECT	TULIP_protect(curtulip,TRUE)
#define T_UNPROTECT	TULIP_protect(curtulip,FALSE)
#define L_PROTECT(ti)	TULIP_protect(ti,TRUE)
#define L_UNPROTECT(ti) TULIP_protect(ti,FALSE)





/************************************************************************/
/*									*/
/*	ASH-TULIP mapping macros					*/
/*									*/
/************************************************************************/


#define ASH_X(x) (LEFTX+(x+MARGIN)*DIRX)
#define ASH_Y(y) (TOPY+(y+MARGIN)*DIRY)
#define TULIP_X(x)  (((x)-LEFTX)*DIRX-MARGIN)
#define TULIP_Y(y)  (((y)-TOPY)*DIRY-MARGIN)





/************************************************************************/
/*									*/
/*	External but local routine definitions				*/
/*									*/
/************************************************************************/


					/* tulipmain.c			*/
extern	void		TULIP_trace();
extern	void		TULIP_error();
extern	Boolean 	TULIP_protect();



					/* tulipuser.c			*/
extern	void		TULIP_user_init();
extern	void		TULIP_saveinput();



					/* tulipexec.c			*/
extern	void		TULIP_exec_init();
extern	void		TULIP_outc();
extern	void		TULIP_outs();
extern	void		TULIP_control();
extern	void		TULIP_escape();



					/* tuliplines.c 		*/
extern	void		TULIP_line_init();
extern	Integer 	TULIP_getfont();
extern	void		TULIP_setup_space();
extern	void		TULIP_setup_lines();
extern	void		TULIP_setup_font();
extern	void		TULIP_draw_all();
extern	void		TULIP_scroll_up();
extern	void		TULIP_scroll_down();
extern	void		TULIP_clear_lines();
extern	void		TULIP_out_char();
extern	void		TULIP_out_string();
extern	void		TULIP_insert_char();
extern	void		TULIP_insert_string();
extern	void		TULIP_delete_char();
extern	void		TULIP_clear_chars();
extern	void		TULIP_set_cursor();
extern	void		TULIP_out_bell();



					/* tulipbox.c			*/
extern	void		TULIP_box_init();
extern	void		TULIP_box_new();
extern	void		TULIP_box_elim();
extern	void		TULIP_box_extend();






/* end of tulip_local.h */
