/************************************************************************/
/*									*/
/*		stemstate.c						*/
/*									*/
/*	State menu interface routines for STEM menu package		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_BTN 	128
#define BTN_EXTRA_SPACE 2
#define TOP_MARGIN	2
#define LEFT_MARGIN	2
#define LINE_MARGIN	2
#define SPACE_WIDTH	6
#define CH_EVENTS	RIP_NO_CHARS
#define BTN_EVENTS	RIP_BTN_ANY_UP
#define NUM_COLOR	10
#define NEW_ADDON	4

#define STD_BOX_SIZE	12





/************************************************************************/
/*									*/
/*	Data types							*/
/*									*/
/************************************************************************/


typedef struct _STEM_STATE_WIN *	STEM_STATE_WIN;
typedef struct _STEM_STATE_BTN *	STEM_STATE_BTN;
typedef struct _STEM_STATE_FONT *	STEM_STATE_FONT;



typedef struct _STEM_STATE_BTN {
   Integer lx,by,rx,ty;
   RIP_REGION rgn;
   STEM_BTN btn;
   Integer state_id;
   Integer state_num;
   String text;
   Integer color;
   Boolean enabled;
   Boolean box;
   Integer fill;
} STEM_STATE_BTN_B;




typedef struct _STEM_STATE_WIN {
   ASH_WINDOW window;
   Universal udata;
   Function_Ptr rtn;
   Integer * data;
   STEM_STATE_WIN next;
   String menu;
   Integer nbtn;
   Integer backcolor;
   STEM_STATE_BTN_B * btn;
   STEM_BTN dummy_btn;
   ASH_COLOR fg;
   ASH_COLOR bg;
} STEM_STATE_WIN_B;




typedef struct _STEM_STATE_FONT {
   String name;
   Integer id;
   Integer ashid;
   STEM_STATE_FONT next;
} STEM_STATE_FONT_B;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	STEM_STATE_WIN	all_state;
static	STEM_STATE_FONT all_state_font;
static	Integer 	base_state_font;
static	Boolean 	box_default;

static	Integer 	std_box_size;



typedef struct _COLOR_INFO {
   String name;
   String backname;
} COLOR_INFO;

static	COLOR_INFO	color_table[NUM_COLOR] = {
   { "black", "white" },
   { "red", "white" },
   { "orange", "white" },
   { "green", "white" },
   { "blue", "white" },
   { "maroon", "white" },
   { "black", "yellow" },
   { "yellow", "green" },
   { "white", "red" },
   { "black", "white" },
};





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		state_refresh();
static	Integer 	state_control();
static	Integer 	state_btn();
static	void		scan_menu();
static	Integer 	scan_btns();
static	void		free_state_menu();
static	void		draw_state_menu();
static	Integer 	out_text();
static	void		hilite();
static	Integer 	setup_state_font();
static	Integer 	get_font();





/************************************************************************/
/*									*/
/*	STEM_state_init -- module initialization			*/
/*									*/
/************************************************************************/


void
STEM_state_init()
{
   String s;

   ITRACE("STEM_state_init");

   all_state = NULL;
   all_state_font = NULL;

   box_default = TRUE;

   base_state_font = setup_state_font(ASHinq_base_font());

   s = ASHinq_resource("state.foreground");
   if (s != NULL) color_table[0].name = s;
   s = ASHinq_resource("state.background");
   if (s != NULL) color_table[0].backname = s;

   s = ASHinq_resource("state.box_size");
   if (s != NULL) std_box_size = atol(s);
   else std_box_size = 0;
   if (std_box_size == 0) std_box_size = STD_BOX_SIZE;
   std_box_size += NEW_ADDON;
};





/************************************************************************/
/*									*/
/*	STEMstate -- set up a state menu				*/
/*									*/
/************************************************************************/


void
STEMstate(w,menu,rtn,data,udata)
   ASH_WINDOW w;
   String menu[];
   Function_Ptr rtn;
   Integer * data;
   Universal udata;
{
   register STEM_STATE_WIN sw;
   Character buf[2048],nbuf[2048];
   STEM_STATE_BTN_B btns[MAX_BTN];
   register Integer i;

   ENTER("STEMstate 0x%x %s 0x%x 0x%x",w,menu[0],rtn,data);

   sw = (STEM_STATE_WIN) calloc(1,sizeof(STEM_STATE_WIN_B));

   sw->window = w;
   sw->rtn = rtn;
   sw->data = data;
   sw->udata = udata;
   sw->backcolor = -1;
   sw->dummy_btn = NULL;
   sw->fg = ASHinq_color(w);
   sw->bg = ASHinq_background_color(w);

   scan_menu(menu,buf);

   sw->nbtn = scan_btns(sw,buf,nbuf,btns);
   if (sw->nbtn < 0) {
      fprintf(stderr,"Bad stem state menu definition:\n%s\n",buf);
      exit(1);
    };

   sw->menu = SALLOC(nbuf);
   sw->btn = (STEM_STATE_BTN_B *) calloc(sw->nbtn,sizeof(STEM_STATE_BTN_B));
   for (i = 0; i < sw->nbtn; ++i) {
      sw->btn[i] = btns[i];
    };

   PROTECT;
   sw->next = all_state;
   all_state = sw;
   UNPROTECT;

   BIOnew_input_window(w);
   BIOset_cursor_standard(w,ASH_CURSOR_SMALL_XHAIRS);

   draw_state_menu(sw);

   ASHset_menu_data(w,sw);
   ASHset_region_refresh(w,state_refresh);
   ASHset_control(w,state_control);
};






/************************************************************************/
/*									*/
/*	STEMstate_inq -- get current value for state			*/
/*	STEMstate_set -- set current value for state			*/
/*									*/
/************************************************************************/


int
STEMstate_inq(w,num)
   ASH_WINDOW w;
   Integer num;
{
   register STEM_STATE_WIN sw;
   register Integer i;

   ENTER("STEMstate_inq 0x%x %d",w,num);

   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return -1;

   for (i = 0; i < sw->nbtn; ++i) {
      if (sw->btn[i].state_num == num && sw->btn[i].enabled)
	 return sw->btn[i].state_id;
    };

   return -1;
};





void
STEMstate_set(w,num,id)
   ASH_WINDOW w;
   Integer num;
   Integer id;
{
   register STEM_STATE_WIN sw;
   register Integer i;

   ENTER("STEMstate_set 0x%x %d %d",w,num,id);

   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return;

   for (i = 0; i < sw->nbtn; ++i) {
      if (sw->btn[i].state_num == num) {
	 if (sw->btn[i].state_id == id) {
	    if (!sw->btn[i].enabled) hilite(sw,&sw->btn[i],TRUE);
	  }
	 else if (sw->btn[i].enabled) hilite(sw,&sw->btn[i],FALSE);
       };
    };

   if (sw->data != NULL) {
      sw->data[num] = id;
    };
};





/************************************************************************/
/*									*/
/*	STEMstate_color -- define color for state windows		*/
/*									*/
/************************************************************************/


void
STEMstate_color(val,color,backg)
   Integer val;
   String color;
   String backg;
{
   if (val < 0 || val >= NUM_COLOR) return;

   if (color != NULL) color_table[val].name = SALLOC(color);
   if (backg != NULL) color_table[val].backname = SALLOC(backg);
};





/************************************************************************/
/*									*/
/*	STEMstate_remove -- remove state window 			*/
/*									*/
/************************************************************************/


void
STEMstate_remove(w)
   ASH_WINDOW w;
{
   register STEM_STATE_WIN sw;

   ENTER("STEMstate_remove 0x%x",w);

   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return;

   if (sw->dummy_btn != NULL) {
      STEMbtn_remove(sw->dummy_btn);
      sw->dummy_btn = NULL;
    };

   ASHremove_control(w,state_control);
   ASHset_menu_data(w,NULL);
   ASHset_refresh(w,ASH_REFRESH_CLEAR);
   free_state_menu(sw);
};





/************************************************************************/
/*									*/
/*	state_refresh -- refresh state window				*/
/*									*/
/************************************************************************/


static void
state_refresh(w)
   ASH_WINDOW w;
{
   register STEM_STATE_WIN sw;

   ITRACE("state_refresh 0x%x",w);

   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return;

   draw_state_menu(sw);
};





/************************************************************************/
/*									*/
/*	state_control -- handle control messages			*/
/*									*/
/************************************************************************/


static int
state_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register STEM_STATE_WIN sw;

   ITRACE("state_control %s 0x%x",msg,w);

   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      ASHset_menu_data(w,NULL);
      free_state_menu(sw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	state_btn -- handle RIP hits					*/
/*									*/
/************************************************************************/


static Integer
state_btn(x,y,ch,btns,rgn)
   Integer x,y;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   register ASH_WINDOW w;
   register STEM_STATE_WIN sw;
   register Boolean fg;
   STEM_STATE_BTN sb;

   ITRACE("state_btn %d %d 0x%x 0x%x 0x%x",x,y,ch,btns,rgn);

   w = RIPinq_window(rgn);
   sw = (STEM_STATE_WIN) ASHinq_menu_data(w);
   if (sw == NULL) return FALSE;
   sb = (STEM_STATE_BTN) RIPinq_data(rgn);
   if (sb == NULL) return FALSE;

   if (sw->rtn != NULL) {
      fg = (*sw->rtn)(sw->udata,sb->state_num,sb->state_id,btns,w);
      if (!fg) return FALSE;
    };

   STEMstate_set(w,sb->state_num,sb->state_id);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	scan_menu -- put menu array into single string			*/
/*									*/
/************************************************************************/


static void
scan_menu(menu,buf)
   String menu[];
   String buf;
{
   register Integer i,j;

   buf[0] = 0;
   for (i = 0; menu[i] != NULL; ++i) {
      strcat(buf,menu[i]);
      j = strlen(buf);
      if (buf[j-2] != '%' || buf[j-1] != 'N') strcat(buf,"\n");
      else buf[j-2] = 0;
    };
};






/************************************************************************/
/*									*/
/*	scan_btns -- set up button array				*/
/*									*/
/************************************************************************/


static Integer
scan_btns(sw,menu,nmenu,btns)
   STEM_STATE_WIN sw;
   String menu;
   String nmenu;
   STEM_STATE_BTN_B btns[];
{
   register Integer nbtn;
   register String s,t,th;
   Character txt[72],c;
   register Integer v1,v2,v3,v4,j;
   Integer nx,ny,ft,h;

   nbtn = 0;
   t = nmenu;

   th = t++;
   h = 0;
   ft = get_font(base_state_font);

   for (s = menu; *s != 0; ++s) {
      if (*s == '\n') {
	 *t++ = *s;
	 if (h == 0) {
	    ASHinq_text_extent(ft," ",&nx,&ny);
	    h = ny;
	  };
	 *th = h;
	 th = t++;
       }
      else if (*s != '%') {
	 *t++ = *s;
	 *t = 0;
	 ASHinq_text_extent(ft,&t[-1],&nx,&ny);
	 if (ny > h) h = ny;
       }
      else {
	 *t++ = *s++;
	 v1 = v2 = v3 = v4 = 0;
	 txt[0] = 0;
	 if (*s == 0 || *s == '\n') return -1;
	 while (isdigit(*s)) v1 = v1*10 + (*s++ - '0');
	 if (*s == '.') {
	    ++s;
	    while(isdigit(*s)) v2 = v2*10 + (*s++ - '0');
	  };
	 if (*s == '.') {
	    ++s;
	    while(isdigit(*s)) v3 = v3*10 + (*s++ - '0');
	  };
	 if (*s == '.') {
	    ++s;
	    while(isdigit(*s)) v4 = v4*10 + (*s++ - '0');
	  };
	 if (!isalpha(*s) && *s != 0 && *s != '\n' && *s != '%') {
	    c = *s++;
	    j = 0;
	    while (*s != c && *s != 0 && *s != '\n' && j < 64) {
	       txt[j++] = *s++;
	     };
	    if (*s != c) return -1;
	    ++s;
	    txt[j] = 0;
	  };
	 *t++ = *s;
	 switch (*s) {
	    case 'C' :
	    case 'R' :
	    case '%' :
	       break;
	    case 'B' :
	       if (v1 >= 0 && v1 < NUM_COLOR && sw->backcolor < 0) sw->backcolor = v1;
	       break;
	    case 'M' :
	       if (v1 == 0) v1 = 1;
	       if (v2 == 0) v2 = v1+1;
	       *t++ = v1;
	       *t++ = v2;
	       break;
	    case 'F' :
	       if (txt[0] == 0) v1 = base_state_font;
	       else v1 = setup_state_font(txt);
	       *t++ = v1;
	       ft = get_font(v1);
	       break;
	    case 'O' :
	    case 'o' :
	    case 'b' :
	    case 'f' :
	    case 'c' :
	       t[-1] = 'b';
	       btns[nbtn].lx = -1;
	       btns[nbtn].rgn = NULL;
	       btns[nbtn].btn = NULL;
	       if (txt[0] == 0) btns[nbtn].text = NULL;
	       else btns[nbtn].text = SALLOC(txt);
	       btns[nbtn].state_id = v2;
	       btns[nbtn].state_num = v1;
	       btns[nbtn].enabled = FALSE;
	       if (*s == 'O') btns[nbtn].box = FALSE;
	       else if (*s != 'o') btns[nbtn].box = TRUE;
	       else if (!box_default) btns[nbtn].box = FALSE;
	       else if (ASHinq_configuration_depth(sw->window) == 1) btns[nbtn].box = TRUE;
	       else btns[nbtn].box = FALSE;
	       btns[nbtn].color = v3;
	       if (*s == 'f' && v4 == 0) v4 = 1;
	       btns[nbtn].fill = v4;
	       *t++ = nbtn+1;
	       ++nbtn;
	       if (txt[0] != 0) {
		  STEMbtn_size(ft,txt,&nx,&ny);
		  ny += BTN_EXTRA_SPACE;
		}
	       else ny = std_box_size + BTN_EXTRA_SPACE;
	       if (ny > h) h = ny;
	       break;
	    default :
	       return -1;
	  };
       };
    };

   *th = h;

   *t = 0;

   return nbtn;
};





/************************************************************************/
/*									*/
/*	free_state_menu -- release menu 				*/
/*									*/
/************************************************************************/


static void
free_state_menu(sw)
   STEM_STATE_WIN sw;
{
   register Integer i;

   ASHset_menu_data(sw->window,NULL);
   ASHremove_control(sw->window,state_control);
   sw->window = NULL;

   SFREE(sw->menu);

   for (i = 0; i < sw->nbtn; ++i) {
      SFREE(sw->btn[i].text);
      if (sw->btn[i].rgn != NULL) {
	 RIPremove_region(sw->btn[i].rgn);
	 sw->btn[i].rgn = NULL;
       };
      if (sw->btn[i].btn != NULL) {
	 STEMbtn_remove(sw->btn[i].btn);
	 sw->btn[i].btn = NULL;
       };
    };

   if (sw->dummy_btn != NULL) {
      STEMbtn_remove(sw->dummy_btn);
      sw->dummy_btn = NULL;
    };

   free(sw->btn);

   free(sw);
};





/************************************************************************/
/*									*/
/*	draw_state_menu -- draw a state menu window			*/
/*									*/
/************************************************************************/


static void
draw_state_menu(sw)
   STEM_STATE_WIN sw;
{
   Byte * s;
   Integer lx,by,rx,ty;
   Integer dx,dy;
   Integer just,h,ft;
   Boolean havebtn;
   Character txt[1024];
   Integer i,v1,v2,x,j;
   Integer c,bc;
   STEM_BTN sb;

   if (sw->window == NULL) return;

   ASHbatch_mode(TRUE);

   ASHinq_size(sw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);

   i = (sw->backcolor < 0 ? 0 : sw->backcolor);
   if (i == 0) {
      c = sw->fg;
      bc = sw->bg;
    }
   else {
      c = ASHlookup_color(sw->window,color_table[i].name);
      bc = ASHlookup_color(sw->window,color_table[i].backname);
    };
   ASHcolor(sw->window,c);
   ASHtext_color(sw->window,c);
   ASHbackground_color(sw->window,bc);
   ASHtext_background_color(sw->window,bc);

   if (sw->dummy_btn != NULL) STEMbtn_remove(sw->dummy_btn);
   sb = STEMbtn_define(sw->window,1,1,1,1,"",STEM_BTN_NO_DRAW,NULL,NULL);
   if (sw->backcolor >= 0 && ASHinq_configuration_depth(sw->window) > 1) {
      STEMbtn_set_gray(sb,sw->backcolor);
    };
   STEMbtn_background(sb,0,0,0,0);
   sw->dummy_btn = sb;

   ft = get_font(base_state_font);

   s = (Byte *) sw->menu;
   h = *s++;
   ty += (TOP_MARGIN + h)*dy;
   j = 0;
   x = lx + LEFT_MARGIN*dx;
   just = 0;
   havebtn = FALSE;
   txt[0] = 0;

   while (*s != 0) {
      if (*s == '\n') {
	 x = out_text(sw,x,ty,ft,just,txt,NULL);
	 just = 0;
	 j = 0;
	 h = *++s;
	 ty += (LINE_MARGIN + h)*dy;
	 if (havebtn) ty += BTN_EXTRA_SPACE*dy;
	 x = lx + LEFT_MARGIN*dx;
	 havebtn = FALSE;
       }
      else if (*s != '%') {
	 txt[j++] = *s;
       }
      else if (s[1] == '%') {
	 txt[j++] = *++s;
       }
      else {
	 x = out_text(sw,x,ty,ft,just,txt,NULL);
	 j = 0;
	 switch (*++s) {
	    case 'F' :
	       ft = get_font(*++s);
	       break;
	    case 'M' :
	       v1 = *++s;
	       v2 = *++s;
	       v1 = lx + (rx-lx)*v1/v2;
	       x += SPACE_WIDTH;
	       if ((v1-x)*dx > 0) x = v1;
	       break;
	    case 'C' :
	       just = 1;
	       break;
	    case 'R' :
	       just = 2;
	       break;
	    case 'b' :
	       i = *++s - 1;
	       x = out_text(sw,x,ty,ft,just,sw->btn[i].text,&sw->btn[i]);
	       havebtn = TRUE;
	       break;
	  };
       };
      txt[j] = 0;
      ++s;
    };

   if (j != 0) {
      out_text(sw,x,ty,ft,just,txt,NULL);
    };

   ASHcolor(sw->window,c);
   ASHtext_color(sw->window,c);
   ASHbackground_color(sw->window,bc);
   ASHtext_background_color(sw->window,bc);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	out_text -- output text to window				*/
/*									*/
/************************************************************************/


static Integer
out_text(sw,x,y,ft,just,txt,btn)
   STEM_STATE_WIN sw;
   Integer x,y;
   Integer ft;
   Integer just;
   String txt;
   STEM_STATE_BTN btn;
{
   Integer lx,by,rx,ty;
   Integer nx,ny,ox,oy,ex,ey;
   Integer dx,dy;
   Integer rslt,dlta,x0;
   Integer i,f,color;
   STEM_BTN sb;

   if (txt == NULL || txt[0] == 0) return x;

   ASHinq_size(sw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);
   rx -= (LEFT_MARGIN)*dx;

   if (txt != NULL) {
      if (btn != NULL) {
	 STEMbtn_size(ft,txt,&ex,&ey);
	 nx = ex+2;
	 ny = ey+2;
	 ox = oy = 0;
       }
      else {
	 ASHinq_text_offset(ft,txt,&ox,&oy);
	 ASHinq_text_next(ft,txt,&nx,&ny);
	 ASHinq_text_extent(ft,txt,&ex,&ey);
       };
    }
   else {
      ox = oy = 0;
      ex = std_box_size;
      ey = std_box_size;
      nx = ex+2;
      ny = ey+2;
    };

   if (btn != NULL) {
      ox += BTN_EXTRA_SPACE;
      nx += 2*BTN_EXTRA_SPACE;
      ex += 2*BTN_EXTRA_SPACE;
    };

   rslt = x+nx*dx;

   dlta = (rx-x)*dx - ex;

   switch (just) {
      case 0 :
	 break;
      case 1 :
	 x += dlta/2;
	 rslt += dlta/2;
	 break;
      case 2 :
	 x += dlta;
	 rslt = rx;
	 break;
    };

   if (ASHinq_configuration_depth(sw->window) > 1) {
      if (btn == NULL) i = 0;
      else i = btn->color;
      if (i == 0) color = sw->fg;
      else color = ASHlookup_color(sw->window,color_table[i].name);
      ASHcolor(sw->window,color);
      ASHtext_color(sw->window,color);
      if (i == 0) color = sw->bg;
      else color = ASHlookup_color(sw->window,color_table[i].backname);
      ASHbackground_color(sw->window,color);
      ASHtext_background_color(sw->window,color);
    };

   if (btn != NULL && btn->btn != NULL) {
      STEMbtn_remove(btn->btn);
      btn->btn = NULL;
    };

   if (txt != NULL) {
      ASHfont(sw->window,ft);
      if (btn != NULL) {
	 sb = STEMbtn_define(sw->window,x,y,x+ex*dx,y-ey*dy,txt,
				STEM_BTN_SENSE_FLIP,
				btn,state_btn);
	 btn->btn = sb;
       }
      else ASHtext(sw->window,x+ox*dx,y-oy*dy,txt);
    }
   else {
      x0 = x+ox*dx;
      if (btn->fill != 0) {
	 f = ASHfill(sw->window,btn->fill);
	 ASHrectangle(sw->window,x0,y,x0+ex*dx,y-ey*dy);
	 ASHfill(sw->window,f);
       }
      else if (btn->color == 0) {
	 sb = STEMbtn_define(sw->window,x,y,x+ex*dx,y-ey*dy,NULL,
				STEM_BTN_SENSE_INOUT,
				btn,state_btn);
	 STEMbtn_set(sb,btn->enabled);
	 btn->btn = sb;
       }
      else {
	 ASHbox(sw->window,x0,y,x0+ex*dx,y-ey*dy);
       };
    };

   if (btn != NULL) {
      btn->lx = x;
      btn->by = y+BTN_EXTRA_SPACE*dy;
      btn->rx = x+ex*dx;
      btn->ty = y-(ey+BTN_EXTRA_SPACE)*dy;
      if (btn->rgn != NULL) RIPremove_region(btn->rgn);
      if (btn->btn == NULL) {
	 btn->rgn = RIPdefine_region(sw->window,btn->lx,btn->by,btn->rx,btn->ty,
					CH_EVENTS,BTN_EVENTS,state_btn,TRUE);
	 RIPset_data(btn->rgn,btn);
	 if (txt != NULL) RIPset_region_id(btn->rgn,txt);
       };
      if (btn->enabled) hilite(sw,btn,TRUE);
    };

   return rslt;
};




/************************************************************************/
/*									*/
/*	hilite -- highlight a button					*/
/*									*/
/************************************************************************/


static void
hilite(sw,btn,fg)
   STEM_STATE_WIN sw;
   STEM_STATE_BTN btn;
   Boolean fg;
{
   if (btn->btn != NULL) {
      STEMbtn_set(btn->btn,fg);
    }
   else if (btn->box) {
      ASHcombination_rule(sw->window,10);
      ASHbox(sw->window,btn->lx,btn->by,btn->rx,btn->ty);
      ASHcombination_rule(sw->window,3);
    }
   else {
      ASHhilite_box(sw->window,btn->lx,btn->by,btn->rx,btn->ty);
    };

   btn->enabled = fg;
};





/************************************************************************/
/*									*/
/*	setup_state_font -- get font id given font name 		*/
/*									*/
/************************************************************************/


static Integer
setup_state_font(nm)
   String nm;
{
   register STEM_STATE_FONT sf;

   PROTECT;
   for (sf = all_state_font; sf != NULL; sf = sf->next) {
      if (STREQL(sf->name,nm)) break;
    };
   if (sf == NULL) {
      sf = (STEM_STATE_FONT) calloc(1,sizeof(STEM_STATE_FONT_B));
      sf->name = SALLOC(nm);
      sf->ashid = ASHloadfont(sf->name);
      if (all_state_font == NULL) sf->id = 1;
      else sf->id = all_state_font->id+1;
      sf->next = all_state_font;
      all_state_font = sf;
    };
   UNPROTECT;

   return sf->id;
};





/************************************************************************/
/*									*/
/*	get_font -- find font from id					*/
/*									*/
/************************************************************************/


static Integer
get_font(id)
   Integer id;
{
   register STEM_STATE_FONT sf;

   for (sf = all_state_font; sf != NULL; sf = sf->next) {
      if (sf->id == id) return sf->ashid;
    };

   return get_font(base_state_font);
};





/* end of stemstate.c */





