/************************************************************************/
/*									*/
/*		stemscroll.c						*/
/*									*/
/*	Scroll bar routines for STEM menu package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define CH_EVENTS	RIP_NO_CHARS
#define BTN_EVENTS	(RIP_BTN_ANY_EITHER)
#define CEN_EVENTS	(RIP_BTN_ANY_EITHER)

#define CENTER_FILL	ASH_FILL_HALFTONE
#define BACK_FILL	ASH_FILL_LIGHT_HALFTONE
#define MIN_WIDTH_3	30
#define MIN_SPACE_VH	40
#define END_SPACE	10

#define MIN_SIZE	12




/************************************************************************/
/*									*/
/*	Data types							*/
/*									*/
/************************************************************************/


typedef struct _STEM_SCROLL *	STEM_SCROLL;



typedef enum {
   SCROLL_TYPE_3,
   SCROLL_TYPE_V,
   SCROLL_TYPE_H
} STEM_SCROLL_TYPE;





typedef struct _STEM_SCROLL {
   ASH_WINDOW	window;
   ASH_WINDOW	left_win;
   ASH_WINDOW	right_win;
   Function_Ptr call_rtn;
   Integer	user_data;
   Integer	first,last;
   Integer	top,bottom;
   Integer	clx,crx;
   Integer	cby,cty;
   Integer	by,ty;
   Integer	rby,rty;
   Integer	wdy,wdx;
   ASH_COLOR	bar_color;
   RIP_REGION	left_region;
   RIP_REGION	right_region;
   RIP_REGION	center_region;
   STEM_SCROLL_TYPE type;
   THREAD	thread;
   STEM_BTN	btn;
} STEM_SCROLL_B;




typedef struct _SCROLL_DATA {
   Integer x;
   Integer y;
   STEM_SCROLL ss;
   Integer lastx;
   Integer lasty;
} SCROLL_DATA;





/********************************************************************************/
/*										*/
/*	Local storage								*/
/*										*/
/********************************************************************************/


static	String		bar_color;
static	Integer 	bar_space;
static	Integer 	min_size;

static	Integer 	handle_icon[] = {		/* 9x9		*/
   0x1c000000,
   0x77000000,
   0x41000000,
   0xc1800000,
   0x80800000,
   0xc1800000,
   0x41000000,
   0x77000000,
   0x1c000000,
};





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Integer 	scroll_control();
static	void		scroll_refresh();
static	void		scroll_refresh_rgn();
static	Integer 	scroll_btn();
static	int		scroll_cont();
static	Integer 	scroll_move_btn();
static	Integer 	move_rtn();

static	void		free_scroll();
static	void		layout_scroll();
static	void		draw_scroll();
static	void		draw_update_scroll();
static	void		draw_triangle();
static	void		draw_arrow();
static	void		draw_scroll_body();





/************************************************************************/
/*									*/
/*	STEM_scroll_init -- module initialization			*/
/*									*/
/************************************************************************/


void
STEM_scroll_init()
{
   String s;

   ITRACE("STEM_scroll_init");

   ASHicon_define("SCROLL_HANDLE",9,9,handle_icon);

   bar_color = ASHinq_resource("scroll.color");

   s = ASHinq_resource("scroll.space");
   if (s != NULL) bar_space = atol(s);
   else bar_space = 2;

   min_size = MIN_SIZE;
};





/************************************************************************/
/*									*/
/*	STEMscroll -- define a scroll bar				*/
/*									*/
/************************************************************************/


void
STEMscroll(window,rtn,id)
   ASH_WINDOW window;
   Function_Ptr rtn;
   Integer id;
{
   register STEM_SCROLL ss;

   ENTER("STEMscroll 0x%x 0x%x",window,id);

   ss = (STEM_SCROLL) calloc(1,sizeof(STEM_SCROLL_B));

   ss->window = window;
   ss->user_data = id;
   ss->call_rtn = rtn;
   ss->first = 0;
   ss->last = -1;
   ss->top = 0;
   ss->bottom = -1;
   ss->thread = THREADinq_current();
   ss->btn = NULL;

   if (bar_color == NULL) ss->bar_color = -1;
   else ss->bar_color = ASHlookup_color(window,bar_color);

   if (!ASHlock(window)) return;

   ASHremove_control(window,scroll_control);
   ASHset_control(window,scroll_control);
   ASHset_refresh(window,scroll_refresh);

   layout_scroll(ss);

   ASHunlock(window);
   ASHset_menu_data(window,ss);
};





/************************************************************************/
/*									*/
/*	STEMscroll_set -- set limits for a scroll region		*/
/*									*/
/************************************************************************/


void
STEMscroll_set(w,top,bottom,first,last)
   ASH_WINDOW w;
   Integer top,bottom;
   Integer first,last;
{
   register STEM_SCROLL ss;
   register Integer x;

   ENTER("STEMscroll_set 0x%x %d %d %d %d",w,top,bottom,first,last);

   ss = (STEM_SCROLL) ASHinq_menu_data(w);

   if (ss == NULL) return;

   if (top < bottom) {
      if (first < top) first = top;
      if (last > bottom) last = bottom;
    }
   else {
      if (first < bottom) first = bottom;
      if (last > top) last = top;
    };

   x = ss->last - ss->first;

   ss->top = top;
   ss->bottom = bottom;
   ss->first = first;
   ss->last = last;

   if (x <= 0) draw_scroll(ss);
   else draw_update_scroll(ss);
};




/************************************************************************/
/*									*/
/*	STEMscroll_thread -- set thread for scroll bars 		*/
/*									*/
/************************************************************************/


void
STEMscroll_thread(w,th)
   ASH_WINDOW w;
   THREAD th;
{
   STEM_SCROLL ss;

   ENTER("STEMscroll_thread 0x%x 0x%x",w,th);

   ss = (STEM_SCROLL) ASHinq_menu_data(w);

   ss->thread = th;
   BIOset_window_thread(ss->window,ss->thread);
   if (ss->left_win != NULL) BIOset_window_thread(ss->left_win,ss->thread);
   if (ss->right_win != NULL) BIOset_window_thread(ss->right_win,ss->thread);
};





/********************************************************************************/
/*										*/
/*	STEMscroll_color -- specify color of scroll bar 			*/
/*										*/
/********************************************************************************/


void
STEMscroll_color(col)
   String col;
{
   if (col == NULL) bar_color = NULL;
   else bar_color = SALLOC(col);
};





/************************************************************************/
/*									*/
/*	scroll_control -- handle window changes for scroll bars 	*/
/*									*/
/************************************************************************/


static Integer
scroll_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register STEM_SCROLL ss;

   ITRACE("scroll_control %s 0x%x",msg,w);

   ss = (STEM_SCROLL) ASHinq_menu_data(w);

   if (ss == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$RESIZE")) {
      layout_scroll(ss);
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      ASHset_menu_data(w,NULL);
      free_scroll(ss);
    };

   return ASH_CONTROL_REJECT;
};




/************************************************************************/
/*									*/
/*	scroll_refresh -- refresh scroll bar				*/
/*	scroll_refresh_rgn -- handle refresh for arrows 		*/
/*									*/
/************************************************************************/


static void
scroll_refresh(w)
   ASH_WINDOW w;
{
   register STEM_SCROLL ss;

   ITRACE("scroll_refresh 0x%x",w);

   ss = (STEM_SCROLL) ASHinq_menu_data(w);

   if (ss != NULL && ss->top <= ss->bottom) draw_scroll(ss);
};




static void
scroll_refresh_rgn(w)
   ASH_WINDOW w;
{
   STEM_SCROLL ss;

   ITRACE("scroll_refresh_rgn 0x%x",w);


   ss = (STEM_SCROLL) ASHinq_menu_data(w);

   if (ss == NULL || ss->window == NULL) return;

   if (ss->type == SCROLL_TYPE_H) {
      if (w == ss->left_win) draw_triangle(ss->left_win,0,ss);
      else if (w == ss->right_win) draw_triangle(ss->right_win,2,ss);
    }
   else if (ss->type == SCROLL_TYPE_V) {
      if (w == ss->left_win) draw_triangle(ss->left_win,1,ss);
      else if (w == ss->right_win) draw_triangle(ss->right_win,3,ss);
    }
   else {
      if (w == ss->left_win) draw_arrow(ss->left_win,0,ss);
      else if (w == ss->right_win) draw_arrow(ss->right_win,1,ss);
    };
};





/************************************************************************/
/*									*/
/*	scroll_btn -- handle left/right button				*/
/*									*/
/************************************************************************/


static Integer
scroll_btn(x,y,ch,btns,rgn)
   Integer x,y;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   register STEM_SCROLL ss;
   register Integer d,m;
   Boolean mwdfg;
   ASH_WINDOW w;

   ITRACE("scroll_btn %d %d 0x%x 0x%x 0x%x",x,y,ch,btns,rgn);

   if (btns & RIP_BTN_DOWN) {
      w = RIPinq_window(rgn);
      ASHset_user_data(w,0);
      mwdfg = BPIOmove_when_down(TRUE);
      RIPtrack(scroll_cont,-1,w,FALSE);
      BPIOmove_when_down(mwdfg);
      return TRUE;
    };

   ss = (STEM_SCROLL) RIPinq_data(rgn);

   d = (rgn == ss->left_region ? -1 : 1);

   if (btns & RIP_BTN_MID) m = abs(ss->last-ss->first)/3;
   else if (btns & RIP_BTN_RIGHT) m = abs(ss->last - ss->first);
   else m = 1;

   if (m <= 0) m = 1;

   if (ss->call_rtn != NULL) {
      (*(ss->call_rtn))(ss->user_data,d,m);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	scroll_cont -- handle continuous scroll requests		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
scroll_cont(x,y,ct,act,max,w,btn,ch)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
   Integer btn;
   Integer ch;
{
   STEM_SCROLL ss;
   Integer i,d;

   ss = (STEM_SCROLL) ASHinq_menu_data(w);
   if (ss == NULL) return FALSE;

   switch (act) {
      case RIP_TRANS_NONE :
      case RIP_TRANS_MOVE :
      case RIP_TRANS_DOWN :
      case RIP_TRANS_TAP :
	 i = (Integer) ASHinq_user_data(w);
	 ASHset_user_data(w,i+1);
	 d = (w == ss->left_win ? -1 : 1);
	 if (ss->call_rtn != NULL) {
	    (*(ss->call_rtn))(ss->user_data,d,1);
	  };
	 break;
      case RIP_TRANS_UP :
	 return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	scroll_move_btn -- handle moving the scroll region		*/
/*									*/
/************************************************************************/


static Integer
scroll_move_btn(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch,btn;
   RIP_REGION rgn;
{
   register STEM_SCROLL ss;
   register Boolean fg;
   register Integer v;
   register ASH_WINDOW w;
   register Boolean cfg;
   ASH_CURSOR crsr;
   SCROLL_DATA sdata;
   ASH_WINDOW tw;
   Integer wd;
   Integer nx,ny;

   ITRACE("scroll_move_btn %d %d 0x%x 0x%x 0x%x",x,y,ch,btn,rgn);

   ss = (STEM_SCROLL) RIPinq_data(rgn);

   if (btn & RIP_BTN_TAP) {
      ASHmap(ASHinq_top_window(RIPinq_window(rgn)),x,y,ss->window,&nx,&ny);
      if (ss->type == SCROLL_TYPE_H) ny = nx;
      ny = ny - (ss->rby - ss->rty)/2;
      v = ((ny - ss->ty)*(ss->bottom - ss->top+1)+(ss->by - ss->ty)/2)/
	 (ss->by - ss->ty+1) + ss->top;
    }
   else if (btn & RIP_BTN_DOWN) {
      tw = BIOinq_top_window();
      wd = ASHinq_user_data(tw);
      ASHset_user_data(tw,&sdata);
      if (ss->btn != NULL) STEMbtn_set(ss->btn,TRUE);

      sdata.x = x;
      sdata.y = y;
      sdata.ss = ss;

      w = RIPgrab_window(NULL);
      cfg = BIOset_cursor(NULL,TRUE);
      crsr = BIOset_cursor_standard(NULL,ASH_CURSOR_ARROWS);
      fg = RIPtrack(move_rtn,1,NULL,FALSE);
      BIOset_cursor_pattern(NULL,crsr);
      BIOset_cursor(NULL,cfg);
      RIPgrab_window(w);
      if (ss->btn != NULL) {
	 if (ss->btn->bg == ss->btn->gray) draw_scroll_body(ss,FALSE);
	 else STEMbtn_set(ss->btn,FALSE);
       };

      ASHset_user_data(tw,wd);

      if (!fg) return FALSE;

      v = ((ss->rty - ss->ty)*(ss->bottom - ss->top+1)+(ss->by - ss->ty)/2)/
	 (ss->by - ss->ty+1) + ss->top;
    }
   else return FALSE;

   if (ss->call_rtn != NULL) {
      (*(ss->call_rtn))(ss->user_data,0,v);
    };

   return TRUE;
};





static Integer
move_rtn(x,y,n,type,max)
   Integer x,y;
   Integer n;
   RIP_TRANS type;
   Integer max;
{
   register Integer dy,dx;
   register STEM_SCROLL ss;
   register SCROLL_DATA * sdata;
   Boolean chng;

   DTRACE("move_rtn %d %d %d %d %d",x,y,n,type,max);

   sdata = (SCROLL_DATA *) ASHinq_user_data(BIOinq_top_window());

   ss = sdata->ss;

   dy = 0;
   dx = 0;
   switch (type) {
      case RIP_TRANS_NONE :
	 sdata->lasty = sdata->y;
	 sdata->lastx = sdata->x;
	 break;
      case RIP_TRANS_MOVE :
	 dy = (y-sdata->lasty)*ss->wdy;
	 sdata->lasty = y;
	 dx = (x-sdata->lastx)*ss->wdx;
	 sdata->lastx = x;
	 break;
      case RIP_TRANS_UP :
      case RIP_TRANS_DOWN :
	 break;
      default :
	 return FALSE;
    };

   ASHbatch_mode(TRUE);

   chng = FALSE;
   if (dy != 0 && ss->type != SCROLL_TYPE_H) {
      if (ss->ty < ss->by) {
	 if (ss->rty+dy < ss->ty) dy = ss->ty - ss->rty;
	 if (ss->rby+dy > ss->by) dy = ss->by - ss->rby;
       }
      else {
	 if (ss->rty+dy > ss->ty) dy = ss->ty - ss->rty;
	 if (ss->rby+dy < ss->by) dy = ss->by - ss->rby;
       };
      if (dy != 0) chng = TRUE;
      ss->rby += dy;
      ss->rty += dy;
    }
   else if (dx != 0 && ss->type == SCROLL_TYPE_H) {
      if (ss->ty < ss->by) {
	 if (ss->rty+dx < ss->ty) dx = ss->ty - ss->rty;
	 if (ss->rby+dx > ss->by) dx = ss->by - ss->rby;
       }
      else {
	 if (ss->rty+dx > ss->ty) dx = ss->ty - ss->rty;
	 if (ss->rby+dx < ss->by) dx = ss->by - ss->rby;
       };
      if (dx != 0) chng = TRUE;
      ss->rby += dx;
      ss->rty += dx;
    };

   if (chng) draw_scroll_body(ss,TRUE);

   ASHbatch_mode(FALSE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	free_scroll -- free scroll bar when window disappears		*/
/*									*/
/************************************************************************/


static void
free_scroll(ss)
   STEM_SCROLL ss;
{
   DTRACE("free_scroll 0x%x",ss);

   free(ss);
};





/************************************************************************/
/*									*/
/*	layout_scroll -- layout a scroll bar				*/
/*									*/
/************************************************************************/


static void
layout_scroll(ss)
   STEM_SCROLL ss;
{
   register Integer x;
   Integer lx,by,rx,ty;
   register Integer dx,dy;
   String sd1,sd2,sd3;
   String sd4,sd5;
   Character buf1[128],buf2[128],buf3[128];

   DTRACE("layout_scroll 0x%x",ss);

   ASHbatch_mode(TRUE);

   if (ss->left_win != NULL) ASHremove(ss->left_win);
   if (ss->right_win != NULL) ASHremove(ss->right_win);
   if (ss->center_region != NULL) RIPremove_region(ss->center_region);

   ASHinq_size(ss->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);
   ss->wdx = dx;
   ss->wdy = dy;

   if (abs(rx-lx) < MIN_WIDTH_3 && abs(ty-by) >= MIN_SPACE_VH) {
      ss->type = SCROLL_TYPE_V;
      ss->clx = lx;
      ss->crx = rx;
      ss->cty = ty+END_SPACE*dy;
      ss->cby = by-END_SPACE*dy;
      ss->ty = ss->cty;
      ss->by = ss->cby;
      ss->left_win = ASHcreate(ss->window,lx,ss->cty-ss->wdy,
				  lx,ss->cty-ss->wdy,rx,ty,0,
				  ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      ss->right_win = ASHcreate(ss->window,lx,by,
				   lx,by,rx,ss->cby+ss->wdy,0,
				   ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      sd1 = "ScrollUp";
      sd2 = "ScrollUpDown";
      sd3 = "ScrollDown";
      sd4 = "up";
      sd5 = "down";
    }
   else if (abs(rx-lx) > abs(ty-by) && abs(rx-lx) >= MIN_SPACE_VH) {
      ss->type = SCROLL_TYPE_H;
      ss->clx = lx+END_SPACE*dx;
      ss->crx = rx-END_SPACE*dx;
      ss->cty = ty;
      ss->cby = by;
      ss->ty = ss->clx;
      ss->by = ss->crx;
      ss->left_win = ASHcreate(ss->window,lx,by,
				  lx,by,ss->clx-ss->wdx,ty,0,
				  ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      ss->right_win = ASHcreate(ss->window,ss->crx+ss->wdx,by,
				   ss->crx+ss->wdx,by,rx,ty,0,
				   ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      sd1 = "ScrollLeft";
      sd2 = "ScrollLeftRight";
      sd3 = "ScrollRight";
      sd4 = "left";
      sd5 = "right";
    }
   else {
      ss->type = SCROLL_TYPE_3;
      x = abs(rx-lx)/3;
      ss->by = by;
      ss->ty = ty;
      ss->clx = lx+x*dx;
      ss->crx = rx-x*dx;
      ss->cty = ty;
      ss->cby = by;
      ss->left_win = ASHcreate(ss->window,lx,by,lx,by,ss->clx-ss->wdx,ty,0,
				  ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      ss->right_win = ASHcreate(ss->window,ss->crx+ss->wdx,by,
				   ss->crx+ss->wdx,by,rx,ty,0,
				   ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR);
      sd1 = "ScrollUp3";
      sd2 = "ScrollUpDown3";
      sd3 = "ScrollDown3";
      sd4 = "up";
      sd5 = "down";
    };

   if (ss->left_win != NULL) {
      BIOnew_input_window(ss->left_win);
      BIOset_window_thread(ss->left_win,ss->thread);
      BIOset_cursor_standard(ss->left_win,ASH_CURSOR_HALF_ARROW_L);
      ss->left_region = RIPdefine_region(ss->left_win,0,0,0,0,
					    CH_EVENTS,BTN_EVENTS,scroll_btn,FALSE);
      ASHset_refresh(ss->left_win,scroll_refresh_rgn);
      RIPset_data(ss->left_region,ss);
      RIPset_region_id(ss->left_region,sd1);
      ASHset_menu_data(ss->left_win,ss);
      sprintf(buf1,"Scroll %s one line",sd4);
      sprintf(buf2,"Scroll %s partail page",sd4);
      sprintf(buf3,"Scroll %s full page",sd4);
      HELPmouse_register(ss->left_win,buf1,buf2,buf3);
    };

   if (ss->right_win != NULL) {
      BIOnew_input_window(ss->right_win);
      BIOset_window_thread(ss->right_win,ss->thread);
      BIOset_cursor_standard(ss->right_win,ASH_CURSOR_HALF_ARROW_R);
      ss->right_region = RIPdefine_region(ss->right_win,0,0,0,0,
					     CH_EVENTS,BTN_EVENTS,scroll_btn,FALSE);
      ASHset_refresh(ss->right_win,scroll_refresh_rgn);
      RIPset_data(ss->right_region,ss);
      RIPset_region_id(ss->right_region,sd3);
      ASHset_menu_data(ss->right_win,ss);
      sprintf(buf1,"Scroll %s one line",sd5);
      sprintf(buf2,"Scroll %s partail page",sd5);
      sprintf(buf3,"Scroll %s full page",sd5);
      HELPmouse_register(ss->right_win,buf1,buf2,buf3);
    };

   BIOnew_input_window(ss->window);
   BIOset_window_thread(ss->window,ss->thread);
   BIOset_cursor_standard(ss->window,ASH_CURSOR_SMALL_XHAIRS);
   ASHset_window_id(ss->window,"scroll");

   ss->center_region = RIPdefine_region(ss->window,
					   ss->clx,ss->cby,ss->crx,ss->cty,
					   CH_EVENTS,CEN_EVENTS,
					   scroll_move_btn,FALSE);
   RIPset_data(ss->center_region,ss);
   RIPset_region_id(ss->center_region,sd2);

   HELPmouse_register(ss->window,"Move visible region","^","^");

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	draw_scroll -- draw the scroll bar				*/
/*									*/
/************************************************************************/


static void
draw_scroll(ss)
   STEM_SCROLL ss;
{
   register Integer q;
   Integer d;
   Integer lx,by,rx,ty;

   DTRACE("draw_scroll 0x%x",ss);

   if (ss->last < ss->first) return;

   ASHbatch_mode(TRUE);

   ASHwindow_draw_thru(ss->window,TRUE);
   ASHinq_size(ss->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   q = (ss->bottom - ss->top +1);
   if (q <= 0) q = 1;

   ss->rty = ss->ty + ((ss->first - ss->top)*(ss->by - ss->ty))/q;
   ss->rby = ss->ty + ((ss->last - ss->top + 1)*(ss->by - ss->ty))/q;
   while (abs(ss->rty - ss->rby) < min_size) {
      d = (ss->ty < ss->by ? 1 : -1);
      ss->rty -= d;
      ss->rby += d;
    };

   ASHclear_box(ss->window,lx,by,rx,ty);

   draw_scroll_body(ss,FALSE);

   if (ss->type == SCROLL_TYPE_H) {
      draw_triangle(ss->left_win,0,ss);
      draw_triangle(ss->right_win,2,ss);
    }
   else if (ss->type == SCROLL_TYPE_V) {
      draw_triangle(ss->left_win,1,ss);
      draw_triangle(ss->right_win,3,ss);
    };

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	draw_update_scroll -- draw the scroll bar			*/
/*									*/
/************************************************************************/


static void
draw_update_scroll(ss)
   STEM_SCROLL ss;
{
   register Integer y0,y1;
   register Integer q;
   Integer d;

   DTRACE("draw_update_scroll 0x%x",ss);

   if (ss->last < ss->first) return;

   q = (ss->bottom - ss->top +1);
   if (q <= 0) q = 1;

   y0 = ss->rty;
   y1 = ss->rby;
   ss->rty = ss->ty + ((ss->first - ss->top)*(ss->by - ss->ty))/q;
   ss->rby = ss->ty + ((ss->last - ss->top + 1)*(ss->by - ss->ty))/q;
   while (abs(ss->rty - ss->rby) < min_size) {
      d = (ss->ty < ss->by ? 1 : -1);
      ss->rty -= d;
      ss->rby += d;
    };

   if (y0 == ss->rty && y1 == ss->rby) return;

   ASHbatch_mode(TRUE);

   draw_scroll_body(ss,FALSE);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	draw_triangle -- draw triangle in scroll bar end region 	*/
/*									*/
/************************************************************************/


static void
draw_triangle(w,dir,ss)
   ASH_WINDOW w;
   Integer dir;
   STEM_SCROLL ss;
{
   Integer lx,by,rx,ty;
   Integer x[3],y[3];
   Integer dx,dy;
   Integer fl;
   ASH_COLOR bc;

   DTRACE("draw_triangle 0x%x %d",w,dir);

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   if (ss->bar_color >= 0) bc = ASHbackground_color(w,ss->bar_color);
   ASHclear_box(w,lx,by,rx,ty);

   dx = (lx < rx ? 1 : -1);
   dy = (by < ty ? 1 : -1);

   switch (dir) {
      case 0 :
	 x[0] = lx+dx;
	 y[0] = (by+ty)/2;
	 x[1] = x[2] = rx-dx;
	 y[1] = ty-dy;
	 y[2] = by+dy;
	 break;
      case 1 :
	 x[0] = (lx+rx)/2;
	 y[0] = ty-dy;
	 x[1] = lx+dx;
	 y[1] = y[2] = by+dy;
	 x[2] = rx-dx;
	 break;
      case 2 :
	 x[0] = rx-dx;
	 y[0] = (by+ty)/2;
	 x[1] = x[2] = lx+dx;
	 y[1] = ty-dy;
	 y[2] = by+dy;
	 break;
      case 3 :
	 x[0] = (lx+rx)/2;
	 y[0] = by+dy;
	 x[1] = lx+dx;
	 y[1] = y[2] = ty-dy;
	 x[2] = rx-dx;
	 break;
    };

   fl = ASHfill(w,1);
   ASHpolygon(w,3,x,y);
   ASHfill(w,fl);
   if (ss->bar_color >= 0) ASHbackground_color(w,bc);
};





/************************************************************************/
/*									*/
/*	draw_arrow -- draw arrow up or down for 3-sided scroll bar	*/
/*									*/
/************************************************************************/


#define ARROW_WIDTH		6
#define ARROW_LINES		12


static void
draw_arrow(w,dir,ss)
   ASH_WINDOW w;
   Integer dir;
   STEM_SCROLL ss;
{
   Integer lx,by,rx,ty;
   Integer x[3],y[3];
   Integer dy,cx,cy;
   Integer fl;
   ASH_COLOR bc;

   DTRACE("draw_arrow 0x%x %d",w,dir);

   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   if (ss->bar_color >= 0) bc = ASHbackground_color(w,ss->bar_color);
   ASHclear_box(w,lx,by,rx,ty);

   dy = (by < ty ? 1 : -1);

   cx = (lx+rx)/2;
   cy = (by+ty)/2;

   if (dir != 0) dy = -dy;

   x[0] = cx - ARROW_WIDTH/2;
   y[0] = cy + dy*(ARROW_LINES-ARROW_WIDTH)/2;
   x[1] = cx + ARROW_WIDTH/2;
   y[1] = y[0];
   x[2] = cx;
   y[2] = cy + dy*(ARROW_LINES+ARROW_WIDTH)/2;
   fl = ASHfill(w,1);
   ASHpolygon(w,3,x,y);

   ASHline(w,cx-1,cy-dy*(ARROW_LINES+ARROW_WIDTH)/2,cx-1,cy+dy*(ARROW_LINES-ARROW_WIDTH)/2);
   ASHline(w,cx+1,cy-dy*(ARROW_LINES+ARROW_WIDTH)/2,cx+1,cy+dy*(ARROW_LINES-ARROW_WIDTH)/2);

   ASHfill(w,fl);
   if (ss->bar_color >= 0) ASHbackground_color(w,bc);
};






/************************************************************************/
/*									*/
/*	draw_scroll_body -- draw/update body of scroll bar		*/
/*									*/
/************************************************************************/


static void
draw_scroll_body(ss,fg)
   STEM_SCROLL ss;
   Boolean fg;
{
   Integer blx,bby,brx,bty;
   ASH_FONT ft;
   STEM_BTN sb;
   ASH_COLOR bc;

   if (ss->type != SCROLL_TYPE_H) {
      blx = ss->clx + (1+bar_space)*ss->wdx;
      bby = ss->rby - ss->wdy;
      brx = ss->crx - (1+bar_space)*ss->wdx;
      bty = ss->rty + ss->wdy;
    }
   else {
      blx = ss->rty + ss->wdx;
      bby = ss->cby - (1+bar_space)*ss->wdy;
      brx = ss->rby - ss->wdx;
      bty = ss->cty + (1+bar_space)*ss->wdy;
    };

   if (ss->btn != NULL) STEMbtn_remove(ss->btn);
   ft = ASHfont(ss->window,ASH_ICON_FONT);
   if (ss->bar_color >= 0) bc = ASHbackground_color(ss->window,ss->bar_color);

   sb = STEMbtn_define(ss->window,blx,bby,brx,bty,"SCROLL_HANDLE",
			  STEM_BTN_FULL_SIZE|STEM_BTN_NO_DRAW,
			  NULL,NULL);
   STEMbtn_set(sb,fg);
   ss->btn = sb;
   STEMbtn_background(sb,ss->clx+ss->wdx,ss->cby-ss->wdy,
			 ss->crx-ss->wdx,ss->cty+ss->wdy);
   STEMbtn_draw(sb);
   if (sb->bg == sb->gray && !fg) {
      ASHbox(ss->window,blx,bby,brx,bty);
    };
   ASHfont(ss->window,ft);

   ASHbox(ss->window,ss->clx,ss->cby,ss->crx,ss->cty);

   if (ss->bar_color >= 0) ASHbackground_color(ss->window,bc);
};





/* end of stemscroll.c */
