/************************************************************************/
/*									*/
/*		stempop.c						*/
/*									*/
/*	Pop up menu module for STEM menu package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define CH_EVENTS	RIP_NO_CHARS
#define BTN_EVENT_DOWN_ONLY	(~RIP_BTN_UP)
#define BTN_EVENT_UP_ONLY	(~RIP_BTN_DOWN)
#define BTN_DOWN	RIP_BTN_DOWN
#define BTN_UP		RIP_BTN_UP
#define BTN_NONE	RIP_BTN_NONE

#define TRACK_CURSOR	ASH_CURSOR_SMALL_BULLSEYE
#define ARROW_FONT	"fixed",15,ASH_FONT_ROMAN
#define LEFT_ARROW	"<"
#define RIGHT_ARROW	">"

#define HALF_TONE_PAT	2

#define MAX_PANEL	16
#define MAX_BUTTON	16
#define MAX_BUTTON_LENGTH	1024

#define TTL_X_PAD	10
#define TTL_Y_PAD	10
#define BTN_X_PAD	20
#define BTN_Y_PAD	10
#define ARROW_X_SIZE	16
#define ARROW_X_PAD	32
#define ARROW_Y_MIN	20
#define ARROW_X_OFF	3
#define ARROW_Y_OFF	5





/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef  struct _STEM_POP_PANEL * STEM_PAN;
typedef struct _STEM_POP_MENU * STEM_PUM;



typedef struct _STEM_POP_BTN {
   String	name;
   Boolean	enable;
   Boolean	select;
   Function_Ptr routine;
   String	invoke;
   RIP_REGION	invrgn;
   Integer	xsize;
   Integer	ysize;
} STEM_POP_BTN;


typedef struct _STEM_POP_PANEL {
   String	name;
   Integer	ttlsize;
   Integer	xsize,ysize;
   Integer	xoff;
   Integer	numbtn;
   STEM_POP_BTN btn[MAX_BUTTON];
} STEM_POP_PANEL;


typedef struct _STEM_POP_MENU {
   String	name;
   ASH_WINDOW	window;
   Integer	data;
   Integer	mask;
   Function_Ptr routine;
   STEM_PUM	next;
   RIP_REGION	region;
   ASH_WINDOW	menuwin;
   Integer	xsize,ysize;
   Boolean	enable;
   Integer	ttlfont;
   Integer	btnfont;
   Integer	curpan;
   Integer	btn_picked;
   Boolean	outside;
   Integer	numpanels;
   STEM_PAN	panel[MAX_PANEL];
} STEM_POP_MENU;



/************************************************************************/
/*									*/
/*	Local Variables 						*/
/*									*/
/************************************************************************/

static int	arrow_font = -1;






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 	pop_top_control();
static	void		free_menu();
static	void		free_panel();
static	void		free_button();
static	void		size_menu();
static	void		resize_menu();
static	void		size_panel();
static	void		resize_panel();
static	void		size_button();
static	STEM_PUM	find_menu();
static	STEM_PAN	find_panel();
static	Integer 	lookup_panel();
static	STEM_POP_BTN *	find_button();
static	STEM_POP_BTN *	lookup_button();
static	void		enable_menu();
static	void		disable_menu();
static	void		disable_panel();
static	void		enable_panel();
static	void		enable_button();
static	void		disable_button();
static	Integer 	pop_up_menu();
static	Integer 	invoke_btn();
static	void		draw_pop_panel();
static	void		draw_title();
static	void		draw_button();
static	void		top_panel();
static	Integer 	pop_track();
static	Integer 	find_pick();
static	void		track_move();
static	Boolean 	inside();


/************************************************************************/
/*									*/
/*	STEM_pop_init -- module initialization				*/
/*									*/
/************************************************************************/


void
STEM_pop_init()
{
   ITRACE("STEM_pop_init");
};





/************************************************************************/
/*									*/
/*	STEMpop_define -- define a new pop-up menu			*/
/*									*/
/************************************************************************/


void
STEMpop_define(window,userdata,btnmask,title,btns,rtn)
   ASH_WINDOW window;
   Integer userdata;
   Integer btnmask;
   String  title;
   STEM_POP_DATA btns[];
   Function_Ptr rtn;
{
   register STEM_PUM p;
   register STEM_PAN pan;
   register STEM_POP_BTN * bp;

   register Integer i;

   TRACE("STEMpop_define 0x%x %d %d %s 0x%x ...",window,userdata,btnmask,title,
	    rtn);

   PROTECT;
   if (arrow_font < 0) arrow_font = ASHfont_id(ARROW_FONT);
   UNPROTECT;

   p = (STEM_PUM) calloc(1,sizeof(STEM_POP_MENU));
   p->name = SALLOC(title);
   p->window = window;
   p->data = userdata;
   p->mask = (btnmask & BTN_EVENT_DOWN_ONLY) | BTN_DOWN;
   p->routine = rtn;
   p->region = NULL;
   p->xsize = p->ysize = 0;
   p->enable = TRUE;
   p->ttlfont = ASH_DEFAULT_FONT;
   p->btnfont = ASH_DEFAULT_FONT;
   p->curpan = p->btn_picked = -1;
   p->outside = FALSE;
   p->numpanels = 0;
   pan = NULL;

   i = 0;
   while (p->numpanels < MAX_PANEL && btns[i].state == STEM_POP_STATE_PANEL)
      {
      pan = (STEM_PAN) calloc(1,sizeof(STEM_POP_PANEL));
      p->panel[p->numpanels++] = pan;
      pan->name = SALLOC(btns[i].name);
      pan->ttlsize = pan->xsize = pan->ysize = 0;
      pan->xoff = 0;
      pan->numbtn = 0;
      ++i;

      while (pan->numbtn < MAX_BUTTON && btns[i].name != NULL &&
		btns[i].state != STEM_POP_STATE_END &&
		btns[i].state != STEM_POP_STATE_PANEL)
	 {
	 bp = &pan->btn[pan->numbtn++];
	 bp->name = SALLOC(btns[i].name);
	 bp->routine = btns[i].routine;
	 bp->invoke = SALLOC(btns[i].invoke);
	 bp->invrgn = NULL;
	 if (btns[i].state == STEM_POP_STATE_BUTTON ||
		btns[i].state == STEM_POP_STATE_SELECT)
	    bp->enable = TRUE;
	 else
	    bp->enable = FALSE;
	 if (btns[i].state == STEM_POP_STATE_SELECT ||
		btns[i].state == STEM_POP_STATE_DIS_SEL)
	    bp->select = TRUE;
	 else
	    bp->select = FALSE;
	 bp->xsize = 0;
	 bp->ysize = 0;
	 ++i;
      }
   }

   PROTECT;
   p->next = (STEM_PUM) ASHinq_menu_data(window);
   ASHset_menu_data(window,p);
   UNPROTECT;

   if (p->next == NULL)
      ASHset_control(window,pop_top_control);

   size_menu(p);
   enable_menu(p);

};





/************************************************************************/
/*									*/
/*	STEMpop_enable - enable/disable a pop-up menu			*/
/*									*/
/************************************************************************/


void
STEMpop_enable(window,menu,flag,mask)
   ASH_WINDOW window;
   String menu;
   Boolean flag;
   Integer mask;
{
   register STEM_PUM p;

   TRACE("STEMpop_enable 0x%x %s %d 0x%x",window,menu,flag,mask);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   if (mask != NULL)
      p->mask = (mask & BTN_EVENT_DOWN_ONLY) | BTN_DOWN;

   p->enable = flag;
   if (flag)
      enable_menu(p);
   else
      disable_menu(p);
};






/************************************************************************/
/*									*/
/*	STEMpop_remove -- explicitly remove a pop up menu		*/
/*									*/
/************************************************************************/


void
STEMpop_remove(window,menu)
   ASH_WINDOW window;
   String menu;
{
   register STEM_PUM p, prev, curr;

   TRACE("STEMpop_remove 0x%x %s",window,menu);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   p->enable = FALSE;
   disable_menu(p);

   prev = NULL;
   PROTECT;
   for (curr = (STEM_PUM) ASHinq_menu_data(window);
	curr != NULL && curr != p;
	curr = curr->next)
      prev = curr;
   if (curr != NULL)
      if (prev != NULL)
	 prev->next = curr->next;
      else
	 ASHset_menu_data(window,curr->next);
   UNPROTECT;

   free_menu(p);
};




/************************************************************************/
/*									*/
/*	STEMpop_panel_add -- add a new panel to a menu			*/
/*	STEMpop_panel_remove -- remove a panel from a menu		*/
/*	STEMpop_panel_select -- set the current panel			*/
/*									*/
/************************************************************************/

void
STEMpop_panel_add(window,menu,panel,before)
   ASH_WINDOW window;
   String menu;
   String panel;
   String before;
{
   STEM_PUM mp;
   STEM_PAN pan;
   Integer i,j;

   TRACE("STEMpop_panel_add 0x%x %s %s %s",window,menu,panel,before);

   if ((mp=find_menu(window,menu)) == NULL)
      return;

   if (mp->numpanels >= MAX_PANEL) {
      ERROR("Too many panels.");
      return;
   }

   pan = (STEM_PAN) calloc(1,sizeof(STEM_POP_PANEL));
   pan->name = SALLOC(panel);
   pan->ttlsize = pan->xsize = pan->ysize = 0;
   pan->xoff = 0;
   pan->numbtn = 0;

   if (STREMPTY(before))
      i = mp->numpanels;
   else
      if ((i=lookup_panel(mp,before)) < 0)
	     i = mp->numpanels;

   for (j=mp->numpanels; j>i; j--)
      mp->panel[j] = mp->panel[j-1];

   mp->panel[i] = pan;
   mp->numpanels++;

   size_panel(mp,pan);
   resize_menu(mp);
}



void
STEMpop_panel_remove(window,menu,panel)
   ASH_WINDOW window;
   String menu;
   String panel;
{
   STEM_PUM mp;
   STEM_PAN pan;
   Integer i;

   TRACE("STEMpop_panel_remove 0x%x %s %s",window,menu,panel);

   if ((mp=find_menu(window,menu)) == NULL)
      return;

   if ((pan=find_panel(mp,panel)) == NULL)
      return;

   for (i=lookup_panel(mp,pan->name); i<mp->numpanels-1; i++)
      mp->panel[i] = mp->panel[i+1];

   mp->numpanels--;
   disable_panel(mp,pan);
   free_panel(pan);

   resize_menu(mp);
}



void
STEMpop_panel_select(window,menu,panel)
   ASH_WINDOW window;
   String menu;
   String panel;
{
   STEM_PUM mp;
   STEM_PAN pan;

   TRACE("STEMpop_panel_select 0x%x %s %s",window,menu,panel);

   if ((mp=find_menu(window,menu)) == NULL)
      return;

   if ((pan=find_panel(mp,panel)) == NULL)
      return;

   mp->curpan = lookup_panel(mp,pan->name);
}





/************************************************************************/
/*									*/
/*	STEMpop_btn_enable -- set button enable/disable 		*/
/*	STEMpop_btn_select -- set button select/deselect		*/
/*									*/
/************************************************************************/


void
STEMpop_btn_enable(window,menu,panel,button,flag)
   ASH_WINDOW window;
   String menu;
   String panel;
   String button;
   Boolean flag;
{
   register STEM_PUM p;
   register STEM_POP_BTN *b;

   TRACE("STEMpop_btn_enable 0x%x %s %s %s %d",window,menu,panel,button,flag);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   if ((b=find_button(p,panel,button)) == NULL)
      return;

   b->enable = flag;

   if (flag && p->enable)
      enable_button(p,b);
   else
      disable_button(p,b);
};






void
STEMpop_btn_select(window,menu,panel,button,flag)
   ASH_WINDOW window;
   String menu;
   String panel;
   String button;
   Boolean flag;
{
   register STEM_PUM p;
   register STEM_POP_BTN *b;

   TRACE("STEMpop_btn_select 0x%x %s %s %s %d",window,menu,panel,button,flag);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   if ((b=find_button(p,panel,button)) == NULL)
      return;

   b->select = flag;

};



/************************************************************************/
/*									*/
/*	STEMpop_btn_remove -- remove button from menu			*/
/*	STEMpop_btn_add -- add button to menu				*/
/*									*/
/************************************************************************/


void
STEMpop_btn_remove(window,menu,panel,button)
   ASH_WINDOW window;
   String menu;
   String panel;
   String button;
{
   register STEM_PUM p;
   register STEM_PAN pan;
   register int fg,i;

   TRACE("STEMpop_btn_remove 0x%x %s %s %s",window,menu,panel,button);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   if ((pan=find_panel(p,panel)) == NULL)
      return;

   fg = FALSE;
   for (i = 0; i < pan->numbtn; ++i) {
      if (STREQL(pan->btn[i].name,button)) {
	 fg = TRUE;
	 disable_button(p,&pan->btn[i]);
	 free_button(&pan->btn[i]);
      }
      else if (fg)
	 pan->btn[i-1] = pan->btn[i];
   }

   if (fg) {
      pan->numbtn--;
      resize_panel(p,pan);
      resize_menu(p);
   }
};





void
STEMpop_btn_add(window,menu,panel,button,routine,invoke)
   ASH_WINDOW window;
   String menu;
   String panel;
   String button;
   Function_Ptr routine;
   String invoke;
{
   register STEM_PUM p;
   register STEM_PAN pan;
   register STEM_POP_BTN *b;

   TRACE("STEMpop_btn_add 0x%x %s %s 0x%x %s",window,menu,button,routine,invoke);

   if ((p=find_menu(window,menu)) == NULL)
      return;

   if ((pan=find_panel(p,panel)) == NULL)
      return;

   b = lookup_button(pan,button);

   if (b == NULL) {
      if (pan->numbtn < MAX_BUTTON)
	 b = &pan->btn[pan->numbtn++];
      else
	 return;
   }
   else {
      disable_button(p,b);
      free_button(b);
   }

   b->name = SALLOC(button);
   b->routine = routine;
   b->invoke = SALLOC(invoke);
   b->invrgn = NULL;
   b->select = FALSE;
   b->enable = TRUE;

   size_button(p,b);
   resize_panel(p,pan);
   resize_menu(p);

   if (b->enable && p->enable)
      enable_button(p,b);

};





/************************************************************************/
/*									*/
/*	STEMpop_inq_size	- return # of panels/buttons in menu	*/
/*	STEMpop_inq_buttons	- return panels/buttons from menu	*/
/*									*/
/************************************************************************/

Integer
STEMpop_inq_size(window,menu)
   ASH_WINDOW window;
   String menu;
{
   register int i,n;
   register STEM_PUM p;

   TRACE("STEMpop_inq_size 0x%x %s",window,menu);

   if ((p=find_menu(window,menu)) == NULL)
      return 0;

   n = 0;
   for (i=0; i<p->numpanels; i++)
      n += 1 + p->panel[i]->numbtn;

   return n;
}



Integer
STEMpop_inq_buttons(window,menu,maxbtn,btns)
   ASH_WINDOW window;
   String menu;
   Integer maxbtn;
   STEM_POP_DATA btns[];
{
   register STEM_PUM p;
   register STEM_PAN pan;
   register int i, j, n;

   TRACE("STEMpop_inq_buttons 0x%x %s %d ...",window,menu,maxbtn);

   if ((p=find_menu(window,menu)) == NULL)
      return 0;

   n = 0;

   for (i=0; i<p->numpanels && n<maxbtn; i++) {
      pan = p->panel[i];
      btns[n].name = pan->name;
      btns[n].routine = NULL;
      btns[n].invoke = NULL;
      btns[n].state = STEM_POP_STATE_PANEL;
      n++;
      for (j=0; j<pan->numbtn && n<maxbtn; j++) {
	 btns[n].name = pan->btn[j].name;
	 btns[n].routine = pan->btn[j].routine;
	 btns[n].invoke = pan->btn[j].invoke;
	 if (pan->btn[j].enable)
	    btns[n].state = (pan->btn[j].select)
			       ? STEM_POP_STATE_SELECT : STEM_POP_STATE_BUTTON;
	 else
	    btns[n].state = (pan->btn[j].select)
			       ? STEM_POP_STATE_DIS_SEL : STEM_POP_STATE_DISABLE;
	 n++;
      }
   }

   return n;

}







/************************************************************************/
/*									*/
/*	STEMpop_fonts -- set menu and button fonts			*/
/*									*/
/************************************************************************/


void
STEMpop_fonts(window,menu,ttl,btn)
   ASH_WINDOW window;
   String menu;
   Integer ttl;
   Integer btn;
{
   register STEM_PUM p;

   TRACE("STEMpop_fonts 0x%x %s %d %d",window,menu,ttl,btn);

   if ((p=find_menu(window,menu)) == NULL) return;

   if (ttl > 0) p->ttlfont = ttl;
   if (btn > 0) p->btnfont = btn;

   size_menu(p);
};





/************************************************************************/
/*									*/
/*	pop_top_control -- handle control for pop up menus		*/
/*	free_menu	-- free up a menu				*/
/*	free_panel	-- free up a panel
/*	free_button	-- free up a button				*/
/*									*/
/************************************************************************/


static Integer
pop_top_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register STEM_PUM p;

   ITRACE("pop_top_control %s 0x%x",msg,w);

   if (STREQL(msg,"ASH$REMOVE")) {
      for (p = (STEM_PUM) ASHinq_menu_data(w); p != NULL; p = p->next) {
	 disable_menu(p);
	 free_menu(p);
      };
   };

   return ASH_CONTROL_REJECT;
};



static void
free_menu(mp)
   STEM_PUM mp;
{
   register Integer i;

   ITRACE("free_menu 0x%x",mp);

   for (i=0; i<mp->numpanels; i++)
      free_panel(mp->panel[i]);

   SFREE(mp->name);
   free(mp);
}


static void
free_panel(pan)
   STEM_PAN pan;
{
   register Integer i;

   ITRACE("free_panel 0x%x",pan);

   for (i=0; i<pan->numbtn; i++)
      free_button(&pan->btn[i]);

   SFREE(pan->name);
   free(pan);
}


static void
free_button(bp)
   STEM_POP_BTN *bp;
{
   ITRACE("free_button 0x%x",bp);

   SFREE(bp->name);
   SFREE(bp->invoke);
}




/************************************************************************/
/*									*/
/*	size_menu - compute size of all buttons and total menu		*/
/*	resize_menu - recompute menu size				*/
/*	size_button - compute size of a button				*/
/*									*/
/************************************************************************/

static void
size_menu(mp)
   STEM_PUM mp;
{
   int i;

   ITRACE("size_menu 0x%x",mp);

   for (i=0; i<mp->numpanels; i++)
      size_panel(mp,mp->panel[i]);

   resize_menu(mp);
}



static void
resize_menu(mp)
   STEM_PUM mp;
{
   register int i;

   ITRACE("resize_menu 0x%x",mp);


   mp->xsize = 0;
   mp->ysize = 0;

   for (i=0; i<mp->numpanels; i++)
      {
      if (mp->xsize < mp->panel[i]->xsize)
	 mp->xsize = mp->panel[i]->xsize;
      if (mp->ysize < mp->panel[i]->ysize)
	 mp->ysize = mp->panel[i]->ysize;
      }
}

static void
size_panel(mp,pan)
   STEM_PUM mp;
   STEM_PAN pan;
{
   int i;

   ITRACE("size_panel 0x%x",pan);

   for (i=0; i<pan->numbtn; i++)
      size_button(mp,&pan->btn[i]);

   resize_panel(mp,pan);
}


static void
resize_panel(mp,pan)
   STEM_PUM mp;
   STEM_PAN pan;
{
   int i, xsz, ysz;

   ITRACE("resize_menu 0x%x",mp);

   ASHinq_text_extent(mp->ttlfont,pan->name,&xsz,&ysz);
   pan->xsize = MAX((xsz + TTL_X_PAD),ARROW_X_PAD);
   pan->ttlsize = ysz + TTL_Y_PAD;
   pan->ysize = 2 * pan->ttlsize;

   for (i=0; i<pan->numbtn; i++) {
      pan->ysize += pan->btn[i].ysize;
      if (pan->xsize < (pan->btn[i].xsize+ARROW_X_PAD))
	 pan->xsize = pan->btn[i].xsize + ARROW_X_PAD;
   }

   ysz = ARROW_Y_MIN + (2 * pan->ttlsize);
   if (pan->ysize < ysz)
      pan->ysize = ysz;

}




static void
size_button(mp,bp)
   STEM_PUM mp;
   STEM_POP_BTN *bp;
{
   int xsz,ysz;

   ITRACE("size_button 0x%x 0x%x",mp,bp);

   ASHinq_text_extent(mp->btnfont,bp->name,&xsz,&ysz);
   bp->xsize = xsz + BTN_X_PAD;
   bp->ysize = ysz + BTN_Y_PAD;
}






/************************************************************************/
/*									*/
/*	find_menu	- find a given menu for a window		*/
/*	find_panel	- find a panel of a menu			*/
/*	lookup_panel	- return index of panel in menu 		*/
/*	find_button	- find a button in a menu			*/
/*	lookup_button	- find a button in a panel			*/
/*									*/
/************************************************************************/

static STEM_PUM
find_menu(window,name)
   ASH_WINDOW window;
   String name;
{
   STEM_PUM mp;

   ITRACE("find_menu 0x%x %s",window,name);

   for (mp = (STEM_PUM) ASHinq_menu_data(window); mp != NULL; mp = mp->next)
      if (STREQL(mp->name,name))
	 break;

   if (mp == NULL)
      ERROR("Bad pop up menu name");

   return mp;
}


static STEM_PAN
find_panel(mp,panel)
   STEM_PUM mp;
   String panel;
{
   register int i;

   ITRACE("find_panel 0x%x %s",mp,panel);

   if (STREMPTY(panel))
      return (mp->numpanels>0 ? mp->panel[0] : NULL);

   i = lookup_panel(mp,panel);

   if (i<0) {
      ERROR("Bad panel name");
      return NULL;
   }
   else
      return mp->panel[i];
}



static Integer
lookup_panel(mp,panel)
   STEM_PUM mp;
   String panel;
{
   register int i;

   ITRACE("lookup_panel 0x%x %s",mp,panel);

   for (i=0; i<mp->numpanels; i++)
      if (STREQL(mp->panel[i]->name,panel))
	 return i;

   return -1;
}



static STEM_POP_BTN *
find_button(mp,panel,button)
   STEM_PUM mp;
   String panel;
   String button;
{
   register STEM_PAN p;

   ITRACE("find_button 0x%x %s %s",mp,panel,button);

   if ((p=find_panel(mp,panel)) == NULL)
      return (STEM_POP_BTN *) NULL;

   return lookup_button(p,button);
}

static STEM_POP_BTN *
lookup_button(p,button)
   STEM_PAN p;
   String button;
{
   register int i;

   ITRACE("lookup_button 0x%x %s",p,button);

   for (i=0; i<p->numbtn; i++)
      if (STREQL(p->btn[i].name,button))
	 return (&p->btn[i]);

   return NULL;
}




/************************************************************************/
/*									*/
/*	disable_menu - Turn off a pop up menu				*/
/*									*/
/************************************************************************/

static void
disable_menu(mp)
   STEM_PUM mp;
{
   Integer i;

   ITRACE("disable_menu 0x%x",mp);

   if (mp->region != NULL) {
      RIPremove_region(mp->region);
      mp->region = NULL;
   }

   for (i=0; i<mp->numpanels; i++)
      disable_panel(mp,mp->panel[i]);
}




/************************************************************************/
/*									*/
/*	enable_menu - Turn on a pop up menu				*/
/*									*/
/************************************************************************/

static void
enable_menu(mp)
   STEM_PUM mp;
{
   int lx,by,rx,ty,i;

   ITRACE("enable_menu 0x%x",mp);

   if (mp->region != NULL)
      RIPremove_region(mp->region);

   ASHinq_size(mp->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   mp->region = RIPdefine_region(mp->window,lx,by,rx,ty,CH_EVENTS,mp->mask,
				    pop_up_menu,ASH_SENSE_NO_CHANGE);
   RIPset_data(mp->region,mp);

   for (i=0; i<mp->numpanels; i++)
      enable_panel(mp,mp->panel[i]);

}



/************************************************************************/
/*									*/
/*	disable_panel - Turn off all buttons in a panel 		*/
/*	enable_panel - turn on all buttons in a panel			*/
/*									*/
/************************************************************************/

static void
disable_panel(mp,p)
   STEM_PUM mp;
   STEM_PAN p;
{
   register int i;

   ITRACE("disable_panel 0x%x",p);

   for (i=0; i<p->numbtn; i++)
      disable_button(mp,&p->btn[i]);
}


static void
enable_panel(mp,p)
   STEM_PUM mp;
   STEM_PAN p;
{
   register int i;

   ITRACE("enable_panel 0x%x",p);

   for (i=0; i<p->numbtn; i++)
      if (p->btn[i].enable)
	  enable_button(mp,&p->btn[i]);

}



/************************************************************************/
/*									*/
/*	enable_button  - turn on automatic button activation		*/
/*	disable_button - turn off auto button activiation		*/
/*									*/
/************************************************************************/

static void
enable_button(mp,bp)
   STEM_PUM mp;
   STEM_POP_BTN *bp;
{
   int lx,by,rx,ty;

   ITRACE("enable_button 0x%x 0x%x",mp,bp);

   if (bp->invoke != NULL && strlen(bp->invoke) > 0 && bp->invrgn == NULL) {
      ASHinq_size(mp->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      bp->invrgn = RIPdefine_region(mp->window,lx,by,rx,ty,bp->invoke,BTN_NONE,
				       invoke_btn,ASH_SENSE_NO_CHANGE);
      RIPset_data(bp->invrgn,mp);
   }
}


static void
disable_button(mp,bp)
   STEM_PUM mp;
   STEM_POP_BTN *bp;
{
   ITRACE("disable_button 0x%x 0x%x",mp,bp);

   if (bp->invrgn != NULL) {
      RIPremove_region(bp->invrgn);
      bp->invrgn = NULL;
   }

}








/************************************************************************/
/*									*/
/*	pop_up_menu - handle rip event to call up menu			*/
/*									*/
/************************************************************************/


static Integer
pop_up_menu(fx,fy,ch,btns,rgn)
   Integer fx,fy;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   STEM_PUM mp;
   STEM_PAN pan;
   ASH_WINDOW gw,gfw;
   Integer x,y,b,lx,by,rx,ty,dx,dy,fg;
   register Boolean cfg;
   ASH_CURSOR crsr;

   ITRACE("pop_up_menu %d %d 0x%x 0x%x 0x%x",fx,fy,ch,btns,rgn);

   mp = (STEM_PUM) RIPinq_data(rgn);

   if (mp->routine != NULL) {
      fg = (*(mp->routine))(mp->data,mp->name,btns);
      if (!fg)
	 return FALSE;
   }

   if (mp->numpanels <= 0)
      return FALSE;

   ASHmap(ASHinq_top_window(mp->window),fx,fy,mp->window,&x,&y);
   ASHinq_size(mp->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dx = (lx < rx) ? 1 : -1;
   dy = (by > ty) ? 1 : -1;

   mp->menuwin = ASHcreate(mp->window,(x-(dx*mp->xsize/2)),(y+dy*mp->ysize),
			      1,mp->ysize,mp->xsize,1,
			      ASH_BORDER_NONE,
			      ASH_WINDOW_COURTEOUS|ASH_WINDOW_NOSAVE|
			      ASH_WINDOW_INDEPENDENT|
			      ASH_WINDOW_NOREDIRECT|ASH_WINDOW_NOCLEAR);
   ASHset_user_data(mp->menuwin,mp);
   ASHset_window_name(mp->menuwin,mp->name);
   ASHset_window_id(mp->menuwin,mp->name);
   gfw = ASHgrab_from(mp->window,mp->menuwin);
   BIOnew_input_window(mp->menuwin);
   while (!ASHinq_viewable(mp->menuwin)) ;
   ASHinq_size(mp->menuwin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclear_box(mp->menuwin,lx,by,rx,ty);

   b = mp->curpan;
   if (b < 0 || b >= mp->numpanels)
      b = 0;
   mp->curpan = -1;
   top_panel(mp,b);

   gw = RIPgrab_window(NULL);
   cfg = BIOset_cursor(NULL,TRUE);
   crsr = BIOset_cursor_standard(NULL,TRACK_CURSOR);

   fg = RIPtrack(pop_track,-1,mp->menuwin,FALSE);  /* always returns 0	*/

   ASHremove(mp->menuwin);
   BIOset_cursor_pattern(NULL,crsr);
   BIOset_cursor(NULL,cfg);
   RIPgrab_window(gw);
   ASHgrab_from(mp->window,gfw);

   b = mp->btn_picked;
   if (b >= 0 && mp->curpan >= 0) {
      pan = mp->panel[mp->curpan];
      if (pan->btn[b].enable && pan->btn[b].routine != NULL) {
	 fg = (*(pan->btn[b].routine))(
		     mp->data,
		     mp->name,
		     pan->name,
		     pan->btn[b].name);
      };
   };

   return fg;
};



/************************************************************************/
/*									*/
/*	invoke_btn - automatic button invocation			*/
/*									*/
/************************************************************************/

static Integer
invoke_btn(x,y,ch,btns,rgn)
   Integer x,y;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   STEM_PUM mp;
   STEM_PAN pan;
   STEM_POP_BTN *b;
   register int i, j, flag;

   ITRACE("invoke_btn %d %d %d %d 0x%x",x,y,ch,btns,rgn);

   mp = (STEM_PUM) RIPinq_data(rgn);

   flag = FALSE;

   for (j=0; j<mp->numpanels; j++) {
      pan = mp->panel[j];
      for (i=0; i<pan->numbtn; i++)
	 if (pan->btn[i].invrgn == rgn) {
	    flag = TRUE;
	    b = &pan->btn[i];
	    break;
	 };
      if (flag)
	 break;
   };

   if (flag) {
      if (b->enable && b->routine != NULL) {
	 flag = (*(b->routine))(
		     mp->data,
		     mp->name,
		     pan->name,
		     b->name);
      };
   };

   return flag;
}





/************************************************************************/
/*									*/
/*	draw_pop_panel - draw a panel					*/
/*									*/
/************************************************************************/

static void
draw_pop_panel(mp,pan)
   STEM_PUM mp;
   STEM_PAN pan;
{
   Integer i, sy1, sy2;
   Integer lx,by,rx,ty;

   ITRACE("draw_pop_panel 0x%x 0x%x",mp,pan);

   pan->xoff = (mp->xsize - pan->xsize) / 2;

   ASHbatch_mode(TRUE);
   ASHinq_size(mp->menuwin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclear_box(mp->menuwin,lx,by,rx,ty);

   ty = 0;
   draw_title(mp,pan,(pan->xoff+1),&ty,TRUE);
   sy1 = ty;
   if (pan->numbtn == 0)
      ty += ARROW_Y_MIN;
   else {
      for (i=0; i<pan->numbtn; i++)
	 draw_button(mp,pan,&pan->btn[i],(pan->xoff+1+ARROW_X_SIZE),&ty);
   }
   sy2 = ty - 1;
   draw_title(mp,pan,(pan->xoff+1),&ty,FALSE);

   ASHline(mp->menuwin,(pan->xoff+1),sy1,(pan->xoff+pan->xsize),sy1);
   ASHline(mp->menuwin,(pan->xoff+1),sy2,(pan->xoff+pan->xsize),sy2);
   ASHline(mp->menuwin,(pan->xoff+ARROW_X_SIZE),sy1,(pan->xoff+ARROW_X_SIZE),sy2);
   ASHline(mp->menuwin,(pan->xoff+1),sy1,(pan->xoff+1),sy2);
   ASHline(mp->menuwin,(pan->xoff+pan->xsize-ARROW_X_SIZE),sy1,
	      (pan->xoff+pan->xsize-ARROW_X_SIZE),sy2);
   ASHline(mp->menuwin,(pan->xoff+pan->xsize),sy1,(pan->xoff+pan->xsize),sy2);

   ASHfont(mp->menuwin,arrow_font);
   if (pan != mp->panel[0])
      ASHtext(mp->menuwin,(pan->xoff+ARROW_X_OFF),(ARROW_Y_OFF+pan->ysize/2),
		 LEFT_ARROW);
   if (pan != mp->panel[mp->numpanels-1])
      ASHtext(mp->menuwin,(pan->xoff+ARROW_X_OFF+pan->xsize-ARROW_X_SIZE),
		 (ARROW_Y_OFF+pan->ysize/2),RIGHT_ARROW);
   ASHbatch_mode(FALSE);
}


/************************************************************************/
/*									*/
/*	draw_title - display title of pop up menu			*/
/*	draw_button - display button in pop up menu			*/
/*									*/
/************************************************************************/


static void
draw_title(mp, pan, x, y, mfg)
   STEM_PUM mp;
   STEM_PAN pan;
   Integer x, *y;
   Boolean mfg;
{
   register Integer ty,rx,by;
   String s;

   ITRACE("draw_title 0x%x 0x%x %d %d",mp,pan,x,*y);

   ASHfont(mp->menuwin,mp->ttlfont);
   ty = *y;
   rx = x + pan->xsize - 1;
   by = ty + pan->ttlsize - 1;
   *y = by + 1;

   s = (mfg ? mp->name : pan->name);
   ASHcenter_text(mp->menuwin,s,x,by,rx,ty);

   ASHhilite_box(mp->menuwin,x,by,rx,ty);
};





static void
draw_button(mp,pan,bp,x,y)
   STEM_PUM mp;
   STEM_PAN pan;
   STEM_POP_BTN *bp;
   Integer x, *y;
{
   char *n,buff[MAX_BUTTON_LENGTH];
   Integer lx,rx,ty,by;

   ITRACE("draw_button 0x%x 0x%x 0x%x %d %d",mp,pan,bp,x,*y);

   ASHfont(mp->menuwin,mp->btnfont);
   lx = x;
   rx = x + bp->xsize - 1;
   ty = *y;
   by = ty + bp->ysize - 1;

   if (bp->select) {
      strcpy(buff,"*");
      strcat(buff,bp->name);
      strcat(buff,"*");
      n = buff;
      }
   else
      n = bp->name;

   ASHcenter_text(mp->menuwin,n,lx,by,rx,ty);
   *y += bp->ysize;

   rx = x + pan->xsize - (1 + 2 * ARROW_X_SIZE);
   if (bp->enable) {
      ASHsensitive_area(mp->menuwin,lx,by,rx,ty,ASH_SENSE_FLIP);
    }
   else {
      ASHstipple_box(mp->menuwin,lx,by,rx,ty);
   }
};





/************************************************************************/
/*									*/
/*	top_panel - set the current panel and bring it to the top	*/
/*									*/
/************************************************************************/


static void
top_panel(mp,p)
   STEM_PUM mp;
   Integer p;
{
   register STEM_PAN pan;

   ITRACE("top_panel 0x%x %d",mp,p);

   if (mp->curpan >= 0)
      ASHsensitive_remove_all(mp->menuwin);

   pan = mp->panel[p];
   mp->curpan = p;

   draw_pop_panel(mp,pan);
}



/************************************************************************/
/*									*/
/*	pop_track -- tracking function for button selection		*/
/*	find_pick -- determine which button was picked			*/
/*									*/
/************************************************************************/


static Integer
pop_track(x,y,n,type,max,window)
   Integer x,y;
   Integer n;
   RIP_TRANS type;
   Integer max;
   ASH_WINDOW window;
{
   register STEM_PUM mp;
   register Integer r;

   DTRACE("pop_track %d %d %d %d %d 0x%x",x,y,n,type,max,window);

   mp = (STEM_PUM) ASHinq_user_data(window);

   r = TRUE;

   switch (type) {
      case RIP_TRANS_NONE:
	 mp->outside = TRUE;
	 break;

      case RIP_TRANS_UP:
	 mp->btn_picked = find_pick(x,y,mp);
	 r = FALSE;
	 break;

      case RIP_TRANS_MOVE:
	 track_move(x,y,mp);
	 break;

      default:
	 break;
   }

   return r;
};



static Integer
find_pick(x,y,mp)
   Integer x,y;
   STEM_PUM mp;
{
   STEM_PAN pan;
   register int i,ty;

   DTRACE("find_pick %d %d 0x%x",x,y,mp);

   if (mp->curpan < 0)
      return -1;

   pan = mp->panel[mp->curpan];
   x -= pan->xoff;

   if (!(inside(x,(1+ARROW_X_SIZE),(pan->xsize-ARROW_X_SIZE)) &&
	    inside(y,(1+pan->ttlsize),(pan->ysize-pan->ttlsize))))
      return -1;

   ty = 1 + pan->ttlsize;
   for (i=0; i<pan->numbtn; i++)
      if (inside(y,ty,(ty+pan->btn[i].ysize-1)))
	 return i;
      else
	 ty += pan->btn[i].ysize;


   return -1;
}


/************************************************************************/
/*									*/
/*	track_move - handle cursor movement while tracking		*/
/*									*/
/************************************************************************/

static void
track_move(x,y,mp)
   Integer x,y;
   STEM_PUM mp;
{
   register STEM_PAN p;
   register int flag, new;

   DTRACE("track_move %d %d 0x%x",x,y,mp);

   p = mp->panel[mp->curpan];
   x -= p->xoff;

   if (inside(x,1,ARROW_X_SIZE) &&
	  inside(y,(p->ttlsize+1),(p->ysize-p->ttlsize)))
      flag = -1;
   else if (inside(x,(p->xsize-ARROW_X_SIZE),p->xsize) &&
	       inside(y,(p->ttlsize+1),(p->ysize-p->ttlsize)))
      flag = 1;
   else {
      flag = 0;
      mp->outside = TRUE;
   }

   if (flag != 0 && mp->outside) {
      new = mp->curpan + flag;
      if (new >= 0 && new < mp->numpanels) {
	 top_panel(mp,new);
	 mp->outside = FALSE;
      }
   }
}





/************************************************************************/
/*									*/
/*	inside -- check for containment in a line			*/
/*									*/
/************************************************************************/


static Boolean
inside(x,lx,rx)
   Integer x;
   Integer lx,rx;
{
   DTRACE("inside %d %d %d",x,lx,rx);

   if (lx < rx) {
      if (x < lx || x > rx) return FALSE;
    }
   else {
      if (x > lx || x < rx) return FALSE;
    };

   return TRUE;
};






/* end of stempdm.c */

