/************************************************************************/
/*									*/
/*		stempdm.c						*/
/*									*/
/*	Pull down menu module for STEM menu package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"
#include <leaf.h>




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define CH_EVENTS	RIP_NO_CHARS
#define BTN_EVENTS	RIP_BTN_ANY_DOWN
#define DEFAULT_RIPOFF	TRUE

#define TRACK_CURSOR	ASH_CURSOR_SMALL_BULLSEYE

#define MAX_PMENU	8
#define MAX_PBTN	32

#define LEFT_MARGIN	5
#define DELTA_SPACE	16
#define DELTA_ICON	5
#define ICON_WIDTH	9
#define ICON_HEIGHT	8
#define MIN_X_SIZE	50
#define BTN_SIZE	20
#define BTN_OFFSET	10

#define NUM_COLORS	16

#define BORDER_X	2
#define BORDER_Y	2




/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _STEM_PDM *	STEM_PDM;



typedef struct _STEM_PBTN {
   String	name;
   Boolean	enable;
   Boolean	select;
   Boolean	info;
   Function_Ptr routine;
   Integer	color;
   Integer	bcolor;
   STEM_BTN	btn;
} STEM_PBTN;





typedef struct _STEM_PKEY {
   String	name;
   String	mname;
   Integer	key;
   Function_Ptr routine;
} STEM_PKEY;





typedef struct _STEM_PMENU {
   String	name;
   ASH_WINDOW	window;
   Integer	lx,rx;
   Integer	xsize,ysize;
   Boolean	enable;
   Boolean	ripoff;
   Integer	ripx;
   Integer	numbtn;
   Integer	color;
   Integer	bcolor;
   STEM_PBTN	btn[MAX_PBTN];
   STEM_BTN	tbtn;
   Function_Ptr routine;
   ASH_WINDOW	ripwin;
   STEM_PDM	pdm;
} STEM_PMENU;





typedef struct _STEM_PDM {
   ASH_WINDOW	window;
   ASH_WINDOW	topwin;
   Integer	data;
   Integer	nummenu;
   Integer	numkey;
   Integer	ttlfont;
   Integer	btnfont;
   Integer	yoffset;
   Integer	bsize;
   STEM_PMENU	menu[MAX_PMENU];
   STEM_PKEY	key[MAX_PBTN];
   RIP_REGION	key_region;
   Integer	color;
   Integer	bcolor;
} STEM_PDM_B;





typedef struct _PDM_DATA {
   STEM_PDM	pdm;
   Integer	menu;
   Integer	btn;
   ASH_WINDOW	menuwin;
   ASH_WINDOW	rootwin;
   Boolean	ripoff;
} PDM_DATA;




/************************************************************************/
/*									*/
/*	Local Variables 						*/
/*									*/
/************************************************************************/


static	PDM_DATA *	cur_pdm_data;
static	Boolean 	root_windows;
static	ASH_WINDOW	root_window;
static	ASH_FONT	ttl_default_font;
static	ASH_FONT	btn_default_font;
static	Integer 	btn_size;

static	Integer 	border_x;
static	Integer 	border_y;
static	Boolean 	default_ripoff;

static	Sequence	all_pdm_data;




/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


typedef struct _PDM_COLORS {
   String foreg;
   String backg;
} PDM_COLORS;


static	PDM_COLORS	pdmcolors[NUM_COLORS] = {
   { "black", "white" },
   { "red", "white" },
   { "black", "white" },
   { "orange", "white" },
   { "green", "white" },
   { "blue", "white" },
   { "maroon", "white" },
   { "black", "yellow" },
   { "yellow", "green" },
   { "white", "red" },
   { "white", "black" },
   { "black", "white" },
   { "red", "white" },
   { "red", "white" },
   { "red", "white" },
};



static	Integer 	ripoff_icon[] = {	/* 9 x 10 up		*/
   0x3e000000,					/* 9 x 7 down		*/
   0x1c000000,
   0x1c000000,
   0x3e000000,
   0x7f000000,
   0xff800000,
   0x1c000000,
   0x1c000000,
   0x08000000,
   0x08000000,
};



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 	pdm_top_control();
static	Integer 	pdm_topw_control();
static	void		pdm_top_refresh();
static	Integer 	pdm_button();
static	void		display_title();
static	void		display_title_btn();
static	void		size_menus();
static	Integer 	pdm_track();
static	Boolean 	track_pmove();
static	Boolean 	inside();
static	void		display_pmenu();
static	void		hilite();
static	void		hilite_btn();
static	void		setup_keys();
static	int		key_routine();
static	void		hilite_ripoff();
static	void		handle_ripoff();
static	Integer 	handle_ripoff_title();
static	Integer 	handle_ripoff_remove();
static	Integer 	handle_ripoff_btn();
static	void		ripoff_reset();
static	void		ripoff_visible();






/************************************************************************/
/*									*/
/*	STEM_pdm_init -- module initialization				*/
/*									*/
/************************************************************************/


void
STEM_pdm_init()
{
   String s;
   Integer i,x,y;

   ITRACE("STEM_pdm_init");

   all_pdm_data = NULL;

   HELPregister_info(STEMpdm_help_info);

   ASHicon_define("RIPOFF_UP",ICON_WIDTH,10,ripoff_icon);
   ASHicon_define("RIPOFF_DOWN",ICON_WIDTH,7,ripoff_icon);

   s = ASHinq_resource("pdm.foreground");
   if (s != NULL) pdmcolors[0].foreg = s;
   s = ASHinq_resource("pdm.background");
   if (s != NULL) pdmcolors[0].backg = s;

   s = ASHinq_resource("pdm.reverseVideo");
   if (s != NULL && STREQL(s,"on")) {
      for (i = 0; i < NUM_COLORS; ++i) {
	 s = pdmcolors[i].foreg;
	 pdmcolors[i].foreg = pdmcolors[i].backg;
	 pdmcolors[i].backg = s;
       };
    };

   cur_pdm_data = NULL;
   root_windows = TRUE;
   root_window = NULL;

   ttl_default_font = ASH_DEFAULT_FONT;
   btn_default_font = ASH_DEFAULT_FONT;
   s = ASHinq_resource("pdm.ttl_font");
   if (s != NULL) ttl_default_font = ASHloadfont(s);
   s = ASHinq_resource("pdm.btn_font");
   if (s != NULL) btn_default_font = ASHloadfont(s);

   STEMbtn_size(btn_default_font,"Xp",&x,&y);
   btn_size = y;

   s = ASHinq_resource("pdm.border_x");
   if (s != NULL) border_x = atol(s);
   else border_x = BORDER_X;
   s = ASHinq_resource("pdm.border_y");
   if (s != NULL) border_y = atol(s);
   else border_y = BORDER_Y;

   s = ASHinq_resource("pdm.ripoff");
   if (s != NULL) default_ripoff = STREQL(s,"on");
   else default_ripoff = DEFAULT_RIPOFF;
};





/************************************************************************/
/*									*/
/*	STEMpdm_define -- define a new pdm				*/
/*									*/
/************************************************************************/


void
STEMpdm_define(window,userdata,mdata)
   ASH_WINDOW window;
   Integer userdata;
   STEM_PDM_DATA mdata[];
{
   STEM_PDM p;
   STEM_PMENU * mp;
   STEM_PBTN * bp;
   STEM_PKEY * kp;
   Integer i,st;
   Integer co;
   Boolean simple;
   String s;
   ASH_WINDOW par;

   ENTER("STEMpdm_define 0x%x %s ...",window,mdata[0].name);

   ASHnew_drawinfo(window);

   s = ASHinq_window_resource(window,"simple");
   if (s == NULL || STREQL(s,"off")) simple = FALSE;
   else simple = TRUE;

   p = (STEM_PDM) calloc(1,sizeof(STEM_PDM_B));
   p->window = window;
   p->data = userdata;
   p->nummenu = 0;
   p->numkey = 0;
   p->ttlfont = ttl_default_font;
   p->btnfont = btn_default_font;
   p->yoffset = 0;
   p->bsize = btn_size;
   p->key_region = NULL;
   p->color = ASHinq_color(window);
   p->bcolor = ASHinq_background_color(window);
   for (p->topwin = window; (par = ASHinq_parent(p->topwin)) != NULL; p->topwin = par);

   s = ASHinq_window_resource(window,"pdm.foreground");
   if (s != NULL) p->color = ASHlookup_color(window,s);
   s = ASHinq_window_resource(window,"pdm.background");
   if (s != NULL) p->bcolor = ASHlookup_color(window,s);

   mp = NULL;
   bp = NULL;

   for (i = 0; mdata[i].name != NULL && mdata[i].state != STEM_PSTATE_END; ++i) {
      st = mdata[i].state;
      switch (st & STEM_PSTATE_TYPE) {
	 case STEM_PSTATE_MENU_DISABLE_OLD :
	    st = (st & STEM_PSTATE_FLAGS) | STEM_PSTATE_DISABLE | STEM_PSTATE_MENU;
	    break;
	 case STEM_PSTATE_BTN_DISABLE_OLD :
	    st = (st & STEM_PSTATE_FLAGS) | STEM_PSTATE_DISABLE | STEM_PSTATE_BTN;
	    break;
	 case STEM_PSTATE_BTN_SELECT_OLD :
	    st = (st & STEM_PSTATE_FLAGS) | STEM_PSTATE_SELECT | STEM_PSTATE_BTN;
	    break;
	 default :
	    break;
       };

      if (((st & STEM_PSTATE_SIMPLE) != 0 && !simple) ||
	     ((st & STEM_PSTATE_COMPLEX) != 0 && simple)) {
	 if ((st & STEM_PSTATE_TYPE) == STEM_PSTATE_MENU) mp = NULL;
	 st = STEM_PSTATE_IGNORE;
       };

      switch (st & STEM_PSTATE_TYPE) {
	 case STEM_PSTATE_MENU :
	    mp = &p->menu[p->nummenu++];
	    mp->name = SALLOC(mdata[i].name);
	    mp->window = NULL;
	    mp->enable = ((st & STEM_PSTATE_DISABLE) == 0);
	    if ((st & STEM_PSTATE_RIPOFF) != 0) mp->ripoff = TRUE;
	    else if ((st & STEM_PSTATE_NO_RIPOFF) != 0) mp->ripoff = FALSE;
	    else mp->ripoff = default_ripoff;
	    mp->numbtn = 0;
	    mp->lx = 0;
	    mp->rx = 0;
	    mp->xsize = 0;
	    mp->ysize = 0;
	    co = STEM_PSTATE_INQ_COLOR(st);
	    if (co < 0) {
	       mp->color = p->color;
	       mp->bcolor = p->bcolor;
	     }
	    else {
	       mp->color = ASHlookup_color(window,pdmcolors[co].foreg);
	       mp->bcolor = ASHlookup_color(window,pdmcolors[co].backg);
	       if (p->nummenu == 1 && co == 0) {
		  p->color = mp->color;
		  p->bcolor = mp->bcolor;
		};
	     };
	    mp->routine = mdata[i].routine;
	    mp->tbtn = NULL;
	    mp->ripwin = NULL;
	    mp->pdm = p;
	    break;

	 case STEM_PSTATE_BTN :
	 case STEM_PSTATE_INFO :
	    if (mp == NULL || mp->numbtn >= MAX_PBTN) break;
	    bp = &mp->btn[mp->numbtn++];
	    bp->name = SALLOC(mdata[i].name);
	    bp->routine = mdata[i].routine;
	    bp->select = ((st & STEM_PSTATE_SELECT) != 0);
	    bp->enable = ((st & STEM_PSTATE_DISABLE) == 0);
	    bp->info = ((st & STEM_PSTATE_TYPE) == STEM_PSTATE_INFO);
	    bp->btn = NULL;
	    co = STEM_PSTATE_INQ_COLOR(st);
	    if (co < 0) {
	       bp->color = mp->color;
	       bp->bcolor = mp->bcolor;
	     }
	    else {
	       bp->color = ASHlookup_color(window,pdmcolors[co].foreg);
	       bp->bcolor = ASHlookup_color(window,pdmcolors[co].backg);
	     };
	    break;
       };

      if ((st & STEM_PSTATE_CHAR_FIELD) != 0 && mdata[i].routine != NULL && mp != NULL) {
	 kp = &p->key[p->numkey++];
	 kp->name = SALLOC(mdata[i].name);
	 kp->routine = mdata[i].routine;
	 kp->key = STEM_PSTATE_INQ_CHAR(st);
	 if (mp != NULL) kp->mname = mp->name;
	 else kp->mname = NULL;
       };
    };

   BIOnew_input_window(window);
   BIOset_window_thread(window,THREADinq_current());
   ASHset_menu_data(window,p);
   ASHset_control(window,pdm_top_control);
   ASHset_control(p->topwin,pdm_topw_control);
   ASHset_region_refresh(window,pdm_top_refresh);
   ASHfont(window,p->ttlfont);
   ASHcolor(window,p->color);
   ASHbackground_color(window,p->bcolor);

   PROTECT;
   all_pdm_data = CONS(p,all_pdm_data);
   UNPROTECT;

   setup_keys(p);
   size_menus(p);

   display_title(p);
};





/************************************************************************/
/*									*/
/*	STEMpdm_menu_enable -- enable a specific menu			*/
/*									*/
/************************************************************************/


void
STEMpdm_menu_enable(w,m,fg)
   ASH_WINDOW w;
   String m;
   Boolean fg;
{
   STEM_PDM p;
   Integer i;
   Integer lx,by,rx,ty;

   TRACE("STEMpdm_menu_enable 0x%x %s %d",w,m,fg);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;
   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(m,p->menu[i].name)) {
	 if (p->menu[i].enable != fg) {
	    p->menu[i].enable = fg;
	    display_title_btn(p,i,by,ty);
	  };
	 break;
       };
    };
};






/************************************************************************/
/*									*/
/*	STEMpdm_remove -- explicitly remove a pull down menu		*/
/*									*/
/************************************************************************/


void
STEMpdm_remove(w)
   ASH_WINDOW w;
{
   STEM_PDM p;
   Integer m;

   TRACE("STEMpdm_remove 0x%x",w);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;
   ASHset_menu_data(w,NULL);

   PROTECT;
   all_pdm_data = REMOB(p,all_pdm_data);
   UNPROTECT;

   for (m = 0; m < p->nummenu; ++m) {
      if (p->menu[m].ripwin != NULL) {
	 if (ASHinq_valid_window(p->menu[m].ripwin)) ASHremove(p->menu[m].ripwin);
	 p->menu[m].ripwin = NULL;
       };
    };

   free(p);
};





/************************************************************************/
/*									*/
/*	STEMpdm_btn_enable -- set button enable/disable 		*/
/*	STEMpdm_btn_select -- set button select/deselect		*/
/*									*/
/************************************************************************/


void
STEMpdm_btn_enable(w,m,b,fg)
   ASH_WINDOW w;
   String m;
   String b;
   Boolean fg;
{
   STEM_PDM p;
   Integer i,j;

   TRACE("STEMpdm_btn_enable 0x%x %s %s %d",w,m,b,fg);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(p->menu[i].name,m)) {
	 for (j = 0; j < p->menu[i].numbtn; ++j) {
	    if (STREQL(p->menu[i].btn[j].name,b)) {
	       p->menu[i].btn[j].enable = fg;
	       ripoff_reset(&p->menu[i]);
	       break;
	     };
	  };
	 break;
       };
    };
};






void
STEMpdm_btn_select(w,m,b,fg)
   ASH_WINDOW w;
   String m;
   String b;
   Boolean fg;
{
   STEM_PDM p;
   Integer i,j;

   TRACE("STEMpdm_btn_select 0x%x %s %s %d",w,m,b,fg);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(p->menu[i].name,m)) {
	 for (j = 0; j < p->menu[i].numbtn; ++j) {
	    if (STREQL(p->menu[i].btn[j].name,b)) {
	       p->menu[i].btn[j].select = fg;
	       ripoff_reset(&p->menu[i]);
	       break;
	     };
	  };
	 break;
       };
    };
};






/************************************************************************/
/*									*/
/*	STEMpdm_menu_remove -- remove a menu				*/
/*	STEMpdm_menu_add -- add a new menu				*/
/*	STEMpdm_menu_remove_btns -- remove all buttons from a menu	*/
/*									*/
/************************************************************************/


void
STEMpdm_menu_remove(w,m)
   ASH_WINDOW w;
   String m;
{
   STEM_PDM p;
   Integer i,j;

   TRACE("STEMpdm_menu_remove 0x%x %s",w,m);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   j = 0;
   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(m,p->menu[i].name)) {
	 if (p->menu[i].ripwin != NULL) {
	    if (ASHinq_valid_window(p->menu[i].ripwin)) ASHremove(p->menu[i].ripwin);
	    p->menu[i].ripwin = NULL;
	  };
	 j = 1;
       }
      else if (j != 0) {
	 p->menu[i-j] = p->menu[i];
       };
    };

   size_menus(p);

   if (j != 0) {
      p->nummenu -= j;
      display_title(p);
    };
};





void
STEMpdm_menu_add(w,m)
   ASH_WINDOW w;
   String m;
{
   STEM_PDM p;
   STEM_PMENU * mp;

   TRACE("STEMpdm_menu_add 0x%x %s",w,m);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   mp = &p->menu[p->nummenu++];
   mp->name = SALLOC(m);
   mp->window = NULL;
   mp->enable = TRUE;
   mp->numbtn = 0;
   mp->tbtn = NULL;
   mp->lx = 0;
   mp->rx = 0;
   mp->xsize = 0;
   mp->ysize = 0;
   mp->color = p->color;
   mp->bcolor = p->bcolor;
   mp->ripoff = default_ripoff;
   mp->pdm = p;

   size_menus(p);

   display_title(p);
};





void
STEMpdm_menu_remove_btns(w,m)
   ASH_WINDOW w;
   String m;
{
   STEM_PDM p;
   Integer i,j;

   TRACE("STEMpdm_menu_remove_btns 0x%x %s",w,m);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(m,p->menu[i].name)) {
	 if (p->menu[i].ripwin != NULL) {
	    if (ASHinq_valid_window(p->menu[i].ripwin)) ASHremove(p->menu[i].ripwin);
	    p->menu[i].ripwin = NULL;
	  };
	 for (j = 0; j < p->menu[i].numbtn; ++j) {
	    p->menu[i].btn[j].btn = NULL;
	  };
	 p->menu[i].numbtn = 0;
       }
    };

   size_menus(p);
};





/************************************************************************/
/*									*/
/*	STEMpdm_btn_remove -- remove button from menu			*/
/*	STEMpdm_btn_add -- add button to menu				*/
/*									*/
/************************************************************************/


void
STEMpdm_btn_remove(w,m,b)
   ASH_WINDOW w;
   String m;
   String b;
{
   STEM_PDM p;
   Integer i,j,k;

   TRACE("STEMpdm_btn_remove 0x%x %s %s",w,m,b);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(p->menu[i].name,m)) {
	 k = 0;
	 for (j = 0; j < p->menu[i].numbtn; ++j) {
	    if (k == 0 && STREQL(p->menu[i].btn[j].name,b)) k = 1;
	    else if (k != 0) p->menu[i].btn[j-k] = p->menu[i].btn[j];
	  };
	 p->menu[i].btn[j].btn = NULL;
	 p->menu[i].numbtn -= k;
	 if (k != 0) break;
       };
    };

   size_menus(p);
};





void
STEMpdm_btn_add(w,m,b,rtn)
   ASH_WINDOW w;
   String m;
   String b;
   Function_Ptr rtn;
{
   STEM_PDM p;
   STEM_PMENU * mp;
   STEM_PBTN * bp;
   Integer i;

   TRACE("STEMpdm_btn_add 0x%x %s %s 0x%x",w,m,b,rtn);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   for (i = 0; i < p->nummenu; ++i) {
      if (STREQL(p->menu[i].name,m)) {
	 mp = &p->menu[i];
	 if (mp->numbtn >= MAX_PBTN) break;
	 bp = &mp->btn[mp->numbtn++];
	 bp->name = SALLOC(b);
	 bp->routine = rtn;
	 bp->select = FALSE;
	 bp->enable = TRUE;
	 bp->color = mp->color;
	 bp->bcolor = mp->bcolor;
	 break;
       };
    };

   size_menus(p);
};





/************************************************************************/
/*									*/
/*	STEMpdm_fonts -- set menu and button fonts			*/
/*									*/
/************************************************************************/


void
STEMpdm_fonts(w,ttl,btn)
   ASH_WINDOW w;
   Integer ttl;
   Integer btn;
{
   STEM_PDM p;

   TRACE("STEMpdm_fonts 0x%x %d %d",w,ttl,btn);

   p = (STEM_PDM) ASHinq_menu_data(w);
   if (p == NULL) return;

   if (ttl > 0) p->ttlfont = ttl;
   if (btn > 0) p->btnfont = btn;

   size_menus(p);

   display_title(p);
};





/************************************************************************/
/*									*/
/*	STEMpdm_help_info -- check for help information 		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

String
STEMpdm_help_info(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   PDM_DATA * pdm;
   String s;
   Character buf[1024],xbuf[1024];
   ASH_WINDOW p;

   if (cur_pdm_data != NULL && cur_pdm_data->menuwin == w) pdm = cur_pdm_data;
   else return NULL;

   buf[0] = 0;
   if (pdm->rootwin != NULL) {
      for (p = pdm->pdm->window; p != NULL; p = ASHinq_parent(p)) {
	 if (ASHinq_parent(p) == NULL) break;
	 s = ASHinq_window_id(p);
	 if (s != NULL) {
	    if (buf[0] == 0) strcpy(buf,s);
	    else {
	       strcpy(xbuf,buf);
	       sprintf(buf,"%s.%s",s,xbuf);
	     };
	  };
       };
    };

   if (pdm != NULL && pdm->pdm != NULL && pdm->menu >= 0 && pdm->btn >= 0) {
      strcpy(xbuf,buf);
      if (xbuf[0] != 0) strcat(xbuf,".");
      sprintf(buf,"%s%s.%s",xbuf,
		 pdm->pdm->menu[pdm->menu].name,
		 pdm->pdm->menu[pdm->menu].btn[pdm->btn].name);
      s = SALLOC(buf);
    }
   else s = NULL;

   return s;
};






/************************************************************************/
/*									*/
/*	pdm_top_control -- handle control for top of pull down menus	*/
/*	pdm_topw_control -- handle control for root window		*/
/*									*/
/************************************************************************/


static Integer
pdm_top_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_PDM p;

   ITRACE("pdm_top_control %s 0x%x",msg,w);

   p = (STEM_PDM) ASHinq_menu_data(w);

   if (p != NULL) {
      if (STREQL(msg,"ASH$REMOVE")) {
	 STEMpdm_remove(w);
       };
    };

   return ASH_CONTROL_REJECT;
};





static Integer
pdm_topw_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_PDM p;
   Sequence l;

   if (STREQL(msg,"ASH$VISIBLE") || STREQL(msg,"ASH$INVISIBLE")) {
      forin (p,STEM_PDM,l,all_pdm_data) {
	 if (p->topwin == w) {
	    ripoff_visible(p,STREQL(msg,"ASH$VISIBLE"));
	    break;
	  };
       };
    };

   return ASH_CONTROL_REJECT;
};






/************************************************************************/
/*									*/
/*	pdm_top_refresh -- handle refresh routine for top window	*/
/*									*/
/************************************************************************/


static void
pdm_top_refresh(w)
   ASH_WINDOW w;
{
   STEM_PDM p;

   ITRACE("pdm_top_refresh 0x%x",w);

   p = (STEM_PDM) ASHinq_menu_data(w);

   if (p == NULL) return;

   display_title(p);
};





/************************************************************************/
/*									*/
/*	pdm_button -- handle rip events on top buttons			*/
/*									*/
/************************************************************************/


static Integer
pdm_button(x,y,ch,btns,rgn)
   Integer x,y;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   ASH_WINDOW w,gw,root;
   Integer m,b;
   Boolean fg,cfg,ripfg;
   ASH_CURSOR crsr;
   STEM_PDM sp;
   PDM_DATA cur;

   ITRACE("pdm_button %d %d 0x%x 0x%x 0x%x",x,y,ch,btns,rgn);

   if (btns & RIP_BTN_UP) return FALSE;
   if (btns & RIP_BTN_NONE) return FALSE;
   if (btns & RIP_BTN_TAP) return FALSE;

   w = RIPinq_window(rgn);
   cur.pdm = (STEM_PDM) ASHinq_menu_data(w);
   cur.menu = RIPinq_data(rgn);
   cur.btn = -1;
   cur.menuwin = NULL;
   cur.ripoff = FALSE;

   cur.rootwin = NULL;
   if (root_windows) {
      PROTECT;
      root = root_window;
      if (root == NULL || ASHinq_display(root) != ASHinq_display(w)) {
	 root_window = NULL;
	 UNPROTECT;
	 if (root != NULL) ASHremove(root);
	 root = ASHroot_X_window(w);
       }
      else {
	 UNPROTECT;
       };
      cur.rootwin = root;
    };

   PROTECT;
   if (cur_pdm_data != NULL) {
      UNPROTECT;
      return FALSE;
    };
   cur_pdm_data = &cur;
   UNPROTECT;

   gw = RIPgrab_window(NULL);
   cfg = BIOset_cursor(NULL,TRUE);
   crsr = BIOset_cursor_standard(NULL,TRACK_CURSOR);
   fg = RIPtrack(pdm_track,1,NULL,FALSE);

   cur_pdm_data = NULL;

   m = cur.menu;
   b = cur.btn;
   sp = cur.pdm;
   ripfg = cur.ripoff;
   display_pmenu(&cur,-1);
   BIOset_cursor_pattern(NULL,crsr);
   BIOset_cursor(NULL,cfg);
   RIPgrab_window(gw);

   if (cur.rootwin != NULL) {
      PROTECT;
      if (root_window != NULL && root_window != cur.rootwin) {
	 UNPROTECT;
	 ASHremove(cur.rootwin);
       }
      else {
	 root_window = cur.rootwin;
	 UNPROTECT;
       };
      cur.rootwin = NULL;
    };

   if (fg && m >= 0 && b >= 0) {
      if (sp->menu[m].btn[b].enable &&
	     sp->menu[m].btn[b].routine != NULL) {
	 fg = (*(sp->menu[m].btn[b].routine))(
		     sp->data,
		     sp->menu[m].name,
		     sp->menu[m].btn[b].name,
		     sp->window);
       };
    }
   else if (fg && m >= 0 && b == -2) {
      if (sp->menu[m].routine != NULL) {
	 fg = (*(sp->menu[m].routine))(sp->data,sp->menu[m].name,NULL,sp->window);
       };
    }
   else if (fg && m >= 0 && ripfg) {
      handle_ripoff(sp,m);
      fg = TRUE;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	display_title -- display title line for pull down menu		*/
/*									*/
/************************************************************************/


static void
display_title(p)
   STEM_PDM p;
{
   Integer i,x;
   Integer dx,yd,dy;
   Integer xs,ys,xo,yo;
   Integer lx,by,rx,ty;
   STEM_BTN sb;

   DTRACE("display_title 0x%x",p);

   if (!ASHlock(p->window)) return;
   ASHbatch_mode(TRUE);

   ASHcolor(p->window,p->color);
   ASHbackground_color(p->window,p->bcolor);
   ASHfont(p->window,p->ttlfont);

   ASHinq_size(p->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);

   STEMbtn_window_remove(p->window);
   x = lx+ dx*LEFT_MARGIN;

   yd = 1;
   for (i = 0; i < p->nummenu; ++i) {
      ASHinq_text_offset(p->ttlfont,p->menu[i].name,&xo,&yo);
      if (yo > yd) yd = yo;
    };
   p->yoffset = yd;

   sb = STEMbtn_define(p->window,1,1,1,1,"",STEM_BTN_NO_DRAW,NULL,NULL);
   STEMbtn_background(sb,0,0,0,0);

   for (i = 0; i < p->nummenu; ++i) {
      p->menu[i].tbtn = NULL;
      STEMbtn_size(p->ttlfont,p->menu[i].name,&xs,&ys);
      p->menu[i].lx = x;
      p->menu[i].rx = x+xs*dx;
      if (p->menu[i].ripoff) xs += ICON_WIDTH+DELTA_ICON;
      display_title_btn(p,i,by,ty);
      x += xs*dx+dx*DELTA_SPACE;
    };

   ASHline(p->window,lx,by,rx,by);
   i = ASHcolor(p->window,p->bcolor);
   ASHline(p->window,lx,by-dy,rx,by-dy);
   ASHcolor(p->window,i);

   ASHunlock(p->window);
   ASHbatch_mode(FALSE);
};






/************************************************************************/
/*									*/
/*	display_title_btn -- display single button in title		*/
/*									*/
/************************************************************************/


static void
display_title_btn(p,i,by,ty)
   STEM_PDM p;
   Integer i;
   Integer by,ty;
{
   Integer y,rx;
   Integer dx,dy;
   STEM_BTN sb;
   STEM_BTN_FLAGS fgs;
   ASH_COLOR cl,bcl;

   DTRACE("display_title_btn 0x%x %d %d %d",p,i,by,ty);

   dx = (p->menu[i].lx < p->menu[i].rx ? 1 : -1);
   dy = (by < ty ? 1 : -1);

   y = by+dy*(2+p->yoffset);

   rx = p->menu[i].rx;

   if (p->menu[i].tbtn != NULL) STEMbtn_remove(p->menu[i].tbtn);

   fgs = STEM_BTN_NO_REFRESH|STEM_BTN_SENSE_FLIP|STEM_BTN_NO_DRAW;
   cl = ASHcolor(p->window,p->menu[i].color);
   bcl = ASHbackground_color(p->window,p->menu[i].bcolor);
   sb = STEMbtn_define(p->window,p->menu[i].lx,by+dy+dy,rx,ty,
			  p->menu[i].name,fgs,i,pdm_button);
   if (!p->menu[i].enable) STEMbtn_enable(sb,FALSE);
   STEMbtn_draw(sb);
   ASHcolor(p->window,cl);
   ASHbackground_color(p->window,bcl);
   p->menu[i].tbtn = sb;

   if (p->menu[i].ripoff) {
      rx += DELTA_ICON*dx;
      if (p->menu[i].ripwin != NULL) ASHicon_draw(p->window,rx,y,"RIPOFF_UP");
      else ASHicon_draw(p->window,rx,y,"RIPOFF_DOWN");
    };
};






/************************************************************************/
/*									*/
/*	size_menus -- get sizes of the various menus			*/
/*									*/
/************************************************************************/


static void
size_menus(p)
   STEM_PDM p;
{
   Integer m,b;
   Integer x,y;
   Integer xsz,ysz,bsz;

   STEMbtn_size(p->btnfont,"Xp",&x,&y);
   p->bsize = y+border_y*2;
   bsz = 4;

   for (m = 0; m < p->nummenu; ++m) {
      p->menu[m].ysize = p->bsize*p->menu[m].numbtn+bsz;
      x = MIN_X_SIZE;
      for (b = 0; b < p->menu[m].numbtn; ++b) {
	 STEMbtn_size(p->btnfont,p->menu[m].btn[b].name,&xsz,&ysz);
	 xsz += 2*border_x;
	 x = MAX(x,xsz);
       };
      p->menu[m].xsize = x+bsz;
      ripoff_reset(&p->menu[m]);
    };
};





/************************************************************************/
/*									*/
/*	pdm_track -- tracking function for button selection		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
pdm_track(x,y,n,type,max,w)
   Integer x,y;
   Integer n;
   RIP_TRANS type;
   Integer max;
   ASH_WINDOW w;
{
   Boolean fg;
   PDM_DATA * cur;

   DTRACE("pdm_track %d %d %d %d %d",x,y,n,type,max);

   fg = TRUE;

   cur = cur_pdm_data;
   if (cur == NULL) return FALSE;

   switch (type) {
      case RIP_TRANS_NONE :		/* first time			*/
	 display_pmenu(cur,cur->menu);
	 break;
      case RIP_TRANS_MOVE :
	 fg = track_pmove(cur,x,y);
	 break;
      case RIP_TRANS_UP :
	 break;
      default :
	 fg = FALSE;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	track_pmove -- handle cursor movements while tracking		*/
/*									*/
/************************************************************************/


static Boolean
track_pmove(cur,x,y)
   PDM_DATA * cur;
   Integer x,y;
{
   Integer lx,by,rx,ty;
   Integer nx,ny;
   Integer i,dy;
   ASH_WINDOW w;
   ASH_WINDOW tw;
   Integer btn;
   STEM_PMENU * mp;

   DTRACE("track_pmove 0x%x %d %d",cur,x,y);

   tw = ASHinq_top_window(cur->pdm->window);

   if (cur->menuwin != NULL) {
      ASHinq_size(cur->menuwin,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      ASHmap(tw,x,y,cur->menuwin,&nx,&ny);
      if (inside(nx,lx,rx) && inside(ny,by,ty)) {
	 if (cur->ripoff) hilite_ripoff(cur,FALSE);
	 btn = ny/cur->pdm->bsize;
	 if (btn >= cur->pdm->menu[cur->menu].numbtn) btn = -1;
	 if (cur->btn != btn) {
	    hilite_btn(cur,FALSE);
	    cur->btn = btn;
	    hilite_btn(cur,TRUE);
	  };
	 return TRUE;
       };
    };

   hilite_btn(cur,FALSE);
   cur->btn = -1;

   ASHinq_size(cur->pdm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHmap(tw,x,y,cur->pdm->window,&nx,&ny);
   dy = (by < ty ? 1 : -1);

   if (cur->menu >= 0) {
      mp = &cur->pdm->menu[cur->menu];
      if (mp->ripoff) {
	 if (inside(nx,mp->rx+DELTA_ICON,mp->rx+DELTA_ICON+ICON_WIDTH) &&
		inside(ny,by+2*dy,by+(2+ICON_HEIGHT)*dy)) {
	    if (cur->ripoff) return TRUE;
	    hilite_ripoff(cur,TRUE);
	    return TRUE;
	  }
	 else if (cur->ripoff) hilite_ripoff(cur,FALSE);
       };
    };

   if (inside(nx,lx,rx) && inside(ny,by+2*dy,ty-2*dy)) {
      for (i = 0; i < cur->pdm->nummenu; ++i) {
	 if (inside(nx,cur->pdm->menu[i].lx,cur->pdm->menu[i].rx)) {
	    if (i != cur->menu) display_pmenu(cur,i);
	    else cur->btn = -2;
	    break;
	  };
       };
      return TRUE;
    };

   if (cur->rootwin == NULL) {
      w = ASHinq_parent(cur->pdm->window);
      ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      ASHmap(tw,x,y,w,&nx,&ny);
      if (!inside(nx,lx,rx) || !inside(ny,by,ty)) {
	 display_pmenu(cur,-1);
	 return FALSE;
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	inside -- check for containment in a line			*/
/*									*/
/************************************************************************/


static Boolean
inside(x,lx,rx)
   Integer x;
   Integer lx,rx;
{
   DTRACE("inside %d %d %d",x,lx,rx);

   if (lx < rx) {
      if (x < lx || x > rx) return FALSE;
    }
   else {
      if (x > lx || x < rx) return FALSE;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	display_pmenu -- display a pull-down menu			*/
/*									*/
/************************************************************************/


static void
display_pmenu(cur,m)
   PDM_DATA * cur;
   Integer m;
{
   Integer nx,ny;
   Integer lx,by,rx,ty;
   Integer rlx,rby,rrx,rty;
   Integer i,y,dy;
   STEM_PBTN * bp;
   ASH_WINDOW w;
   STEM_BTN sb;
   Integer brdr;
   ASH_COLOR cl,bcl;

   DTRACE("display_pmenu 0x%x %d",cur,m);

   if (cur->menuwin != NULL) {
      hilite_ripoff(cur,FALSE);
      hilite(cur,cur->menu,FALSE);
      ASHremove(cur->menuwin);
      cur->menuwin = NULL;
    };

   if (!cur->pdm->menu[m].enable) m = -1;

   cur->menu = m;
   cur->btn = -2;
   if (m < 0) return;

   ASHbatch_mode(TRUE);

   ASHinq_size(cur->pdm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dy = (ty < by ? 1 : -1);
   hilite(cur,cur->menu,TRUE);
   w = ASHinq_parent(cur->pdm->window);
   ASHmap(cur->pdm->window,cur->pdm->menu[m].lx,by+cur->pdm->menu[m].ysize*dy+2*dy,
	     w,&nx,&ny);
   if (cur->rootwin != NULL) {
      ASHmap(w,nx,ny,cur->rootwin,&nx,&ny);
      w = cur->rootwin;
      ASHinq_size(cur->rootwin,ASH_SIZE_WINDOW,&rlx,&rby,&rrx,&rty);
      if (ny > rby) ny = rby;
      if (ny - cur->pdm->menu[m].ysize < 0) ny = cur->pdm->menu[m].ysize;
      if (nx + cur->pdm->menu[m].xsize > rrx) nx = rrx - cur->pdm->menu[m].xsize;
    };

   cur->menuwin =
      ASHcreate(w,nx,ny,
		   0,cur->pdm->menu[m].ysize,cur->pdm->menu[m].xsize,0,
		   0,
		   ASH_WINDOW_COURTEOUS|ASH_WINDOW_NOSAVE|ASH_WINDOW_NOCLEAR|
		   ASH_WINDOW_INDEPENDENT|
		   ASH_WINDOW_NOREDIRECT|ASH_WINDOW_INVISIBLE);
/* ASHset_window_name(cur->menuwin,"STEM_PDM");         */
   ASHvisible(cur->menuwin,TRUE);

/* ASHbatch_mode(FALSE);	*/

   if (cur->rootwin != NULL) {
/*    while (!ASHinq_viewable(cur->menuwin)) BPIOwait(100);	*/
    };

/* ASHbatch_mode(TRUE); 	*/

   ASHfont(cur->menuwin,cur->pdm->btnfont);
   ASHcolor(cur->menuwin,cur->pdm->menu[m].color);
   ASHbackground_color(cur->menuwin,cur->pdm->menu[m].bcolor);
   BIOnew_input_window(cur->menuwin);

   y = 2;
   brdr = 2;

   for (i = 0; i < cur->pdm->menu[m].numbtn; ++i) {
      bp = &cur->pdm->menu[m].btn[i];
      cl = ASHcolor(cur->menuwin,bp->color);
      bcl = ASHbackground_color(cur->menuwin,bp->bcolor);
      sb = STEMbtn_define(cur->menuwin,border_x+brdr,y+cur->pdm->bsize-1-border_y,
			     cur->pdm->menu[m].xsize-border_x-brdr,y+border_y,
			     bp->name,
			     STEM_BTN_NO_DRAW|STEM_BTN_FULL_SIZE|STEM_BTN_LEFT,
			     NULL,NULL);
      if (bp->select) STEMbtn_select(sb,TRUE);
      if (!bp->enable) STEMbtn_enable(sb,FALSE);
      bp->btn = sb;
      ASHcolor(cur->menuwin,cl);
      ASHbackground_color(cur->menuwin,bcl);
      y += cur->pdm->bsize;
    };

   STEMbtn_window_draw(cur->menuwin);
   ASHbox(cur->menuwin,0,cur->pdm->menu[m].ysize,cur->pdm->menu[m].xsize,0);
   i = ASHcolor(cur->menuwin,cur->pdm->menu[m].bcolor);
   ASHbox(cur->menuwin,1,cur->pdm->menu[m].ysize-1,cur->pdm->menu[m].xsize-1,1);
   ASHcolor(cur->menuwin,i);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	hilite -- inverse video the title area of selected menu 	*/
/*									*/
/************************************************************************/


static void
hilite(cur,m,fg)
   PDM_DATA * cur;
   Integer m;
   Boolean fg;
{
   Integer lx,by,rx,ty;

   DTRACE("hilite 0x%x %d",cur,m);

   if (m < 0) return;

   ASHinq_size(cur->pdm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   STEMbtn_hilite(cur->pdm->menu[m].tbtn,fg);
};





/************************************************************************/
/*									*/
/*	hilite_btn -- hilite a button in menu				*/
/*									*/
/************************************************************************/


static void
hilite_btn(cur,fg)
   PDM_DATA * cur;
   Boolean fg;
{
   STEM_PBTN * bp;

   if (cur->btn < 0 || cur->menu < 0 || cur->menuwin == NULL) return;

   bp = &cur->pdm->menu[cur->menu].btn[cur->btn];

   if (bp->enable && !bp->info) {
      if (bp->btn != NULL) STEMbtn_set(bp->btn,fg);
    };
};





/************************************************************************/
/*									*/
/*	setup_keys -- handle keys in the menu				*/
/*									*/
/************************************************************************/


static void
setup_keys(p)
   STEM_PDM p;
{
   Integer i;
   Character keys[1024],buf[128];
   ASH_WINDOW par;

   if (p->numkey == 0) return;

   keys[0] = 0;
   for (i = 0; i < p->numkey; ++i) {
      sprintf(buf,"\\%o",p->key[i].key);
      strcat(keys,buf);
    };

   par = ASHinq_parent(p->window);

   p->key_region = RIPdefine_region(par,0,0,0,0,keys,RIP_BTN_NONE,key_routine,
				       ASH_SENSE_NO_CHANGE);
   RIPset_data(p->key_region,p);
};





/*ARGSUSED*/

static int
key_routine(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   STEM_PDM p;
   Integer i;

   if ((btn & RIP_BTN_NONE) == 0) return FALSE;

   p = (STEM_PDM) RIPinq_data(rgn);

   for (i = 0; i < p->numkey; ++i) {
      if (ch == p->key[i].key) {
	 if (p->key[i].mname != NULL) {
	    (*p->key[i].routine)(p->data,p->key[i].mname,p->key[i].name,p->window);
	  }
	 else {
	    (*p->key[i].routine)(p->data,p->key[i].name,NULL,p->window);
	  };
	 break;
       };
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	hilite_ripoff -- hilight ripoff icon for current menu		*/
/*									*/
/************************************************************************/


static void
hilite_ripoff(cur,fg)
   PDM_DATA * cur;
   Boolean fg;
{
   Integer lx,by,rx,ty;
   Integer c,bc,dy,dx;
   STEM_PMENU * mp;

   if (cur->menu < 0) {
      cur->ripoff = FALSE;
      return;
    };
   if (cur->ripoff && fg) return;
   if (!cur->ripoff && !fg) return;

   ASHinq_size(cur->pdm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dy = (by < ty ? 1 : -1);
   dx = (lx < rx ? 1 : -1);

   by += 2*dy;

   mp = &cur->pdm->menu[cur->menu];
   if (!mp->ripoff) {
      cur->ripoff = FALSE;
      return;
    };

   lx = mp->rx+DELTA_ICON*dx;
   rx = lx + ICON_WIDTH*dx;
   ty = by + ICON_HEIGHT*dy;

   c = ASHcolor(cur->pdm->window,mp->color);
   bc = ASHbackground_color(cur->pdm->window,mp->bcolor);

   ASHhilite_box(cur->pdm->window,lx,by,rx,ty);

   ASHbackground_color(cur->pdm->window,bc);
   ASHcolor(cur->pdm->window,c);

   cur->ripoff = fg;
};






/************************************************************************/
/*									*/
/*	handle_ripoff -- handle ripoff menu for given menu		*/
/*	handle_ripoff_title -- handle hits in title area		*/
/*	handle_ripoff_remove -- handle hits in remove area		*/
/*	handle_ripoff_btn -- handle hits on buttons			*/
/*	ripoff_reset -- setup ripoff buttons for menu			*/
/*	ripoff_visible -- make ripoff windows follow menu visibility	*/
/*									*/
/************************************************************************/


#define RIPOFF_HT	16

#define RIPOFF_TITLE	1
#define RIPOFF_ICON	2
#define RIPOFF_BTNS	3

static LEAF_DATA	ripoff_leaf[] = {
   LEAF_ROOT(NULL),
   { RIPOFF_TITLE, LEAF_TYPE_TITLE,
	{ LEAF_COORD_LX, LEAF_COORD_SAME(RIPOFF_ICON),
	     LEAF_COORD_NEXT(RIPOFF_ICON), LEAF_COORD_TY },
	handle_ripoff_title, NULL },
   { RIPOFF_ICON, LEAF_TYPE_TEXT|LEAF_TYPE_ICONIC|LEAF_TYPE_SENSE_FLIP,
	{ LEAF_COORD_TEXT, LEAF_COORD_SIZE(LEAF_VALUE_PARAM(0)),
	     LEAF_COORD_RX, LEAF_COORD_TY },
	handle_ripoff_remove, (int) "RIPOFF_UP" },
   { RIPOFF_BTNS, LEAF_TYPE_STEM|LEAF_TYPE_INVISIBLE,
	{ LEAF_COORD_LX, LEAF_COORD_BY,
	     LEAF_COORD_RX, LEAF_COORD_NEXT_LINE(RIPOFF_TITLE) },
	handle_ripoff_btn, NULL },
   LEAF_END
};






static void
handle_ripoff(sp,m)
   STEM_PDM sp;
   Integer m;
{
   STEM_PMENU * mp;
   Character buf[256];
   ASH_WINDOW w;
   Integer x,y;

   mp = &sp->menu[m];

   if (!mp->ripoff) return;
   if (mp->ripwin != NULL) {
      ASHpop(mp->ripwin);
      return;
    };

   ASHinq_text_next(mp->pdm->ttlfont,mp->name,&x,&y);
   x += ICON_WIDTH+DELTA_ICON;
   if (mp->xsize >= x) x = mp->xsize;
   if (y < RIPOFF_HT) y = RIPOFF_HT;

   sprintf(buf,"%s_Menu -geometry %dx%d -query",mp->name,x+4,mp->ysize+y+4);
   w = ASHroot_window(ASHinq_display_name(sp->window),buf);
   mp->ripwin = w;

   if (w == NULL) return;

   ASHinq_window_full_id(mp->pdm->window,buf);
   ASHset_window_id(w,buf);
   ASHset_window_defaults(w);
   ASHinput_lock_by(w,mp->pdm->window);

   LEAFsetup_window(w,ripoff_leaf,mp);
   LEAFset_data(mp->name,w,RIPOFF_TITLE,0);
   LEAFset_font(mp->pdm->ttlfont,w,RIPOFF_TITLE,0);
   LEAFset_parameters(w,1,y);
   LEAFredraw(w);

   ripoff_reset(mp);

   display_title(mp->pdm);
};






/*ARGSUSED*/

static Integer
handle_ripoff_title(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   STEM_PMENU * mp;
   STEM_PDM sp;
   Boolean fg;

   w = RIPinq_window(rgn);
   mp = (STEM_PMENU *) ASHinq_user_data(w);
   if (mp != NULL && mp->ripwin == w) {
      ASHpop(w);
      if (mp->routine != NULL) {
	 sp = mp->pdm;
	 fg = (*(mp->routine))(sp->data,mp->name,NULL,sp->window);
       }
      else fg = TRUE;
    }
   else fg = FALSE;

   return fg;
};





/*ARGSUSED*/

static Integer
handle_ripoff_remove(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   STEM_PMENU * mp;

   w = RIPinq_window(rgn);
   mp = (STEM_PMENU *) ASHinq_user_data(w);
   if (mp != NULL && mp->ripwin == w) {
      ASHremove(mp->ripwin);
      mp->ripwin = NULL;
      display_title(mp->pdm);
    };

   return TRUE;
};





static Integer
handle_ripoff_btn(mp,b,sw)
   STEM_PMENU * mp;
   Integer b;
   Integer sw;
{
   STEM_PDM sp;
   Boolean fg;

   if (sw & RIP_BTN_DOWN) return FALSE;

   sp = mp->pdm;

   if (!mp->enable || !mp->btn[b].enable) fg = FALSE;
   else if (mp->btn[b].routine != NULL) {
      fg = (*(mp->btn[b].routine))(sp->data,mp->name,
					      mp->btn[b].name,
					      sp->window);
    }
   else fg = TRUE;

   return fg;
};





static void
ripoff_reset(mp)
   STEM_PMENU * mp;
{
   STEM_BUTTON * sbtn;
   ASH_WINDOW w;
   Integer i;

   if (!ASHinq_valid_window(mp->ripwin)) {
      mp->ripwin = NULL;
      return;
    };

   sbtn = (STEM_BUTTON *) alloca((mp->numbtn+1)*sizeof(STEM_BUTTON));

   for (i = 0; i < mp->numbtn; ++i) {
      sbtn[i].name = mp->btn[i].name;
      sbtn[i].flags = STEM_FLAG_NORMAL;
      if (!mp->btn[i].enable) sbtn[i].flags |= STEM_FLAG_LOLITE;
      if (mp->btn[i].select) sbtn[i].flags |= STEM_FLAG_CHOICE;
    };
   sbtn[i].name = NULL;
   sbtn[i].flags = 0;

   w = LEAFinq_window(mp->ripwin,RIPOFF_BTNS,0);

   if (w == NULL) {
      ASHremove(mp->ripwin);
      mp->ripwin = NULL;
      return;
    };

   ASHfont(w,mp->pdm->btnfont);
   STEMmenu_refresh(w,sbtn);
   ASHvisible(w,TRUE);
};






static void
ripoff_visible(p,fg)
   STEM_PDM p;
   Boolean fg;
{
   Integer m;

   for (m = 0; m < p->nummenu; ++m) {
      if (p->menu[m].ripwin != NULL) {
	 if (ASHinq_valid_window(p->menu[m].ripwin))
	    ASHvisible(p->menu[m].ripwin,fg);
       };
    };
};






/* end of stempdm.c */

