/************************************************************************/
/*									*/
/*		stempanel.c						*/
/*									*/
/*	Panel tool -- build a panel from a set of buttons		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define EXTRA_SPACE	2






/************************************************************************/
/*									*/
/*	Data types							*/
/*									*/
/************************************************************************/


typedef struct _STEM_PANEL *	STEM_PANEL;



typedef struct _STEM_PANEL {
   ASH_WINDOW window;
   Integer nbtn;
   Function_Ptr rtn;
   STEM_PANEL_BTN * btns;
   STEM_BTN * btndata;
   Universal data;
   ASH_FONT font;
   ASH_COLOR fg,bg;
   Boolean resize;
   Boolean samesize;
   STEM_PANEL_BTN * downbtn;
} STEM_PANEL_INFO;






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 	panel_control();
static	void		panel_refresh();
static	Integer 	panel_btn_handler();
static	void		free_panel();
static	void		free_buttons();
static	void		copy_buttons();
static	void		draw_panel();
static	STEM_BTN	draw_panel_btn();






/************************************************************************/
/*									*/
/*	STEM_panel_init -- module initialization			*/
/*									*/
/************************************************************************/


void
STEM_panel_init()
{
};





/************************************************************************/
/*									*/
/*	STEMpanel_define -- define a panel				*/
/*									*/
/************************************************************************/


void
STEMpanel_define(w,btns,rtn,data,flags)
   ASH_WINDOW w;
   STEM_PANEL_BTN btns[];
   Function_Ptr rtn;
   Universal data;
   STEM_PANEL_FLAGS flags;
{
   STEM_PANEL sp;

   sp = PALLOC(STEM_PANEL_INFO);

   sp->window = w;
   sp->rtn = rtn;
   sp->data = data;
   sp->font = ASHinq_font(w);
   sp->fg = ASHinq_color(w);
   sp->bg = ASHinq_background_color(w);
   sp->resize = TRUE;
   sp->samesize = ((flags & STEM_PANEL_SAME_SIZE) != 0);
   sp->downbtn = NULL;

   copy_buttons(sp,btns);

   ASHset_menu_data(w,sp);
   ASHset_region_refresh(w,panel_refresh);
   ASHset_control(w,panel_control);

   if (ASHinq_visible(w)) draw_panel(sp);
};





/************************************************************************/
/*									*/
/*	STEMpanel_reset -- redefine a panel				*/
/*									*/
/************************************************************************/


void
STEMpanel_reset(w,btns)
   ASH_WINDOW w;
   STEM_PANEL_BTN btns[];
{
   STEM_PANEL sp;

   sp = (STEM_PANEL) ASHinq_menu_data(w);
   if (sp == NULL) return;

   free_buttons(sp);
   copy_buttons(sp,btns);

   if (ASHinq_visible(w)) draw_panel(sp);
};





/************************************************************************/
/*									*/
/*	STEMpanel_btn_select -- select a button on a panel		*/
/*									*/
/************************************************************************/


void
STEMpanel_btn_select(w,btn,fg)
   ASH_WINDOW w;
   String btn;
   Boolean fg;
{
   STEM_PANEL sp;
   Integer i;

   sp = (STEM_PANEL) ASHinq_menu_data(w);
   if (sp == NULL) return;

   for (i = 0; i < sp->nbtn; ++i) {
      if (STREQL(sp->btns[i].text,btn)) {
	 if (sp->btndata[i] != NULL) {
	    STEMbtn_select(sp->btndata[i],fg);
	    break;
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	panel_control -- handle control messages for panel		*/
/*									*/
/************************************************************************/


static Integer
panel_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_PANEL sp;

   sp = (STEM_PANEL) ASHinq_menu_data(w);
   if (sp == NULL) return ASH_CONTROL_REJECT;

   sp->downbtn = NULL;

   if (STREQL(msg,"ASH$RESIZE")) {
      sp->resize = TRUE;
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      free_panel(sp);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	panel_refresh -- handle refresh of a panel			*/
/*									*/
/************************************************************************/


static void
panel_refresh(w)
   ASH_WINDOW w;
{
   STEM_PANEL sp;

   sp = (STEM_PANEL) ASHinq_menu_data(w);
   if (sp == NULL) return;

   if (sp->resize) draw_panel(sp);
   else STEMbtn_window_draw(sp->window);
};





/************************************************************************/
/*									*/
/*	panel_btn_handler -- handle hits on panel buttons		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static Integer
panel_btn_handler(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   ASH_WINDOW w;
   STEM_PANEL_BTN * sb;
   STEM_PANEL sp;
   Boolean fg;

   w = RIPinq_window(rgn);
   sb = (STEM_PANEL_BTN *) RIPinq_data(rgn);
   sp = (STEM_PANEL) ASHinq_menu_data(w);

   if (btn & RIP_BTN_NONE) {
      sp->downbtn = NULL;
      return FALSE;
    };

   if (sb == NULL || sp == NULL) return FALSE;

   if (btn & RIP_BTN_DOWN) {
      sp->downbtn = sb;
      return FALSE;
    };

   if ((btn & RIP_BTN_TAP) == 0) {
      if (sp->downbtn != NULL && sp->downbtn != sb) {
	 sp->downbtn = NULL;
	 return FALSE;
       }
    };

   sp->downbtn = NULL;

   if (sp->rtn != NULL) {
      fg = (*sp->rtn)(sb->data,sp->data,w,sb->text);
    }
   else fg = TRUE;

   return fg;
};





/************************************************************************/
/*									*/
/*	free_panel -- remove a panel when window is cleared		*/
/*	free_buttons -- free panel buttons				*/
/*									*/
/************************************************************************/


static void
free_panel(sp)
   STEM_PANEL sp;
{
   if (sp->window != NULL) {
      ASHset_menu_data(sp->window,NULL);
      sp->window = NULL;
    };

   free_buttons(sp);

   free(sp);
};





static void
free_buttons(sp)
   STEM_PANEL sp;
{
   Integer i,ct;

   ct = sp->nbtn;
   if (ct != 0) {
      sp->nbtn = 0;
      for (i = 0; i < ct; ++i) {
	 SFREE(sp->btns[i].text);
	 if (sp->btns[i].color != NULL) SFREE(sp->btns[i].color);
       };
      free(sp->btns);
      sp->btns = NULL;
      free(sp->btndata);
      sp->btndata = NULL;
    };
};





/************************************************************************/
/*									*/
/*	copy_buttons -- copy panel buttons from user to our structure	*/
/*									*/
/************************************************************************/


static void
copy_buttons(sp,btns)
   STEM_PANEL sp;
   STEM_PANEL_BTN btns[];
{
   Integer i,ct;

   if (btns == NULL) ct = 0;
   else {
      for (ct = 0; ; ++ct) {
	 if (btns[ct].text == NULL) break;
       };
    };
   sp->nbtn = ct;

   if (ct == 0) {
      sp->btns = NULL;
      sp->btndata = NULL;
    }
   else {
      sp->btns = (STEM_PANEL_BTN *) calloc(ct,sizeof(STEM_PANEL_BTN));
      sp->btndata = (STEM_BTN *) calloc(ct,sizeof(STEM_BTN));
    };

   for (i = 0; i < ct; ++i) {
      sp->btns[i] = btns[i];
      sp->btns[i].text = SALLOC(btns[i].text);
      if (btns[i].color != NULL)
	 sp->btns[i].color = SALLOC(btns[i].color);
      sp->btndata[i] = NULL;
    };
};





/************************************************************************/
/*									*/
/*	draw_panel -- layout and draw the panel 			*/
/*									*/
/************************************************************************/


static void
draw_panel(sp)
   STEM_PANEL sp;
{
   Integer mxx,mxy,cx,cy;
   Integer lx,by,rx,ty;
   Integer i,j,k,ncol,nrow;
   STEM_BTN sb;
   Integer xsz,ysz;
   Integer dx,dy;

   ASHfont(sp->window,sp->font);
   ASHcolor(sp->window,sp->fg);
   ASHbackground_color(sp->window,sp->bg);
   ASHtext_background_color(sp->window,-1);

   mxx = 1;
   mxy = 1;
   for (i = 0; i < sp->nbtn; ++i) {
      STEMbtn_size(sp->window,sp->btns[i].text,&xsz,&ysz);
      mxx = MAX(mxx,xsz);
      mxy = MAX(mxy,ysz);
      sp->btndata[i] = NULL;
    };

   ASHinq_size(sp->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);

   ncol = abs(rx-lx)/(mxx+EXTRA_SPACE);
   if (ncol == 0) ncol = 1;
   nrow = abs(by-ty)/(mxy+EXTRA_SPACE);
   if (nrow == 0) nrow = 1;

   while (ncol > 1 && nrow > 1 && sp->nbtn <= (ncol-1)*(nrow-1)) {
      ncol--;
      nrow--;
    };

   while (ncol > 1 && sp->nbtn <= (ncol-1)*nrow) ncol--;
   while (nrow > 1 && sp->nbtn <= ncol*(nrow-1)) nrow--;

   if (sp->nbtn > ncol*nrow) {
      ncol = (sp->nbtn+nrow-1)/nrow;
    };

   cx = (rx-lx)/ncol;
   cy = (by-ty)/nrow;

   STEMbtn_window_remove(sp->window);
   RIPremove_window_regions(sp->window);

   sb = STEMbtn_define(sp->window,1,1,1,1,"",
			  STEM_BTN_NO_DRAW,
			  NULL,NULL);
   STEMbtn_background(sb,0,0,0,0);

   for (i = 0; i < nrow; ++i) {
      for (j = 0; j < ncol; ++j) {
	 k = i*ncol+j;
	 if (k >= sp->nbtn) continue;
	 sb = draw_panel_btn(sp,&sp->btns[k],
				lx+cx*j,ty+cy*(i+1)-dy,lx+cx*(j+1)-dx,ty+cy*i,mxx,mxy);
	 sp->btndata[k] = sb;
       };
    };

   sp->resize = FALSE;
};





/************************************************************************/
/*									*/
/*	draw_panel_btn -- draw a single button				*/
/*									*/
/************************************************************************/


static STEM_BTN
draw_panel_btn(sp,sb,lx,by,rx,ty,mxx,mxy)
   STEM_PANEL sp;
   STEM_PANEL_BTN * sb;
   Integer lx,by,rx,ty;
   Integer mxx,mxy;
{
   ASH_COLOR c,cf,tc;
   STEM_BTN sbtn;
   Integer dx,dy;
   STEM_BTN_FLAGS fgs;

   dx = (lx < rx ? 1 : -1);
   dy = (ty < by ? 1 : -1);
   if (mxx & 1) mxx++;
   if (mxy & 1) mxy++;

   fgs = STEM_BTN_SENSE_INOUT | STEM_BTN_NO_FILL;

   if (sb->color != 0) {
      c = ASHlookup_color(sp->window,sb->color);
    }
   else {
      c = sp->fg;
    };

   if (sp->samesize) {
      if (abs(rx-lx) > mxx) {
	 lx += (abs(rx-lx) - mxx)/2*dx;
	 rx = lx + (mxx-1)*dx;
       };
      if (abs(by-ty) > mxy) {
	 ty += (abs(by-ty) - mxy)/2*dy;
	 by = ty + (mxy-1)*dy;
       };
      fgs |= STEM_BTN_FULL_SIZE;
    };

   cf = ASHcolor(sp->window,c);
   tc = ASHinq_text_color(sp->window);
   if (tc == c) ASHtext_color(sp->window,tc^1);

   sbtn = STEMbtn_define(sp->window,lx,by,rx,ty,sb->text,fgs,
			    sb,panel_btn_handler);

   ASHcolor(sp->window,cf);
   ASHtext_color(sp->window,tc);

   return sbtn;
};





/* end of stempanel.c */
