/************************************************************************/
/*									*/
/*		stemmenu.c						*/
/*									*/
/*	Sticky menu module for STEM menu package			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_DIFF		4	/* max buttons to rewrite	*/

#define AVG_SIZE_STRING "Xp^_XXXXXXXXXXXXXXXX"
#define SCROLL_BAR_WIDTH	40

#define BORDER_X	2
#define BORDER_Y	2





/************************************************************************/
/*									*/
/*	Data Type Definitions						*/
/*									*/
/************************************************************************/


typedef struct _STEM_MENU *	STEM_MENU;



typedef struct _STEM_IBTN {
   String	name;
   Integer	flags;
   STEM_BTN	btn;
} STEM_IBTN;





typedef struct _STEM_MENU {
   ASH_WINDOW	window;
   Integer	btn_count;
   Integer	user_data;
   Function_Ptr call_rtn;
   Integer	btnmask;
   STEM_IBTN *	buttons;
   Integer	ncol;
   Integer	nrow;
   Integer	toprow;
   Integer	colwidth;
   Integer	rowht;
   Integer	dx;
   Integer	dy;
   ASH_WINDOW	scrollwin;
   Boolean	scroll;
   Boolean	resize;
   Boolean	onecol;
   Integer	norm_color;
   Integer	lo_color;
   Integer	choice_color;
   ASH_FONT	font;
   THREAD	thread;
   STEM_BTN	dummy_btn;
} STEM_MENU_B;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 	scroll_bar_width;
static	String		norm_color;
static	String		lo_color;
static	String		hi_color;
static	Integer 	border_x;
static	Integer 	border_y;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 	stem_control();
static	void		stem_refresh();
static	Integer 	stem_btn();
static	void		stem_scroll();

static	void		copy_buttons();
static	void		free_data();
static	void		layout_menu();
static	void		draw_menu();
static	void		redraw_button();
static	void		put_button();





/************************************************************************/
/*									*/
/*	STEM_menu_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
STEM_menu_init()
{
   String s;

   ITRACE("STEM_menu_init");

   s = ASHinq_resource("menu.scroll_width");
   if (s == NULL) scroll_bar_width = 0;
   else scroll_bar_width = atol(s);
   if (scroll_bar_width == 0) scroll_bar_width = SCROLL_BAR_WIDTH;

   s = ASHinq_resource("menu.norm_color");
   if (s == NULL) norm_color = NULL;
   else norm_color = s;
   s = ASHinq_resource("menu.lo_color");
   if (s == NULL) lo_color = NULL;
   else lo_color = s;
   s = ASHinq_resource("menu.hi_color");
   if (s == NULL) hi_color = NULL;
   else hi_color = s;

   s = ASHinq_resource("menu.border_x");
   if (s != NULL) border_x = atol(s);
   else border_x = BORDER_X;
   s = ASHinq_resource("menu.border_y");
   if (s != NULL) border_y = atol(s);
   else border_y = BORDER_Y;
};





/************************************************************************/
/*									*/
/*	STEMmenu -- define a new menu					*/
/*									*/
/************************************************************************/


void
STEMmenu(w,rtn,data,bmsk,btns)
   ASH_WINDOW w;
   Function_Ptr rtn;
   Integer data;
   Integer bmsk;
   STEM_BUTTON *btns;
{
   register STEM_MENU sm;

   ENTER("STEMmenu 0x%x 0x%x 0x%x 0x%x",w,data,bmsk,btns);

   sm = (STEM_MENU) calloc(1,sizeof(STEM_MENU_B));

   sm->window = w;
   sm->call_rtn = rtn;
   sm->user_data = data;
   sm->btnmask = bmsk;
   sm->thread = THREADinq_current();
   sm->onecol = FALSE;
   sm->font = ASHinq_font(w);
   sm->dummy_btn = NULL;

   ASHset_control(w,stem_control);
   ASHset_menu_data(w,sm);
   ASHset_region_refresh(w,stem_refresh);
   ASHnew_drawinfo(w);
   BIOnew_input_window(w);
   BIOset_window_thread(w,sm->thread);

   copy_buttons(sm,btns);

   sm->ncol = 0;
   sm->nrow = 0;
   sm->toprow = 0;
   sm->scroll = FALSE;
   sm->scrollwin = NULL;
   sm->resize = TRUE;

   STEMmenu_colors(w,norm_color,lo_color,hi_color);

   layout_menu(sm);
   draw_menu(sm,TRUE,TRUE);
};





/************************************************************************/
/*									*/
/*	STEMmenu_refresh -- refresh display				*/
/*									*/
/************************************************************************/


void
STEMmenu_refresh(w,btns)
   ASH_WINDOW w;
   STEM_BUTTON * btns;
{
   register STEM_MENU sm;
   register Integer i,ndiff;
   register Integer fdiff;
   STEM_BUTTON dummy[1];

   ENTER("STEMmenu_refresh 0x%x",w);

   sm = (STEM_MENU) ASHinq_menu_data(w);
   sm->font = ASHinq_font(w);

   if (btns == NULL) {
      btns = dummy;
      dummy[0].name = NULL;
      dummy[0].flags = 0;
    };

   ndiff = 0;
   fdiff = 0;

   for (i = 0; btns != NULL && btns[i].name != NULL; ++i);
   if (sm->btn_count != i) ndiff = MAX_DIFF;

   for (i = 0; btns != NULL && btns[i].name != NULL && ndiff < MAX_DIFF; ++i) {
      if (STRNEQ(btns[i].name,sm->buttons[i].name) ||
	     btns[i].flags != sm->buttons[i].flags) {
	 if (ndiff == 0) fdiff = i;
	 ++ndiff;
       };
    };

   if (ndiff >= MAX_DIFF) {
      copy_buttons(sm,btns);
      layout_menu(sm);
      draw_menu(sm,TRUE,TRUE);
    }
   else if (ndiff > 0) {
      ASHbatch_mode(TRUE);
      for (i = fdiff; btns[i].name != NULL && ndiff > 0; ++i) {
	 if (STRNEQ(btns[i].name,sm->buttons[i].name) ||
		btns[i].flags != sm->buttons[i].flags) {
	    sm->buttons[i].flags = btns[i].flags;
	    free(sm->buttons[i].name);
	    sm->buttons[i].name = SALLOC(btns[i].name);
	    redraw_button(sm,i);
	    --ndiff;
	  };
       };
      ASHbatch_mode(FALSE);
    };
};





/************************************************************************/
/*									*/
/*	STEMmenu_colors -- set menu colors				*/
/*									*/
/************************************************************************/


void
STEMmenu_colors(w,norm,lo,choice)
   ASH_WINDOW w;
   String norm;
   String lo;
   String choice;
{
   STEM_MENU sm;

   ENTER("STEMmenu_colors 0x%x",w);

   sm = (STEM_MENU) ASHinq_menu_data(w);

   if (sm == NULL) return;

   if (ASHinq_configuration_depth(w) == 1) {
      sm->norm_color = -1;
      sm->lo_color = -1;
      sm->choice_color = -1;
    }
   else {
      if (norm == NULL) sm->norm_color = -1;
      else sm->norm_color = ASHlookup_color(w,norm);
      if (lo == NULL) sm->lo_color = -1;
      else sm->lo_color = ASHlookup_color(w,lo);
      if (choice == NULL) sm->choice_color = -1;
      else sm->choice_color = ASHlookup_color(w,choice);
    };
};





/********************************************************************************/
/*										*/
/*	STEMmenu_remove -- remove a static menu 				*/
/*										*/
/********************************************************************************/


void
STEMmenu_remove(w)
   ASH_WINDOW w;
{
   STEM_MENU sm;

   ENTER("STEMmenu_remove 0x%x",w);

   sm = (STEM_MENU) ASHinq_menu_data(w);

   if (sm == NULL) return;

   ASHset_menu_data(w,NULL);

   free_data(sm);
};





/************************************************************************/
/*									*/
/*	stem_control -- handle ASH control messages for window		*/
/*									*/
/************************************************************************/


static Integer
stem_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register STEM_MENU sm;

   ITRACE("stem_control %s 0x%x",msg,w);

   sm = (STEM_MENU) ASHinq_menu_data(w);

   if (sm == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$RESIZE")) {
      layout_menu(sm);
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      ASHset_menu_data(w,NULL);
      free_data(sm);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	stem_refresh -- handle window refresh request			*/
/*									*/
/************************************************************************/


static void
stem_refresh(w)
   ASH_WINDOW w;
{
   register STEM_MENU sm;

   ITRACE("stem_refresh 0x%x",w);

   sm = (STEM_MENU) ASHinq_menu_data(w);

   if (sm != NULL) {
      draw_menu(sm,FALSE,TRUE);
    };
};





/************************************************************************/
/*									*/
/*	stem_btn -- handle button press 				*/
/*									*/
/************************************************************************/


static Integer
stem_btn(x,y,ch,btns,rgn)
   Integer x,y;
   Integer ch;
   Integer btns;
   RIP_REGION rgn;
{
   register ASH_WINDOW w;
   register STEM_MENU sm;
   register Integer btn;
   register Integer fg;

   ITRACE("stem_btn %d %d 0x%x 0x%x 0x%x",x,y,ch,btns,rgn);

   w = RIPinq_window(rgn);
   btn = (Integer) RIPinq_data(rgn);

   sm = (STEM_MENU) ASHinq_menu_data(w);

   if ((btns & sm->btnmask) != btns) return FALSE;

   if (sm->call_rtn != NULL) {
      fg = (*(sm->call_rtn))(sm->user_data,btn,btns,w);
    }
   else fg = TRUE;

   return fg;
};





/************************************************************************/
/*									*/
/*	stem_scroll -- handle scroll request				*/
/*									*/
/************************************************************************/


static void
stem_scroll(sm,dir,val)
   STEM_MENU sm;
   Integer dir;
   Integer val;
{
   register Integer i;

   ITRACE("stem_scroll 0x%x %d %d",sm,dir,val);

   if (dir < 0) sm->toprow = sm->toprow-val;
   else if (dir > 0) {
      i = (sm->btn_count+sm->ncol-1)/sm->ncol - 1;
      val = MIN(val,i-sm->toprow);
      if (val > 0) sm->toprow += val;
    }
   else sm->toprow = val;

   if (sm->toprow < 0) sm->toprow = 0;

   draw_menu(sm,FALSE,FALSE);
};





/************************************************************************/
/*									*/
/*	copy_buttons -- copy user buttons to our own structure		*/
/*									*/
/************************************************************************/


static void
copy_buttons(sm,btns)
   STEM_MENU sm;
   STEM_BUTTON * btns;
{
   register Integer i;
   STEM_BTN sb;

   DTRACE("copy_buttons 0x%x",sm);

   if (sm->buttons != NULL) {
      for (i = 0; i < sm->btn_count; ++i) {
	 free(sm->buttons[i].name);
	 if (sm->buttons[i].btn != NULL && btns != NULL && sm->window != NULL)
	    STEMbtn_remove(sm->buttons[i].btn);
       };
      free(sm->buttons);
      sm->buttons = NULL;
    };

   if (sm->dummy_btn != NULL && sm->window != NULL) {
      sb = sm->dummy_btn;
      sm->dummy_btn = NULL;
      STEMbtn_remove(sb);
    };

   if (btns == NULL) {
      sm->btn_count = 0;
      sm->buttons = NULL;
      sm->onecol = TRUE;
      return;
    };

   for (i = 0; btns[i].name != NULL; ++i);
   sm->btn_count = i;
   sm->onecol = FALSE;

   sm->buttons = (STEM_IBTN *) calloc(i+1,sizeof(STEM_IBTN));
   for (i = 0; btns[i].name != NULL; ++i) {
      sm->buttons[i].name = SALLOC(btns[i].name);
      sm->buttons[i].flags = btns[i].flags;
      sm->buttons[i].btn = NULL;
      if (btns[i].flags & STEM_FLAG_ONECOL) {
	 sm->onecol = TRUE;
	 sm->buttons[i].flags &= ~STEM_FLAG_ONECOL;
       };
    };
};




/************************************************************************/
/*									*/
/*	free_data -- free STEM DATA for a window being deleted		*/
/*									*/
/************************************************************************/


static void
free_data(sm)
   STEM_MENU sm;
{
   DTRACE("free_data 0x%x",sm);

   ASHset_menu_data(sm->window,NULL);
   ASHremove_control(sm->window,stem_control);

   sm->window = NULL;
   copy_buttons(sm,NULL);

   free(sm);
};





/************************************************************************/
/*									*/
/*	layout_menu -- layout a new menu				*/
/*									*/
/************************************************************************/


static void
layout_menu(sm)
   STEM_MENU sm;
{
   Integer x,y;
   Integer lx,by,rx,ty;

   DTRACE("layout_menu 0x%x",sm);

   ASHfont(sm->window,sm->font);
   ASHinq_size(sm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   STEMbtn_size(sm->font,AVG_SIZE_STRING,&x,&y);
   y += 2*border_y;
   x += 2*border_x;

   sm->nrow = (abs(ty-by)+1)/y;
   sm->ncol = (abs(rx-lx)+1)/x;
   if (sm->onecol) sm->ncol = 1;
   if (sm->nrow < 1) sm->nrow = 1;
   if (sm->ncol < 1) sm->ncol = 1;

   if (sm->ncol*sm->nrow < sm->btn_count) {
      sm->scroll = TRUE;
      sm->ncol = (abs(rx-lx)+1-scroll_bar_width)/x;
      if (sm->onecol || sm->ncol < 1) sm->ncol = 1;
    }
   else {
      sm->scroll = FALSE;
      if (sm->btn_count == 0) sm->ncol = 1;
      else while (sm->btn_count < (sm->ncol-1)*sm->nrow) sm->ncol--;
    };

   sm->toprow = 0;
   sm->resize = TRUE;
};





/************************************************************************/
/*									*/
/*	draw_menu -- actually draw a menu				*/
/*									*/
/************************************************************************/



static void
draw_menu(sm,rsz,rdr)
   STEM_MENU sm;
   Boolean rsz;
   Boolean rdr;
{
   Integer lx,by,rx,ty;
   register Integer x,y;
   register Integer i,j,k;
   STEM_BTN sb;

   DTRACE("draw_menu 0x%x %d %d",sm,rsz,rdr);

   if (sm->resize) {
      sm->resize = FALSE;
      rsz = TRUE;
    };

   if (!ASHinq_visible(sm->window)) {
      if (rsz) sm->resize = TRUE;
      return;
    };

   ASHbatch_mode(TRUE);

   ASHfont(sm->window,sm->font);
   ASHinq_size(sm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   if (sm->norm_color >= 0) ASHcolor(sm->window,sm->norm_color);

   if (rsz) {
      sm->dx = (lx < rx ? 1 : -1);
      sm->dy = (ty < by ? 1 : -1);

      if (sm->scroll) {
	 sm->colwidth = (abs(rx-lx)+1-scroll_bar_width)/sm->ncol;
	 x = rx-sm->dx*scroll_bar_width+1+3;
	 if (sm->scrollwin == NULL) {
	    sm->scrollwin = ASHcreate(sm->window,x,by,0,0,rx-x,abs(ty-by),0,
					 ASH_WINDOW_NOSAVE);
	    STEMscroll(sm->scrollwin,stem_scroll,sm);
	    STEMscroll_thread(sm->scrollwin,sm->thread);
	  }
	 else {
	    ASHresize(sm->scrollwin,x,by,0,0,rx-x,abs(ty-by));
	    ASHvisible(sm->scrollwin,TRUE);
	  };
       }
      else {
	 sm->colwidth = (abs(rx-lx)+1)/sm->ncol;
       };

      sm->rowht = (abs(ty-by)+1)/sm->nrow;
    };

   if (rdr) {
      if (sm->scroll) {
	 x = rx-sm->dx*(scroll_bar_width+1);
	 for (i = 0; i < 3; ++i) {
	    ASHline(sm->window,x,by,x,ty);
	    ++x;
	  };
       };
      for (i = 1; i < sm->ncol; ++i) {	/* separate columns		*/
	 x = lx+sm->dx*i*sm->colwidth;
       };
    };

   STEMbtn_window_remove(sm->window);
   sm->dummy_btn = NULL;
   for (i = 0; i < sm->btn_count; ++i) {
      sm->buttons[i].btn = NULL;
    };

   sb = STEMbtn_define(sm->window,1,1,1,1,"",STEM_BTN_NO_DRAW,NULL,NULL);
   STEMbtn_background(sb,0,0,0,0);
   sm->dummy_btn = sb;
   if (sm->scrollwin != NULL && !sm->scroll) {
      ASHvisible(sm->scrollwin,FALSE);
    };

   for (i = 0; i < sm->ncol; ++i) {
      x = lx+sm->dx*i*sm->colwidth;
      for (j = 0; j < sm->nrow; ++j) {
	 y = ty+sm->dy*j*sm->rowht;
	 k = i+(j+sm->toprow)*sm->ncol;
	 if (k >= sm->btn_count) continue;
	 put_button(sm,k,x,y);
       };
    };

   if (sm->scroll) {
      i = (sm->btn_count+sm->ncol-1)/sm->ncol-1;
      STEMscroll_set(sm->scrollwin,0,i,sm->toprow,sm->toprow+sm->nrow-1);
    };

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	redraw_button -- redraw a button				*/
/*									*/
/************************************************************************/


static void
redraw_button(sm,k)
   STEM_MENU sm;
   Integer k;
{
   Integer lx,by,rx,ty;
   register Integer x,y;
   register Integer i,j;

   DTRACE("redraw_button 0x%x %d",sm,k);

   if (!ASHinq_visible(sm->window)) return;

   ASHinq_size(sm->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHfont(sm->window,sm->font);

   i = k % sm->ncol;
   j = (k / sm->ncol) - sm->toprow;

   if (j >= 0 && j < sm->nrow) {
      x = lx+sm->dx*i*sm->colwidth;
      y = ty+sm->dy*j*sm->rowht;

      put_button(sm,k,x,y);
    };
};





/************************************************************************/
/*									*/
/*	put_button -- actually output a button				*/
/*									*/
/************************************************************************/


static void
put_button(sm,k,x,y)
   STEM_MENU sm;
   Integer k;
   Integer x,y;
{
   Integer by,rx;
   STEM_BTN sb;

   DTRACE("put_button 0x%x %d %d %d",sm,k,x,y);

   rx = x+sm->dx*(sm->colwidth-2);
   by = y+sm->dy*(sm->rowht-1);

   if (sm->buttons[k].btn != NULL) {
      STEMbtn_remove(sm->buttons[k].btn);
      sm->buttons[k].btn = NULL;
    };

   sb = STEMbtn_define(sm->window,x+sm->dx*border_x,by-sm->dy*border_y,
			  rx-sm->dx*border_x,y+sm->dy*border_y,
			  sm->buttons[k].name,
			  STEM_BTN_NO_DRAW|STEM_BTN_LEFT|STEM_BTN_FULL_SIZE|
			     STEM_BTN_SENSE_FLIP,
			  k,stem_btn);
   if (sm->buttons[k].flags & STEM_FLAG_LOLITE) STEMbtn_enable(sb,FALSE);
   if (sm->buttons[k].flags & STEM_FLAG_CHOICE) STEMbtn_select(sb,TRUE);
   if (sm->choice_color >= 0) STEMbtn_set_select(sb,sm->choice_color);
   STEMbtn_draw(sb);
};





/* end of stemmenu.c */
