/************************************************************************/
/*									*/
/*		stemdialog.c						*/
/*									*/
/*	Dialog interface routines for STEM menu package 		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"
#include "edt.h"
#include <varargs.h>



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TOP_MARGIN	8

#define BORDER_WIDTH	3
#define BORDER_COLOR	"black"

#define EDIT_BORDER	2

#define MAX_FIELD	256
#define NUM_COLOR	10

#define NUM_FONT	4
#define FONT_FAMILY	"fixed"
#define FONT_DELTA	2
#define FONT_PROP	ASH_FONT_ROMAN|ASH_FONT_BOLD
#define FONT0		"fixed",10,ASH_FONT_ROMAN|ASH_FONT_BOLD
#define FONT1		"fixed",12,ASH_FONT_ROMAN|ASH_FONT_BOLD
#define FONT2		"fixed",14,ASH_FONT_ROMAN|ASH_FONT_BOLD
#define FONT3		"fixed",16,ASH_FONT_ROMAN|ASH_FONT_BOLD

#define BOX_SIZE	17
#define BOX_X_SIZE	box_x_size
#define BOX_Y_SIZE	box_y_size
#define BOX_Y_OFFSET	0

#define RECT_SIZE	21
#define RECT_X_SIZE	rect_x_size
#define RECT_Y_SIZE	rect_y_size
#define RECT_Y_OFFSET	0

#define SELECT_X_SIZE(k) (select_x_size+(k)*fonts[1].x_size)
#define SELECT_Y_SIZE	(fonts[1].y_size+select_y_size)
#define SELECT_Y_OFFSET (fonts[1].y_offset+select_y_size/2-1)

#define EDIT_X_SIZE(k)	(text_x_size+(k)*fonts[1].x_size+16)
#define EDIT_Y_SIZE(k)	(fonts[1].y_size*(k)+text_y_size)
#define EDIT_Y_OFFSET	(fonts[1].y_offset+text_y_size/2-1)
#define DFLT_EDIT_SIZE	32
#define DFLT_EDIT_LINES 1

#define LIST_X_SIZE(k)	(text_x_size+(k)*fonts[1].x_size)
#define LIST_Y_SIZE(k)	(fonts[1].y_size*(k)+text_y_size)
#define LIST_Y_OFFSET	(fonts[1].y_offset+text_y_size/2-1)
#define DFLT_LIST_SIZE	32
#define DFLT_LIST_LINES 1

#define TEXT_X_DELTA	6
#define TEXT_Y_DELTA	6
#define TEXT_X_SIZE(k)	(text_x_size+(k)*fonts[1].x_size)
#define TEXT_Y_SIZE	EDIT_Y_SIZE(1)
#define TEXT_Y_OFFSET	EDIT_Y_OFFSET
#define DFLT_TEXT_SIZE	32

#define INT_X_SIZE(k)	TEXT_X_SIZE(k)
#define INT_Y_SIZE	EDIT_Y_SIZE(1)
#define INT_Y_OFFSET	EDIT_Y_OFFSET
#define DFLT_INT_SIZE	8

#define FLT_X_SIZE(k)	TEXT_X_SIZE(k)
#define FLT_Y_SIZE	EDIT_Y_SIZE(1)
#define FLT_Y_OFFSET	EDIT_Y_OFFSET
#define DFLT_FLT_SIZE	16

#define MARGIN		4
#define BORDER		1
#define NEW_ADDON	4

#define CURSOR_COLOR	"red"
#define TEXTON_COLOR	"blue"
#define TEXTOFF_COLOR	"gray"

#define MAX_COL 	10




/************************************************************************/
/*									*/
/*	Data types							*/
/*									*/
/************************************************************************/


typedef struct _STEM_DIALOG *	STEM_DIALOG;
typedef struct _STEM_DFLD   *	STEM_DFLD;



typedef enum _STEM_DTYPE {
   STEM_DTYPE_OPTION,
   STEM_DTYPE_TEXT,
   STEM_DTYPE_BOXOPT,
   STEM_DTYPE_BBOXOPT,
   STEM_DTYPE_ACCEPT,
   STEM_DTYPE_CANCEL,
   STEM_DTYPE_BOPT,
   STEM_DTYPE_FLAG,
   STEM_DTYPE_INT,
   STEM_DTYPE_FLT,
   STEM_DTYPE_DBL,
   STEM_DTYPE_EDIT,
   STEM_DTYPE_LIST,
} STEM_DTYPE;




typedef struct _STEM_DFLD {
   Integer	lx,by,rx,ty;
   STEM_DTYPE	type;
   STEM_BTN	btn;
   Integer	index;
   Universal	value;
   Integer	color;
   Integer	fill;
   EDT_ID	editid;
   ASH_WINDOW	subwin;
   Boolean	hilite;
   Boolean	select;
   Boolean	cancel;
   Boolean	accept;
   Boolean	edited;
} STEM_DFLD_B;




typedef struct _STEM_DIALOG {
   STEM_DIALOG	next;
   ASH_WINDOW	window;
   ASH_WINDOW	user_window;
   ASH_WINDOW	cover_window;
   ASH_WINDOW	root_window;
   Universal ** user_data;
   String	user_text;
   Integer	num_field;
   Integer	xsize,ysize;
   Integer	ypos[MAX_FIELD];
   Integer	linesize[MAX_FIELD];
   Integer	xcol[MAX_COL];
   Integer	text_fld;
   Integer	text_pos;
   Integer	backcolor;
   Integer	hilite;
   Integer	cursor_color;
   Integer	texton_color;
   Integer	textoff_color;
   Character	text_buf[256];
   Boolean	done;
   Boolean	abort;
   Boolean	drawn;
   Function_Ptr wait_rtn;
   STEM_DFLD_B	field[MAX_FIELD];
} STEM_DIALOG_B;




typedef struct	_STEM_DFONT {
   Integer	fontid;
   Integer	x_size;
   Integer	y_size;
   Integer	x_offset;
   Integer	y_offset;
} STEM_DFONT_B;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	STEM_DFONT_B	fonts[NUM_FONT];
static	STEM_DIALOG	all_dialog;
static	Boolean 	move_mouse;
static	Boolean 	root_windows;


typedef struct _COLOR_INFO {
   String name;
   String backname;
} COLOR_INFO;

static	COLOR_INFO	color_table[NUM_COLOR] = {
   { "black", "white" },
   { "red", "white" },
   { "orange", "white" },
   { "green", "white" },
   { "blue", "white" },
   { "maroon", "white" },
   { "black", "yellow" },
   { "yellow", "green" },
   { "white", "red" },
   { "white", "black" },
};


static	Integer 	box_x_size;
static	Integer 	box_y_size;

static	Integer 	rect_x_size;
static	Integer 	rect_y_size;

static	Integer 	text_x_size;
static	Integer 	text_y_size;

static	Integer 	select_x_size;
static	Integer 	select_y_size;

static	String		cursor_color;
static	String		texton_color;
static	String		textoff_color;





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	stem_do_dialog();
static	int		dialog_control();
static	int		dialog_user_control();
static	void		dialog_refresh();
static	void		setup_font();
static	Integer 	size_dialog();
static	Boolean 	setup_dialog();
static	Boolean 	draw_dialog();
static	Integer 	flushbuf();
static	void		draw_check();
static	void		draw_box();
static	void		draw_edit();
static	void		update_edit();
static	void		draw_list();
static	int		list_button();
static	void		draw_rect();
static	Boolean 	process_dialog();
static	void		remove_dialog();
static	int		dialog_hit();
static	int		dialog_track();
static	void		dialog_down();
static	Boolean 	dialog_btn();
static	void		update_btns();
static	Boolean 	dialog_typein();
static	void		select_text();
static	void		edit_border();
static	void		fix_edits();
static	void		fix_edit_field();
static	void		move_window();
static	void		get_button_size();
static	void		dialog_edt_mouse();
static	void		set_next_field();





/************************************************************************/
/*									*/
/*	STEM_dialog_init -- module initialization			*/
/*									*/
/************************************************************************/


void
STEM_dialog_init()
{
   String s,t;
   ASH_FONT ft;
   Integer i,xd,yd,xn,yn;
   Integer l,m,h;

   ITRACE("STEM_dialog_init");

   all_dialog = NULL;
   move_mouse = FALSE;
   root_windows = TRUE;

   s = ASHinq_resource("move_mouse");
   if (s == NULL) s = ASHinq_resource("MOVE_MOUSE");
   if (s == NULL) s = (String) getenv("MOVE_MOUSE");
   if (s != NULL && STRNEQ(s,"off")) move_mouse = TRUE;

   cursor_color = ASHinq_resource("dialog.cursor_color");
   if (cursor_color == NULL) cursor_color = CURSOR_COLOR;
   texton_color = ASHinq_resource("dialog.texton_color");
   if (texton_color == NULL) texton_color = TEXTON_COLOR;
   textoff_color = ASHinq_resource("dialog.textoff_color");
   if (textoff_color == NULL) textoff_color = TEXTOFF_COLOR;

   s = ASHinq_resource("font");
   t = ASHinq_resource("dialog.fontsize");
   if (t != NULL) {
      i = atol(t);
      if (i <= 0) t = NULL;
    };

   if (s == NULL && t == NULL) {
      setup_font(0,ASHfont_name(FONT0));
      setup_font(1,ASHfont_name(FONT1));
      setup_font(2,ASHfont_name(FONT2));
      setup_font(3,ASHfont_name(FONT3));
    }
   else if (s == NULL && t != NULL) {
      setup_font(0,ASHfont_name(FONT_FAMILY,i-FONT_DELTA,FONT_PROP));
      setup_font(1,ASHfont_name(FONT_FAMILY,i,FONT_PROP));
      setup_font(2,ASHfont_name(FONT_FAMILY,i+FONT_DELTA,FONT_PROP));
      setup_font(3,ASHfont_name(FONT_FAMILY,i+2*FONT_DELTA,FONT_PROP));
    }
   else {
      ft = ASHloadfont(s);
      ASHinq_text_next(ft,"Xp",&xd,&yd);
      l = 4;
      h = 36;
      while (l < h) {
	 m = (l+h)/2;
	 ft = ASHfont_id(FONT_FAMILY,m,FONT_PROP);
	 ASHinq_text_next(ft,"Xp",&xn,&yn);
	 if (xn < xd || yn < yd) l = m+1;
	 else h = m;
       };
      i = l;
      setup_font(0,ASHfont_name(FONT_FAMILY,i-FONT_DELTA,FONT_PROP));
      setup_font(1,ASHfont_name(FONT_FAMILY,i,FONT_PROP));
      setup_font(2,ASHfont_name(FONT_FAMILY,i+FONT_DELTA,FONT_PROP));
      setup_font(3,ASHfont_name(FONT_FAMILY,i+2*FONT_DELTA,FONT_PROP));
    };

   s = ASHinq_resource("dialog.box_size");
   if (s == NULL) box_x_size = BOX_SIZE;
   else box_x_size = atol(s);
   if (box_x_size == 0) box_x_size = BOX_SIZE;
   box_x_size += NEW_ADDON;
   box_y_size = box_x_size;

   s = ASHinq_resource("dialog.rect_size");
   if (s == NULL) rect_x_size = RECT_SIZE;
   else rect_x_size = atol(s);
   if (rect_x_size == 0) rect_x_size = RECT_SIZE;
   rect_x_size += NEW_ADDON;
   rect_y_size = rect_x_size;

   s = ASHinq_resource("dialog.text_x_size");
   if (s == NULL) text_x_size = TEXT_X_DELTA;
   else text_x_size = atol(s);
   if (text_x_size == 0) text_x_size = TEXT_X_DELTA;
   s = ASHinq_resource("dialog.text_y_size");
   if (s == NULL) text_y_size = TEXT_Y_DELTA;
   else text_y_size = atol(s);
   if (text_y_size == 0) text_y_size = TEXT_Y_DELTA;

   s = ASHinq_resource("dialog.select_x_size");
   if (s == NULL) select_x_size = text_x_size;
   else select_x_size = atol(s);
   if (select_x_size == 0) select_x_size = text_x_size;
   s = ASHinq_resource("dialog.select_y_size");
   if (s == NULL) select_y_size = text_y_size;
   else select_y_size = atol(s);
   if (select_y_size == 0) select_y_size = text_y_size;
};





/************************************************************************/
/*									*/
/*	STEMdialog -- set up and handle a dialog box			*/
/*									*/
/************************************************************************/


/* VARARGS */

int
STEMdialog(va_alist)
   va_dcl
{
   va_list ap;
   ASH_WINDOW uwindow;
   String * utext;
   Integer i,j;
   Character buf[10240];

   va_start(ap);
   uwindow = va_arg(ap,ASH_WINDOW);
   utext = va_arg(ap,String *);

   buf[0] = 0;
   for (i = 0; utext[i] != 0; ++i) {
      strcat(buf,utext[i]);
      j = strlen(buf);
      if (buf[j-2] != '%' || buf[j-1] != 'N') strcat(buf,"\n");
      else buf[j-2] = 0;
    };

   ENTER("STEMdialog 0x%x %s",uwindow,buf);

   return stem_do_dialog(uwindow,buf,ap,NULL,NULL);
};





/************************************************************************/
/*									*/
/*	STEMdialog1 -- entry for single string				*/
/*									*/
/************************************************************************/


/* VARARGS */

int
STEMdialog1(va_alist)
   va_dcl
{
   va_list ap;
   ASH_WINDOW uwindow;
   String utext;

   va_start(ap);
   uwindow = va_arg(ap,ASH_WINDOW);
   utext = va_arg(ap,String);

   ENTER("STEMdialog1 0x%x %s",uwindow,utext);

   return stem_do_dialog(uwindow,utext,ap,NULL,NULL);
};






/************************************************************************/
/*									*/
/*	STEMdialog_array -- entry for strings and array of variables	*/
/*	STEMdialog1_array -- entry for string and array of variables	*/
/*									*/
/************************************************************************/


int
STEMdialog_array(uwindow,utext,udata)
   ASH_WINDOW uwindow;
   String utext[];
   Universal * udata[];
{
   Integer i,j;
   Character buf[10240];

   buf[0] = 0;
   for (i = 0; utext[i] != 0; ++i) {
      strcat(buf,utext[i]);
      j = strlen(buf);
      if (buf[j-2] != '%' || buf[j-1] != 'N') strcat(buf,"\n");
      else buf[j-2] = 0;
    };

   ENTER("STEMdialog_array 0x%x %s",uwindow,buf);

   return stem_do_dialog(uwindow,buf,NULL,udata,NULL);
};





int
STEMdialog1_array(uwindow,utext,udata)
   ASH_WINDOW uwindow;
   String utext;
   Universal * udata[];
{
   ENTER("STEMdialog1_array 0x%x %s",uwindow,utext);

   return stem_do_dialog(uwindow,utext,NULL,udata,NULL);
};





/************************************************************************/
/*									*/
/*	STEMdialog_wait -- set up and handle a dialog box w/ user wait	*/
/*	STEMdialog1_wait -- set up and handle dialog box w/ user wait	*/
/*									*/
/************************************************************************/


/* VARARGS */
int
STEMdialog_wait(va_alist)
   va_dcl
{
   va_list ap;
   ASH_WINDOW uwindow;
   String * utext;
   Integer i,j;
   Character buf[10240];
   Function_Ptr wfct;

   va_start(ap);
   uwindow = va_arg(ap,ASH_WINDOW);
   wfct = va_arg(ap,Function_Ptr);
   utext = va_arg(ap,String *);

   buf[0] = 0;
   for (i = 0; utext[i] != 0; ++i) {
      strcat(buf,utext[i]);
      j = strlen(buf);
      if (buf[j-2] != '%' || buf[j-1] != 'N') strcat(buf,"\n");
      else buf[j-2] = 0;
    };

   ENTER("STEMdialog 0x%x %s",uwindow,buf);

   return stem_do_dialog(uwindow,buf,ap,NULL,wfct);
};





/* VARARGS */

int
STEMdialog1_wait(va_alist)
   va_dcl
{
   va_list ap;
   ASH_WINDOW uwindow;
   String utext;
   Function_Ptr wfct;

   va_start(ap);
   uwindow = va_arg(ap,ASH_WINDOW);
   wfct = va_arg(ap,Function_Ptr);
   utext = va_arg(ap,String);

   ENTER("STEMdialog1 0x%x %s",uwindow,utext);

   return stem_do_dialog(uwindow,utext,ap,NULL,wfct);
};






/************************************************************************/
/*									*/
/*	STEMdialog_mouse -- indicate whether to move mouse or not	*/
/*									*/
/************************************************************************/


void
STEMdialog_mouse(fg)
   Boolean fg;
{
   ENTER("STEMdialog_mouse %d",fg);

   move_mouse = fg;
};





/************************************************************************/
/*									*/
/*	STEMdialog_color -- define color for dialog boxes		*/
/*									*/
/************************************************************************/


void
STEMdialog_color(val,color,backg)
   Integer val;
   String color;
   String backg;
{
   if (val < 0 || val >= NUM_COLOR) return;

   if (color != NULL) color_table[val].name = SALLOC(color);
   if (backg != NULL) color_table[val].backname = SALLOC(backg);
};





/************************************************************************/
/*									*/
/*	stem_do_dialog -- main entry to handle dialogs			*/
/*									*/
/************************************************************************/


static Boolean
stem_do_dialog(uwindow,utext,ap,ud,wrtn)
   ASH_WINDOW uwindow;
   String utext;
   Universal * ap[];
   Universal * ud[];
   Function_Ptr wrtn;
{
   Universal * udata[MAX_FIELD];
   Integer i,j;
   STEM_DIALOG_B ddef;

   DTRACE("stem_do_dialog 0x%x %s 0x%x",uwindow,utext,ap);

   if (!ASHinq_valid_window(uwindow)) return FALSE;

   ddef.user_window = uwindow;
   ddef.cover_window = NULL;
   ddef.root_window = NULL;
   ddef.user_data = NULL;
   ddef.user_text = utext;
   ddef.window = NULL;
   ddef.num_field = 0;
   ddef.text_fld = -1;
   ddef.text_pos = -1;
   ddef.text_buf[0] = 0;
   ddef.next = NULL;
   ddef.backcolor = -1;
   ddef.hilite = -1;
   ddef.wait_rtn = wrtn;
   ddef.drawn = FALSE;

   if (ASHinq_configuration_depth(uwindow) > 1) {
      ddef.cursor_color = ASHlookup_color(uwindow,cursor_color);
      ddef.texton_color = ASHlookup_color(uwindow,texton_color);
      ddef.textoff_color = ASHlookup_color(uwindow,textoff_color);
    }
   else {
      ddef.cursor_color = ASHinq_color(uwindow);
      ddef.texton_color = ASHinq_color(uwindow);
      ddef.textoff_color = ASHinq_background_color(uwindow);
    };

   i = size_dialog(&ddef,utext);
   if (i < 0 || i >= MAX_FIELD) {
      ERROR("Bad dialog definition");
      ERROR(utext);
      return FALSE;
    };

   for (j = 0; j < MAX_FIELD; ++j) {
      ddef.field[j].editid = NULL;
      ddef.field[j].lx = -1;
      ddef.field[j].by = -1;
      ddef.field[j].index = -1;
      ddef.field[j].value = 0;
      ddef.field[j].editid = NULL;
      ddef.field[j].subwin = NULL;
      ddef.field[j].hilite = FALSE;
      ddef.field[j].select = FALSE;
      ddef.field[j].cancel = FALSE;
      ddef.field[j].accept = FALSE;
      ddef.field[j].edited = FALSE;
      ddef.field[j].color = -1;
      ddef.field[j].fill = -1;
      ddef.field[j].btn = NULL;
    };

   if (ud != NULL) ddef.user_data = ud;
   else if (ap != NULL) {
      for (j = 0; j <= i; ++j) {
	 udata[j] = *ap++;
       };
      ddef.user_data = udata;
    };

   if (!setup_dialog(&ddef)) {
      ERROR("Couldn't setup dialog");
      return FALSE;
    };

   if (ddef.cover_window != NULL) ASHgrab_from(ddef.cover_window,ddef.window);
   ASHgrab_text_from(ddef.window,ddef.window);

   PROTECT;
   ddef.next = all_dialog;
   all_dialog = &ddef;
   UNPROTECT;

   ASHvisible(ddef.window,TRUE);
   ASHpop(ddef.window);

   i = process_dialog(&ddef);

   remove_dialog(&ddef);

   return i;
};





/************************************************************************/
/*									*/
/*	dialog_control -- handle ASH messages for dialog window 	*/
/*									*/
/************************************************************************/


static int
dialog_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_DIALOG dd;

   ITRACE("dialog_control %s 0x%x",msg,w);

   dd = (STEM_DIALOG) ASHinq_user_data(w);
   if (dd == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      dd->window = NULL;
      if (dd->cover_window != NULL) {
	 ASHgrab_from(dd->cover_window,NULL);
	 dd->cover_window = NULL;
       };
      if (!dd->abort) remove_dialog(dd);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	dialog_user_control -- handle ASH messages for user window	*/
/*									*/
/************************************************************************/


static int
dialog_user_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_DIALOG dd;

   ITRACE("dialog_user_control %s 0x%x",msg,w);

   PROTECT;
   for (dd = all_dialog;
	dd != NULL && dd->user_window != w && dd->cover_window != w;
	dd = dd->next);
   UNPROTECT;
   if (dd == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$REMOVE")) {
      dd->user_window = NULL;
      if (!ASHinq_valid_window(dd->cover_window)) dd->cover_window = NULL;
      if (!dd->abort) remove_dialog(dd);
    }
   else if (STREQL(msg,"ASH$RESIZE") || STREQL(msg,"ASH$VIEW")) {
      move_window(dd);
    }
   else if (STREQL(msg,"ASH$VISIBLE")) {
      ASHvisible(dd->window,TRUE);
      ASHpop(dd->window);
    }
   else if (STREQL(msg,"ASH$INVISIBLE")) {
      ASHvisible(dd->window,FALSE);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	dialog_refresh -- handle ASH refresh requests for dialog window */
/*									*/
/************************************************************************/


static void
dialog_refresh(w)
   ASH_WINDOW w;
{
   STEM_DIALOG dd;

   ITRACE("dialog_refresh 0x%x",w);

   dd = (STEM_DIALOG) ASHinq_user_data(w);
   if (dd == NULL || dd->window == NULL) return;

   draw_dialog(dd);
};





/************************************************************************/
/*									*/
/*	setup_font -- setup a font for later use			*/
/*									*/
/************************************************************************/


static void
setup_font(idx,fnt)
   Integer idx;
   String fnt;
{
   Integer x,y;

   DTRACE("setup_font %d %s",idx,fnt);

   fonts[idx].fontid = ASHloadfont(fnt);
   ASHinq_text_next(fonts[idx].fontid,"X",&x,&y);
   fonts[idx].x_size = x;
   fonts[idx].y_size = y;
   ASHinq_text_offset(fonts[idx].fontid,"X",&x,&y);
   fonts[idx].x_offset = x;
   fonts[idx].y_offset = y;
};





/************************************************************************/
/*									*/
/*	size_dialog -- get size of dialog box				*/
/*									*/
/************************************************************************/


static Integer
size_dialog(dd,txt)
   STEM_DIALOG dd;
   String txt;
{
   Integer xmx,ysz,ymx,yof;
   Integer colsz[MAX_COL],endsz[MAX_COL],ncol,midfg;
   Integer font;
   Integer line,ch,cha;
   Integer csz;
   Boolean rectfg;
   Integer i,j,k;
   Integer maxarg;
   Integer color;
   Integer tx,ty;

   DTRACE("size_dialog 0x%x %s",dd,txt);

   xmx = ymx = 0;
   line = 0;
   maxarg = 0;

   for (i = 0; i < MAX_COL; ++i) colsz[i] = endsz[i] = 0;
   ncol = 0;

   ysz = TOP_MARGIN;
   font = 1;
   csz = 0;
   midfg = 0;
   ymx = fonts[0].y_size;
   yof = 0;
   while (*txt != 0) {
      if (*txt != '%' || *++txt == '%') {
	 if (*txt == '\n') {
	    if (csz > endsz[midfg]) endsz[midfg] = csz;
	    if (midfg > ncol) ncol = midfg;
	    ysz += ymx;
	    dd->linesize[line] = csz;
	    dd->ypos[line++] = ysz-yof;
	    ysz += 1;
	    font = 1;
	    csz = 0;
	    midfg = 0;
	    ymx = fonts[0].y_size;
	  }
	 else {
	    csz += fonts[font].x_size;
	    ymx = MAX(ymx,fonts[font].y_size);
	    yof = MAX(yof,fonts[font].y_offset);
	  };
       }
      else if (isupper(*txt)) {
	 switch (*txt) {
	    case 'M' :
	       font = 1;
	       if (csz > colsz[midfg]) colsz[midfg] = csz;
	       csz = 0;
	       ++midfg;
	       break;
	    case 'F' :
	       ++txt;
	       font = *txt - '0';
	       if (font < 0 || font > NUM_FONT) return -1;
	       break;
	    case 'B' :
	       ++txt;
	       color = *txt - '0';
	       if (color < 0 || color > NUM_COLOR) dd->backcolor = -1;
	       else dd->backcolor = color;
	       break;
	    case 'P' :
	       ++txt;
	       color = *txt - '0';
	       if (color < 0 || color > NUM_COLOR) return -1;
	       break;
	    case 'L' :
	    case 'C' :
	    case 'R' :
	       csz = 0;
	       break;
	    default :
	       return -1;
	  };
       }
      else {
	 i = 0;
	 while (isdigit(*txt)) i = i*10 + *txt++ - '0';
	 if (i >= maxarg) maxarg = i;
	 i = 0;
	 if (*txt == '.') {
	    ++txt;
	    while (isdigit(*txt)) i = i*10 + *txt++ - '0';
	  };
	 rectfg = FALSE;
	 k = 0;
	 if (*txt == '.') {
	    rectfg = TRUE;
	    ++txt;
	    while (isdigit(*txt)) k = k*10 + *txt++ - '0';
	    while (*txt == '.') {
	       ++txt;
	       while (isdigit(*txt)) ++txt;
	     };
	  };
	 j = 0;
	 if (!isalpha(*txt) && *txt != '+') {
	    j = 0;
	    ch = *txt++;
	    while (*txt != 0 && *txt != ch) {
	       ++txt;
	       ++j;
	     };
	    if (*txt == 0) return -1;
	    ++txt;
	  };
	 if (*txt == '+') ++txt;

	 cha = *txt;
	 switch (cha) {
	    case 'o' :
	    case 'f' :
	       if (rectfg) {
		  csz += RECT_X_SIZE;
		  ymx = MAX(ymx,RECT_Y_SIZE+RECT_Y_OFFSET);
		  yof = MAX(yof,RECT_Y_OFFSET);
		}
	       else {
		  csz += BOX_X_SIZE;
		  ymx = MAX(ymx,BOX_Y_SIZE+BOX_Y_OFFSET);
		  yof = MAX(yof,BOX_Y_OFFSET);
		};
	       break;
	    case 't' :
	       if (i == 0) i = DFLT_TEXT_SIZE;
	       csz += TEXT_X_SIZE(i);
	       ymx = MAX(ymx,TEXT_Y_SIZE);
	       yof = MAX(yof,TEXT_Y_OFFSET);
	       break;
	    case 'a' :
	    case 'c' :
	    case 'b' :
	       if (j == 0) {
		  if (*txt == 'a' || *txt == 'c') j = 6;
		  else return -1;
		};
	       get_button_size(j,&tx,&ty);
	       csz += tx;
	       ymx = MAX(ymx,ty+1);
	       yof = MAX(yof,1);
	       break;
	    case 'd' :
	       if (i == 0) i = DFLT_INT_SIZE;
	       csz += INT_X_SIZE(i);
	       ymx = MAX(ymx,INT_Y_SIZE);
	       yof = MAX(yof,INT_Y_OFFSET);
	       break;
	    case 'g' :
	    case 'h' :
	       if (i == 0) i = DFLT_FLT_SIZE;
	       csz += FLT_X_SIZE(i);
	       ymx = MAX(ymx,FLT_Y_SIZE);
	       yof = MAX(yof,FLT_Y_OFFSET);
	       break;
	    case 'e' :
	       if (i == 0) i = DFLT_EDIT_SIZE;
	       if (k == 0) k = DFLT_EDIT_LINES;
	       csz += EDIT_X_SIZE(i);
	       ymx = MAX(ymx,EDIT_Y_SIZE(k));
	       yof = MAX(yof,EDIT_Y_OFFSET);
	       break;
	    case 'l' :
	       if (i == 0) i = DFLT_LIST_SIZE;
	       if (k == 0) k = DFLT_LIST_LINES;
	       csz += LIST_X_SIZE(i);
	       ymx = MAX(ymx,LIST_Y_SIZE(k));
	       yof = MAX(yof,LIST_Y_OFFSET);
	       break;
	    default :
	       return -1;
	  };
       };
      ++txt;
    };

   if (csz > 0) {
      if (csz > endsz[midfg]) endsz[midfg] = csz;
      if (midfg > ncol) ncol = midfg;
      ysz += ymx;
      dd->linesize[line] = csz;
      dd->ypos[line++] = ysz-yof;
    };


   ++ncol;
   for (i = 0; i < ncol; ++i) colsz[i] += MARGIN;
   xmx = endsz[0];
   csz = colsz[0];
   for (i = 1; i < ncol; ++i) {
      if (endsz[i] + csz > xmx) xmx = endsz[i]+csz;
      csz += colsz[i];
    };

   dd->xsize = xmx+2*MARGIN+2*BORDER;
   dd->ysize = ysz+2*MARGIN+2*BORDER;

   i = dd->xsize-2*BORDER-2*MARGIN;
   colsz[ncol-1] = endsz[ncol-1];
   csz = 0;
   k = 0;
   for (j = 0; j < ncol; ++j) {
      if (colsz[j] > i/ncol) k = 1;
      csz += colsz[j];
    };

   if (k == 0) {
      for (j = 0; j < ncol; ++j) {
	 dd->xcol[j] = (i*(j+1)/ncol)+(j+2)*MARGIN;
       };
    }
   else {
      k = i-csz;
      csz = MARGIN;
      for (j = 0; j < ncol; ++j) {
	 csz += colsz[j] + k/(ncol-j);
	 dd->xcol[j] = csz;
	 k -= k/(ncol-j);
       };
    };

   return maxarg;
};





/************************************************************************/
/*									*/
/*	setup_dialog -- setup windows for dialog			*/
/*									*/
/************************************************************************/


static Boolean
setup_dialog(dd)
   STEM_DIALOG dd;
{
   Integer ulx,uby,urx,uty;
   Integer tlx,tby,trx,tty;
   Integer lx,by;
   Integer plx,pby;
   Integer cx,cy;
   RIP_REGION rgn;
   ASH_WINDOW root,w;

   DTRACE("setup_dialog 0x%x",dd);

   if (USE_THREADS) {
      root = BIOinq_top_window();
      for (w = dd->user_window; w != NULL; w = ASHinq_parent(w)) {
	 if (w == root) break;
       };
      if (w == NULL) root = ASHinq_top_window(dd->user_window);
    }
   else root = ASHinq_top_window(dd->user_window);

   dd->cover_window = root;

   if (root_windows) {
      root = ASHroot_X_window(dd->user_window);
      dd->root_window = root;
      ASHinq_size(root,ASH_SIZE_WINDOW,&tlx,&tby,&trx,&tty);
    }
   else dd->root_window = NULL;

   ASHinq_size(root,ASH_SIZE_WINDOW,&tlx,&tby,&trx,&tty);
   ASHinq_size(dd->user_window,ASH_SIZE_WINDOW,&ulx,&uby,&urx,&uty);

   lx = (abs(urx-ulx)-dd->xsize)/2;
   lx = ulx + (ulx < urx ? lx : -lx);
   by = (abs(uty-uby)-dd->ysize)/2;
   by = uby + (uby < uty ? by : -by);

   ASHmap(dd->user_window,lx,by,root,&plx,&pby);

   if (plx < 0) plx = 0;
   else if (plx+dd->xsize > trx) plx = trx - dd->xsize;
   if (pby < dd->ysize) pby = dd->ysize;
   else if (pby > tby) pby = tby;

   dd->window =
      ASHcreate(root,plx,pby,0,dd->ysize-1,dd->xsize-1,0,BORDER_WIDTH,
		   ASH_WINDOW_NOREDIRECT|ASH_WINDOW_INDEPENDENT|
		   ASH_WINDOW_INVISIBLE|ASH_WINDOW_NOSAVE);

   ASHset_window_name(dd->window,"STEM_DIALOG");
   ASHset_window_id(dd->window,"dialog");
   ASHset_window_defaults(dd->window);
   ASHset_user_data(dd->window,dd);
   ASHset_control(dd->window,dialog_control);
   ASHset_refresh(dd->window,dialog_refresh);

   if (ASHinq_configuration_depth(dd->window) > 1) {
      ASHwindow_border(dd->window,BORDER_WIDTH,
			  ASHlookup_color(dd->window,BORDER_COLOR),NULL);
    };

   ASHset_control(dd->user_window,dialog_user_control);
   if (dd->user_window != dd->cover_window) {
      ASHset_control(dd->cover_window,dialog_user_control);
    };

   BIOnew_input_window(dd->window);
   BIOset_cursor(dd->window,TRUE);
   BIOset_cursor_standard(dd->window,ASH_CURSOR_SMALL_ARROW);

   if (move_mouse) {
      cx = dd->xsize/2;
      cy = dd->ysize/2;
      ASHcursor_move(dd->window,cx,cy);
    };

   rgn = RIPdefine_region(dd->window,0,dd->ysize-1,dd->xsize-1,0,RIP_ALL_CHARS,
			     RIP_BTN_ANY_EITHER,dialog_hit,0);
   RIPset_data(rgn,dd);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	draw_dialog -- output text into dialog box			*/
/*									*/
/************************************************************************/


static Boolean
draw_dialog(dd)
   STEM_DIALOG dd;
{
   String s;
   Character buf[1024];
   Integer idx,value,i,v1,v2;
   Character vbuf[1024];
   Integer ch,cha,vlen,xpos,font,cct,midct,line;
   Integer c,tc,bc,tbc,color;
   STEM_BTN sb;
   Integer tx,ty;
   Boolean havevalue;

   DTRACE("draw_dialog 0x%x",dd);

   dd->drawn = TRUE;

   if (!ASHinq_visible(dd->window)) return TRUE;

   ASHbatch_mode(TRUE);

   s = dd->user_text;
   line = 0;
   cct = 0;
   midct = 0;
   font = 1;
   xpos = MARGIN;
   dd->num_field = 0;
   c = ASHinq_color(dd->window);
   tc = ASHinq_text_color(dd->window);
   bc = ASHinq_background_color(dd->window);
   tbc = ASHinq_text_background_color(dd->window);

   sb = STEMbtn_define(dd->window,1,1,1,1,"",STEM_BTN_NO_DRAW,NULL,NULL);
   if (dd->backcolor >= 0 && ASHinq_configuration_depth(dd->window) > 1) {
      color = ASHlookup_color(dd->window,color_table[dd->backcolor].backname);
      STEMbtn_set_gray(sb,color);
    }
   else STEMbtn_set_gray(sb,bc);
   STEMbtn_background(sb,0,0,0,0);

   c = ASHcolor(dd->window,ASHlookup_color(dd->window,BORDER_COLOR));
   ASHbox(dd->window,1,dd->ysize-2,dd->xsize-2,1);
   ASHcolor(dd->window,c);

   while (*s != 0) {
      if (*s != '%' || *++s == '%') {
	 if (*s == '\n') {
	    xpos = flushbuf(dd,xpos,buf,cct,line,font);
	    ++line;
	    font = 1;
	    cct = 0;
	    midct = 0;
	    xpos = MARGIN;
	  }
	 else if (*s == ' ' && s[1] == ' ') {
	    xpos = flushbuf(dd,xpos,buf,cct,line,font);
	    cct = 0;
	    buf[cct++] = *s;
	  }
	 else if (*s == '\t') buf[cct++] = ' ';
	 else buf[cct++] = *s;
       }
      else if (isupper(*s)) {
	 xpos = flushbuf(dd,xpos,buf,cct,line,font);
	 cct = 0;
	 switch (*s) {
	    case 'M' :
	       font = 1;
	       xpos = dd->xcol[midct];
	       ++midct;
	       break;
	    case 'F' :
	       ++s;
	       font = *s - '0';
	       break;
	    case 'P' :
	       ++s;
	       i = *s - '0';
	       if (ASHinq_configuration_depth(dd->window) > 1) {
		  color = ASHlookup_color(dd->window,color_table[i].name);
		  ASHcolor(dd->window,color);
		  ASHtext_color(dd->window,color);
		  color = ASHlookup_color(dd->window,color_table[i].backname);
		  ASHbackground_color(dd->window,color);
		  ASHtext_background_color(dd->window,color);
		};
	       break;
	    case 'B' :
	       ++s;
	       break;
	    case 'L' :
	       break;
	    case 'C' :
	       xpos = (dd->xsize - dd->linesize[line])/2;
	       break;
	    case 'R' :
	       xpos = (dd->xsize - dd->linesize[line] - BORDER - MARGIN);
	       break;
	  };
       }
      else {
	 xpos = flushbuf(dd,xpos,buf,cct,line,font);
	 cct = 0;
	 if (isdigit(*s)) idx = 0;
	 else idx = -1;
	 value = -1;
	 havevalue = FALSE;
	 v1 = -1;
	 v2 = -1;
	 while (isdigit(*s)) idx = idx*10+ *s++ - '0';
	 if (*s == '.') {
	    ++s;
	    if (isdigit(*s)) {
	       havevalue = TRUE;
	       value = 0;
	       while (isdigit(*s)) value = value*10+ *s++ - '0';
	     };
	  };
	 if (*s == '.') {
	    ++s;
	    if (isdigit(*s)) {
	       v1 = 0;
	       while (isdigit(*s)) v1 = v1*10+ *s++ - '0';
	     };
	  };
	 if (*s == '.') {
	    ++s;
	    if (isdigit(*s)) {
	       v2 = 0;
	       while (isdigit(*s)) v2 = v2*10+ *s++ - '0';
	     };
	  };
	 vlen = 0;
	 if (!isalpha(*s) && *s != '+') {
	    ch = *s++;
	    while (*s != ch) vbuf[vlen++] = *s++;
	    ++s;
	  };
	 dd->field[dd->num_field].lx = xpos;
	 dd->field[dd->num_field].by = dd->ypos[line];
	 dd->field[dd->num_field].index = idx;
	 dd->field[dd->num_field].value = value;
	 dd->field[dd->num_field].hilite = FALSE;
	 dd->field[dd->num_field].select = FALSE;
	 dd->field[dd->num_field].cancel = FALSE;
	 dd->field[dd->num_field].accept = FALSE;
	 dd->field[dd->num_field].color = -1;
	 dd->field[dd->num_field].fill = -1;
	 dd->field[dd->num_field].btn = NULL;
	 if (*s == '+') {
	    dd->field[dd->num_field].select = TRUE;
	    ++s;
	  };

	 cha = *s;
	 switch (cha) {
	    case 'o' :
	    case 'f' :
	       if (v1 > 0 || v2 > 0) {
		  xpos += RECT_X_SIZE;
		  dd->field[dd->num_field].by -= RECT_Y_OFFSET;
		  dd->field[dd->num_field].rx = xpos-1;
		  dd->field[dd->num_field].ty = dd->field[dd->num_field].by-RECT_Y_SIZE+1;
		}
	       else {
		  xpos += BOX_X_SIZE;
		  dd->field[dd->num_field].by -= BOX_Y_OFFSET;
		  dd->field[dd->num_field].rx = xpos-1;
		  dd->field[dd->num_field].ty = dd->field[dd->num_field].by-BOX_Y_SIZE+1;
		};
	       if (*s == 'o') dd->field[dd->num_field].type = STEM_DTYPE_OPTION;
	       else dd->field[dd->num_field].type = STEM_DTYPE_FLAG;
	       if (!havevalue) {
		  dd->field[dd->num_field].value = 1;
		  dd->field[dd->num_field].type = STEM_DTYPE_BOPT;
		};
	       dd->field[dd->num_field].color = v1;
	       dd->field[dd->num_field].fill = v2;
	       draw_check(dd,dd->num_field,TRUE);
	       break;
	    case 't' :
	       if (dd->text_fld < 0) dd->text_fld = dd->num_field;
	       if (value <= 0) value = DFLT_EDIT_SIZE;
	       if (v1 <= 0) v1 = value;
	       dd->field[dd->num_field].value = v1;
	       xpos += TEXT_X_SIZE(value);
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-TEXT_Y_SIZE+1;
	       dd->field[dd->num_field].type = STEM_DTYPE_TEXT;
	       draw_edit(dd,dd->num_field);
	       break;
	    case 'd' :
	       if (dd->text_fld < 0) dd->text_fld = dd->num_field;
	       if (value <= 0) {
		  value = DFLT_INT_SIZE;
		  dd->field[dd->num_field].value = value;
		};
	       xpos += INT_X_SIZE(value);
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-INT_Y_SIZE+1;
	       dd->field[dd->num_field].type = STEM_DTYPE_INT;
	       draw_edit(dd,dd->num_field);
	       break;
	    case 'g' :
	    case 'h' :
	       if (dd->text_fld < 0) dd->text_fld = dd->num_field;
	       if (value <= 0) {
		  value = DFLT_FLT_SIZE;
		  dd->field[dd->num_field].value = value;
		};
	       xpos += FLT_X_SIZE(value);
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-FLT_Y_SIZE+1;
	       if (*s == 'g') dd->field[dd->num_field].type = STEM_DTYPE_FLT;
	       else dd->field[dd->num_field].type = STEM_DTYPE_DBL;
	       draw_edit(dd,dd->num_field);
	       break;
	    case 'a' :
	    case 'c' :
	    case 'b' :
	       if (vlen == 0) {
		  if (*s == 'a') {
		     strcpy(vbuf,"Accept");
		     vlen = 6;
		     dd->field[dd->num_field].type = STEM_DTYPE_ACCEPT;
		   }
		  else if (*s == 'c') {
		     strcpy(vbuf,"Cancel");
		     vlen = 6;
		     dd->field[dd->num_field].type = STEM_DTYPE_CANCEL;
		   }
		}
	       else if (!havevalue) {
		  vbuf[vlen] = 0;
		  dd->field[dd->num_field].value = 1;
		  dd->field[dd->num_field].type = STEM_DTYPE_BBOXOPT;
		}
	       else {
		  vbuf[vlen] = 0;
		  dd->field[dd->num_field].type = STEM_DTYPE_BOXOPT;
		};
	       if (*s == 'a') dd->field[dd->num_field].accept = TRUE;
	       else if (*s == 'c') dd->field[dd->num_field].cancel = TRUE;
	       get_button_size(vlen,&tx,&ty);
	       xpos += tx;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-ty-1+1;
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].color = v1;
	       dd->field[dd->num_field].fill = v2;
	       draw_box(dd,dd->num_field,vbuf);
	       break;
	    case 'e' :
	       if (value <= 0) value = DFLT_EDIT_SIZE;
	       if (v1 <= 0) v1 = DFLT_EDIT_LINES;
	       xpos += EDIT_X_SIZE(value);
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-EDIT_Y_SIZE(v1)+1;
	       dd->field[dd->num_field].type = STEM_DTYPE_EDIT;
	       draw_edit(dd,dd->num_field);
	       break;
	    case 'l' :
	       if (value <= 0) value = DFLT_LIST_SIZE;
	       if (v1 <= 0) v1 = DFLT_LIST_LINES;
	       xpos += LIST_X_SIZE(value);
	       dd->field[dd->num_field].rx = xpos-1;
	       dd->field[dd->num_field].ty = dd->field[dd->num_field].by-LIST_Y_SIZE(v1)+1;
	       dd->field[dd->num_field].type = STEM_DTYPE_LIST;
	       draw_list(dd,dd->num_field);
	       break;
	  };
	 ++dd->num_field;
       };
      ++s;
    };

   if (cct != 0) {
      flushbuf(dd,xpos,buf,cct,line,font);
    };

   ASHcolor(dd->window,c);
   ASHtext_color(dd->window,tc);
   ASHbackground_color(dd->window,bc);
   ASHtext_background_color(dd->window,tbc);

   ASHbatch_mode(FALSE);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	flushbuf -- flush output buffer 				*/
/*									*/
/************************************************************************/


static Integer
flushbuf(dd,xpos,buf,cct,line,font)
   STEM_DIALOG dd;
   Integer xpos;
   String buf;
   Integer cct;
   Integer line;
   Integer font;
{
   String s;
   Integer y;
   Integer xct;

   buf[cct] = 0;

   DTRACE("flushbuf 0x%x %d %s %d %d %d",dd,xpos,buf,cct,line,font);

   s = buf;
   while (*s == ' ') {
      xpos += fonts[font].x_size;
      ++s;
      --cct;
    };

   xct = 0;
   while (cct > 0 && s[cct-1] == ' ') {
      ++xct;
      s[cct-1] = 0;
      --cct;
    };

   if (cct > 0) {
      y = dd->ypos[line] - text_y_size/2 + 1 - fonts[font].y_offset;
      ASHfont(dd->window,fonts[font].fontid);
      ASHtext(dd->window,xpos+fonts[font].x_offset,y,s);
      xpos += cct*fonts[font].x_size;
    };

   xpos += fonts[font].x_size*xct;

   return xpos;
};





/************************************************************************/
/*									*/
/*	draw_check -- draw a check box for field			*/
/*									*/
/************************************************************************/


static void
draw_check(dd,idx,fg)
   STEM_DIALOG dd;
   Integer idx;
   Boolean fg;
{
   Boolean vfg;
   STEM_BTN sb;

   DTRACE("draw_check 0x%x %d",dd,idx);

   if (dd->field[idx].type == STEM_DTYPE_FLAG) {
      vfg = ((*dd->user_data[dd->field[idx].index] & dd->field[idx].value) ==
		dd->field[idx].value);
    }
   else {
      vfg = (*dd->user_data[dd->field[idx].index] == dd->field[idx].value);
    };

   if (dd->field[idx].color > 0 || dd->field[idx].fill > 0) {
      draw_rect(dd,idx,fg,vfg);
      return;
    };

   if (dd->field[idx].btn != NULL) STEMbtn_remove(dd->field[idx].btn);
   sb = STEMbtn_define(dd->window,dd->field[idx].lx,dd->field[idx].by,
			  dd->field[idx].rx,dd->field[idx].ty," ",
			  STEM_BTN_NO_DRAW|STEM_BTN_FULL_SIZE,
			  NULL,NULL);
   dd->field[idx].btn = sb;
   STEMbtn_select(sb,dd->field[idx].select);
   STEMbtn_set(sb,vfg);
   STEMbtn_draw(sb);

   if (sb->bg == sb->gray && !vfg) {
      ASHbox(dd->window,dd->field[idx].lx,dd->field[idx].by,
			  dd->field[idx].rx,dd->field[idx].ty);
    };
};





/************************************************************************/
/*									*/
/*	draw_box -- draw a selection box				*/
/*									*/
/************************************************************************/


static void
draw_box(dd,idx,txt)
   STEM_DIALOG dd;
   Integer idx;
   String txt;
{
   Boolean fg;
   STEM_BTN sb;

   DTRACE("draw_box 0x%x %d %s",dd,idx,txt);

   if (dd->field[idx].index < 0) fg = FALSE;
   else fg = (*dd->user_data[dd->field[idx].index] == dd->field[idx].value);

   ASHfont(dd->window,fonts[1].fontid);

   if (dd->field[idx].btn == NULL) {
      sb = STEMbtn_define(dd->window,dd->field[idx].lx,dd->field[idx].by,
			     dd->field[idx].rx,dd->field[idx].ty,txt,
			     STEM_BTN_NO_DRAW|STEM_BTN_FULL_SIZE,
			     NULL,NULL);
      dd->field[idx].btn = sb;
      STEMbtn_set(sb,fg);
      STEMbtn_select(sb,dd->field[idx].select);
      STEMbtn_draw(sb);

      if (sb->bg == sb->gray && !fg && txt == NULL) {
	 ASHbox(dd->window,dd->field[idx].lx,dd->field[idx].by,
		   dd->field[idx].rx,dd->field[idx].ty);
       };
    }
   else {
      sb = dd->field[idx].btn;
      STEMbtn_set(sb,fg);
    };
};





/************************************************************************/
/*									*/
/*	draw_edit -- draw an edit region				*/
/*	update_edit -- update text in an edit region			*/
/*									*/
/************************************************************************/


static void
draw_edit(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   ASH_WINDOW w;
   EDT_DATA ed;
   EDT_ID ei;
   Integer r,r1;
   Boolean efg;

   DTRACE("draw_edit 0x%x %d",dd,idx);

   if (dd->field[idx].subwin != NULL) return;

   r = (dd->field[idx].rx - dd->field[idx].lx)-2*EDIT_BORDER;
   r1 = (dd->field[idx].by - dd->field[idx].ty)-2*EDIT_BORDER;
   efg = (dd->field[idx].type == STEM_DTYPE_EDIT);

   w = ASHcreate(dd->window,dd->field[idx].lx,dd->field[idx].by, 0,r1,r,0,
		    ASH_BORDER_WIDTH(EDIT_BORDER),
		    ASH_WINDOW_NOSAVE|ASH_WINDOW_INVISIBLE);

   dd->field[idx].subwin = w;

   ed = PALLOC(EDT_DATA_INFO);
   ed->readonly = FALSE;
   ed->create = FALSE;
   ed->temporary = TRUE;
   ed->scroll = efg;
   ed->raw = FALSE;
   ed->echo = FALSE;
   ed->savectl = FALSE;
   ed->mode = "DIALOG,TEXT";
   ed->bounds_rtn = NULL;
   ed->input_rtn = NULL;
   ed->mouse_rtn = (Function_Ptr) dialog_edt_mouse;
   ed->file_rtn = NULL;
   ed->input_filter = NULL;
   ed->output_filter = NULL;

   ei = EDTdefine_editor(w,NULL,ed,&dd->field[idx],NULL,NULL);
   dd->field[idx].editid = ei;
   edit_border(dd,idx);

   if (ei != NULL) {
      EDTprohibit_close(ei);
      update_edit(dd,idx);
    };

   ASHvisible(w,TRUE);
};





static void
update_edit(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   EDT_ID ei;
   String s;
   Character intbuf[64];
   double f;

   DTRACE("update_edit 0x%x %d",dd,idx);

   if (dd->field[idx].type == STEM_DTYPE_EDIT) {
      s = *((String *) dd->user_data[dd->field[idx].index]);
    }
   else if (dd->field[idx].type == STEM_DTYPE_INT) {
      sprintf(intbuf,"%d",*((Integer *)dd->user_data[dd->field[idx].index]));
      s = intbuf;
    }
   else if (dd->field[idx].type == STEM_DTYPE_FLT) {
      f = *((float *) dd->user_data[dd->field[idx].index]);
      sprintf(intbuf,"%f",f);
      s = intbuf;
    }
   else if (dd->field[idx].type == STEM_DTYPE_DBL) {
      f = *((double *) dd->user_data[dd->field[idx].index]);
      sprintf(intbuf,"%f",f);
      s = intbuf;
    }
   else {
      s = (String) dd->user_data[dd->field[idx].index];
    };

   ei = dd->field[idx].editid;
   if (ei != NULL) {
      EDTauto_select_all(ei,FALSE);
      EDTtruncate_at(ei,1);
      if (s != NULL && *s != 0) {
	 EDTinsert_line(ei,1,s);
	 EDTauto_select_all(ei,TRUE);
       };
      if (idx == dd->text_fld) {
	 EDTkeep_position(ei,TRUE);
       }
      else {
	 EDTkeep_position(ei,FALSE);
       };
    };
};





/************************************************************************/
/*									*/
/*	draw_list -- draw a list region 				*/
/*	list_button -- handle STEM menu button presses			*/
/*									*/
/************************************************************************/


static void
draw_list(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   ASH_WINDOW w;
   Integer r,r1,i;
   String *v;
   STEM_DIALOG_LIST * sdl;
   STEM_BUTTON btns[1024];

   DTRACE("draw_list 0x%x %d",dd,idx);

   r = (dd->field[idx].rx - dd->field[idx].lx);
   r1 = (dd->field[idx].by - dd->field[idx].ty);

   if (dd->field[idx].subwin != NULL) return;

   w = ASHcreate(dd->window,dd->field[idx].lx,dd->field[idx].by, 0,r1,r,0,
		    ASH_BORDER_THIN,
		    ASH_WINDOW_NOSAVE|ASH_WINDOW_INVISIBLE);
   ASHset_user_data(w,dd);

   dd->field[idx].subwin = w;

   sdl = (STEM_DIALOG_LIST *) dd->user_data[dd->field[idx].index];
   v = sdl->btns;

   for (i = 0; i < 1023; ++i) {
      if (*v == NULL) break;
      btns[i].name = *v;
      btns[i].flags = (sdl->choice == i ? STEM_FLAG_CHOICE : 0);
      if (**v == '^') {
	 btns[i].flags |= STEM_FLAG_LOLITE;
	 ++btns[i].name;
       };
      ++v;
    };
   btns[i].name = NULL;
   btns[i].flags = 0;

   STEMmenu(w,list_button,sdl,RIP_BTN_ANY_EITHER,btns);

   ASHvisible(w,TRUE);
};



static int
list_button(sdl,idx,btn,w)
   STEM_DIALOG_LIST * sdl;
   Integer idx;
   Integer btn;
   ASH_WINDOW w;
{
   String *v;
   STEM_BUTTON btns[1024];
   Integer i;
   STEM_DIALOG dd;

   DTRACE("list_button 0x%x %d 0x%x 0x%x",sdl,idx,btn,w);

   if (btn & RIP_BTN_DOWN) return FALSE;

   if (sdl->choice == idx) {		/* button already chosen	*/
      dd = (STEM_DIALOG) ASHinq_user_data(w);
      for (i = 0; i < dd->num_field; ++i) {
	 if (dd->field[i].type == STEM_DTYPE_TEXT ||
		dd->field[i].type == STEM_DTYPE_EDIT ||
		dd->field[i].type == STEM_DTYPE_INT ||
		dd->field[i].type == STEM_DTYPE_FLT ||
		dd->field[i].type == STEM_DTYPE_DBL) break;
       };
      if (i >= dd->num_field && dd->wait_rtn == NULL) dd->done = TRUE;
      return TRUE;
    };

   v = sdl->btns;

   sdl->choice = idx;

   for (i = 0; i < 1023; ++i) {
      if (*v == NULL) break;
      btns[i].name = *v;
      btns[i].flags = (sdl->choice == i ? STEM_FLAG_CHOICE : 0);
      if (**v == '^') {
	 btns[i].flags |= STEM_FLAG_LOLITE;
	 ++btns[i].name;
       };
      ++v;
    };
   btns[i].name = NULL;
   btns[i].flags = 0;

   STEMmenu_refresh(w,btns);

   return TRUE;
};





/********************************************************************************/
/*										*/
/*	draw_rect -- draw a filled/colored rectangle box			*/
/*										*/
/********************************************************************************/


static void
draw_rect(dd,idx,newfg,highfg)
   STEM_DIALOG dd;
   Integer idx;
   Boolean newfg;
   Boolean highfg;
{
   Integer ncol,nfil,ocol;
   Integer lx,by,rx,ty;

   ncol = dd->field[idx].color;
   if (ncol < 0) ncol = ASHinq_color(dd->window);
   nfil = dd->field[idx].fill;
   if (nfil < 0) nfil = ASH_FILL_SOLID;

   lx = dd->field[idx].lx;
   by = dd->field[idx].by;
   rx = dd->field[idx].rx;
   ty = dd->field[idx].ty;

   if (newfg) {
      ASHclear_box(dd->window,lx,by,rx,ty);
      ASHfill(dd->window,1);
      ASHbox(dd->window,lx,by,rx,ty);
      ocol = ASHcolor(dd->window,ncol);
      ASHfill(dd->window,nfil);
      ASHrectangle(dd->window,lx+2,by-2,rx-2,ty+2);
      ASHfill(dd->window,1);
      ASHcolor(dd->window,ocol);
    };

   if (highfg) {
      ASHfill(dd->window,1);
      ASHbox(dd->window,lx+1,by-1,rx-1,ty+1);
    }
   else {
      ASHfill(dd->window,0);
      ASHbox(dd->window,lx+1,by-1,rx-1,ty+1);
      ASHfill(dd->window,1);
    };
};





/************************************************************************/
/*									*/
/*	process_dialog -- do the input part of the dialog		*/
/*									*/
/************************************************************************/


static Boolean
process_dialog(dd)
   STEM_DIALOG dd;
{
   Integer fg;

   DTRACE("process_dialog 0x%x",dd);

   ASHsync(dd->window);

   dd->abort = FALSE;
   dd->done = FALSE;

   while (!dd->abort && !dd->done) {
      if (dd->wait_rtn != NULL) {
	 if (!dd->drawn) BIOnext();
	 else {
	    fg = (*dd->wait_rtn)(dd->user_window);
	    if (fg < 0) dd->abort = TRUE;
	    else if (fg > 0) dd->done = TRUE;
	  };
       }
      else RIPuser_pick(NULL);
      if (ASHinq_valid_window(dd->window)) {
/*	 if (!dd->drawn && getenv("NOFORCE_REFRESH") == NULL) ASHrefresh(dd->window);   */
       }
      else dd->abort = TRUE;
    };

   if (dd->done) {
      fix_edits(dd);
    };

   return dd->done;
};





/************************************************************************/
/*									*/
/*	remove_dialog -- remove a dialog box				*/
/*									*/
/************************************************************************/


static void
remove_dialog(dd)
   STEM_DIALOG dd;
{
   STEM_DIALOG dd1;

   DTRACE("remove_dialog 0x%x",dd);

   dd->abort = TRUE;

   if (!ASHinq_valid_window(dd->window)) dd->window = NULL;
   if (!ASHinq_valid_window(dd->root_window)) dd->root_window = NULL;
   if (!ASHinq_valid_window(dd->user_window)) dd->user_window = NULL;
   if (!ASHinq_valid_window(dd->cover_window)) dd->cover_window = NULL;

   if (dd->cover_window != NULL) ASHgrab_from(dd->cover_window,NULL);
   dd->cover_window = NULL;

   if (dd->window != NULL) ASHremove(dd->window);
   dd->window = NULL;

   if (dd->user_window != NULL)
      ASHremove_control(dd->user_window,dialog_user_control);

   if (dd->root_window != NULL) ASHremove(dd->root_window);
   dd->root_window = NULL;

   PROTECT;
   if (all_dialog == dd) all_dialog = dd->next;
   else {
      for (dd1 = all_dialog; dd1 != NULL && dd1->next != dd; dd1 = dd1->next);
      if (dd1 != NULL) dd1->next = dd->next;
    };
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	dialog_hit -- handle button hits and text in dialog box 	*/
/*									*/
/************************************************************************/


static int
dialog_hit(x,y,ch,btn,rgn)
   Integer x,y;
   Integer ch;
   Integer btn;
   RIP_REGION rgn;
{
   STEM_DIALOG dd;
   Boolean fg;

   ITRACE("dialog_hit %d %d 0x%x 0x%x 0x%x",x,y,ch,btn,rgn);

   dd = (STEM_DIALOG) RIPinq_data(rgn);
   if (dd == NULL) return FALSE;

   ASHmap(ASHinq_top_window(RIPinq_window(rgn)),x,y,dd->window,&x,&y);

   if (btn & RIP_BTN_DOWN) {
      dialog_down(dd,x,y);
      RIPtrack(dialog_track,-1,dd->window,FALSE);
      fg = TRUE;
    }
   else if (btn & RIP_BTN_NONE) {
      fg = dialog_typein(dd,x,y,ch);
    }
   else {
      fg = dialog_btn(dd,x,y);
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	dialog_track -- handle mouse tracking in dialog box		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static int
dialog_track(x,y,ct,act,max,w,btn,ch)
   Integer x,y;
   Integer ct;
   RIP_TRANS act;
   Integer max;
   ASH_WINDOW w;
   Integer btn;
   Integer ch;
{
   STEM_DIALOG dd;

   dd = (STEM_DIALOG) ASHinq_user_data(w);
   if (dd == NULL) return FALSE;

   switch (act) {
      case RIP_TRANS_NONE :
	 break;
      case RIP_TRANS_MOVE :
      case RIP_TRANS_DOWN :
      case RIP_TRANS_TAP :
	 dialog_down(dd,x,y);
	 break;
      case RIP_TRANS_UP :
	 dialog_btn(dd,x,y);
	 return FALSE;
      default :
	 break;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	dialog_down -- handle down buttons and tracks in dialog box	*/
/*									*/
/************************************************************************/


static void
dialog_down(dd,x,y)
   STEM_DIALOG dd;
   Integer x,y;
{
   Integer i;

   ITRACE("dialog_down 0x%x %d %d",dd,x,y);

   for (i = 0; i < dd->num_field; ++i) {
      if (x >= dd->field[i].lx && x <= dd->field[i].rx &&
	     y >= dd->field[i].ty && y <= dd->field[i].by)
	 break;
    };
   if (i >= dd->num_field) i = -1;
   else switch (dd->field[i].type) {
      case STEM_DTYPE_ACCEPT :
      case STEM_DTYPE_CANCEL :
	 break;
      default :
	 i = -1;
	 break;
    };

   if (i == dd->hilite) return;

   if (dd->hilite >= 0) {
      if (dd->field[dd->hilite].btn != NULL) {
	 STEMbtn_set(dd->field[dd->hilite].btn,FALSE);
       }
      else {
	 ASHhilite_box(dd->window,dd->field[dd->hilite].lx,dd->field[dd->hilite].by,
			  dd->field[dd->hilite].rx,dd->field[dd->hilite].ty);
       };
    };

   dd->hilite = i;

   if (dd->hilite >= 0) {
      if (dd->field[dd->hilite].btn != NULL) {
	 STEMbtn_set(dd->field[dd->hilite].btn,TRUE);
       }
      else {
	 ASHhilite_box(dd->window,dd->field[dd->hilite].lx,dd->field[dd->hilite].by,
			  dd->field[dd->hilite].rx,dd->field[dd->hilite].ty);
       };
    };
};





/************************************************************************/
/*									*/
/*	dialog_btn -- handle button hits in dialog box			*/
/*									*/
/************************************************************************/


static Boolean
dialog_btn(dd,x,y)
   STEM_DIALOG dd;
   Integer x,y;
{
   Integer i;

   ITRACE("dialog_btn 0x%x %d %d",dd,x,y);

   for (i = 0; i < dd->num_field; ++i) {
      if (x >= dd->field[i].lx && x <= dd->field[i].rx &&
	     y >= dd->field[i].ty && y <= dd->field[i].by)
	 break;
    };
   if (i >= dd->num_field) return FALSE;

   switch (dd->field[i].type) {
      case STEM_DTYPE_OPTION :
      case STEM_DTYPE_BOXOPT :
      case STEM_DTYPE_ACCEPT :
      case STEM_DTYPE_CANCEL :
	 if (dd->field[i].index < 0) break;
	 *dd->user_data[dd->field[i].index] = dd->field[i].value;
	 break;
      case STEM_DTYPE_FLAG :
	 if (dd->field[i].index < 0) break;
	 if ((*dd->user_data[dd->field[i].index] & dd->field[i].value) ==
		dd->field[i].value)
	    *dd->user_data[dd->field[i].index] ^= dd->field[i].value;
	 else
	    *dd->user_data[dd->field[i].index] |= dd->field[i].value;
	 break;
      case STEM_DTYPE_TEXT :
      case STEM_DTYPE_INT :
      case STEM_DTYPE_FLT :
      case STEM_DTYPE_DBL :
      case STEM_DTYPE_EDIT :
	 select_text(dd,i);
	 break;
      case STEM_DTYPE_BBOXOPT :
      case STEM_DTYPE_BOPT :
	 *dd->user_data[dd->field[i].index] = !*dd->user_data[dd->field[i].index];
	 break;
    };

   if (dd->field[i].accept) dd->done = TRUE;
   else if (dd->field[i].cancel) dd->abort = TRUE;

   update_btns(dd,i);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	update_btns -- update buttons after value changes		*/
/*									*/
/************************************************************************/


static void
update_btns(dd,index)
   STEM_DIALOG dd;
   Integer index;
{
   Integer i,j;

   j = dd->field[index].index;

   ASHbatch_mode(TRUE);

   for (i = 0; i < dd->num_field; ++i) {
      if (dd->field[i].index == j) {
	 switch (dd->field[i].type) {
	    case STEM_DTYPE_OPTION :
	    case STEM_DTYPE_FLAG :
	    case STEM_DTYPE_BOPT :
	       draw_check(dd,i,FALSE);
	       break;
	    case STEM_DTYPE_BOXOPT :
	    case STEM_DTYPE_BBOXOPT :
	       draw_box(dd,i,NULL);
	       break;
	    case STEM_DTYPE_INT :
	    case STEM_DTYPE_FLT :
	    case STEM_DTYPE_DBL :
	       update_edit(dd,i);
	       break;
	    default :
	       break;
	  };
       };
    };

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	dialog_typein -- handle typein for dialog box			*/
/*									*/
/************************************************************************/


static Boolean
dialog_typein(dd,x,y,ch)
   STEM_DIALOG dd;
   Integer x,y;
   Integer ch;
{
   Integer i;

   DTRACE("dialog_typein 0x%x %d %d 0x%x",dd,x,y,ch);

   i = dd->text_fld;

   if (ch == '\33') {
      if (dd->wait_rtn == NULL) dd->abort = TRUE;
      return TRUE;
    };

   if (i >= 0 && dd->field[i].editid != NULL) {
      dd->field[i].edited = TRUE;
      EDTtypein(dd->field[i].editid,ch);
    }
   else if (ch == '\n' || ch == '\r') {
      if (dd->wait_rtn == NULL) dd->done = TRUE;
    }

   return TRUE;
};





/************************************************************************/
/*									*/
/*	select_text -- select text area to use next			*/
/*									*/
/************************************************************************/


static void
select_text(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   Integer oidx;

   DTRACE("select_text 0x%x %d",dd,idx);

   if (dd->text_fld == idx) return;

   oidx = dd->text_fld;
   dd->text_fld = idx;

   edit_border(dd,oidx);
   edit_border(dd,idx);

   if (oidx >= 0) {
      switch (dd->field[oidx].type) {
	 case STEM_DTYPE_INT :
	 case STEM_DTYPE_FLT :
	 case STEM_DTYPE_DBL :
	    fix_edit_field(dd,oidx);
	    update_btns(dd,oidx);
	    break;
	 default :
	    break;
       };
    };
};




static void
edit_border(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   ASH_COLOR col;

   if (idx < 0 || dd == NULL) return;

   if (dd->field[idx].subwin == NULL) return;

   if (idx == dd->text_fld) {
      col = dd->texton_color;
      EDTkeep_position(dd->field[idx].editid,TRUE);
    }
   else {
      col = dd->textoff_color;
      EDTkeep_position(dd->field[idx].editid,FALSE);
    };

   ASHwindow_border(dd->field[idx].subwin,EDIT_BORDER,col,NULL);
};






/************************************************************************/
/*									*/
/*	fix_edits -- get final result from various editors		*/
/*	fix_edit_field -- fix single edit field 			*/
/*									*/
/************************************************************************/


static void
fix_edits(dd)
   STEM_DIALOG dd;
{
   Integer idx;

   DTRACE("fix_edits 0x%x",dd);

   for (idx = 0; idx < dd->num_field; ++idx) {
      if (dd->field[idx].editid != NULL) {
	 fix_edit_field(dd,idx);
       };
    };
};





static void
fix_edit_field(dd,idx)
   STEM_DIALOG dd;
   Integer idx;
{
   String s,t,p;
   String *sp;
   Integer i;

   DTRACE("fix_edit_field 0x%x %d",dd,idx);

   if (dd->field[idx].editid == NULL) return;

   if (dd->field[idx].type == STEM_DTYPE_EDIT) {
      s = EDTinq_contents(dd->field[idx].editid);
      dd->field[idx].editid = NULL;
      sp = (String *) dd->user_data[dd->field[idx].index];
      *sp = s;
    }
   else {
      s = EDTinq_contents(dd->field[idx].editid);
      dd->field[idx].editid = NULL;
      if (dd->field[idx].type == STEM_DTYPE_INT) {
	 *((Integer *) dd->user_data[dd->field[idx].index]) = atol(s);
       }
      else if (dd->field[idx].type == STEM_DTYPE_FLT) {
	 *((float *) dd->user_data[dd->field[idx].index]) = atof(s);
       }
      else if (dd->field[idx].type == STEM_DTYPE_DBL) {
	 *((double *) dd->user_data[dd->field[idx].index]) = atof(s);
       }
      else {
	 t = (String) dd->user_data[dd->field[idx].index];
	 p = s;
	 for (i = 0; i < dd->field[idx].value-1; ++i) {
	    if (*p == 0 || *p == '\n') break;
	    *t++ = *p++;
	  };
	 *t = 0;
       };
      free(s);
    };
};





/************************************************************************/
/*									*/
/*	move_window -- reposition dialog box after window moves 	*/
/*									*/
/************************************************************************/


static void
move_window(dd)
   STEM_DIALOG dd;
{
   Integer ulx,uby,urx,uty;
   Integer lx,by;
   Integer tlx,tby;
   ASH_WINDOW r;

   DTRACE("move_window 0x%x",dd);

   ASHinq_size(dd->user_window,ASH_SIZE_WINDOW,&ulx,&uby,&urx,&uty);

   lx = (abs(urx-ulx)-dd->xsize)/2;
   lx = ulx + (ulx < urx ? lx : -lx);
   by = (abs(uty-uby)-dd->ysize)/2;
   by = uby + (uby < uty ? by : -by);

   if (dd->root_window != NULL) r = dd->root_window;
   else r = ASHinq_top_window(dd->user_window); ASHmap(dd->user_window,lx,by,r,&tlx,&tby);

   if (tlx < 0) tlx = 0;
   if (tby < dd->ysize) tby = dd->ysize;

   ASHview(dd->window,tlx,tby,0,dd->ysize-1,dd->xsize-1,0);
   ASHpop(dd->window);
};





/************************************************************************/
/*									*/
/*	get_button_size -- get size of new type of button		*/
/*									*/
/************************************************************************/


static void
get_button_size(nchar,xsz,ysz)
   Integer nchar;
   Integer * xsz, * ysz;
{
   Character buf[256];
   Integer i;

   for (i = 0; i < nchar; ++i) {
      if (i & 1) buf[i] = 'X';
      else buf[i] = 'p';
    };
   buf[i] = 0;

   STEMbtn_size(fonts[1].fontid,buf,xsz,ysz);
};





/************************************************************************/
/*									*/
/*	dialog_edt_mouse -- handle mouse hits in EDT sub windows	*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
dialog_edt_mouse(df,btn,l,c,ssl,ssc,sel,sec)
   STEM_DFLD df;
   Integer btn;
   Integer l,c;
   Integer ssl,ssc;
   Integer sel,sec;
{
   STEM_DIALOG dd;
   Integer idx;

   for (dd = all_dialog; dd != NULL; dd = dd->next) {
      for (idx = 0; idx < dd->num_field; ++idx) {
	 if (df == &dd->field[idx]) break;
       };
      if (idx < dd->num_field) break;
    };

   if (dd == NULL) return;

   if (idx != dd->text_fld) select_text(dd,idx);
};





/************************************************************************/
/*									*/
/*	STEM_dialog_edt_cmd -- handle DIALOG commands from editor	*/
/*									*/
/************************************************************************/


int
STEMdialog_edt_cmd(df,cmd)
   STEM_DFLD df;
   String cmd;
{
   STEM_DIALOG dd;
   Integer idx;

   for (dd = all_dialog; dd != NULL; dd = dd->next) {
      for (idx = 0; idx < dd->num_field; ++idx) {
	 if (df == &dd->field[idx]) break;
       };
      if (idx < dd->num_field) break;
    };

   if (dd == NULL) return FALSE;

   if (STREQL(cmd,"NEXT")) set_next_field(dd,'\t');
   else if (STREQL(cmd,"DONE")) set_next_field(dd,'\n');
   else if (STREQL(cmd,"CANCEL")) dd->abort = TRUE;
   else if (STREQL(cmd,"ACCEPT")) dd->done = TRUE;
   else return FALSE;

   return TRUE;
};





static void
set_next_field(dd,ch)
   STEM_DIALOG dd;
   Integer ch;
{
   Integer i;

   for (i = dd->text_fld+1; i < dd->num_field; ++i) {
      if (dd->field[i].type == STEM_DTYPE_TEXT ||
	     dd->field[i].type == STEM_DTYPE_EDIT ||
	     dd->field[i].type == STEM_DTYPE_INT ||
	     dd->field[i].type == STEM_DTYPE_FLT ||
	     dd->field[i].type == STEM_DTYPE_DBL) break;
    };

   if (ch == '\t' && i >= dd->num_field) {
      for (i = 0; i < dd->num_field; ++i) {
	 if (dd->field[i].type == STEM_DTYPE_TEXT ||
		dd->field[i].type == STEM_DTYPE_EDIT ||
		dd->field[i].type == STEM_DTYPE_INT ||
		dd->field[i].type == STEM_DTYPE_FLT ||
		dd->field[i].type == STEM_DTYPE_DBL) break;
       };
    };

   if (i >= dd->num_field) {
      if (dd->wait_rtn == NULL) dd->done = TRUE;
    }
   else {
      select_text(dd,i);
    };
};




/* end of stemdialog.c */
