/************************************************************************/
/*									*/
/*		stembtn.c						*/
/*									*/
/*	Routines for drawing and using 3-D buttons			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define BW_FILL_STYLE		ASH_FILL_LIGHT_HALFTONE
#define FG_COLOR		"black"
#define BG_COLOR		"white"
#define GRAY_COLOR		"lightgray"
#define SELECT_COLOR		"red"
#define ON_COLOR		"gray"
#define BASE_FONT_FAMILY	"times"
#define BASE_FONT_SIZE		14
#define BASE_FONT_FLAGS 	ASH_FONT_ROMAN
#define DEFAULT_STYLE		"3D"

#define MAX_LINES		32





/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct _STEM_BTN_STYLE {
   String name;
   STEM_BTN_STYLE next;
   Function_Ptr setup_rtn;
   Function_Ptr size_rtn;
   Function_Ptr draw_rtn;
   Function_Ptr set_rtn;
   Function_Ptr hilite_rtn;
   Function_Ptr select_rtn;
   Function_Ptr enable_rtn;
} STEM_BTN_STYLE_INFO;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	ASH_FONT		btn_font;
static	Sequence		all_stem_btns;

static	STEM_BTN_STYLE		all_styles;
static	STEM_BTN_STYLE		cur_style;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void			get_btn_size();
static	void			set_btn_size();
static	void			btn_sense_set();
static	void			btn_sense_hilite();
static	STEM_BTN		find_btn();
static	void			setup_window();
static	void			stem_btn_refresh();
static	Integer 		stem_btn_control();
static	void			btn_text_size();
static	ASH_FONT		get_btn_font();
static	void			set_cur_style();
static	STEM_BTN_STYLE		find_style();





/************************************************************************/
/*									*/
/*	STEM_btn_init -- module initialization				*/
/*									*/
/************************************************************************/


void
STEM_btn_init()
{
   btn_font = ASHfont_id(BASE_FONT_FAMILY,BASE_FONT_SIZE,BASE_FONT_FLAGS);
   all_stem_btns = NULL;
   all_styles = NULL;
   cur_style = NULL;
};





/************************************************************************/
/*									*/
/*	STEMbtn_define -- create a new STEM 3D button			*/
/*									*/
/************************************************************************/


STEM_BTN
STEMbtn_define(w,lx,by,rx,ty,text,fgs,data,rtn)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
   String text;
   STEM_BTN_FLAGS fgs;
   Universal data;
   Function_Ptr rtn;
{
   STEM_BTN sb;
   String s;
   Integer i;

   set_cur_style();

   sb = PALLOC(STEM_BTN_INFO);
   sb->window = w;
   sb->text = (text == NULL ? NULL : SALLOC(text));
   sb->rtn = rtn;
   sb->data = data;
   sb->refresh = ((fgs & STEM_BTN_REFRESH) != 0);
   sb->fullsize = ((fgs & STEM_BTN_FULL_SIZE) != 0);
   sb->left = ((fgs & STEM_BTN_LEFT) != 0);
   sb->nofill = ((fgs & STEM_BTN_NO_FILL) != 0);
   sb->resize = FALSE;
   sb->status = FALSE;
   sb->hilite = FALSE;
   sb->drawn = FALSE;
   sb->enable = TRUE;
   sb->dummy = (lx == rx && by == ty && lx != 0);
   sb->id = NULL;
   sb->fill_style = BW_FILL_STYLE;
   sb->gray_size = 0;
   sb->style = cur_style;
   sb->font = ASHinq_font(w);

   if (ASHinq_configuration_depth(w) <= 1) sb->fill = TRUE;
   else sb->fill = FALSE;

   if (lx == 0 && by == 0 && rx == 0 && ty == 0) {
      if ((fgs & STEM_BTN_NO_REFRESH) == 0) sb->refresh = TRUE;
      ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      sb->resize = TRUE;
    };

   if (lx != rx || by != ty) {
      s = ASHinq_window_resource(w,"btn.set_colors");
      if (s != NULL && (s[0] == 't' || s[0] == 'T' || s[0] == '1'))
	 fgs |= STEM_BTN_DEFAULT_COLORS;
    };

   if ((fgs & (STEM_BTN_DEFAULT|STEM_BTN_DEFAULT_COLORS)) != 0) {
      s = ASHinq_window_resource(w,"btn.foreground");
      if (s == NULL) s = FG_COLOR;
      sb->fg = ASHlookup_color(w,s);
      s = ASHinq_window_resource(w,"btn.background");
      if (s == NULL) s = BG_COLOR;
      sb->bg = ASHlookup_color(w,s);
      if (!sb->fill) {
	 s = ASHinq_window_resource(w,"btn.gray");
	 if (s == NULL) s = GRAY_COLOR;
	 sb->gray = ASHlookup_color(w,s);
	 s = ASHinq_window_resource(w,"btn.select");
	 if (s == NULL) s = SELECT_COLOR;
	 sb->selcol = ASHlookup_color(w,s);
	 s = ASHinq_window_resource(w,"btn.on");
	 if (s == NULL) s = ON_COLOR;
	 sb->oncol = ASHlookup_color(w,s);
       };
    }
   else {
      sb->fg = ASHinq_color(w);
      sb->bg = ASHinq_background_color(w);
      if (!sb->fill) {
	 s = ASHinq_window_resource(w,"btn.gray_mix");
	 if (s == NULL || *s == 0) i = 3;
	 else i = atol(s);
	 sb->gray = ASHaverage_color(w,sb->fg,sb->bg,1,i);
	 s = ASHinq_window_resource(w,"btn.on_mix");
	 if (s == NULL || *s == 0) i = 7;
	 else i = atol(s);
	 sb->oncol = ASHaverage_color(w,sb->fg,sb->bg,1,i);
	 s = ASHinq_window_resource(w,"btn.select");
	 if (s == NULL) s = SELECT_COLOR;
	 sb->selcol = ASHlookup_color(w,s);
	 if (sb->selcol == sb->fg || sb->selcol == sb->bg ||
		sb->selcol == sb->gray) {
	    s = ASHinq_window_resource(w,"btn.select_mix");
	    if (s == NULL || *s == 0) i = 1;
	    else i = atol(s);
	    sb->selcol = ASHaverage_color(w,sb->fg,sb->bg,1,i);
	  };
       };
    };

   if ((fgs & STEM_BTN_DEFAULT) != 0) {
      if ((fgs & STEM_BTN_WINDOW_FONT) != 0) sb->font = ASHinq_font(w);
      else {
	 s = ASHinq_window_resource(w,"btn.font");
	 if (s != NULL) sb->font = ASHloadfont(s);
	 else sb->font = btn_font;
       };
    };

   if (sb->fg == sb->bg) sb->fg = sb->bg ^ 1;

   if ((fgs & STEM_BTN_SENSE_INOUT) != 0)
      sb->sense = (ASH_SENSE_TYPE) btn_sense_set;
   else if ((fgs & STEM_BTN_SENSE_FLIP) != 0)
      sb->sense = (ASH_SENSE_TYPE) btn_sense_hilite;
   else sb->sense = ASH_SENSE_NO_CHANGE;

   sb->xdl = (lx < rx ? 1 : -1);
   sb->ydl = (ty < by ? 1 : -1);

   sb->lx = lx;
   sb->by = by;
   sb->rx = rx;
   sb->ty = ty;
   set_btn_size(sb);

   sb->rgn = NULL;
   sb->sensebox = NULL;

   if (sb->style->setup_rtn != NULL) {
      (*sb->style->setup_rtn)(sb);
    };

   PROTECT;
   setup_window(sb);
   UNPROTECT;

   if ((fgs & STEM_BTN_NO_DRAW) == 0) {
      if (sb->refresh && sb->resize) STEMbtn_window_draw(sb->window);
      else STEMbtn_draw(sb);
    };

   return sb;
};





/************************************************************************/
/*									*/
/*	STEMbtn_background -- fill background region for buttons	*/
/*									*/
/************************************************************************/


void
STEMbtn_background(sb,lx,by,rx,ty)
   STEM_BTN sb;
   Integer lx,by,rx,ty;
{
   Integer fl,cl,bcl;
   Integer bk,wt;
   ASH_WINDOW w;

   ENTER("STEMbtn_background 0x%x %d %d %d %d",sb,lx,by,rx,ty);

   if (sb->window == NULL) return;

   w = sb->window;

   if (lx == 0 && rx == 0 && ty == 0 && by == 0) {
      ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
    };

   bk = sb->fg;
   wt = sb->bg;

   if (sb->fill) {
      fl = ASHfill(w,sb->fill_style);
      cl = ASHcolor(w,bk);
      bcl = ASHbackground_color(w,wt);
    }
   else {
      fl = ASHfill(w,ASH_FILL_SOLID);
      cl = ASHcolor(w,sb->gray);
      bcl = ASHbackground_color(w,wt);
    };

   ASHrectangle(w,lx,by,rx,ty);
   ASHbackground_color(w,bcl);
   ASHcolor(w,cl);
   ASHfill(w,fl);
};





/************************************************************************/
/*									*/
/*	STEMbtn_size -- return size required for a button given name	*/
/*	STEMbtn_size_style -- return size when given specific style	*/
/*									*/
/************************************************************************/


void
STEMbtn_size(worf,txt,xszp,yszp)
   ASH_FONT worf;	      /* can be ASH_WINDOW */
   String txt;
   Integer * xszp, * yszp;
{
   ENTER("STEMbtn_size 0x%x %s",worf,txt);

   set_cur_style();

   get_btn_size(cur_style,worf,txt,xszp,yszp);
};






void
STEMbtn_size_style(sty,worf,txt,xszp,yszp)
   String sty;
   ASH_FONT worf;	      /* can be ASH_WINDOW */
   String txt;
   Integer * xszp, * yszp;
{
   STEM_BTN_STYLE sbs;

   ENTER("STEMbtn_size_style 0x%x %s",worf,txt);

   sbs = find_style(sty);

   get_btn_size(sbs,worf,txt,xszp,yszp);
};





static void
get_btn_size(sbs,worf,txt,xszp,yszp)
   STEM_BTN_STYLE sbs;
   ASH_FONT worf;	      /* can be ASH_WINDOW */
   String txt;
   Integer * xszp, * yszp;
{
   Integer xs,ys;

   ENTER("STEMbtn_size 0x%x %s",worf,txt);

   if (worf == STEM_BTN_DEFAULT_FONT) worf = btn_font;

   btn_text_size(worf,txt,&xs,&ys);

   if (sbs->size_rtn != NULL) {
      (*sbs->size_rtn)(worf,txt,&xs,&ys);
    };

   if (xszp != NULL) *xszp = xs;
   if (yszp != NULL) *yszp = ys;
};






/************************************************************************/
/*									*/
/*	STEMbtn_draw -- draw a button					*/
/*									*/
/************************************************************************/


void
STEMbtn_draw(sb)
   STEM_BTN sb;
{
   ASH_WINDOW w;
   String s,t;

   ENTER("STEMbtn_draw 0x%x",sb);

   w = sb->window;
   if (w == NULL) return;

   ASHbatch_mode(TRUE);

   if (sb->center && !sb->nofill) STEMbtn_background(sb,sb->lx,sb->by,sb->rx,sb->ty);

   if (sb->style->draw_rtn != NULL && !sb->dummy) {
      (*sb->style->draw_rtn)(sb);
    };

   if (!sb->enable) {
      if (sb->rgn != NULL) RIPremove_region(sb->rgn);
      sb->rgn = NULL;
      if (sb->sensebox != NULL) ASHsensitive_remove(sb->sensebox);
      sb->sensebox = NULL;
    }
   else if (sb->sensebox == NULL && sb->rgn == NULL &&
	  (sb->sense != NULL || sb->rtn != NULL)) {
      if (sb->rtn != NULL) {
	 sb->rgn = RIPdefine_region(w,sb->blx,sb->bby,sb->brx,sb->bty,
				       RIP_NO_CHARS,RIP_BTN_ANY_EITHER,
				       sb->rtn,sb->sense);
	 RIPset_data(sb->rgn,sb->data);
	 if (sb->id != NULL) RIPset_region_id(sb->rgn,sb->id);
	 else if (sb->text != NULL) {
	    s = rindex(sb->text,'\n');
	    if (s == NULL) s = sb->text;
	    else ++s;
	    t = rindex(s,'\f');
	    if (t != NULL) s = ++t;
	    RIPset_region_id(sb->rgn,s);
	  };
       }
      else if (sb->sense != NULL) {
	 sb->sensebox = ASHsensitive_area(w,sb->blx,sb->bby,sb->brx,sb->bty,
					     sb->sense);
       };
    };

   sb->drawn = TRUE;

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	STEMbtn_window_draw -- draw all buttons in a window		*/
/*									*/
/************************************************************************/


void
STEMbtn_window_draw(w)
   ASH_WINDOW w;
{
   Integer ct;
   STEM_BTN sb;
   Sequence l;

   ct = 0;
   forin (sb,STEM_BTN,l,all_stem_btns) {
      if (sb->window == w) {
	 if (ct == 0) STEMbtn_background(sb,0,0,0,0);
	 STEMbtn_draw(sb);
	 ++ct;
       };
    };
};





/************************************************************************/
/*									*/
/*	STEMbtn_remove -- remove a stem button				*/
/*									*/
/************************************************************************/


void
STEMbtn_remove(sb)
   STEM_BTN sb;
{
   if (sb == NULL || sb->window == NULL) return;

   PROTECT;
   all_stem_btns = REMOB(sb,all_stem_btns);
   UNPROTECT;

   if (sb->rgn != NULL) RIPremove_region(sb->rgn);
   sb->rgn = NULL;
   if (sb->sensebox != NULL) ASHsensitive_remove(sb->sensebox);
   sb->sensebox = NULL;

   if (USE_THREADS) {
      sb->window = NULL;
      sb = (STEM_BTN) realloc(sb,4);
    }
   else {
      sb->window = NULL;
      free(sb);
    };
};





/************************************************************************/
/*									*/
/*	STEMbtn_window_remove -- remove all buttons for a window	*/
/*									*/
/************************************************************************/


void
STEMbtn_window_remove(w)
   ASH_WINDOW w;
{
   STEM_BTN sb;
   Sequence l;

   forin (sb,STEM_BTN,l,all_stem_btns) {
      if (sb->window == w) {
	 STEMbtn_remove(sb);
       };
    };
};





/************************************************************************/
/*									*/
/*	STEMbtn_set -- turn a button on or off				*/
/*	STEMbtn_hilite -- flip hilighting on a button			*/
/*	STEMbtn_select -- show button as selected			*/
/*	STEMbtn_enable -- show button as enabled or not 		*/
/*									*/
/************************************************************************/


void
STEMbtn_set(sb,fg)
   STEM_BTN sb;
   Boolean fg;
{
   if (sb->window == NULL) return;

   sb->status = fg;
   if (sb->drawn && !sb->dummy) {
      if (sb->style->set_rtn != NULL) {
	 (*sb->style->set_rtn)(sb);
       }
      else if (sb->style->draw_rtn != NULL) {
	 (*sb->style->draw_rtn)(sb);
       }
    };
};





void
STEMbtn_hilite(sb,fg)
   STEM_BTN sb;
   Boolean fg;
{
   if (sb->window == NULL) return;

   sb->hilite = fg;
   if (sb->drawn && !sb->dummy) {
      if (sb->style->hilite_rtn != NULL) {
	 (*sb->style->hilite_rtn)(sb);
       }
      else if (sb->style->draw_rtn != NULL) {
	 (*sb->style->draw_rtn)(sb);
       }
    };
};





void
STEMbtn_select(sb,fg)
   STEM_BTN sb;
   Boolean fg;
{
   if (sb->window == NULL) return;

   sb->select = fg;
   if (sb->drawn && !sb->dummy) {
      if (sb->style->select_rtn != NULL) {
	 (*sb->style->select_rtn)(sb);
       }
      else if (sb->style->draw_rtn != NULL) {
	 (*sb->style->draw_rtn)(sb);
       }
    };
};





void
STEMbtn_enable(sb,fg)
   STEM_BTN sb;
   Boolean fg;
{
   if (sb->window == NULL) return;

   sb->enable = fg;
   if (sb->drawn && !sb->dummy) {
      if (sb->style->enable_rtn != NULL) {
	 (*sb->style->enable_rtn)(sb);
       }
      else if (sb->style->draw_rtn != NULL) {
	 (*sb->style->draw_rtn)(sb);
       }
    };
};





/************************************************************************/
/*									*/
/*	STEMbtn_set_gray -- set gray color for button			*/
/*	STEMbtn_set_select -- set select color for button		*/
/*	STEMbtn_set_id -- set ID for region help, etc.			*/
/*									*/
/************************************************************************/


void
STEMbtn_set_gray(sb,c)
   STEM_BTN sb;
   ASH_COLOR c;
{
   sb->gray = c;
};





void
STEMbtn_set_select(sb,c)
   STEM_BTN sb;
   ASH_COLOR c;
{
   sb->selcol = c;
};





void
STEMbtn_set_id(sb,id)
   STEM_BTN sb;
   String id;
{
   if (id == NULL) sb->id = NULL;
   else sb->id = SALLOC(id);

   if (sb->rgn != NULL) RIPset_region_id(sb->rgn,sb->id);
};





/************************************************************************/
/*									*/
/*	STEM_btn_define_style -- define a new button style		*/
/*									*/
/************************************************************************/


void
STEM_btn_define_style(name,setup,size,draw,set,hilite,select,enable)
   String name;
   Function_Ptr setup;
   Function_Ptr size;
   Function_Ptr draw;
   Function_Ptr set;
   Function_Ptr hilite;
   Function_Ptr select;
   Function_Ptr enable;
{
   STEM_BTN_STYLE sbs;

   sbs = PALLOC(STEM_BTN_STYLE_INFO);
   sbs->name = SALLOC(name);
   sbs->setup_rtn = setup;
   sbs->size_rtn = size;
   sbs->draw_rtn = draw;
   sbs->set_rtn = set;
   sbs->hilite_rtn = hilite;
   sbs->select_rtn = select;
   sbs->enable_rtn = enable;

   PROTECT;
   sbs->next = all_styles;
   all_styles = sbs;
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	STEM_btn_size_string -- get sizes on a complex string		*/
/*									*/
/************************************************************************/


Integer
STEM_btn_size_string(ft0,txt,xp,yp,xsz,ysz)
   ASH_FONT ft0;
   String txt;
   Integer * xp, * yp;
   Integer * xsz, * ysz;
{
   Integer ct,bct,fct;
   Integer maxx,maxy,totx,toty;
   Integer x,y;
   Character buf[1024],fbuf[128];
   ASH_FONT ft;

   ct = 0;
   maxx = 0;
   maxy = 0;
   totx = 0;
   toty = 0;
   bct = 0;
   ft = ft0;

   for ( ; ; ) {
      if (*txt != '\f' && *txt != '\n' && *txt != 0) {
	 buf[bct++] = *txt++;
	 buf[bct] = 0;
       }
      else {
	 if (bct > 0) {
	    if (*txt != 0) ASHinq_text_next(ft,buf,&x,&y);
	    else ASHinq_text_next(ft,buf,&x,&y);
	    totx += x;
	    maxy = MAX(maxy,y);
	    bct = 0;
	    buf[0] = 0;
	  };
	 if (*txt == '\f') {
	    fct = 0;
	    ++txt;
	    while (*txt != '\f' && *txt != 0) fbuf[fct++] = *txt++;
	    fbuf[fct] = 0;
	    if (*txt == '\f') ++txt;
	    if (fct == 0) ft = ft0;
	    else ft = get_btn_font(fbuf);
	  }
	 else {
	    if (xsz != NULL) xsz[ct] = totx;
	    if (ysz != NULL) ysz[ct] = maxy;
	    ++ct;
	    maxx = MAX(maxx,totx);
	    toty += maxy;
	    totx = 0;
	    maxy = 0;
	    if (*txt == 0) break;
	    else ++txt;
	  };
       };
    };

   *xp = maxx;
   *yp = toty;

   return ct;
};





/************************************************************************/
/*									*/
/*	STEM_btn_draw_string -- draw a complex string in a box		*/
/*									*/
/************************************************************************/


void
STEM_btn_draw_string(w,txt,lx,by,rx,ty,left)
   ASH_WINDOW w;
   String txt;
   Integer lx,by,rx,ty;
   Boolean left;
{
   Integer xtot,ytot;
   Integer xsz[MAX_LINES],ysz[MAX_LINES];
   Integer ct,i,x,y,dx,dy;
   Integer bct,fct;
   Character buf[1024],fbuf[128];
   ASH_FONT ft,ft0;

   if (txt == NULL) txt = "";

   ASHclip_region(w,lx,by,rx,ty);
   ASHclip(w,TRUE);

   ct = STEM_btn_size_string(ASHinq_font(w),txt,&xtot,&ytot,xsz,ysz);

   dy = (ty < by ? 1 : -1);
   dx = (lx < rx ? 1 : -1);

   ysz[ct] = by;
   y = (abs(ty-by)+1-ytot)/2;
   for (i = ct-1; i >= 0; --i) {
      x = ysz[i];
      ysz[i] = ysz[i+1]-y*dy;
      y = x;
    };

   for (i = 0; i < ct; ++i) {
      if (left) xsz[i] = lx;
      else {
	 x = (abs(rx-lx)+1-xsz[i])/2;
	 xsz[i] = lx+x*dx;
       };
    };

   ct = 0;
   bct = 0;
   ft0 = ASHinq_font(w);

   for ( ; ; ) {
      if (*txt != '\f' && *txt != '\n' && *txt != 0) {
	 buf[bct++] = *txt++;
	 buf[bct] = 0;
       }
      else {
	 if (bct > 0) {
	    ASHinq_text_offset(w,buf,&x,&y);
	    ASHtext(w,xsz[ct]+x*dx,ysz[ct]-y*dy,buf);
	    if (*txt == '\f') {
	       ASHinq_text_next(w,buf,&x,&y);
	       xsz[ct] += x;
	     };
	    bct = 0;
	    buf[0] = 0;
	  };
	 if (*txt == '\f') {
	    fct = 0;
	    ++txt;
	    while (*txt != '\f' && *txt != 0) fbuf[fct++] = *txt++;
	    fbuf[fct] = 0;
	    if (*txt == '\f') ++txt;
	    if (fct == 0) ft = ft0;
	    else ft = get_btn_font(fbuf);
	    ASHfont(w,ft);
	  }
	 else {
	    ++ct;
	    if (*txt == 0) break;
	    else ++txt;
	  };
       };
    };

   ASHclip(w,FALSE);
   ASHfont(w,ft0);
};





/************************************************************************/
/*									*/
/*	set_btn_size -- set size of actual button in its region 	*/
/*									*/
/************************************************************************/


static void
set_btn_size(sb)
   STEM_BTN sb;
{
   Integer x,y,d;

   get_btn_size(sb->style,sb->font,sb->text,&x,&y);

   sb->center = FALSE;
   d = (abs(sb->rx-sb->lx) - x);
   if (d <= 0 || sb->fullsize) {
      sb->blx = sb->lx;
      sb->brx = sb->rx;
    }
   else {
      sb->center = TRUE;
      if (sb->rx > sb->lx) {
	 sb->blx = sb->lx + d/2;
	 sb->brx = sb->blx+x-1;
       }
      else {
	 sb->blx = sb->lx - d/2;
	 sb->brx = sb->blx-x+1;
       };
    };

   d = abs(sb->ty-sb->by)-y;
   if (d <= 0 || sb->fullsize) {
      sb->bty = sb->ty;
      sb->bby = sb->by;
    }
   else {
      sb->center = TRUE;
      if (sb->ty < sb->by) {
	 sb->bty = sb->ty + d/2;
	 sb->bby = sb->bty + y - 1;
       }
      else {
	 sb->bty = sb->ty - d/2;
	 sb->bby = sb->bty - y + 1;
       };
    };
};





/************************************************************************/
/*									*/
/*	btn_sense_set -- handle sensitive flip				*/
/*	btn_sense_hilite -- handle sensitive hilite			*/
/*	find_btn -- find button given region				*/
/*									*/
/************************************************************************/


static void
btn_sense_set(w,lx,by,rx,ty,fg)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
   Boolean fg;
{
   STEM_BTN sb;

   sb = find_btn(w,lx,by,rx,ty);
   if (sb != NULL) STEMbtn_set(sb,fg);
};





static void
btn_sense_hilite(w,lx,by,rx,ty,fg)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
   Boolean fg;
{
   STEM_BTN sb;

   sb = find_btn(w,lx,by,rx,ty);
   if (sb != NULL) STEMbtn_hilite(sb,fg);
};





static STEM_BTN
find_btn(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   STEM_BTN sb;
   Sequence l;

   forin (sb,STEM_BTN,l,all_stem_btns) {
      if (sb->window == w && sb->blx == lx && sb->bby == by &&
	     sb->brx == rx && sb->bty == ty) break;
    };

   return sb;
};





/************************************************************************/
/*									*/
/*	setup_window -- handle window control and refresh		*/
/*									*/
/************************************************************************/


static void
setup_window(sb)
   STEM_BTN sb;
{
   Boolean refresh,ctl;
   STEM_BTN nsb;
   Sequence l;

   refresh = sb->refresh;
   ctl = TRUE;

   PROTECT;
   forin (nsb,STEM_BTN,l,all_stem_btns) {
      if (nsb->window == sb->window) {
	 if (nsb->refresh) refresh = FALSE;
	 ctl = FALSE;
	 if (!refresh) break;
       };
    };

   all_stem_btns = APPEND(sb,all_stem_btns);
   UNPROTECT;

   if (refresh) ASHset_region_refresh(sb->window,stem_btn_refresh);
   if (ctl) ASHset_control(sb->window,stem_btn_control);
};





/************************************************************************/
/*									*/
/*	stem_btn_refresh -- handle window refresh request		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
stem_btn_refresh(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   STEMbtn_window_draw(w);
};





/************************************************************************/
/*									*/
/*	stem_btn_control -- handle control messages for window		*/
/*									*/
/************************************************************************/


static Integer
stem_btn_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   STEM_BTN sb;
   Sequence l;

   if (STREQL(msg,"ASH$RESIZE")) {
      forin (sb,STEM_BTN,l,all_stem_btns) {
	 if (sb->window == w && sb->resize) {
	    ASHinq_size(w,ASH_SIZE_WINDOW,&sb->lx,&sb->by,&sb->rx,&sb->ty);
	    set_btn_size(sb);
	  };
       };
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      forin (sb,STEM_BTN,l,all_stem_btns) {
	 if (sb->window == w) {
	    sb->rgn = NULL;
	    sb->sensebox = NULL;
	    STEMbtn_remove(sb);
	  };
       };
    };

   return ASH_CONTROL_REJECT;
};






/************************************************************************/
/*									*/
/*	btn_text_size -- get size of text for a button			*/
/*									*/
/************************************************************************/


static void
btn_text_size(ft,txt,xp,yp)
   ASH_FONT ft;
   String txt;
   Integer * xp, * yp;
{
   if (txt == NULL) {
      *xp = 5;
      *yp = 5;
    }
   else {
      STEM_btn_size_string(ft,txt,xp,yp,NULL,NULL);
    };
};





/************************************************************************/
/*									*/
/*	get_btn_font -- get font from name				*/
/*									*/
/************************************************************************/


static ASH_FONT
get_btn_font(s)
   String s;
{
   String t;

   for (t = s; isdigit(*t); ++t);

   if (*t == 0) return (ASH_FONT) atol(s);

   return ASHloadfont(s);
};





/************************************************************************/
/*									*/
/*	set_cur_style -- set default style				*/
/*	find_style -- find style given name				*/
/*									*/
/************************************************************************/


static void
set_cur_style()
{
   String s;

   if (cur_style == NULL) {
      s = ASHinq_resource("style");
      cur_style = find_style(s);
    };
};





static STEM_BTN_STYLE
find_style(s)
   String s;
{
   STEM_BTN_STYLE sbs;

   if (s == NULL) s = DEFAULT_STYLE;
   for (sbs = all_styles; sbs != NULL; sbs = sbs->next) {
      if (STREQL(s,sbs->name)) break;
    };
   if (sbs == NULL) sbs = all_styles;

   return sbs;
};





/* end of stembtn.c */

