/************************************************************************/
/*									*/
/*		stem3d.c						*/
/*									*/
/*	Three-dimensional button routines for STEM_BTN types		*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define GRAY_SIZE		4
#define GRAY_INSIDE_FRACTION	33
#define SHADOW_SIZE		2
#define X_BORDER_SIZE		2
#define Y_BORDER_SIZE		1
#define BW_ON_FILL_STYLE	ASH_FILL_HALFTONE





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 		gray_size;
static	Integer 		gray_fract;
static	Integer 		shadow_size;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void			setup_3d();
static	void			size_3d();
static	void			draw_3d();
static	void			set_3d();
static	void			hilite_3d();
static	void			select_3d();

static	void			draw_set();
static	void			draw_hilite();
static	void			draw_select();
static	void			btn_text_draw();




/************************************************************************/
/*									*/
/*	STEM_3d_init -- module initialization				*/
/*									*/
/************************************************************************/


void
STEM_3d_init()
{
   String s;

   s = ASHinq_resource("btn.shadow_size");
   if (s == NULL) shadow_size = SHADOW_SIZE;
   else {
      shadow_size = atol(s);
      if (shadow_size == 0) shadow_size = SHADOW_SIZE;
    };

   s = ASHinq_resource("btn.gray_size");
   if (s == NULL) {
      gray_size = GRAY_SIZE;
      gray_fract = 0;
    }
   else if (s[0] == '%') {
      gray_size = 0;
      gray_fract = atol(&s[1]);
      if (gray_fract == 0) gray_fract = GRAY_INSIDE_FRACTION;
    }
   else {
      gray_size = atol(s);
      gray_fract = 0;
    };

   STEM_btn_define_style("3D",setup_3d,size_3d,draw_3d,set_3d,hilite_3d,select_3d,
			    draw_3d);
};






/************************************************************************/
/*									*/
/*	setup_3d -- handle button definitions				*/
/*									*/
/************************************************************************/


static void
setup_3d(sb)
   STEM_BTN sb;
{
   Integer x,y;

   if (gray_fract == 0) sb->gray_size = gray_size;
   else {
      ASHinq_text_next(sb->font,"Xp",&x,&y);
      sb->gray_size = gray_fract*y/100;
    };
};





/************************************************************************/
/*									*/
/*	size_3d -- handle sizing request				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
size_3d(ft,txt,xszp,yszp)
   ASH_FONT ft;
   String txt;
   Integer *xszp;
   Integer *yszp;
{
   Integer xs,ys;
   Integer x,y,g;

   if (xszp != NULL) xs = *xszp;
   else xs = 0;
   if (yszp != NULL) ys = *yszp;
   else ys = 0;

   if (gray_fract == 0) g = gray_size;
   else {
      ASHinq_text_next(ft,"Xp",&x,&y);
      g = gray_fract*y/100;
    };

   xs += 2*X_BORDER_SIZE;
   ys += 2*Y_BORDER_SIZE;

   xs += 2*g;
   ys += 2*g;

   xs += 2*shadow_size;
   ys += 2*shadow_size;

   if (xszp != NULL) *xszp = xs;
   if (yszp != NULL) *yszp = ys;
};





/************************************************************************/
/*									*/
/*	draw_3d -- handle 3d button full draw				*/
/*									*/
/************************************************************************/


static void
draw_3d(sb)
   STEM_BTN sb;
{
   draw_set(sb);
   draw_select(sb);
   btn_text_draw(sb);
};





/************************************************************************/
/*									*/
/*	set_3d -- handle 3d button set					*/
/*									*/
/************************************************************************/


static void
set_3d(sb)
   STEM_BTN sb;
{
   draw_set(sb);
   btn_text_draw(sb);
};





/************************************************************************/
/*									*/
/*	hilite_3d -- handle 3d button hilite				*/
/*									*/
/************************************************************************/


static void
hilite_3d(sb)
   STEM_BTN sb;
{
   draw_hilite(sb);
};





/************************************************************************/
/*									*/
/*	select_3d -- handle 3d button select				*/
/*									*/
/************************************************************************/


static void
select_3d(sb)
   STEM_BTN sb;
{
   draw_select(sb);
   btn_text_draw(sb);
};






/************************************************************************/
/*									*/
/*	draw_set -- draw the 3d portion of a button on or off		*/
/*									*/
/************************************************************************/


static void
draw_set(sb)
   STEM_BTN sb;
{
   ASH_WINDOW w;
   Integer fl,cl,bcl;
   ASH_COLOR bk,wt;
   Integer lx,by,rx,ty;
   Integer xdl,ydl;

   ASHbatch_mode(TRUE);

   if (!sb->status) {
      bk = sb->fg;
      wt = sb->bg;
    }
   else {
      wt = sb->fg;
      bk = sb->bg;
    }

   w = sb->window;
   lx = sb->blx;
   by = sb->bby;
   rx = sb->brx;
   ty = sb->bty;
   xdl = sb->xdl;
   ydl = sb->ydl;

   fl = ASHfill(w,ASH_FILL_SOLID);
   cl = ASHcolor(w,wt);
   bcl = ASHbackground_color(w,wt);

   ASHrectangle(w,lx,by,lx+(SHADOW_SIZE-1)*xdl,ty);
   ASHrectangle(w,lx+shadow_size*xdl,ty+(shadow_size-1)*ydl,rx,ty);

   ASHcolor(w,bk);
   ASHrectangle(w,lx+shadow_size*xdl,by,rx,by-(shadow_size-1)*ydl);
   ASHrectangle(w,rx-(shadow_size-1)*xdl,by-shadow_size*ydl,rx,ty+shadow_size*ydl);

   ASHcolor(w,cl);
   ASHbackground_color(w,bcl);
   ASHfill(w,fl);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	draw_hilite -- highlight a button				*/
/*									*/
/************************************************************************/


static void
draw_hilite(sb)
   STEM_BTN sb;
{
   Integer lx,by,rx,ty;

   lx = sb->blx;
   rx = sb->brx;
   ty = sb->bty;
   by = sb->bby;

   lx += shadow_size*sb->xdl;
   rx -= shadow_size*sb->xdl;
   by -= shadow_size*sb->ydl;
   ty += shadow_size*sb->ydl;

   lx += sb->gray_size*sb->xdl;
   rx -= sb->gray_size*sb->xdl;
   by -= sb->gray_size*sb->ydl;
   ty += sb->gray_size*sb->ydl;

   ASHhilite_box(sb->window,lx,by,rx,ty);
};





/************************************************************************/
/*									*/
/*	draw_select -- draw inside gray area, possibly in select mode	*/
/*									*/
/************************************************************************/


static void
draw_select(sb)
   STEM_BTN sb;
{
   Integer lx,by,rx,ty;
   Integer cl,fl,bcl;

   bcl = ASHbackground_color(sb->window,sb->bg);

   lx = sb->blx + shadow_size*sb->xdl;
   rx = sb->brx - shadow_size*sb->xdl;
   by = sb->bby - shadow_size*sb->ydl;
   ty = sb->bty + shadow_size*sb->ydl;

   if (sb->fill) {
      cl = ASHcolor(sb->window,sb->fg);
      if (sb->select) fl = ASHfill(sb->window,ASH_FILL_SOLID);
      else fl = ASHfill(sb->window,sb->fill_style);
    }
   else {
      if (sb->select) cl = ASHcolor(sb->window,sb->selcol);
      else cl = ASHcolor(sb->window,sb->gray);
      fl = ASHfill(sb->window,ASH_FILL_SOLID);
    };

   ASHrectangle(sb->window,lx,by,rx,ty);

   ASHfill(sb->window,fl);
   ASHcolor(sb->window,cl);
   ASHbackground_color(sb->window,bcl);
};






/************************************************************************/
/*									*/
/*	btn_text_draw -- draw text inside a 3d button			*/
/*									*/
/************************************************************************/


static void
btn_text_draw(sb)
   STEM_BTN sb;
{
   Integer ft,fl,cl,bcl,tcl,tbcl;
   Integer lx,by,rx,ty;

   lx = sb->blx + shadow_size*sb->xdl;
   rx = sb->brx - shadow_size*sb->xdl;
   by = sb->bby - shadow_size*sb->ydl;
   ty = sb->bty + shadow_size*sb->ydl;

   lx += sb->gray_size*sb->xdl;
   rx -= sb->gray_size*sb->xdl;
   by -= sb->gray_size*sb->ydl;
   ty += sb->gray_size*sb->ydl;

   tcl = ASHtext_color(sb->window,sb->fg);
   bcl = ASHbackground_color(sb->window,sb->bg);
   ft = ASHfont(sb->window,sb->font);

   if (sb->fill) {
      tbcl = ASHtext_background_color(sb->window,-1);
      if (sb->status) {
	 fl = ASHfill(sb->window,BW_ON_FILL_STYLE);
	 cl = ASHcolor(sb->window,sb->fg);
       }
      else {
	 fl = ASHfill(sb->window,ASH_FILL_SOLID);
	 cl = ASHcolor(sb->window,sb->bg);
       };
    }
   else {
      fl = ASHfill(sb->window,ASH_FILL_SOLID);
      if (sb->status) {
	 cl = ASHcolor(sb->window,sb->oncol);
	 tbcl = ASHtext_background_color(sb->window,sb->oncol);
       }
      else {
	 cl = ASHcolor(sb->window,sb->bg);
	 tbcl = ASHtext_background_color(sb->window,sb->bg);
       };
    };

   ASHrectangle(sb->window,lx,by,rx,ty);
   ASHfill(sb->window,ASH_FILL_SOLID);

   lx += X_BORDER_SIZE*sb->xdl;
   rx -= X_BORDER_SIZE*sb->xdl;
   by -= Y_BORDER_SIZE*sb->ydl;
   ty += Y_BORDER_SIZE*sb->ydl;

   STEM_btn_draw_string(sb->window,sb->text,lx,by,rx,ty,sb->left);

   if (!sb->enable) ASHstipple_box(sb->window,lx,by,rx,ty);
   else if (sb->hilite) ASHhilite_box(sb->window,lx,by,rx,ty);

   ASHfont(sb->window,ft);
   ASHfill(sb->window,fl);
   ASHcolor(sb->window,cl);
   ASHbackground_color(sb->window,bcl);
   ASHtext_color(sb->window,tcl);
   ASHtext_background_color(sb->window,tbcl);
};






/* end of stem3d.c */
