/************************************************************************/
/*									*/
/*		stem2d.c						*/
/*									*/
/*	Standard 2d button routines for STEM_BTN types			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "stem_local.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define BORDER_SIZE		1
#define BW_ON_FILL_STYLE	ASH_FILL_VERY_LIGHT





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	Integer 		border_size;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void			setup_2d();
static	void			size_2d();
static	void			draw_2d();
static	void			set_2d();
static	void			hilite_2d();
static	void			select_2d();

static	void			draw_set();
static	void			draw_hilite();
static	void			draw_select();
static	void			btn_text_draw();




/************************************************************************/
/*									*/
/*	STEM_2d_init -- module initialization				*/
/*									*/
/************************************************************************/


void
STEM_2d_init()
{
   border_size = BORDER_SIZE;

   STEM_btn_define_style("2D",setup_2d,size_2d,draw_2d,
			    set_2d,hilite_2d,select_2d,
			    draw_2d);
};






/************************************************************************/
/*									*/
/*	setup_2d -- handle button definitions				*/
/*									*/
/************************************************************************/


static void
setup_2d(sb)
   STEM_BTN sb;
{
   sb->gray = sb->bg;
};





/************************************************************************/
/*									*/
/*	size_2d -- handle sizing request				*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
size_2d(ft,txt,xszp,yszp)
   ASH_FONT ft;
   String txt;
   Integer *xszp;
   Integer *yszp;
{
   Integer xs,ys;

   if (xszp != NULL) xs = *xszp;
   else xs = 0;
   if (yszp != NULL) ys = *yszp;
   else ys = 0;

   xs += 2*border_size;
   ys += 2*border_size;

   if (xszp != NULL) *xszp = xs;
   if (yszp != NULL) *yszp = ys;
};





/************************************************************************/
/*									*/
/*	draw_2d -- handle 2d button full draw				*/
/*									*/
/************************************************************************/


static void
draw_2d(sb)
   STEM_BTN sb;
{
   draw_set(sb);
   draw_select(sb);
   btn_text_draw(sb);
};





/************************************************************************/
/*									*/
/*	set_2d -- handle 2d button set				*/
/*									*/
/************************************************************************/


static void
set_2d(sb)
   STEM_BTN sb;
{
   draw_set(sb);
   btn_text_draw(sb);
};





/************************************************************************/
/*									*/
/*	hilite_2d -- handle 2d button hilite			*/
/*									*/
/************************************************************************/


static void
hilite_2d(sb)
   STEM_BTN sb;
{
   draw_hilite(sb);
};





/************************************************************************/
/*									*/
/*	select_2d -- handle 2d button select			*/
/*									*/
/************************************************************************/


static void
select_2d(sb)
   STEM_BTN sb;
{
   draw_select(sb);
};






/************************************************************************/
/*									*/
/*	draw_set -- draw the 2d portion of a button on or off		*/
/*									*/
/************************************************************************/


static void
draw_set(sb)
   STEM_BTN sb;
{
   Integer fl,cl;
   ASH_COLOR bk;

   ASHbatch_mode(TRUE);

   bk = (sb->status ? sb->fg : sb->bg);

   fl = ASHfill(sb->window,ASH_FILL_SOLID);
   cl = ASHcolor(sb->window,bk);
   ASHbox(sb->window,sb->blx,sb->bby,sb->brx,sb->bty);
   ASHcolor(sb->window,cl);
   ASHfill(sb->window,fl);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	draw_hilite -- highlight a button				*/
/*									*/
/************************************************************************/


static void
draw_hilite(sb)
   STEM_BTN sb;
{
   Integer lx,by,rx,ty;

   lx = sb->blx;
   rx = sb->brx;
   ty = sb->bty;
   by = sb->bby;

   lx += border_size*sb->xdl;
   rx -= border_size*sb->xdl;
   by -= border_size*sb->ydl;
   ty += border_size*sb->ydl;

   ASHhilite_box(sb->window,lx,by,rx,ty);
};





/************************************************************************/
/*									*/
/*	draw_select -- draw inside gray area, possibly in select mode	*/
/*									*/
/************************************************************************/


static void
draw_select(sb)
   STEM_BTN sb;
{
   Integer lx,by,rx,ty;
   Integer cl,fl,bcl;

   bcl = ASHbackground_color(sb->window,sb->bg);

   lx = sb->blx + border_size*sb->xdl;
   rx = sb->brx - border_size*sb->xdl;
   by = sb->bby - border_size*sb->ydl;
   ty = sb->bty + border_size*sb->ydl;

   if (sb->fill) {
      cl = ASHcolor(sb->window,sb->fg);
      if (sb->select) fl = ASHfill(sb->window,ASH_FILL_SOLID);
      else fl = ASHfill(sb->window,sb->fill_style);
    }
   else {
      if (sb->select) cl = ASHcolor(sb->window,sb->selcol);
      else cl = ASHcolor(sb->window,sb->gray);
      fl = ASHfill(sb->window,ASH_FILL_SOLID);
    };

   ASHrectangle(sb->window,lx,by,rx,ty);

   ASHfill(sb->window,fl);
   ASHcolor(sb->window,cl);
   ASHbackground_color(sb->window,bcl);
};






/************************************************************************/
/*									*/
/*	btn_text_draw -- draw text inside a 2d button		    */
/*									*/
/************************************************************************/


static void
btn_text_draw(sb)
   STEM_BTN sb;
{
   Integer ft,fl,cl,bcl,tcl,tbcl;
   Integer lx,by,rx,ty;

   lx = sb->blx + border_size*sb->xdl;
   rx = sb->brx - border_size*sb->xdl;
   by = sb->bby - border_size*sb->ydl;
   ty = sb->bty + border_size*sb->ydl;

   tcl = ASHtext_color(sb->window,sb->fg);
   bcl = ASHbackground_color(sb->window,sb->bg);
   ft = ASHfont(sb->window,sb->font);
   tbcl = ASHtext_background_color(sb->window,-1);

   if (sb->fill) {
      if (sb->select) {
	 fl = ASHfill(sb->window,sb->fill_style);
	 cl = ASHcolor(sb->window,sb->fg);
       }
      else if (sb->status) {
	 fl = ASHfill(sb->window,BW_ON_FILL_STYLE);
	 cl = ASHcolor(sb->window,sb->fg);
       }
      else {
	 fl = ASHfill(sb->window,ASH_FILL_SOLID);
	 cl = ASHcolor(sb->window,sb->bg);
       };
    }
   else {
      fl = ASHfill(sb->window,ASH_FILL_SOLID);
      if (sb->select) {
	 cl = ASHcolor(sb->window,sb->selcol);
       }
      else if (sb->status) {
	 cl = ASHcolor(sb->window,sb->oncol);
       }
      else {
	 cl = ASHcolor(sb->window,sb->bg);
       };
    };

   ASHrectangle(sb->window,lx,by,rx,ty);
   ASHfill(sb->window,ASH_FILL_SOLID);

   STEM_btn_draw_string(sb->window,sb->text,lx,by,rx,ty,sb->left);

   if (!sb->enable) ASHstipple_box(sb->window,lx,by,rx,ty);
   else if (sb->hilite) ASHhilite_box(sb->window,lx,by,rx,ty);

   ASHfont(sb->window,ft);
   ASHfill(sb->window,fl);
   ASHcolor(sb->window,cl);
   ASHbackground_color(sb->window,bcl);
   ASHtext_color(sb->window,tcl);
   ASHtext_background_color(sb->window,tbcl);
};






/* end of stem2d.c */
