/************************************************************************/
/*									*/
/*				  R I P 				*/
/*									*/
/*		    Region-oriented Input/output Package		*/
/*									*/
/*									*/
/*			    Stefan K. S. Tucker 			*/
/*									*/
/*									*/
/************************************************************************/
/*									*/
/*			     region routines				*/
/*									*/
/*  RIPdefine_region()							*/
/*									*/
/*  RIPset_data()							*/
/*  RIPset_sense()							 */
/*  RIPinq_data()							*/
/*  RIPinq_window()							*/
/*  RIPset_region_id()							*/
/*  RIPinq_region_id()							*/
/*									*/
/*  RIPremove_region()							*/
/*  RIPremove_window_regions()						*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "rip_local.h"






/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


	RIP_REGION	RIP__head = NULL;





/************************************************************************/
/*									*/
/*  local routines							*/
/*									*/
/************************************************************************/


static	Short * 	scan_events();
static	int		remove_window();

#ifdef LIB
static			cleanup_rtn();
#endif



/************************************************************************/
/*									*/
/*  RIP_region_init -- module initialization				*/
/*									*/
/************************************************************************/


void
RIP_region_init()
{
   ITRACE("RIP_region_init");

   RIP__head = NULL;
};





/************************************************************************/
/*									*/
/*  RIPdefine_region:							*/
/*									*/
/*	defines a new region. If the window that is passed has		*/
/*  not been used before, a control routine is registered for it.	*/
/*  This control routine will trap any ASH$REMOVE messages and, 	*/
/*  therefore, be able to remove the window's regions.                  */
/*									*/
/************************************************************************/


RIP_REGION
RIPdefine_region(window, lx, by, rx, ty, events_ch, events_btn, rtn, sense)
   ASH_WINDOW window;
   Integer lx, by, rx, ty, events_btn;
   String events_ch;
   Function_Ptr rtn;
   ASH_SENSE_TYPE sense;
{
   Boolean used;
   register RIP_REGION nrgn, rgn;

   CHECKINIT;

   ETRACE("RIPdefine_region: (%d, %d) -> (%d, %d), events_ch: '%s', events_btn: 0x%x, sense = 0x%x",
	     lx, by, rx, ty, events_ch, events_btn, sense);

   if (window == NULL) window = ASHinq_top();

   nrgn = RGNALLOC();
   if (nrgn == NULL) {
      ERROR("RIPdefine_region", NOSTORAGE);
      return RIP_BAD;
    }

   if (lx == 0 && rx == 0 && by == 0 && ty == 0) {
      ASHinq_size(window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
      nrgn->whole_window = TRUE;
    }
   else nrgn->whole_window = FALSE;

   nrgn->window = window;
   nrgn->lx = lx;
   nrgn->by = by;
   nrgn->rx = rx;
   nrgn->ty = ty;

   nrgn->events.all_chars = STREQL(events_ch,RIP_ALL_CHARS);
   nrgn->events.events_ch = scan_events(events_ch);

   if ((events_btn != RIP_BTN_NONE) && !(events_btn & RIP_BTN_EITHER))
      events_btn |= RIP_BTN_EITHER;
   nrgn->events.events_btn = events_btn;

   nrgn->rtn = rtn;
   nrgn->data = NULL;
   nrgn->id = NULL;
   nrgn->enable = TRUE;
   nrgn->sense = sense;
   nrgn->local_window = FALSE;

   used = FALSE;
   for (rgn = RIP__head; !used && rgn != NULL; rgn = rgn->next) {
      if (rgn->window == window) used = TRUE;
    };
   if (!used) {
      BIOnew_input_window(window);
      ASHset_control(window,remove_window);
    };

   if (sense != 0) {
      nrgn->area_id = ASHsensitive_area(window, lx, by, rx, ty, sense);
    }
   else nrgn->area_id = NULL;

   PROTECT;
   nrgn->next = RIP__head;
   RIP__head = nrgn;
   UNPROTECT;

   ETRACE("<-- RIPdefine_region: returned: 0x%x", nrgn);

   return nrgn;
}




/************************************************************************/
/*									*/
/*  RIPset_data:							*/
/*									*/
/*	associates the passed data with the passed region.		*/
/*									*/
/************************************************************************/


void
RIPset_data(region_id, data)
   RIP_REGION region_id;
   RIP_DATA data;
{
   CHECKINIT;

   ETRACE("RIPset_data: region_id = 0x%x, data: 0x%x", region_id, data);

   region_id->data = data;
};






/************************************************************************/
/*									*/
/*  RIPset_sense:							*/
/*									*/
/*	changes the sense value for the passed region			*/
/*									*/
/************************************************************************/


void
RIPset_sense(region_id, sense)
   RIP_REGION region_id;
   ASH_SENSE_TYPE sense;
{
   CHECKINIT;

   ETRACE("RIPset_sense: region_id = 0x%x, sense: %d", region_id, sense);

   region_id->sense = sense;

   if (region_id->area_id != NULL) {
      ASHsensitive_remove(region_id->area_id);
      region_id->area_id = NULL;
    };

   if (region_id->sense != 0) {
      region_id->area_id =
	 ASHsensitive_area(region_id->window,
			      region_id->lx,region_id->by,
			      region_id->rx,region_id->ty,
			      region_id->sense);
    };
};






/************************************************************************/
/*									*/
/*  RIPinq_data:							*/
/*									*/
/*	returns the data that is associated with the passed region.	*/
/*									*/
/************************************************************************/


RIP_DATA
RIPinq_data(region_id)
   RIP_REGION region_id;
{
   CHECKINIT;

   ETRACE("RIPinq_data: region_id = 0x%x", region_id);

   return region_id->data;
};





/************************************************************************/
/*									*/
/*  RIPinq_window:							*/
/*									*/
/*	returns the window associated with the passed region.		*/
/*									*/
/************************************************************************/


ASH_WINDOW
RIPinq_window(region_id)
   RIP_REGION region_id;
{
   CHECKINIT;

   ETRACE("RIPinq_window: region_id = 0x%x", region_id);

   return region_id->window;
};





/************************************************************************/
/*									*/
/*  RIPset_region_id							*/
/*									*/
/*	associates the passed string as the id of the region (for help) */
/*									*/
/************************************************************************/


void
RIPset_region_id(region_id, data)
   RIP_REGION region_id;
   String data;
{
   CHECKINIT;

   ETRACE("RIPset_region_id: region_id = 0x%x, data: %s", region_id, data);

   region_id->id = data;
};






/************************************************************************/
/*									*/
/*  RIPinq_region_id							*/
/*									*/
/*	returns the string associated as the id of the region		*/
/*									*/
/************************************************************************/


char *
RIPinq_region_id(region_id)
   RIP_REGION region_id;
{
   CHECKINIT;

   ETRACE("RIPinq_region_id: region_id = 0x%x", region_id);

   return region_id->id;
};





/********************************************************************************/
/*										*/
/*	RIPuse_local_window -- indicate that region should use its local window */
/*										*/
/********************************************************************************/


void
RIPuse_local_window(region_id)
   RIP_REGION region_id;
{
   CHECKINIT;

   ETRACE("RIPuse_local_window: region_id = 0x%x", region_id);

   region_id->local_window = TRUE;
};





/************************************************************************/
/*									*/
/*  RIPremove_region:							*/
/*									*/
/*	removes the passed region from the list of defined regions.	*/
/*									*/
/************************************************************************/


void
RIPremove_region(region_id)
   RIP_REGION region_id;
{
   register RIP_REGION rgn;
   Boolean used;

   CHECKINIT;

   ETRACE("RIPremove_region: region_id = 0x%x", region_id);

   PROTECT;
   if (RIP__head == region_id) RIP__head = region_id->next;
   else if (RIP__head != NULL) {
      for (rgn = RIP__head; rgn->next != NULL; rgn = rgn->next) {
	 if (rgn->next == region_id) break;
       };
      if (rgn->next == NULL) {
	 UNPROTECT;
	 return;
       };
      rgn->next = region_id->next;
    };
   RIP_region_invalidate(region_id);

   UNPROTECT;

   if (region_id->area_id != NULL) ASHsensitive_remove(region_id->area_id);

   used = FALSE;
   for (rgn = RIP__head; !used && rgn != NULL; rgn = rgn->next) {
      if (rgn->window == region_id->window) used = TRUE;
    };
   if (!used) ASHremove_control(region_id->window,remove_window);

   FREE(region_id);
};





/************************************************************************/
/*									*/
/*  RIPremove_window_regions:						*/
/*									*/
/*	removes all of the regions associated with the passed window	*/
/*									*/
/************************************************************************/


void
RIPremove_window_regions(window)
   ASH_WINDOW window;
{
   register RIP_REGION rgn,nxt;

   CHECKINIT;

   ETRACE("RIPremove_window_regions");

   RIP_region_invalidate(NULL);

   for (rgn = RIP__head; rgn != NULL; rgn = nxt) {
      nxt = rgn->next;
      if (rgn->window == window) RIPremove_region(rgn);
    };
};





/************************************************************************/
/*									*/
/*  scan_events:							*/
/*									*/
/*	converts escape sequences into hex codes.			*/
/*									*/
/************************************************************************/


static Short *
scan_events(events_ch)
   unsigned char *events_ch;
{
   register Integer offset, i;
   Short *new_events_ch, *new2;
   Character key_type, temp_code[2];
   Integer code, last_code;

   ITRACE("scan_events: events: '%s'", events_ch);

   if (STREMPTY(events_ch)) return NULL;

   new_events_ch = new2 = (Short *) malloc(2*(strlen(events_ch) + 1 + 128));
   *new_events_ch = EOS;

   if (*events_ch == DASH)
      *new2++ = *events_ch++;

   while (!STREMPTY(events_ch) && (*events_ch == DASH))
      ++events_ch;

   if (!STREMPTY(events_ch) && (*(events_ch + strlen(events_ch) - 1) == DASH)) {
      if (*new_events_ch != DASH)
	 *new2++ = DASH;
      *(events_ch + strlen(events_ch) - 1) = EOS;
    }

   while (!STREMPTY(events_ch) && (*(events_ch + strlen(events_ch) - 1) == DASH))
      *(events_ch + strlen(events_ch) - 1) = EOS;

   while (!STREMPTY(events_ch)) {
      if (*events_ch == ESC) {
	 ++events_ch;
	 if ((*events_ch == ESC) || (*events_ch == DASH))
	    *new2++ = *events_ch++;
	 else if (*events_ch >= '0' && *events_ch <= '7') {
	    offset = 0;
	    i = 0;
	    while (i < 3 && *events_ch >= '0' && *events_ch <= '7') {
	       offset = offset*8 + *events_ch - '0';
	       ++events_ch;
	       ++i;
	     };
	    *new2++ = offset;
	  }
	 else if ((*events_ch == fKEY) || (*events_ch == LKEY) || (*events_ch == RKEY)) {
	    key_type = *events_ch++;

	    if (*events_ch == DOWN)
	       offset = FKEY_FIRST;
	    else if (*events_ch == SHIFT)
	       offset = FKEY_FIRST_SHIFT;
	    else if (*events_ch == UP)
	       offset = FKEY_FIRST_UP;
	    else if ( (*events_ch == CTRL) && (key_type == fKEY) )
	       offset = FKEY_FIRST_CTRL;
	    else {
	       ++events_ch;
	       continue;
	     }

	    ++events_ch;
	    temp_code[0] = *events_ch & 0xff;
	    temp_code[1] = EOS;
	    sscanf(temp_code, "%x", &code);

	    if (key_type == fKEY) code += BPIO_KEY_BASE;
	    else if (key_type == LKEY) code += BPIO_LEFT_BASE;
	    else if (key_type == RKEY) code += BPIO_RIGHT_BASE;
	    else printf("This should never have happened\n");

	    *new2++ = code+offset;

	    ++events_ch;

	    if (*events_ch == DASH) {
	       ++events_ch;

	       temp_code[0] = *events_ch & 0xff;
	       temp_code[1] = EOS;
	       sscanf(temp_code, "%x", &last_code);

	       if (last_code < FIELD_SIZE) {
		  for (i = code + 1; i <= last_code; ++i) {
		     *new2++ = i+offset;
		   }
		};

	       ++events_ch;
	     }
	  }

	 ++events_ch;
       }
      else
	 *new2++ = *events_ch++;
    }

   *new2++ = EOS;

   i = ((int) new2) - ((int) new_events_ch);
   new2 = (Short *) malloc(i+2);
   bcopy(new_events_ch,new2,i);
   FREE(new_events_ch);

   return new2;
}





/************************************************************************/
/*									*/
/*  remove_window:							*/
/*									*/
/*	traps ASH$REMOVE messages for windows with regions and removes	*/
/*  that window's regions.                                              */
/*									*/
/************************************************************************/


static int
remove_window(msg, window)
   char *msg;
   ASH_WINDOW window;
{
   register RIP_REGION rgn;
   Integer lx,by,rx,ty;

   ITRACE("--> remove_window: msg: '%s'", msg);

   if (STREQL(msg, "ASH$REMOVE")) {
      RIPremove_window_regions(window);
    }
   else if (STREQL(msg, "ASH$RESIZE")) {
      for (rgn = RIP__head; rgn != NULL; rgn = rgn->next) {
	 if (rgn->window == window && rgn->whole_window) {
	    ASHinq_size(window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
	    rgn->lx = lx;
	    rgn->by = by;
	    rgn->rx = rx;
	    rgn->ty = ty;
	    if (rgn->sense != 0) {
	       ASHsensitive_remove(rgn->area_id);
	       rgn->area_id = ASHsensitive_area(window,lx,by,rx,ty,rgn->sense);
	     };
	  };
       };
    };

   return ASH_CONTROL_REJECT;
}





/* end of ripregion.c */

