/************************************************************************/
/*									*/
/*				  R I P 				*/
/*									*/
/*		    Region-oriented Input/output Package		*/
/*									*/
/*									*/
/*			    Stefan K. S. Tucker 			*/
/*									*/
/*									*/
/************************************************************************/
/*									*/
/*			 picking/tracking routines			*/
/*									*/
/*  RIPenable_region()							*/
/*  RIPenable_window()							*/
/*									*/
/*  RIPgrab_window()							*/
/*  RIPrelease_window() 						*/
/*									*/
/*  RIPtrack()								*/
/*									*/
/*  RIPuser_pick()							*/
/*  RIPprog_pick()							*/
/*									*/
/*  RIPinq_region()							*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "rip_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_WAIT	1024

#define WAIT_DATA1(th)	(*((WAIT_FOR *) THREADdynamic(th,rip_thread_offset)))
#define WAIT_DATA()	(USE_THREADS ? WAIT_DATA1(THREADinq_current()) : wait_data)





/************************************************************************/
/*									*/
/*	local storage							*/
/*									*/
/************************************************************************/


static	RIP_REGION	last_region = NULL;

typedef struct _WAIT_FOR {
   RIP_REGION	   region;
   Integer	   x;
   Integer	   y;
   Integer	   ch;
   Integer	   btns;
   Integer	   ctr;
} WAIT_FOR_INFO, *WAIT_FOR;

static	WAIT_FOR	wait_data = NULL;
static	Integer 	rip_thread_offset;






/************************************************************************/
/*									*/
/*  local routines							*/
/*									*/
/************************************************************************/


static	void		rip_thread_initer();
static	Boolean 	user_pick();
static	int		match_event();
static	Boolean 	next_pick();
static	Boolean 	call_rtn();
static	void		do_help();



/************************************************************************/
/*									*/
/*	RIP_pick_init -- module initialization				*/
/*									*/
/************************************************************************/


void
RIP_pick_init()
{
   ITRACE("RIP_pick_init");

   HELPregister_info(RIPhelp_info);

   last_region = NULL;
   if (!USE_THREADS) {
      wait_data = PALLOC(WAIT_FOR_INFO);
      wait_data->region = NULL;
      wait_data->ctr = 0;
    };
};





/************************************************************************/
/*									*/
/*	RIPthread_init -- handle thread-based initialization		*/
/*									*/
/************************************************************************/


void
RIPthread_init()
{
   rip_thread_offset = THREADtype1_register(4,rip_thread_initer,NULL,NULL,NULL);
};





static void
rip_thread_initer(th)
   THREAD th;
{
   WAIT_DATA1(th) = PALLOC(WAIT_FOR_INFO);

   WAIT_DATA1(th)->region = NULL;
};






/************************************************************************/
/*									*/
/*  RIPgrab_window:							*/
/*									*/
/*	makes all picks seem as if they were done in the passed window. */
/*  Thus, all picks have their coordinates mapped into the coordinate	*/
/*  system of the grabbed window.					*/
/*	Regions in other windows are not highlighted or picked. 	*/
/*									*/
/************************************************************************/


ASH_WINDOW
RIPgrab_window(window)
   ASH_WINDOW window;
{
   CHECKINIT;

   ETRACE("RIPgrab_window");

   return BIOgrab_io(window);
};




/************************************************************************/
/*									*/
/*  RIPtrack:								*/
/*									*/
/*	first calls the passed routine with the current x- and		*/
/*  y-coordinates of the mouse, the current number of clicks (zero),	*/
/*  a flag as RIP_TRANS_NONE (indicating no recent event), and the	*/
/*  maximum number of clicks.						*/
/*	Then it tracks the cursor until a mouse button is clicked or a	*/
/*  key is pressed calling the routine with the x- and y-coordinates,	*/
/*  0 as the number of clicks, the transition is RIP_TRANS_MOVE, and	*/
/*  the maximum number of clicks. Then it calls the routine with the	*/
/*  current x and y, 1 for the current number of clicks, the flag as	*/
/*  RIP_TRANS_DOWN (a down transition), and the maximum number of	*/
/*  clicks.								*/
/*	Then it tracks the cursor again with the transition as		*/
/*  RIP_TRANS_MOVE (just mouse movement). When the mouse button is	*/
/*  released, the routine is called with the current x and y, 2 for	*/
/*  the number of clicks, the flag as RIP_TRANS_UP (an up transition),	*/
/*  and the maximum number of clicks.					*/
/*	Simply, up and down transitions of a mouse button each count	*/
/*  as a separate click. A key press also counts as a click and the	*/
/*  flag is passed as RIP_TRANS_CHAR.					*/
/*	This continues until the routine returns zero, in which case	*/
/*  RIPtrack() aborts and returns zero, or until the number of clicks	*/
/*  equals the passed maximum number of clicks, in which case		*/
/*  RIPtrack() stops and returns one.					*/
/*	If the maximum number of clicks is less than zero, RIPtrack()	*/
/*  will not return until the routine returns zero.			*/
/*	If the passed routine is NULL, then no routine is called,	*/
/*  but the tracking is still done.					*/
/*									*/
/************************************************************************/


int
RIPtrack(rtn, maxclick, window, allup)
   Function_Ptr rtn;
   int maxclick;
   ASH_WINDOW window;
   Boolean allup;
{
   Integer x_old, y_old, click, flag;
   Integer x, y, btns;
   Integer xw,yw;
   Boolean cfg;
   Integer ch;
   RIP_TRANS trans;
   ASH_WINDOW scrn,w,sw;
   ASH_WINDOW inwin;

   CHECKINIT;

   ETRACE("RIPtrack: maxclick = %d, window = 0x%x, allup = %d", maxclick,window,allup);

   if (window != NULL) scrn = ASHinq_top_window(window);
   else scrn = ASHinq_top();

   w = (window == NULL ? scrn : window);

   if (rtn != NULL) {
      flag = (*rtn)(0, 0, 0, RIP_TRANS_NONE, maxclick, w, 0, 0);
      if (flag == FALSE) return FALSE;
    };

   cfg = BIOset_cursor(NULL,TRUE);

   /* set old coordinates */
   x_old = -1;
   y_old = -1;

   flag = TRUE;
   click = 0;
   while ((click < maxclick) || (maxclick < 0)) {
      do {
	 trans = RIP_get_input(FALSE, &x, &y, &btns, &ch, &inwin);
       }
      while (trans == RIP_TRANS_NONE);

      if ((trans == RIP_TRANS_CHAR) || (trans == RIP_TRANS_DOWN) ||
	     (trans == RIP_TRANS_UP) || (trans == RIP_TRANS_TAP)) {
	 if (!allup || trans != RIP_TRANS_UP) ++click;
	 if (trans != RIP_TRANS_CHAR) allup = FALSE;
       };

      if ((x != x_old) || (y != y_old)) {
	 x_old = x;
	 y_old = y;
       }

      if (rtn != NULL) {
	 sw = ASHinq_top_window(inwin);
	 if (window == NULL || sw == window) {
	    flag = (*rtn)(x, y, click, trans, maxclick, sw, (1<<btns), ch);
	  }
	 else {
	    ASHmap(sw,x,y,window,&xw,&yw);
	    flag = (*rtn)(xw, yw, click, trans, maxclick, window, (1<<btns), ch);
	  };
	 if (flag == FALSE) {
	    trans = RIP_TRANS_NONE;
	    break;
	  };
       };
    }

   if (!cfg) BIOset_cursor(NULL,cfg);

   return flag;
};





/************************************************************************/
/*									*/
/*	RIPidler -- a general purpose idle routine			*/
/*									*/
/************************************************************************/


void
RIPidler(fg)
   Integer fg;
{
   CMPXselect(fg);

   RIPuser_poll(NULL);
};






/************************************************************************/
/*									*/
/*  RIPuser_pick:							*/
/*									*/
/*	waits for a pick.						*/
/*	If a region was picked, it then calls the routine associated	*/
/*  with the picked region. It passes the coordinates of the pick, the	*/
/*  btns or char that caused the pick, and the id of the picked region. */
/*	If no region was picked, it passes NULL as the region id.	*/
/*	If there is no routine associated with the picked region or no	*/
/*  region was picked, the passed routine is called.			*/
/*	It returns the value that the called routine returns.		*/
/*									*/
/*	RIPuser_poll -- handle input if present, return otherwise	*/
/*									*/
/*									*/
/************************************************************************/


int
RIPuser_pick(dflt_rtn)
   Function_Ptr dflt_rtn;
{
   Boolean flag;

   CHECKINIT;

   ETRACE("RIPuser_pick");

   flag = user_pick(dflt_rtn,TRUE);

   return flag;
};





int
RIPuser_poll(dflt_rtn)
   Function_Ptr dflt_rtn;
{
   Boolean flag;

   CHECKINIT;

   ETRACE("RIPuser_poll");

   flag = user_pick(dflt_rtn,FALSE);

   return flag;
};







static Boolean
user_pick(dflt_rtn,waitfg)
   Function_Ptr dflt_rtn;
   Boolean waitfg;
{
   Integer x, y, btns, ch;
   Boolean flag;
   RIP_REGION region_id;
   ASH_WINDOW win;
   ASH_LOCK_STATUS sts;
   WAIT_FOR wait;
   Integer oct;

   PROTECT;
   wait = WAIT_DATA();
   if (wait->region != NULL) {
      region_id = wait->region;
      x = wait->x;
      y = wait->y;
      ch = wait->ch;
      btns = wait->btns;
    }
   else {
      UNPROTECT;
      if (!next_pick(waitfg,&x,&y,&btns,&ch,&win)) region_id = NULL;
      else {
	 region_id = RIPinq_region(NULL,x,y,btns,ch,win);
	 if (region_id != NULL) RIPstatus(NULL);
       };
      PROTECT;
    };

   if (region_id == NULL) {
      UNPROTECT;
      if (!waitfg) flag = call_rtn(dflt_rtn,region_id,x,y,ch,btns);
    }
   else {
      oct = wait->ctr;
      wait->ctr = 0;
      wait->region = NULL;
      win = region_id->window;
      sts = ASHinput_lock(win,FALSE);
      flag = FALSE;
      if (sts == ASH_LOCK_GRANTED) {
	 UNPROTECT;
	 flag = call_rtn(dflt_rtn, region_id, x, y, ch, btns);
	 ASHinput_unlock(win);
       }
      else if (sts == ASH_LOCK_FAIL) {
	 UNPROTECT;
       }
      else if (oct >= MAX_WAIT) {
	 UNPROTECT;
	 RIPstatus("RIP:  Timeout:  Event Discarded");
       }
      else {
	 wait->region = region_id;
	 wait->x = x;
	 wait->y = y;
	 wait->ch = ch;
	 wait->btns = btns;
	 wait->ctr = oct+1;
	 UNPROTECT;
       };
    };

   return flag;
};






/************************************************************************/
/*									*/
/*  RIPinq_region:							*/
/*									*/
/*	returns the id of the region in the passed window at the	*/
/*  passed coordinates. The coordinates are assumed to be in the	*/
/*  screen's coordinate system. If the passed window is NULL, all       */
/*  of the windows below the passed location are checked. If the	*/
/*  passed window is not NULL, just the topmost window at the		*/
/*  passed location is checked. 					*/
/*	To be found, a region must be enabled, defined in the window	*/
/*  being searched, and either the passed 'ignore' flag must be         */
/*  non-zero or the most recent pick (during RIPnext_pick() or		*/
/*  RIPuser_pick()) must be an event that it's sensitive to.            */
/*									*/
/************************************************************************/


RIP_REGION
RIPinq_region(window, x, y, btns, ch, inwin)
   ASH_WINDOW window;
   Integer x, y;
   Integer btns, ch;
   ASH_WINDOW inwin;
{
   Integer x_window, y_window;
   ASH_WINDOW w,top;
   RIP_REGION rgn;
   Boolean flag;

   CHECKINIT;

   ETRACE("RIPinq_region: (%d, %d) %d %d", x, y, btns, ch);

   if (inwin != NULL) top = ASHinq_top_window(inwin);
   else if (window != NULL) top = ASHinq_top_window(window);
   else top = ASHinq_top();

   if (inwin != NULL) w = inwin;
   else w = ASHfind_child_window(top, x, y);

   if (last_region != NULL && btns == RIP_BTN_NONE) {
      rgn = last_region;
      if (rgn->enable && (rgn->window == w) && match_event(rgn->events, ch, btns)) {
	 flag = ASHmap(top, x, y, w, &x_window, &y_window);
	 if (flag)
	    if (BTWN(x_window, rgn->lx, rgn->rx))
	       if (BTWN(y_window, rgn->by, rgn->ty)) {
		  return rgn;
		};
       };
    };

   PROTECT;
   rgn = NULL;
   while (w != NULL) {
      if (window == NULL || w == window) {
	 for (rgn = RIP__head; rgn != NULL; rgn = rgn->next) {
	    if (rgn->enable && (rgn->window == w) && match_event(rgn->events, ch, btns)) {
	       flag = ASHmap(top, x, y, w, &x_window, &y_window);
	       if (flag)
		  if (BTWN(x_window, rgn->lx, rgn->rx))
		     if (BTWN(y_window, rgn->by, rgn->ty))
			break;
	     };
	  };
       };
      if ((window != NULL) || (rgn != NULL)) break;
      w = ASHinq_parent(w);
    };

   last_region = rgn;
   UNPROTECT;

   return rgn;
}


/************************************************************************/
/*									*/
/*	RIPhelp_info -- handle info requests for help			*/
/*									*/
/************************************************************************/


char *
RIPhelp_info(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   RIP_REGION rgn;
   Boolean flag;
   Integer x_window,y_window;
   ASH_WINDOW hw;

   if (w == NULL) return NULL;

   for (hw = w; hw != NULL; hw = ASHinq_parent(hw)) {
      for (rgn = RIP__head; rgn != NULL; rgn = rgn->next) {
	 if (rgn->enable && (rgn->window == hw)) {
	    flag = ASHmap(w, x, y, hw, &x_window, &y_window);
	    if (flag)
	       if (BTWN(x_window, rgn->lx, rgn->rx))
	       if (BTWN(y_window, rgn->by, rgn->ty))
	       break;
	  };
       };
      if (rgn != NULL && rgn->id != NULL) return rgn->id;
      if (ASHinq_window_id(hw) != NULL) break;
    };

   return NULL;
};






/************************************************************************/
/*									*/
/*	RIP_region_invalidate -- invalidate last_region if it matches	*/
/*									*/
/************************************************************************/


void
RIP_region_invalidate(rgn)
   RIP_REGION rgn;
{
   WAIT_FOR wait;

   ITRACE("RIP_region_invalidate 0x%x",rgn);

   if (rgn == NULL) last_region = NULL;
   else if (last_region == rgn) last_region = NULL;

   wait = WAIT_DATA();
   if (wait->region == rgn) {
      wait->region = NULL;
      wait->ctr = 0;
    };
};




/************************************************************************/
/*									*/
/*	RIP_track_event -- get next relevant event for window		*/
/*									*/
/************************************************************************/


RIP_TRANS
RIP_track_event(waitfg,window,xp,yp,btn,ch,winp)
   Boolean waitfg;
   ASH_WINDOW window;
   Integer *xp, *yp, *btn, *ch;
   ASH_WINDOW * winp;
{
   Integer x_old, y_old, click;
   Boolean cfg;
   RIP_TRANS trans;
   ASH_WINDOW scrn;

   CHECKINIT;

   ITRACE("RIP_track_event: 0x%x",window);

   if (window != NULL) scrn = ASHinq_top_window(window);
   else scrn = ASHinq_top();
   if (window == scrn) window = NULL;

   cfg = BIOset_cursor(NULL,TRUE);

   /* set old coordinates */
   x_old = -1;
   y_old = -1;

   click = 0;
   while (click == 0) {
      do {
	 trans = RIP_get_input(waitfg, xp,yp,btn,ch,winp);
	 if (!waitfg) break;
       }
      while (trans == RIP_TRANS_NONE);

      if (trans == RIP_TRANS_NONE) break;

      if ((trans == RIP_TRANS_CHAR) || (trans == RIP_TRANS_DOWN) ||
	     (trans == RIP_TRANS_UP) || (trans == RIP_TRANS_TAP))
	 ++click;

      if ((*xp != x_old) || (*yp != y_old)) {
	 x_old = *xp;
	 y_old = *yp;
       }
    };

   if (!cfg) BIOset_cursor(NULL,cfg);

   return trans;
};





/************************************************************************/
/*									*/
/*  match_event:							*/
/*									*/
/*	checks to see if the passed events are in the passed structure	*/
/*  of possible events. If it is, it returns a non-zero integer. If it	*/
/*  is not, it returns zero.						*/
/*									*/
/************************************************************************/

static int
match_event(events, event_ch, event_btn)
   EVENTS events;
   unsigned int event_ch;
   int event_btn;
{
   Boolean match;
   Short * ix;

   event_ch &= 0xff;

   match = FALSE;

   if (event_btn != RIP_BTN_NONE) {
      if ((events.events_btn & event_btn) == event_btn)
	 match = TRUE;
    }
   else if (events.all_chars && event_ch != EOS) {
      match = TRUE;
    }
   else if (event_ch != EOS && events.events_ch != NULL) {
      /* scan event.events_ch looking for event_ch */
      ix = events.events_ch;

      if (*ix == DASH) {
	 if (event_ch == DASH)
	    match = TRUE;
	 else
	    ++ix;
       };

      while (!STREMPTY(ix) && !match) {
	 /* interpret dashes as ranges */
	 if (*ix == DASH) {
	    if ((event_ch >= *(ix - 1)) && (event_ch <= *(ix + 1)))
	       match = TRUE;
	    else
	       ++ix;
	  }
	 else if (event_ch == *ix)
	    match = TRUE;
	 else
	    ++ix;
       };
    };

   return match;
}





/************************************************************************/
/*									*/
/*	next_pick -- get next relevant pick in RIP terms		*/
/*									*/
/************************************************************************/


static Boolean
next_pick(waitfg,xp,yp,btnp,chp,winp)
   Boolean waitfg;
   Integer *xp, *yp;
   Integer *btnp;
   Integer *chp;
   ASH_WINDOW * winp;
{
   Integer btns, ch,xbt;
   RIP_TRANS trans;

   ITRACE("next_pick");

   trans = RIP_track_event(waitfg,NULL,xp,yp,&btns,&ch,winp);

   if (trans == RIP_TRANS_NONE) return FALSE;

   if ((btns & 0xf000) != 0) {
      xbt = RIP_BTN_META;
      btns >>= 12;
    }
   else if ((btns & 0x0f00) != 0) {
      xbt = RIP_BTN_CTRL;
      btns >>= 8;
    }
   else if ((btns & 0x00f0) != 0) {
      xbt = RIP_BTN_SHIFT;
      btns >>= 4;
    }
   else xbt = 0;

   switch (trans) {
      case RIP_TRANS_UP :
	 *btnp = (1 << btns) | RIP_BTN_UP | xbt;
	 *chp = EOS;
	 break;
      case RIP_TRANS_DOWN :
	 *btnp = (1 << btns) | RIP_BTN_DOWN | xbt;
	 *chp = EOS;
	 break;
      case RIP_TRANS_CHAR :
	 *btnp = RIP_BTN_NONE;
	 *chp = ch;
	 break;
      case RIP_TRANS_TAP :
	 *btnp = (1 << btns) | RIP_BTN_UP | xbt | RIP_BTN_TAP;
	 *chp = EOS;
	 break;
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*  call_rtn:								*/
/*									*/
/*	calls the routine associated with the passed region id. If this */
/*  routine is NULL, it calls the passed default routine. If the	*/
/*  default routine is NULL, the returned flag is zero. Otherwise,	*/
/*  it returns what the called routine returns. The routine is passed	*/
/*  the passed coordinates, character, and buttons.			*/
/*									*/
/************************************************************************/


static Boolean
call_rtn(dflt_rtn, region_id, x, y, ch, btns)
   Function_Ptr dflt_rtn;
   RIP_REGION region_id;
   int x, y, btns;
   int ch;
{
   Boolean flag;
   Function_Ptr rtn;

   ITRACE("call_rtn: region_id = 0x%x, (%d, %d), ch: <%c>, btns = 0x%x", region_id, x, y, ch, btns);

   if (region_id == NULL)
      rtn = dflt_rtn;
   else {
      rtn = region_id->rtn;
      if (rtn == NULL) rtn = dflt_rtn;
      if (region_id->local_window) {
	 ASHmap(ASHinq_top_window(region_id->window),x,y,region_id->window,&x,&y);
       };
    }

   if (rtn == NULL)
      flag = 0;
   else {
      flag = (*rtn)(x, y, ch, btns, region_id);
    };

   return flag;
};





/* end of rippick.c */
