/************************************************************************/
/*									*/
/*				  R I P 				*/
/*									*/
/*		    Region-oriented Input/output Package		*/
/*									*/
/*									*/
/*			    Stefan K. S. Tucker 			*/
/*									*/
/*									*/
/************************************************************************/
/*									*/
/*			  miscellaneous routines			*/
/*									*/
/*  RIPinit()								*/
/*									*/
/*  RIPtrace()								*/
/*									*/
/*  RIPregister_clock() 						*/
/*									*/
/*  RIPstatus() 							*/
/*  RIPprompt() 							*/
/*									*/
/*  RIParea()								*/
/*  RIPinq_area_window()						*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "rip_local.h"





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


	Boolean 	RIP__initialized = FALSE;
	Integer 	RIP__trace_level = 0;

	ASH_WINDOW	RIP__areas[NUM_AREAS];

	PROT_DECL;




/************************************************************************/
/*									*/
/*  local routines							*/
/*									*/
/************************************************************************/

#ifdef LIB
static	void	cleanup_rtn();
#endif




/************************************************************************/
/*									*/
/*  RIP_misc_init -- module initialization				*/
/*									*/
/************************************************************************/


void
RIP_misc_init()
{
   register Integer i;

   ITRACE("RIP_misc_init");

   for (i = 0; i < NUM_AREAS; ++i)
      RIP__areas[i] = NULL;
}





/************************************************************************/
/*									*/
/*  RIPinit:								*/
/*									*/
/*	initializes the RIP package.					*/
/*									*/
/************************************************************************/


void
RIPinit()
{
   ETRACE("RIPinit");

   PROT_INIT;

   PROTECT;
   if (!RIP__initialized) {
      BIOinit();
      BIOset_mask(NULL,USE_BIO_MASK,NULL);

      BIOflush_same(USE_BIO_MASK,FLUSH_BIO_MASK);

#ifdef LIB
      BROWNregister_cleanup(cleanup_rtn);
#endif

      RIP_region_init();
      RIP_pick_init();
      RIP_misc_init();
      RIP_inout_init();

      RIP__initialized = TRUE;
    };
   UNPROTECT;
}


#ifdef LIB

static void
cleanup_rtn()
{
   RIP__initialized = FALSE;
   RIP__trace_level = 0;
   RIPset_help(0,NULL);
}

#endif





/************************************************************************/
/*									*/
/*  RIPtrace:								*/
/*									*/
/*	sets the tracing level. 					*/
/*									*/
/************************************************************************/


void
RIPtrace(level)
   int level;
{
   CHECKINIT;

   RIP__trace_level = level;
};





/************************************************************************/
/*									*/
/*  RIPregister_clock:							*/
/*									*/
/*	registers a new clock with RIP. Every time that the passed	*/
/*  number of ticks pass, the routine that was passed will be called.	*/
/*									*/
/************************************************************************/


void
RIPregister_clock(rtn, trigger)
   Function_Ptr rtn;
   int trigger;
{
   CHECKINIT;

   ETRACE("RIPregister_clock: trigger = %d", trigger);

   CMPXregister_timeout(trigger,rtn);
}





/************************************************************************/
/*									*/
/*  RIPstatus:								*/
/*									*/
/*	displays the passed string in the status window. If there is no */
/*  window currently defined, the default is created. If the string is	*/
/*  empty, the status window is cleared.				*/
/*									*/
/************************************************************************/


void
RIPstatus(text)
   String text;
{
   CHECKINIT;

   ETRACE("RIPstatus: '%s'", text);

   PROTECT;
   if (RIP__areas[RIP_AREA_STATUS] == NULL && !STREMPTY(text)) {
      if (text != NULL) fprintf(stderr,"RIP: %s\n",text);
      return;
    };
   UNPROTECT;

   RIP_display_text(RIP__areas[RIP_AREA_STATUS], text);
};





/************************************************************************/
/*									*/
/*  RIPprompt:								*/
/*									*/
/*	displays the passed string in the prompt window. If there is no */
/*  window currently defined, the default is created. If the string is	*/
/*  empty, the prompt window is cleared.				*/
/*									*/
/************************************************************************/


void
RIPprompt(text)
   String text;
{
   CHECKINIT;

   ETRACE("RIPprompt: '%s'", text);

   PROTECT;
   if (RIP__areas[RIP_AREA_PROMPT] == NULL &&!STREMPTY(text)) {
      if (text != NULL) fprintf(stderr,"RIP: %s\n",text);
      return;
    };
   UNPROTECT;

   RIP_display_text(RIP__areas[RIP_AREA_PROMPT], text);
};





/************************************************************************/
/*									*/
/*  RIParea:								*/
/*									*/
/*	associates the passed area with the passed window.		*/
/*	'area' is either RIP_AREA_STATUS, RIP_AREA_PROMPT, or           */
/*  RIP_AREA_TYPEIN.							*/
/*	Thus, any text passed to RIPstatus() will be displayed in the	*/
/*  RIP_AREA_STATUS window; any text passed to RIPprompt() will be	*/
/*  displayed in the RIP_AREA_PROMPT window; any input that comes from	*/
/*  the keyboard will be echoed in the RIP_AREA_TYPEIN window.		*/
/*  (The prompt and typein areas can be in the same window.)		*/
/*	The id of the window previously associated with the named area	*/
/*  is returned.							*/
/*									*/
/************************************************************************/

ASH_WINDOW
RIParea(area, wndw)
   int area;
   ASH_WINDOW wndw;
{
   int lx, by, rx, ty, dy, w_max, h_max, width, height, base, vert_sp, horiz_sp;
   ASH_WINDOW old_wndw;

   CHECKINIT;

   ETRACE("RIParea: area = 0x%x", area);

   PROTECT;
   old_wndw = RIP__areas[area];

   if (wndw == NULL) {

      /* set default window */

      ASHinq_size(ASHinq_top(),ASH_SIZE_WINDOW, &lx, &by, &rx, &ty);

      dy = (by < ty) ? 1 : -1;

      w_max = ABS(lx - rx);
      h_max = ABS(by - ty);

      width = w_max * 98 / 100;
      height = h_max / 40;

      base = (by < ty) ? ty : by;
      base += (dy * (h_max / 80));

      horiz_sp = (w_max - width) / 2;
      vert_sp = height / 4;

      RIP__areas[area] = ASHcreate(ASHinq_top(),
				      horiz_sp, base + ( dy * (height * (2 - area)) - (vert_sp * (2 - area)) ),
				      0, 0, width, height, ASH_BORDER_THIN,
				      ASH_WINDOW_VISIBLE);
    }
   else if (RIP__areas[area] != wndw)
      RIP__areas[area] = wndw;

   UNPROTECT;

   return old_wndw;
};





/************************************************************************/
/*									*/
/*  RIPinq_area_window: 						*/
/*									*/
/*	returns the window associated with the passed area.		*/
/*									*/
/************************************************************************/


ASH_WINDOW
RIPinq_area_window(area)
   int area;
{
   CHECKINIT;

   ETRACE("RIPinq_area_window: area = 0x%x", area);

   return RIP__areas[area];
};






/************************************************************************/
/*									*/
/*  RIP_trace:								*/
/*									*/
/*	prints the passed parameters in the passed format to stdout.	*/
/*									*/
/************************************************************************/


void
RIP_trace(fmt, a0, a1, a2, a3, a4, a5, a6, a7, a8, a9)
   char *fmt;
   int a0, a1, a2, a3, a4, a5, a6, a7, a8, a9;
{
   printf("RIP trace: ");
   printf(fmt, a0, a1, a2, a3, a4, a5, a6, a7, a8, a9);
   printf("\n");
};





/************************************************************************/
/*									*/
/*  RIP_error:								*/
/*									*/
/*	prints the passed error message on standard output.		*/
/*									*/
/************************************************************************/


void
RIP_error(name, msg)
   char *name, *msg;
{
   printf("RIP: ERROR in %s: %s\n", name, msg);
   fflush(stdout);
}






/* end of ripmisc.c */
