/************************************************************************/
/*									*/
/*				  R I P 				*/
/*									*/
/*		    Region-oriented Input/output Package		*/
/*									*/
/*									*/
/*			    Stefan K. S. Tucker 			*/
/*									*/
/*									*/
/************************************************************************/
/*									*/
/*			   input/output routines			*/
/*									*/
/*  RIPinput_track()							*/
/*									*/
/*  RIPinput_point()							*/
/*  RIPinput_line()							*/
/*  RIPinput_box()							*/
/*  RIPinput_polygon()							*/
/*									*/
/*  RIPinput_ival()							*/
/*  RIPinput_rval()							*/
/*  RIPinput_text()							*/
/*									*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "rip_local.h"





/************************************************************************/
/*									*/
/*	Local data structures						*/
/*									*/
/************************************************************************/


typedef struct _POINTS {
   Integer count;
   Integer x[MAX_PTS];
   Integer y[MAX_PTS];
} POINTS_B,  * POINTS;


typedef struct _DRAWDATA {
   ASH_CURSOR	   old_cursor;
   Integer	   old_cr;
   Integer	   old_data;
   Integer	   old_ls;
   Integer	   old_fill;
   Integer	   old_thru;
   THREAD	   old_thread;
} DRAWDATA_B, * DRAWDATA;




/************************************************************************/
/*									*/
/*	local storage							*/
/*									*/
/************************************************************************/


static	ASH_CURSOR	track_cursor;




/************************************************************************/
/*									*/
/*  local routines							*/
/*									*/
/************************************************************************/


static	int		get_two_pts();
static	int		rubberband_line();
static	int		rubberband_box();
static	int		rubberband_polygon();
static	String		get_text();
static	ASH_WINDOW	make_overlay();
static	void		remove_overlay();





/************************************************************************/
/*									*/
/*	RIP_inout_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
RIP_inout_init()
{
   ITRACE("RIP_inout_init");

   track_cursor = ASHcursor_standard(INPUT_CURSOR);
};





/************************************************************************/
/*									*/
/*  RIPinput_track:							*/
/*									*/
/*	prints the passed prompt, and tracks the user input.		*/
/*  (See RIPtrack().) It returns the value that RIPtrack() returns.	*/
/*									*/
/************************************************************************/


int
RIPinput_track(window, prompt, rtn, maxclick)
   ASH_WINDOW window;
   String prompt;
   Function_Ptr rtn;
   Integer maxclick;
{
   register Boolean flag;
   register ASH_WINDOW overlay, old_window;
   DRAWDATA_B dd;

   CHECKINIT;

   ETRACE(" RIPinput_track: prompt: '%s', maxclick = %d", prompt, maxclick);

   old_window = RIPgrab_window(window);

   RIPprompt(prompt);

   overlay = make_overlay(window,NULL,&dd);

   flag = RIPtrack(rtn, maxclick, window, TRUE);

   remove_overlay(overlay,&dd);

   RIPprompt(EMPTY);

   RIPgrab_window(old_window);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_track_down:						*/
/*									*/
/*	similar to input track, but starts with button down.		*/
/*  (See RIPtrack().) It returns the value that RIPtrack() returns.	*/
/*									*/
/************************************************************************/

int
RIPinput_track_down(window, prompt, rtn, maxclick)
   ASH_WINDOW window;
   String prompt;
   Function_Ptr rtn;
   Integer maxclick;
{
   register Boolean flag;
   register ASH_WINDOW overlay, old_window;
   DRAWDATA_B dd;

   CHECKINIT;

   ETRACE("RIPinput_track_down: prompt: '%s', maxclick = %d", prompt, maxclick);

   old_window = RIPgrab_window(window);

   RIPprompt(prompt);

   overlay = make_overlay(window,NULL,&dd);

   flag = RIPtrack(rtn, maxclick, window, FALSE);

   remove_overlay(overlay,&dd);

   RIPprompt(EMPTY);

   RIPgrab_window(old_window);

   return flag;
};






/************************************************************************/
/*									*/
/*  RIPinput_point:							*/
/*									*/
/*	prints the passed prompt, requests the user to enter a point,	*/
/*  and returns the coordinates in the passed pointers. 		*/
/*	If the coordinates are inside of the passed window, it		*/
/*  returns 1. Otherwise, it returns 0. 				*/
/*									*/
/************************************************************************/


int
RIPinput_point(window, prompt, x_ptr, y_ptr)
   ASH_WINDOW window;
   String prompt;
   Integer *x_ptr, *y_ptr;
{
   register Boolean flag;
   register ASH_WINDOW old_window,overlay;
   Integer nx,ny,btn,ch;
   register RIP_TRANS trans;
   ASH_WINDOW w;
   DRAWDATA_B dd;

   CHECKINIT;

   ETRACE("RIPinput_point: 0x%x %s", window, prompt);

   RIPprompt(prompt);

   overlay = make_overlay(window,NULL,&dd);
   old_window = RIPgrab_window(overlay);

   trans = RIP_track_event(TRUE,overlay,x_ptr,y_ptr,&btn,&ch,NULL);
   if (trans == RIP_TRANS_DOWN) {
      while (RIP_track_event(TRUE,overlay,&nx,&ny,&btn,&ch,NULL) != RIP_TRANS_UP);
    };

   RIPgrab_window(old_window);
   remove_overlay(overlay,&dd);

   flag = TRUE;
   if (window != NULL) {
      for (w = ASHfind_child_window(ASHinq_top_window(window),*x_ptr,*y_ptr);
	   w != NULL && w != window;
	   w = ASHinq_parent(w));
      if (w == NULL) flag = FALSE;
    };

   /* need to map pts into coord syst of passed window? */
   if ((window != NULL) && (window != ASHinq_top_window(window)))
      ASHmap(ASHinq_top_window(window), *x_ptr, *y_ptr, window, x_ptr, y_ptr);

   RIPprompt(EMPTY);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_line:							*/
/*									*/
/*	prints the passed prompt, requests the user to enter a line,	*/
/*  and returns the coordinates of the endpoints in the passed pointers.*/
/*	If the coordinates are inside of the passed window, it		*/
/*  returns 1. Otherwise, it returns 0. 				*/
/*									*/
/************************************************************************/


int
RIPinput_line(window, prompt, x0_ptr, y0_ptr, x1_ptr, y1_ptr)
   ASH_WINDOW window;
   String prompt;
   Integer *x0_ptr, *y0_ptr, *x1_ptr, *y1_ptr;
{
   register Boolean flag;

   CHECKINIT;

   ETRACE("RIPinput_line: prompt: '%s'", prompt);

   flag = get_two_pts(prompt, window, rubberband_line, x0_ptr, y0_ptr, x1_ptr, y1_ptr);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_box:							*/
/*									*/
/*	prints the passed prompt, requests the user to enter a box,	*/
/*  and returns the coordinates of the opposite corners in the passed	*/
/*  pointers.								*/
/*	If the coordinates are inside of the passed window, it		*/
/*  returns 1. Otherwise, it returns 0. 				*/
/*									*/
/************************************************************************/


int
RIPinput_box(window, prompt, x0_ptr, y0_ptr, x1_ptr, y1_ptr)
   ASH_WINDOW window;
   String prompt;
   Integer *x0_ptr, *y0_ptr, *x1_ptr, *y1_ptr;
{
   register Boolean flag;

   CHECKINIT;

   ETRACE("RIPinput_box: prompt: '%s'", prompt);

   flag = get_two_pts(prompt, window, rubberband_box, x0_ptr, y0_ptr, x1_ptr, y1_ptr);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_polygon:							*/
/*									*/
/*	prints the passed prompt, requests the user to enter a polygon	*/
/*  by selecting subsequent vertices, and returns the coordinates of	*/
/*  the vertices in the passed arrays. The number of vertices is also	*/
/*  passed.								*/
/*	If the coordinates are inside of the passed window, it		*/
/*  returns 1. If an illegal number of points was requested, it returns */
/*  RIP_BAD_PTS. Otherwise, it returns 0.				*/
/*									*/
/************************************************************************/


int
RIPinput_polygon(window, prompt, numpts, x_pt, y_pt)
   ASH_WINDOW window;
   String prompt;
   Integer *numpts, x_pt[], y_pt[];
{
   int pt, flag;
   ASH_WINDOW overlay, old_window;
   POINTS_B pts;
   ASH_WINDOW w;
   DRAWDATA_B dd;

   CHECKINIT;

   ETRACE("RIPinput_polygon: prompt: '%s', numpts = %d", prompt, *numpts);

   /* is numpts out of bounds? */
   if ( ((*numpts >= 0) && (*numpts <= 2)) || (*numpts > MAX_PTS) ) {
      ERROR("RIPinput_polygon", "Illegal number of points");
      return RIP_BAD_PTS;
    };

   pts.count = 0;

   RIPprompt(prompt);

   overlay = make_overlay(window,&pts,&dd);
   old_window = RIPgrab_window(overlay);

   if (*numpts < 0)
      RIPtrack(rubberband_polygon, -1, overlay, TRUE);
   else
      RIPtrack(rubberband_polygon, (*numpts * 2) - 1, overlay, TRUE);

   if (*numpts < 0) {
      for (*numpts = 0;
	   (pts.x[*numpts] != -1) || (pts.y[*numpts] != -1);
	   ++(*numpts)) ;
      *numpts -= 3;
    };

   pts.x[*numpts] = pts.x[*numpts + 1];
   pts.y[*numpts] = pts.y[*numpts + 1];
   ASHpolyline(overlay,*numpts + 1, pts.x, pts.y);

   flag = TRUE;
   /* copy coords into passed arrays */
   for (pt = 0; (pt <= *numpts); ++pt) {
      /* pt not in passed window? */
      if (window != NULL) {
	 for (w = ASHfind_child_window(ASHinq_top_window(window),pts.x[pt],pts.y[pt]);
	      w != NULL && w != window;
	      w = ASHinq_parent(w));
	 if (w == NULL) flag = FALSE;
       };

      /* erase pts left over by ASHpolygon() */
      if (pt <= *numpts - 2)
	 ASHpoint(overlay,pts.x[pt], pts.y[pt]);

      if (window != NULL && window != ASHinq_top()) {
	 ASHmap(overlay, pts.x[pt], pts.y[pt], window, &(x_pt[pt]), &(y_pt[pt]));
       }
      else {
	 x_pt[pt] = pts.x[pt];
	 y_pt[pt] = pts.y[pt];
       };
    };

   RIPgrab_window(old_window);
   remove_overlay(overlay,&dd);
   RIPprompt(EMPTY);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_ival:							*/
/*									*/
/*	requests the user to enter an integer number. This is echoed in */
/*  the typein window. If only a carriage return was entered, zero is	*/
/*  returned. Otherwise, one is returned.				*/
/*									*/
/************************************************************************/


int
RIPinput_ival(prompt, ival)
   String prompt;
   Integer *ival;
{
   register Boolean flag;
   register String text;

   CHECKINIT;

   ETRACE("RIPinput_ival: prompt: '%s'", prompt);

   text = get_text(prompt);

   flag = !STREMPTY(text);

   *ival = atoi(text);
   FREE(text);

   RIPprompt(EMPTY);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_rval:							*/
/*									*/
/*	requests the user to enter a real number. This is echoed in the */
/*  typein window. If only a carriage return was entered, zero is	*/
/*  returned. Otherwise, one is returned.				*/
/*									*/
/************************************************************************/


int
RIPinput_rval(prompt, rval)
   String prompt;
   double *rval;
{
   register Boolean flag;
   register String text;

   CHECKINIT;

   ETRACE("RIPinput_rval: prompt: '%s'", prompt);

   text = get_text(prompt);

   flag = !STREMPTY(text);

   *rval = atof(text);
   FREE(text);

   RIPprompt(EMPTY);

   return flag;
};





/************************************************************************/
/*									*/
/*  RIPinput_text:							*/
/*									*/
/*	requests the user to enter a line of text. This is echoed in	*/
/*  the typein window. If only a carriage return was entered, zero is	*/
/*  returned. Otherwise, one is returned.				*/
/*									*/
/************************************************************************/


int
RIPinput_text(prompt, text)
   String prompt, *text;
{
   register Boolean flag;
   register String textlong;

   CHECKINIT;

   ETRACE("RIPinput_text: prompt: '%s'", prompt);

   textlong = get_text(prompt);

   flag = !STREMPTY(textlong);

   *text = SALLOC(textlong);
   FREE(textlong);

   RIPprompt(EMPTY);

   return flag;
};





/********************************************************************************/
/*										*/
/*	RIPset_tracking_cursor -- set cursor for tracking purposes		*/
/*										*/
/********************************************************************************/


ASH_CURSOR
RIPset_tracking_cursor(c)
   ASH_CURSOR c;
{
   track_cursor = c;
};





/************************************************************************/
/*									*/
/*  RIP_get_input:							*/
/*									*/
/*	mimics BIOnext(),	but returns a RIP_TRANS.		*/
/*									*/
/*	It is, however, far more informative than BIO.			*/
/*									*/
/*  returned value			meaning 			*/
/*  --------------			------- 			*/
/*  RIP_TRANS_NONE			no new input was encountered	*/
/*  RIP_TRANS_MOVE			the locator was moved		*/
/*  RIP_TRANS_CHAR			a key was pressed		*/
/*  RIP_TRANS_DOWN			a locator button was pressed	*/
/*  RIP_TRANS_UP			a locator button was released	*/
/*									*/
/*	In addition, it checks to see if any global functions are	*/
/*  active. If so, if a global event has occurred, it calls the routine */
/*  associated with that key.						*/
/*									*/
/*	WARNING: The x and y positions can change even though an event	*/
/*  such as a locator button being pressed or released or a key being	*/
/*  pressed has occurred. Therefore, when using this routine, it is	*/
/*  best to check separately for the x and y in addition to an event	*/
/*  occuring.								*/
/*									*/
/************************************************************************/


RIP_TRANS
RIP_get_input(waitfg,x_ptr, y_ptr, btns_ptr, ch_ptr, win_ptr)
   Boolean waitfg;
   Integer *x_ptr, *y_ptr, *btns_ptr;
   Integer *ch_ptr;
   ASH_WINDOW * win_ptr;
{
   register Integer event;
   register RIP_TRANS trans;

   ITRACE("RIP_get_input %d",waitfg);

   if (waitfg) {
      event = BIOnext_wait();
    }
   else {
      event = BIOnext();
    };

   BIOinq_position(x_ptr,y_ptr);
   if (win_ptr != NULL) *win_ptr = BIOinq_window();

   if (FKEY_TEST_ASCII(event) || FKEY_TEST_META(event) || FKEY_TEST(event)) {
      trans = RIP_TRANS_CHAR;
      *btns_ptr = 0;
      *ch_ptr = event;
    }
   else if (FKEY_TEST_MOUSE_DOWN(event)) {
      trans = RIP_TRANS_DOWN;
      *btns_ptr = 1 << (event - FKEY_MOUSE_FIRST);
      *ch_ptr = 0;
    }
   else if (FKEY_TEST_MOUSE_UP(event)) {
      trans = RIP_TRANS_UP;
      *btns_ptr = 1 << (event - FKEY_MOUSE_UP_FIRST);
      *ch_ptr = 0;
    }
   else if (FKEY_TEST_MOUSE_TAP(event)) {
      trans = RIP_TRANS_TAP;
      *btns_ptr = (1 << (event - FKEY_MOUSE_TAP_FIRST));
      *ch_ptr = 0;
    }
   else {
      *ch_ptr = 0;
      *btns_ptr = 0;
      switch (event) {
	 case FKEY_EVENT_MOVE :
	    trans = RIP_TRANS_MOVE;
	    break;
	 default :
	    trans = RIP_TRANS_NONE;
	    break;
       };
    };

   return trans;
};





/************************************************************************/
/*									*/
/*  RIP_display_text:							*/
/*									*/
/*	displays the passed string in the passed window. The window is	*/
/*  first cleared. If the entire string will fit across it is centered. */
/*  Otherwise, it is split up into smaller segments (separated by	*/
/*  blanks if possible) and printed in a column.			*/
/*									*/
/************************************************************************/


void
RIP_display_text(window, show)
   ASH_WINDOW window;
   String show;
{
   Integer xlen, ylen, x_delta, y_delta, lx, by, rx, ty,
       dx, dy;
   Integer font;

   ITRACE("RIP_display_text: text: '%s'", show);

   if (window == NULL) return;

   ASHinq_size(window, ASH_SIZE_WINDOW, &lx, &by, &rx, &ty);
   ASHclear_box(window,lx,by,rx,ty);

   if (STREMPTY(show)) return;


   dx = (lx < rx) ? 1 : -1;
   dy = (by < ty) ? 1 : -1;

   font = ASHinq_font(window);

   ASHinq_text_extent(font, show, &xlen, &ylen);

   ASHinq_text_offset(font, show, &x_delta, &y_delta);
   ASHtext(window, lx + x_delta * dx + 2, by + y_delta * dy + 1, show);
};





/************************************************************************/
/*									*/
/*  get_two_pts:							*/
/*									*/
/*	gets a two point figure by calling RIPtrack() with passed	*/
/*  rubberbanding routine and returning the two points along with a	*/
/*  flag indicating whether or not the points are in the passed window. */
/*	It takes care of all the busy work, too.			*/
/*									*/
/************************************************************************/


static int
get_two_pts(prompt, window, rtn, x0_ptr, y0_ptr, x1_ptr, y1_ptr)
   String prompt;
   ASH_WINDOW window;
   Function_Ptr rtn;
   Integer *x0_ptr, *y0_ptr, *x1_ptr, *y1_ptr;
{
   register Boolean flag;
   register ASH_WINDOW overlay, old_window,w;
   Integer i;
   POINTS_B pts;
   DRAWDATA_B dd;

   ITRACE("get_two_pts: prompt: '%s'", prompt);

   overlay = make_overlay(window,&pts,&dd);
   old_window = RIPgrab_window(overlay);
   RIPprompt(prompt);

   RIPtrack(rtn, 2, overlay, TRUE);

   /* need to map pts into coord syst of passed window? */
   if ((window == NULL) || (window == ASHinq_top_window(window))) {
      *x0_ptr = pts.x[0];
      *y0_ptr = pts.y[0];

      *x1_ptr = pts.x[1];
      *y1_ptr = pts.y[1];
    }
   else {
      ASHmap(overlay, pts.x[0], pts.y[0], window, x0_ptr, y0_ptr);
      ASHmap(overlay, pts.x[1], pts.y[1], window, x1_ptr, y1_ptr);
    }

   if (window == NULL) flag = TRUE;
   else {
      flag = TRUE;
      for (i = 0; i < 2; ++i) {
	 for (w = ASHfind_child_window(ASHinq_top_window(window),pts.x[i],pts.y[i]);
	      w != NULL && w != window;
	      w = ASHinq_parent(w));
	 if (w == NULL) flag = FALSE;
       };
    };

   RIPgrab_window(old_window);
   remove_overlay(overlay,&dd);

   RIPprompt(EMPTY);

   return flag;
};





/************************************************************************/
/*									*/
/*  rubberband_line:							*/
/*									*/
/*	called by a tracking routine to rubberband a line and store	*/
/*  the endpoints in a global array.					*/
/*									*/
/************************************************************************/


static int
rubberband_line(x, y, click, trans, max, w)
   Integer x, y, click, max;
   RIP_TRANS trans;
   ASH_WINDOW w;
{
   register POINTS pts;

   ITRACE("rubberband_line: (%d, %d), click = %d, trans: %d, maxclick = %d, 0x%x",
	     x, y, click, trans, max, w);

   pts = (POINTS) ASHinq_user_data(w);

   if (click) {
      if ((click == 1) && (trans == RIP_TRANS_DOWN)) {
	 pts->x[0] = x;
	 pts->y[0] = y;
       }

      /* erase old line if r-banding */
      if ((click == 2) || (trans == RIP_TRANS_MOVE))
	 ASHline(w, pts->x[0], pts->y[0], pts->x[1], pts->y[1]);

      /* save new pt */
      pts->x[1] = x;
      pts->y[1] = y;

      /* if not last pt, draw new line */
      if (click == 1)
	 ASHline(w, pts->x[0], pts->y[0], pts->x[1], pts->y[1]);
    };

   return RIP_OK;
};





/************************************************************************/
/*									*/
/*  rubberband_box:							*/
/*									*/
/*	called by a tracking routine to rubberband a box and store	*/
/*  the endpoints in a global array.					*/
/*									*/
/************************************************************************/


static int
rubberband_box(x, y, click, trans, max, w)
   Integer x, y, click, max;
   RIP_TRANS trans;
   ASH_WINDOW w;
{
   register POINTS pts;

   ITRACE("rubberband_box: (%d, %d), %d %d %d 0x%x",x,y,click,trans,max,w);

   pts = (POINTS) ASHinq_user_data(w);

   if (click) {
      if ((click == 1) && (trans == RIP_TRANS_DOWN)) {
	 pts->x[0] = x;
	 pts->y[0] = y;
       }

      /* erase old box if r-banding */
      if ((click == 2) || (trans == RIP_TRANS_MOVE))
	 ASHbox(w,pts->x[0], pts->y[0], pts->x[1], pts->y[1]);

      /* save new pt */
      pts->x[1] = x;
      pts->y[1] = y;

      /* if not last pt, draw new box */
      if (click == 1)
	 ASHbox(w,pts->x[0], pts->y[0], pts->x[1], pts->y[1]);
    }

   return RIP_OK;
};





/************************************************************************/
/*									*/
/*  rubberband_polygon: 						*/
/*									*/
/*	called by a tracking routine to rubberband a polygon and store	*/
/*  the vertices in a global array.					*/
/*									*/
/************************************************************************/


static int
rubberband_polygon(x, y, click, trans, max, w)
   Integer x, y, click, max;
   RIP_TRANS trans;
   ASH_WINDOW w;
{
   register POINTS pts;
   register Integer n;
   register Boolean good, doit;

   ITRACE("rubberband_polygon: (%d, %d), %d %d %d 0x%x",x,y,click,trans,max,w);

   good = TRUE;

   pts = (POINTS) ASHinq_user_data(w);

   /* tracking for 1st point */
   if (pts->count == 0) {
      if (click == 1) {
	 pts->x[0] = x;
	 pts->y[0] = y;
	 pts->x[1] = x;
	 pts->y[1] = y;
	 ASHpoint(w, x, y);
	 pts->count = 1;
       };
    }
   else if ((trans == RIP_TRANS_MOVE) || (trans == RIP_TRANS_UP)) {
      /* number of points being drawn */
      n = (pts->count - 1) / 2 + 2;

      ASHpolyline(w, ((n > 2) ? (n + 1) : n), pts->x, pts->y);
      pts->x[n - 1] = x;
      pts->y[n - 1] = y;
      ASHpolyline(w, ((n > 2) ? (n + 1) : n), pts->x, pts->y);
      if (trans == RIP_TRANS_UP) ++pts->count;
    }
   else {
      /* number of points being drawn */
      n = (pts->count - 1) / 2 + 2;

      ASHpolyline(w, ((n > 2) ? (n + 1) : n), pts->x, pts->y);
      pts->x[n - 1] = x;
      pts->y[n - 1] = y;

      doit = FALSE;
      if (max < 0) {
	 if (n + 1 > MAX_PTS) doit = TRUE;
	 else if (BTWN(pts->x[n-1],pts->x[n-2]-POLY_TLC,pts->x[n-2]+POLY_TLC)) {
	    if (BTWN(pts->y[n-1],pts->y[n-2]-POLY_TLC,pts->y[n-2]+POLY_TLC))
	       doit = TRUE;
	  };
       };
      if (doit) {
	 pts->x[n + 2] = -1;
	 pts->y[n + 2] = -1;
	 good = FALSE;
       };

      if (n > 2)
	 ASHpolyline(w, (n + 1), pts->x, pts->y);

      pts->x[n] = x;
      pts->y[n] = y;
      pts->x[n + 1] = pts->x[0];
      pts->y[n + 1] = pts->y[0];

      if (trans == RIP_TRANS_DOWN)
	 ++pts->count;
      else
	 pts->count += 2;
    };

   if (good) {
      if ((pts->count >= max) && (click < max) && (max != -1))
	 good = FALSE;
      else
	 good = TRUE;
    };

   return good;
};





/************************************************************************/
/*									*/
/*  get_text:								*/
/*									*/
/*	returns a string of characters that were obtained from the	*/
/*  typein window.							*/
/*									*/
/************************************************************************/


static String
get_text(prompt)
   String prompt;
{
   Integer x, y, btns, pos, promptlen;
   Integer ch;
   String text;
   ASH_WINDOW overlay;
   DRAWDATA_B dd;

   ITRACE("get_text: prompt: '%s'", prompt);

   PROTECT;

   if (RIP__areas[RIP_AREA_TYPEIN] == NULL || RIP__areas[RIP_AREA_PROMPT] == NULL)
      return NULL;

   overlay = make_overlay(NULL,NULL,&dd);

   RIPprompt(prompt);

   if (RIP__areas[RIP_AREA_TYPEIN] == RIP__areas[RIP_AREA_PROMPT]) {
      promptlen = strlen(prompt);
      text = (char *) malloc(promptlen + MAXINPUT + 2);
      strcpy(text, prompt);
      pos = promptlen;
    }
   else {
      promptlen = 0;
      text = (char *) malloc(MAXINPUT + 1);
      *text = EOS;
      pos = 0;
    }

   for ( ; ; ) {
      while (RIP_get_input(TRUE,&x,&y,&btns,&ch,NULL) != RIP_TRANS_CHAR);

      if ((ch & 0x80) != 0) continue;

      if (ch == '\b') {
	 if (pos > promptlen) {
	    --pos;
	    *(text + pos) = EOS;
	    RIP_display_text(RIP__areas[RIP_AREA_TYPEIN], text);
	  };
       }
      else if (ch == '\n' || ch == '\r') break;
      else if (pos < MAXINPUT) {
	 *(text + pos) = ch;
	 *(text + pos + 1) = EOS;
	 ++pos;
	 RIP_display_text(RIP__areas[RIP_AREA_TYPEIN], text);
       };
    };

   RIP_display_text(RIP__areas[RIP_AREA_PROMPT], EMPTY);
   RIP_display_text(RIP__areas[RIP_AREA_TYPEIN], EMPTY);

   if (RIP__areas[RIP_AREA_TYPEIN] == RIP__areas[RIP_AREA_PROMPT])
      strcpy(text, (text + promptlen));

   remove_overlay(overlay,&dd);

   UNPROTECT;

   return text;
};





/************************************************************************/
/*									*/
/*	make_overlay:							*/
/*	remove_overlay							*/
/*									*/
/*	makes a frame window over the top window for tracking purposes. */
/*									*/
/************************************************************************/


static ASH_WINDOW
make_overlay(w,pts,dd)
   ASH_WINDOW w;
   POINTS pts;
   DRAWDATA dd;
{
   register ASH_WINDOW overlay;

   ITRACE("make_overlay");

   if (USE_THREADS) overlay = BIOinq_top_window();
   else overlay = (w == NULL ? ASHinq_top() : ASHinq_top_window(w));

   dd->old_thru = ASHwindow_draw_thru(overlay,TRUE);
   dd->old_cr = ASHcombination_rule(overlay,10);
   dd->old_ls = ASHline_style(overlay,0);
   dd->old_data = ASHinq_user_data(overlay);
   dd->old_fill = ASHfill(overlay,1);

   if (USE_THREADS)
      dd->old_thread = (THREAD) BIOset_window_thread(overlay,THREADinq_current());

   ASHset_user_data(overlay,pts);
   dd->old_cursor = BIOset_cursor_pattern(overlay,track_cursor);

   return overlay;
};





static void
remove_overlay(w,dd)
   ASH_WINDOW w;
   DRAWDATA dd;
{
   ASHfill(w,dd->old_fill);
   ASHline_style(w,dd->old_ls);
   ASHcombination_rule(w,dd->old_cr);
   ASHset_user_data(w,dd->old_data);
   BIOset_cursor_pattern(w,dd->old_cursor);
   ASHwindow_draw_thru(w,dd->old_thru);
   if (USE_THREADS) BIOset_window_thread(w,dd->old_thread);
};





/* end of ripinout.c */
