/************************************************************************/
/*									*/
/*		SEQUENCE.C						*/
/*									*/
/*	This module contains the various subroutines required for	*/
/*	list processing.  The actual calls to these routines are	*/
/*	through the header file sequence.h				*/
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/

/* LINTLIBRARY */

				/* global definitions			*/
extern	char *	malloc();

#include "sequence.h"
#include <varargs.h>
#include "DATATYPES.h"

#define PROT_NAME	SEQ__sema
#include <bwethread.h>




/************************************************************************/
/*									*/
/*	local storage							*/
/*									*/
/************************************************************************/


static	Sequence    freelist = NULL;
static	Boolean     initflag = FALSE;

static	PROT_DECL;




/************************************************************************/
/*									*/
/*	Internal Functions  and  Constants				*/
/*									*/
/************************************************************************/


#define LALLOC	newlistcell()
#define FREEL(l) freelistcell(l)

#define LBLKSIZE (512-2)
#define rplaca(list,v)	((list)->Lcar) = ((Universal)(v))
#define rplacd(list,v)	((list)->Lcdr) = ((Sequence)(v))






/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Sequence	listalloc();
static	Sequence	newlistcell();
static	Sequence	freelistcell();

#ifdef LIB
static			listcleanup();
extern			BROWNregister_cleanup();
#endif




/************************************************************************/
/*									*/
/*	SEQnth -- return nth element of a list	(origin 1)		*/
/*									*/
/************************************************************************/


Universal
SEQnth(list,n)
   Sequence list;
   Integer n;
{
   while (list != NULL && --n > 0) list = CDR(list);

   if (list == NULL) return NULL;

   return list->Lcar;
};






/************************************************************************/
/*									*/
/*	SEQcons -- add element to front of list 			*/
/*									*/
/************************************************************************/


Sequence
SEQcons(item,list)
   Universal item;
   Sequence list;
{
   Sequence l;

   l = LALLOC;
   l->Lcar = item;
   l->Lcdr = list;

   return l;
};





/************************************************************************/
/*									*/
/*	SEQnew -- construct a new list with the given elements		*/
/*									*/
/************************************************************************/

/* VARARGS */

Sequence
SEQnew(args)
   Universal args;
{
   Integer narg,i;
   Universal *arg;
   Sequence l;

   arg = &args;
   narg = (Integer) arg[-1];

   l = NULL;
   for (i = narg-1; i >= 0; --i)
      l = CONS(arg[i],l);

   return l;
};





/************************************************************************/
/*									*/
/*	SEQnewc -- construct a new list with the given elements 	*/
/*									*/
/************************************************************************/

/* VARARGS */

Sequence
SEQnewc(va_alist)
   va_dcl
{
   Universal val[128];
   Integer i,narg;
   va_list pv;
   Sequence l;

   va_start(pv);
   narg = va_arg(pv,Integer);

   for (i = 0; i < narg; ++i) val[i] = va_arg(pv,Universal);
   va_end(pv);

   l = NULL;
   for (i = narg-1; i >= 0; --i)
      l = CONS(val[i],l);

   return l;
};





/************************************************************************/
/*									*/
/*	SEQappend -- add element to end of list 			*/
/*									*/
/************************************************************************/


Sequence
SEQappend(item,list)
   Universal item;
   Sequence list;
{
   Sequence l;

   if (EMPTY(list)) return CONS(item,list);

   for (l = list; !EMPTY(CDR(l)); l = CDR(l));

   RPLACD(l,CONS(item,NULL));

   return list;
};





/************************************************************************/
/*									*/
/*	SEQconc -- append two lists (destructive)			*/
/*									*/
/************************************************************************/


Sequence
SEQconc(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence l;

   if (la == NULL) return lb;
      else if (lb == NULL) return la;

   for (l = la; !EMPTY(CDR(l)); l = CDR(l));
   RPLACD(l,lb);

   return la;
};





/************************************************************************/
/*									*/
/*	SEQfree -- free a list						*/
/*									*/
/************************************************************************/


SEQfree(list)
   Sequence list;
{
   Sequence l;

   while (list != NULL)
    { l = list;
      list = CDR(list);
      FREEL(l);
    };
};





/************************************************************************/
/*									*/
/*	SEQcdrf -- return cdr of a list after freeing the car		*/
/*									*/
/************************************************************************/


Sequence
SEQcdrf(list)
   Sequence list;
{
   Sequence l;

   if (list == NULL) return NULL;

   l = CDR(list);

   return l;
};





/************************************************************************/
/*									*/
/*	SEQmemq -- check for item being EQ an element of a list 	*/
/*									*/
/************************************************************************/


Boolean
SEQmemq(item,list)
   Universal item;
   Sequence list;
{
   Sequence l;

   for (l = list; !EMPTY(l); l = CDR(l))
      if (CAR(Universal,l) == item) return TRUE;

   return FALSE;
};





/************************************************************************/
/*									*/
/*	SEQsame -- check for two lists being the same as sets		*/
/*									*/
/************************************************************************/


Boolean
SEQsame(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence l;

   for (l = la; !EMPTY(l); l = CDR(l))
      if (!MEMQ(CAR(Universal,l),lb)) return FALSE;
   for (l = lb; !EMPTY(l); l = CDR(l))
      if (!MEMQ(CAR(Universal,l),la)) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	SEQequal -- check for equality of two lists			*/
/*									*/
/************************************************************************/


Boolean
SEQequal(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence lx,ly;

   for (lx = la, ly = lb; !EMPTY(lx) && !EMPTY(ly); lx = CDR(lx), ly = CDR(ly))
      if (CAR(Universal,lx) != CAR(Universal,ly)) break;

   if (!EMPTY(lx) || !EMPTY(ly)) return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	SEQcontain -- test containment					*/
/*									*/
/************************************************************************/


Boolean
SEQcontain(x,y)
   Sequence x,y;
{
   Sequence l;
   Universal a;

   forin (a,Universal,l,x)
      if (!MEMQ(a,y)) return FALSE;

   return TRUE;
};






/************************************************************************/
/*									*/
/*	SEQnullint -- check for null intersection of two lists		*/
/*									*/
/************************************************************************/


Boolean
SEQnullint(x,y)
   Sequence x,y;
{
   Sequence l;
   Universal a;

   forin (a,Universal,l,x)
      if (MEMQ(a,y)) return FALSE;

   return TRUE;
};






/************************************************************************/
/*									*/
/*	SEQminus -- compute set difference				*/
/*									*/
/************************************************************************/


Sequence
SEQminus(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence l,lc;

   if (la == NULL) return NULL;
      else if (lb == NULL) return LCOPY(la);

   l = lc = NULL;
   for ( ; la != NULL; la = CDR(la)) {
      if (!MEMQ(CAR(Universal,la),lb)) {
	 if (l == NULL) l = lc = CONS(CAR(Universal,la),NULL);
	 else {
	    l->Lcdr = CONS(CAR(Universal,la),NULL);
	    l = l->Lcdr;
	  };
       };
    };

   return lc;
};





/************************************************************************/
/*									*/
/*	SEQinter -- compute intersection of two lists			*/
/*									*/
/************************************************************************/


Sequence
SEQinter(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence l,lc;

   if (la == NULL) return NULL;
      else if (lb == NULL) return NULL;

   l = lc = NULL;
   for ( ; la != NULL; la = CDR(la)) {
      if (MEMQ(CAR(Universal,la),lb)) {
	 if (l == NULL) l = lc = CONS(CAR(Universal,la),NULL);
	 else {
	    l->Lcdr = CONS(CAR(Universal,la),NULL);
	    l = l->Lcdr;
	  };
       };
    };

   return lc;
};





/************************************************************************/
/*									*/
/*	SEQunion -- compute the union of two lists			*/
/*									*/
/************************************************************************/


Sequence
SEQunion(la,lb)
   Sequence la;
   Sequence lb;
{
   Sequence l,lc;

   lc = NULL;
   for (l = la; !EMPTY(l); l = CDR(l))
      if (!MEMQ(CAR(Universal,l),lc)) lc = CONS(CAR(Universal,l),lc);
   for (l = lb; !EMPTY(l); l = CDR(l))
      if (!MEMQ(CAR(Universal,l),lc)) lc = CONS(CAR(Universal,l),lc);

   return REVERSE(lc);
};





/************************************************************************/
/*									*/
/*	SEQremob -- remove object from a list				*/
/*									*/
/************************************************************************/


Sequence
SEQremob(item,list)
   Universal item;
   Sequence list;
{
   Sequence la;

   if (EMPTY(list)) return list;

   if (list->Lcar == item) return CDRF(list);

   for (la = list; !EMPTY(CDR(la)); la = CDR(la))
      if (la->Lcdr->Lcar == item)
       { RPLACD(la,CDRF(CDR(la)));
	 break;
       };

   return list;
};





/************************************************************************/
/*									*/
/*	SEQrplaca -- replace car of a list element			*/
/*									*/
/************************************************************************/


Sequence
SEQrplaca(l,v)
   Sequence l;
   Universal v;
{
   l->Lcar = v;

   return l;
};





/************************************************************************/
/*									*/
/*	SEQrplacd -- replace cdr of a list element			*/
/*									*/
/************************************************************************/


Sequence
SEQrplacd(l,v)
   Sequence l;
   Sequence v;
{
   l->Lcdr = v;

   return l;
};






/************************************************************************/
/*									*/
/*	SEQreverse -- compute the reverse of a list			*/
/*									*/
/************************************************************************/


Sequence
SEQreverse(list)
   Sequence list;
{
   Sequence l,la,lb;

   la = NULL;
   for ( ; list != NULL; list = CDR(list))
      la = CONS(CAR(Universal,list),la);

   return la;
};





/************************************************************************/
/*									*/
/*	SEQcopy -- copy a list						*/
/*									*/
/************************************************************************/


Sequence
SEQcopy(list)
   Sequence list;
{
   return list == NULL ? NULL :
			 CONS(CAR(Universal,list),SEQcopy(CDR(list)));
};






/************************************************************************/
/*									*/
/*	SEQlength -- return the length of a list			*/
/*									*/
/************************************************************************/


Integer
SEQlength(list)
   Sequence list;
{
   Integer ln;

   ln = 0;
   for ( ; !EMPTY(list); list = CDR(list)) ++ln;

   return ln;
};






/************************************************************************/
/*									*/
/*	SEQprint -- debugging routine to print a list			*/
/*									*/
/************************************************************************/


SEQprint(l)
   Sequence l;
{
   Integer i,x;
   Sequence la;

   printf("\t( ");
   i = 1;
   forin (x,Integer,la,l) {
      printf("0x%x ",x);
      if (i++ % 8 == 0 && !EMPTY(CDR(la)))
	 printf("\n\t  ");
    };
   printf(")\n\n");
};






/************************************************************************/
/*									*/
/*	listalloc -- allocate list block				*/
/*									*/
/************************************************************************/


static Sequence
listalloc()
{
   SequenceB *lb;
   Integer i;

   lb = (SequenceB *) malloc(LBLKSIZE*sizeof(SequenceB));

   for (i = 1; i < LBLKSIZE; ++i)
      rplacd(&lb[i-1],&lb[i]);

   rplacd(&lb[LBLKSIZE-1],freelist);
   freelist = &lb[1];
   rplaca(&lb[0],0);
   rplacd(&lb[0],0);

   return &lb[0];
};





/************************************************************************/
/*									*/
/*	newlistcell -- allocate list cell				*/
/*	freelistcell-- remove list cell 				*/
/*									*/
/************************************************************************/


static Sequence
newlistcell()
{
   register Sequence l;

   if (!initflag) {
      PROT_INIT;
    };

   PROTECT;

   if (!initflag) {
#ifdef LIB
      BROWNregister_cleanup(listcleanup);
#endif
      initflag = TRUE;
      freelist = NULL;
    };

   if (freelist == NULL) l = listalloc();
   else {
      l = freelist;
      freelist = CDR(freelist);
    };

   UNPROTECT;

   return l;
};





static Sequence
freelistcell(l)
   Sequence l;
{
   PROTECT;

   rplacd(l,freelist);
   freelist = l;

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	listcleanup -- cleanup list module on exit			*/
/*									*/
/************************************************************************/



#ifdef LIB

static
listcleanup()
{
   initflag = FALSE;
   freelist = NULL;
};

#endif





/* end of lists.c */
