/************************************************************************/
/*									*/
/*		hashsearch.c						*/
/*									*/
/*	General purpose extensible hashing routines			*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "hashsearch.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_PERCENT_FULL	80

#define HASH_FUNCTION(x,ht)	((((unsigned)(x))*2654435769)>>((ht)->shift))
#define HASH_FUNCTION_2(x,ht)	(1|((((unsigned)(x))*1818367229)>>((ht)->shift)))

#define NULL			0




/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct _HASH_TABLE {
   int num_entry;
   int max_entry;
   int shift;
   int mask;
   ENTRY * entry;
} HASH_TABLE_B;





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	int		string_hash();





/************************************************************************/
/*									*/
/*	HASHcreate -- create a new hash table				*/
/*									*/
/************************************************************************/


HASH_TABLE
HASHcreate(nel)
   int nel;
{
   int sz,shft;
   HASH_TABLE ht;

   sz = 16;
   shft = 32-4;

   nel = (nel * 100)/ MAX_PERCENT_FULL;

   while (sz < nel) {
      sz *= 2;
      shft -= 1;
    };

   ht = (HASH_TABLE) calloc(1,sizeof(HASH_TABLE_B));
   ht->num_entry = 0;
   ht->max_entry = sz;
   ht->shift = shft;
   ht->mask = ht->max_entry - 1;
   ht->entry = (ENTRY *) calloc(sizeof(ENTRY),sz);

   if (ht->entry == NULL) return NULL;

   return ht;
};





/************************************************************************/
/*									*/
/*	HASHdestroy -- remove a hash table				*/
/*									*/
/************************************************************************/


void
HASHdestroy(ht)
   HASH_TABLE ht;
{
   if (ht == NULL) return;

   if (ht->entry != NULL) {
      free(ht->entry);
      ht->entry = NULL;
    };

   ht->max_entry = 0;
   ht->num_entry = 0;

   free(ht);
};





/************************************************************************/
/*									*/
/*	HASHsearch -- add or find an item in hash table 		*/
/*									*/
/************************************************************************/


ENTRY *
HASHsearch(ht,item,action)
   HASH_TABLE ht;
   ENTRY item;
   ACTION action;
{
   register int h,h1,h2;
   ENTRY * oe;
   int i,ct;

   h = string_hash(item.key);
   h1 = HASH_FUNCTION(h,ht);
   h2 = -1;

   while (ht->entry[h1].key != NULL) {
      if (strcmp(ht->entry[h1].key,item.key) == 0) break;
      if (h2 < 0) h2 = HASH_FUNCTION_2(h,ht);
      h1 = (h1+h2) & (ht->mask);
    };

   if (ht->entry[h1].key == NULL) {
      if (action == FIND) return NULL;
      ht->entry[h1] = item;
      ht->num_entry++;
      if (ht->num_entry * 100 / MAX_PERCENT_FULL > ht->max_entry) {
	 oe = ht->entry;
	 ct = ht->max_entry;
	 ht->max_entry *= 2;
	 ht->shift -= 1;
	 ht->mask = ht->max_entry - 1;
	 ht->entry = (ENTRY *) calloc(sizeof(ENTRY),ht->max_entry);
	 ht->num_entry = 0;
	 for (i = 0; i < ct; ++i) {
	    if (oe[i].key != NULL) HASHsearch(ht,oe[i],ENTER);
	  };
	 free(oe);
       };
    };

   return &ht->entry[h1];
};






/************************************************************************/
/*									*/
/*	hash_string -- standard hash function for strings		*/
/*									*/
/************************************************************************/


static int
string_hash(s)
   char * s;
{
   register int wd,sh;
   register char * txt;

   txt = s;

   wd = 0;
   sh = 24;
   while (*txt != 0)
    { wd ^= (*txt++)<<sh;
      if ((sh -= 8) < 0) sh = 24;
    };

   wd &= 0x3FFFFFFF;

   return wd;
};






/* end of hashsearch.c */
