/************************************************************************/
/*									*/
/*		bwethread.c					       */
/*								       */
/*	Entries to handle thread management within BWE		       */
/*									*/
/************************************************************************/
/*	Copyright 1988 Brown University -- Steven P. Reiss		*/


#include <signal.h>

#include "bwethread.h"
#include "DATATYPES.h"



/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/






/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	SEMAPHORE	bwe_sema = NULL;
static	BWE_SEMA *	first_sema = NULL;
static	Integer 	thread_count = 0;
static Integer		waitct = 0;




/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		bwe_thread_destroy();
static	void		bwe_thread_create();

extern	String		getenv();





/************************************************************************/
/*									*/
/*	BWE_thread_init -- begin threads while using BWE		*/
/*									*/
/************************************************************************/


void
BWE_thread_init()
{
   first_sema = NULL;
   thread_count = 0;
   THREADtype1_register(0,bwe_thread_create,bwe_thread_destroy,NULL,NULL);
   bwe_sema = THREADseminit(1);
};





/************************************************************************/
/*									*/
/*	BWEthread_semaphore -- safely create a module semaphore 	*/
/*									*/
/************************************************************************/


BWE_SEMA *
BWEthread_semaphore(sem_loc)
   BWE_SEMA * sem_loc;
{
   if (sem_loc == NULL) return NULL;

   if (!USE_THREADS) {
      sem_loc->real_sema = NULL;
      sem_loc->owner = NULL;
      sem_loc->next = NULL;
      return sem_loc;
    };

   if (sem_loc->real_sema != NULL) return sem_loc;

   THREADpsem(bwe_sema);

   if (sem_loc->real_sema == NULL) {
      sem_loc->real_sema = THREADseminit(1);
      sem_loc->owner = NULL;
      sem_loc->count = 0;
      sem_loc->next = first_sema;
      first_sema = sem_loc;
    };

   THREADvsem(bwe_sema);

   return sem_loc;
};





/************************************************************************/
/*									*/
/*	BWEthread_protect -- P the BWE semaphore passed 		*/
/*	BWEthread_unprotect -- V the BWE semaphore			*/
/*									*/
/************************************************************************/


void
BWEthread_protect(bs)
   BWE_SEMA * bs;
{
   register THREAD t,t1;

   if (bs == NULL || bs->real_sema == NULL) return;

   t1 = THREADinq_current();
   do {
      t = t1;
      t1 = THREADfrozenparent(t);
    }
   while (t1 != NULL);

   if (bs->owner == t) {
      ++bs->count;
      return;
    };

   if (bs->count > 0) {
      ++waitct;
    };

   THREADpsem(bs->real_sema);
   bs->owner = t;
   bs->count = 1;
};




void
BWEthread_unprotect(bs)
   BWE_SEMA * bs;
{
   register THREAD t,t1;

   if (bs == NULL || bs->real_sema == NULL) return;

   t1 = THREADinq_current();
   do {
      t = t1;
      t1 = THREADfrozenparent(t);
    }
   while (t1 != NULL);

   if (bs->owner != t) {
      fprintf(stderr,"UNPROTECT attempted when not holding semaphore\n");
      abort();						
    };

   if (--bs->count > 0) return;

   bs->owner = NULL;
   THREADvsem(bs->real_sema);
};




void
BWEthread_unprotect_ok(bs)
   BWE_SEMA * bs;
{
   register THREAD t,t1;

   if (bs == NULL || bs->real_sema == NULL) return;

   t1 = THREADinq_current();
   do {
      t = t1;
      t1 = THREADfrozenparent(t);
    }
   while (t1 != NULL);

   if (bs->owner != t) {
      return;				/* called after reset		  */
    };

   if (--bs->count > 0) return;

   bs->owner = NULL;
   THREADvsem(bs->real_sema);
};




/************************************************************************/
/*									*/
/*	BWEthread_reset -- handle reset on thread by releasing sema	*/
/*									*/
/************************************************************************/


void
BWEthread_reset(t)
   THREAD t;
{
   register BWE_SEMA * bs;

   for (bs = first_sema; bs != NULL; bs = bs->next) {
      if (bs->owner == t && bs->real_sema != NULL) {
	 THREADvsem(bs->real_sema);
	 bs->owner = NULL;
	 bs->count = 0;
       };
    };
};





/************************************************************************/
/*									*/
/*	BWEthread_sema_remove -- remove BWE semaphore			*/
/*									*/
/************************************************************************/


void
BWEthread_sema_remove(bs)
   BWE_SEMA * bs;
{
   Integer i;

   if (bs == NULL || bs->real_sema == NULL) return;

   for (i = 0; i < thread_count; ++i) {
      THREADvsem(bs->real_sema);
    };

   bs->real_sema = NULL;
   bs->owner = NULL;
   bs->count = -1;
};




/************************************************************************/
/*									*/
/*	bwe_thread_destroy -- handle thread departing			*/
/*	bwe_thread_create -- handle thread creation			*/
/*									*/
/************************************************************************/


static void
bwe_thread_destroy(t)
   THREAD t;
{
   register BWE_SEMA * bs;

   for (bs = first_sema; bs != NULL; bs = bs->next) {
      if (bs->owner == t) {
	 THREADvsem(bs->real_sema);
	 bs->owner = NULL;
	 bs->count = 0;
       };
    };

   --thread_count;
};





static void
bwe_thread_create(t)
   THREAD t;
{
   ++thread_count;
};





/* end of bwethread.c */
