/************************************************************************/
/*									*/
/*		leaflayout.c						*/
/*									*/
/*	Layout routines for LEAF window layout package			*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "leaf_local.h"
#include <varargs.h>






/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/


static	Integer 	leaf_font;


#define CHECK_ID(ld,x)	   if ((x) >= (ld)->numwin || (x) < 0) FATAL("Bad LEAF window ID")

#define SCROLL_WIDTH		24




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		setup_leaf_def();
static	Integer 	check_user_def();
static	void		setup_components();
static	void		compute_layout();
static	void		layout_component();
static	Integer 	compute_coord();
static	Integer 	text_size();





/************************************************************************/
/*									*/
/*	LEAF_layout_init -- module initialization			*/
/*									*/
/************************************************************************/


void
LEAF_layout_init()
{
   ITRACE("LEAF_layout_init");

   leaf_font = ASHloadfont(ASHinq_base_font());
};





/************************************************************************/
/*									*/
/*	LEAFsetup_window -- setup the current window from table 	*/
/*	LEAFlayout_window -- setup and layout current window		*/
/*									*/
/************************************************************************/


void
LEAFsetup_window(w,ld,data)
   ASH_WINDOW w;
   LEAF_DATA * ld;
   Universal data;
{
   register LEAF_DEF ldef;
   register Integer i;

   ENTER("LEAFsetup_window 0x%x 0x%x 0x%x",w,ld,data);

   ldef = (LEAF_DEF) calloc(1,sizeof(LEAF_DEF_B));

   ldef->data = data;
   ldef->leafdata = ld;
   ldef->ctrl_rtn = NULL;
   ldef->refresh_rtn = NULL;
   ldef->info = NULL;
   ldef->infosize = 0;
   ldef->font = -1;
   ldef->resize_only = FALSE;
   ldef->active = FALSE;

   for (i = 0; i < MAX_PARAM; ++i) {
      ldef->param[i] = 0;
    };

   setup_leaf_def(ldef,ld,w);
};





void
LEAFlayout_window(w,ld,data)
   ASH_WINDOW w;
   LEAF_DATA * ld;
   Universal data;
{
   register LEAF_DEF ldef;

   ENTER("LEAFlayout_window 0x%x 0x%x 0x%x",w,ld,data);

   LEAFsetup_window(w,ld,data);

   ldef = LEAF_find_window_defn(w);

   compute_layout(ldef);
};





/************************************************************************/
/*									*/
/*	LEAFset_parameters -- reset parameters				*/
/*									*/
/************************************************************************/


/*VARARGS*/

void
LEAFset_parameters(va_alist)
   va_dcl
{
   va_list ap;
   register LEAF_DEF ldef;
   register Integer np,i;
   register ASH_WINDOW w;

   va_start(ap);
   w = va_arg(ap,ASH_WINDOW);
   np = va_arg(ap,Integer);

   TRACE("LEAFset_parameters 0x%x %d",w,np);

   ldef = LEAF_find_window_defn(w);

   for (i = 0; i < np; ++i) {
      ldef->param[i] = va_arg(ap,Integer);
      ITRACE("\t%d",ldef->param[i]);
    };

   setup_leaf_def(ldef,ldef->leafdata,w);

   compute_layout(ldef);
};





/************************************************************************/
/*									*/
/*	LEAFset_control -- set control routine for leaf window		*/
/*									*/
/************************************************************************/


void
LEAFset_control(w,rtn)
   ASH_WINDOW w;
   Function_Ptr rtn;
{
   register LEAF_DEF ldef;

   TRACE("LEAFset_control 0x%x 0x%x",w,rtn);

   ldef = LEAF_find_window_defn(w);

   ldef->ctrl_rtn = rtn;
};





/************************************************************************/
/*									*/
/*	LEAFset_refresh -- set refresh routine for leaf window		*/
/*									*/
/************************************************************************/


void
LEAFset_refresh(w,rtn)
   ASH_WINDOW w;
   Function_Ptr rtn;
{
   register LEAF_DEF ldef;

   TRACE("LEAFset_refresh 0x%x 0x%x",w,rtn);

   ldef = LEAF_find_window_defn(w);

   ldef->refresh_rtn = rtn;
};





/************************************************************************/
/*									*/
/*	LEAFset_active -- set active flag  for leaf window		*/
/*									*/
/************************************************************************/


void
LEAFset_active(w,fg)
   ASH_WINDOW w;
   Boolean fg;
{
   register LEAF_DEF ldef;
   Integer i;

   TRACE("LEAFset_active 0x%x %d",w,fg);

   ldef = LEAF_find_window_defn(w);
   if (ldef == NULL) return;

   if (ldef->active == fg) return;

   ldef->active = fg;

   for (i = 0; i < ldef->numwin; ++i) {
      if (ldef->info[i].defn != NULL &&
	     (ldef->info[i].defn->type & LEAF_TYPE_USE_TITLE) != 0 &&
	     ldef->info[i].status == LEAF_STATUS_DRAWN) {
	 ldef->info[i].status = LEAF_STATUS_LAYOUT;
       };
    };
};





/************************************************************************/
/*									*/
/*	LEAFinq_window -- return window of specified component		*/
/*									*/
/************************************************************************/


/* VARARGS2 */

ASH_WINDOW
LEAFinq_window(w,c,idx)
   ASH_WINDOW w;
   Integer c;
   Integer idx;
{
   register LEAF_DEF ldef;

   TRACE("LEAFinq_window 0x%x %d %d",w,c,idx);

   ldef = LEAF_find_window_defn(w);
   if (ldef == NULL) return NULL;

   CHECK_ID(ldef,c);

   if (ldef->info[c].defn->type & LEAF_TYPE_ARRAY) c += idx;

   return ldef->info[c].window;
};



/************************************************************************/
/*									*/
/*	LEAFset_font -- set font for component				*/
/*	LEAFset_data -- set data for component				*/
/*	LEAFinq_data -- return data for component			*/
/*	LEAFinq_rip_region -- return rip region defined for component	*/
/*									*/
/************************************************************************/


/* VARARGS3 */

void
LEAFset_font(font,w,c,idx)
   Integer font;
   ASH_WINDOW w;
   Integer c;
   Integer idx;
{
   register LEAF_DEF ldef;

   TRACE("LEAFset_font 0x%x 0x%x %d %d",font,w,c,idx);

   ldef = LEAF_find_window_defn(w);
   CHECK_ID(ldef,c);

   if (ldef->info[c].defn->type & LEAF_TYPE_ARRAY) c += idx;

   ldef->info[c].font = font;

   if (c == 0) {
      ldef->font = font;
      ASHfont(w,font);
    };

   if (ldef->info[c].status == LEAF_STATUS_DRAWN) {
      ldef->info[c].status = LEAF_STATUS_LAYOUT;
    };
};





/* VARARGS3 */

void
LEAFset_data(data,w,c,idx)
   Universal data;
   ASH_WINDOW w;
   Integer c;
   Integer idx;
{
   register LEAF_DEF ldef;

   TRACE("LEAFset_data 0x%x 0x%x %d %d",data,w,c,idx);

   ldef = LEAF_find_window_defn(w);
   CHECK_ID(ldef,c);

   if (ldef->info[c].defn->type & LEAF_TYPE_ARRAY) c += idx;

   ldef->info[c].data = data;

   if (ldef->info[c].status == LEAF_STATUS_DRAWN) {
      ldef->info[c].status = LEAF_STATUS_LAYOUT;
    };
};





/* VARARGS2 */

int
LEAFinq_data(w,c,idx)
   ASH_WINDOW w;
   Integer c;
   Integer idx;
{
   register LEAF_DEF ldef;

   TRACE("LEAFinq_data 0x%x %d %d",w,c,idx);

   ldef = LEAF_find_window_defn(w);
   CHECK_ID(ldef,c);

   if (ldef->info[c].defn->type & LEAF_TYPE_ARRAY) c += idx;

   return (int) ldef->info[c].data;
};





/* VARARGS2 */

RIP_REGION
LEAFinq_rip_region(w,c,idx)
   ASH_WINDOW w;
   Integer c;
   Integer idx;
{
   register LEAF_DEF ldef;

   TRACE("LEAFinq_rip_region 0x%x %d %d",w,c,idx);

   ldef = LEAF_find_window_defn(w);
   CHECK_ID(ldef,c);

   if (ldef == NULL) return NULL;

   if (ldef->info[c].defn->type & LEAF_TYPE_ARRAY) c += idx;

   return ldef->info[c].region;
};





/************************************************************************/
/*									*/
/*	LEAF_recompute_layout -- redo the layout			*/
/*									*/
/************************************************************************/


void
LEAF_recompute_layout(ldef)
   LEAF_DEF ldef;
{
   Integer lx,by,rx,ty;

   ITRACE("LEAF_recompute_layout 0x%x",ldef);

   ASHinq_size(ldef->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ldef->coordlx = lx;
   ldef->coordby = by;
   ldef->coordrx = rx;
   ldef->coordty = ty;
   ldef->dx = (lx < rx ? 1 : -1);
   ldef->dy = (by < ty ? 1 : -1);

   compute_layout(ldef);
};





/************************************************************************/
/*									*/
/*	LEAF_find_window_defn -- get leaf def for given window		*/
/*									*/
/************************************************************************/


LEAF_DEF
LEAF_find_window_defn(w)
   ASH_WINDOW w;
{
   register LEAF_DEF ldef;

   ITRACE("LEAF_find_window_defn 0x%x",w);

   ldef = (LEAF_DEF) ASHinq_menu_data(w);

   return ldef;
};




/************************************************************************/
/*									*/
/*	LEAF_get_value -- get value from user field			*/
/*									*/
/************************************************************************/


Integer
LEAF_get_value(v,ldef)
   Integer v;
   LEAF_DEF ldef;
{
   ITRACE("LEAF_get_value 0x%x 0x%x",v,ldef);

   if (v > 0 && (v & LEAF_VFLAGS) != 0) {
      if (v & LEAF_VFLAG_PARAM) v = ldef->param[v&0xffff];
    };

   return v;
};





/************************************************************************/
/*									*/
/*	LEAF_get_font -- get proper font for subwindow			*/
/*									*/
/************************************************************************/


Integer
LEAF_get_font(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   register Integer f;

   ITRACE("LEAF_get_font 0x%x %d",ldef,i);

   if (ldef->info[i].font > 0) f = ldef->info[i].font;
   else if (ldef->font > 0) f = ldef->font;
   else f = leaf_font;

   return f;
};





/************************************************************************/
/*									*/
/*	setup_leaf_def -- setup a definition from user structure	*/
/*									*/
/************************************************************************/


static void
setup_leaf_def(ldef,ld,w)
   LEAF_DEF ldef;
   LEAF_DATA ld[];
   ASH_WINDOW w;
{
   Integer lx,by,rx,ty;

   DTRACE("setup_leaf_def 0x%x 0x%x 0x%x",ldef,ld,w);

   ldef->window = w;
   ldef->numwin = check_user_def(ld,ldef);

   ASHset_menu_data(w,ldef);
   ASHinq_size(w,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
/* if (ldef->font == -1) ldef->font = ASHinq_font(w);	*/
   ldef->coordlx = lx;
   ldef->coordby = by;
   ldef->coordrx = rx;
   ldef->coordty = ty;
   ldef->dx = (lx < rx ? 1 : -1);
   ldef->dy = (by < ty ? 1 : -1);

   setup_components(ldef,ld);
};




/************************************************************************/
/*									*/
/*	check_user_def -- check user defn and return max window number	*/
/*									*/
/************************************************************************/


static Integer
check_user_def(ld,ldef)
   LEAF_DATA ld[];
   LEAF_DEF ldef;
{
   register Integer mx;
   register Boolean haveroot;
   register Integer i;

   DTRACE("check_user_def 0x%x",ld);

   haveroot = FALSE;
   mx = -1;

   for (i = 0; ld[i].id != LEAF_BOX_ID_LAST && i < MAX_BOX; ++i) {
      if (ld[i].id == LEAF_BOX_ID_ROOT) haveroot = TRUE;
      else if (ld[i].type & LEAF_TYPE_ARRAY) {
	 mx = MAX(mx,ld[i].id + LEAF_get_value(ld[i].data,ldef));
       }
      else mx = MAX(mx,ld[i].id);
    };

   if (mx >= 2*i) {
      FATAL("LEAF id numbers are too sparse");
    }
   else if (!haveroot) {
      FATAL("LEAF specification must include the root window descriptor");
    }
   else if (i >= MAX_BOX) {
      FATAL("Too many boxes (or missing end box) for LEAF");
    };

   return mx+1;
};





/************************************************************************/
/*									*/
/*	setup_components -- setup the components of a box		*/
/*									*/
/************************************************************************/


static void
setup_components(ldef,ldata)
   LEAF_DEF ldef;
   LEAF_DATA ldata[];
{
   register Integer i,j,k;

   DTRACE("setup_components 0x%x 0x%x",ldef,ldata);

   if (ldef->info == NULL) {
      ldef->info= (LEAF_WIN) calloc(ldef->numwin,sizeof(LEAF_WIN_B));
      ldef->infosize = ldef->numwin;
      for (i = 0; i < ldef->numwin; ++i) {
	 ldef->info[i].status = LEAF_STATUS_NEW;
       };
    }
   else if (ldef->numwin <= ldef->infosize) {
      for (i = ldef->numwin; i < ldef->infosize; ++i) {
	 LEAF_free_component(ldef,i);
	 ldef->info[i].defn = NULL;
       };
    }
   else {
      ldef->info = (LEAF_WIN) realloc(ldef->info,ldef->numwin*sizeof(LEAF_WIN_B));
      for (i = ldef->infosize; i < ldef->numwin; ++i) {
	 ldef->info[i].status = LEAF_STATUS_NEW;
       };
      ldef->infosize = ldef->numwin;
    };

   for (i = 0; i < ldef->numwin; ++i) ldef->info[i].defn = NULL;

   for (i = 0; ldata[i].id != LEAF_BOX_ID_LAST; ++i) {
      if (ldata[i].type & LEAF_TYPE_ARRAY)
	 k = ldata[i].id+LEAF_get_value(ldata[i].data,ldef);
      else k = ldata[i].id+1;
      for (j = ldata[i].id; j < k; ++j) {
	 ldef->info[j].defn = &ldata[i];
	 if (ldef->info[j].status == LEAF_STATUS_NEW) {
	    ldef->info[j].window = NULL;
	    ldef->info[j].region = NULL;
	    if (ldata[i].type & LEAF_TYPE_ICONIC) ldef->info[j].font = ASH_ICON_FONT;
	    else ldef->info[j].font = -1;
	    ldef->info[j].data = ldata[i].data;
	  };
	 ldef->info[j].status = LEAF_STATUS_UNDONE;
       };
    };
};





/************************************************************************/
/*									*/
/*	compute_layout -- compute the coordinates for each component	*/
/*									*/
/************************************************************************/


static void
compute_layout(ldef)
   LEAF_DEF ldef;
{
   register Integer i;

   DTRACE("compute_layout 0x%x",ldef);

   for (i = 0; i < ldef->numwin; ++i) {
      ldef->info[i].status = LEAF_STATUS_UNDONE;
    };

   for (i = 0; i < ldef->numwin; ++i) {
      if (ldef->info[i].defn != NULL &&
	     ldef->info[i].status == LEAF_STATUS_UNDONE)
	 layout_component(ldef,i);
    };
};





/************************************************************************/
/*									*/
/*	layout_component -- layout a single component of window 	*/
/*									*/
/************************************************************************/


static void
layout_component(ldef,num)
   LEAF_DEF ldef;
   Integer num;
{
   register Integer i,j,k;
   Integer co[4];

   DTRACE("layout_component 0x%x %d",ldef,num);

   if (ldef->info[num].defn->id != num) {
      layout_component(ldef,ldef->info[num].defn->id);
    };

   if (ldef->info[num].status == LEAF_STATUS_LAYOUT) return;
   else if (ldef->info[num].status != LEAF_STATUS_UNDONE) {
      FATAL("Circular layout definition");
    };

   if (ldef->info[num].window != NULL && num != LEAF_BOX_ID_ROOT) {
      if ((ldef->info[num].defn->type & LEAF_TYPE_INVISIBLE) == 0)
	 ASHvisible(ldef->info[num].window,FALSE);
    };

   ldef->info[num].status = LEAF_STATUS_WORK;

   if ((ldef->info[num].defn->type & LEAF_TYPE_ARRAY)  == 0) {
      for (i = 0; i < 4; ++i) {
	 ldef->info[num].coord[i] = compute_coord(ldef,num,i,FALSE);
       };
      ldef->info[num].status = LEAF_STATUS_LAYOUT;
    }
   else {
      for (i = 0; i < 4; ++i) {
	 co[i] = compute_coord(ldef,num,i,FALSE);
       };
      j = LEAF_get_value(ldef->info[num].defn->data,ldef);
      k = ((ldef->info[num].defn->type & LEAF_TYPE_SEPARATE) ? 1 : 0);
      for (i = 0; i < j; ++i) {
	 if (i+1 == j) k = 0;
	 ldef->info[num+i].coordlx = co[LEAF_LOC_LX];
	 ldef->info[num+i].coordty = co[LEAF_LOC_TY];
	 if (ldef->info[num].defn->type & LEAF_TYPE_VERTICAL) {
	    ldef->info[num+i].coordrx = co[LEAF_LOC_RX];
	    ldef->info[num+i].coordby =
	       co[LEAF_LOC_TY]-(co[LEAF_LOC_TY]-co[LEAF_LOC_BY])/(j-i)+k*ldef->dy;
	    co[LEAF_LOC_TY] = ldef->info[num+i].coordby-(k+1)*ldef->dy;
	  }
	 else {
	    ldef->info[num+i].coordby = co[LEAF_LOC_BY];
	    ldef->info[num+i].coordrx =
	       co[LEAF_LOC_LX]+(co[LEAF_LOC_RX]-co[LEAF_LOC_LX])/(j-i)-k*ldef->dx;
	    co[LEAF_LOC_LX] = ldef->info[num+i].coordrx+(k+1)*ldef->dx;
	  };
	 ldef->info[num+i].status = LEAF_STATUS_LAYOUT;
       };
    };
};





/************************************************************************/
/*									*/
/*	compute_coord -- compute single coordinate			*/
/*									*/
/************************************************************************/


static Integer
compute_coord(ldef,num,cnm,boxfg)
   LEAF_DEF ldef;
   Integer num;
   Integer cnm;
   Boolean boxfg;
{
   register Integer x,c0,c1,v,v1,delta;
   register LEAF_COORD cd;

   DTRACE("compute_coord 0x%x %d %d",ldef,num,cnm);

   c0 = (cnm&1);
   c1 = c0+2;
   cd = &ldef->info[num].defn->coord[cnm];
   delta = (c0 == 0 ? ldef->dx : ldef->dy);

   v = LEAF_get_value(cd->value,ldef);

   switch ((cd->flags)&LEAF_FLAG_TYPE) {
      case LEAF_FLAG_ABSOLUTE :
	 if (cnm == c0) {
	    x = ldef->coord[c0] + delta*v;
	  }
	 else {
	    x = ldef->coord[c1] - delta*v;
	  }
	 break;
      case LEAF_FLAG_SIZE :
	 if (cnm == c0) {
	    x = compute_coord(ldef,num,c1,TRUE);
	    x = x - (v-1)*delta;
	  }
	 else {
	    x = ldef->info[num].coord[c0] + (v-1)*delta;
	  };
	 break;
      case LEAF_FLAG_REL_PERCENT :
	 v = (abs(ldef->coord[c1]-ldef->coord[c0])*v)/100;
	 if (cnm == c0) {
	    x = compute_coord(ldef,num,c1,TRUE);
	    x = x - (v-1)*delta;
	  }
	 else {
	    x = ldef->info[num].coord[c0] + (v-1)*delta;
	  };
	 break;
      case LEAF_FLAG_PERCENT :
	 x = ((ldef->coord[c1]-ldef->coord[c0])*v)/100+ldef->coord[c0];
	 if (cnm == c0) x += delta;
	 break;
      case LEAF_FLAG_TEXT_SIZE :
	 v = text_size(ldef,num,(c0 == 0));
	 if (cnm == c0) {
	    x = compute_coord(ldef,num,c1,TRUE);
	    x = x - (v-1)*delta;
	  }
	 else {
	    x = ldef->info[num].coord[c0] + (v-1)*delta;
	  };
	 break;
      case LEAF_FLAG_NEXT :
	 layout_component(ldef,v);
	 if (cnm == c0)
	    x = compute_coord(ldef,v,c1,TRUE)+delta;
	 else
	    x = compute_coord(ldef,v,c0,TRUE)-delta;
	 break;
      case LEAF_FLAG_SAME :
	 layout_component(ldef,v);
	 x = compute_coord(ldef,v,cnm,TRUE);
	 break;
      case LEAF_FLAG_PERCENT|LEAF_FLAG_NEXT :
	 v1 = v / 1024;
	 v = v % 1024;
	 v = (abs(ldef->coord[c1]-ldef->coord[c0])*v)/100;
	 layout_component(ldef,v1);
	 if (cnm == c0) {
	    x = compute_coord(ldef,v1,c1,TRUE)+delta;
	    x = x + (v-1)*delta;
	  }
	 else {
	    x = compute_coord(ldef,v1,c0,TRUE)-delta;
	    x = x - (v-1)*delta;
	  };
	 break;
      case LEAF_FLAG_SIZE|LEAF_FLAG_NEXT :
	 v1 = v / 1024;
	 v = v % 1024;
	 layout_component(ldef,v1);
	 if (cnm == c0) {
	    x = compute_coord(ldef,v1,c1,TRUE)+delta;
	    x = x + (v-1)*delta;
	  }
	 else {
	    x = compute_coord(ldef,v1,c0,TRUE)-delta;
	    x = x - (v-1)*delta;
	  };
	 break;
      case LEAF_FLAG_WINDOW :
	 x = ldef->coord[v];
	 break;
      case LEAF_FLAG_SCROLL_WIDTH :
	 v = SCROLL_WIDTH;
	 if (cnm == c0) {
	    x = compute_coord(ldef,num,c1,TRUE);
	    x = x - (v-1)*delta;
	  }
	 else {
	    x = ldef->info[num].coord[c0] + (v-1)*delta;
	  };
	 break;

      default :
	 FATAL("Bad coordinate type speification");
	 break;
    };

   if (!boxfg && (cd->flags)&LEAF_FLAG_DESCRIPTOR) {
      c1 = (cnm == c0 ? 1 : -1)*delta;
      if (cd->flags&LEAF_FLAG_LINE) x += c1;
      if (cd->flags&LEAF_FLAG_DOUBLE_LINE) x += 2*c1;
    };

   return x;
};





/************************************************************************/
/*									*/
/*	text_size -- get size of text window				*/
/*									*/
/************************************************************************/


static Integer
text_size(ldef,i,xfg)
   LEAF_DEF ldef;
   Integer i;
   Boolean xfg;
{
   register Integer font;
   register String s;
   Integer x,y;
   LEAF_TYPE typ;

   DTRACE("text_size 0x%x %d %d",ldef,i,xfg);

   font = LEAF_get_font(ldef,i);

   s = (String) ldef->info[i].data;
   if (s == NULL || *s == 0) {
      if (xfg) s = "M";
      else s = "Mp_";
    };

   typ = (ldef->info[i].defn->type & LEAF_TYPE_BASE);

   if (typ == LEAF_TYPE_BTN || typ == LEAF_TYPE_PDM || typ == LEAF_TYPE_STEM) {
      STEMbtn_size(font,s,&x,&y);
      if (typ == LEAF_TYPE_STEM) y += 4;
    }
   else {
      ASHinq_text_next(font,s,&x,&y);
    };

   return (xfg ? x+1 : y+1);
};





/* end of leaflayout.c */
