/************************************************************************/
/*									*/
/*		leafdraw.c						*/
/*									*/
/*	Drawing routines for LEAF window layout package 		*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "leaf_local.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define DEFAULT_TITLE_FILL	ASH_FILL_VERY_LIGHT





/************************************************************************/
/*									*/
/*	Variable definitions						*/
/*									*/
/************************************************************************/


static	ASH_FILL_STYLE	title_fill;



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Integer 	leaf_root_control();
static	void		leaf_root_refresh();
static	void		redraw_component();
static	void		redraw_box();
static	void		redraw_window();
static	void		setup_rip_region();
static	void		box_lines();
static	void		draw_line();
static	void		setup_window();




/************************************************************************/
/*									*/
/*	LEAF_layout_init -- module initialization			*/
/*									*/
/************************************************************************/


void
LEAF_draw_init()
{
   String s;

   ITRACE("LEAF_draw_init");

   s = ASHinq_resource("leaf.titlefill");
   if (s != NULL) title_fill = atol(s);
   else title_fill = DEFAULT_TITLE_FILL;
};





/************************************************************************/
/*									*/
/*	LEAFredraw -- draw or redraw a window layout			*/
/*									*/
/************************************************************************/


void
LEAFredraw(w)
   ASH_WINDOW w;
{
   register LEAF_DEF ldef;
   register Integer i;

   TRACE("LEAFredraw 0x%x",w);

   ldef = LEAF_find_window_defn(w);
   if (ldef == NULL) return;

   if (!ASHlock(w)) return;

   ASHbatch_mode(TRUE);
   ASHcombination_rule(w,3);
   ASHline_style(w,ASH_STYLE_SOLID);
   ASHfill(w,ASH_FILL_SOLID);

   for (i = 0; i < ldef->numwin; ++i) {
      if (!ASHinq_valid_window(ldef->window)) break;
      if (ldef->info[i].defn != NULL) {
	 if (ldef->info[i].status != LEAF_STATUS_DRAWN) {
	    redraw_component(ldef,i);
	    if (!ldef->resize_only) box_lines(ldef,i);
	  };
       };
    };

   ASHunlock(w);

   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	LEAFremove -- remove leaf control from a window 		*/
/*									*/
/************************************************************************/


void
LEAFremove(w)
   ASH_WINDOW w;
{
   register LEAF_DEF ldef;

   TRACE("LEAFremove 0x%x",w);

   ldef = LEAF_find_window_defn(w);

   LEAF_remove(ldef,FALSE);
};





/************************************************************************/
/*									*/
/*	LEAF_free_component -- release a component			*/
/*									*/
/************************************************************************/


void
LEAF_free_component(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   ITRACE("LEAF_free_component 0x%x %d",ldef,i);

   if (ldef->info[i].defn != NULL) {
      if (ldef->info[i].window != NULL) {
	 if (i != 0) ASHremove(ldef->info[i].window);
	 ldef->info[i].window = NULL;
	 ldef->info[i].region = NULL;
       };
      if (ldef->info[i].region != NULL) {
	 RIPremove_region(ldef->info[i].region);
	 ldef->info[i].region = NULL;
       };
      ldef->info[i].defn = NULL;
    };
};





/************************************************************************/
/*									*/
/*	LEAF_remove -- remove a LEAF window				*/
/*									*/
/************************************************************************/


void
LEAF_remove(ldef,rfg)
   LEAF_DEF ldef;
   Boolean rfg;
{
   register Integer i;

   ITRACE("LEAF_remove 0x%x",ldef);

   if (!rfg) {
      ASHremove_control(ldef->window,leaf_root_control);
      ASHset_refresh(ldef->window,ASH_REFRESH_CLEAR);
      ASHset_menu_data(ldef->window,NULL);
      ASHset_user_data(ldef->window,NULL);
      for (i = 0; i < ldef->numwin; ++i) {
	 LEAF_free_component(ldef,i);
       };
    };

   free(ldef->info);
   ldef->info = NULL;
   free(ldef);
};




/************************************************************************/
/*									*/
/*	leaf_root_control -- control function for root windows		*/
/*									*/
/************************************************************************/


static Integer
leaf_root_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register LEAF_DEF ldef;
   register Integer sts;

   ITRACE("leaf_root_control %s 0x%x",msg,w);

   ldef = (LEAF_DEF) ASHinq_menu_data(w);
   if (ldef == NULL || ldef->window != w) return ASH_CONTROL_REJECT;

   sts = ASH_CONTROL_REJECT;

   if (ldef->ctrl_rtn != NULL) {
      sts = (*(ldef->ctrl_rtn))(msg,w);
    };

   if (STREQL(msg,"ASH$RESIZE")) {
      if (ASHlock(w)) {
	 LEAF_recompute_layout(ldef);
	 ASHunlock(w);
       };
#ifdef RESIZE_INVISIBLE
      ldef->resize_only = TRUE;
#endif
      LEAFredraw(ldef->window); 	/* do we need this */
      ldef->resize_only = FALSE;
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      ASHremove_control(ldef->window,leaf_root_control);
      LEAF_remove(ldef,TRUE);
    };

   return sts;
};





/************************************************************************/
/*									*/
/*	leaf_root_refresh -- handle refresh requests for root window	*/
/*									*/
/************************************************************************/


static void
leaf_root_refresh(w)
   ASH_WINDOW w;
{
   register Integer i;
   register LEAF_DEF ldef;

   ITRACE("leaf_root_refresh");

   ldef = (LEAF_DEF) ASHinq_menu_data(w);
   for (i = 0; i < ldef->numwin; ++i) {
      if (ldef->info[i].status == LEAF_STATUS_DRAWN)
	 ldef->info[i].status = LEAF_STATUS_LAYOUT;
    };

   LEAFredraw(w);

   if (ldef->refresh_rtn != NULL) (*(ldef->refresh_rtn))(w);
};





/************************************************************************/
/*									*/
/*	redraw_component -- redraw arbitrary component			*/
/*									*/
/************************************************************************/


static void
redraw_component(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   DTRACE("redraw_component 0x%x %d",ldef,i);

   if (ldef->info[i].status == LEAF_STATUS_DRAWN) return;
   else if (ldef->info[i].status != LEAF_STATUS_LAYOUT)
      LEAF_recompute_layout(ldef);

   switch (ldef->info[i].defn->type & LEAF_TYPE_BASE) {
      case LEAF_TYPE_ROOT :
	 if (ldef->info[i].window == NULL) {
	    ldef->info[i].window = ldef->window;
	    ASHset_control(ldef->info[i].window,leaf_root_control);
	    ASHset_region_refresh(ldef->info[i].window,leaf_root_refresh);
	    ASHset_user_data(ldef->info[i].window,ldef->data);
	    setup_window(ldef,i);
	  };
	 redraw_box(ldef,i);
	 break;
      case LEAF_TYPE_NONE :
      case LEAF_TYPE_TEXT :
      case LEAF_TYPE_OLD_TEXT :
      case LEAF_TYPE_TEXT_CENTERED :
      case LEAF_TYPE_TEXT_RIGHT :
      case LEAF_TYPE_SHADE :
	 redraw_box(ldef,i);
	 break;
      case LEAF_TYPE_WINDOW :
      case LEAF_TYPE_PDM :
      case LEAF_TYPE_STEM :
      case LEAF_TYPE_BTN :
	 redraw_window(ldef,i);
	 break;
    };

   if (!ldef->resize_only) ldef->info[i].status = LEAF_STATUS_DRAWN;
};





/************************************************************************/
/*									*/
/*	redraw_box -- redraw a single box-like component		*/
/*									*/
/************************************************************************/


static void
redraw_box(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   Integer x,y,xs,ys,xt,yt;
   register Integer font;
   register String s;
   Integer fl;

   DTRACE("redraw_box 0x%x %d",ldef,i);

   if (ldef->resize_only) return;

   if (ASHinq_visible(ldef->window)) {
      ASHclip_region(ldef->window,ldef->info[i].coordlx,ldef->info[i].coordby,
			ldef->info[i].coordrx,ldef->info[i].coordty);
      ASHclip(ldef->window,TRUE);

      if (ldef->info[i].defn->type & LEAF_TYPE_SHADE) {
	 fl = ASHfill(ldef->window,ldef->info[i].data);
	 ASHrectangle(ldef->window,ldef->info[i].coordlx,ldef->info[i].coordby,
			 ldef->info[i].coordrx,ldef->info[i].coordty);
	 ASHfill(ldef->window,fl);
	 s = NULL;
       }
      else if ((ldef->info[i].defn->type & LEAF_TYPE_USE_TITLE) != 0 && ldef->active) {
	 fl = ASHfill(ldef->window,title_fill);
	 ASHrectangle(ldef->window,ldef->info[i].coordlx,ldef->info[i].coordby,
			 ldef->info[i].coordrx,ldef->info[i].coordty);
	 ASHfill(ldef->window,fl);
	 s = (String) ldef->info[i].data;
       }
      else {
	 ASHclear_box(ldef->window,ldef->info[i].coordlx,ldef->info[i].coordby,
			 ldef->info[i].coordrx,ldef->info[i].coordty);
	 s = (String) ldef->info[i].data;
       };

      if (s != NULL) {
	 font = LEAF_get_font(ldef,i);
	 ASHfont(ldef->window,font);
	 ASHinq_text_offset(font,s,&x,&y);
	 ASHinq_text_extent(font,s,&xs,&ys);
	 if (font != ASH_ICON_FONT) ASHinq_text_extent(font,"Mp_",&xt,&yt);
	 else ASHinq_text_extent(font,s,&xt,&yt);

	 y = (ldef->info[i].coordby+ldef->info[i].coordty-yt*ldef->dy)/2+y*ldef->dy;

	 switch (ldef->info[i].defn->type & LEAF_TYPE_BASE) {
	    case LEAF_TYPE_TEXT :
	    case LEAF_TYPE_OLD_TEXT :
	       x = ldef->info[i].coordlx+(x+1)*ldef->dx;
	       break;
	    case LEAF_TYPE_TEXT_CENTERED :
	       x = (ldef->info[i].coordrx+ldef->info[i].coordlx-xs*ldef->dx)/2+x*ldef->dx;
	       if (x < ldef->info[i].coordlx) x = ldef->info[i].coordlx;
	       break;
	    case LEAF_TYPE_TEXT_RIGHT :
	       ASHinq_text_extent(font,s,&xs,&ys);
	       x = ldef->info[i].coordrx-xs*ldef->dx+(x-3)*ldef->dx;
	       if (x < ldef->info[i].coordlx) x= ldef->info[i].coordlx;
	       break;
	    default :
	       x = y = -1;
	       break;
	  };

	 if (x >= 0) {
	    ASHtext(ldef->window,x,y,s);
	  };

	 ASHfont(ldef->window,ldef->font);
       };

      ASHclip(ldef->window,FALSE);
    };

   if (ldef->info[i].defn->type & LEAF_TYPE_USE_TITLE) {
      s = ASHinq_window_name(ldef->window);
      if ((s == NULL && ldef->info[i].data != NULL) ||
	     (s != NULL && ldef->info[i].data == NULL) ||
	     (s != NULL && STRNEQ(s,ldef->info[i].data)))
	 ASHset_window_name(ldef->window,ldef->info[i].data);
    };

   setup_rip_region(ldef,i);

   if (ldef->info[i].region != NULL && s != NULL)
      RIPset_region_id(ldef->info[i].region,s);
};





/************************************************************************/
/*									*/
/*	redraw_window -- redraw a window-like component 		*/
/*									*/
/************************************************************************/


static void
redraw_window(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   Boolean rsz,new;
   Integer lx,by,rx,ty;
   Integer olx,oby,orx,oty;
   Integer fgs;
   ASH_WINDOW w;

   DTRACE("redraw_window 0x%x %d",ldef,i);

   lx = 0;
   rx = abs(ldef->info[i].coordrx-ldef->info[i].coordlx);
   if (ldef->info[i].defn->type & LEAF_TYPE_UPPER_LEFT) {
      ty = 0;
      by = abs(ldef->info[i].coordty-ldef->info[i].coordby);
    }
   else {
      by = 0;
      ty = abs(ldef->info[i].coordty-ldef->info[i].coordby);
    };

   rsz = TRUE;

   if (!ASHinq_valid_window(ldef->info[i].window)) ldef->info[i].window = NULL;

   if (ldef->dx*(ldef->info[i].coordrx - ldef->info[i].coordlx) <= 0 ||
	  ldef->dy*(ldef->info[i].coordty - ldef->info[i].coordby) <= 0) {
      if (ldef->info[i].window != NULL) {
	 w = ldef->info[i].window;
	 ldef->info[i].window = NULL;
	 ASHremove(w);
       };
    }
   else if (ldef->info[i].window == NULL) {
      new = TRUE;
      ldef->info[i].window =
	 ASHcreate(ldef->window,ldef->info[i].coordlx,ldef->info[i].coordby,
		      lx,by,rx,ty,
		      ASH_BORDER_NONE,
		      ASH_WINDOW_NOSAVE|ASH_WINDOW_INVISIBLE);
      setup_window(ldef,i);
    }
   else {
      new = FALSE;
      ASHinq_size(ldef->info[i].window,ASH_SIZE_WINDOW,&olx,&oby,&orx,&oty);
      ASHmap(ldef->info[i].window,olx,oby,ldef->window,&olx,&oby);
      ASHmap(ldef->info[i].window,orx,oty,ldef->window,&orx,&oty);
      if (orx-olx != ldef->info[i].coordrx-ldef->info[i].coordlx ||
	     oty-oby != ldef->info[i].coordty-ldef->info[i].coordby) {
	 ASHresize(ldef->info[i].window,
		      ldef->info[i].coordlx,ldef->info[i].coordby,
		      lx,by,rx,ty);
       }
      else {
	 rsz = FALSE;
	 if (olx != ldef->info[i].coordlx || oby != ldef->info[i].coordby) {
	    ASHview(ldef->info[i].window,
		       ldef->info[i].coordlx,ldef->info[i].coordby,
		       lx,by,rx,ty);
	  };
       };
    };

   if (ldef->info[i].window != NULL) {
      if (new || rsz) setup_rip_region(ldef,i);

      switch (ldef->info[i].defn->type & LEAF_TYPE_BASE) {
	 case LEAF_TYPE_STEM :
	    if (new) {
	       STEMmenu(ldef->info[i].window,ldef->info[i].defn->button_fct,
			   ldef->data,RIP_BTN_ANY_EITHER,NULL);
	     };
	    break;
	 case LEAF_TYPE_BTN :
	    if (new) {
	       if (ldef->info[i].defn->type & LEAF_TYPE_SENSE_FLIP) fgs = STEM_BTN_SENSE_FLIP;
	       else fgs = 0;
	       STEMbtn_define(ldef->info[i].window,0,0,0,0,ldef->info[i].data,
				 STEM_BTN_DEFAULT|fgs,
				 ldef->data,ldef->info[i].defn->button_fct);
	     };
       };

      if ((ldef->info[i].defn->type & LEAF_TYPE_INVISIBLE) == 0) {
	 if (!ASHinq_visible(ldef->info[i].window,TRUE))
	    ASHvisible(ldef->info[i].window,TRUE);
       };
    };
};





/************************************************************************/
/*									*/
/*	setup_rip_region -- set up rip region for area			*/
/*									*/
/************************************************************************/


static void
setup_rip_region(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   register ASH_WINDOW w;
   Integer lx,by,rx,ty;

   DTRACE("setup_rip_region 0x%x %d",ldef,i);

   switch (ldef->info[i].defn->type & LEAF_TYPE_BASE) {
      case LEAF_TYPE_STEM :
      case LEAF_TYPE_BTN :
	 return;
      default :
	 if (ldef->info[i].defn->button_fct == NULL &&
		(ldef->info[i].defn->type & LEAF_TYPE_SENSE_FLIP) == 0) return;
	 break;
    };

   if (ldef->info[i].region != NULL) {
      RIPremove_region(ldef->info[i].region);
    };

   if (ldef->info[i].window == NULL) {
      w = ldef->window;
      lx = ldef->info[i].coordlx;
      rx = ldef->info[i].coordrx;
      by = ldef->info[i].coordby;
      ty = ldef->info[i].coordty;
    }
   else {
      w = ldef->info[i].window;
      lx = by = rx = ty = 0;			/* use whole window	*/
    };

   ldef->info[i].region =
      RIPdefine_region(w,lx,by,rx,ty,
			  RIP_ALL_CHARS, RIP_BTN_ANY_EITHER,
			  ldef->info[i].defn->button_fct,FALSE);

   RIPset_data(ldef->info[i].region,ldef->data);
   if (ldef->info[i].defn->type & LEAF_TYPE_SENSE_FLIP) {
      RIPset_sense(ldef->info[i].region,ASH_SENSE_FLIP);
    };
};





/************************************************************************/
/*									*/
/*	box_lines -- draw lines for box as needed			*/
/*									*/
/************************************************************************/


static void
box_lines(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   register Integer j;
   register LEAF_BOX lb;
   Integer rx,by;
   Integer dl[4];

   DTRACE("box_lines 0x%x %d",ldef,i);

   lb = ldef->info[i].defn;

   for (j = 0; j < 4; ++j) {
      dl[j] = 0;
      if (lb->coord[j].flags&LEAF_FLAG_LINE) dl[j] += 1;
      if (lb->coord[j].flags&LEAF_FLAG_DOUBLE_LINE) dl[j] += 2;
      DTRACE("\tflags = 0x%x dl = %d", lb->coord[j].flags,dl[j]);
    };

   if (lb->id != i) {
      if (lb->type & LEAF_TYPE_SEPARATE) {
	 if (lb->type & LEAF_TYPE_VERTICAL) {
	    dl[LEAF_LOC_TY] = 1;
	    draw_line(ldef,i,dl,LEAF_LOC_TY);
	  }
	 else {
	    dl[LEAF_LOC_LX] = 1;
	    draw_line(ldef,i,dl,LEAF_LOC_LX);
	  };
       };
    }
   else if (lb->type & LEAF_TYPE_ARRAY) {
      j = LEAF_get_value(lb->data,ldef);
      if (j != 0) {
	 rx = ldef->info[i].coordrx;
	 by = ldef->info[i].coordby;
	 ldef->info[i].coordrx = ldef->info[i+j-1].coordrx;
	 ldef->info[i].coordby = ldef->info[i+j-1].coordby;
	 for (j = 0; j < 4; ++j) {
	    if (dl[j] != 0) draw_line(ldef,i,dl,j);
	  };
	 ldef->info[i].coordrx = rx;
	 ldef->info[i].coordby = by;
       };
    }
   else {
      for (j = 0; j < 4; ++j) {
	 if (dl[j] != 0) draw_line(ldef,i,dl,j);
       };
    };
};





/************************************************************************/
/*									*/
/*	draw_line -- draw a bounding line				*/
/*									*/
/************************************************************************/


static void
draw_line(ldef,i,dl,loc)
   LEAF_DEF ldef;
   Integer i;
   Integer dl[4];
   Integer loc;
{
   register Integer x0,y0,x1,y1;
   register Integer dx,dy,j;

   DTRACE("draw_line 0x%x %d %d",ldef,i,loc);

   if (loc&1) {
      x0 = ldef->info[i].coordlx-dl[LEAF_LOC_LX]*ldef->dx;
      x1 = ldef->info[i].coordrx+dl[LEAF_LOC_RX]*ldef->dx;
      dx = 0;
      dy = (loc == LEAF_LOC_BY ? -1 : 1)*ldef->dy;
      y0 = y1 = ldef->info[i].coord[loc]+dy;
    }
   else {
      y0 = ldef->info[i].coordby-dl[LEAF_LOC_BY]*ldef->dy;
      y1 = ldef->info[i].coordty+dl[LEAF_LOC_TY]*ldef->dy;
      dy = 0;
      dx = (loc == LEAF_LOC_LX ? -1 : 1)*ldef->dx;
      x0 = x1 = ldef->info[i].coord[loc]+dx;
    };

   for (j = 0; j < dl[loc]; ++j) {
      ASHline(ldef->window,x0,y0,x1,y1);
      x0 += dx;
      x1 += dx;
      y0 += dy;
      y1 += dy;
    };
};





/************************************************************************/
/*									*/
/*	setup_window -- setup a new window				*/
/*									*/
/************************************************************************/


static void
setup_window(ldef,i)
   LEAF_DEF ldef;
   Integer i;
{
   DTRACE("setup_window");

   ASHfont(ldef->info[i].window,LEAF_get_font(ldef,i));
};





/* end of leafdraw.c */
