/************************************************************************/
/*									*/
/*		helpitem.c						*/
/*									*/
/*	Item manipulations for HELP faciltiy				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/



#include "help_local.h"

#include <sys/types.h>
#include <sys/stat.h>
#include <sys/file.h>
#include <sys/dir.h>




/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	HELP_DATA	top_data;

static	String		ignore_files[] = {
   "bBACKUP",
   "bCONTROL",
   "bBUFFERS",
   0
};





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	set_file_type();
static	Boolean 	read_directory();
static	Boolean 	read_file();
static	String		scan_name();





/************************************************************************/
/*									*/
/*	HELP_item_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
HELP_item_init()
{
   top_data = NULL;
};




/************************************************************************/
/*									*/
/*	HELPsetup -- declare a item to have help on			*/
/*									*/
/************************************************************************/


void
HELPsetup(name,file)
   String name;
   String file;
{
   Character buf[128];

   HELPinit();

   if (file == NULL) {
      sprintf(buf,"%s/%s/%s",BWEbwe_project(),HELP_DIRECTORY,name);
      file = buf;
    };

   PROTECT;
   HELP_item_add(name,file,0,NULL);
   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	HELP_item_add -- add item to current help structure		*/
/*									*/
/************************************************************************/


HELP_DATA
HELP_item_add(name,file,pos,level)
   String name;
   String file;
   Integer pos;
   HELP_DATA level;
{
   HELP_DATA d,b;

   d = PALLOC(HELP_DATA_INFO);
   d->name = SALLOC(name);
   d->filename = SALLOC(file);
   d->filepos = pos;
   d->textfile = NULL;
   d->textpos = 0;
   d->flags = HELP_FLAG_INITIAL;

   d->parent = level;
   d->brother = NULL;
   d->son = NULL;

   if (level != NULL) {
      if (level->son == NULL) {
	 level->son = d;
       }
      else {
	 for (b = level->son;
	      b->brother != NULL;
	      b = b->brother);
	 b->brother = d;
       };
    }
   else {
      if (top_data == NULL) {
	 top_data = d;
       }
      else {
	 for (b = top_data; b->brother != NULL; b = b->brother);
	 b->brother = d;
       };
    };

   return d;
};





/************************************************************************/
/*									*/
/*	HELP_item_expand -- expand item 				*/
/*									*/
/************************************************************************/


void
HELP_item_expand(d)
   HELP_DATA d;
{
   PROTECT;

   if ((d->flags & HELP_FLAG_EXPANDED) == 0) {
      if (set_file_type(d)) {
	 if (d->flags & HELP_FLAG_DIRECTORY) {
	    read_directory(d);
	  }
	 else {
	    read_file(d);
	  };
       };

      d->flags |= HELP_FLAG_EXPANDED;
    };

   UNPROTECT;
};





/************************************************************************/
/*									*/
/*	HELP_inq_top_data -- return topmost help data block		*/
/*									*/
/************************************************************************/


HELP_DATA
HELP_inq_top_data()
{
   return top_data;
};






/************************************************************************/
/*									*/
/*	HELP_item_find -- find item given list of names 		*/
/*									*/
/************************************************************************/


HELP_DATA
HELP_item_find(ct,ids)
   Integer ct;
   String ids[];
{
   HELP_DATA hd;
   Integer i;

   hd = NULL;
   for (i = ct-1; i >= 0; --i) {
      for (hd = (hd == NULL ? top_data : hd->son); hd != NULL; hd = hd->brother) {
	 if (STREQL(ids[i],hd->name)) break;
       };
      if (hd == NULL) break;
      HELP_item_expand(hd);
    };

   return hd;
};





/************************************************************************/
/*									*/
/*	set_file_type -- determine type of file for help data block	*/
/*									*/
/************************************************************************/


static Boolean
set_file_type(d)
   HELP_DATA d;
{
   struct stat sbuf;

   if (stat(d->filename,&sbuf) < 0) {
      fprintf(stderr,"HELP: Can't find help file %s\n",d->filename);
      d->flags |= HELP_FLAG_IGNORE;
    }
   else {
      switch (sbuf.st_mode & S_IFMT) {
	 case S_IFDIR :
	    d->flags |= HELP_FLAG_DIRECTORY;
	    break;

	 case S_IFBLK :
	 case S_IFCHR :
	    d->flags |= HELP_FLAG_IGNORE;
	    break;

	 default :
	    d->flags |= HELP_FLAG_FILE;
	    break;
       };
    };

   return ((d->flags & HELP_FLAG_IGNORE) == 0);
};





/************************************************************************/
/*									*/
/*	read_directory -- get information from HELP directory		*/
/*									*/
/************************************************************************/


static Boolean
read_directory(d)
   HELP_DATA d;
{
   Character buf[256],buf1[256];
   DIR *dir;
   struct direct *dp;
   Integer i;

   dir = opendir(d->filename);
   if (dir == NULL) return FALSE;

   for (dp = readdir(dir); dp != NULL; dp = readdir(dir)) {
      if (*dp->d_name == '.') continue;
      for (i = 0; ignore_files[i] != NULL; ++i) {
	 if (STREQL(dp->d_name,ignore_files[i])) break;
       };
      if (ignore_files[i] != NULL) continue;
      sprintf(buf,"%s/%s",d->filename,dp->d_name);
      if (STREQL(dp->d_name,HELP_TEXT_NAME)) {
	 d->textfile = SALLOC(buf);
	 d->textpos = 0;
       }
      else {
	 i = strlen(dp->d_name);
	 if (i > 5 && STREQL(&dp->d_name[i-5],".help")) {
	    strcpy(buf1,dp->d_name);
	    buf1[i-5] = 0;
	    HELP_item_add(buf1,buf,0,d);
	  }
	 else HELP_item_add(dp->d_name,buf,0,d);
       };
    };

   closedir(dir);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	read_file -- get information from HELP file			*/
/*									*/
/************************************************************************/


static Boolean
read_file(d)
   HELP_DATA d;
{
   FILE *inf;
   Character buf[1024],nam[128],fnam[128];
   HELP_DATA stack[256];
   Integer level,pos;
   Boolean textfg;
   String p;

   inf = fopen(d->filename,"r");
   if (inf == NULL) return FALSE;

   if (d->filepos != 0) fseek(inf,0,d->filepos);

   level = 0;
   stack[level] = d;
   textfg = FALSE;

   while (((pos = ftell(inf)),(fgets(buf,1024,inf) != NULL))) {
      if (textfg && buf[0] != HELP_TEXT_FLAG) continue;
      textfg = FALSE;
      if (buf[0] == '\n') continue;
      if (buf[0] != HELP_TEXT_FLAG) {
	 stack[level]->textfile = d->filename;
	 stack[level]->textpos = pos;
	 textfg = TRUE;
       }
      else if (buf[1] == HELP_TEXT_FLAG) {
	 if (level > 0) --level;
       }
      else {
	 p = scan_name(&buf[1],nam);
	 if (nam[0] == 0) continue;
	 p = scan_name(p,fnam);
	 if (fnam[0] == 0) {
	    ++level;
	    stack[level] = HELP_item_add(nam,d->filename,pos,stack[level-1]);
	    stack[level]->flags |= HELP_FLAG_EXPANDED;
	    stack[level]->flags |= HELP_FLAG_FILE;
	  }
	 else {
	    HELP_item_add(nam,fnam,0,stack[level]);
	  };
       };
    };

   fclose(inf);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	scan_name -- scan a name from the text string			*/
/*									*/
/************************************************************************/


static String
scan_name(s,buf)
   String s;
   String buf;
{
   Integer ch;

   while (isspace(*s)) ++s;

   if (*s != '"' && *s != '\'') {
      while (isalnum(*s) || ispunct(*s)) *buf++ = *s++;
    }
   else {
      ch = *s++;
      while (*s != ch && *s != 0) *buf++ = *s++;
      if (*s == ch) ++s;
    };

   *buf = 0;

   return s;
};





/* end of helpitem.c */
