/************************************************************************/
/*									*/
/*		gelowin.c						*/
/*									*/
/*	Window management routines for GELO				*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"






/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define WORLD_SIZE_PARAM	3





/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Tables								*/
/*									*/
/************************************************************************/


static Integer	fill_styles[] = {	/* fill styles for hilighting	*/
   0,					/*    ignore			*/
   1,					/*    main			*/
   2,					/*    secondard 		*/
   3,					/*    tertiary			*/
   4,					/*    temporary 		*/
   5,					/*    execution 		*/
   6,					/*    current			*/
};





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		gelo_refresh();
static	int		gelo_control();
static	void		resize_gelo_window();
static	void		free_gelo_window();
static	void		redraw_object();





/************************************************************************/
/*									*/
/*	GELO_win_init -- module initialization				*/
/*									*/
/************************************************************************/


void
GELO_win_init()
{
   ITRACE("GELO_win_init");
};





/************************************************************************/
/*									*/
/*	GELOwindow_open -- create a GELO window 			*/
/*									*/
/************************************************************************/


void
GELOwindow_open(w)
   ASH_WINDOW w;
{
   register GELO_WINDOW gw;
   Integer lx,by,rx,ty;

   ENTER("GELOwindow_open");

   gw = (GELO_WINDOW) calloc(1,sizeof(GELO_WINDOW_B));

   gw->realwindow = w;
   gw->object = NULL;
   gw->size_valid = FALSE;
   gw->select = NULL;

   ASHinq_size(gw->realwindow,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   gw->window = ASHcreate(gw->realwindow,
			     lx,by,0,abs(ty-by),abs(rx-lx),0,0,
			     ASH_WINDOW_NOSAVE|ASH_WINDOW_INVISIBLE);
   ASHset_region_refresh(gw->window,gelo_refresh);
   ASHinq_size(gw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHset_control(gw->realwindow,gelo_control);
   ASHset_user_data(gw->window,gw);
   ASHset_user_data(gw->realwindow,gw);

   ASHvisible(gw->window,TRUE);

   gw->real_lx = lx;
   gw->real_by = by;
   gw->real_rx = rx;
   gw->real_ty = ty;

   gw->view_lx = lx;
   gw->view_by = by;
   gw->view_rx = rx;
   gw->view_ty = ty;

   gw->world_x = (rx+1);
   gw->world_y = (by+1);
};





/************************************************************************/
/*									*/
/*	GELOwindow_close -- close a given GELO window			*/
/*									*/
/************************************************************************/


void
GELOwindow_close(w)
   ASH_WINDOW w;
{
   GELO_WINDOW gw;

   TRACE("GELOwindow_close 0x%x",w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   if (gw != NULL) {
      ASHremove_control(gw->realwindow,gelo_control);
      ASHset_user_data(gw->realwindow,NULL);
      ASHremove(gw->window);
      gw->window = NULL;
      free_gelo_window(gw);
      ASHclear(w);
    };
};





/************************************************************************/
/*									*/
/*	GELOwindow_draw -- draw given gelo object in the window 	*/
/*									*/
/************************************************************************/


void
GELOwindow_draw(w,go)
   ASH_WINDOW w;
   GELO_OBJECT go;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_draw 0x%x 0x%x",w,go);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   if (gw == NULL) return;
   if (gw->object != NULL && gw->object != go) {
      GELO_set_x_position(gw->object,65536);		/* move off screen */
    };

   gw->object = go;
   gw->size_valid = FALSE;
   if (gw->select != NULL) {
      LFREE(gw->select);
      gw->select = NULL;
    };

   ASHclear_damage(gw->window);

   GELOwindow_redraw(w);
};




/************************************************************************/
/*									*/
/*	GELOwindow_redraw -- redraw the given window			*/
/*									*/
/************************************************************************/


void
GELOwindow_redraw(w)
   ASH_WINDOW w;
{
   register GELO_WINDOW gw;
   register Integer xsz,ysz;
   register GELO_OBJECT gp;
   register Sequence l;
   Integer lx,by,rx,ty;

   TRACE("GELOwindow_redraw 0x%x",w);

   if (w == NULL) return;
   gw = (GELO_WINDOW) ASHinq_user_data(w);
   if (gw == NULL || (gw->window != w && gw->realwindow != w)) return;

   if (!ASHlock(gw->window)) return;

   ASHbatch_mode(TRUE);
   PROTECT;

   ASHinq_size(gw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);
   ASHclear_box(gw->window,lx,by,rx,ty);

   if (gw->object != NULL) {
      if (!gw->size_valid) {
	 xsz = gw->world_x;
	 ysz = gw->world_y;
	 GELO_size(gw->object,xsz,ysz);
	 GELO_set_x_size(gw->object,xsz);
	 GELO_set_y_size(gw->object,ysz);
	 gw->size_valid = TRUE;
       };

      GELO_set_x_position(gw->object,0);
      GELO_set_y_position(gw->object,0);
      GELO_relayout(gw,gw->object);

      gp = gw->object->parent;
      gw->object->parent = NULL;
      GELO_draw(gw,gw->object,GELO_DRAW_NORM);
      gw->object->parent = gp;

      if (gw->select != NULL) {
	 l = gw->select;
	 gw->select = NULL;
	 while (l != NULL) {
	    GELOwindow_select(w,CAR(GELO_OBJECT,l),CADR(Integer,l),TRUE);
	    l = CDRF(CDRF(l));
	  };
       };
    };

   UNPROTECT;
   ASHunlock(gw->window);
   ASHbatch_mode(FALSE);
};





/************************************************************************/
/*									*/
/*	GELOwindow_set_view -- set view for given window		*/
/*	GELOwindow_inq_view -- return view for given window		*/
/*	GELOwindow_inq_real -- return real view for given window	*/
/*									*/
/************************************************************************/


void
GELOwindow_set_view(w,lx,by,rx,ty)
   ASH_WINDOW w;
   Integer lx,by,rx,ty;
{
   register GELO_WINDOW gw;
   register Integer x0,x1;

   TRACE("GELOwindow_set_view 0x%x %d %d %d %d",w,lx,by,rx,ty);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   gw->view_lx = lx;
   gw->view_by = by;
   gw->view_rx = rx;
   gw->view_ty = ty;

   x0 = abs(gw->real_rx - gw->real_lx)+1;
   x0 *= x0;
   x1 = abs(rx-lx)+1;
   x1 *= WORLD_SIZE_PARAM;
   x0 /= x1;

   x1 = 1;
   while (gw->world_x < x0) {
      x1 *= 2;
      gw->world_x *= 2;
      gw->world_y *= 2;
      gw->size_valid = FALSE;
    };

   if (x1 != 1) {
      gw->view_lx *= x1;
      gw->view_by *= x1;
      gw->view_rx *= x1;
      gw->view_ty *= x1;
    };
};





void
GELOwindow_inq_real(w,lxp,byp,rxp,typ)
   ASH_WINDOW w;
   Integer *lxp,*byp,*rxp,*typ;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_inq_real 0x%x",w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   *lxp = gw->real_lx;
   *byp = gw->real_by;
   *rxp = gw->real_rx;
   *typ = gw->real_ty;
};





void
GELOwindow_inq_view(w,lxp,byp,rxp,typ)
   ASH_WINDOW w;
   Integer *lxp,*byp,*rxp,*typ;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_inq_view 0x%x",w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   *lxp = gw->view_lx;
   *byp = gw->view_by;
   *rxp = gw->view_rx;
   *typ = gw->view_ty;
};





void
GELOwindow_inq_world(w,lxp,byp,rxp,typ)
   ASH_WINDOW w;
   Integer *lxp,*byp,*rxp,*typ;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_inq_world 0x%x",w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   *lxp = 0;
   *byp = gw->world_y-1;
   *rxp = gw->world_x-1;
   *typ = 0;
};





/************************************************************************/
/*									*/
/*	GELOwindow_select -- hilite item in window			*/
/*									*/
/************************************************************************/


void
GELOwindow_select(w,go,sty,fg)
   ASH_WINDOW w;
   GELO_OBJECT go;
   Integer sty;
   Boolean fg;
{
   register GELO_WINDOW gw;
   register Sequence l,ll;

   TRACE("GELOwindow_select 0x%x 0x%x %d %d",w,go,sty,fg);

   gw = (GELO_WINDOW) ASHinq_user_data(w);
   if (go == NULL) return;

   if (!ASHlock(gw->window)) return;

   if (!fg) {
      ll = NULL;
      for (l = gw->select; l != NULL; l = CDDR(l)) {
	 if (CAR(GELO_OBJECT,l) == go && CADR(Integer,l) == sty) break;
	 ll = l;
       };
      if (l == NULL) {
	 ASHunlock(gw->window);
	 return;
       };
      if (ll == NULL) {
	 gw->select = CDRF(CDRF(gw->select));
       }
      else {
	 RPLACD(CDR(ll),CDDR(l));
	 RPLACD(CDR(l),NULL);
	 LFREE(l);
       };
    }
   else {
      gw->select = APPEND(go,gw->select);
      gw->select = APPEND(sty,gw->select);
    };

   sty = fill_styles[sty];

   PROTECT;
   GELO_hilite(gw,go,sty,fg);
   UNPROTECT;

   ASHunlock(gw->window);
};






void
GELO_deselect(gw,go)
   GELO_WINDOW gw;
   GELO_OBJECT go;
{
   register Sequence l,ll;

   ITRACE("GELO_deselect 0x%x 0x%x",gw,go);

   if (go == NULL) return;

   for ( ; ; ) {
      ll = NULL;
      for (l = gw->select; l != NULL; l = CDDR(l)) {
	 if (CAR(GELO_OBJECT,l) == go) break;
	 ll = l;
       };
      if (l == NULL) break;
      if (ll == NULL) {
	 gw->select = CDRF(CDRF(gw->select));
       }
      else {
	 RPLACD(CDR(ll),CDDR(l));
	 RPLACD(CDR(l),NULL);
	 LFREE(l);
       };
    };
};






/************************************************************************/
/*									*/
/*	GELOcorrelate -- find gelo object at given position		*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELOcorrelate(w,x,y)
   ASH_WINDOW w;
   Integer x,y;
{
   register GELO_WINDOW gw;
   register GELO_OBJECT g,gp;

   TRACE("GELOcorrelate 0x%x %d %d",w,x,y);

   gw = (GELO_WINDOW) ASHinq_user_data(w);
   if (gw->object == NULL) return NULL;

   ASHmap(gw->realwindow,x,y,gw->window,&x,&y);

   gp = gw->object->parent;
   gw->object->parent = NULL;
   g = GELO_test_correlate(gw,gw->object,x,y);
   gw->object->parent = gp;

   return g;
};




/************************************************************************/
/*									*/
/*	GELOwindow_resize_object -- resize object in window		*/
/*									*/
/************************************************************************/


void
GELOwindow_resize_object(w,o)
   ASH_WINDOW w;
   GELO_OBJECT o;
{
   register GELO_WINDOW gw;
   register Integer x,y;

   TRACE("GELOwindow_resize_object 0x%x 0x%x",w,o);

   if (w == NULL) gw = NULL;
   else gw = (GELO_WINDOW) ASHinq_user_data(w);

   if (o->parent == NULL) {
      if (gw != NULL) gw->size_valid = FALSE;
    }
   else {
      x = GELO_inq_x_size(o->parent)*WORLD_SIZE_PARAM*2;
      y = GELO_inq_x_size(o->parent)*WORLD_SIZE_PARAM*2;
      PROTECT;
      GELO_size(o,x,y);
      GELOlayout_validate(o->parent,FALSE);
      UNPROTECT;
    };
};





/************************************************************************/
/*									*/
/*	GELOwindow_free -- free gelo object in given window		*/
/*									*/
/************************************************************************/


void
GELOwindow_free(w,o)
   ASH_WINDOW w;
   GELO_OBJECT o;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_free 0x%x 0x%x",w,o);

   if (w == NULL) gw = NULL;
   else {
      gw = (GELO_WINDOW) ASHinq_user_data(w);
      if (gw == NULL) return;
    };

   GELO_object_free(gw,o,NULL);
};





/************************************************************************/
/*									*/
/*	GELOwindow_free_except -- free most of an object tree		*/
/*									*/
/************************************************************************/


void
GELOwindow_free_except(w,o,keep)
   ASH_WINDOW w;
   GELO_OBJECT o;
   GELO_OBJECT keep;
{
   register GELO_WINDOW gw;

   TRACE("GELOwindow_free 0x%x 0x%x 0x%x",w,o,keep);

   if (w == NULL) gw = NULL;
   else gw = (GELO_WINDOW) ASHinq_user_data(w);

   GELO_object_free(gw,o,keep);
};





/************************************************************************/
/*									*/
/*	GELOwindow_replace -- replace gelo object in given window	*/
/*									*/
/************************************************************************/


void
GELOwindow_replace(w,old,new)
   ASH_WINDOW w;
   GELO_OBJECT old;
   GELO_OBJECT new;
{
   register GELO_WINDOW gw;
   register GELO_OBJECT p,np;

   TRACE("GELOwindow_replace 0x%x 0x%x 0x%x",w,old,new);

   gw = (GELO_WINDOW) ASHinq_user_data(w);
   if (gw == NULL) return;

   p = GELOinq_parent(old);

   if (!ASHlock(gw->window)) return;

   gw->size_valid = FALSE;
   np = GELO_replace_object(old,new);

   if (gw->object == old || p == NULL) {
      GELOwindow_draw(w,new);
    }
   else {
      redraw_object(gw,np);
    };

   ASHunlock(gw->window);
};





/************************************************************************/
/*									*/
/*	GELOcompare -- compare two gelo trees				*/
/*									*/
/************************************************************************/


int
GELOcompare(old,new,oldp,newp)
   GELO_OBJECT old,new;
   GELO_OBJECT * oldp;
   GELO_OBJECT * newp;
{
   GELO_OBJECT o1,o2;
   register Boolean fg;

   TRACE("GELOcompare 0x%x 0x%x",old,new);

   fg = GELO_object_compare(old,new,&o1,&o2);

   if (!fg) {
      if (oldp != NULL) *oldp = o1;
      if (newp != NULL) *newp = o2;
    }
   else {
      *oldp = old;
      *newp = new;
    };

   return fg;
};





/************************************************************************/
/*									*/
/*	GELO_inq_position -- get actual position of object		*/
/*	GELO_scale_position -- scale a given x,y position rel to object */
/*									*/
/************************************************************************/


#define SCALEX(x)	\
  (((((x)-gw->view_lx)*(gw->real_rx-gw->real_lx+1))/ \
     (gw->view_rx-gw->view_lx+1))+gw->real_lx)
#define SCALEY(y)	\
  (((((y)-gw->view_ty)*(gw->real_by-gw->real_ty+1))/ \
     (gw->view_by-gw->view_ty+1))+gw->real_ty)


Boolean
GELO_inq_position(gw,o,lxp,byp,rxp,typ)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer *lxp,*byp,*rxp,*typ;
{
   register Integer lx,by,rx,ty;
   GELO_OBJECT gp;

   ITRACE("GELO_inq_position 0x%x 0x%x",gw,o);

   lx = GELO_inq_absolute_x_position(o);
   ty = GELO_inq_absolute_y_position(o);
   rx = lx + GELO_inq_x_size(o) - 1;
   by = ty + GELO_inq_y_size(o) - 1;

   *lxp = SCALEX(lx);
   *rxp = SCALEX(rx);
   *typ = SCALEY(ty);
   *byp = SCALEY(by);

   if (rx < gw->view_lx || lx > gw->view_rx ||
	  ty > gw->view_by || by < gw->view_ty)
      return FALSE;

   for (gp = o->parent; gp != NULL; gp = gp->parent) {
      if (gp->too_small) return FALSE;
    };

   return TRUE;
};





Boolean
GELO_scale_position(gw,o,x,y,nxp,nyp)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer x,y;
   Integer *nxp,*nyp;
{
   ITRACE("GELO_scale_position 0x%x 0x%x %d %d",gw,o,x,y);

   x += GELO_inq_absolute_x_position(o);
   y += GELO_inq_absolute_y_position(o);

   *nxp = SCALEX(x);
   *nyp = SCALEY(y);

   if (x < gw->view_lx || x > gw->view_rx ||
	  y > gw->view_by || y < gw->view_ty)
      return FALSE;

   return TRUE;
};





/************************************************************************/
/*									*/
/*	gelo_refresh -- refresh function for a gelo window		*/
/*									*/
/************************************************************************/


static void
gelo_refresh(w)
   ASH_WINDOW w;
{
   register GELO_WINDOW gw;

   ITRACE("gelo_refresh 0x%x",w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   if (gw == NULL) return;

   GELOwindow_redraw(w);
};





/************************************************************************/
/*									*/
/*	gelo_control -- handle control messages for real window 	*/
/*									*/
/************************************************************************/


static int
gelo_control(msg,w)
   String msg;
   ASH_WINDOW w;
{
   register GELO_WINDOW gw;

   ITRACE("gelo_control %s 0x%x",msg,w);

   gw = (GELO_WINDOW) ASHinq_user_data(w);

   if (gw == NULL) return ASH_CONTROL_REJECT;

   if (STREQL(msg,"ASH$RESIZE")) {
      resize_gelo_window(gw);
    }
   else if (STREQL(msg,"ASH$REMOVE")) {
      free_gelo_window(gw);
    };

   return ASH_CONTROL_REJECT;
};





/************************************************************************/
/*									*/
/*	resize_gelo_window -- handle resizing of window 		*/
/*									*/
/************************************************************************/


static void
resize_gelo_window(gw)
   GELO_WINDOW gw;
{
   Integer lx,by,rx,ty;

   DTRACE("resize_gelo_window 0x%x",gw);

   ASHinq_size(gw->realwindow,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   ASHresize(gw->window,lx,by,0,abs(ty-by),abs(rx-lx),0);

   ASHinq_size(gw->window,ASH_SIZE_WINDOW,&lx,&by,&rx,&ty);

   gw->real_lx = lx;
   gw->real_by = by;
   gw->real_rx = rx;
   gw->real_ty = ty;

   gw->view_lx = lx;
   gw->view_by = by;
   gw->view_rx = rx;
   gw->view_ty = ty;

   gw->world_x = rx+1;
   gw->world_y = by+1;

   gw->size_valid = FALSE;
};





/************************************************************************/
/*									*/
/*	free_gelo_window -- handle window disappearing			*/
/*									*/
/************************************************************************/


static void
free_gelo_window(gw)
   GELO_WINDOW gw;
{
   DTRACE("free_gelo_window 0x%x",gw);

   if (gw->window != NULL && gw->object != NULL) {
      GELO_object_free(gw,gw->object,NULL);
      gw->object = NULL;
    };

   gw->window = NULL;
   gw->realwindow = NULL;
};





/************************************************************************/
/*									*/
/*	redraw_object -- redraw component of diagram			*/
/*									*/
/************************************************************************/


static void
redraw_object(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   Integer lx,by,rx,ty;
   Integer xsz,ysz;

   DTRACE("redraw_object 0x%x 0x%x",gw,o);

   GELO_inq_position(gw,o,&lx,&by,&rx,&ty);

   ASHclear_box(gw->window,lx,by,rx,ty);

   xsz = GELO_inq_x_size(o);
   ysz = GELO_inq_y_size(o);
   GELO_size(o,xsz,ysz);
   GELO_set_x_size(o,xsz);
   GELO_set_y_size(o,ysz);
   gw->size_valid = TRUE;

   GELO_layout(gw,o);
   GELO_draw(gw,o,GELO_DRAW_NORM);
};





/* end of gelowin.c */
