/************************************************************************/
/*									*/
/*		gelotiled.c						*/
/*									*/
/*	TILED flavor object routines for GELO				*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"






/************************************************************************/
/*									*/
/*	Local attributes						*/
/*									*/
/************************************************************************/


#define GELO_TILED_ATTR_NUMXPOS 	4
#define GELO_TILED_ATTR_XPOS		5
#define GELO_TILED_ATTR_NUMYPOS 	6
#define GELO_TILED_ATTR_YPOS		7





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_COMPONENTS		512
#define MIN_SIZE		5




/************************************************************************/
/*									*/
/*	Access macro definitions					*/
/*									*/
/************************************************************************/


#define BOX(o)		((Integer) ((o)->attributes[GELO_TILED_ATTR_BOX]))
#define COMPONENTS(o)	((Sequence) ((o)->attributes[GELO_TILED_ATTR_COMPONENTS]))
#define SET_COMPONENTS(o,v)   ((o)->attributes[GELO_TILED_ATTR_COMPONENTS] = (Universal) (v))
#define CONSTRAINTS(o)	((Sequence) ((o)->attributes[GELO_TILED_ATTR_CONSTRAINTS]))
#define SET_CONSTRAINTS(o,v)   ((o)->attributes[GELO_TILED_ATTR_CONSTRAINTS] = (Universal) (v))
#define CONNECTS(o)	((Sequence) ((o)->attributes[GELO_TILED_ATTR_CONNECTS]))
#define SET_CONNECTS(o,v)   ((o)->attributes[GELO_TILED_ATTR_CONNECTS] = (Universal) (v))
#define NUMXPOS(o)	((Integer) ((o)->attributes[GELO_TILED_ATTR_NUMXPOS]))
#define XPOS(o) 	((Integer *) ((o)->attributes[GELO_TILED_ATTR_XPOS]))
#define SETXPOS(o,x)	((o)->attributes[GELO_TILED_ATTR_XPOS] = ((Integer)(x)))
#define NUMYPOS(o)	((Integer) ((o)->attributes[GELO_TILED_ATTR_NUMYPOS]))
#define YPOS(o) 	((Integer *) ((o)->attributes[GELO_TILED_ATTR_YPOS]))
#define SETYPOS(o,x)	((o)->attributes[GELO_TILED_ATTR_YPOS] = ((Integer)(x)))





/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


typedef struct _COMPONENT {
   GELO_OBJECT	object;
   Integer lx,by,rx,ty;
   float   x_exp,y_exp;
   float   shrinkage;
} COMPONENT;




typedef struct _CONSTRAINT {
   GELO_OBJECT	from;
   GELO_OBJECT	to;
   Boolean	from_y,to_y;
   float	mult,add;
} CONSTRAINT;





/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	GELO_ATTR_DATA_B	tiled_attrs[] = {
   { "BOX" },
   { "COMPONENTS" },
   { "CONSTRAINTS" },
   { "CONNECTS" },
   { "NUMXPOS" },
   { "XPOS" },
   { "NUMYPOS" },
   { "YPOS" },
   { NULL }
};



/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	gelo_tiled_size();
static	void		gelo_tiled_layout();
static	void		gelo_tiled_draw();
static	void		setup_tiling();
static	Integer 	add_coord();
static	Boolean 	compute_size();
static	void		find_expansion();
static	void		set_positions();
static	void		set_components();
static	Integer 	find_index();
static	void		gelo_tiled_free();
static	GELO_OBJECT	gelo_tiled_replace();
static	Boolean 	gelo_tiled_compare();





/************************************************************************/
/*									*/
/*	GELO_tiled_init -- module initialization			*/
/*									*/
/************************************************************************/


void
GELO_tiled_init()
{
   ITRACE("GELO_tiled_init");

   GELO_flavor_define(GELO_FLAVOR_TILED,
			 "TILED",
			 tiled_attrs,
			 gelo_tiled_size,
			 gelo_tiled_layout,
			 gelo_tiled_draw,
			 gelo_tiled_free,gelo_tiled_replace,NULL,NULL,NULL,
			 gelo_tiled_compare);
};





/************************************************************************/
/*									*/
/*	GELOtiled_component -- define a new component			*/
/*									*/
/************************************************************************/


int
GELOtiled_component(o,lx,by,rx,ty,xe,ye,sh)
   GELO_OBJECT o;
   Integer lx,by,rx,ty;
   float xe,ye;
   float sh;
{
   register COMPONENT * c;

   c = (COMPONENT *) calloc(1,sizeof(COMPONENT));

   c->object = o;
   c->lx = lx;
   c->by = by;
   c->rx = rx;
   c->ty = ty;
   c->x_exp = xe;
   c->y_exp = ye;
   c->shrinkage = sh;

   TRACE("GELOtiled_component 0x%x %d %d %d %d %f %f %f => 0x%x",
	    o,lx,by,rx,ty,xe,ye,sh,c);

   return ((int) c);
};





/************************************************************************/
/*									*/
/*	GELOtiled_constraint -- define a new constraint 		*/
/*									*/
/************************************************************************/


int
GELOtiled_constraint(from,to,fromy,toy,mult,add)
   GELO_OBJECT from;
   GELO_OBJECT to;
   Boolean fromy,toy;
   Float mult,add;
{
   register CONSTRAINT * c;

   c = (CONSTRAINT *) calloc(1,sizeof(CONSTRAINT));

   c->from = from;
   c->to = to;
   c->from_y = fromy;
   c->to_y = toy;
   c->mult = mult;
   c->add = add;

   TRACE("GELOtiled_constraint 0x%x 0x%x %d %d %f %f => 0x%x",
	    from,to,fromy,toy,mult,add,c);

   return ((int) c);
};





/************************************************************************/
/*									*/
/*	gelo_tiled_size -- size a tiled object				*/
/*									*/
/************************************************************************/


#define SHRINK(x)	((int)((x)*c->shrinkage))


static Boolean
gelo_tiled_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   register Sequence l;
   register COMPONENT * c;
   register Boolean fg;

   ITRACE("gelo_tiled_size 0x%x",o);

   if (mxx < MIN_SIZE || mxy < MIN_SIZE) return FALSE;

   setup_tiling(o);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      fg = GELO_size(c->object,SHRINK(mxx),SHRINK(mxy));
      if (!fg) return FALSE;
    };

   fg = compute_size(o,mxx,mxy);

   return fg;
};





/************************************************************************/
/*									*/
/*	gelo_tiled_layout -- layout a tiled object			*/
/*									*/
/************************************************************************/


static void
gelo_tiled_layout(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   Float * solution;
   Integer * nx;
   Integer * ny;
   Integer ln,ln1;

   ITRACE("gelo_tiled_layout 0x%x 0x%x",gw,o);

   if (GELO_inq_x_size(o) == 0) return;

   ln = (LENGTH(COMPONENTS(o))+1)*2;
   ln1 = NUMXPOS(o)+NUMYPOS(o)+2;
   solution = (Float *) alloca(ln*sizeof(Float));
   nx = (Integer *) alloca(ln1*sizeof(Integer));
   ny = (Integer *) alloca(ln1*sizeof(Integer));

   find_expansion(o,solution);

   set_positions(o,solution,nx,ny);

   set_components(gw,o,nx,ny);
};





/************************************************************************/
/*									*/
/*	gelo_tiled_draw -- draw a tiled object				*/
/*									*/
/************************************************************************/


static void
gelo_tiled_draw(gw,o,hifg)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_DRAW_FLAGS hifg;
{
   register COMPONENT * c;
   register Sequence l;
   Integer lx,by,rx,ty;

   ITRACE("gelo_tiled_draw 0x%x 0x%x",gw,o);

   if (!GELO_inq_position(gw,o,&lx,&by,&rx,&ty)) return;

   if (BOX(o) != 0) {
      ASHbox(gw->window,lx,by,rx,ty);
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      GELO_draw(gw,c->object,hifg);
    };

   GELO_draw_connections(gw,CONNECTS(o),GELO_DRAW_NORM);
};





/************************************************************************/
/*									*/
/*	setup_tiling -- setup the tiling for object			*/
/*									*/
/************************************************************************/


static void
setup_tiling(o)
   GELO_OBJECT o;
{
   register Integer i;
   register Integer nx,ny;
   register COMPONENT * c;
   register Sequence l;
   Integer * xpos;
   Integer * ypos;

   i = (LENGTH(COMPONENTS(o)) + 1) * 2;
   xpos = (Integer *) alloca(i*sizeof(Integer));
   ypos = (Integer *) alloca(i*sizeof(Integer));

   nx = ny = 0;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      nx = add_coord(c->lx,nx,xpos);
      nx = add_coord(c->rx,nx,xpos);
      ny = add_coord(c->ty,ny,ypos);
      ny = add_coord(c->by,ny,ypos);
    };

   if (XPOS(o) == NULL || nx > NUMXPOS(o)) {
      if (XPOS(o) != NULL) free(XPOS(o));
      o->attributes[GELO_TILED_ATTR_NUMXPOS] = nx;
      SETXPOS(o,calloc(nx,sizeof(int)));
      for (i = 0; i < nx; ++i) XPOS(o)[i] = xpos[i];
    };

   if (YPOS(o) == NULL || ny > NUMYPOS(o)) {
      if (YPOS(o) != NULL) free(YPOS(o));
      o->attributes[GELO_TILED_ATTR_NUMYPOS] = ny;
      SETYPOS(o,calloc(ny,sizeof(int)));
      for (i = 0; i < ny; ++i) YPOS(o)[i] = ypos[i];
    };
};





/************************************************************************/
/*									*/
/*	add_coord -- add a coordinate to coordinate list		*/
/*									*/
/************************************************************************/


static Integer
add_coord(x,n,p)
   Integer x;
   Integer n;
   Integer p[];
{
   register Integer mn,mx,i;

   DTRACE("add_coord %d %d",x,n);

   mn = 0;
   mx = n-1;
   while (mn <= mx) {
      i = (mn+mx)/2;
      if (x == p[i]) return n;
      else if (x < p[i]) mx = i-1;
      else mn = i+1;
    };

   for (i = n; i > mn; --i) p[i] = p[i-1];

   p[mn] = x;

   return n+1;
};





/************************************************************************/
/*									*/
/*	compute_size -- compute size for tiled object			*/
/*									*/
/************************************************************************/


static Boolean
compute_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   register Sequence l;
   register COMPONENT * c;
   register Integer i;
   register Integer i0,i1,sz,msz;
   Integer mnx,mny,xsz,ysz;
   Integer * xszs;
   Integer * yszs;
   Integer * xmns;
   Integer * ymns;

   DTRACE("compute_size 0x%x %d %d",o,mxx,mxy);

   i = NUMXPOS(o)+NUMYPOS(o)+2;
   xszs = (Integer *) alloca(i*sizeof(Integer));
   yszs = (Integer *) alloca(i*sizeof(Integer));
   xmns = (Integer *) alloca(i*sizeof(Integer));
   ymns = (Integer *) alloca(i*sizeof(Integer));

   i0 = (BOX(o) ? 2 : 0);

   for (i = 0; i < NUMXPOS(o); ++i) {
      yszs[i] = i0;
      ymns[i] = i0;
    };
   for (i = 0; i < NUMYPOS(o); ++i) {
      xszs[i] = i0;
      xmns[i] = i0;
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      i0 = find_index(c->ty,NUMYPOS(o),YPOS(o));
      i1 = find_index(c->by,NUMYPOS(o),YPOS(o));
      sz = GELO_inq_x_size(c->object);
      msz = GELO_inq_x_min_size(c->object);
      for (i = i0; i < i1; ++i) {
	 xszs[i] += sz;
	 xmns[i] += msz;
       };
      i0 = find_index(c->lx,NUMXPOS(o),XPOS(o));
      i1 = find_index(c->rx,NUMXPOS(o),XPOS(o));
      sz = GELO_inq_y_size(c->object);
      msz = GELO_inq_y_min_size(c->object);
      for (i = i0; i < i1; ++i) {
	 yszs[i] += sz;
	 ymns[i] += msz;
       };
    };

   xsz = xszs[0];
   ysz = yszs[0];
   mnx = xmns[0];
   mny = ymns[0];

   for (i = 1; i < NUMXPOS(o); ++i) {
      ysz = MAX(ysz,yszs[i]);
      mny = MAX(mny,ymns[i]);
    };

   for (i = 1; i < NUMYPOS(o); ++i) {
      xsz = MAX(xsz,xszs[i]);
      mnx = MAX(mnx,xmns[i]);
    };

   if (mnx > mxx || mny > mxy) return FALSE;
   if (xsz > mxx) {
      ysz = (ysz*mxx)/xsz;
      xsz = mxx;
    };
   if (ysz > mxy) {
      xsz = (xsz*mxy)/ysz;
      ysz = mxy;
    };

   GELO_set_x_size(o,xsz);
   GELO_set_y_size(o,ysz);
   GELO_set_x_min_size(o,mnx);
   GELO_set_y_min_size(o,mny);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	find_expansion -- solve for sizes of components 		*/
/*									*/
/************************************************************************/


#define mat(x,y) matp[x*ncol+y]

static void
find_expansion(o,solution)
   GELO_OBJECT o;
   Float solution[];
{
   register Integer i,j;
   register COMPONENT * c;
   register Sequence l,la;
   register CONSTRAINT * cnst;
   Float * rhs;
   Integer ncom,nrow,ncol,nuse;
   Integer mn,df,i0,i1,sz;
   Float z;
   Boolean badx,bady;
   Integer rptct;
   float * matp;

   DTRACE("find_expansion 0x%x",o);

   ncom = LENGTH(COMPONENTS(o));
   ncol = ncom*2;
   nrow = NUMXPOS(o)+NUMYPOS(o)-2;
   i = nrow + 2 + LENGTH(CONSTRAINTS(o));
   matp = (float *) alloca(ncol*i*sizeof(float));
   rhs = (Float *) alloca(i*sizeof(Float));

   badx = FALSE;
   bady = FALSE;
   rptct = 0;

   do {
      nrow = NUMXPOS(o)+NUMYPOS(o)-2;
      i0 = (BOX(o) ? 1 : 0);
      for (i = 0; i < nrow; ++i) {
	 if (i == 0) rhs[i] = GELO_inq_x_size(o)-i0;
	 else if (i == NUMYPOS(o)-1) rhs[i] = GELO_inq_y_size(o)-i0;
	 else rhs[i] = rhs[i-1];
	 for (j = 0; j < ncol; ++j) {
	    mat(i,j) = 0.0;
	  };
       };

      j = 0;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 i0 = find_index(c->ty,NUMYPOS(o),YPOS(o));
	 i1 = find_index(c->by,NUMYPOS(o),YPOS(o));
	 mn = GELO_inq_x_min_size(c->object);
	 df = GELO_inq_x_size(c->object);
	 z = c->x_exp*(df-mn)*GELOinq_priority_x(c->object);
	 for (i = i0; i < i1; ++i) {
	    rhs[i] -= mn;
	    mat(i,j) = z;
	  };
	 i0 = NUMYPOS(o)-1+find_index(c->lx,NUMXPOS(o),XPOS(o));
	 i1 = NUMYPOS(o)-1+find_index(c->rx,NUMXPOS(o),XPOS(o));
	 mn = GELO_inq_y_min_size(c->object);
	 df = GELO_inq_y_size(c->object);
	 z = c->y_exp*(df-mn)*GELOinq_priority_y(c->object);
	 for (i = i0; i < i1; ++i) {
	    rhs[i] -= mn;
	    mat(i,j+1) = z;
	  };
	 j += 2;
       };

      nuse = nrow;
      forin (cnst,CONSTRAINT *,l,CONSTRAINTS(o)) {
	 for (i = 0; i < ncol; ++i) mat(nrow,i) = 0.0;
	 rhs[nrow] = cnst->add;
	 i0 = -1;
	 i1 = -1;
	 j = 0;
	 forin (c,COMPONENT *,la,COMPONENTS(o)) {
	    if (c->object == cnst->from) {
	       if (cnst->from_y) {
		  i0 = 2*j+1;
		  mn = GELO_inq_y_min_size(c->object);
		  sz = GELO_inq_y_size(c->object);
		  rhs[nrow] -= mn;
		  mat(nrow,i0) = c->y_exp * (sz - mn) * GELOinq_priority_y(c->object);
		}
	       else {
		  i0 = 2*j;
		  mn = GELO_inq_x_min_size(c->object);
		  sz = GELO_inq_x_size(c->object);
		  rhs[nrow] -= mn;
		  mat(nrow,i0) = c->x_exp * (sz - mn) * GELOinq_priority_x(c->object);
		};
	     };
	    if (c->object == cnst->to) {
	       if (cnst->from_y) {
		  i1 = 2*j+1;
		  mn = GELO_inq_y_min_size(c->object);
		  sz = GELO_inq_y_size(c->object);
		  rhs[nrow] += mn;
		  mat(nrow,i1) = -(cnst->mult) * c->y_exp * (sz - mn) * GELOinq_priority_y(c->object);
		}
	       else {
		  i1 = 2*j;
		  mn = GELO_inq_x_min_size(c->object);
		  sz = GELO_inq_x_size(c->object);
		  rhs[nrow] += mn;
		  mat(nrow,i1) = -(cnst->mult) * c->x_exp * (sz - mn) * GELOinq_priority_x(c->object);
		};
	     };
	    ++j;
	  };
	 if (i0 >= 0 && i1 >= 0 && i0 != i1) nrow++;
       };

      GELO_solve_system(nrow,ncol,nuse,matp,rhs,solution);

      if (rptct == 1) rptct = 2;

      for (i = 0; i < ncol; ++i) {
	 if (solution[i] < 0.0) {
	    if (rptct > 1) {
	       solution[i] = 0.0;
	       rptct = 3;
	     }
	    else {
	       if (i & 1) bady = TRUE;
	       else badx = TRUE;
	       rptct = 1;
	     };
	  };
       };

      if (rptct == 1) {
	 i = 0;
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    if (GELOinq_use_min_size(c->object)) {
	       ++i;
	       GELOuse_min_size(c->object,FALSE);
	     };
	    if (badx && GELOinq_use_default_x(c->object)) {
	       ++i;
	       GELOuse_default_x(c->object,FALSE);
	     };
	    if (bady && GELOinq_use_default_y(c->object)) {
	       ++i;
	       GELOuse_default_y(c->object,FALSE);
	     };
	  };
	 if (i == 0) rptct = 3;
      };
    }
   while (rptct == 1);

   if (rptct == 3) {
      GELO_set_too_small(o);
    };
};





/************************************************************************/
/*									*/
/*	set_positions -- set coordinate positions from solution 	*/
/*									*/
/************************************************************************/


static void
set_positions(o,solution,nx,ny)
   GELO_OBJECT o;
   Float solution[];
   Integer nx[];
   Integer ny[];
{
   register Integer i,j;
   register COMPONENT * c;
   register Sequence l;
   Integer i0,i1,ct;
   Integer x0,x1,z;
   Integer maxx,maxy;

   DTRACE("set_positions 0x%x",o);

   for (i = 0; i < NUMXPOS(o); ++i) nx[i] = -1;
   for (i = 0; i < NUMYPOS(o); ++i) ny[i] = -1;

   ct = NUMXPOS(o)+NUMYPOS(o)-2;

   if (BOX(o)) {
      nx[0] = 1;
      ny[0] = 1;
      maxx = GELO_inq_x_size(o)-1;
      maxy = GELO_inq_y_size(o)-1;
    }
   else {
      nx[0] = 0;
      ny[0] = 0;
      maxx = GELO_inq_x_size(o);
      maxy = GELO_inq_y_size(o);
     };

   while (ct > 0) {
      j = 0;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 i0 = find_index(c->lx,NUMXPOS(o),XPOS(o));
	 if (nx[i0] >= 0) {
	    i1 = find_index(c->rx,NUMXPOS(o),XPOS(o));
	    if (nx[i1] < 0) {
	       x0 = GELO_inq_x_size(c->object);
	       x1 = GELO_inq_x_min_size(c->object);
	       z = solution[j]*c->x_exp*(x0-x1)*GELOinq_priority_x(c->object);
	       nx[i1] = nx[i0] + z + x1;
	       if (nx[i1] >= maxx) nx[i1] = maxx;
	       --ct;
	     };
	  };
	 i0 = find_index(c->ty,NUMYPOS(o),YPOS(o));
	 if (ny[i0] >= 0) {
	    i1 = find_index(c->by,NUMYPOS(o),YPOS(o));
	    if (ny[i1] < 0) {
	       x0 = GELO_inq_y_size(c->object);
	       x1 = GELO_inq_y_min_size(c->object);
	       z = solution[j+1]*c->y_exp*(x0-x1)*GELOinq_priority_y(c->object);
	       ny[i1] = ny[i0] + z + x1;
	       if (ny[i1] >= maxy) ny[i1] = maxy;
	       --ct;
	     };
	  };
	 if (ct == 0) break;
	 j += 2;
       };
    };

   i1 = NUMXPOS(o);
   i0 = maxx - nx[i1-1];
   if (i0 > 0) for (i = 1; i < i1; ++i) nx[i] += i0*i/(i1-1);
   i1 = NUMYPOS(o);
   i0 = maxy - ny[i1-1];
   if (i0 > 0) for (i = 1; i < i1; ++i) ny[i] += i0*i/(i1-1);
};





/************************************************************************/
/*									*/
/*	set_components -- set size and position for components		*/
/*									*/
/************************************************************************/


static void
set_components(gw,o,nx,ny)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer nx[];
   Integer ny[];
{
   register COMPONENT * c;
   register Sequence l;
   register Integer i0,i1;

   DTRACE("set_components 0x%x",o);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      i0 = find_index(c->lx,NUMXPOS(o),XPOS(o));
      i1 = find_index(c->rx,NUMXPOS(o),XPOS(o));
      GELO_set_x_size(c->object,nx[i1]-nx[i0]);
      GELO_set_x_position(c->object,nx[i0]);

      i0 = find_index(c->ty,NUMYPOS(o),YPOS(o));
      i1 = find_index(c->by,NUMYPOS(o),YPOS(o));
      GELO_set_y_size(c->object,ny[i1]-ny[i0]);
      GELO_set_y_position(c->object,ny[i0]);

      GELO_layout(gw,c->object);
    };
};





/************************************************************************/
/*									*/
/*	find_index -- find index to a given coordinate			*/
/*									*/
/************************************************************************/


static Integer
find_index(x,n,p)
   Integer x;
   Integer n;
   Integer p[];
{
   register Integer mn,mx,i;

   DTRACE("find_index %d %d",x,n);

   mn = 0;
   mx = n-1;
   while (mn <= mx) {
      i = (mn+mx)/2;
      if (x == p[i]) return i;
      else if (x < p[i]) mx = i-1;
      else mn = i+1;
    };

   FATAL("Tiled index not found");

   return 0;
};





/************************************************************************/
/*									*/
/*	gelo_tiled_free -- free a tiled object				*/
/*									*/
/************************************************************************/


static void
gelo_tiled_free(gw,o,keep)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_OBJECT keep;
{
   register Sequence l;
   register COMPONENT * c;

   DTRACE("gelo_tiled_free 0x%x 0x%x 0x%x",gw,o,keep);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      GELO_object_free(gw,c->object,keep);
      c->object = NULL;
      free(c);
    };

   LFREE(COMPONENTS(o));
   SET_COMPONENTS(o,NULL);

   GELO_free_connects(gw,CONNECTS(o),keep);

   LFREE(CONNECTS(o));
   SET_CONNECTS(o,NULL);
};





/************************************************************************/
/*									*/
/*	gelo_tiled_replace -- replace component of object		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
gelo_tiled_replace(o,old,new)
   GELO_OBJECT o;
   GELO_OBJECT old;
   GELO_OBJECT new;
{
   register Sequence l,la;
   register COMPONENT * c;
   register CONSTRAINT * cnst;
   register GELO_OBJECT r;

   DTRACE("gelo_tiled_replace 0x%x 0x%x 0x%x",o,old,new);

   r = new;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->object == old) {
	 c->object = new;
	 GELO_connect_replace(CONNECTS(o),old,new);
	 forin (cnst,CONSTRAINT *,la,CONSTRAINTS(o)) {
	    if (cnst->from == old) cnst->from = new;
	    if (cnst->to == old) cnst->to = new;
	  };
	 break;
       };
    };

   if (EMPTY(l)) {
      if (MEMQ(old,CONNECTS(o))) {
	 r = o;
	 SET_CONNECTS(o,REMOB(old,CONNECTS(o)));
	 SET_CONNECTS(o,APPEND(new,CONNECTS(o)));
       };
    };

   return r;
};





/************************************************************************/
/*									*/
/*	gelo_tiled_compare -- compare two tiled objects 		*/
/*									*/
/************************************************************************/


static Boolean
gelo_tiled_compare(o1,o2,o1p,o2p)
   GELO_OBJECT o1;
   GELO_OBJECT o2;
   GELO_OBJECT * o1p;
   GELO_OBJECT * o2p;
{
   register Sequence l,la,lb;
   register COMPONENT * c0, * c1;
   register CONSTRAINT * cnsta, * cnstb;
   register Boolean fg,nfg;
   register Integer i;
   GELO_OBJECT o1a,o2a;

   DTRACE("gelo_tiled_compare 0x%x 0x%x",o1,o2);

   *o1p = NULL;
   *o2p = NULL;

   if (BOX(o1) != BOX(o2)) fg = FALSE;
   else if (LENGTH(CONSTRAINTS(o1)) != LENGTH(CONSTRAINTS(o2))) fg = FALSE;
   else if (LENGTH(COMPONENTS(o1)) != LENGTH(COMPONENTS(o2))) fg = FALSE;
   else fg = TRUE;

   if (fg) {
      la = CONSTRAINTS(o2);
      forin (cnsta,CONSTRAINT *,l,CONSTRAINTS(o1)) {
	 cnstb = CAR(CONSTRAINT *,la);
	 la = CDR(la);
	 if (cnsta->from_y != cnstb->from_y ||
		cnsta->to_y != cnstb->to_y ||
		cnsta->mult != cnstb->mult ||
		cnsta->add != cnstb->add) {
	    fg = FALSE;
	    break;
	  };
	 i = 1;
	 forin (c0,COMPONENT *,lb,COMPONENTS(o1)) {
	    if (c0->object == cnsta->from) {
	       c1 = NTH(COMPONENT *,COMPONENTS(o2),i);
	       if (c1->object != cnstb->from) {
		  fg = FALSE;
		  break;
		};
	     };
	    if (c0->object == cnsta->to) {
	       c1 = NTH(COMPONENT *,COMPONENTS(o2),i);
	       if (c1->object != cnstb->to) {
		  fg = FALSE;
		  break;
		};
	     };
	    ++i;
	  };
       };
    };

   if (fg) {
      la = COMPONENTS(o2);
      forin (c0,COMPONENT *,l,COMPONENTS(o1)) {
	 c1 = CAR(COMPONENT *,la);
	 la = CDR(la);
	 if (c0->lx != c1->lx || c0->by != c1->by ||
		c0->rx != c1->rx || c0->ty != c1->ty ||
		c0->x_exp != c1->x_exp || c0->y_exp != c1->y_exp ||
		c0->shrinkage != c1->shrinkage) {
	    fg = FALSE;
	    break;
	  };
       };
    };

   if (!fg) {
      *o1p = o1;
      *o2p = o2;
    }
   else {
      la = COMPONENTS(o2);
      forin (c0,COMPONENT *,l,COMPONENTS(o1)) {
	 c1 = CAR(COMPONENT *,la);
	 la = CDR(la);
	 nfg = GELO_object_compare(c0->object,c1->object,&o1a,&o2a);
	 if (fg) {
	    if (!nfg) {
	       *o1p = o1a;
	       *o2p = o2a;
	       fg = FALSE;
	     };
	  }
	 else if (!nfg) {
	    *o1p = o1;
	    *o2p = o2;
	    break;
	  };
       };
    };

   return fg;
};





/* end of gelotiled.c */


