/************************************************************************/
/*									*/
/*		geloobject.c						*/
/*									*/
/*	Object processing for the GELO general layout package		*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"






/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define MAX_FLAVORS	8





/************************************************************************/
/*									*/
/*	Local type definitions						*/
/*									*/
/************************************************************************/


typedef struct FLAVOR_DATA {
   String	name;
   Integer	num_attr;
   Function_Ptr size_rtn;
   Function_Ptr layout_rtn;
   Function_Ptr draw_rtn;
   Function_Ptr free_rtn;
   Function_Ptr replace_rtn;
   Function_Ptr corr_rtn;
   Function_Ptr hilite_rtn;
   Function_Ptr offset_rtn;
   Function_Ptr compare_rtn;
   GELO_ATTR_DATA attrs;
} FLAVOR_DATA;




/************************************************************************/
/*									*/
/*	Table definitions						*/
/*									*/
/************************************************************************/


static	FLAVOR_DATA	flavor_data[MAX_FLAVORS];

static	Function_Ptr	notify_fct;

typedef struct COLOR_DATA {
   String name;
   Integer value;
} COLOR_DATA;

static	COLOR_DATA	colors[] = {
   { "white", 0 },
   { "LightBlue", 0 },
   { "green", 0 },
   { "yellow", 0 },
   { "thistle", 0 },
   { "red", 0 },
   { "orange", 0 },
   { 0,0 }
};




/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	GELO_object_init -- module initialization			*/
/*									*/
/************************************************************************/


void
GELO_object_init()
{
   Integer i,dp;
   ASH_WINDOW top;

   ITRACE("GELO_object_init");

   for (i = 0; i < MAX_FLAVORS; ++i) {
      flavor_data[i].num_attr = -1;
    };

   notify_fct = NULL;
   top = ASHinq_top();
   dp = ASHinq_configuration_depth(top);

   for (i = 0; colors[i].name != NULL; ++i) {
      if (dp == 1 && STRNEQ(colors[i].name,"white"))
	 colors[i].value = ASHlookup_color(top,"black");
      else
	 colors[i].value = ASHlookup_color(top,colors[i].name);
    };
};





/************************************************************************/
/*									*/
/*	GELOdefine -- define a new object of the given flavor		*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELOdefine(f)
   GELO_FLAVOR f;
{
   register GELO_OBJECT g;

   ENTER("GELOdefine %d",f);

   g = (GELO_OBJECT) calloc(1,sizeof(GELO_OBJECT_B));

   g->flavor = f;
   g->parent = NULL;
   g->son = NULL;
   g->brother = NULL;
   g->layout_valid = FALSE;
   g->too_small = FALSE;
   g->use_min = FALSE;
   g->use_default_x = FALSE;
   g->use_default_y = FALSE;
   g->zero_size_x = FALSE;
   g->zero_size_y = FALSE;
   g->priority_x = 1.0;
   g->priority_y = 1.0;
   g->x_size = 1;
   g->y_size = 1;
   g->x_min_size = 1;
   g->y_min_size = 1;
   g->x_pos = 0;
   g->y_pos = 0;

   g->attributes = (Universal *) calloc(flavor_data[f].num_attr,sizeof(Universal));
   if (g->attributes == NULL) {
      FATAL("Out of memory");
    };

   g->object = NULL;
   g->owner = NULL;
   g->contents = NULL;
   g->source.object = NULL;
   g->source.field = NULL;
   g->source.index = NULL;
   g->source.type = NULL;

   TRACE("\t=> 0x%x",g);

   return g;
};





/************************************************************************/
/*									*/
/*	GELOset_nofify_fct -- set function for notifications		*/
/*									*/
/************************************************************************/


void
GELOset_notify_fct(fct)
   Function_Ptr fct;
{
   ENTER("GELOset_notify_fct 0x%x",fct);

   notify_fct = fct;
};





/************************************************************************/
/*									*/
/*	GELOset_attribute -- set attribute to given value		*/
/*	GELOadd_attribute -- add to a list attribute			*/
/*	GELOinq_attribute -- return attribute				*/
/*									*/
/************************************************************************/


void
GELOset_attribute(o,a,v)
   GELO_OBJECT o;
   GELO_ATTRIBUTE a;
   Universal v;
{
   TRACE("GELOset_attribute 0x%x 0x%x 0x%x",o,a,v);

   o->attributes[a] = v;
};





void
GELOadd_attribute(o,a,v)
   GELO_OBJECT o;
   GELO_ATTRIBUTE a;
   Universal v;
{
   TRACE("GELOadd_attribute 0x%x 0x%x 0x%x",o,a,v);

   o->attributes[a] = (Universal) APPEND(v,o->attributes[a]);
};





int
GELOinq_attribute(o,a)
   GELO_OBJECT o;
   GELO_ATTRIBUTE a;
{
   register Universal v;

   v = o->attributes[a];

   TRACE("GELOinq_attribute 0x%x 0x%x => 0x%x",o,a,v);

   return ((int) v);
};





/************************************************************************/
/*									*/
/*	GELOlayout_validate -- validate or invalidate layout		*/
/*									*/
/************************************************************************/


void
GELOlayout_validate(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOlayout_validate 0x%x %d",o,fg);

   o->layout_valid = fg;
};





/************************************************************************/
/*									*/
/*	GELOadd_son -- add son object to parent 			*/
/*									*/
/************************************************************************/


void
GELOadd_son(p,s)
   GELO_OBJECT p;
   GELO_OBJECT s;
{
   register GELO_OBJECT s1;

   TRACE("GELOadd_son 0x%x 0x%x",p,s);

   s->parent = p;
   s->brother = NULL;

   if (p->son == NULL) {
      p->son = s;
    }
   else {
      for (s1 = p->son; s1->brother != NULL; s1 = s1->brother);
      s1->brother = s;
    };
};





/************************************************************************/
/*									*/
/*	GELOset_user_structure -- set association for a GELO_OBJECT	*/
/*	GELOinq_user_structure -- return association for a GELO_OBJECT	*/
/*	GELOset_source -- set source association for a GELO_OBJECT	*/
/*	GELOinq_source_object -- return source object			*/
/*	GELOinq_source_field -- return source field			*/
/*	GELOinq_source_index -- return source index			*/
/*	GELOinq_source_type -- return source type			*/
/*	GELOset_contents -- set content for a GELO_OBJECT		*/
/*	GELOinq_contents -- return contents of an object		*/
/*	GELOset_owner -- set owner for a GELO_OBJECT			*/
/*									*/
/************************************************************************/


void
GELOset_user_structure(o,s)
   GELO_OBJECT o;
   Universal s;
{
   TRACE("GELOset_user_structure 0x%x 0x%x",o,s);

   o->object = s;
};





int
GELOinq_user_structure(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_user_structure 0x%x => 0x%x",o,o->object);

   return (int) o->object;
};






void
GELOset_source(o,so,fld,idx,typ)
   GELO_OBJECT o;
   Universal so;
   Universal fld;
   Universal idx;
   Universal typ;
{
   TRACE("GELOset_source 0x%x 0x%x 0x%x 0x%x",o,so,fld,idx);

   o->source.object = so;
   o->source.field = fld;
   o->source.index = idx;
   o->source.type = typ;
};





int
GELOinq_source_object(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_source_object 0x%x => 0x%x",o,o->source.object);

   return (int) o->source.object;
};






int
GELOinq_source_field(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_source_field 0x%x => 0x%x",o,o->source.field);

   return (int) o->source.field;
};






int
GELOinq_source_index(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_source_index 0x%x => 0x%x",o,o->source.index);

   return (int) o->source.index;
};






int
GELOinq_source_type(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_source_type 0x%x => 0x%x",o,o->source.type);

   return (int) o->source.type;
};






void
GELOset_contents(o,conts)
   GELO_OBJECT o;
   Universal conts;
{
   TRACE("GELOset_contents 0x%x 0x%x",o,conts);

   o->contents = conts;
};





int
GELOinq_contents(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_contents 0x%x",o);

   return (int) o->contents;
};





void
GELOset_owner(o,owner)
   GELO_OBJECT o;
   Universal owner;
{
   TRACE("GELOset_owner 0x%x 0x%x",o,owner);

   o->owner = owner;
};





/************************************************************************/
/*									*/
/*	GELOinq_flavor -- return object flavor				*/
/*									*/
/************************************************************************/


GELO_FLAVOR
GELOinq_flavor(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_flavor 0x%x => 0x%x",o,o->flavor);

   return o->flavor;
};




/************************************************************************/
/*									*/
/*	GELOset_priority_x -- set priority for object in x		*/
/*	GELOinq_priority_x -- return priority for object in x		*/
/*	GELOset_priority_y -- set priority for object in y		*/
/*	GELOinq_priority_y -- return priority for object in y		*/
/*									*/
/************************************************************************/


void
GELOset_priority_x(o,pr)
   GELO_OBJECT o;
   Float pr;
{
   TRACE("GELOset_priority_x 0x%x %f",o,pr);

   o->priority_x = pr;
};





double
GELOinq_priority_x(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_priority_x 0x%x => %f",o,o->priority_x);

   return o->priority_x;
};





void
GELOset_priority_y(o,pr)
   GELO_OBJECT o;
   Float pr;
{
   TRACE("GELOset_priority_y 0x%x %f",o,pr);

   o->priority_y = pr;
};





double
GELOinq_priority_y(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_priority_y 0x%x => %f",o,o->priority_y);

   return o->priority_y;
};





/************************************************************************/
/*									*/
/*	GELOuse_min_size -- use given min size rather than default	*/
/*	GELOinq_use_min_size -- return value of flag			*/
/*	GELOuse_default_x -- force to be size of default in x		*/
/*	GELOinq_use_default_x -- return value of this flag		*/
/*	GELOuse_default_y -- force to be size of default in y		*/
/*	GELOinq_use_default_y -- return value of this flag		*/
/*	GELOuse_zero_size_x -- allow object to be zero size in x	*/
/*	GELOinq_zero_size_x -- return value of this flag		*/
/*	GELOuse_zero_size_y -- allow object to be zero size in y	*/
/*	GELOinq_zero_size_y -- return value of this flag		*/
/*	GELO_set_too_small -- indicate an object is too big		*/
/*									*/
/************************************************************************/


void
GELOuse_min_size(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOuse_min_size 0x%x %d",o,fg);

   o->use_min = fg;
};





int
GELOinq_use_min_size(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_use_min_size 0x%x => %d",o,o->use_min);

   return o->use_min;
};





void
GELOuse_default_x(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOuse_default_x 0x%x %d",o,fg);

   o->use_default_x = fg;
};





int
GELOinq_use_default_x(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_use_default_x 0x%x => %d",o,o->use_default_x);

   return o->use_default_x;
};





void
GELOuse_default_y(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOuse_default_y 0x%x %d",o,fg);

   o->use_default_y = fg;
};





int
GELOinq_use_default_y(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_use_default_y 0x%x => %d",o,o->use_default_y);

   return o->use_default_y;
};





void
GELOuse_zero_size_x(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOuse_zero_size_x 0x%x %d",o,fg);

   o->zero_size_x = fg;
};





int
GELOinq_zero_size_x(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_zero_size_x 0x%x => %d",o,o->zero_size_x);

   return o->zero_size_x;
};





void
GELOuse_zero_size_y(o,fg)
   GELO_OBJECT o;
   Boolean fg;
{
   TRACE("GELOuse_zero_size_y 0x%x %d",o,fg);

   o->zero_size_y = fg;
};





int
GELOinq_zero_size_y(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_zero_size_y 0x%x => %d",o,o->zero_size_y);

   return o->zero_size_y;
};





void
GELO_set_too_small(o)
   GELO_OBJECT o;
{
   ITRACE("GELO_set_too_small 0x%x",o);

   o->too_small = TRUE;
};





/************************************************************************/
/*									*/
/*	GELOinq_parent -- return parent object of given object		*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELOinq_parent(o)
   GELO_OBJECT o;
{
   TRACE("GELOinq_parent 0x%x",o);

   if (o == NULL) return NULL;

   return o->parent;
};





/************************************************************************/
/*									*/
/*	GELO_size -- size an object					*/
/*	GELO_layout -- layout an object 				*/
/*	GELO_relayout -- layout only what hasn't been done yet          */
/*	GELO_draw -- draw an object					*/
/*									*/
/************************************************************************/


int
GELO_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   register Boolean fg;

   ITRACE("GELO_size 0x%x %d %d",o,mxx,mxy);

   o->too_small = FALSE;
   o->layout_valid = FALSE;

   if (mxx <= 0 || mxy <= 0) return FALSE;

   fg = (*(flavor_data[o->flavor].size_rtn))(o,mxx,mxy);

   if (!fg) {
      o->too_small = TRUE;
      fg = (*(flavor_data[GELO_FLAVOR_EMPTY].size_rtn))(o,mxx,mxy);
    };

   return fg;
};





void
GELO_layout(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   register GELO_FLAVOR f;

   ITRACE("GELO_layout 0x%x",o);

   o->layout_valid = FALSE;

   if (o->x_size < o->x_min_size || o->y_size < o->y_min_size) {
      o->too_small = TRUE;
    };

   f = (o->too_small ? GELO_FLAVOR_EMPTY : o->flavor);

   (*(flavor_data[f].layout_rtn))(gw,o);

   o->layout_valid = TRUE;
};





void
GELO_relayout(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   register GELO_OBJECT gs;

   ITRACE("GELO_relayout 0x%x",o);

   if (!o->layout_valid) {
      GELO_layout(gw,o);
    }
   else {
      for (gs = o->son; gs != NULL; gs = gs->brother) {
	 GELO_relayout(gw,gs);
       };
    };
};





void
GELO_draw(gw,o,hifg)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_DRAW_FLAGS hifg;
{
   register GELO_FLAVOR f;

   ITRACE("GELO_draw 0x%x",o);

   f = (o->too_small ? GELO_FLAVOR_EMPTY : o->flavor);

   if (hifg  == GELO_DRAW_NORM &&
	  notify_fct != NULL && o->object != NULL && o->owner != NULL) {
      (*notify_fct)(o->owner,o,o->object,gw->realwindow,TRUE);
    };

   (*(flavor_data[f].draw_rtn))(gw,o,hifg);
};





/************************************************************************/
/*									*/
/*	GELO_set_x_size -- set x size of object 			*/
/*	GELO_set_y_size -- set y size of object 			*/
/*	GELO_inq_x_size -- return x size of object			*/
/*	GELO_inq_y_size -- return y size of object			*/
/*	GELO_set_x_min_size -- set minimum x size of object		*/
/*	GELO_set_y_min_size -- set minimum y size of object		*/
/*	GELO_inq_x_min_size -- return minimum x size of object		*/
/*	GELO_inq_y_min_size -- return minimum y size of object		*/
/*									*/
/************************************************************************/


void
GELO_set_x_size(o,sz)
   GELO_OBJECT o;
   GELO_COORD sz;
{
   ITRACE("GELO_set_x_size 0x%x %d",o,sz);

   if (sz < 0) sz = 0;

   o->x_size = sz;
};





void
GELO_set_y_size(o,sz)
   GELO_OBJECT o;
   GELO_COORD sz;
{
   ITRACE("GELO_set_y_size 0x%x %d",o,sz);

   if (sz < 0) sz = 0;

   o->y_size = sz;
};





GELO_COORD
GELO_inq_x_size(o)
   GELO_OBJECT o;
{
   ITRACE("GELO_inq_x_size 0x%x => %d",o,o->x_size);

   return o->x_size;
};





GELO_COORD
GELO_inq_y_size(o)
   GELO_OBJECT o;
{
   ITRACE("GELO_inq_y_size 0x%x => %d",o,o->y_size);

   return o->y_size;
};





void
GELO_set_x_min_size(o,sz)
   GELO_OBJECT o;
   GELO_COORD sz;
{
   ITRACE("GELO_set_x_min_size 0x%x %d",o,sz);

   if (sz < 0) sz = 0;

   o->x_min_size = sz;
};





void
GELO_set_y_min_size(o,sz)
   GELO_OBJECT o;
   GELO_COORD sz;
{
   ITRACE("GELO_set_y_min_size 0x%x %d",o,sz);

   if (sz < 0) sz = 0;

   o->y_min_size = sz;
};





GELO_COORD
GELO_inq_x_min_size(o)
   GELO_OBJECT o;
{
   GELO_COORD x;

   if (o->use_min) x = o->x_min_size;
   else if (o->use_default_x) x = o->x_size;
   else if (o->zero_size_x) x = 0;
   else x = MIN_COMPONENT_SIZE;

   ITRACE("GELO_inq_x_min_size 0x%x => %d",o,x);

   return x;
};





GELO_COORD
GELO_inq_y_min_size(o)
   GELO_OBJECT o;
{
   GELO_COORD y;

   if (o->use_min) y = o->y_min_size;
   else if (o->use_default_y) y = o->y_size;
   else if (o->zero_size_y) y = 0;
   else y = MIN_COMPONENT_SIZE;

   ITRACE("GELO_inq_y_min_size 0x%x => %d",o,y);

   return y;
};





/************************************************************************/
/*									*/
/*	GELO_set_x_position -- set relative x position			*/
/*	GELO_set_y_position -- set relative y position			*/
/*	GELO_inq_x_position -- inquire relative x position		*/
/*	GELO_inq_y_position -- inquire relative y position		*/
/*	GELO_inq_absolute_x_position -- inquire absolute x position	*/
/*	GELO_inq_absolute_y_position -- inquire absolute y position	*/
/*									*/
/************************************************************************/


void
GELO_set_x_position(o,ps)
   GELO_OBJECT o;
   GELO_COORD ps;
{
   ITRACE("GELO_set_x_position 0x%x %d",o,ps);

   o->x_pos = ps;
};





void
GELO_set_y_position(o,ps)
   GELO_OBJECT o;
   GELO_COORD ps;
{
   ITRACE("GELO_set_y_position 0x%x %d",o,ps);

   o->y_pos = ps;
};





GELO_COORD
GELO_inq_x_position(o)
   GELO_OBJECT o;
{
   ITRACE("GELO_inq_x_position 0x%x => %d",o,o->x_pos);

   return o->x_pos;
};





GELO_COORD
GELO_inq_y_position(o)
   GELO_OBJECT o;
{
   ITRACE("GELO_inq_y_position 0x%x => %d",o,o->y_pos);

   return o->y_pos;
};





GELO_COORD
GELO_inq_absolute_x_position(o)
   GELO_OBJECT o;
{
   register GELO_OBJECT p;
   register Integer x;

   x = o->x_pos;

   for (p = o->parent; p != NULL; p = p->parent) {
      if (p->too_small) x = 0;
      x += p->x_pos;
    };

   ITRACE("GELO_inq_absolute_x_position 0x%x => %d",o,x);

   return x;
};





GELO_COORD
GELO_inq_absolute_y_position(o)
   GELO_OBJECT o;
{
   register GELO_OBJECT p;
   register Integer y;

   y = o->y_pos;

   for (p = o->parent; p != NULL; p = p->parent) {
      if (p->too_small) y = 0;
      y += p->y_pos;
    };

   ITRACE("GELO_inq_absolute_y_position 0x%x => %d",o,y);

   return y;
};





/************************************************************************/
/*									*/
/*	GELO_flavor_define -- define a flavor				*/
/*									*/
/************************************************************************/


GELO_FLAVOR
GELO_flavor_define(id,name,attrs,srtn,lrtn,drtn,frtn,rrtn,crtn,hrtn,offrtn,cmprtn)
   GELO_FLAVOR id;
   String name;
   GELO_ATTR_DATA attrs;
   Function_Ptr srtn;
   Function_Ptr lrtn;
   Function_Ptr drtn;
   Function_Ptr frtn;
   Function_Ptr rrtn;
   Function_Ptr crtn;
   Function_Ptr hrtn;
   Function_Ptr offrtn;
   Function_Ptr cmprtn;
{
   register Integer i,j;

   if (id < 0) {
      for (i = 0; i < MAX_FLAVORS; ++i) {
	 if (flavor_data[i].num_attr < 0) break;
       };
      if (i >= MAX_FLAVORS) {
	 FATAL("Too many flavors defined");
       };
    }
   else if (flavor_data[id].num_attr >= 0) {
      FATAL("Redefinition of flavor");
    }
   else i = id;

   flavor_data[i].name = name;
   flavor_data[i].size_rtn = srtn;
   flavor_data[i].layout_rtn = lrtn;
   flavor_data[i].draw_rtn = drtn;
   flavor_data[i].free_rtn = frtn;
   flavor_data[i].replace_rtn = rrtn;
   flavor_data[i].corr_rtn = crtn;
   flavor_data[i].hilite_rtn = hrtn;
   flavor_data[i].offset_rtn = offrtn;
   flavor_data[i].compare_rtn = cmprtn;
   flavor_data[i].attrs = attrs;

   for (j = 0; attrs[j].name != NULL; ++j);
   flavor_data[i].num_attr = j;

   ITRACE("GELO_flavor_define 0x%x 0x%x 0x%x => %d",srtn,lrtn,drtn,i);

   return (GELO_FLAVOR) i;
};





/************************************************************************/
/*									*/
/*	GELO_replace_object -- replace object with new object		*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELO_replace_object(old,new)
   GELO_OBJECT old;
   GELO_OBJECT new;
{
   register GELO_OBJECT p,s,r;

   ITRACE("GELO_replace_object 0x%x 0x%x",old,new);

   p = old->parent;
   r = new;

   if (p != NULL && flavor_data[p->flavor].replace_rtn != NULL) {
      r = (GELO_OBJECT) (*(flavor_data[p->flavor].replace_rtn))(p,old,new);
    };

   GELOlayout_validate(new,FALSE);

   new->source = old->source;
   new->owner = old->owner;

   new->x_size = old->x_size;
   new->y_size = old->y_size;
   new->x_min_size = old->x_min_size;
   new->y_min_size = old->y_min_size;
   new->x_pos = old->x_pos;
   new->y_pos = old->y_pos;

   new->parent = p;
   if (p != NULL) {
      if (p->son == old) p->son = new;
      else {
	 for (s = p->son; s != NULL && s->brother != old; s = s->brother);
	 if (s != NULL) s->brother = new;
	 else abort();
       };
    };
   new->brother = old->brother;

   GELO_layout_fix_subconnect(p,old,new);

   return r;
};





/************************************************************************/
/*									*/
/*	GELO_object_free -- free a gelo object				*/
/*									*/
/************************************************************************/


void
GELO_object_free(gw,o,keep)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_OBJECT keep;
{
   ASH_WINDOW w;

   ITRACE("GELO_object_free 0x%x 0x%x 0x%x",gw,o,keep);

   if (keep != NULL && o == keep) return;

   if (gw != NULL) {
      GELO_deselect(gw,o);
      w = gw->realwindow;
    }
   else w = NULL;

   if (notify_fct != NULL && o != NULL &&
	  o->object != NULL && o->owner != NULL) {
      (*notify_fct)(o->owner,o,o->object,w,FALSE);
    };

   if (o != NULL && flavor_data[o->flavor].free_rtn != NULL) {
      (*(flavor_data[o->flavor].free_rtn))(gw,o,keep);
    };

   o->x_size = -1;
   o->y_size = -2;

   if (o != NULL) {
      if (o->attributes != NULL) free(o->attributes);
      o->attributes = NULL;
      free(o);
    };

   if (gw != NULL && gw->object == o) gw->object = NULL;
};





/************************************************************************/
/*									*/
/*	GELO_test_correlate -- check for correlation			*/
/*									*/
/************************************************************************/


GELO_OBJECT
GELO_test_correlate(gw,g,x,y)
   GELO_WINDOW gw;
   GELO_OBJECT g;
   Integer x,y;
{
   Integer lx,by,rx,ty;
   register GELO_OBJECT gs,gr;

   if (g == NULL) return NULL;

   if (flavor_data[g->flavor].corr_rtn != NULL && !g->too_small) {
      gr = (GELO_OBJECT) (*(flavor_data[g->flavor].corr_rtn))(gw,g,x,y);
      return gr;
    };

   if (!GELO_inq_position(gw,g,&lx,&by,&rx,&ty)) return NULL;

   if (lx < rx) {
      if (x < lx || x > rx) return NULL;
    }
   else {
      if (x > lx || x < rx) return NULL;
    };

   if (by < ty) {
      if (y < by || y > ty) return NULL;
    }
   else {
      if (y > by || y < ty) return NULL;
    };

   if (g->too_small) return g;

   for (gs = g->son; gs != NULL; gs = gs->brother) {
      gr = GELO_test_correlate(gw,gs,x,y);
      if (gr != NULL) {
	 g = gr;
	 break;
       };
    };

   return g;
};





/************************************************************************/
/*									*/
/*	GELO_hilite -- hilight specified object in specified style	*/
/*									*/
/************************************************************************/


void
GELO_hilite(gw,g,sty,fg)
   GELO_WINDOW gw;
   GELO_OBJECT g;
   Integer sty;
   Boolean fg;
{
   Integer lx,by,rx,ty;
   Integer fl,bg,tbg;
   GELO_OBJECT gp;
   Sequence l;

   ITRACE("GELO_hilite 0x%x 0x%x %d %d",gw,g,sty,fg);

   if (gw->window == NULL) return;

   if (flavor_data[g->flavor].hilite_rtn != NULL) {
      if ((*(flavor_data[g->flavor].hilite_rtn))(gw,g,sty,fg)) return;
    };

   gp = gw->object->parent;
   gw->object->parent = NULL;
   if (!GELO_inq_position(gw,g,&lx,&by,&rx,&ty)) return;
   gw->object->parent = gp;

   if (ASHinq_configuration_depth(gw->window) == 1) {
      ASHdraw_hilite(gw->window,TRUE);
      if (sty > 1 && sty < ASH_FILL_STIPPLE_OFFSET) sty += ASH_FILL_STIPPLE_OFFSET;
      fl = ASHfill(gw->window,sty);
      ASHrectangle(gw->window,lx,by,rx,ty);
      ASHfill(gw->window,fl);
      ASHdraw_hilite(gw->window,FALSE);
    }
   else {
      if (!fg) {
	 for (gp = g; gp != NULL; gp = gp->parent) {
	    for (l = gw->select; l != NULL; l = CDDR(l)) {
	       if (CAR(GELO_OBJECT,l) == gp) {
		  sty = CADR(Integer,l);
		  fg = TRUE;
		  break;
		};
	     };
	    if (l != NULL) break;
	  };
       };

      if (fg) {
	 bg = ASHbackground_color(gw->window,colors[sty].value);
	 tbg = ASHtext_background_color(gw->window,colors[sty].value);
       };
      GELO_draw(gw,g,(fg ? GELO_DRAW_HILITE : GELO_DRAW_HILITE_OFF));
      if (fg) {
	 ASHbackground_color(gw->window,bg);
	 ASHtext_background_color(gw->window,tbg);
       };
    };
};





/************************************************************************/
/*									*/
/*	GELO_inq_port_offset -- get port offset for object		*/
/*									*/
/************************************************************************/


void
GELO_inq_port_offset(obj,port,lx,by,rx,ty,dxp,dyp)
   GELO_OBJECT obj;
   GELO_PORT port;
   Integer lx,by,rx,ty;
   Integer * dxp, * dyp;
{
   ITRACE("GELO_inq_port_offset 0x%x %d %d %d %d %d",obj,port,lx,by,rx,ty);

   if (flavor_data[obj->flavor].offset_rtn != NULL) {
      (*(flavor_data[obj->flavor].offset_rtn))(obj,port,lx,by,rx,ty,dxp,dyp);
      return;
    };

   *dxp = 0;
   *dyp = 0;
};





/************************************************************************/
/*									*/
/*	GELO_object_compare -- compare two gelo objects for equality	*/
/*									*/
/************************************************************************/


Boolean
GELO_object_compare(o1,o2,o1p,o2p)
   GELO_OBJECT o1;
   GELO_OBJECT o2;
   GELO_OBJECT *o1p;
   GELO_OBJECT *o2p;
{
   register Boolean fg;

   ITRACE("GELO_object_compare 0x%x 0x%x",o1,o2);

   if (o1 == o2) fg = TRUE;
   else if (o1 == NULL || o2 == NULL || o1->flavor != o2->flavor ||
	       flavor_data[o1->flavor].compare_rtn == NULL) {
      fg = FALSE;
      *o1p = o1;
      *o2p = o2;
    }
   else {
      fg = (*(flavor_data[o1->flavor].compare_rtn))(o1,o2,o1p,o2p);
    };

   return fg;
};





/* end of geloobject.c */


