/************************************************************************/
/*									*/
/*		gelolevel.c						*/
/*									*/
/*	Level graph heuristics for graph layout 			*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"
#include "gelo_layout.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		set_levels();
static	Integer 	set_level();
static	Integer 	push_level();
static	void		opt_level();
static	Integer 	move_level_info();
static	void		move_up();
static	void		move_down();
static	void		set_ranks();
static	Integer 	rank_compare();
static	Float		avg_rank();
static	void		expand_rank();








/************************************************************************/
/*									*/
/*	GELO_level_init -- module initialization			*/
/*									*/
/************************************************************************/


void
GELO_level_init()
{
};





/************************************************************************/
/*									*/
/*	GELO_level_layout -- do level graph layout			*/
/*									*/
/************************************************************************/


Boolean
GELO_level_layout(o,mthd)
   GELO_OBJECT o;
   GELO_METHOD mthd;
{
   set_levels(o,mthd);

   set_ranks(o,mthd);

   return TRUE;
};






/************************************************************************/
/*									*/
/*	set_levels -- compute level for each non-portal node		*/
/*									*/
/************************************************************************/


static void
set_levels(o,mthd)
   GELO_OBJECT o;
   GELO_METHOD mthd;
{
   Boolean bi;
   Integer mx;
   COMPONENT * c;
   Sequence l;

   bi = (mthd & GELO_METHOD_BIT_2WAY ? TRUE : FALSE);

   mx = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->portal) continue;
      set_level(o,c,bi,1,mthd);
      if (c->pos_y > mx) mx = c->pos_y;
    };

   if (mthd & GELO_METHOD_BIT_VAR0) {
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 c->pos_y = -1;
       };
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 push_level(o,c,bi,mx);
       };
    };

   if (mthd & GELO_METHOD_BIT_VAR1) {
      opt_level(o,bi,mx);
    };
};






/************************************************************************/
/*									*/
/*	set_level -- assign a level to each node			*/
/*									*/
/************************************************************************/


static Integer
set_level(o,c,bi,lvl,mthd)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer lvl;
   GELO_METHOD mthd;
{
   Integer mx,j;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;

   if (c->portal) return -2;

   if (c->pos_y != -1) {
      if (c->pos_y < 0 || c->pos_y >= lvl) return c->pos_y;
      if (mthd & GELO_METHOD_BIT_VAR4) return c->pos_y;
    };

   if (mthd & GELO_METHOD_BIT_VAR4) c->pos_y = lvl;
   else c->pos_y = -2;

   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL) {
	 j = set_level(o,c1,bi,lvl+1,mthd);
       };
    };

   c->pos_y = lvl;

   return c->pos_y;
};





/************************************************************************/
/*									*/
/*	push_level -- assign a level to each node starting at bottom	*/
/*									*/
/************************************************************************/


static Integer
push_level(o,c,bi,bot)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer bot;
{
   Integer mn,j;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;

   if (c->portal) return -2;

   if (c->pos_y != -1) return c->pos_y;

   c->pos_y = -2;

   mn = bot+1;
   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL) {
	 j = push_level(o,c1,bi,bot);
	 if (j > 0 && j < mn) mn = j;
       };
    };

   c->pos_y = mn-1;

   return c->pos_y;
};







/************************************************************************/
/*									*/
/*	opt_level -- do leveling optimizations				*/
/*									*/
/************************************************************************/


static void
opt_level(o,bi,mx)
   GELO_OBJECT o;
   Boolean bi;
   Integer mx;
{
   COMPONENT * c;
   Sequence l;
   Integer * cts;
   Integer i,j,k,ln;
   Boolean chng;
   Integer up,down;

   cts = (Integer *) alloca((mx+4)*sizeof(Integer));

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->cost = -1;
      c->pos_x = -1;
    };
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      move_level_info(o,c,bi,1,mx);
    };

   ln = LENGTH(COMPONENTS(o));

   for (k = 0; k < ln; ++k) {
      chng = FALSE;

      for (i = 0; i <= mx; ++i) cts[i] = 0;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (!c->portal) cts[c->pos_y]++;
       };

      j = 1;
      for (i = 2; i <= mx; ++i) {
	 if (cts[i] > cts[j]) j = i;
	 else if (cts[i] == cts[j] && i != mx) j = i;
       };

      up = (j == 1 ? 0 : 1);
      down = (j == mx ? 0 : 1);
      if (up && cts[j-1] == cts[j]) up = 0;
      if (down && cts[j+1] == cts[j]) down = 0;
      if (!up && !down) break;
      if (up && down) {
	 if (cts[j-1] < cts[j+1]) ++up;
	 else ++down;
       };

      for (i = 2; i > 0; --i) {
	 if (i == 2 && (up == 0 || down == 0)) continue;
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    if (c->portal) continue;
	    if (c->pos_y != j) continue;
	    if (up == i && c->pos_x == j) continue;
	    if (down == i && c->cost == j) continue;
	    if (up == i) move_up(o,c,bi);
	    else if (down == i) move_down(o,c,bi,mx);
	    chng = TRUE;
	    break;
	  };
	 if (chng) break;
       };

      if (!chng) break;
    };
};






/************************************************************************/
/*									*/
/*	move_level_info -- get topmost and bottommost positions 	*/
/*									*/
/************************************************************************/


static Integer
move_level_info(o,c,bi,up,mx)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer up;
   Integer mx;
{
   GELO_CONNECT conn;
   COMPONENT * c1;
   Integer j,ct;
   Sequence l;

   if (c->pos_x < 0) c->pos_x = up;
   else {
      c->pos_x = MAX(c->pos_x,up);
      return;
    };

   ct = mx+1;
   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && c1->pos_y > c->pos_y) {
	 j = move_level_info(o,c1,bi,up+1,mx);
	 if (j < ct) ct = j;
       };
    };

   c->cost = ct-1;

   return c->cost;
};






/************************************************************************/
/*									*/
/*	move_up -- move a node up one level				*/
/*	move_down -- move a node down one level 			*/
/*									*/
/************************************************************************/


static void
move_up(o,c,bi)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
{
   GELO_CONNECT conn;
   COMPONENT * c1;
   Sequence l;

   if (c->portal || c->pos_y == 1) return;
   c->pos_y -= 1;

   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->to_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);

      if (c1 != NULL && !c1->portal && c1->pos_y == c->pos_y - 1) {
	 move_up(o,c1,bi);
       };
    };
};







static void
move_down(o,c,bi,mx)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer mx;
{
   GELO_CONNECT conn;
   COMPONENT * c1;
   Sequence l;

   if (c->portal || c->pos_y == mx) return;
   c->pos_y += 1;

   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);

      if (c1 != NULL && !c1->portal && c1->pos_y == c->pos_y + 1) {
	 move_down(o,c1,bi,mx);
       };
    };
};







/************************************************************************/
/*									*/
/*	set_ranks -- set ranks for all components			*/
/*									*/
/************************************************************************/







typedef struct _rank_info {
   COMPONENT * component;
   Float value;
} RANK_INFO;



static void
set_ranks(o,mthd)
   GELO_OBJECT o;
   GELO_METHOD mthd;
{
   Integer * ranks;
   RANK_INFO * row;
   Integer lvl,i,j,ln,rown;
   COMPONENT * c;
   Sequence l;
   Integer mxlvl,mxrow;

   ln = LENGTH(COMPONENTS(o))+10;
   row = (RANK_INFO *) alloca(ln*sizeof(RANK_INFO));
   ranks = (Integer *) alloca(ln*sizeof(Integer));

   for (i = 0; i < ln; ++i) ranks[i] = 0;

   mxlvl = 0;
   mxrow = 0;
   rown = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (!c->portal && c->pos_y > 0) {
	 if (c->pos_y > mxlvl) mxlvl = c->pos_y;
	 ranks[c->pos_y] += 1;
	 c->pos_x = ranks[c->pos_y];
	 if (c->pos_x >= mxrow) {
	    mxrow = c->pos_x;
	    rown = c->pos_y;
	  };
       };
     };


   if (mthd & GELO_METHOD_BIT_VAR2) {
      for (lvl = mxlvl-1; lvl > 0; --lvl) {
	 j = 0;
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    if (!c->portal && c->pos_y == lvl) {
	       row[j].component = c;
	       row[j].value = avg_rank(o,c,lvl+1);
	       ++j;
	     };
	  };
	 if (j == 0) continue;

	 qsort(row,j,sizeof(RANK_INFO),rank_compare);

	 for (i = 0; i < j; ++i) {
	    row[i].component->pos_x = i+1;
	  };
       };
    };

   if (mthd & GELO_METHOD_BIT_VAR3) {
      for (lvl = rown-1; lvl > 0; --lvl) {
	 expand_rank(o,lvl,lvl+1,mxrow);
       }
      for (lvl = rown+1; lvl <= mxlvl; ++lvl) {
	 expand_rank(o,lvl,lvl-1,mxrow);
       };
    };
};





/************************************************************************/
/*									*/
/*	rank_compare -- sort comparison function for ranks		*/
/*									*/
/************************************************************************/


static Integer
rank_compare(r1,r2)
   RANK_INFO *r1;
   RANK_INFO *r2;
{
   Float v;
   Integer i;

   v = r1->value - r2->value;

   if (v < 0) i = -1;
   else if (v == 0) i = 0;
   else if (v > 0) i = 1;

   return i;
};






/************************************************************************/
/*									*/
/*	avg_rank -- compute the average position for a node		*/
/*									*/
/************************************************************************/


static Float
avg_rank(o,c,flvl)
   GELO_OBJECT o;
   COMPONENT * c;
   Integer flvl;
{
   GELO_CONNECT conn;
   Sequence l;
   COMPONENT *c1;
   Integer i,j;

   i = 0;
   j = 0;

   forin (conn,GELO_CONNECT,l,c->all_arcs) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && !c1->portal && c1->pos_y == flvl) {
	 i += c1->pos_x;
	 j += 1;
       };
    };

   if (j == 0) return 10000.0;

   return ((Float) i)/((Float) j);
};





/************************************************************************/
/*									*/
/*	expand_rank -- expand ranks to line up nodes			*/
/*									*/
/************************************************************************/


static void
expand_rank(o,lvl,nlvl,mxrow)
   GELO_OBJECT o;
   Integer lvl,nlvl;
   Integer mxrow;
{
   COMPONENT ** row, ** nextrow, * c, * c1, * c2;
   Integer j,nj,i,pos,k,m,slack;
   GELO_CONNECT conn;
   Sequence l;
   Integer sum,ct;

   row = (COMPONENT **) alloca((mxrow+4) * sizeof(COMPONENT *));
   nextrow = (COMPONENT **) alloca((mxrow+4) * sizeof(COMPONENT *));

   for (i = 0; i <= mxrow; ++i) {
      nextrow[i] = NULL;
      row[i] = NULL;
    };

   j = 0;
   nj = 0;
   ct = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (!c->portal) {
	 if (c->pos_y == lvl) {
	    row[c->pos_x] = c;
	    if (c->pos_x > j) j = c->pos_x;
	    ++ct;
	  }
	 else if (c->pos_y == nlvl) {
	    nextrow[c->pos_x] = c;
	    if (c->pos_x > nj) nj = c->pos_x;
	  };
       };
    };

   pos = 1;
   slack = mxrow - ct;
   if (slack == 0) return;

   for (i = 0; i <= j; ++i) {
      c = row[i];
      if (c == NULL) continue;
      sum = 0;
      ct = 0;
      for (m = 0; m <= nj; ++m) {
	 c1 = nextrow[m];
	 if (c1 != NULL) {
	    forin (conn,GELO_CONNECT,l,c->all_arcs) {
	       c2 = (COMPONENT *) GELO_conn_inq_from_data(conn);
	       if (c2 == c) c2 = (COMPONENT *) GELO_conn_inq_to_data(conn);
	       if (c2 == c1) break;
	     };
	    if (conn != NULL) {
	       sum += m;
	       ct += 1;
	     };
	  };
       };

      if (ct == 0) k = pos;
      else {
	 sum /= ct;
	 if (sum <= pos) k = pos;
	 else if (sum >= pos+slack) k = pos+slack;
	 else k = sum;
       };

      slack -= k-pos;
      c->pos_x = k;
      pos = k+1;
    };
};






/* end of gelolevel.c */
