/************************************************************************/
/*									*/
/*		gelolayout.c						*/
/*									*/
/*	LAYOUT flavor object routines for GELO				*/
/*									*/
/************************************************************************/
/*	Copyright 1985 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"
#include "gelo_layout.h"





/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


/************************************************************************/
/*									*/
/*	Local storage							*/
/*									*/
/************************************************************************/


static	GELO_ATTR_DATA_B	layout_attrs[] = {
   { "METHOD" },
   { "CONN_METHOD" },
   { "SHRINKAGE" },
   { "WHITE_SPACE" },
   { "BOX" },
   { "COMPONENTS" },
   { "CONNECTS" },
   { "FIXED" },
   { "STANDARD" },
   { "CENTERED" },
   { NULL }
};





/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	Boolean 	gelo_layout_size();
static	void		gelo_layout_layout();
static	void		gelo_layout_draw();
static	Boolean 	compute_layout_size();
static	void		compute_white_space();
static	void		layout_connections();
static	COMPONENT *	find_component();
static	Integer 	seq_port();
static	void		gelo_layout_free();
static	GELO_OBJECT	gelo_layout_replace();
static	Boolean 	gelo_layout_compare();





/************************************************************************/
/*									*/
/*	GELO_layout_init -- module initialization			*/
/*									*/
/************************************************************************/


void
GELO_layout_init()
{
   ITRACE("GELO_layout_init");

   GELO_flavor_define(GELO_FLAVOR_LAYOUT,
			 "LAYOUT",
			 layout_attrs,
			 gelo_layout_size,
			 gelo_layout_layout,
			 gelo_layout_draw,
			 gelo_layout_free,gelo_layout_replace,NULL,NULL,NULL,
			 gelo_layout_compare);
};





/************************************************************************/
/*									*/
/*	GELOlayout_component -- define a new component			*/
/*									*/
/************************************************************************/


int
GELOlayout_component(o)
   GELO_OBJECT o;
{
   register COMPONENT * c;

   c = (COMPONENT *) calloc(1,sizeof(COMPONENT));

   c->object = o;
   c->done = FALSE;
   c->portal = FALSE;
   c->port = GELO_PORT_ANY;
   c->from_arcs = NULL;
   c->to_arcs = NULL;
   c->all_arcs = NULL;

   TRACE("GELOlayout_component 0x%x => 0x%x",o,c);

   return ((int) c);
};





/************************************************************************/
/*									*/
/*	GELOlayout_portal -- define a new portal component		*/
/*									*/
/************************************************************************/


int
GELOlayout_portal(o,port)
   GELO_OBJECT o;
   GELO_PORT port;
{
   register COMPONENT * c;

   c = (COMPONENT *) calloc(1,sizeof(COMPONENT));

   c->object = o;
   c->done = FALSE;
   c->portal = TRUE;
   c->port = port;

   TRACE("GELOlayout_portal 0x%x %d => 0x%x",o,port,c);

   return ((int) c);
};





/************************************************************************/
/*									*/
/*	GELOinq_layout_heuristics -- return layout heuristics		*/
/*									*/
/************************************************************************/


int
GELOinq_layout_heuristics(o,mp,cmp,wp)
   GELO_OBJECT o;
   GELO_METHOD *mp;
   GELO_CONN_METHOD *cmp;
   Integer *wp;
{
   TRACE("GELOinq_layout_heuristics 0x%x",o);

   if (o->flavor != GELO_FLAVOR_LAYOUT) return FALSE;

   if (mp != NULL) *mp = METHOD(o);
   if (cmp != NULL) *cmp = CONNMETHOD(o);
   if (wp != NULL) *wp = WHITE_SPACE(o);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	GELO_layout_connect_correlate -- correlate connections		*/
/*									*/
/************************************************************************/


GELO_CONNECT
GELO_layout_connect_correlate(gw,lay,x,y)
   GELO_WINDOW gw;
   GELO_OBJECT lay;
   Integer x,y;
{
   ITRACE("GELO_layout_connect_correlate 0x%x 0x%x %d %d",gw,lay,x,y);

   return GELO_connect_list_correlate(gw,CONNECTS(lay),x,y);
};






/************************************************************************/
/*									*/
/*	GELO_layout_fix_subconnect -- handle sub connections to node	*/
/*									*/
/************************************************************************/


GELO_layout_fix_subconnect(o,old,new)
   GELO_OBJECT o;
   GELO_OBJECT old;
   GELO_OBJECT new;
{
   while (o != NULL) {
      if (o->flavor == GELO_FLAVOR_LAYOUT) {
	 GELO_connect_replace(CONNECTS(o),old,new);
       };
      o = o->parent;
    };
};





/************************************************************************/
/*									*/
/*	gelo_layout_size -- size a layout object			 */
/*									*/
/************************************************************************/


static Boolean
gelo_layout_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   register Sequence l;
   register COMPONENT * c;
   register Integer x,y;
   register Boolean fg;
   register Float sh;

   ITRACE("gelo_layout_size 0x%x %d %d",o,mxx,mxy);

   if (mxx < MIN_SIZE || mxy < MIN_SIZE) return FALSE;

   GELO_heur_layout(o,mxx,mxy);

   if (SHRINKAGE(o) == 0) sh = 1.0;
   else sh = SHRINKAGE(o)/100.0;

   x = mxx*sh;
   y = mxy*sh;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      fg = GELO_size(c->object,x,y);
      if (!fg) return FALSE;
    };

   fg = compute_layout_size(o,mxx,mxy);

   return fg;
};





/************************************************************************/
/*									*/
/*	gelo_layout_layout -- layout a layout object			*/
/*									*/
/************************************************************************/


static void
gelo_layout_layout(gw,o)
   GELO_WINDOW gw;
   GELO_OBJECT o;
{
   Integer * mx;
   Integer * mn;
   Integer * co;
   Integer * rw;
   Integer nm,totx,totn,wx,wy;
   Float a;
   register Integer i,j,k;
   register COMPONENT * c;
   register Sequence l;
   Integer ln;

   ITRACE("gelo_layout_layout 0x%x 0x%x",gw,o);

   ln = LENGTH(COMPONENTS(o));
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->pos_x > ln) ln = c->pos_x;
      if (c->pos_y > ln) ln = c->pos_y;
    };
   ln += 4;
   mx = (Integer *) alloca(ln*sizeof(Integer));
   mn = (Integer *) alloca(ln*sizeof(Integer));
   co = (Integer *) alloca(ln*sizeof(Integer));
   rw = (Integer *) alloca(ln*sizeof(Integer));
   for (j = 0; j < ln; ++j) {
      mx[j] = mn[j] = co[j] = rw[j] = 0;
    };

   compute_white_space(o,&wx,&wy);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      for (i = 0; i < 4; ++i) {
	 c->seqs[i] = 0;
	 c->seqct[i] = 0;
       };
    };


   nm = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->portal) nm = MAX(nm,c->pos_x);
      else nm = MAX(nm,c->pos_x+1);
      j = GELO_inq_x_size(c->object);
      j *= GELOinq_priority_x(c->object);
      mx[c->pos_x] = MAX(mx[c->pos_x],j);
      mn[c->pos_x] = MAX(mn[c->pos_x],GELO_inq_x_min_size(c->object));
    };

   totx = wx*(nm);
   totn = MIN_WHITE_SPACE*(nm);
   for (i = 0; i <= nm; ++i) {
      totx += mx[i];
      totn += mn[i];
    };
   k = GELO_inq_x_size(o);
   if (totn > k) {
      GELO_set_too_small(o);
      return;
    };
   if (totn == totx) a = 1.0;
   else {
      a = k-totn;
      a /= (totx-totn);
      if (a > 1.0 && STANDARD(o)) a = 1.0;
    };
   for (i = 0; i < nm; ++i) {
      mx[i] = mn[i] + (mx[i]-mn[i])*a;
    };
   wx = (wx-MIN_WHITE_SPACE)*a+MIN_WHITE_SPACE;
   k = 1;
   for (i = 0; i <= nm; ++i) {
      j = mx[i];
      co[i] = k-wx;
      mn[i] = k;
      k += j+wx;
    };
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      j = GELO_inq_x_size(c->object);
      j *= GELOinq_priority_x(c->object);
      k = GELO_inq_x_min_size(c->object);
      if (FIXED(o) && j != k) k = mx[c->pos_x];
      else k += (j-k)*a;
      if (k > mx[c->pos_x]) k = mx[c->pos_x];
      GELO_set_x_size(c->object,k);
      j = mn[c->pos_x];
      if (CENTERED(o)) j += (mx[c->pos_x]-k)/2;
      GELO_set_x_position(c->object,j);
    };

   for (i = 0; i < ln; ++i) mx[i] = mn[i] = 0;
   nm = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->portal) nm = MAX(nm,c->pos_y);
      else nm = MAX(nm,c->pos_y+1);
      j = GELO_inq_y_size(c->object);
      j *= GELOinq_priority_y(c->object);
      mx[c->pos_y] = MAX(mx[c->pos_y],j);
      mn[c->pos_y] = MAX(mn[c->pos_y],GELO_inq_y_min_size(c->object));
    };
   totx = wy*(nm);
   totn = MIN_WHITE_SPACE*(nm);
   for (i = 0; i <= nm; ++i) {
      totx += mx[i];
      totn += mn[i];
    };
   k = GELO_inq_y_size(o);
   if (totn > k) {
      GELO_set_too_small(o);
      return;
    };
   if (totn == totx) a = 1.0;
   else {
      a = k-totn;
      a /= (totx-totn);
      if (a > 1.0 && STANDARD(o)) a = 1.0;
    };
   for (i = 0; i < nm; ++i) {
      mx[i] = mn[i] + (mx[i]-mn[i])*a;
    };
   wy = (wy-MIN_WHITE_SPACE)*a+MIN_WHITE_SPACE;
   k = 1;
   for (i = 0; i <= nm; ++i) {
      j = mx[i];
      rw[i] = k-wy;
      mn[i] = k;
      k += j+wy;
    };
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      j = GELO_inq_y_size(c->object);
      j *= GELOinq_priority_y(c->object);
      k = GELO_inq_y_min_size(c->object);
      if (FIXED(o) && j != k) k = mx[c->pos_y];
      else k += (j-k)*a;
      if (k > mx[c->pos_y]) k = mx[c->pos_y];
      GELO_set_y_size(c->object,k);
      j = mn[c->pos_y];
      if (CENTERED(o)) j += (mx[c->pos_y]-k)/2;
      GELO_set_y_position(c->object,j);
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      GELO_layout(gw,c->object);
    };

   layout_connections(gw,o,wx,wy,co,rw);
};





/************************************************************************/
/*									*/
/*	gelo_layout_draw -- draw a layout object			*/
/*									*/
/************************************************************************/


static void
gelo_layout_draw(gw,o,hifg)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_DRAW_FLAGS hifg;
{
   register Sequence l;
   register COMPONENT * c;
   Integer lx,by,rx,ty;

   ITRACE("gelo_layout_draw 0x%x",o);

   if (!GELO_inq_position(gw,o,&lx,&by,&rx,&ty)) return;

   if (BOX(o)) {
      ASHbox(gw->window,lx,by,rx,ty);
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      GELO_draw(gw,c->object,hifg);
    };

   GELO_draw_connections(gw,CONNECTS(o),hifg);
};





/************************************************************************/
/*									*/
/*	compute_layout_size -- estimate size for given layout		*/
/*									*/
/************************************************************************/


static Boolean
compute_layout_size(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   Integer * xsz;
   Integer * ysz;
   Integer * xmns;
   Integer * ymns;
   register Integer i,j;
   register Sequence l;
   register COMPONENT * c;
   Integer ctx,cty;
   Integer mnx,mny;
   Integer nx,ny;
   Integer wx,wy;
   Integer ln;
   Integer rptct;

   DTRACE("compute_layout_size 0x%x %d %d",o,mxx,mxy);

   ln = LENGTH(COMPONENTS(o));
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->pos_x > ln) ln = c->pos_x;
      if (c->pos_y > ln) ln = c->pos_y;
    };
   ln += 2;
   xsz = (Integer *) alloca(ln*sizeof(Integer));
   ysz = (Integer *) alloca(ln*sizeof(Integer));
   xmns = (Integer *) alloca(ln*sizeof(Integer));
   ymns = (Integer *) alloca(ln*sizeof(Integer));

   for (rptct = 0; ; ++rptct) {
      for (i = 0; i < ln; ++i) {
	 xsz[i] = ysz[i] = 0;
	 xmns[i] = ymns[i] = 0;
       };
      ctx = cty = 1;

      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 j = GELO_inq_x_size(c->object);
	 xsz[c->pos_x] = MAX(xsz[c->pos_x],j);
	 j = GELO_inq_x_min_size(c->object);
	 xmns[c->pos_x] = MAX(xmns[c->pos_x],j);
	 j = GELO_inq_y_size(c->object);
	 ysz[c->pos_y] = MAX(ysz[c->pos_y],j);
	 j = GELO_inq_y_min_size(c->object);
	 ymns[c->pos_y] = MAX(ymns[c->pos_y],j);
	 if (c->portal) {
	    ctx = MAX(ctx,c->pos_x);
	    cty = MAX(cty,c->pos_y);
	  }
	 else {
	    ctx = MAX(ctx,c->pos_x+1);
	    cty = MAX(cty,c->pos_y+1);
	  };
       };

      compute_white_space(o,&wx,&wy);

      nx = ny = 0;
      mnx = mny = 0;
      for (i = 0; i <= ctx; ++i) {
	 nx += xsz[i];
	 mnx += xmns[i];
       };
      for (i = 0; i <= cty; ++i) {
	 ny += ysz[i];
	 mny += ymns[i];
       };
      nx += (ctx)*wx+2;
      ny += (cty)*wy+2;
      mnx += (ctx)*MIN_WHITE_SPACE+2;
      mny += (cty)*MIN_WHITE_SPACE+2;

      if (mnx <= mxx && mny <= mxy) break;

      if (rptct == 0) {
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    GELOuse_min_size(c->object,FALSE);
	    GELOuse_default_x(c->object,FALSE);
	    GELOuse_default_y(c->object,FALSE);
	  };
       }
      else if (rptct == 1) {
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    GELO_set_x_min_size(c->object,MIN_COMPONENT_SIZE);
	    GELO_set_y_min_size(c->object,MIN_COMPONENT_SIZE);
	  };
       }
      else if (rptct == 2) {
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    GELOuse_zero_size_x(c->object,TRUE);
	    GELOuse_zero_size_y(c->object,TRUE);
	  };
       }
      else return FALSE;
    };

   if (nx <= mnx) nx = mnx+1;
   if (ny <= mny) ny = mny+1;

   if (nx > mxx) {
      ny = (ny*mxx)/nx;
      nx = mxx;
    };
   if (ny > mxy) {
      nx = (nx*mxy)/ny;
      ny = mxy;
    };

   GELO_set_x_size(o,nx);
   GELO_set_y_size(o,ny);
   GELO_set_x_min_size(o,mnx);
   GELO_set_y_min_size(o,mny);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	compute_white_space -- compute white space for object		*/
/*									*/
/************************************************************************/


static void
compute_white_space(o,wxp,wyp)
   GELO_OBJECT o;
   Integer *wxp, *wyp;
{
   register Sequence l;
   register COMPONENT * c;
   Integer xtot,ytot,ct;
   register Integer w,wx;
   GELO_CONN_METHOD cmthd;
   GELO_METHOD mthd;

   DTRACE("compute_white_space 0x%x",o);

   ct = 0;
   xtot = ytot = 0;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      xtot += GELO_inq_x_size(c->object);
      ytot += GELO_inq_y_size(c->object);
      ct += 1;
    };

   if (ct == 0) {
      *wxp = MIN_WHITE_SPACE;
      *wyp = MIN_WHITE_SPACE;
    }
   else {
      w = WHITE_SPACE(o);
      if (w <= 0) {
	 cmthd = CONNMETHOD(o);
	 mthd = METHOD(o);
	 if (CONNECTS(o) == NULL) w = DEFAULT_WHITE_SPACE_NOCONN;
	 else if (cmthd != GELO_CONN_METHOD_DIRECT &&
		     (mthd & (GELO_METHOD_PERTBITS|GELO_METHOD_BIT_GIOTTO)) != 0)
	    w = DEFAULT_WHITE_SPACE_PERT;
	 else w = DEFAULT_WHITE_SPACE;
       };

      wx = xtot*w/(ct*100);
      if (wx <= MIN_WHITE_SPACE) wx = MIN_WHITE_SPACE;
      *wxp = wx;

      wx = ytot*w/(ct*100);
      if (wx <= MIN_WHITE_SPACE) wx = MIN_WHITE_SPACE;
      *wyp = wx;
    };

   DTRACE("\t=> %d %d,%d",w,*wxp,*wyp);
};





/************************************************************************/
/*									*/
/*	layout_connections -- layout the actual connection		*/
/*	find_component -- find component for connection 		*/
/*									*/
/************************************************************************/


static void
layout_connections(gw,o,wx,wy,copos,rwpos)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer wx,wy;
   Integer copos[];
   Integer rwpos[];
{
   register GELO_CONNECT conn;
   register Sequence l;
   register COMPONENT * c1, * c2;
   register Integer i,j,seq;
   GELO_METHOD mthd;
   GELO_CONN_METHOD cmthd;

   DTRACE("layout_connections 0x%x 0x%x %d %d",gw,o,wx,wy);

   mthd = METHOD(o);
   cmthd = CONNMETHOD(o);

   seq = 0;
   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      c1 = find_component(o,GELO_inq_connect_from(conn));
      if (c1 == NULL) continue;
      c2 = find_component(o,GELO_inq_connect_to(conn));
      if (c2 == NULL) continue;

      GELO_conn_set_ports(conn,c1,c2,mthd,cmthd,FALSE);

      i = seq_port(GELO_inq_connect_from_port(conn));
      if (i >= 0) {
	 c1->seqs[i]++;
	 ++seq;
       };
      i = seq_port(GELO_inq_connect_to_port(conn));
      if (i >= 0) {
	 c2->seqs[i]++;
	 ++seq;
       };
    };

   if (seq > 0) {
      forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
	 c1 = find_component(o,GELO_inq_connect_from(conn));
	 if (c1 == NULL) continue;
	 c2 = find_component(o,GELO_inq_connect_to(conn));
	 if (c2 == NULL) continue;
	 i = seq_port(GELO_inq_connect_from_port(conn));
	 if (i >= 0) {
	    j = GELO_conn_inq_from_subport(conn);
	    if (j < 0) j = c1->seqct[i]++;
	    GELO_conn_set_from_subport(conn,j,c1->seqs[i]);
	  };
	 i = seq_port(GELO_inq_connect_to_port(conn));
	 if (i >= 0) {
	    j = GELO_conn_inq_to_subport(conn);
	    if (j < 0) j = c2->seqct[i]++;
	    GELO_conn_set_to_subport(conn,j,c2->seqs[i]);
	  };
       };
    };

   GELO_heur_connect(gw,o,wx,wy,copos,rwpos);
};





static COMPONENT *
find_component(o,obj)
   GELO_OBJECT o;
   GELO_OBJECT obj;
{
   register Sequence l;
   register COMPONENT * c;

   DTRACE("find_component 0x%x 0x%x",o,obj);

   while (obj != NULL) {
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (c->object == obj) return c;
       };
      obj = GELOinq_parent(obj);
    };

   return NULL;
};





static Integer
seq_port(pt)
   GELO_PORT pt;
{
   register Integer i;

   DTRACE("seq_port %d",pt);

   switch (pt) {
      case GELO_PORT_TOP_SEQ :
	 i = 0;
	 break;
      case GELO_PORT_BOTTOM_SEQ :
	 i = 1;
	 break;
      case GELO_PORT_LEFT_SEQ :
	 i = 2;
	 break;
      case GELO_PORT_RIGHT_SEQ :
	 i = 3;
	 break;
      default :
	 i = -1;
	 break;
    };

   return i;
};





/************************************************************************/
/*									*/
/*	gelo_layout_free -- free a layout object			*/
/*									*/
/************************************************************************/


static void
gelo_layout_free(gw,o,keep)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_OBJECT keep;
{
   register Sequence l;
   register COMPONENT * c;

   DTRACE("gelo_free_layout 0x%x 0x%x 0x%x",gw,o,keep);

   if (o != NULL) {
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (c->object != NULL) GELO_object_free(gw,c->object,keep);
	 c->object = NULL;
	 if (c->from_arcs != NULL) LFREE(c->from_arcs);
	 c->from_arcs = NULL;
	 if (c->to_arcs != NULL) LFREE(c->to_arcs);
	 c->to_arcs = NULL;
	 if (c->all_arcs != NULL) LFREE(c->all_arcs);
	 c->all_arcs = NULL;
	 free(c);
       };
      LFREE(COMPONENTS(o));
      SET_COMPONENTS(o,NULL);

      GELO_free_connects(gw,CONNECTS(o),keep);
      LFREE(CONNECTS(o));
      SET_CONNECTS(o,NULL);
    };
};





/************************************************************************/
/*									*/
/*	gelo_layout_replace -- replace component of object		*/
/*									*/
/************************************************************************/


static GELO_OBJECT
gelo_layout_replace(o,old,new)
   GELO_OBJECT o;
   GELO_OBJECT old;
   GELO_OBJECT new;
{
   register Sequence l;
   register COMPONENT * c;
   register GELO_OBJECT r;

   DTRACE("gelo_layout_replace 0x%x 0x%x 0x%x",o,old,new);

   r = new;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->object == old) {
	 c->object = new;
	 GELO_connect_replace(CONNECTS(o),old,new);
	 break;
       };
    };

   if (EMPTY(l)) {
      if (MEMQ(old,CONNECTS(o))) {
	 r = o;
	 SET_CONNECTS(o,REMOB(old,CONNECTS(o)));
	 SET_CONNECTS(o,APPEND(new,CONNECTS(o)));
       };
    };

   return r;
};





/************************************************************************/
/*									*/
/*	gelo_layout_compare -- compare two layout objects		*/
/*									*/
/************************************************************************/


static Boolean
gelo_layout_compare(o1,o2,o1p,o2p)
   GELO_OBJECT o1,o2;
   GELO_OBJECT * o1p, *o2p;
{
   register Sequence l,la,lb;
   register COMPONENT * c1,* c2;
   register Boolean fg,nfg;
   register Integer i;
   register GELO_CONNECT conn1,conn2;
   GELO_OBJECT o1a,o2a,o1b;

   DTRACE("gelo_layout_compare 0x%x 0x%x",o1,o2);

   if (METHOD(o1) != METHOD(o2)) fg = FALSE;
   else if (CONNMETHOD(o1) != CONNMETHOD(o2)) fg = FALSE;
   else if (SHRINKAGE(o1) != SHRINKAGE(o2)) fg = FALSE;
   else if (WHITE_SPACE(o1) != WHITE_SPACE(o2)) fg = FALSE;
   else if (FIXED(o1) != FIXED(o2)) fg = FALSE;
   else if (STANDARD(o1) != STANDARD(o2)) fg = FALSE;
   else if (CENTERED(o1) != CENTERED(o2)) fg = FALSE;
   else if (LENGTH(COMPONENTS(o1)) != LENGTH(COMPONENTS(o2))) fg = FALSE;
   else if (LENGTH(CONNECTS(o1)) != LENGTH(CONNECTS(o2))) fg = FALSE;
   else {
      fg = TRUE;
      la = COMPONENTS(o2);
      forin (c1,COMPONENT *,l,COMPONENTS(o1)) {
	 c2 = CAR(COMPONENT *,la);
	 la = CDR(la);
	 if (c1->portal != c2->portal || c1->port != c2->port) {
	    fg = FALSE;
	    break;
	  };
       };
    };

   if (fg) {
      la = CONNECTS(o2);
      forin (conn1,GELO_CONNECT,l,CONNECTS(o1)) {
	 conn2 = CAR(GELO_CONNECT,la);
	 la = CDR(la);
	 if (!GELO_connect_compare(conn1,conn2)) {
	    fg = FALSE;
	    break;
	  };
	 i = 1;
	 forin (c1,COMPONENT *,lb,COMPONENTS(o1)) {
	    if (c1->object == GELO_inq_connect_from(conn1)) {
	       c2 = NTH(COMPONENT *,COMPONENTS(o2),i);
	       if (c2->object != GELO_inq_connect_from(conn2)) {
		  fg = FALSE;
		  break;
		};
	     };
	    if (c1->object == GELO_inq_connect_to(conn1)) {
	       c2 = NTH(COMPONENT *,COMPONENTS(o2),i);
	       if (c2->object != GELO_inq_connect_to(conn2)) {
		  fg = FALSE;
		  break;
		};
	     };
	    ++i;
	  };
       };
    };

   if (!fg) {
      *o1p = o1;
      *o2p = o2;
    }
   else {
      la = COMPONENTS(o2);
      forin (c1,COMPONENT *,l,COMPONENTS(o1)) {
	 c2 = CAR(COMPONENT *,la);
	 la = CDR(la);
	 nfg = GELO_object_compare(c1->object,c2->object,&o1a,&o2a);
	 if (fg) {
	    if (!nfg) {
	       *o1p = o1a;
	       *o2p = o2a;
	       fg = FALSE;
	     };
	  }
	 else if (!nfg) {
	    *o1p = o1;
	    *o2p = o2;
	    break;
	  };
       };

      if (!fg && *o1p != o1) {
	 nfg = FALSE;
	 o1a = *o1p;
	 forin (conn1,GELO_CONNECT,l,CONNECTS(o1)) {
	    o1b = GELO_inq_connect_from(conn1);
	    for (o1b = o1b->parent; o1b != NULL && o1b != o1; o1b = o1b->parent) {
	       if (o1b == o1a) nfg = TRUE;
	     };
	    o1b = GELO_inq_connect_to(conn1);
	    for (o1b = o1b->parent; o1b != NULL && o1b != o1; o1b = o1b->parent) {
	       if (o1b == o1a) nfg = TRUE;
	     };
	    if (nfg) break;
	  };
	 if (nfg) {
	    *o1p = o1;
	    *o2p = o2;
	  };
       };
    };

   return fg;
};






/* end of gelolayout.c */
