/************************************************************************/
/*									*/
/*		geloheur.c						*/
/*									*/
/*	Heuristics for layout objects in GELO				*/
/*									*/
/************************************************************************/
/*	Copyright 1989 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"
#include "gelo_layout.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define TEMP_BASE	1
#define TEMP_MAX	10
#define TEMP_INCR	2

#define ANNEAL_COUNT(x,y)	(x)*(y)*2

#define INTERSECT_COST	10

#define check(i,j)	checkp[(i)*checkct+(j)]

#define SWAP(p1,p2)	{ int xxx; xxx = p1; p1 = p2; p2 = xxx; }

#define OC_TOP			010
#define OC_BOTTOM		004
#define OC_RIGHT		002
#define OC_LEFT 		001
#define OC_INSIDE		000
#define OC_ACCEPT(c1,c2)	((c1|c2) == 0)
#define OC_REJECT(c1,c2)	((c1&c2) != 0)





/************************************************************************/
/*									*/
/*	Forward Definitions						*/
/*									*/
/************************************************************************/


static	void		heuristic_dfsinsert();
static	void		dfsset();
static	void		heuristic_bfsinsert();
static	void		bfsset();
static	void		heuristic_avginsert();
static	void		avgset();
static	void		nextposition();
static	Integer 	dfsorder();
static	Integer 	dfsorderset();
static	Integer 	bfsorder();
static	Integer 	bfsorderset();
static	void		start_portals();
static	void		finish_portals();
static	void		conn_heuristic_rect();
static	Sequence	connect_path();
static	Integer 	channel_pos();
static	void		fix_pivots();
static	Integer 	channel_location();
static	void		conn_heuristic_pert();
static	Boolean 	minimize_pivots();
static	Boolean 	check_clear();
static	Integer 	check_intersect();
static	Integer 	get_outcode();
static	void		get_coords();
static	void		minimize();
static	Boolean 	swap_comp();
static	Integer 	comp_cost();
static	Integer 	path_cost();
static	void		anneal();
static	Boolean 	random_swap();
static	GELO_OBJECT	component_of();
static	void		get_conn_pos();





/************************************************************************/
/*									*/
/*	GELO_heur_init -- module initialization 			*/
/*									*/
/************************************************************************/


void
GELO_heur_init()
{
   ITRACE("GELO_heur_init");
};




/************************************************************************/
/*									*/
/*	GELO_heur_layout -- do the general layout positions		*/
/*									*/
/************************************************************************/


void
GELO_heur_layout(o,mxx,mxy)
   GELO_OBJECT o;
   Integer mxx,mxy;
{
   Sequence l,la;
   COMPONENT * c;
   Integer i,j;
   GELO_METHOD mthd;
   GELO_CONNECT conn;
   Integer dx,dy,bi;
   GELO_OBJECT co;
   COMPONENT ** checkp;
   Integer checkct;
   Boolean done;

   ITRACE("GELO_heur_layout 0x%x %d %d",o,mxx,mxy);

   mthd = METHOD(o);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->done = FALSE;
      c->pos_x = -1;
      c->pos_y = -1;
      c->cost = -1;
      if (c->from_arcs != NULL) LFREE(c->from_arcs);
      c->from_arcs = NULL;
      if (c->to_arcs != NULL) LFREE(c->to_arcs);
      c->to_arcs = NULL;
      if (c->all_arcs != NULL) LFREE(c->all_arcs);
      c->all_arcs = NULL;
    };

   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      GELO_conn_set_from_data(conn,NULL);
      GELO_conn_set_to_data(conn,NULL);
      GELO_clear_pivots(conn);
      forin (c,COMPONENT *,la,COMPONENTS(o)) {
	 i = 0;
	 co = component_of(o,GELO_inq_connect_from(conn));
	 if (co == c->object) {
	    i = 1;
	    c->from_arcs = CONS(conn,c->from_arcs);
	    GELO_conn_set_from_data(conn,c);
	  };
	 co = component_of(o,GELO_inq_connect_to(conn));
	 if (co == c->object) {
	    i = 2;
	    c->to_arcs = CONS(conn,c->to_arcs);
	    GELO_conn_set_to_data(conn,c);
	  };
	 if (i != 0) c->all_arcs = CONS(conn,c->all_arcs);
       };
    };

   if ((mthd & (GELO_METHOD_BIT_HOR|GELO_METHOD_BIT_VERT)) ==
	  (GELO_METHOD_BIT_HOR|GELO_METHOD_BIT_VERT))
      mthd &= ~(GELO_METHOD_BIT_HOR|GELO_METHOD_BIT_VERT);

   if ((mthd & (GELO_METHOD_BIT_HOR|GELO_METHOD_BIT_VERT)) == 0 &&
	  (mthd & GELO_METHOD_PERTBITS) == 0) {
      if (CONNECTS(o) == NULL) {
	 mthd |= (mxx < mxy) ? GELO_METHOD_BIT_VERT : GELO_METHOD_BIT_HOR;
       }
      else {
	 mthd |= (mxx < mxy) ? GELO_METHOD_BIT_HOR : GELO_METHOD_BIT_VERT;
       };
    };

   done = FALSE;
   if ((mthd & GELO_METHOD_BIT_GIOTTO) != 0) {
      if (GELO_giotto_layout(o,mthd)) done = TRUE;
      else {
	 i = METHOD(o);
	 i &= ~GELO_METHOD_BIT_GIOTTO;
	 GELOdefine_layout_method(o,i);
       };
    }
   else if ((mthd & GELO_METHOD_PERTBITS) != 0) {
      if (GELO_grid_layout(o,mthd)) done = TRUE;
      else {
	 i = METHOD(o);
	 i &= ~GELO_METHOD_PERTBITS;
	 GELOdefine_layout_method(o,i);
       };
    }
   else if ((mthd & GELO_METHOD_BIT_LEVEL) != 0) {
      if (GELO_level_layout(o,mthd)) done = TRUE;
      else {
	 i = METHOD(o);
	 i &= ~GELO_METHOD_BIT_LEVEL;
	 GELOdefine_layout_method(o,i);
       };
    };

   if (done) {
      checkct = 0;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (checkct < c->pos_x) checkct = c->pos_x;
	 else if (checkct < c->pos_y) checkct = c->pos_y;
       };
    }
   else checkct = 2*LENGTH(COMPONENTS(o));
   checkct += 4;
   i = checkct*checkct*sizeof(COMPONENT *);
   checkp = (COMPONENT **) alloca(i);
   bzero(checkp,i);

   if (done) {
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (!c->portal && (mthd & GELO_METHOD_BIT_HOR) != 0) {
	    dx = c->pos_x;
	    c->pos_x = c->pos_y;
	    c->pos_y = dx;
	  };
	 check(c->pos_x,c->pos_y) = c;
       };
      if ((mthd & GELO_METHOD_BIT_HOR) != 0) {
	 forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
	    la = LCOPY(GELO_connect_inq_pivots(conn));
	    GELO_clear_pivots(conn);
	    while (la != NULL) {
	       i = CAR(Integer,la);
	       la = CDRF(la);
	       j = CAR(Integer,la);
	       la = CDRF(la);
	       GELO_connect_pivot(conn,j,i);
	     };
	  };
       };
    }
   else {
      if (mthd & GELO_METHOD_BIT_VERT) {
	 dx = 0;
	 dy = 1;
       }
      else {
	 dx = 1;
	 dy = 0;
       };
      bi = (mthd & GELO_METHOD_BIT_2WAY ? TRUE : FALSE);

      if (mthd & GELO_METHOD_BIT_AVG) {
	 heuristic_avginsert(dx,dy,bi,mthd,o,checkp,checkct);
       }
      else if (mthd & GELO_METHOD_BIT_BFS) {
	 heuristic_bfsinsert(dx,dy,bi,mthd,o,checkp,checkct);
       }
      else {
	 heuristic_dfsinsert(dx,dy,bi,mthd,o,checkp,checkct);
       };

      if (mthd & GELO_METHOD_BIT_ANNEAL) {
	 anneal(o,mthd,checkp,checkct);
       };

      if (mthd & GELO_METHOD_BIT_OPT) {
	 minimize(o,mthd,checkp,checkct);
       };
    };
};





/************************************************************************/
/*									*/
/*	GELO_heur_connect -- do the connections 		       */
/*									*/
/************************************************************************/


void
GELO_heur_connect(gw,o,wx,wy,copos,rwpos)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   Integer wx,wy;
   Integer copos[];
   Integer rwpos[];
{
   GELO_CONNECT conn;
   Sequence l;
   GELO_METHOD mthd;
   GELO_CONN_METHOD cmthd;
   Boolean fg,stfg;

   ITRACE("GELO_heur_connect 0x%x 0x%x %d %d",gw,o,wx,wy);

   mthd = METHOD(o);
   cmthd = CONNMETHOD(o);
   fg = ((mthd & (GELO_METHOD_PERTBITS|GELO_METHOD_BIT_GIOTTO)) != 0);

   if (cmthd == GELO_CONN_METHOD_PERT) cmthd = GELO_CONN_METHOD_DIRECT_RECT;

   stfg = (cmthd == GELO_CONN_METHOD_DIRECT_RECT);

   if (cmthd == GELO_CONN_METHOD_DIRECT_RECT) {
      if (fg) cmthd = GELO_CONN_METHOD_PERT;
      else cmthd = GELO_CONN_METHOD_DIRECT_RECT;
    };


   if (cmthd == GELO_CONN_METHOD_DIRECT) {
      forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
	 GELO_clear_pivots(conn);
       };
    }
   else if (fg) {
      conn_heuristic_pert(gw,stfg,o,wx,wy,copos,rwpos);
    }
   else {
      conn_heuristic_rect(gw,stfg,o,wx,wy,copos,rwpos);
    };
};





/************************************************************************/
/*									*/
/*		Depth-first heuristic					*/
/*									*/
/*	heuristic_dfsinsert -- insert nodes in dfs order		*/
/*	dfsset -- position a single node				*/
/*									*/
/************************************************************************/


static void
heuristic_dfsinsert(dx,dy,bi,mthd,o,checkp,checkct)
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   GELO_OBJECT o;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;
   Integer x,y;

   DTRACE("heuristic_dfsinsert %d %d %d 0x%x 0x%x",dx,dy,bi,mthd,o);

   start_portals(o,checkp,checkct);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->done) continue;
      x = y = 1;
      while (check(x,y) != NULL) {
	 x += dx;
	 y += dy;
       };
      dfsset(o,c,x,y,dx,dy,bi,mthd,checkp,checkct);
    };

   finish_portals(o,checkp,checkct);
};




static void
dfsset(o,c,x,y,dx,dy,bi,mthd,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT * c;
   Integer x,y;
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   COMPONENT ** checkp;
   Integer checkct;
{
   Integer i;
   Integer x1,y1;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;

   DTRACE("dfsset 0x%x 0x%x %d %d %d %d %d 0x%x",o,c,x,y,dx,dy,bi,mthd);

   c->pos_x = x;
   c->pos_y = y;
   c->done = TRUE;
   check(x,y) = c;

   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && !c1->done) {
	 for (i = 0; ; ++i) {
	    if (i == 1 && (mthd & GELO_METHOD_BIT_FWD) != 0) continue;
	    nextposition(i,x,y,dx,dy,&x1,&y1);
	    if (check(x1,y1) == NULL) break;
	  };
	 dfsset(o,c1,x1,y1,dx,dy,bi,mthd,checkp,checkct);
       };
    };
};





/************************************************************************/
/*									*/
/*		Breadth-first heuristic 				*/
/*									*/
/*	heuristic_bfsinsert -- insert in breadth-first order		*/
/*	bfsset -- position a single node				*/
/*									*/
/************************************************************************/


static void
heuristic_bfsinsert(dx,dy,bi,mthd,o,checkp,checkct)
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   GELO_OBJECT o;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;
   Integer x,y;

   DTRACE("heuristic_bfsinsert %d %d %d 0x%x 0x%x",dx,dy,bi,mthd,o);

   start_portals(o,checkp,checkct);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->done) continue;
      x = y = 1;
      while (check(x,y) != NULL) {
	 x += dx;
	 y += dy;
       };
      c->pos_x = x;
      c->pos_y = y;
      c->done = TRUE;
      check(x,y) = c;
      bfsset(o,c,x,y,dx,dy,bi,mthd,checkp,checkct);
    };

   finish_portals(o,checkp,checkct);
};




static void
bfsset(o,c,x,y,dx,dy,bi,mthd,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT * c;
   Integer x,y;
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   COMPONENT ** checkp;
   Integer checkct;
{
   Integer i;
   Integer x1,y1;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;
   COMPONENT ** queue;
   Integer qct;

   DTRACE("bfsset 0x%x 0x%x %d %d %d %d %d 0x%x",o,c,x,y,dx,dy,bi,mthd);

   i = LENGTH(c->all_arcs)+10;
   queue = (COMPONENT **) alloca(i*sizeof(COMPONENT *));

   qct = 0;
   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && !c1->done) {
	 for (i = 0; ; ++i) {
	    if (i == 1 && (mthd & GELO_METHOD_BIT_FWD) != 0) continue;
	    nextposition(i,x,y,dx,dy,&x1,&y1);
	    if (check(x1,y1) == NULL) break;
	  };
	 queue[qct++] = c1;
	 check(x1,y1) = c1;
	 c1->done = TRUE;
	 c1->pos_x = x1;
	 c1->pos_y = y1;
       };
    };

   for (i = 0; i < qct; ++i) {
      c1 = queue[i];
      bfsset(o,c1,c1->pos_x,c1->pos_y,dx,dy,bi,mthd,checkp,checkct);
    };
};





/************************************************************************/
/*									*/
/*		Average fit heuristic					*/
/*									*/
/*	heuristic_avginsert -- insert nodes in average fit oredr	*/
/*	avgset -- position a single node				*/
/*									*/
/************************************************************************/


static void
heuristic_avginsert(dx,dy,bi,mthd,o,checkp,checkct)
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   GELO_OBJECT o;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;
   COMPONENT ** queue;
   Integer ct,i;

   DTRACE("heuristic_avginsert %d %d %d %d 0x%x",dx,dy,bi,mthd,o);

   i = LENGTH(COMPONENTS(o))+10;
   queue = (COMPONENT **) alloca(i*sizeof(COMPONENT *));

   start_portals(o,checkp,checkct);

   if (mthd & GELO_METHOD_BIT_DFS) {
      ct = dfsorder(o,bi,queue);
    }
   else if (mthd & GELO_METHOD_BIT_BFS) {
      ct = bfsorder(o,bi,queue);
    }
   else {
      ct = 0;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 queue[ct++] = c;
       };
    };

   for (i = 0; i < ct; ++i) {
      c = queue[i];
      avgset(o,c,dx,dy,bi,mthd,checkp,checkct);
    };

   finish_portals(o,checkp,checkct);
};




static void
avgset(o,c,dx,dy,bi,mthd,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT * c;
   Integer dx,dy;
   Boolean bi;
   GELO_METHOD mthd;
   COMPONENT ** checkp;
   Integer checkct;
{
   Sequence l;
   Integer i,x,y;
   COMPONENT * c1;
   Integer xsum,ysum,ct;
   Integer x1,y1;
   GELO_CONNECT conn;

   DTRACE("avgset 0x%x 0x%x %d %d %d 0x%x",o,c,dx,dy,bi,mthd);

   if (c->done) return;

   ct = 0;
   xsum = ysum = 0;
   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && c1->done && !c1->portal) {
	 xsum += c1->pos_x;
	 ysum += c1->pos_y;
	 ++ct;
       };
    };

   if (ct == 0) {
      x = y = 1;
      while (check(x,y) != NULL) {
	 x += dx;
	 y += dy;
       };
    }
   else {
      x = xsum/ct;
      y = ysum/ct;
    };

   if (check(x,y) != NULL) {
      for (i = 0; ; ++i) {
	 if (i == 1 && (mthd & GELO_METHOD_BIT_FWD) != 0) continue;
	 nextposition(i,x,y,dx,dy,&x1,&y1);
	 if (check(x1,y1) == NULL) break;
       };
    }
   else {
      x1 = x;
      y1 = y;
    };

   c->pos_x = x1;
   c->pos_y = y1;
   c->done = TRUE;
   check(x1,y1) = c;
};




/************************************************************************/
/*									*/
/*	nextposition -- find next position from current one		*/
/*									*/
/************************************************************************/


static void
nextposition(i,x,y,dx,dy,xp,yp)
   Integer i;
   Integer x,y;
   Integer dx,dy;
   Integer *xp, *yp;
{
   Integer i0,i1,i2;

   DTRACE("nextposition %d %d %d %d %d",i,x,y,dx,dy);

   i1 = i;
   i2 = 1;
   for (i0 = 3; i1-i0 >= 0; i0 += 2) {
      i1 -= i0;
      i2 += 1;
    };

   if ((i1 & 1) != dx) {
      y += i2;
      x += i1/2;
    }
   else {
      x += i2;
      y += i1/2;
    };

   *xp = x;
   *yp = y;
};






/************************************************************************/
/*									*/
/*	dfsorder -- compute dfs order of nodes				*/
/*	dfsorderset -- set order for dfs				*/
/*									*/
/************************************************************************/


static Integer
dfsorder(o,bi,queue)
   GELO_OBJECT o;
   Boolean bi;
   COMPONENT * queue[];
{
   Sequence l;
   COMPONENT * c;
   Integer ct;

   DTRACE("dfsorder 0x%x %d",o,bi);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->done = FALSE;
    };

   ct = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      ct = dfsorderset(o,c,bi,ct,queue);
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->done = FALSE;
    };

   return ct;
};




static Integer
dfsorderset(o,c,bi,ct,queue)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer ct;
   COMPONENT * queue[];
{
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;

   DTRACE("dfsorderset 0x%x 0x%x %d %d",o,c,bi,ct);

   if (c->done) return ct;

   queue[ct++] = c;
   c->done = TRUE;

   forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && !c1->done) ct = dfsorderset(o,c1,bi,ct,queue);
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	bfsorder -- compute bfs order of nodes				*/
/*	bfsorderset -- set order for bfs				*/
/*									*/
/************************************************************************/


static Integer
bfsorder(o,bi,queue)
   GELO_OBJECT o;
   Boolean bi;
   COMPONENT * queue[];
{
   Sequence l;
   COMPONENT * c;
   Integer ct;

   DTRACE("bfsorder 0x%x %d",o,bi);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->done = FALSE;
    };

   ct = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      ct = bfsorderset(o,c,bi,ct,queue);
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->done = FALSE;
    };

   return ct;
};




static Integer
bfsorderset(o,c,bi,ct,queue)
   GELO_OBJECT o;
   COMPONENT * c;
   Boolean bi;
   Integer ct;
   COMPONENT * queue[];
{
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1;
   Integer i,j;

   DTRACE("bfsorderset 0x%x 0x%x %d %d",o,c,bi,ct);

   if (c->done) return ct;

   i = ct;
   queue[ct++] = c;
   c->done = TRUE;

   for (j = i; j < ct; ++j) {
      c = queue[j];
      if (c->done) continue;
      forin (conn,GELO_CONNECT,l,(bi ? c->all_arcs : c->from_arcs)) {
	 c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
	 if (bi && c1 == c) c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
	 if (c1 != NULL && !c1->done) {
	    queue[ct++] = c1;
	    c1->done = TRUE;
	  };
       };
    };

   return ct;
};





/************************************************************************/
/*									*/
/*	start_portals -- setup portals for layout			*/
/*	finish_portals -- assign portal positions for layout		*/
/*									*/
/************************************************************************/


/*ARGSUSED*/

static void
start_portals(o,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;

   DTRACE("start_portals 0x%x",o);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->portal) {
	 c->done = TRUE;
       };
    };
};





static void
finish_portals(o,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;
   Integer x,y,mx,my;
   Integer z,i;
   Boolean dx;
   GELO_PORT port;

   DTRACE("finish_portals 0x%x",o);

   mx = my = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (!c->portal) {
	 if (mx < c->pos_x) mx = c->pos_x;
	 if (my < c->pos_y) my = c->pos_y;
       };
    };

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->portal) {
	 port = GELO_portal_location(c->object,c->port);
	 dx = FALSE;
	 switch (port) {
	    case GELO_PORT_LEFT :
	    case GELO_PORT_TOP_LEFT :
	    case GELO_PORT_BOTTOM_LEFT :
	    case GELO_PORT_LEFT_SEQ :
	       x = 0;
	       break;
	    case GELO_PORT_RIGHT :
	    case GELO_PORT_TOP_RIGHT :
	    case GELO_PORT_BOTTOM_RIGHT :
	    case GELO_PORT_RIGHT_SEQ :
	       x = mx+1;
	       break;
	    default :
	       x = 1;
	       dx = TRUE;
	       break;
	  };
	 switch (port) {
	    case GELO_PORT_TOP :
	    case GELO_PORT_TOP_LEFT :
	    case GELO_PORT_TOP_RIGHT :
	    case GELO_PORT_TOP_SEQ :
	       y = 0;
	       break;
	    case GELO_PORT_BOTTOM :
	    case GELO_PORT_BOTTOM_LEFT :
	    case GELO_PORT_BOTTOM_RIGHT :
	    case GELO_PORT_BOTTOM_SEQ :
	       y = my+1;
	       break;
	    default :
	       y = 1;
	       break;
	  };
	 for (i = 0; ; ++i) {
	    z = ((i&1) ? i : -i);
	    if (dx) x += z;
	    else y += z;
	    if (x >= 0 && y >= 0 && check(x,y) == NULL) break;
	  };
	 c->pos_x = x;
	 c->pos_y = y;
	 check(x,y) = c;
       };
    };
};





/************************************************************************/
/*									*/
/*		Rectangular layout heuristics				*/
/*									*/
/*	conn_heuristic_rect -- entry for layout connections		*/
/*	connect_path -- establish connection for a path 		*/
/*	channel_pos -- location for channel				*/
/*	fix_pivots -- fix pivots for a channel by setting their locs	*/
/*	channel_location -- get absolute location of a channel		*/
/*									*/
/************************************************************************/


static void
conn_heuristic_rect(gw,straight,o,wx,wy,copos,rwpos)
   GELO_WINDOW gw;
   Boolean straight;
   GELO_OBJECT o;
   Integer wx,wy;
   Integer copos[];
   Integer rwpos[];
{
   Integer * cop;
   Integer * com;
   Integer * rwp;
   Integer * rwm;
   Sequence * pvt;
   Integer i;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1, *c2, *c;
   Integer j;
   Integer checkct;
   GELO_PORT p1,p2;

   ITRACE("conn_heuristic_rect 0x%x %d 0x%x 0x%x %d %d",gw,straight,o,wx,wy);

   checkct = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (checkct < c->pos_x) checkct = c->pos_x;
      else if (checkct < c->pos_y) checkct = c->pos_y;
    };
   checkct += 4;

   cop = (Integer *) alloca(checkct*sizeof(Integer));
   com = (Integer *) alloca(checkct*sizeof(Integer));
   rwp = (Integer *) alloca(checkct*sizeof(Integer));
   rwm = (Integer *) alloca(checkct*sizeof(Integer));

   i = LENGTH(CONNECTS(o))+4;
   pvt = (Sequence *) alloca(i*sizeof(Sequence));

   for (i = 0; i < checkct; ++i) {
      cop[i] = com[i] = rwp[i] = rwm[i] = 0;
    };

   i = 0;
   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      pvt[i] = NULL;
      ++i;
      c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 == NULL) continue;
      c2 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c2 == NULL) continue;
      p1 = GELO_inq_connect_from_port(conn);
      p2 = GELO_inq_connect_to_port(conn);
      if (!straight || !check_clear(o,c1,p1,c2,p2,copos,rwpos)) {
	 pvt[i-1] = connect_path(conn,c1,c2,cop,com,rwp,rwm);
       };
    };

   i = 0;
   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      GELO_clear_pivots(conn);
      if (pvt[i] != NULL) {
	 fix_pivots(gw,o,conn,pvt[i],rwp,rwm,cop,com,rwpos,copos,wx,wy);
       };
      ++i;
    };
};





static Sequence
connect_path(conn,c1,c2,cop,com,rwp,rwm)
   GELO_CONNECT conn;
   COMPONENT * c1, * c2;
   Integer cop[], com[], rwp[], rwm[];
{
   Integer co,rw,tco,trw,mco,mrw;
   Integer fdir,tdir,mdir;
   Sequence l;

   DTRACE("connect_path 0x%x 0x%x 0x%x 0x%x",conn,c1,c2);

   co = rw = -1;
   switch (GELO_inq_connect_from_port(conn)) {
      case GELO_PORT_LEFT :
      case GELO_PORT_LEFT_SEQ :
	 co = c1->pos_x;
	 fdir = 1;
	 break;
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_TOP :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_TOP_SEQ :
	 rw = c1->pos_y;
	 fdir = 1;
	 break;
      case GELO_PORT_RIGHT :
      case GELO_PORT_RIGHT_SEQ :
	 co = c1->pos_x + 1;
	 fdir = -1;
	 break;
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_BOTTOM :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_BOTTOM_SEQ :
      case GELO_PORT_CENTER :
	 rw = c1->pos_y+1;
	 fdir = -1;
	 break;
    };
   tco = trw = -1;
   switch (GELO_inq_connect_to_port(conn)) {
      case GELO_PORT_LEFT :
      case GELO_PORT_LEFT_SEQ :
	 tco = c2->pos_x;
	 tdir = 1;
	 break;
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_TOP :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_TOP_SEQ :
	 trw = c2->pos_y;
	 tdir = 1;
	 break;
      case GELO_PORT_RIGHT :
      case GELO_PORT_RIGHT_SEQ :
	 tco = c2->pos_x + 1;
	 tdir = -1;
	 break;
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_BOTTOM :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_BOTTOM_SEQ :
      case GELO_PORT_CENTER :
	 trw = c2->pos_y+1;
	 tdir = -1;
	 break;
    };
   mco = mrw = -1;
   if (co  < 0 && tco < 0 && rw != trw) {
      if (c2->pos_x > 0 && c2->portal) {
	 mco = c2->pos_x;
	 mdir = -1;
       }
      else if (c2->pos_x < c1->pos_x || c2->pos_x == 0 ||
	     (c2->pos_x == c1->pos_x && rw < trw)) {
	 mco = c2->pos_x + 1;
	 mdir = 1;
       }
      else {
	 mco = c2->pos_x;
	 mdir = -1;
       };
    }
   else if (rw < 0 && trw < 0 && co != tco) {
      if (c2->pos_y > 0 && c2->portal) {
	 mrw = c2->pos_y;
	 mdir = -1;
       }
      else if (c2->pos_y < c1->pos_y || c2->pos_y == 0 ||
	     (c2->pos_y == c1->pos_y && co < tco)) {
	 mrw = c2->pos_y + 1;
	 mdir = 1;
       }
      else {
	 mrw = c2->pos_y;
	 mdir = -1;
       };
    };

   l = NULL;

   if (co >= 0) {
      l = APPEND(1,l);
      l = APPEND(co,l);
      l = APPEND(channel_pos(co,fdir,cop,com),l);
    }
   else if (rw >= 0) {
      l = APPEND(2,l);
      l = APPEND(rw,l);
      l = APPEND(channel_pos(rw,fdir,rwp,rwm),l);
    };
   if (mco >= 0) {
      l = APPEND(1,l);
      l = APPEND(mco,l);
      l = APPEND(channel_pos(mco,mdir,cop,com),l);
    }
   else if (mrw >= 0) {
      l = APPEND(2,l);
      l = APPEND(mrw,l);
      l = APPEND(channel_pos(mrw,mdir,rwp,rwm),l);
    };
   if (tco >= 0) {
      if (co != tco || mrw >= 0) {
	 l = APPEND(1,l);
	 l = APPEND(tco,l);
	 l = APPEND(channel_pos(tco,tdir,cop,com),l);
       };
      l = APPEND(4,l);
      l = APPEND(tco,l);
      l = APPEND(0,l);
    }
   else if (trw >= 0) {
      if (rw != trw || mco >= 0) {
	 l = APPEND(2,l);
	 l = APPEND(trw,l);
	 l = APPEND(channel_pos(trw,tdir,rwp,rwm),l);
       };
      l = APPEND(3,l);
      l = APPEND(trw,l);
      l = APPEND(0,l);
    };

   return l;
};





static Integer
channel_pos(ch,dir,pct,mct)
   Integer ch;
   Integer dir;
   Integer pct[], mct[];
{
   Integer i;

   if (dir >= 0) {
      ++pct[ch];
      i = pct[ch];
    }
   else {
      ++mct[ch];
      i = -mct[ch];
    };

   DTRACE("channel_pos %d %d => %d",ch,dir,i);

   return i;
};





static void
fix_pivots(gw,o,conn,pvts,rwp,rwm,cop,com,rwpos,copos,wx,wy)
   GELO_WINDOW gw;
   GELO_OBJECT o;
   GELO_CONNECT conn;
   Sequence pvts;
   Integer rwp[],rwm[],cop[],com[];
   Integer rwpos[],copos[];
   Integer wx,wy;
{
   Integer x,y,nx,ny,dx,dy;
   Integer fg,loc,pos;

   DTRACE("fix_pivots 0x%x 0x%x 0x%x 0x%x %d %d",gw,o,conn,pvts,wx,wy);

   if (!GELO_inq_connect_from_location(gw,o,conn,&x,&y)) return;
   if (!GELO_inq_connect_to_location(gw,o,conn,&nx,&ny)) return;
   dx = 0;
   dy = 0;

   while (pvts != NULL) {
      fg = CAR(Integer,pvts);
      pvts = CDRF(pvts);
      loc = CAR(Integer,pvts);
      pvts = CDRF(pvts);
      pos = CAR(Integer,pvts);
      pvts = CDRF(pvts);
      switch (fg) {
	 case 1 :
	    x = channel_location(pos,cop[loc],com[loc],wx,copos[loc]+dx);
	    break;
	 case 2 :
	    y = channel_location(pos,rwp[loc],rwm[loc],wy,rwpos[loc]+dy);
	    break;
	 case 3 :
	    x = nx;
	    break;
	 case 4 :
	    y = ny;
	    break;
       };
      GELO_connect_pivot(conn,x,y);
    };
};





static Integer
channel_location(pos,pct,mct,width,base)
   Integer pos;
   Integer pct,mct;
   Integer width;
   Integer base;
{
   Integer dlta;

   dlta = width/(pct+mct+1);

   if (pos > 0) {
      dlta *= pos;
    }
   else {
      dlta = width+dlta*pos;
    };

   DTRACE("channel_location %d %d %d %d %d => %d",pos,pct,mct,width,base,base+dlta);

   return base+dlta;
};





/************************************************************************/
/*									*/
/*	conn_heuristic_pert -- pert connection heuristics		*/
/*									*/
/************************************************************************/


static void
conn_heuristic_pert(gw,straight,o,wx,wy,copos,rwpos)
   GELO_WINDOW gw;
   Boolean straight;
   GELO_OBJECT o;
   Integer wx,wy;
   Integer copos[];
   Integer rwpos[];
{
   Integer * cop;
   Integer * com;
   Integer * rwp;
   Integer * rwm;
   Sequence pl;
   Integer i;
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1, *c2, *c;
   Integer j,k,vx,vy;
   Integer mxc,mxr;
   GELO_PORT pt1,pt2;
   Integer checkct;

   DTRACE("conn_heuristic_pert 0x%x 0x%x %d %d",gw,o,wx,wy);

   checkct = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (checkct < c->pos_x) checkct = c->pos_x;
      else if (checkct < c->pos_y) checkct = c->pos_y;
    };
   checkct += 4;

   cop = (Integer *) alloca(checkct*sizeof(Integer));
   com = (Integer *) alloca(checkct*sizeof(Integer));
   rwp = (Integer *) alloca(checkct*sizeof(Integer));
   rwm = (Integer *) alloca(checkct*sizeof(Integer));

   for (i = 0; i < checkct; ++i) {
      cop[i] = com[i] = rwp[i] = rwm[i] = 0;
    };

   mxc = mxr = 0;
   for (i = 1; i < checkct; ++i) {
      if (mxc == 0 && copos[i] == 0) mxc = i;
      if (mxr == 0 && rwpos[i] == 0) mxr = i;
      if (mxc != 0 && mxr != 0) break;
    };

   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 == NULL) continue;
      c2 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c2 == NULL) continue;

      pl = GELO_connect_inq_pivots(conn);
      pt1 = GELO_inq_connect_from_port(conn);
      pt2 = GELO_inq_connect_to_port(conn);

      if (pl != NULL && straight && check_clear(o,c1,pt1,c2,pt2,copos,rwpos)) {
	 GELO_clear_pivots(conn);
       }
      else if (pl != NULL && straight) {
	 if (!minimize_pivots(o,conn,c1,c2,copos,rwpos)) {
	    GELO_clear_pivots(conn);
	    pl = connect_path(conn,c1,c2,cop,com,rwp,rwm);
	    fix_pivots(gw,o,conn,pl,rwp,rwm,cop,com,rwpos,copos,wx,wy);
	    continue;
	  };
       }
      else if (pl != NULL) {
	 GELO_conn_set_ports(conn,c1,c2,METHOD(o),CONNMETHOD(o),TRUE);
       };

      pl = GELO_connect_inq_pivots(conn);
      if (pl == NULL) {
	 if (c1 == c2) {
	    pl = connect_path(conn,c1,c2,cop,com,rwp,rwm);
	    fix_pivots(gw,o,conn,pl,rwp,rwm,cop,com,rwpos,copos,wx,wy);
	  };
       }
      else {
	 pl = LCOPY(pl);
	 GELO_clear_pivots(conn);
	 while (!EMPTY(pl)) {
	    j = CAR(Integer,pl);
	    pl = CDRF(pl);
	    k = CAR(Integer,pl);
	    pl = CDRF(pl);
	    if (j+1 < mxc) vx = (copos[j] + copos[j+1])/2;
	    else vx = copos[mxc-1] + 2*(j-mxc+2);
	    if (k+1 < mxr) vy = (rwpos[k] + rwpos[k+1])/2;
	    else vy = rwpos[mxr-1] + 2*(k-mxr+2);
	    GELO_connect_pivot(conn,vx,vy);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	minimize_pivots -- remove pert/giotto pivots			*/
/*									*/
/************************************************************************/


static Boolean
minimize_pivots(o,conn,c1,c2,copos,rwpos)
   GELO_OBJECT o;
   GELO_CONNECT conn;
   COMPONENT * c1;
   COMPONENT * c2;
   Integer * copos;
   Integer * rwpos;
{
   Sequence pl;
   COMPONENT dummy1,dummy2;
   COMPONENT * ca, * cb;
   Integer x,y;
   Boolean chng;
   Integer npvt;
   GELO_PORT p1,p2;

   for ( ; ; )	{
      pl = GELO_connect_inq_pivots(conn);

      if (pl == NULL) return TRUE;

      pl = LCOPY(pl);
      GELO_clear_pivots(conn);

      chng = FALSE;
      ca = c1;
      p1 = GELO_inq_connect_from_port(conn);
      npvt = 0;

      while (pl != NULL) {
	 x = CAR(Integer,pl);
	 y = CADR(Integer,pl);
	 pl = CDDR(pl);
	 if (EMPTY(pl)) {
	    cb = c2;
	    p2 = GELO_inq_connect_to_port(conn);
	  }
	 else {
	    dummy2.pos_x = CAR(Integer,pl);
	    dummy2.pos_y = CADR(Integer,pl);
	    dummy2.object = NULL;
	    cb = &dummy2;
	    p2 = GELO_PORT_CENTER;
	  };

	 if (check_clear(o,ca,p1,cb,p2,copos,rwpos)) {
	    chng = TRUE;
	  }
	 else {
	    ++npvt;
	    GELO_connect_pivot(conn,x,y);
	    dummy1.pos_x = x;
	    dummy1.pos_y = y;
	    dummy1.object = NULL;
	    ca = &dummy1;
	    p1 = GELO_PORT_CENTER;
	  };
       };

      if (npvt == 0 || !chng) break;
    };

   if (npvt > 0) {
      GELO_conn_set_ports(conn,c1,c2,METHOD(o),CONNMETHOD(o),TRUE);
      pl = GELO_connect_inq_pivots(conn);
      dummy1.pos_x = CAR(Integer,pl);
      dummy1.pos_y = CADR(Integer,pl);
      dummy1.object = NULL;
      if (!check_clear(o,c1,GELO_inq_connect_from_port(conn),&dummy1,GELO_PORT_CENTER,
			  copos,rwpos)) return FALSE;

      while (CDDR(pl) != NULL) pl = CDDR(pl);
      dummy1.pos_x = CAR(Integer,pl);
      dummy1.pos_y = CADR(Integer,pl);
      dummy1.object = NULL;
      if (!check_clear(o,&dummy1,GELO_PORT_CENTER,c2,GELO_inq_connect_to_port(conn),
			  copos,rwpos)) return FALSE;

      GELO_conn_set_ports(conn,c1,c2,METHOD(o),CONNMETHOD(o),TRUE);
    };

   return TRUE;
};





/************************************************************************/
/*									*/
/*	check_clear -- check for clear path between points		*/
/*	check_intersect -- intersect path with box			*/
/*	get_outcode -- get outcodes for intersection			*/
/*	get_coords -- get coordinates for component/port		*/
/*									*/
/************************************************************************/




static Boolean
check_clear(o,c1,p1,c2,p2,copos,rwpos)
   GELO_OBJECT o;
   COMPONENT * c1;
   GELO_PORT p1;
   COMPONENT * c2;
   GELO_PORT p2;
   Integer * copos;
   Integer * rwpos;
{
   Integer x1,y1,x2,y2;
   Integer xp1,yp1,xp2,yp2;
   Integer i;
   COMPONENT * c;
   Sequence l;

   get_coords(c1,p1,copos,rwpos,&x1,&y1);
   get_coords(c2,p2,copos,rwpos,&x2,&y2);

   xp1 = c1->pos_x;
   yp1 = c1->pos_y;
   xp2 = c2->pos_x;
   yp2 = c2->pos_y;

   if (xp1 > xp2) SWAP(xp1,xp2);
   if (yp1 > yp2) SWAP(yp1,yp2);

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (c->pos_x < xp1 || c->pos_y < yp1 || c->pos_x > xp2 || c->pos_y > yp2) continue;

      i = check_intersect(x1,y1,x2,y2,c,copos,rwpos);

      if (i == 0) continue;
      if (i == 1 && (c == c1 || c == c2)) continue;		/* point only	*/
      if (c == c1 && p1 == GELO_PORT_CENTER) continue;
      if (c == c2 && p2 == GELO_PORT_CENTER) continue;

      return FALSE;
    };

   return TRUE;
};






static Integer
check_intersect(x1,y1,x2,y2,c,copos,rwpos)
   Integer x1,y1,x2,y2;
   COMPONENT * c;
   Integer * copos;
   Integer * rwpos;
{
   Integer xmin,ymin,xmax,ymax;
   Integer oc1,oc2;

   get_coords(c,GELO_PORT_TOP_LEFT,copos,rwpos,&xmin,&ymin);
   get_coords(c,GELO_PORT_BOTTOM_RIGHT,copos,rwpos,&xmax,&ymax);

   oc1 = get_outcode(x1,y1,xmin,ymin,xmax,ymax);
   oc2 = get_outcode(x2,y2,xmin,ymin,xmax,ymax);

   if (!OC_REJECT(oc1,oc2)) {
      while (!OC_ACCEPT(oc1,oc2)) {
	 if (oc1 == OC_INSIDE) {
	    SWAP(x1,x2);
	    SWAP(y1,y2);
	    SWAP(oc1,oc2);
	  };
	 if (oc1 & OC_TOP) {
	    x1 += (x2-x1)*(ymax-y1)/(y2-y1);
	    y1 = ymax;
	  }
	 else if (oc1 & OC_BOTTOM) {
	    x1 += (x2-x1)*(ymin-y1)/(y2-y1);
	    y1 = ymin;
	  }
	 else if (oc1 & OC_RIGHT) {
	    y1 += (y2-y1)*(xmax-x1)/(x2-x1);
	    x1 = xmax;
	  }
	 else if (oc1 & OC_LEFT) {
	    y1 += (y2-y1)*(xmin-x1)/(x2-x1);
	    x1 = xmin;
	  };
	 oc1 = get_outcode(x1,y1,xmin,ymin,xmax,ymax);
	 if (OC_REJECT(oc1,oc2)) break;
       };
    };

   if (!OC_ACCEPT(oc1,oc2)) return 0;

   if (x1 == x2 && y1 == y2) return 1;

   return -1;
};





static Integer
get_outcode(x,y,xmin,ymin,xmax,ymax)
   Integer x,y;
   Integer xmin,ymin,xmax,ymax;
{
   Integer out;

   if (y > ymax) out = OC_TOP;
   else if (y < ymin) out = OC_BOTTOM;
   else out = OC_INSIDE;

   if (x > xmax) out |= OC_RIGHT;
   else if (x < xmin) out |= OC_LEFT;

   return out;
};





static void
get_coords(c,p,copos,rwpos,x,y)
   COMPONENT * c;
   GELO_PORT p;
   Integer * copos;
   Integer * rwpos;
   Integer *x, *y;
{
   Integer mxc,mxr;
   Integer i,j,k;
   Integer vx,vy,xs,ys;

   if (c->object == NULL) {
      mxc = mxr = 0;
      for (i = 1; ; ++i) {
	 if (mxc == 0 && copos[i] == 0) mxc = i;
	 if (mxr == 0 && rwpos[i] == 0) mxr = i;
	 if (mxc != 0 && mxr != 0) break;
       };

      j = c->pos_x;
      if (j+1 < mxc) vx = (copos[j] + copos[j+1])/2;
      else vx = copos[mxc-1] + 2*(j-mxc+2);
      k = c->pos_y;
      if (k+1 < mxr) vy = (rwpos[k] + rwpos[k+1])/2;
      else vy = rwpos[mxr-1] + 2*(k-mxr+2);
    }
   else {
      vx = GELO_inq_x_position(c->object);
      vy = GELO_inq_y_position(c->object);
      xs = GELO_inq_x_size(c->object);
      ys = GELO_inq_y_size(c->object);
      switch (p) {
	 case GELO_PORT_LEFT :
	 case GELO_PORT_TOP_LEFT :
	 case GELO_PORT_BOTTOM_LEFT :
	 case GELO_PORT_LEFT_SEQ :
	    break;
	 case GELO_PORT_TOP :
	 case GELO_PORT_BOTTOM :
	 case GELO_PORT_TOP_SEQ :
	 case GELO_PORT_BOTTOM_SEQ :
	    vx += xs/2;
	    break;
	 case GELO_PORT_RIGHT :
	 case GELO_PORT_TOP_RIGHT :
	 case GELO_PORT_BOTTOM_RIGHT :
	 case GELO_PORT_RIGHT_SEQ :
	    vx += xs;
	    break;
       };

      switch (p) {
	 case GELO_PORT_TOP :
	 case GELO_PORT_TOP_LEFT :
	 case GELO_PORT_TOP_RIGHT :
	 case GELO_PORT_TOP_SEQ :
	    break;
	 case GELO_PORT_LEFT :
	 case GELO_PORT_RIGHT :
	 case GELO_PORT_LEFT_SEQ :
	 case GELO_PORT_RIGHT_SEQ :
	    vy += ys/2;
	    break;
	 case GELO_PORT_BOTTOM :
	 case GELO_PORT_BOTTOM_LEFT :
	 case GELO_PORT_BOTTOM_RIGHT :
	 case GELO_PORT_BOTTOM_SEQ :
	    vy += ys;
	    break;
       };
    };

   *x = vx;
   *y = vy;
};







/************************************************************************/
/*									*/
/*	minimize -- using swaping to minimize costs			*/
/*									*/
/************************************************************************/


static void
minimize(o,m,checkp,checkct)
   GELO_OBJECT o;
   GELO_METHOD m;
   COMPONENT ** checkp;
   Integer checkct;
{
   COMPONENT * c;
   Sequence l;
   Integer i;
   Boolean fg;

   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      c->cost = comp_cost(o,c,m);
    };

   for (i = 0; i < 10; ++i) {
      fg = FALSE;
      forin (c,COMPONENT *,l,COMPONENTS(o)) {
	 if (!c->portal) {
	    while (swap_comp(o,c,m,checkp,checkct)) fg = TRUE;
	  };
       };
      if (!fg) break;
    };
};





/************************************************************************/
/*									*/
/*	swap_comp -- swap component using steepest descent		*/
/*									*/
/************************************************************************/


static Boolean
swap_comp(o,c,m,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT * c;
   GELO_METHOD m;
   COMPONENT ** checkp;
   Integer checkct;
{
   Integer x,y;
   Integer costs[10],cst0,cst1;
   Integer ct,i;
   COMPONENT * c1;
   Integer xd,yd;
   COMPONENT * comps[10];

   DTRACE("swap_comp 0x%x 0x%x 0x%x",o,c,m);

   x = c->pos_x;
   y = c->pos_y;

   ct = 0;
   for (xd = -1; xd <= 1; ++xd) {
      for (yd = -1; yd <= 1; ++yd) {
	 if ((abs(xd) == abs(yd)) || x+xd <= 0 || y+yd <= 0)
	    costs[ct] = 0;
	 else {
	    c1 = check(x+xd,y+yd);
	    if (c1 != NULL && c1->portal) costs[ct] = 0;
	    else {
	       comps[ct] = c1;
	       c->pos_x = x+xd;
	       c->pos_y = y+yd;
	       if (c1 == NULL) {
		  cst0 = comp_cost(o,c,m);
		  costs[ct] = c->cost-cst0;
		}
	       else {
		  c1->pos_x = x;
		  c1->pos_y = y;
		  cst0 = comp_cost(o,c,m);
		  cst1 = comp_cost(o,c1,m);
		  c1->pos_x = x+xd;
		  c1->pos_y = y+yd;
		  costs[ct] = c->cost - cst0 + c1->cost - cst1;
		};
	       c->pos_x = x;
	       c->pos_y = y;
	     };
	  };
	 ++ct;
       };
    };

   cst0 = 0;
   ct = -1;
   for (i = 0; i < 9; ++i) {
      if (costs[i] > cst0) {
	 ct = i;
	 cst0 = costs[i];
       };
    };

   if (ct < 0) return FALSE;

   yd = (ct % 3) -1;
   xd = (ct / 3) -1;

   c->pos_x = x+xd;
   c->pos_y = y+yd;

   if (comps[ct] != NULL) {
      c1 = comps[ct];
      c1->pos_x = x;
      c1->pos_y = y;
      c1->cost = comp_cost(o,c1,m);
    };

   c->cost = comp_cost(o,c,m);

   DTRACE("\tswap 0x%x %d %d 0x%x %d %d",c,x,y,c1,x+xd,y+yd);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	comp_cost -- get cost of a component				*/
/*	path_cost -- cost along a path					*/
/*									*/
/************************************************************************/


static Integer
comp_cost(o,c,m)
   GELO_OBJECT o;
   COMPONENT * c;
   GELO_METHOD m;
{
   Sequence l;
   GELO_CONNECT conn;
   COMPONENT * c1, * c2;
   Integer cost;
   Integer x0,y0,x1,y1;

   DTRACE("comp_cost 0x%x 0x%x",o,c);

   cost = 0;

   forin (conn,GELO_CONNECT,l,c->all_arcs) {
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      c2 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c1 != NULL && c2 != NULL) {
	 get_conn_pos(c1,GELO_inq_connect_to_port(conn),&x1,&y1);
	 get_conn_pos(c2,GELO_inq_connect_from_port(conn),&x0,&y0);
	 cost += path_cost(o,conn,m,x0,y0,x1,y1);
       };
    };

   return cost;
};





static Integer
path_cost(o,conn,m,x0,y0,x1,y1)
   GELO_OBJECT o;
   GELO_CONNECT conn;
   GELO_METHOD m;
   Integer x0,y0;
   Integer x1,y1;
{
   Integer cost,bcost;
   COMPONENT * c1, * c2;
   Sequence l;
   GELO_CONNECT conn1;
   Integer x2,y2,x3,y3;
   Float s,t,u;

   DTRACE("path_cost 0x%x %d %d %d %d",m,x0,y0,x1,y1);

   cost = abs(x1-x0);
   cost += abs(y1-y0);
   bcost = cost;

#ifndef NO_CROSSINGS
   forin (conn1,GELO_CONNECT,l,CONNECTS(o)) {
      if (conn1 == conn) continue;
      c1 = (COMPONENT *) GELO_conn_inq_from_data(conn1);
      c2 = (COMPONENT *) GELO_conn_inq_to_data(conn1);
      if (c1 == NULL || c2 == NULL) continue;
      get_conn_pos(c1,GELO_inq_connect_from_port(conn1),&x2,&y2);
      get_conn_pos(c2,GELO_inq_connect_to_port(conn1),&x3,&y3);
      u = (x3-x2)*(y1-y0) - (y3-y2)*(x1-x0);
      if (u == 0) continue;
      s = (x3-x2)*(y2-y0) - (y3-y2)*(x2-x0);
      s /= u;
      t = (x1-x0)*(y2-y0) - (y1-y0)*(x2-x0);
      t /= u;
      if (s > 0 && s < 1.0 && t > 0 && t < 1.0)
	 cost += INTERSECT_COST*bcost;
    };
#endif

   return cost;
};





/************************************************************************/
/*									*/
/*	anneal -- do simulated annealing in our own funny way		*/
/*									*/
/************************************************************************/


static void
anneal(o,m,checkp,checkct)
   GELO_OBJECT o;
   GELO_METHOD m;
   COMPONENT ** checkp;
   Integer checkct;
{
   Sequence l;
   COMPONENT * c;
   Integer xsz,ysz;
   Integer t,ct,temp;
   Integer i,x,y;

   xsz = 1;
   ysz = 1;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      if (!c->portal) {
	 if (c->pos_x > xsz) xsz = c->pos_x;
	 if (c->pos_y > ysz) ysz = c->pos_y;
       };
    };

   for (t = TEMP_BASE; t < TEMP_MAX; t += TEMP_INCR) {
      ct = ANNEAL_COUNT(xsz,ysz);
      temp = 1<<t;
      for (i = 0; i < ct; ++i) {
	 forin (c,COMPONENT *,l,COMPONENTS(o)) {
	    x = random() % xsz + 1;
	    y = random() % ysz + 1;
	    random_swap(o,c,x,y,temp,m,checkp,checkct);
	  };
       };
    };
};





/************************************************************************/
/*									*/
/*	random_swap -- try a random component swap			*/
/*									*/
/************************************************************************/


static Boolean
random_swap(o,c,nx,ny,pr,m,checkp,checkct)
   GELO_OBJECT o;
   COMPONENT * c;
   Integer nx,ny;
   Integer pr;
   GELO_METHOD m;
   COMPONENT ** checkp;
   Integer checkct;
{
   Integer x,y;
   Integer cost,cst0,cst1;
   COMPONENT * c1;

   DTRACE("random_swap 0x%x 0x%x 0x%x",o,c,m);

   x = c->pos_x;
   y = c->pos_y;

   c1 = check(nx,ny);
   if (c1 != NULL && c1->portal) return FALSE;
   else if (c == c1) return FALSE;

   c->pos_x = nx;
   c->pos_y = ny;
   if (c1 == NULL) {
      cst0 = comp_cost(o,c,m);
      cost = c->cost-cst0;
    }
   else {
      c1->pos_x = x;
      c1->pos_y = y;
      cst0 = comp_cost(o,c,m);
      cst1 = comp_cost(o,c1,m);
      c1->pos_x = nx;
      c1->pos_y = ny;
      cost = c->cost - cst0 + c1->cost - cst1;
    };
   c->pos_x = x;
   c->pos_y = y;

   if (cost < 0) {
      if ((random() & (pr-1)) != 0) return FALSE;
    };

   c->pos_x = nx;
   c->pos_y = ny;
   c->cost = cst0;

   if (c1 != NULL) {
      c1->pos_x = x;
      c1->pos_y = y;
      c1->cost = cst1;
    };

   check(nx,ny) = c;
   check(x,y) = c1;

   DTRACE("\tswap 0x%x %d %d 0x%x %d %d",c,x,y,c1,nx,ny);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	component_of -- find proper component				*/
/*									*/
/************************************************************************/


static GELO_OBJECT
component_of(o,co)
   GELO_OBJECT o;
   GELO_OBJECT co;
{
   while (co != NULL && co->parent != o)
      co = co->parent;

   return co;
};




/************************************************************************/
/*									*/
/*	get_conn_pos -- get position of a connection			*/
/*									*/
/************************************************************************/


static void
get_conn_pos(c,wh,xp,yp)
   COMPONENT * c;
   GELO_PORT wh;
   Integer *xp,*yp;
{
   Integer x,y;

   DTRACE("get_conn_pos 0x%x 0x%x",c,wh);

   x = c->pos_x * 5 + 2;
   y = c->pos_y * 5 + 2;

   switch (wh) {
      case GELO_PORT_LEFT :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_LEFT_SEQ :
	 break;
      case GELO_PORT_TOP :
      case GELO_PORT_BOTTOM :
      case GELO_PORT_TOP_SEQ :
      case GELO_PORT_BOTTOM_SEQ : x += 1;
	 break;
      case GELO_PORT_RIGHT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_RIGHT_SEQ :
	 x += 2;
	 break;
    };

   switch (wh) {
      case GELO_PORT_TOP :
      case GELO_PORT_TOP_LEFT :
      case GELO_PORT_TOP_RIGHT :
      case GELO_PORT_TOP_SEQ :
	 break;
      case GELO_PORT_LEFT :
      case GELO_PORT_RIGHT :
      case GELO_PORT_LEFT_SEQ :
      case GELO_PORT_RIGHT_SEQ :
	 y += 1;
	 break;
      case GELO_PORT_BOTTOM :
      case GELO_PORT_BOTTOM_LEFT :
      case GELO_PORT_BOTTOM_RIGHT :
      case GELO_PORT_BOTTOM_SEQ :
	 y += 2;
	 break;
    };

   *xp = x;
   *yp = y;
};






/* end of geloheur.c */
