/************************************************************************/
/*									*/
/*		gelogiotto.c						*/
/*									*/
/*	Interface for using GIOTTO drawing package for graph layout	*/
/*									*/
/************************************************************************/
/*	Copyright 1990 Brown University -- Steven P. Reiss		*/


#include "gelo_local.h"
#include "gelo_layout.h"




/************************************************************************/
/*									*/
/*	Parameters							*/
/*									*/
/************************************************************************/


#define GIOTTO_PATH		"%s/bin/giotto"
#define OLD_GIOTTO_PATH 	    "%s/bin/giotto.bak"



#define GIOTTO_VERSION		1
#define GIOTTO_SHAPE		5
#define GIOTTO_NODE		1

#define UNSET_POS		(-10000000)




/************************************************************************/
/*									*/
/*	Type definitions						*/
/*									*/
/************************************************************************/


typedef struct _G_SYM {
   COMPONENT * comp;
   Integer node;
   Integer arc;
} G_SYM_INFO, * G_SYM;



typedef struct _G_NODE {
   Integer sym;
   Integer arc;
   Integer type;
   COMPONENT * comp;
   Integer xpos,ypos;
} G_NODE_INFO, * G_NODE;



typedef struct _G_ARC {
   Integer ltarg;
   Integer lsrc;
   Integer len;
   Integer dir;
   Integer next;
   Integer comp;
   Integer fict;
   Integer src;
   Integer targ;
   Integer link;
   Integer snext;
} G_ARC_INFO, * G_ARC;





/************************************************************************/
/*									*/
/*	Local Storage							*/
/*									*/
/************************************************************************/


static	Boolean 	giotto_debug = 0;




/************************************************************************/
/*									*/
/*	Forward definitions						*/
/*									*/
/************************************************************************/


static	void		write_giotto_file();
static	Boolean 	run_giotto();
static	Boolean 	read_giotto_file();
static	void		set_node_pos();
static	Integer 	find_arc_tail();
static	void		giotto_pivots();




/************************************************************************/
/*									*/
/*	GELO_giotto_init -- module initialization			*/
/*									*/
/************************************************************************/


void
GELO_giotto_init()
{
   if (getenv("GIOTTO_DEBUG") != NULL) giotto_debug = 1;
};





/************************************************************************/
/*									*/
/*	GELO_giotto_layout -- layout a graph using GIOTTO		*/
/*									*/
/************************************************************************/


Boolean
GELO_giotto_layout(o,mthd)
   GELO_OBJECT o;
   GELO_METHOD mthd;
{
   FILE * otf;
   FILE * inf;
   Character filnam[64],tmpnam[64],outnam[64];
   Integer nnod,narc;
   COMPONENT * c,* c1;
   GELO_CONNECT conn;
   Integer * connidx;
   Sequence l;
   Boolean fg;

   strcpy(tmpnam,"/tmp/gelogioXXXXXX");
   mktemp(tmpnam);
   otf = fopen(tmpnam,"w");
   if (otf == NULL) return FALSE;
   fclose(otf);

   sprintf(filnam,"%s.mod",tmpnam);
   otf = fopen(filnam,"w");
   if (otf == NULL) {
      unlink(tmpnam);
      return FALSE;
    };
   write_giotto_file(otf,o);
   fclose(otf);

   sprintf(outnam,"%s.gri",tmpnam);

   fg = run_giotto(filnam,outnam);

   if (!giotto_debug) unlink(filnam);

   inf = fopen(outnam,"r");
   if (inf == NULL) fg = FALSE;

   if (fg) fg = read_giotto_file(inf,o);

   unlink(filnam);
   unlink(tmpnam);

   return fg;
};






/************************************************************************/
/*									*/
/*	write_giotto_file -- write input file for giotto		*/
/*									*/
/************************************************************************/


static void
write_giotto_file(otf,o)
   FILE * otf;
   GELO_OBJECT o;
{
   Integer nnod,narc;
   COMPONENT * c,* c1;
   GELO_CONNECT conn;
   Integer * connidx;
   Sequence l;
   Integer i;

   nnod = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      ++nnod;
      c->data = (Universal) 0;
    };

   i = LENGTH(CONNECTS(o));
   connidx = (Integer *) calloc((i+1)*2,sizeof(Integer));

   narc = 0;
   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      c = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c == NULL) continue;
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c1 == NULL) continue;
      if (c == c1) continue;

      ++narc;
      i = (Integer) c->data;
      if (i == 0) {
	 connidx[narc] = narc;
	 c->data = (Universal) narc;
       }
      else {
	 connidx[narc] = connidx[i];
	 connidx[i] = narc;
       };

      ++narc;
      i = (Integer) c1->data;
      if (i == 0) {
	 connidx[narc] = narc;
	 c1->data = (Universal) narc;
       }
      else {
	 connidx[narc] = connidx[i];
	 connidx[i] = narc;
       };
    };

   fprintf(otf,"\n\n\n\n\n");
   fprintf(otf,"%8d\n",GIOTTO_VERSION);
   fprintf(otf,"%8d%8d%8d%8d%8d\n",nnod,nnod,0,0,narc);
   fprintf(otf,"\n\n\n");

   i = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      ++i;
      fprintf(otf,"%8d%8d%8d%8d%8d%8d%8d%8d\n",GIOTTO_SHAPE,0,255,0,i,0,0,0);
      fprintf(otf,"0x%x\n",c);
    };

   fprintf(otf,"\n\n\n");

   i = 0;
   forin (c,COMPONENT *,l,COMPONENTS(o)) {
      ++i;
      fprintf(otf,"%8d%8d%8d%8d%8d\n",GIOTTO_NODE,i,0,0,c->data);
      c->data = (Universal) i;
    };

   fprintf(otf,"\n\n\n");
   fprintf(otf,"\n\n\n");

   i = 0;
   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      c = (COMPONENT *) GELO_conn_inq_from_data(conn);
      if (c == NULL) continue;
      c1 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c1 == NULL) continue;
      if (c == c1) continue;

      ++i;
      fprintf(otf,"%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
		 c1->data,0,-1,-1,connidx[i],0,i+1,0,0);

      ++i;
      fprintf(otf,"%8d%8d%8d%8d%8d%8d%8d%8d%8d\n",
		 c->data,0,-1,-1,connidx[i],0,i-1,0,0);
    };
};





/************************************************************************/
/*									*/
/*	run_giotto -- run the giotto program to get a layout		*/
/*									*/
/************************************************************************/


static Boolean
run_giotto(file,outfile)
   String file;
   String outfile;
{
   Character cmd[1024],sys[1024];
   Integer sts;
   Integer pid;
   Boolean old;

   old = FALSE;

   sprintf(sys,GIOTTO_PATH,BWEbwe_project(),BWEarch());

   if (access(sys,1) < 0) {
      sprintf(sys,OLD_GIOTTO_PATH,BWEbwe_project(),BWEarch());
      if (access(sys,1) < 0) return FALSE;
      old = TRUE;
    };

   if ((pid = vfork()) == 0) {
      if (!old) execl(sys,"giotto",file,outfile,0);
      else execl(sys,"giotto",file,0);
      _exit(127);
    };

   while (getpgrp(pid) > 0) CMPXselect(0);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	read_giotto_file -- read output from giotto			*/
/*									*/
/************************************************************************/


static Boolean
read_giotto_file(inf,o)
   FILE * inf;
   GELO_OBJECT o;
{
   Integer nsym,nnod,nfac,narc;
   Integer i,x,y,j;
   G_SYM syms;
   G_NODE nods;
   G_ARC arcs;
   Integer a,n;

   if (fscanf(inf,"%d",&x) != 1) return FALSE;
   if (fscanf(inf,"%d %d %d %d %d",&nsym,&nnod,&nfac,&x,&narc) != 5) return FALSE;

   syms = (G_SYM) alloca(sizeof(G_SYM_INFO)*nsym);
   nods = (G_NODE) alloca(sizeof(G_NODE_INFO)*nnod);
   arcs = (G_ARC) alloca(sizeof(G_ARC_INFO)*narc);

   for (i = 0; i < nsym; ++i) {
      if (fscanf(inf,"%d %d %d %d %d %d %d %d 0x%x",
		    &x,&x,&x,&x,&x,&x,&x,&x,&syms[i].comp) != 9)
	 return FALSE;
      syms[i].node = -1;
      syms[i].comp->data = (Universal) i;
      syms[i].arc = -1;
    };

   for (i = 0; i < nnod; ++i) {
      if (fscanf(inf,"%d %d %d %d %d",&nods[i].type,&nods[i].sym,&x,&x,&nods[i].arc) != 5)
	 return FALSE;
      nods[i].sym -= 1;
      nods[i].arc -= 1;
      if (nods[i].sym < 0) nods[i].comp = NULL;
      else {
	 if (syms[nods[i].sym].node < 0) syms[nods[i].sym].node = i;
	 nods[i].comp = syms[nods[i].sym].comp;
       };
      nods[i].xpos = UNSET_POS;
      nods[i].ypos = UNSET_POS;
    };

   for (i = 0; i < nfac; ++i) {
      if (fscanf(inf,"%d %d %d",&x,&x,&x) != 3) return FALSE;
    };

   for (i = 0; i < narc; ++i) {
      if (fscanf(inf,"%d %d %d %d %d %d %d %d %d",
		    &arcs[i].ltarg,&x,&arcs[i].len,&arcs[i].dir,&arcs[i].next,
		    &x,&arcs[i].comp,&arcs[i].fict,&x) != 9)
	 return FALSE;
      arcs[i].ltarg -= 1;
      arcs[i].next -= 1;
      arcs[i].comp -= 1;
      arcs[i].src = -1;
      arcs[i].targ = -1;
      arcs[i].link = -1;
      arcs[i].snext = -1;
      arcs[i].lsrc = -1;
    };

   set_node_pos(0,0,0,nods,arcs);

   for (i = 0; i < nnod; ++i) {
      if (nods[i].type == 2 || nods[i].type == 4 || nods[i].type == 5) continue;
      a = nods[i].arc;
      while (a >= 0) {
	 arcs[a].lsrc = i;
	 if (!arcs[a].fict) {
	    find_arc_tail(a,nods,arcs);
	    j = nods[i].sym;
	    arcs[a].src = j;
	    if (j >= 0) {
	       arcs[a].snext = syms[j].arc;
	       syms[j].arc = a;
	     };
	  };
	 a = arcs[a].next;
	 if (a == nods[i].arc) break;
       };
    };

   x = y = 0;
   for (i = 0; i < nnod; ++i) {
      if (nods[i].xpos == UNSET_POS) {
	 continue;
       };
      j = 1 - nods[i].xpos;
      if (j > x) x = j;
      j = 1 - nods[i].ypos;
      if (j > y) y = j;
    };
   for (i = 0; i < nnod; ++i) {
      nods[i].xpos += x;
      nods[i].ypos += y;
    };

   for (i = 0; i < nsym; ++i) {
      syms[i].comp->pos_x = nods[syms[i].node].xpos;
      syms[i].comp->pos_y = nods[syms[i].node].ypos;
    };

   giotto_pivots(o,nsym,syms,nods,arcs);

   return TRUE;
};





/************************************************************************/
/*									*/
/*	set_node_pos -- set node position				*/
/*									*/
/************************************************************************/


static void
set_node_pos(n,x,y,nods,arcs)
   Integer n;
   Integer x,y;
   G_NODE nods;
   G_ARC arcs;
{
   Integer a;

   if (nods[n].xpos != UNSET_POS) {
      if (nods[n].xpos != x || nods[n].ypos != y) abort();
      return;
    };

   nods[n].xpos = x;
   nods[n].ypos = y;

   a = nods[n].arc;
   while (a >= 0) {
      arcs[a].src = n;
      x = nods[n].xpos;
      y = nods[n].ypos;

      switch (arcs[a].dir) {
	 case 0 :
	    y += arcs[a].len;
	    break;
	 case 1 :
	    x -= arcs[a].len;
	    break;
	 case 2 :
	    y -= arcs[a].len;
	    break;
	 case 3 :
	    x += arcs[a].len;
	    break;
       };

      set_node_pos(arcs[a].ltarg,x,y,nods,arcs);

      a = arcs[a].next;
      if (a == nods[n].arc) break;
    };
};





/************************************************************************/
/*									*/
/*	find_arc_tail -- find end of arc				*/
/*									*/
/************************************************************************/


static Integer
find_arc_tail(a,nods,arcs)
   Integer a;
   G_NODE nods;
   G_ARC arcs;
{
   Integer n,i;

   n = arcs[a].ltarg;
   arcs[a].link = -1;

   if (nods[n].type == 2) {		/* Crossing			*/
      i = arcs[a].comp;
      i = arcs[i].next;
      i = arcs[i].next;
      arcs[a].link = i;
      n = find_arc_tail(i,nods,arcs);
    }
   else if (nods[n].type == 4 || nods[n].type == 5) {	/* Bend 	*/
      i = arcs[a].comp;
      i = arcs[i].next;
      if (arcs[i].fict) i = arcs[i].next;
      arcs[a].link = i;
      n = find_arc_tail(i,nods,arcs);
    }

   arcs[a].targ = nods[n].sym;

   return n;
};






/************************************************************************/
/*									*/
/*	giotto_pivots -- set pivot points for arcs			*/
/*									*/
/************************************************************************/


static void
giotto_pivots(o,nsym,syms,nods,arcs)
   GELO_OBJECT o;
   Integer nsym;
   G_SYM syms;
   G_NODE nods;
   G_ARC arcs;
{
   GELO_CONNECT conn;
   COMPONENT * c1, * c2;
   Integer i,j,k;
   Integer a,a0;
   GELO_PORT p1,p2;
   Sequence l;

   forin (conn,GELO_CONNECT,l,CONNECTS(o)) {
      GELO_clear_pivots(conn);
      c1 = (COMPONENT *) GELO_conn_inq_from_data(conn);
      c2 = (COMPONENT *) GELO_conn_inq_to_data(conn);
      if (c1 == NULL || c2 == NULL || c1 == c2) continue;
      j = (Integer) c1->data;
      if (j < 0 || j >= nsym || syms[j].comp != c1) continue;
      k = (Integer) c2->data;
      if (k < 0 || k >= nsym || syms[k].comp != c2) continue;

      for (a = syms[j].arc; a >= 0; a = arcs[a].snext) {
	 if (arcs[a].fict) continue;
	 if (arcs[a].targ == k) break;
       };

      if (a < 0) continue;

      switch (arcs[a].dir) {
	 case 0 : p1 = GELO_PORT_BOTTOM; break;
	 case 1 : p1 = GELO_PORT_LEFT; break;
	 case 2 : p1 = GELO_PORT_TOP; break;
	 case 3 : p1 = GELO_PORT_RIGHT; break;
       };

      i = arcs[a].lsrc;
      if (syms[nods[i].sym].node != i) {
	 GELO_connect_pivot(conn,nods[i].xpos,nods[i].ypos);
       };

      a0 = a;
      for (a = a0; arcs[a].link >= 0; a = arcs[a].link) {
	 i = arcs[a].ltarg;
	 GELO_connect_pivot(conn,nods[i].xpos,nods[i].ypos);
       };

      i = arcs[a].ltarg;
      if (syms[nods[i].sym].node != i) {
	 GELO_connect_pivot(conn,nods[i].xpos,nods[i].ypos);
       };

      switch (arcs[a].dir) {
	 case 0 : p2 = GELO_PORT_TOP; break;
	 case 1 : p2 = GELO_PORT_RIGHT; break;
	 case 2 : p2 = GELO_PORT_BOTTOM; break;
	 case 3 : p2 = GELO_PORT_LEFT; break;
       };

      /**** could set ports to p1,p2 **********/
    };
};





/* end of gelogiotto.c */









